#!/bin/bash
#:set ts=3
#
######################################################################
#
# Script to install Dialogic(R) Software Demo Kit
#
# Copyright (C) 2011 Dialogic Corporation.  All Rights Reserved.
#
# All names, products, and services mentioned herein are the
# trademarks or registered trademarks of their respective
# organizations and are the sole property of their respective
# owners.
#
######################################################################

### SCRIPT VARIABLES AND CONSTANTS

#Using the buildinfo file determine what type of platform
#i.e hmp or tvl2
tplatform=`awk  '/Major Number/ {print $6}' buildinfo`
trel=`awk  '/Release Minor Number/ {print $6}' buildinfo`
tsubrel=`awk  '/Release SubMinor Number/ {print $6}' buildinfo`
GCCVER=

# Release-specific definitions
BUS=""
COMPANY_NAME="dialogic"
FAMILY="Dialogic(R)"
MAINTREL=""
RELNUM=$trel
RELNAME="HMP Software Release ${RELNUM}LIN"
UPDATE=""
PLATFORM="`echo ${tplatform} | tr '[a-z]' '[A-Z]'`"

case "${tplatform}" in
	hmp)
		RELNAME="HMP Software Release ${RELNUM}LIN"
		
		case "$RELNUM" in
			1.5)
				COMPANY_NAME="intel"
				;;
			3.1 | 4.1)
				if [ "$tsubrel" != "-1" ]
				then
					RELNAME="${RELNUM}.${SUBRELNUM}LIN"
					FULLNAME=$RELNAME
				fi
				;;		
		esac
		;;
	ovl | tvl2)
		RELNAME="Dialogic(R) Multimedia Software Release ${RELNUM} for AdvancedTCA"
		FULLNAME="${RELNAME}"
		;;
	zvl)
		RELNAME="Multimedia Software for PCIe"
		FULLNAME="${RELNAME}"
		;;
esac

if [ -z "${UPDATE}" ]
then
        FULLNAME="${FAMILY} ${RELNAME}"
else
        FULLNAME="${FAMILY} ${RELNAME} ${UPDATE}"
fi

BUILDDIR=`pwd`
DEMOERROR=0

#Determine the platform either tvl2 or hmp by 
#looking at the Major Number in the top buildinfo file.
PLATFORM=""
DEMOSILENT=0

######################################################################
######################################################################
### GENERAL PURPOSE FUNCTIONS
######################################################################
######################################################################


#########################################################################
#    Process command line. Check for silent install. If requested,
#########################################################################
process_cmd_line()
{
        if [ $# -gt 0 ]
        then
                if [ "$1" == "--silent" ]
                then
                        DEMOSILENT=1
                fi
        fi
}



######################################################################
# Get OS release information
######################################################################
get_release_info()
{
        OS=`uname -s`
 
        if [ "${OS}" = "Linux" ] ; then
                if [ -f /etc/redhat-release ] ; then
                        DIST='RedHat'
                        PSUEDONAME=`cat /etc/redhat-release`
                elif [ -f /etc/SuSE-release ] ; then
                        DIST="SuSE"
                        PSUEDONAME=`cat /etc/SuSE-release`
                elif [ -f /etc/debian_version ] ; then
                        DIST="Debian"
                        PSUEDONAME=`cat /etc/debian_version`
                elif [ -f /etc/turbolinux-release ] ; then
                        DIST="Turbolinux"
                        PSUEDONAME=`cat /etc/turbolinux-release`
                else
                    DIST="UNKNOWN"
                    PSUEDONAME="UNKNOWN"
                fi
        else
                #Unsupported OS
                DIST="UNKNOWN"
                PSUEDONAME="UNKNOWN"
        fi
}

######################################################################
# Check if the OS release is valid
######################################################################
check_release()
{
   # Supported OS Distributions
   case "${DIST}" in
      Debian)
         # Supported OS Distribution
         ;;
      RedHat)
         # Supported OS Distribution
         ;;
      SuSE)
         # Supported OSD
         ;;
      Turbolinux)
      # Supported OS Distribution
         ;;
      UNKNOWN)
         # Unable to determine OS Distribution
         echo
         echo "Unable to determine the specific distribution of Linux you are running."
         echo
         exit 1
         ;;
      *)
         # Unsupported OSD (i.e., Mandrake)
         echo
         echo "OS Distribution is not supported."
         echo
         exit 1
         ;;
   esac
 
   # Unsupported OS Descriptions
   case "${PSUEDONAME}" in
      "Red Hat Enterprise Linux AS release 3 (Taroon)")
         # Unsupported OS version
         echo
         echo "${PSUEDONAME} is not supported."
         echo
         exit 1
         ;;
      UNKNOWN)
         # Unable to determine OS Version
         echo
         echo "Unable to determine the OS version"
         echo
         exit 1
         ;;
      *)
         # Supported OS Version (i.e., Red Hat AS 3.0 Update 1)
         ;;
   esac
}

######################################################################
# set the package commands
######################################################################
set_package_cmds()
{
   # Supported OS Distributions
   case "${DIST}" in
      Debian)
        PKG_EXT=".deb"
        PKGS=`ls ${LSBPKGNAME}*_*${PKG_EXT}`

        ERASE_PKG="dpkg --purge"
        INSTALL_PKG="dpkg -i"
        QUERY_PKG="dpkg -l"
        UPGRADE_PKG="dpkg -i"

        PKGBASE=${PKG%%_*}
        PKGNAME=${PKG%_*$PKG_EXT}
        ;;
      RedHat)
        PKG_EXT=".rpm"
        PKGS=`ls ${LSBPKGNAME}*-*${PKG_EXT}`

        ERASE_PKG="rpm --erase"
        INSTALL_PKG="rpm --install"
        QUERY_PKG="rpm --query"
        UPGRADE_PKG="rpm --upgrade -vh"

        PKGBASE=${PKG%%-*}
        PKGNAME=${PKG%.*$PKG_EXT}
        ;;
      SuSE)
        PKG_EXT=".rpm"
        PKGS=`ls ${LSBPKGNAME}*-*${PKG_EXT}`

        ERASE_PKG="rpm --erase"
        INSTALL_PKG="rpm --install"
        QUERY_PKG="rpm --query"
        UPGRADE_PKG="rpm --upgrade -vh"

        PKGBASE=${PKG%%-*}
        PKGNAME=${PKG%.*$PKG_EXT}
        ;;
      Turbolinux)
        PKG_EXT=".rpm"
        PKGS=`ls ${LSBPKGNAME}*-*${PKG_EXT}`

        ERASE_PKG="rpm --erase"
        INSTALL_PKG="rpm --install"
        QUERY_PKG="rpm --query"
        UPGRADE_PKG="rpm --upgrade -vh"

        PKGBASE=${PKG%%-*}
        PKGNAME=${PKG%.*$PKG_EXT}
		;;
      UNKNOWN)
        # Unable to determine OS Distribution
        echo
        echo "Unable to determine the specific distribution of Linux you are running."
        echo
        exit 1
        ;;
      *)
        # Unsupported OSD (i.e., Mandrake)
        echo
        echo "OS Distribution is not supported."
        echo
        exit 1
        ;;
   esac

	# determine the lsb package name
	LSBPKGNAME=`ls *${PKG_EXT}` &> /dev/null
	LSBPKGNAME=`echo ${LSBPKGNAME} | sed "s:lsb-${COMPANY_NAME}-::g" | sed 's:-.*::g'`
	LSBPKGNAME=lsb-${COMPANY_NAME}-${LSBPKGNAME}-
}

######################################################################
# query for installed packages
######################################################################
query_for_package()
{
	# Supported OS Distributions
    case "${DIST}" in
        Debian)
            # Supported OS Distribution
            FOUND_VERSION=`${QUERY_PKG} ${PKGBASE} | grep ${LSBPKGNAME}`
            FOUND_VERSION=${FOUND_VERSION##*$LSBPKGNAME }
            FOUND_VERSION=${FOUND_VERSION%% *}
            FOUND_VERSION="${PKGBASE}_${FOUND_VERSION}"
            ;;
        RedHat)
            # Supported OS Distribution
            FOUND_VERSION=`${QUERY_PKG} ${PKGBASE}`
            ;;
        SuSE)
            # Supported OSD
            FOUND_VERSION=`${QUERY_PKG} ${PKGBASE}`
            ;;
        Turbolinux)
            # Supported OS Distribution
            FOUND_VERSION=`${QUERY_PKG} ${PKGBASE}`
            ;;
        UNKNOWN)
            # Unable to determine OS Distribution
            echo
            echo "Unable to determine the specific distribution of Linux you are running."
            echo
            myexit 1
            ;;
        *)
            # Unsupported OSD (i.e., Mandrake)
            echo
            echo "OS Distribution is not supported."
            echo
            exit 1
            ;;
    esac
}

######################################################################
# Check for a supported GCC version
######################################################################
check_gcc()
{	
    # determine from the RPMs in the build which compiler version(s)
	# we are supporting
	SUPP_GCC=`ls ${LSBPKGNAME}*_gcc*${PKG_EXT} | sed -e 's/${LSBPKGNAME}[0-z]*-[.-9]*-[0-9]*_gcc//g' -e 's/\.[0-z]*${PKG_EXT}//g' | sort | uniq`
	
	# check if env DLG_NO_GCC is set
	if [ -n "${DLG_NO_GCC}" ]
	then
                echo "dlg_no_gcc is on, check force_gcc_version"
		if [ -n "${FORCE_GCC_VERSION}" ]
		then
			echo " "
			GCCVER=${FORCE_GCC_VERSION}
		else
			echo "FATAL ERROR:"
			echo "FATAL ERROR: DLG_NO_GCC environment variable is set but FORCE_GCC_VERSION is not set"
			echo "FATAL ERROR: Installation aborted"
			echo "FATAL ERROR:"
			exit 1
		fi
	else
		# see if the gcc compiler RPM is installed
		if ! ${QUERY_PKG} gcc &> /dev/null
		then
			echo "FATAL ERROR:"
			echo "FATAL ERROR: gcc compiler not found"
			echo "FATAL ERROR:"
			for i in ${SUPP_GCC}
			do
				echo -ne "FATAL ERROR:\t\tGCC $i\n"
			done
			echo "FATAL ERROR: Please install the gcc compiler and rerun $0."
			echo "FATAL ERROR:"
			echo "FATAL ERROR: Installation aborted due to missing gcc compiler."
			echo "FATAL ERROR:"
			exit 1
		fi
		
		# we found some version of the gcc compiler, so get it's version
		# fix problem in parsing gcc -v output: the existing one assumes the output contains 4+ columns, 
		# ex "gcc version 4.1.2 20080704 (Red Hat 4.1.2-46)
		# but depending on gcc installation, the 4th column may not show. Also, for none-English version of RH, the output could be different
		# use perl to parse the output and get the 3rd column "4.1.2" in the above example
		#GCCVER=`gcc -v 2>&1 | sed -e '/gcc version/!d' -e '/gcc version/{s/.*gcc version \([0-9,.]*\) .*/\1/;}' | cut -c1-3`
		GCCVER=`gcc -v 2>&1| perl -e 'my $line;while(<>){$line=$_;if ($line=~s/^gcc\p{isSpace}*[^\p{isSpace}]*\p{isSpace}*(\p{isWord}*\p{isPunct}\p{isWord}*).*/$1/){chomp $line;print $line, "\n"; } }'`

		# 4.x -> 3.4
		[ "${GCCVER}" = "4.4" ] && GCCVER=4.1
		[ "${GCCVER}" = "4.3" ] && GCCVER=4.1
		[ "${GCCVER}" = "4.2" ] && GCCVER=4.1
		[ "${GCCVER}" = "4.1" ] && GCCVER=4.1
		[ "${GCCVER}" = "4.0" ] && GCCVER=4.1

		# GCC 3.2 binaries can be used under GCC 3.3
		[ "${GCCVER}" = "3.3" ] && GCCVER=3.2
	fi

	# if the FORCE_GCC_VERSION environment variable is set 
		#
		# If the user wants to force a particular set of RPMs to install
		# then the "FORCE_GCC_VERSION" environment variable should be set
		# before this script is run. The value of this variable must contain
		# only the first three characters of the GCC version (e.g. 3.2 for
		# GCC 3.2.3).
		#
		# Example:
		#	FORCE_GCC_VERSION=3.2 ./install.sh
		#
		# This would force the installation of Dialogic RPM's built with
		# GCC 3.2 to be installed REGARDLESS OF WHICH COMPILER VERSION IS ON YOUR
		# SYSTEM.
		#
		# CAUTION: USE THIS CAPABILITY AT YOUR OWN RISK
		#
		if [ -n "${FORCE_GCC_VERSION}" ]
		then
			# check to see if the specified value matches one of the compiler
			# versions for which we have binary RPMs
			echo "${SUPP_GCC}" | grep -q "${FORCE_GCC_VERSION}"
			RC=$?
			if [ ${RC} -eq 0 ]
			then
				# the user specified a GCC version for which we have binary RPMs
				echo
				echo "WARNING:"
				echo "WARNING: AUTOMATIC GCC VERSION DETECTION OVERRIDDEN BY THE USER."
				echo "WARNING:"
				echo "WARNING: FORCING INSTALLATION OF BINARY RPMS BUILT USING GCC ${FORCE_GCC_VERSION}"
				echo "WARNING: ON A SYSTEM THAT "
				if [ -n "${DLG_NO_GCC}" ]
				then 
					echo "WARNING: DOESN'T HAVE GCC INSTALLED"
				else
					echo "WARNING: HAS GCC ${GCCVER}." 
				fi
				echo "WARNING:"
				echo "WARNING: COMPATIBILITY ISSUES MAY EXIST.  SEE RELEASE DOCUMENTATION"
				echo "WARNING: FOR SUPPORTED GCC VERSIONS."
				echo "WARNING:"
				echo

				# force the compiler version
				GCCVER=${FORCE_GCC_VERSION}

				# tell RPM top ignore GCC version conflicts
				NODEPS="--nodeps"
			else
				# user specified GCC version for which we DO NOT have binary RPMs
				echo "FATAL ERROR:"
				echo "FATAL ERROR: You specified a GCC version using the FORCE_GCC_VERSION environment variable"
				echo "FATAL ERROR: for which binary RPMs are not included in the build."
				echo "FATAL ERROR:"
				echo "FATAL ERROR: Valid values for FORCE_GCC_VERSION are:"
				echo "FATAL ERROR:"
				for i in ${SUPP_GCC}
				do
					echo -ne "FATAL ERROR:\t\t$i\n"
					echo "FATAL ERROR:"
				done
				echo "FATAL ERROR: Installation aborted due to invalid value of FORCE_GCC_VERSION."
				echo "FATAL ERROR:"
				exit 1
			fi
		fi
		
	#
	# SUPP_GCC - list of RPM (DEB on debian) files.
	#
	echo ${SUPP_GCC} | grep -q "gcc${GCCVER}" 
	RC=$?

	if [ ${RC} -eq 1 ]
	then
		echo "FATAL ERROR:"
		echo "FATAL ERROR: Unsupported gcc compiler detected"
		echo "FATAL ERROR:"
		echo "FATAL ERROR: gcc version ${GCCVER} was detected on your system."
		echo "FATAL ERROR:"
		echo "FATAL ERROR: This software requires that one of the following versions"
		echo "FATAL ERROR: of the gcc compiler be installed:"
		echo "FATAL ERROR:"
		for i in ${SUPP_GCC}
		do
			echo -ne "FATAL ERROR:\t\tGCC $i\n"
		done
		echo "FATAL ERROR:"
		echo "FATAL ERROR: Please install a supported version of the gcc compiler and rerun $0."
		echo "FATAL ERROR:"
		echo "FATAL ERROR: Installation aborted due to unsupported gcc compiler."
		echo "FATAL ERROR:"
		exit 1
	fi
}

######################################################################
# Print the error from the installation return code
#     $1 - return code
######################################################################
disp_install_return_code()
{
    case "$1" in
        0)
            echo "Installation successful."
            ;;
        *)
            NUMRPMS=`echo ${2} | wc -w`

            # check if the return code is equal to the number of RPMs
            if [ $NUMRPMS -ne $1  ]
            then
                echo
                echo "Fatal error during installation. Installation terminated."
                exit 1
            else
                # validate that the RPMs got installed
                for RPM in $2
                do
                    # get the 4 part RPM name
                    PKG=${RPM%%-[0-9].*}

                    # verify that the package was actually installed
                    ${QUERY_PKG} $PKG &> /dev/null
                    RET=$?

                    if [ $RET -ne 0  ]
                    then
                        echo
                        echo "Fatal error during installation. Installation terminated."
                        exit 1
                    fi
                done
            fi
            ;;
    esac
}

######################################################################
######################################################################
### BEGINNING OF SCRIPT
######################################################################
######################################################################

process_cmd_line $*
get_release_info
check_release
set_package_cmds
check_gcc

# loop through all RPM's
for PKG in ${PKGS}
do
    # check if the rpm name is for the gcc version on the system
    found=`echo "$PKG" | grep "gcc${GCCVER}"`
    if [ -n "${found}" ]
    then
        set_package_cmds
        query_for_package ${PKGBASE}

        if [ "${FOUND_VERSION}" != "${PKGNAME}" ]
        then
            # what is currently installed does not match that in our package
            RET=0
            if [ "$DEMOSILENT" -eq 0 ]
            then
                ${UPGRADE_PKG} ${PKG} 2>/dev/null
            else
                ${UPGRADE_PKG} ${PKG} &>/dev/null
            fi
            RET=$?
            disp_install_return_code ${RET} ${PKG}
 
            [ ${RET} -ne 0 ] && DEMOERROR=1
        else
            echo "${FULLNAME}"
            echo "Software Demo Kit for Linux is already installed."
            exit "${DEMOERROR}"
        fi
    fi
done


exit "${DEMOERROR}"




