#!/bin/bash
#:set ts=3
#
######################################################################
#
# Top-Level script to install Dialogic(R) Redistributable
# Runtime and Software Development Kit for Linux
#
# Copyright (C) 2002-2013 Dialogic Corporation.  All Rights Reserved.
#
# All names, products, and services mentioned herein are the
# trademarks or registered trademarks of their respective
# organizations and are the sole property of their respective
# owners.
#
######################################################################

#Using the buildinfo file determine what type of platform
#i.e hmp or tvl2
tplatform=`awk  '/Major Number/ {print $6}' buildinfo`
trel=`awk  '/Release Minor Number/ {print $6}' buildinfo`
tsubrel=`awk  '/Release SubMinor Number/ {print $6}'  buildinfo`

# Release-specific definitions

BUS=""
FAMILY="Dialogic(R)"
MAINTREL=""
RELNUM=$trel
SUBRELNUM=$tsubrel
RELNAME=""
UPDATE=""
PLATFORM="`echo ${tplatform} | tr '[a-z]' '[A-Z]'`"

SNMPORT=""
CLIPORT=""
TOPSILENT=0
RUNTIME_SUCCESSFUL=0

#Specific to TVL
amcBusIfName=""
amcBusMac=""
amcIpAddr=""
amcNetMask=""
HpetEnabled=0

#########################################################################
#Determine the platform either by 
#looking at the Major Number in the top buildinfo file.
#########################################################################
setplatformType()
{
	if [ "$tplatform" = "tvl2" ] || [ "$tplatform" = "ovl" ]
	then
		RELNAME="Multimedia Software Release ${RELNUM} for AdvancedTCA";
	elif [ "$tplatform" = "zvl" ]
	then
		RELNAME="Multimedia Kit Software Release ${RELNUM} for PCIe";
	elif [ "$tplatform" = "hmp" ] && [ "$tsubrel" = "-1" ]
	then 
		RELNAME="HMP Software Release ${RELNUM}LIN";
	else
		RELNAME="${RELNUM}.${SUBRELNUM}LIN";
	fi
}

#########################################################################
#    Process command line. Check for silent install. If requested,
#########################################################################
process_cmd_line()
{
        if [ $# -gt 0 ]
        then
                if [ "$1" == "--silent" ]
                then
                        TOPSILENT=1
                fi
        fi
}

checkSharedMemory()
{
    SHARED_MAX_BYTE=`more /proc/sys/kernel/shmmax`
    if [ ${SHARED_MAX_BYTE} -lt 65536000 ]
    then
        SHARED_MAX_MEG=`echo $SHARED_MAX_BYTE 1024000 | awk '{print $1/$2}'`
        echo
        echo "Warning: The shared memory limit on this system is currently set to $SHARED_MAX_MEG MB."
        echo "In order to be able to run more than 120 instances of a single resource"
        echo "type, you will need to increase the shared memory maximum (kernel.shmmax)"
        echo "in /etc/sysctl.conf to 64 MB or more."
        echo
        echo -e "Press ENTER to continue\c"

        read _ANS
    fi
}

islibstdcplplInstalled()
{
     rpm --queryformat='%{NAME}-%{VERSION}-%{RELEASE}.%{arch}\n' -qa | grep ^libstdc | grep "i686\|i586\|i386"
    if [ $? -ne 0 ]
    then
        echo "The 32-bit libstdc++ package, required by HMP, was not found."
        echo "Please correct the issue before attempting HMP installation again."
        exit 1
    fi
}

#########################################################################
# Parse the output of dmesg utility. 
# looks for "Using *** HPET"
# example of mesg output: demsg | grep -i hpet
# ACPI: HPET (v001 INTEL  S5000PAL 0x00000001 INTL 0x01000013) @ 0x7fb2e000
# ACPI: HPET id: 0x8086a201 base: 0xfed00000
# hpet0: at MMIO 0xfed00000 (virtual 0xf8800000), IRQs 2, 8, 0
# hpet0: 3 64-bit timers, 14318180 Hz
# Using HPET for base-timer
# hpet_resources: 0xfed00000 is busy
# Time: hpet clocksource has been installed.
#########################################################################
isHpetEnabled()
{
	# warning on HPET enabled
    	hpetFound=`dmesg |grep -i hpet 2>&1| perl -e 'my $line; while (<>) { $line = $_; if ($line =~ m/Using(\p{isSpace}*[^\p{isSpace}]*\p{isSpace}*)*hpet/i){print "Found HPET"; last;}}'`
    	if [ ! "${hpetFound}" = "" ] 
    	then
       	    HpetEnabled=1
            echo "WARNING: Detected that HPET is used as time source"
       	    echo "WARNING: HPET may be enabled. It is recommended to disable the HPET when running HMP"
            echo "WARNING:"
    	fi
}

#########################################################################
#    Get AMC Informatino from hwconf and ifconfig.
#########################################################################
getAMCInformation()
{

AMCPCIBUSID=`awk  ' /ParamValue/ ' /usr/dialogic/cfg/mitconfig/ipms_pcibus_map_cfg.xml \
	   | sed -ne 's/<ParamValue>\([0-9]*\)<\/ParamValue>/\1/p' | \
	   awk 'BEGIN{ln=0; BMI_EXTIF_TBL_BUSID=8; IPMS_EXT_IF_TYPE_FABRIC_RTM=4 }; 
		      { ln++; if ( ln == BMI_EXTIF_TBL_BUSID  && $1 == IPMS_EXT_IF_TYPE_FABRIC_RTM ) {print busId}; busId=$1  }' `
	 
  
#amcBusIdLine=`awk  ' /class:\s*/  { if( $2 == "NETWORK") { devcls=$2 } } 
#	/device:\s*/ { if( devcls=="NETWORK") { devname=$2 }  } 
#	/network.hwaddr:\s*/ { if (devcls=="NETWORK") {devMac=$2} }
#	/pcibus:\s*/ {if (devcls=="NETWORK") 
#	{ devBus=$2; 
#	  printf ("devBusId=%d:devName=%s:mac=%s\n",devBus,devname, devMac); 
#	  devcls="";} } ' /etc/sysconfig/hwconf | awk '/devBusId='$AMCPCIBUSID'/ {print $1}' ` 


amcBusIfName=`awk  ' /class:\s*/  { if( $2 == "NETWORK") { devcls=$2 } } 
	/device:\s*/ { if( devcls=="NETWORK") { devname=$2 }  } 
	/network.hwaddr:\s*/ { if (devcls=="NETWORK") {devMac=$2} }
	/pcibus:\s*/ {if (devcls=="NETWORK") {devBus=$2} } 
	/pcifn:\s*/ { pcifn =$2; if ( (devcls=="NETWORK") && (pcifn==0) )
	{ printf ("devBusId=%d:devName=%s:mac=%s\n",devBus,devname, devMac); 
	  devcls="";} } ' /etc/sysconfig/hwconf | awk '/devBusId='$AMCPCIBUSID'/ {print $1}' | sed -ne 's/.*devName\=\([a-zA-Z0-9]*\).*$/\1/p' ` 


if [ -z "$amcBusIfName" ]
then
	return 1
fi


amcBusMac=`awk  ' /class:\s*/  { if( $2 == "NETWORK") { devcls=$2 } } 
	/device:\s*/ { if( devcls=="NETWORK") { devname=$2 }  } 
	/network.hwaddr:\s*/ { if (devcls=="NETWORK") {devMac=$2} }
	/pcibus:\s*/ {if (devcls=="NETWORK") {devBus=$2} } 
	/pcifn:\s*/ { pcifn =$2; if ( (devcls=="NETWORK") && (pcifn==0) )
	{ printf ("devBusId=%d:devName=%s:mac=%s\n",devBus,devname, devMac); 
	  devcls="";} } ' \
 /etc/sysconfig/hwconf | awk '/devBusId='$AMCPCIBUSID'/ {print $1}' | sed -ne 's/.*mac\=\(\([a-zA-Z0-9]\{2\}:\)\{5\}[a-zA-Z0-9]\{2\}\).*$/\1/p' ` 



ifd=`/sbin/ifconfig $amcBusIfName`

if [ -z "$ifd" ]
then
	return 2
else
	amcIpAddr=`/sbin/ifconfig $amcBusIfName | sed -ne 's/.*inet addr:\(\([0-9]\{1,3\}\.\)\{3\}[0-9]\{1,3\}\).*$/\1/p' ` 
	amcNetMask=`/sbin/ifconfig $amcBusIfName | sed -ne 's/.*Mask:\(\([0-9]\{1,3\}\.\)\{3\}[0-9]\{1,3\}\).*$/\1/p' ` 
fi

	return 0
}

#########################################################################
#    Execute additional install steps.
#########################################################################
ExecuteAdditionalInstallation()
{

if [ "$PLATFORM" = "HMP" ] && [ "$RELNUM" = "3.1" ] || 
	[ "$RELNUM" = "4.1" ] && [ "$PLATFORM" = "HMP"  ] ||
	[ "$PLATFORM" = "OVL" ] ||
	[ "$PLATFORM" = "TVL2" ] ||
	[ "$PLATFORM" = "ZVL" ] && 
	[ $TOPSILENT -eq 0 ] &&
	[ ${RUNTIME_SUCCESSFUL} -eq 0 ]
then
	changeAgentPorts

	echo
	echo "Installation of $FULLNAME successfully completed."
	echo
fi

}

##################################################################################
#    Execute additional install steps 1) Tvl2StartUp script for ZVL/OVL/TVL
##################################################################################
ExecuteTvl2StartUpScript()
{
 
if 		[ "$PLATFORM" = "OVL" ] || [ "$PLATFORM" = "TVL2" ] || [ "$PLATFORM" = "ZVL" ] &&
        [ $TOPSILENT -eq 0 ] &&
        [ ${RUNTIME_SUCCESSFUL} -eq 0 ]  
then
        TVL2_START="/etc/init.d/tvl2_startup start"
        $TVL2_START
        RC=$?
        
        echo "Execution of startup script tvl2_startup completed."
        echo
fi
 
}

##################################################################################
#    Execute additional install steps 2) 
#    validate the operation of the /dev/rtc driver on the system. 
#    If the test fails, HPET may be enabled. print out warning
##################################################################################
ExecuteRtcDriverTest()
{
	if 	[ "$PLATFORM" = "HMP"  ] && [ ${RUNTIME_SUCCESSFUL} -eq 0 ]  
	then
	    if [ -x /usr/dialogic/bin/rtc_irq_test ]
        then
		    /usr/dialogic/bin/rtc_irq_test &>/dev/null
        	RC=$?
        	if [ ${RC} -ne 0 ]
            then
			    echo "WARNING: "
			    echo "WARNING: RTC IRQ test failed, The HPET may be enabled."
			    echo "WARNING: It is recommended to disable the HPET when running HMP"
				echo "WARNING: Please see HMP Software Installation Guide for more information"
			    echo
			fi
		fi
	fi
 
}

#########################################################################
#    Get SNMP Agent Listening Port from ct_intel.
#########################################################################
findSnmpAgentPort()
{
	if [ -f /etc/init.d/ct_intel ]
	then
		#SNMPORT=`sed -ne 's/.*-sp\s*\([0-9]*\).*/\1/p ' /etc/init.d/ct_intel`
		SNMPORT=`sed -ne 's/\s*SNMP_LISTENER\s*=\s*\([0-9]*\)/\1/p' /etc/init.d/ct_intel`
	else
		SNMPORT="161"
	fi
}

#########################################################################
#    Get CLI Agent Listening Port from ct_intel.
#########################################################################
findCliAgentPort()
{
	if [ -f /etc/init.d/ct_intel ]
	then
		#CLIPORT=`sed -ne 's/.*-ct\s*\([0-9]*\).*/\1/p ' /etc/init.d/ct_intel`
		CLIPORT=`sed -ne 's/\s*CLI_LISTENER\s*=\s*\([0-9]*\)/\1/p' /etc/init.d/ct_intel`
	else
		CLIPORT="23"
	fi
}



#########################################################################
#    Change Agent Listening Port in ct_intel.
#########################################################################
changeAgentPorts()
{
	validPort=""
	findSnmpAgentPort
	echo -ne "Do you wish to configure the SNMP Agent Listening Port (default is $SNMPORT) [y/n] ? \c"
	read line
	resp=`echo ${line} | tr '[A-Z]' '[a-z]' | cut -c1`
	#if [ -z "${resp}" ] || [ "${resp}" = "y" ]
	if [ -z "${resp}" ]
	then
		echo "SNMP port $SNMPORT accepted."
	elif [ "${resp}" = "y" ]
	then
        	wexit=0
        	while [ $wexit = 0 ]
        	do
                	echo -ne "Please Enter SNMP Agent Listening Port? \c"
                	read port
  			validPort=`echo $port | sed -ne 's/^\([0-9]*\)$/\1/p' `
			if [ -n "$validPort" ]
			then
                        	#if [ -n "$validPort" ] && [ "$port" -lt 32767 ]
                        	if [ "$port" -lt 32767 ]
                		then
                        		#sed 's/-sp *[0-9]*/-sp '$port'/' /etc/init.d/ct_intel > footmp
					sed 's/SNMP_LISTENER\s*=\s*[0-9]*/SNMP_LISTENER='$port'/' /etc/init.d/ct_intel > /tmp/footmp	
                        		mv /tmp/footmp /etc/init.d/ct_intel
					chmod 500 /etc/init.d/ct_intel > /dev/null
					SNMPORT=$port
					echo "SNMP port $port accepted."
                        		wexit=1
                		else
                        		echo "Invalid Port: port exceeds range."
				fi
                	else
                        	echo "Invalid Port: must be numeric."
                	fi
        	done
	fi

	findCliAgentPort
	echo -ne "Do you wish to configure the CLI Agent Telnet Port (default is $CLIPORT) [y/n] ? \c"
	read line
	resp=`echo ${line} | tr '[A-Z]' '[a-z]' | cut -c1`
	#if [ -z "${resp}" ] || [ "${resp}" = "y" ]
	if [ -z "${resp}" ]
	then
		echo "CLI port $CLIPORT accepted."
	elif [ "${resp}" = "y" ]
	then
        	wexit=0;
        	while [ $wexit = 0 ]
        	do
                	echo -ne "Please Enter CLI Agent Telnet Port? \c"
                	read port

			validPort=`echo $port | sed -ne 's/^\([0-9]*\)$/\1/p' `
			if [ -n "$validPort" ]
			then
               			if [ "$port" -eq "$SNMPORT" ]
				then
               				echo "Invalid Port: Same as SNMP Agent Port...choose a different port."
				#elif [ "$validPort" ] && [ "$port" -lt 32767 ]
				elif  [ "$port" -lt 32767 ]
               			then
					sed 's/CLI_LISTENER\s*=\s*[0-9]*/CLI_LISTENER='$port'/' /etc/init.d/ct_intel > /tmp/footmp	
               				mv /tmp/footmp /etc/init.d/ct_intel
					chmod 500 /etc/init.d/ct_intel > /dev/null
					echo "CLI port $port accepted."
               				wexit=1
               			else
               				echo "Invalid Port: exceeds port range"
				fi
			else
               			echo "Invalid Port: must be numeric"
			fi
       		done
	fi
}

#########################################################################
#    Install Runtime Packages.
#########################################################################
installRuntime()
{
cd redistributable-runtime

echo
echo "Installing ${FULLNAME}"
echo "Redistributable Runtime"
echo "Please wait, installation is in progress..."
echo

RET=0
if [ -x ./install.sh ]
then
	if [ "$PLATFORM" == "TVL2" ] || [ "$PLATFORM" == "OVL" ]
	then
		./install.sh --silent ALL
		RET=$?
	else
		#./install.sh
		#Added by J. Cruz for silent install to work
		./install.sh $*
		RET=$?
	fi

else
	echo
	echo "ERROR: install.sh not found in the redistributable-runtime directory".
	echo
	RET=1
fi

RUNTIME_SUCCESSFUL=${RET}

if [ ${RET} -eq 0 ] 
then
	if [ "$PLATFORM" != "TVL2" ] || [ "$PLATFORM" != "OVL" ]
	then
		echo
		echo "Installation of ${FULLNAME} Redistributable Runtime was successful."
		echo
	fi
elif    [ ${RET} -eq 1 ]
then
        echo
        echo "An error occurred while installing the Redistributable Runtime"
        echo "Software.  Aborting installation of the Software Development"
        echo "Kit."
        echo
        exit 1
elif    [ ${RET} -eq 2 ]
then
        echo "Based on your selection no runtime packages were installed."
elif    [ ${RET} -eq 3 ]
then
        echo
        echo "Installation of the Redistributable Runtime Software"
        echo "failed due to lack of required disk space. Please"
        echo "free up some disk space and re-run $0 again."
        echo "Aborting installation of the Software Development Kit."
        echo
        exit 3
fi


cd ..
}


#########################################################################
#    Install SDK Package.
#########################################################################
installSdk()
{
# Prompt the user about installing the SDK
if [ "$PLATFORM" == "TVL2" ] || [ "$PLATFORM" == "OVL" ]
then
	RET=0
	if [ -x ./install.sh ]
	then
		cd sdk
		#./install.sh
		#Added by J. Cruz for silent install to work
		#It looks like the sdk silent install does not obey 
		#./install.sh --silent 
		./install.sh --silent
		RET=$?
	else
		echo
		echo "ERROR: SDK install.sh not found in the sdk directory".
		echo
		RET=1
	fi

	if [ ${RET} -eq 0 ]
	then
	   	echo
	else
		echo
	   	echo "An error occurred while installing the Software Development Kit."
	   	echo
	   	exit 1
	fi
	cd ..
else
	echo
	echo -ne "Do you wish to install the Software Development Kit [y/n] ? \c"
	read line
	resp=`echo ${line} | tr '[A-Z]' '[a-z]' | cut -c1`
	if [ -z "${resp}" ] || [ "${resp}" = "y" ]
	then
		# must install sdk after redistributable-runtime to avoid file
		# conflicts with any previously existing RPM's.

		cd sdk

		echo
		echo "Installing ${FULLNAME} Software Development Kit"
		echo

		RET=0
		if [ -x ./install.sh ]
		then
			#./install.sh
			#Added by J. Cruz for silent install to work
			#It looks like the sdk silent install does not obey 
			./install.sh $*
			RET=$?
		else
			echo
			echo "ERROR: install.sh not found in the sdk directory".
			echo

			RET=1
		fi

		if [ ${RET} -eq 0 ]
		then
	   		echo
	   		echo "Installation of ${FULLNAME} Software Development Kit was successful."
	   		echo
		else
	   		echo
	   		echo "An error occurred while installing the Software Development Kit."
	   		echo
	   		exit 1
		fi
		cd ..
	elif [ -z "${resp}" ] || [ "${resp}" = "n" ]
	then
		echo
		echo "Installation of the Software Development Kit was skipped."
		echo
	fi
fi
}
	
#########################################################################
#    Install Demo Package.
#########################################################################
installDemo()
{
# Optional demo package
	echo
	echo -ne "Do you wish to install the Software Demo Kit [y/n] ? \c"
	read line
	resp=`echo ${line} | tr '[A-Z]' '[a-z]' | cut -c1`
	if [ -z "${resp}" ] || [ "${resp}" = "y" ]
	then
		cd demo

		echo
		echo "Installing ${FULLNAME} Software Demo Kit"
		echo

		RET=0
		if [ -x ./install.sh ]
		then
			./install.sh $*
			RET=$?
		else
			echo
			echo "ERROR: install.sh not found in the demos directory".
			echo
			RET=1
		fi

		if [ ${RET} -eq 0 ]
		then
	   		echo
	   		echo "Installation of ${FULLNAME} Software Demo Kit was successful."
	   		echo
		else
	   		echo
	   		echo "An error occurred while installing the Software Demo Kit."
	   		echo
	   		exit 1
		fi
		cd ..
	elif [ -z "${resp}" ] || [ "${resp}" = "n" ]
	then
		echo
		echo "Installation of the Software Demo Kit was skipped."
		echo
	fi
}

#########################################################################
#    Install Media Server Language Prompts Package.
#########################################################################
installMediaServerPrompts()
{
    # only install on HMP4.1 Redhat
    if [ "$PLATFORM" = "HMP" ] && [ "${RELNUM}" = "4.1" ]
    then
        # only for Redhat
        OS=`uname -s`
 
        if [ "${OS}" = "Linux" ] && [ -f /etc/redhat-release ] 
        then  
            #  DIST='RedHat'
            # check if media_server_prompts directory and media prompt files exist 
            if [ -d media_server_prompts ] && [ -f media_server_prompts/*.rpm ]
            then
                cd media_server_prompts
                echo 
                echo "Installing media server prompt package"
                rpm -ivh *.rpm
                cd ..
                echo
            fi
        fi
    fi
}

######################################################################
# Display User Notice
#
######################################################################
displayNotice()
{
if [ ${RUNTIME_SUCCESSFUL} = 0 ]
then
	if [ "$PLATFORM" = "HMP" ] && [ "${RELNUM}" = "3.1" ]
	then
                echo "----------------------------------------------------------------------------"
                echo "--                                                                        --"
                echo "-- NOTE:                                                                  --"
                echo "-- (1) The default Verification License has been automatically installed. --"
                echo "-- (2) The Dialogic(R) system services will automatically start           --"
                echo "--     every time the system is rebooted. To start and stop system        --"
                echo "--     services manually, use the dlstop and dlstart scripts found in     --"
                echo "--     /usr/dialogic/bin.                                                 --"
                echo "-- (3) For administration via SNMP, use any SNMP browser, library or tools--"
                echo "-- (4) For administration via CLI, use the telnet command                 --"
                echo "-- (5) For details, refer to documentation                                --"
                echo "--                                                                        --"
                echo "-- NEXT STEPS:                                                            --"
                echo "-- You need to do the following before using the product:                 --"
                echo "-- (1) Before using the software, you must also ensure that the           --"
                echo "--     environment variables are set by performing the following action:  --"
                echo "--     - Logout and login                                                 --"
                echo "-- (2) The Dialogic(R) system services must be started. To start system   --"
                echo "--     services manually, use the dlstart script found in                 --"
                echo "--     /usr/dialogic/bin.                                                 --"
                echo "-- (3) Configure an appropriate license (higher density) via SNMP or CLI  --"
                echo "--                                                                        --"
                echo "----------------------------------------------------------------------------"
	elif [ "$PLATFORM" = "TVL2" ] || [ "$PLATFORM" == "OVL" ] || [ "$PLATFORM" = "ZVL" ]
	then
		# Notice that config.sh is not executed...
		echo "----------------------------------------------------------------------------"
		echo "--                                                                        --"
		echo "-- NOTE:                                                                  --"
		echo "--                                                                        --"
		echo "-- (1) The default Verification License has been automatically installed. --"
		echo "-- (2) The Dialogic(R) system services will automatically start every     --"
		echo "--     time the system is rebooted. To start and stop system services     --"
		echo "--     manually, use the dlstop and dlstart scripts found in              --"
		echo "--     /usr/dialogic/bin.                                                 --"
		echo "-- (3) For administration via SNMP, use any SNMP browser, library or tools--"
		echo "-- (4) For administration via CLI, use the telnet command                 --"
		echo "-- (5) For details, refer to documentation                                --"
		echo "--                                                                        --"
		echo "-- NEXT STEPS:                                                            --"
		echo "-- (1) Before using the software, you MUST do one of the following:       --"
		echo "--     a) Reboot the system.      -- OR --                                --"
		echo "--     b) Do the following:                                               --"
		echo "--        (i) Ensure that the environment variables are set by performing --"
		echo "--            the following action:                                       --"
		echo "--            - Logout and login                                          --"
		#echo "--        (ii) Manually run the required startup script                  --"
		#echo "--            - /etc/init.d/tvl2_startup start                           --"
		echo "--        (ii) Start the Dialogic(R) system services manually using the   --"
		echo "--             dlstart script found in /usr/dialogic/bin.                 --"
		echo "-- (2) Configure your network                                             --"
		echo "-- (3) Configure an appropriate license (higher density) via SNMP or CLI  --"
		echo "-- (4) Configure the Media Ethernet Interfaces via SNMP or CLI            --"
                echo "-- (5) Ensure that the MMA board interface IP address and DSP IP          --"
                echo "--     addresses are on same sub-net via SNMP or CLI                      --"   
		echo "-- (6) If you are using PSTN, refer documentation on how to configure     --"
		echo "--     PSTN via SNMP or CLI                                               --"
		echo "--                                                                        --"
		echo "----------------------------------------------------------------------------"
	elif [ "$PLATFORM" = "HMP" ] && [ "${RELNUM}" = "4.1" ]
	then
		# Notice that config.sh is not executed...
		echo "----------------------------------------------------------------------------"
		echo "--                                                                        --"
		echo "-- NOTE:                                                                  --"
		echo "--                                                                        --"
		echo "-- (1) The default Verification License has been automatically installed. --"
		echo "-- (2) The Dialogic(R) system services will automatically start every     --"
		echo "--     time the system is rebooted. To start and stop system services     --"
		echo "--     manually, use the dlstop and dlstart scripts found in              --"
		echo "--     /usr/dialogic/bin.                                                 --"
		echo "-- (3) For administration via SNMP, use any SNMP browser, library or tools--"
		echo "-- (4) For administration via CLI, use the telnet command                 --"
		echo "-- (5) For details, refer to documentation                                --"
		echo "--                                                                        --"
		echo "-- NEXT STEPS:                                                            --"
		echo "-- (1) Before using the software, you MUST do one of the following:       --"
		echo "--     a) Reboot the system.      -- OR --                                --"
		echo "--     b) Do the following:                                               --"
		echo "--        (i) Ensure that the environment variables are set by performing --"
		echo "--            the following action:                                       --"
		echo "--            - Logout and login                                          --"
		echo "--        (ii) Start the Dialogic(R) system services manually using the   --"
		echo "--             dlstart script found in /usr/dialogic/bin.                 --"
		echo "-- (2) Configure an appropriate license (higher density) via SNMP or CLI  --"
		echo "-- (3) If you are using PSTN, refer documentation on how to configure     --"
		echo "--     PSTN via SNMP or CLI                                               --"
		echo "--                                                                        --"
		echo "----------------------------------------------------------------------------"
	fi
fi
}





##########################################
# START OF SCRIPT
# MAIN
##########################################
setplatformType
process_cmd_line $*

if [ -z "${UPDATE}" ]
then
        FULLNAME="${FAMILY} ${RELNAME}"
else
        FULLNAME="${FAMILY} ${RELNAME} ${UPDATE}"
fi

checkSharedMemory
islibstdcplplInstalled
isHpetEnabled
installRuntime
if [ ${RUNTIME_SUCCESSFUL} -eq 0 ]
then
    installMediaServerPrompts
    installSdk
    installDemo
    ExecuteAdditionalInstallation
    ExecuteTvl2StartUpScript
    if [ ${HpetEnabled} -eq 0 ]
    then
	    ExecuteRtcDriverTest
    fi
    displayNotice
fi
exit 0

#END oF MAIN SCRIPT
