#!/bin/bash
#:set ts=3
#
######################################################################
#
# Script to uninstall Dialogic(r) software on Linux
#
# Copyright (C) 2002-2012 Dialogic Corporation.  All Rights Reserved.
#
# All names, products, and services mentioned herein are the
# trademarks or registered trademarks of their respective
# organizations and are the sole property of their respective
# owners.
#
######################################################################

### Misc environment variables used in this script
COMPANY_NAME="dialogic"
SILENT=0
REMOTE=0
DO_HMP50=0
HMP41_BUILDINFO_FILE="buildinfo"
RSRV_BUILDINFO_FILE="rsrv_buildinfo"

#########################################################################
#########################################################################
### GENERAL PURPOSE FUNCTIONS
#########################################################################
#########################################################################
#########################################################################
#Determine the platform by 
#looking at the Major Number in the top buildinfo file.
#########################################################################
setplatformType()
{
    # check if HMP5.0 is installed
    if [ -f ${INTEL_DIALOGIC_CFG}/${RSRV_BUILDINFO_FILE} ]
    then
        BUILDINFO=${INTEL_DIALOGIC_CFG}/${RSRV_BUILDINFO_FILE}
        DO_HMP50=1
    elif [ -f ${INTEL_DIALOGIC_CFG}/buildinfo ]
	then
		BUILDINFO=${INTEL_DIALOGIC_CFG}/buildinfo
	else
		echo
		echo "ERROR: buildinfo file not found!"
		echo
		myexit
	fi

	tplatform=`awk  '/Major Number/ {print $6}' ${BUILDINFO}`
	trel=`awk  '/Release Minor Number/ {print $6}' ${BUILDINFO}`
	tsubrel=`awk  '/Release SubMinor Number/ {print $6}' ${BUILDINFO}`
	
	SUBRELNUM=$tsubrel
	RELNUM=$trel
   	FULLNAME="HMP Software Release ${RELNUM}LIN"
	RELNAME=`echo $tplatform | tr '[a-z]' '[A-Z]'`
	
    case "${tplatform}" in
        hmp)
            FULLNAME="Dialogic(R) HMP Software Release ${RELNUM}LIN"
			
            case "$RELNUM" in
                1.5)
                    COMPANY_NAME="intel"
                    ;;
                3.1 | 4.1)
                    if [ "$tsubrel" != "-1" ]
                    then
                        FULLNAME="Dialogic(R) ${RELNUM}.${SUBRELNUM}LIN"
                    fi
                    ;;		
                5.0) 
                    FULLNAME="Dialogic(R) PowerMedia HMP Release ${RELNUM}LIN"				
                    ;;
            esac
			;;
        ovl | tvl2)
            FULLNAME="Dialogic(r) Multimedia Software Release ${RELNUM} for AdvancedTCA"
            ;;
        zvl)
            FULLNAME="Dialogic(r) Multimedia Kit Software Release ${RELNUM} for PCIe"
            ;;
    esac
}




######################################################################
# Check if user is root
######################################################################
check_root()
{
	if [ `id -u` -ne 0 ]
	then
		echo This program requires you to be a Superuser.
		echo Please login as root and and re-run $0.
		myexit 1
	fi
}


######################################################################
# Print the error from the uninstall return code
#     $1 - return code
######################################################################
disp_uninstall_return_code()
{
	case "$1" in
	0)
		myecho "successfully removed $PKGNM"
		;;
	1)
		echo
		echo "Fatal error removing $PKGNM. Uninstall terminated."
		myexit 1
		;;
	*)
		echo Unknown error: $1
		myexit 1
		;; 
	esac
}


#########################################################################
#       Function replaces /bin/echo (or shell builtin) but only         #
#       prints the message if ${SILENT}=0.                              #
#########################################################################
myecho()
{
	[ ${SILENT} -eq 0 ] && echo $*
}

#########################################################################
#       Determine if necessary system utilities are in the path         #
#       ADD ENTRIES AS NECESSARY WHEN THIS SCRIPT IS MODIFIED           #
#########################################################################
check_prerequisite_cmds()
{
	for CMD in cut exit id printf rm tr
	do
		# type is a bash internal command which can be used
		# to verify if the specified program exists in the path.
		type -a ${CMD} > /dev/null 2>&1
		if [ $? -ne 0 ]
		then
			echo "Error: $0: The command ${CMD} is not in the path."
			myexit 1
		fi
	done
}


#########################################################################
#    Process command line. Check for silent uninstall.
#########################################################################
process_cmd_line()
{
	if [ $# -eq 0 ]
	then
		SILENT=0
	else
		if [ "$1" = "--silent" ] && [ $# -eq 1 ] 
		then
			SILENT=1
		elif [ "$1" = "--remote" ] && [ $# -eq 1 ]
		then
			SILENT=1
			REMOTE=1
		else
			echo "Usage: $0 [ --silent ]"
			myexit 
		fi
	fi
}

#########################################################################
#    Remove configuration entries from syslogd.conf
#########################################################################
remove_syslogd_config()
{
	SYSLOG_CONF=/etc/syslog.conf

	# be paranoid and make a backup copy
	cp ${SYSLOG_CONF} ${SYSLOG_CONF}.pre-dialogic-uninstall
	grep -i -v "dialogic" ${SYSLOG_CONF} > ${SYSLOG_CONF}.post-dialogic

	# preserve permissions
	cat ${SYSLOG_CONF}.post-dialogic > ${SYSLOG_CONF}

	# tell syslogd to re-read the updated config file
	if [ -f /var/run/syslogd.pid ]
	then
		kill -SIGHUP `cat /var/run/syslogd.pid` > /dev/null 2>&1
	fi

	# clean up old and temporary files
	rm -f ${SYSLOG_CONF}.pre-dialogic ${SYSLOG_CONF}.post-dialogic
}

#########################################################################
#    Stop all services and unload the drivers
#########################################################################
stop_all_and_unload()
{
   # must set env variables before using this!
                                                                                
   # stop all snmp dlgcsnmpd is found
   if [ -x /etc/init.d/dlgcsnmpd ]
   then
      myecho
      myecho -ne "Stopping Dialogic(r) SNMP software ... "
      /etc/init.d/dlgcsnmpd stop > /dev/null 2>&1
      RET=$?
      [ ${RET} -eq 0 ] && myecho "Successful"
#      [ ${RET} -ne 0 ] && myecho "Failed"
   fi
                                                                                
   # stop all services if dlstop is found
   if [ -x ${INTEL_DIALOGIC_BIN}/dlstop ]
   then
      myecho
      myecho -ne "Stopping Dialogic(r) software ... "
      ${INTEL_DIALOGIC_BIN}/dlstop > /dev/null 2>&1
      RET=$?
      [ ${RET} -eq 0 ] && myecho "Successful"
#      [ ${RET} -ne 0 ] && myecho "Failed"
   else
      myecho
      myecho "Can't find ${INTEL_DIALOGIC_BIN}/dlstop.  Assuming Dialogic(r) software"
      myecho "has already been stopped."
   fi
}

######################################################################
# Save the current tty settings
#
# Sets the global variable TTYSAVE
######################################################################
save_tty_settings()
{
   [ ${SILENT} -eq 0 ] && TTYSAVE=`stty -g 2>/dev/null`
}

######################################################################
# Save the current tty settings
#
# Sets the global variable TTYSAVE
######################################################################
restore_tty_settings()
{
   # restore the previously saved tty settings if they exist
   [ ${SILENT} -eq 0 -a -n "${TTYSAVE}" ] && stty ${TTYSAVE} &> /dev/null
}

######################################################################
# Set the tty settings
#
######################################################################
set_tty_settings()
{
if [ ${REMOTE} -eq 0 ]
then
   # Disable Ctrl+C, Ctrl+Z, Ctrl+D, & Ctrl+\
   stty intr \000 susp \000 eof \000 quit \000
fi
}

######################################################################
# Exit the program, resetting the prior terminal settings.
#
# $1 = Exit value to return to the shell
######################################################################
myexit()
{
   restore_tty_settings
   exit $1
}

######################################################################
# Get OS release information
######################################################################
get_release_info()
{
        OS=`uname -s`
 
        if [ "${OS}" = "Linux" ] ; then
                if [ -f /etc/redhat-release ] ; then
                        DIST='RedHat'
                        PSUEDONAME=`cat /etc/redhat-release`
                elif [ -f /etc/SuSE-release ] ; then
                        DIST="SuSE"
                        PSUEDONAME=`cat /etc/SuSE-release`
                elif [ -f /etc/debian_version ] ; then
                        DIST="Debian"
                        PSUEDONAME=`cat /etc/debian_version`
                elif [ -f /etc/turbolinux-release  ] ; then
                        DIST="Turbolinux"
                        PSUEDONAME=`cat /etc/turbolinux-release`
        else
            DIST="UNKNOWN"
            PSUEDONAME="UNKNOWN"
                fi
        else
                #Unsupported OS
                DIST="UNKNOWN"
                PSUEDONAME="UNKNOWN"
        fi
}
 
######################################################################
# Check if the OS release is valid
######################################################################
check_release()
{
   # Supported OS Distributions
   case "${DIST}" in
      Debian)
         # Supported OS Distribution
         ;;
      RedHat)
         # Supported OS Distribution
         ;;
      SuSE)
         # Supported OSD
         ;;
      Turbolinux)
         # Supported OS Distribution
         ;;
      UNKNOWN)
         # Unable to determine OS Distribution
         echo
         echo "Unable to determine the specific distribution of Linux you are running."
         echo
         myexit 1
         ;;
      *)
         # Unsupported OSD (i.e., Mandrake)
         echo
         echo "OS Distribution is not supported."
         echo
         myexit 1
         ;;
   esac
 
   # Unsupported OS Descriptions
   case "${PSUEDONAME}" in
      "Red Hat Enterprise Linux AS release 3 (Taroon)")
         # Unsupported OS version
         echo
         echo "${OS_DESC} is not supported."
         echo
         myexit 1
         ;;
      UNKNOWN)
         # Unable to determine OS Version
         echo
         echo "Unable to determine the OS version"
         echo
         myexit 1
         ;;
      *)
         # Supported OS Version (i.e., Red Hat AS 3.0 Update 1)
         ;;
   esac
}

######################################################################
# set the package commands
######################################################################
set_package_cmds()
{
	# Supported OS Distributions
	case "${DIST}" in
		Debian)
			TEST_PKG="dpkg -l"
			REMOVE_PKG="dpkg --purge"
			QUERY_PKG="dpkg -l"
			PKG_EXT=".deb"
			PKGBASE=${PKG%%_*}
			;;
		RedHat | SuSE | Turbolinux)
			TEST_PKG="rpm -q"
			PKG_EXT=".rpm"
			REMOVE_PKG="rpm -e --nodeps"
			;;
		*)
			# Unsupported OSD (i.e., Mandrake)
			echo
			echo "OS Distribution is not supported."
			echo
			myexit 1
			;;
	esac
}

remove_remote_server()
{
    #
    # Remove remote server package
    #
    RSRV_LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp50-"
    RSRV_INSTALLEDLIST="${RSRV_LSB_PKG_NAME}rsrv"

    for PKG in ${RSRV_INSTALLEDLIST}
    do
        #  echo "check pkg: $PKG, installedlist=${INSTALLEDLIST}"
        $TEST_PKG $PKG &> /dev/null
        RC=$?
 
        if [ $RC -eq 0 ]
        then
            [ ${SILENT} -eq 0 ] && printf "   %-21s   %s" $PKG...
            case ${PKG} in
                *intel* | *dialogic*)
                    $REMOVE_PKG ${PKG} > /dev/null 2>&1
                    RETVAL=$?
                    [ ${SILENT} -eq 0 ] && disp_uninstall_return_code $RETVAL
				    ;;
                *)
                    echo ""
                    echo "WARNING: Unknown RPM found: ${PKG}."
                    echo "WARNING: Skipping removal of this RPM."
                    echo ""
                    ;;
            esac
     #   else
     #       echo "Test failed, $TEST_PKG $PKG returns $RC"
        fi
    done
}

######################################################################
# post uninstall custom actions
######################################################################
post_uninstall_custom_actions()
{
        # Supported OS Distributions
        case "${DIST}" in
		Debian)
			# remove environment variables from /etc/profile
			grep '^# Dialogic(r) HMP Environment Variables$' /etc/profile &> /dev/null
			if [ $? -eq 0 ]
			then
			    cp /etc/profile /etc/profile.tmp

			    cat /etc/profile.tmp | sed '/# Dialogic(r) HMP Environment Variables/d' &> /etc/profile
			fi
 
			grep '^. /etc/profile.d/ct_intel.sh$' /etc/profile &> /dev/null
			if [ $? -eq 0 ]
			then
			    cp /etc/profile /etc/profile.tmp
			    cat /etc/profile.tmp | sed '/^\. \/etc\/profile.d\/ct_intel.sh$/d' &> /etc/profile
			fi

			grep '^. /etc/profile.d/orbacus.sh$' /etc/profile &> /dev/null
			if [ $? -eq 0 ]
			then
			    cp /etc/profile /etc/profile.tmp
			    cat /etc/profile.tmp | sed '/^\. \/etc\/profile.d\/orbacus.sh$/d' &> /etc/profile
			fi
			rm -f /etc/profile.tmp
	 
			# temp for orbacus and ct_intel
			if [ -d /var/lock/subsys ]
			then
				rm -rf /var/lock/subsys
			fi
                        ;;
                RedHat)
                	if  [ "${DIALOGIC_PLATFORM}" = "HMP" ] && [ "${RELNUM}" = "4.1"  ] ||
						[ "${DIALOGIC_PLATFORM}" = "TVL2" ] || [ "${DIALOGIC_PLATFORM}" = "OVL" ] || [ "${DIALOGIC_PLATFORM}" = "ZVL" ]
                	then
                		rm -f /etc/rc1.d/*tvl2_startup* > /dev/null
                		rm -f /etc/rc2.d/*tvl2_startup* > /dev/null
                		rm -f /etc/rc3.d/*tvl2_startup* > /dev/null
                		rm -f /etc/rc5.d/*tvl2_startup* > /dev/null
                	fi
                        ;;
                SuSE)
                	if [ "${DIALOGIC_PLATFORM}" = "HMP" ] && [ "${RELNUM}" = "4.1"  ] ||
						[ "${DIALOGIC_PLATFORM}" = "TVL2" ] || [ "${DIALOGIC_PLATFORM}" = "OVL" ] || [ "${DIALOGIC_PLATFORM}" = "ZVL" ]
                	then
	                	rm -f /etc/init.d/rc1.d/*tvl2_startup* > /dev/null
	                	rm -f /etc/init.d/rc2.d/*tvl2_startup* > /dev/null
	                	rm -f /etc/init.d/rc3.d/*tvl2_startup* > /dev/null
	                	rm -f /etc/init.d/rc5.d/*tvl2_startup* > /dev/null
	                fi
                	;;
                Turbolinux)
                    ;;
                *)
                        # Unsupported OSD (i.e., Mandrake)
                        echo
                        echo "OS Distribution is not supported."
                        echo
                        myexit 1
                        ;;
        esac
}

######################################################################
# Create a list of all the installed packages.  None of the RPM's have
# internal dependencies for now, so a straight query and remove can be used.
######################################################################
query_for_packages()
{
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-com &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-dmdev &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-docs &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-hmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-lic &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-ooc &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-sdk &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-snmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-t1e1 &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp-demos &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp-"

	${TEST_PKG} lsb-${COMPANY_NAME}-tvl-com &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-tvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-tvl-dmdev &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-tvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-tvl-docs &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-tvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-tvl-hmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-tvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-tvl-lic &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-tvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-tvl-ooc &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-tvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-tvl-sdk &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-tvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-tvl-snmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-tvl-"

	${TEST_PKG} lsb-${COMPANY_NAME}-ovl-com &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-ovl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-ovl-dmdev &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-ovl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-ovl-docs &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-ovl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-ovl-hmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-ovl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-ovl-lic &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-ovl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-ovl-ooc &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-ovl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-ovl-sdk &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-ovl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-ovl-snmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-ovl-"

	${TEST_PKG} lsb-${COMPANY_NAME}-zvl-com &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-zvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-zvl-dmdev &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-zvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-zvl-docs &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-zvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-zvl-hmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-zvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-zvl-lic &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-zvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-zvl-ooc &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-zvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-zvl-sdk &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-zvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-zvl-snmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-zvl-"
	${TEST_PKG} lsb-${COMPANY_NAME}-zvl-t1e1 &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-zvl-"

	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-com &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	if [ "${ARCH}" == "x86_64" ] && [ "${GCCVER}" == "4.1" ] && [ "${RELNUM}" = "4.1" ] || [ "${RELNUM}" = "5.0" ]
	then
		${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-x64com &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	fi
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-dmdev &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-docs &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-hmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	if [ "${ARCH}" == "x86_64" ] && [ "${GCCVER}" == "4.1" ] && [ "${RELNUM}" = "4.1" ] || [ "${RELNUM}" = "5.0" ]
	then
		${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-x64hmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	fi
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-lic &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-ooc &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-sdk &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-snmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-t1e1 &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-demos &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
	${TEST_PKG} lsb-${COMPANY_NAME}-hmp41-msp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-hmp41-"
		
	${TEST_PKG} lsb-${COMPANY_NAME}-311-com &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-311-"
	${TEST_PKG} lsb-${COMPANY_NAME}-311-dmdev &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-311-"
	${TEST_PKG} lsb-${COMPANY_NAME}-311-docs &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-311-"
	${TEST_PKG} lsb-${COMPANY_NAME}-311-hmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-311-"
	${TEST_PKG} lsb-${COMPANY_NAME}-311-lic &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-311-"
	${TEST_PKG} lsb-${COMPANY_NAME}-311-sdk &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-311-"
	${TEST_PKG} lsb-${COMPANY_NAME}-311-snmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-311-"
	${TEST_PKG} lsb-${COMPANY_NAME}-311-t1e1 &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-311-"
	
	${TEST_PKG} lsb-${COMPANY_NAME}-411-com &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-411-"
	${TEST_PKG} lsb-${COMPANY_NAME}-411-dmdev &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-411-"
	${TEST_PKG} lsb-${COMPANY_NAME}-411-docs &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-411-"
	${TEST_PKG} lsb-${COMPANY_NAME}-411-hmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-411-"
	${TEST_PKG} lsb-${COMPANY_NAME}-411-lic &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-411-"
	${TEST_PKG} lsb-${COMPANY_NAME}-411-sdk &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-411-"
	${TEST_PKG} lsb-${COMPANY_NAME}-411-snmp &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-411-"
	${TEST_PKG} lsb-${COMPANY_NAME}-411-t1e1 &> /dev/null && LSB_PKG_NAME="lsb-${COMPANY_NAME}-411-"
	
        # Supported OS Distributions
        case "${DIST}" in
            Debian)
			    # the list command does not work properly so a static list must be maintained.
			    INSTALLEDLIST="${LSB_PKG_NAME}lic ${LSB_PKG_NAME}snmp ${LSB_PKG_NAME}t1e1 ${LSB_PKG_NAME}com ${LSB_PKG_NAME}dmdev ${LSB_PKG_NAME}docs ${LSB_PKG_NAME}hmp ${LSB_PKG_NAME}ooc ${LSB_PKG_NAME}sdk ${LSB_PKG_NAME}demos"
                ;;
            RedHat | SuSE | Turbolinux)
                if [ "${RELNAME}" = "HMP" ] && [ "${RELNUM}" = "1.5"  ]
                then
                    INSTALLEDLIST="lsb-intel-hmp-com lsb-intel-hmp-dmdev lsb-intel-hmp-docs lsb-intel-hmp-hmp lsb-intel-hmp-ooc lsb-intel-hmp-sdk lsb-intel-hmp-snmp"
                elif [ "${RELNAME}" = "HMP" ] && [ "${RELNUM}" = "3.1"  ]
                then
                    if [ "${SUBRELNUM}" = "-1" ]
                    then
                        INSTALLEDLIST="lsb-dialogic-hmp-com lsb-dialogic-hmp-dmdev lsb-dialogic-hmp-docs lsb-dialogic-hmp-hmp lsb-dialogic-hmp-lic lsb-dialogic-hmp-sdk lsb-dialogic-hmp-t1e1 lsb-dialogic-hmp-demos"
                    else
                        INSTALLEDLIST="lsb-dialogic-311-com lsb-dialogic-311-dmdev lsb-dialogic-311-hmp lsb-dialogic-311-lic lsb-dialogic-311-sdk lsb-dialogic-311-t1e1"
                    fi	
                elif [ "${RELNAME}" = "OVL" ]
                then
                    INSTALLEDLIST="lsb-dialogic-ovl-com lsb-dialogic-ovl-dmdev lsb-dialogic-ovl-docs lsb-dialogic-ovl-hmp lsb-dialogic-ovl-ooc lsb-dialogic-ovl-sdk"
                elif [ "${RELNAME}" = "TVL2" ]
                then
                    INSTALLEDLIST="lsb-dialogic-tvl-com lsb-dialogic-tvl-dmdev lsb-dialogic-tvl-docs lsb-dialogic-tvl-hmp lsb-dialogic-tvl-ooc lsb-dialogic-tvl-sdk"
                elif [ "${RELNAME}" = "ZVL" ]
                then
                    INSTALLEDLIST="lsb-dialogic-zvl-com lsb-dialogic-zvl-dmdev lsb-dialogic-zvl-docs lsb-dialogic-zvl-hmp lsb-dialogic-zvl-lic lsb-dialogic-zvl-ooc lsb-dialogic-zvl-sdk lsb-dialogic-zvl-t1e1"
                elif [ "${RELNAME}" = "HMP" ] && [ "${RELNUM}" = "4.1"  ] || [ "${RELNUM}" = "5.0"  ]
                then
                    if [ "${SUBRELNUM}" = "-1" ]
                    then
						if [ "${ARCH}" == "x86_64" ] && [ "${GCCVER}" == "4.1" ] && [ "${RELNUM}" = "4.1" ] || [ "${RELNUM}" = "5.0" ]
						then
							INSTALLEDLIST="lsb-dialogic-hmp41-com lsb-dialogic-hmp41-x64com lsb-dialogic-hmp41-dmdev lsb-dialogic-hmp41-docs lsb-dialogic-hmp41-hmp lsb-dialogic-hmp41-x64hmp lsb-dialogic-hmp41-lic lsb-dialogic-hmp41-sdk lsb-dialogic-hmp41-t1e1 lsb-dialogic-hmp41-demos lsb-dialogic-hmp41-msp"
						else
							INSTALLEDLIST="lsb-dialogic-hmp41-com lsb-dialogic-hmp41-dmdev lsb-dialogic-hmp41-docs lsb-dialogic-hmp41-hmp lsb-dialogic-hmp41-lic lsb-dialogic-hmp41-sdk lsb-dialogic-hmp41-t1e1 lsb-dialogic-hmp41-demos lsb-dialogic-hmp41-msp"
						fi
                    else
                        INSTALLEDLIST="lsb-dialogic-411-com lsb-dialogic-411-dmdev lsb-dialogic-411-hmp lsb-dialogic-411-lic lsb-dialogic-411-sdk lsb-dialogic-411-t1e1"
                    fi
			    else
				    echo
				    echo "ERROR: Unable to determine installed list"
				    echo
				    myexit
			    fi
				;;
            *)
                # Unsupported OSD (i.e., Mandrake)
                echo
                echo "OS Distribution is not supported."
                echo
                myexit 1
                ;;
        esac
}

get_gcc()
{
	GCCVER=`gcc -v 2>&1| perl -e 'my $line;while(<>){$line=$_;if ($line=~s/^gcc\p{isSpace}*[^\p{isSpace}]*\p{isSpace}*(\p{isWord}*\p{isPunct}\p{isWord}*).*/$1/){chomp $line;print $line, "\n"; } }'`

	# 4.x -> 3.4
	[ "${GCCVER}" = "4.4" ] && GCCVER=4.1
	[ "${GCCVER}" = "4.3" ] && GCCVER=4.1
	[ "${GCCVER}" = "4.2" ] && GCCVER=4.1
	[ "${GCCVER}" = "4.1" ] && GCCVER=4.1
	[ "${GCCVER}" = "4.0" ] && GCCVER=4.1

	# GCC 3.2 binaries can be used under GCC 3.3
	[ "${GCCVER}" = "3.3" ] && GCCVER=3.2
}

######################################################################
######################################################################
### BEGINNING OF SCRIPT
######################################################################
######################################################################


# Process command line arguments
process_cmd_line $*

# Determine platform
setplatformType


# Save the current tty settings
TTYSAVE=
save_tty_settings

# Disable Ctrl+C, Ctrl+Z, Ctrl+D, & Ctrl+\
set_tty_settings

# Check for file containing our environment variables.
# If it is not found, abort the execution.
OURVARS=/etc/profile.d/ct_intel.sh
if [ ! -x ${OURVARS} ]
then
        echo "No Dialogic(r) software was found."
        myexit
else
        # get our environment variables
        . ${OURVARS}
fi

# make sure all commands and files needed by this script are available
check_prerequisite_cmds

# Check if user is root
check_root

# Check for supported Linux distributions
get_release_info
check_release

# Get gcc version
get_gcc

# Get architecture
ARCH=`uname -m`

# set OS Distribution package specific commands (rpm, dpkg, etc...)
set_package_cmds

# Display initial banner for interactive uninstall only
if [ ${SILENT} -eq 0 ]
then
	echo
	echo "================================================================================"
	if [ "${RELNAME}" = "HMP"  ]
	then
		echo "                   $FULLNAME"
	elif [ "${RELNAME}" = "TVL2" ] || [ "${RELNAME}" = "OVL" ]
	then
		echo "         $FULLNAME"
	elif [ "${RELNAME}" = "ZVL"  ]
	then
		echo "         $FULLNAME"
	fi
	echo "                                  UNINSTALL"
	echo
	echo "Before you uninstall the software, you may wish to save certain configuration"
	echo "files that you have modified based upon the configuration of your system."
	echo
	echo "If you wish to save these configuration files, answer "Y" at the prompt below to"
	echo "quit the uninstall.  You may then manually back up any files located under"
	echo "${INTEL_DIALOGIC_DIR}."
	echo
	echo "Once this has been done, re-run this script to uninstall the software."
	echo
	echo -ne "Quit the uninstall and back up your configuration files now [y/n] ? \c"

	# read input from the user
	read line

	# abort the install if requested
	resp=`echo ${line} | tr '[A-Z]' '[a-z]' | cut -c1`
	if [ -z "${resp}" ] || [ "${resp}" = "y" ]
	then
		echo
		echo "Please back up your configuration files and re-run $0."
		echo "Uninstall cancelled at the user's request."
		myexit
	fi

	myecho
	myecho "The Dialogic(R) Software will now be uninstalled."
	myecho

fi

# stop software and unload drivers before proceeding to uninstall.
stop_all_and_unload

if [ ${DO_HMP50} -eq 1 ]
then
	remove_remote_server
fi

#
# Handle files delivered outside of RPM's and temporary files
#

[ -n "${INTEL_DIALOGIC_BIN}" ] && rm -f ${INTEL_DIALOGIC_BIN}/config.sh
[ -n "${INTEL_DIALOGIC_CFG}" ] && rm -f ${INTEL_DIALOGIC_CFG}/*buildinfo*
rm -f /cheetah.log
rm -f /tmp/devmap.ref
rm -f /tmp/CheetahIpcMutex
rm -f /tmp/CheetahSharedMemory.dat


query_for_packages

#
# Remove each package
#
for PKGNM in $INSTALLEDLIST
do
        $TEST_PKG $PKGNM &> /dev/null
        RC=$?

        if [ $RC -eq 0 ]
        then
		[ ${SILENT} -eq 0 ] && printf "   %-21s   %s" $PKGNM...

:
		case ${PKGNM} in
			*intel* | *dialogic*)
				$REMOVE_PKG ${PKGNM} > /dev/null 2>&1
				RETVAL=$?
				[ ${SILENT} -eq 0 ] && disp_uninstall_return_code $RETVAL
				;;
			*)
				echo ""
				echo "WARNING: Unknown RPM found: ${PKGNM}."
				echo "WARNING: Skipping removal of this RPM."
				echo ""
		esac
	fi

done	# for PKGNM in $INSTALLEDLIST

post_uninstall_custom_actions

# clean up syslogd config file entries
# no longer need it. No dialog log entry is added to /etc/syslog.conf any more
#remove_syslogd_config

myecho
myecho "To complete the removal of the Dialogic(r) software,"
myecho "the following action is required:"
myecho
myecho "	- Logout and login"
myecho
myecho "NOTE: A reboot is not required to complete the uninstall process."
myecho

# Now that all files installed from the distribution RPM's have been removed, 
# Remove this script and the directories containing it once this occurs, there
# is no way to run it again.

# If any symlinks from ${INTEL_DIALOGIC_INC} or ${INTEL_DIALOGIC_LIB}
# to /usr/{include,lib} still exist, then one of our RPM's is not well behaved.
[ -n "${INTEL_DIALOGIC_BIN}" ] && rm -f ${INTEL_DIALOGIC_BIN}/dlguninstall
[ -n "${INTEL_DIALOGIC_DIR}" ] && rm -rf ${INTEL_DIALOGIC_DIR}


myexit

