#!/bin/bash
#:set ts=3
#
######################################################################
#
# Script to install Dialogic(r) Software
#
# Copyright (C) 2001-2011 Dialogic Corporation.  All Rights Reserved.
#
# All names, products, and services mentioned herein are the
# trademarks or registered trademarks of their respective
# organizations and are the sole property of their respective
# owners.
#
######################################################################
### GLOBAL ENVIRONMENT VARIABLES ###
DLGCROOT=/usr/dialogic
######################################################################
### SCRIPT VARIABLES AND CONSTANTS
TARGETLIST=
NODEPS=
NUMPKGS=0
PNAMELST=
PKGFLAG=0
QUITFLAG=0
UPFLAG=0
SILENT=0
REMOTE=0
TRANSNUM=0
ALL=0
ERROR=0
PKG_SUFFIX=
GCC_VER=
BUILD_DIR=`pwd`
OBSOLETE_PKGS=""
COUNTER=0
BUILDINFO="buildinfo"
I_BUILDINFO="/usr/dialogic/cfg/${BUILDINFO}"
UPGRADE_INSTALL=0
SUB_PROCESS=0
INSTALL_32BIT_ONLY=1

# list of RPM's which need the rpm --replacefiles option
NEEDS_REPLACEFILES=""

#Using the buildinfo file determine what type of platform
#i.e hmp or tvl2
tplatform=`awk  '/Major Number/ {print $6}' ${BUILDINFO}`
trel=`awk  '/Release Minor Number/ {print $6}' ${BUILDINFO}`
tsubrel=`awk  '/Release SubMinor Number/ {print $6}' ${BUILDINFO}`

B_MAJOR=$tplatform
B_MINOR=$trel
B_SUBMINOR=$tsubrel
B_BUILDNUM=`awk  '/Release Build Number/ {print $6}' ${BUILDINFO}`

if [ -f ${I_BUILDINFO} ]
then
	UPGRADE_INSTALL=1

	I_MAJOR=`awk  '/Release Major Number/ {print $6}' ${I_BUILDINFO}`
	I_MINOR=`awk  '/Release Minor Number/ {print $6}' ${I_BUILDINFO}`
	I_SUBMINOR=`awk  '/Release SubMinor Number/ {print $6}' ${I_BUILDINFO}`
	I_BUILDNUM=`awk  '/Release Build Number/ {print $6}' ${I_BUILDINFO}`
fi


# Release-specific definitions
BUS=""
COMPANY_NAME="dialogic"
FAMILY="Dialogic(R)"
MAINTREL=""
SHORTMAINTREL=""
RELNUM=$trel
SUBRELNUM=$tsubrel
RELNAME="HMP Software Release ${RELNUM}LIN"
FULLNAME=$RELNAME
SHORTRELNAME="`echo ${tplatform} | tr '[a-z]' '[A-Z]'`"
UPDATE=""
PLATFORM="`echo ${tplatform} | tr '[a-z]' '[A-Z]'`"

case "${tplatform}" in
	hmp)
		RELNAME="HMP Software Release ${RELNUM}LIN"
		FULLNAME=$RELNAME	
		case "$RELNUM" in
			1.5)
				COMPANY_NAME="intel"
				;;
			3.1 | 4.1)
				OBSOLETE_PKGS="lsb-dialogic-hmp-ooc"
				if [ "$tsubrel" != "-1" ]
				then
					RELNAME="${RELNUM}.${SUBRELNUM}LIN"
					FULLNAME=$RELNAME
				fi
				;;	
		esac
		;;
	tvl2 | ovl)
		RELNAME="Multimedia Software Release ${RELNUM} for AdvancedTCA"
		FULLNAME="${RELNAME}"
		;;
	zvl)
		RELNAME="Multimedia Kit Software Release ${RELNUM} for PCIe"
		FULLNAME="${RELNAME}"
		OBSOLETE_PKGS="lsb-dialogic-zvl-ooc"
		;;
esac


		
if [ -z "${UPDATE}" ]
then
	FULLNAME="${FAMILY} ${RELNAME}"
	SYSREL="${SHORTRELNAME} ${RELNUM}"
else
	FULLNAME="${FAMILY} ${RELNAME} ${UPDATE}"
	SYSREL="${SHORTRELNAME} ${RELNUM} ${UPDATE}"
fi

######################################################################
# Package List/Dependencies/Description Table
#
# IMPORTANT: The contents and format of this part of the file
# controls:
#
# 1) List of packages which can be installed if present in
#    the current directory
# 2) Dependencies of each package
# 3) Description of each package
#
# Format: #^#DLGCpkg:DLGCpkg2 DLGCpkg3 DLGCpkg4 ...:Description
# 64-bit Format: #>#DLGCpkg:DLGCpkg2 DLGCpkg3 DLGCpkg4 ...:Description
# where DLGCpkg depends on DLGCpkg2, DLGCpkg3 and DLGCpkg4 ...
#
# NOTE: Please do not remove the initial "#^#" text
# The Boardwatch package dependencies are handled in the script
# logic elsewhere
#
# NOTE: The first field of each entry is the ORDER how the packages
# will be installed and how the menu is displayed. 
#
# NOTE: Do not remove any entry from the table below which is also
# included in the SILENT_KEYWORDS list unless the functionality
# has been dropped from the release.
#
# Note: June 30 2006 - SNMP Package for TVL2 has being removed 
# This RPM is not needed
#
######################################################################
# START---PACKAGE LIST---

###########
# HMP 1.5 #
###########
#^#lsb-intel-hmp-com::Common Runtime Software for the Dialogic(r) Boards
#^#lsb-intel-hmp-ooc:lsb-intel-hmp-com:Orbacus Support Package 
#^#lsb-intel-hmp-dmdev:lsb-intel-hmp-com lsb-intel-hmp-ooc:Dialogic(r) Base Runtime
#^#lsb-intel-hmp-hmp:lsb-intel-hmp-com lsb-intel-hmp-ooc lsb-intel-hmp-dmdev:Dialogic(r) Host Media Processing Software
#^#lsb-intel-hmp-snmp:lsb-intel-hmp-com lsb-intel-hmp-ooc:SNMP Component Manager
#^#lsb-intel-hmp-docs::Documentation

###########
# HMP 3.1 #
###########
#^#lsb-dialogic-hmp-com::Common Runtime Software for the Dialogic(r) Boards
#^#lsb-dialogic-hmp-dmdev:lsb-dialogic-hmp-com:Dialogic(r) Base Runtime
#^#lsb-dialogic-hmp-hmp:lsb-dialogic-hmp-com lsb-dialogic-hmp-dmdev lsb-dialogic-hmp-lic lsb-dialogic-hmp-docs:Dialogic(r) Host Media Processing Software 
#^#lsb-dialogic-hmp-t1e1:lsb-dialogic-hmp-com lsb-dialogic-hmp-dmdev lsb-dialogic-hmp-hmp lsb-dialogic-hmp-lic lsb-dialogic-hmp-docs:Dialogic(r) DNI Boards & HMP Software 
#^#lsb-dialogic-hmp-lic::License Package
#^#lsb-dialogic-hmp-docs::Documentation

########
# TVL2 #
########
#^#lsb-dialogic-tvl-com::Common Runtime Software for the Dialogic(r) Boards
#^#lsb-dialogic-tvl-ooc:lsb-dialogic-tvl-com:Orbacus Support Package
#^#lsb-dialogic-tvl-dmdev:lsb-dialogic-tvl-com lsb-dialogic-tvl-ooc:Dialogic(r) Base Runtime
#^#lsb-dialogic-tvl-hmp:lsb-dialogic-tvl-com lsb-dialogic-tvl-ooc lsb-dialogic-tvl-dmdev:Dialogic(r) Multimedia Software Release 1.0 for AdvancedTCA
#^#lsb-dialogic-tvl-docs::Documentation

########
# OVL #
########
#^#lsb-dialogic-ovl-com::Common Runtime Software for the Dialogic(r) Boards
#^#lsb-dialogic-ovl-ooc:lsb-dialogic-ovl-com:Orbacus Support Package
#^#lsb-dialogic-ovl-dmdev:lsb-dialogic-ovl-com lsb-dialogic-ovl-ooc:Dialogic(r) Base Runtime
#^#lsb-dialogic-ovl-hmp:lsb-dialogic-ovl-com lsb-dialogic-ovl-ooc lsb-dialogic-ovl-dmdev:Dialogic(r) Multimedia Software Release 1.1 for AdvancedTCA
#^#lsb-dialogic-ovl-docs::Documentation

########
# ZVL #
########
#^#lsb-dialogic-zvl-com::Common Runtime Software for the Dialogic(r) Boards
#^#lsb-dialogic-zvl-dmdev:lsb-dialogic-zvl-com:Dialogic(r) Base Runtime
#^#lsb-dialogic-zvl-hmp:lsb-dialogic-zvl-com lsb-dialogic-zvl-dmdev lsb-dialogic-zvl-lic lsb-dialogic-zvl-docs:Dialogic(r) MMK Software Release 1.0 for PCIe
#^#lsb-dialogic-zvl-t1e1:lsb-dialogic-zvl-com lsb-dialogic-zvl-dmdev lsb-dialogic-zvl-hmp lsb-dialogic-zvl-lic lsb-dialogic-zvl-docs:Dialogic(r) DNI Boards & MMK Software Release 1.0 for PCIe
#^#lsb-dialogic-zvl-lic::License Package
#^#lsb-dialogic-zvl-docs::Documentation

########
# HMP 4.1 #
########
#^#lsb-dialogic-hmp41-com::Common Runtime Software for the Dialogic(r) Boards
#^#lsb-dialogic-hmp41-dmdev:lsb-dialogic-hmp41-com:Dialogic(r) Base Runtime
#^#lsb-dialogic-hmp41-hmp:lsb-dialogic-hmp41-com lsb-dialogic-hmp41-dmdev lsb-dialogic-hmp41-lic lsb-dialogic-hmp41-docs:Dialogic(r) Host Media Processing Software
#^#lsb-dialogic-hmp41-t1e1:lsb-dialogic-hmp41-com lsb-dialogic-hmp41-dmdev lsb-dialogic-hmp41-hmp lsb-dialogic-hmp41-lic lsb-dialogic-hmp41-docs:Dialogic(r) DNI Boards & HMP Software 
#^#lsb-dialogic-hmp41-lic::License Package
#^#lsb-dialogic-hmp41-docs::Documentation

########
# 64-bit HMP 4.1 #
########
#>#lsb-dialogic-hmp41-com::Common Runtime Software for the Dialogic(r) Boards
#>#lsb-dialogic-hmp41-x64com::64-bit Common Runtime Software for the Dialogic(r) Boards
#>#lsb-dialogic-hmp41-dmdev:lsb-dialogic-hmp41-com:Dialogic(r) Base Runtime
#>#lsb-dialogic-hmp41-hmp:lsb-dialogic-hmp41-com lsb-dialogic-hmp41-x64com lsb-dialogic-hmp41-dmdev lsb-dialogic-hmp41-x64hmp lsb-dialogic-hmp41-lic lsb-dialogic-hmp41-docs:Dialogic(r) Host Media Processing Software
#>#lsb-dialogic-hmp41-x64hmp::64-bit Dialogic(r) Host Media Processing Software
#>#lsb-dialogic-hmp41-t1e1:lsb-dialogic-hmp41-com lsb-dialogic-hmp41-x64com lsb-dialogic-hmp41-dmdev lsb-dialogic-hmp41-hmp lsb-dialogic-hmp41-x64hmp lsb-dialogic-hmp41-lic lsb-dialogic-hmp41-docs:Dialogic(r) DNI Boards & HMP Software
#>#lsb-dialogic-hmp41-lic::License Package
#>#lsb-dialogic-hmp41-docs::Documentation

########
# HMP 3.1.1 #
########
#^#lsb-dialogic-311-com::Common Runtime Software for the Dialogic(r) Boards
#^#lsb-dialogic-311-dmdev:lsb-dialogic-311-com:Dialogic(r) Base Runtime
#^#lsb-dialogic-311-hmp:lsb-dialogic-311-com lsb-dialogic-311-dmdev lsb-dialogic-311-lic:Dialogic(r) Host Media Processing Software
#^#lsb-dialogic-311-t1e1:lsb-dialogic-311-com lsb-dialogic-311-dmdev lsb-dialogic-311-hmp lsb-dialogic-311-lic:Dialogic(r) DNI Boards & HMP Software 
#^#lsb-dialogic-311-lic::License Package

#############
# HMP 4.1.1 #
#############
#^#lsb-dialogic-411-com::Common Runtime Software for the Dialogic(r) Boards
#^#lsb-dialogic-411-dmdev:lsb-dialogic-411-com:Dialogic(r) Base Runtime
#^#lsb-dialogic-411-hmp:lsb-dialogic-411-com lsb-dialogic-411-dmdev lsb-dialogic-411-lic:Dialogic(r) Host Media Processing Software
#^#lsb-dialogic-411-t1e1:lsb-dialogic-411-com lsb-dialogic-411-dmdev lsb-dialogic-411-hmp lsb-dialogic-411-lic:Dialogic(r) DNI Boards & HMP Software 
#^#lsb-dialogic-411-lic::License Package

# END---PACKAGE LIST---


######################################################################
######################################################################
### GENERAL PURPOSE FUNCTIONS
######################################################################
######################################################################

######################################################################
# Validate if installing this build is a supported upgrade path
######################################################################
validate_upgrade_path()
{
	if [ ${UPGRADE_INSTALL} -eq 1 ]
	then
		# major number of build is earlier than that of installed release
		if [ ${B_MAJOR} != ${I_MAJOR} ]
		then
			myecho "FATAL ERROR:"
			myecho "FATAL ERROR: Your system is already running a differnt major version"
			myecho "FATAL ERROR: of the Dialogic System Release software.  Update aborted"
			myecho "FATAL ERROR: due to a differnt major version of the software already"
			myecho "FATAL ERROR: installed on the system."
			myecho "FATAL ERROR:"
			myexit 1
		fi
			
		if [ ${B_MINOR} != ${I_MINOR} ]
		then
			myecho "FATAL ERROR:"
			myecho "FATAL ERROR: Your system is already running a differnt minor version"
			myecho "FATAL ERROR: of the Dialogic System Release software.  Update aborted"
			myecho "FATAL ERROR: due to a differnt minor version of the software already"
			myecho "FATAL ERROR: installed on the system."
			myecho "FATAL ERROR:"
			myexit 1
		fi

		if [ ${B_BUILDNUM} -lt ${I_BUILDNUM} ]
		then
			myecho "FATAL ERROR:"
			myecho "FATAL ERROR: Your system is already running a more recent version of the"
			myecho "FATAL ERROR: Dialogic System Release software.  Update aborted due to "
			myecho "FATAL ERROR: later version of the software already installed on the system."
			myecho "FATAL ERROR:"
			myexit 1
		fi
	fi
}

######################################################################
# Check that the system date is later than the build date
######################################################################
check_date()
{
	# get the system date
	SYSTEM_MONTH=`date +%m`
	SYSTEM_DAY=`date +%d`
	SYSTEM_YEAR=`date +%Y`
	SYSTEM_DATE=${SYSTEM_MONTH}/${SYSTEM_DAY}/${SYSTEM_YEAR}

	# get the build date
	SR_DATE=`awk  '/System Release Build Date/ {print $6}' buildinfo`
	SP_DATE=`awk  '/Service Pack Build Date/ {print $6}' buildinfo`
	FP_DATE=`awk  '/Feature Pack Build Date/ {print $6}' buildinfo`
	
	if [ "${SR_DATE}" != "None" ]
	then
		BUILD_DATE=${SR_DATE}
	elif [ "${SP_DATE}" != "None" ]
	then
		BUILD_DATE=${SP_DATE}
	elif [ "${FP_DATE}" != "None" ]
	then
		BUILD_DATE=${FP_DATE}
	else
		myecho
	        myecho "Unable to determine the build creation date."
                myecho "Please check buildinfo and rerun $0."
                myecho
                myexit 1	
	fi
	
	BUILD_MONTH=${BUILD_DATE%%\/*}
	BUILD_DAY_TMP=${BUILD_DATE##$BUILD_MONTH\/}
	BUILD_DAY=${BUILD_DAY_TMP%%\/*}
	BUILD_YEAR=${BUILD_DAY_TMP##$BUILD_DAY\/}

	# Verify that the system date is later than the build date
	if [ ${SYSTEM_YEAR} -lt ${BUILD_YEAR} ] ||
	   [ ${SYSTEM_YEAR} -eq ${BUILD_YEAR} -a ${SYSTEM_MONTH} -lt ${BUILD_MONTH} ] ||
	   [ ${SYSTEM_YEAR} -eq ${BUILD_YEAR} -a ${SYSTEM_MONTH} -eq ${BUILD_MONTH} -a ${SYSTEM_DAY} -lt ${BUILD_DAY} ]
	then
		myecho
	        myecho "The system date (${SYSTEM_DATE}) is older"
		myecho "than the build creation date (${BUILD_DATE})."
                myecho "Please update the system date and rerun $0."
                myecho
                myexit 1
	fi

	return 0
}

######################################################################
# Center a line of text on the screen
######################################################################
center()
{
	# get size of the screen
	SIZE=`stty size`

	# extract number of columns
	COLS=${SIZE##[0-9]*\ }

	# get length of string to be printed
	INPSTR=`echo $1 | wc -c | sed 's:\ ::g'`

	# calculate number of remaining columns
	let DIFF=${COLS}-${INPSTR}

	# take half
	let HDIFF=${DIFF}/2
	
	printf "%${HDIFF}c%s\n" " " "$1"
}


######################################################################
# Check if user is root
######################################################################
check_root()
{
	if [ `id -u` != 0 ]
	then
		echo "FATAL ERROR: This program requires you to be logged in as root."
		echo "FATAL ERROR: Please log in as root and and re-run $0."
		echo "FATAL ERROR: Installation aborted due to insufficient permissions."
		myexit 1
	fi
}

######################################################################
# Check for a supported GCC version
######################################################################
check_gcc()
{
	# determine from the RPMs in the build which compiler version(s)
	# we are supporting
	SUPP_GCC=`ls ${LSBPKGNAME}*_gcc*${PKG_EXT} | sed -e 's/${LSBPKGNAME}[0-z]*-[.-9]*-[0-9]*_gcc//g' -e 's/\.[0-z]*${PKG_EXT}//g' | sort | uniq`
	
	# check if env DLG_NO_GCC is set
	if [ -n "${DLG_NO_GCC}" ]
	then
		if [ -n "${FORCE_GCC_VERSION}" ]
		then
			echo " "
			GCCVER=${FORCE_GCC_VERSION}
		else
			echo "FATAL ERROR:"
			echo "FATAL ERROR: DLG_NO_GCC environment variable is set but FORCE_GCC_VERSION is not set"
			echo "FATAL ERROR: Installation aborted"
			echo "FATAL ERROR:"
			myexit 1
		fi
	else
		# see if the gcc compiler RPM is installed
		if ! ${QUERY_PKG} gcc &> /dev/null
		then
			echo "FATAL ERROR:"
			echo "FATAL ERROR: gcc compiler not found"
			echo "FATAL ERROR:"
			echo "FATAL ERROR: This software requires that one of the following versions"
			echo "FATAL ERROR: of the gcc compiler be installed:"
			echo "FATAL ERROR:"
			for i in ${SUPP_GCC}
			do
				echo -ne "FATAL ERROR:\t\tGCC $i\n"
			done
			echo "FATAL ERROR:"
			echo "FATAL ERROR: Please install the gcc compiler and rerun $0."
			echo "FATAL ERROR:"
			echo "FATAL ERROR: Installation aborted due to missing gcc compiler."
			echo "FATAL ERROR:"
			myexit 1
		fi
		
		# we found some version of the gcc compiler, so get it's version
		# fix problem in parsing gcc -v output: the existing one assumes the output contains 4+ columns, 
		# ex "gcc version 4.1.2 20080704 (Red Hat 4.1.2-46)
		# but depending on gcc installation, the 4th column may not show. Also, for none-English version of RH, the output could be different
		# use perl to parse the output and get the 3rd column "4.1.2" in the above example
		#GCCVER=`gcc -v 2>&1 | sed -e '/gcc version/!d' -e '/gcc version/{s/.*gcc version \([0-9,.]*\) .*/\1/;}' | cut -c1-3`
		GCCVER=`gcc -v 2>&1| perl -e 'my $line;while(<>){$line=$_;if ($line=~s/^gcc\p{isSpace}*[^\p{isSpace}]*\p{isSpace}*(\p{isWord}*\p{isPunct}\p{isWord}*).*/$1/){chomp $line;print $line, "\n"; } }'`

		# 4.x -> 3.4
		[ "${GCCVER}" = "4.4" ] && GCCVER=4.1
		[ "${GCCVER}" = "4.3" ] && GCCVER=4.1
		[ "${GCCVER}" = "4.2" ] && GCCVER=4.1
		[ "${GCCVER}" = "4.1" ] && GCCVER=4.1
		[ "${GCCVER}" = "4.0" ] && GCCVER=4.1

		# GCC 3.2 binaries can be used under GCC 3.3
		[ "${GCCVER}" = "3.3" ] && GCCVER=3.2
	fi

	# if the FORCE_GCC_VERSION environment variable is set 
		#
		# If the user wants to force a particular set of RPMs to install
		# then the "FORCE_GCC_VERSION" environment variable should be set
		# before this script is run. The value of this variable must contain
		# only the first three characters of the GCC version (e.g. 3.2 for
		# GCC 3.2.3).
		#
		# Example:
		#	FORCE_GCC_VERSION=3.2 ./install.sh
		#
		# This would force the installation of Dialogic RPM's built with
		# GCC 3.2 to be installed REGARDLESS OF WHICH COMPILER VERSION IS ON YOUR
		# SYSTEM.
		#
		# CAUTION: USE THIS CAPABILITY AT YOUR OWN RISK
		#
		if [ -n "${FORCE_GCC_VERSION}" ]
		then
			# check to see if the specified value matches one of the compiler
			# versions for which we have binary RPMs
			echo "${SUPP_GCC}" | grep -q "${FORCE_GCC_VERSION}"
			RC=$?
			if [ ${RC} -eq 0 ]
			then
				# the user specified a GCC version for which we have binary RPMs
				myecho
				myecho "WARNING:"
				myecho "WARNING: AUTOMATIC GCC VERSION DETECTION OVERRIDDEN BY THE USER."
				myecho "WARNING:"
				myecho "WARNING: FORCING INSTALLATION OF BINARY RPMS BUILT USING GCC ${FORCE_GCC_VERSION}"
				myecho "WARNING: ON A SYSTEM THAT "
				if [ -n "${DLG_NO_GCC}" ]
				then 
					myecho "WARNING: DOESN'T HAVE GCC INSTALLED"
				else
					myecho "WARNING: HAS GCC ${GCCVER}." 
				fi
				myecho "WARNING:"
				myecho "WARNING: COMPATIBILITY ISSUES MAY EXIST.  SEE RELEASE DOCUMENTATION"
				myecho "WARNING: FOR SUPPORTED GCC VERSIONS."
				myecho "WARNING:"
				myecho

				# force the compiler version
				GCCVER=${FORCE_GCC_VERSION}

				# tell RPM top ignore GCC version conflicts
				NODEPS="--nodeps"
			else
				# user specified GCC version for which we DO NOT have binary RPMs
				echo "FATAL ERROR:"
				echo "FATAL ERROR: You specified a GCC version using the FORCE_GCC_VERSION environment variable"
				echo "FATAL ERROR: for which binary RPMs are not included in the build."
				echo "FATAL ERROR:"
				echo "FATAL ERROR: Valid values for FORCE_GCC_VERSION are:"
				echo "FATAL ERROR:"
				for i in ${SUPP_GCC}
				do
					echo -ne "FATAL ERROR:\t\t$i\n"
					echo "FATAL ERROR:"
				done
				echo "FATAL ERROR: Installation aborted due to invalid value of FORCE_GCC_VERSION."
				echo "FATAL ERROR:"
				myexit 1
			fi
		fi
		
	#
	# SUPP_GCC - list of RPM (DEB on debian) files.
	#
	echo ${SUPP_GCC} | grep -q "gcc${GCCVER}" 
	RC=$?

	if [ ${RC} -eq 1 ]
	then
		echo "FATAL ERROR:"
		echo "FATAL ERROR: Unsupported gcc compiler detected"
		echo "FATAL ERROR:"
		echo "FATAL ERROR: gcc version ${GCCVER} was detected on your system."
		echo "FATAL ERROR:"
		echo "FATAL ERROR: This software requires that one of the following versions"
		echo "FATAL ERROR: of the gcc compiler be installed:"
		echo "FATAL ERROR:"
		for i in ${SUPP_GCC}
		do
			echo -ne "FATAL ERROR:\t\tGCC $i\n"
		done
		echo "FATAL ERROR:"
		echo "FATAL ERROR: Please install a supported version of the gcc compiler and rerun $0."
		echo "FATAL ERROR:"
		echo "FATAL ERROR: Installation aborted due to unsupported gcc compiler."
		echo "FATAL ERROR:"
		myexit 1
	fi
}

######################################################################
# Check if force 32bit packages installation
######################################################################
check_32bit_install()
{
	if [ "${ARCH}" == "x86_64" ] && [ "${GCCVER}" == "4.1" ]
	then
		if [ -n "${DLGCFW_FORCE_32BIT_INSTALL}" ]
		then
			if [ ${DLGCFW_FORCE_32BIT_INSTALL} -eq 0 ]
			then
				INSTALL_32BIT_ONLY=0
			else
				myecho "WARNING: Installing 32-bit components only (DLGCFW_FORCE_32BIT_INSTALL is set)" 
			fi
		else
			INSTALL_32BIT_ONLY=0
		fi
	fi
}

######################################################################
# Check if kernel has been upgraded
######################################################################
check_kernel()
{
	# Deterime the current kernel version
	VERSION=`uname -r` 
	K_MAJOR=${VERSION%%.*}
	K_MINTMP=${VERSION##$K_MAJOR.}
	K_MINOR=${K_MINTMP%%.[0-9]*}
	K_SUBTMP=${VERSION##$K_MAJOR.$K_MINOR.}
	K_SUBTMP=${K_SUBTMP%%.[A-z]*}
	K_SUBTMP=${K_SUBTMP%%-*}
	K_SUBTMP=${K_SUBTMP%%[A-z]*}
	K_SUB1=${K_SUBTMP%%.*}

	# Determine the minimum supported kernel version
	if [ "${DIST}" = "Debian" ]
	then
		SUPPORTED_KERNEL=2.6.8
	elif [ "${DIST}" = "RedHat" ]
	then
		REDHAT_VER_MAJOR=`sed 's/ *.*release \([0-9][0-9]*\).*/\1/' /etc/redhat-release`

		if [ ${REDHAT_VER_MAJOR} -eq 3 ]
		then
			SUPPORTED_KERNEL=2.6.8
		elif [ ${REDHAT_VER_MAJOR} -eq 4 ]
		then
			SUPPORTED_KERNEL=2.6.9
		elif [ ${REDHAT_VER_MAJOR} -eq 5 ]
		then
			SUPPORTED_KERNEL=2.6.18
                elif [ ${REDHAT_VER_MAJOR} -eq 6 ]
                then
                        SUPPORTED_KERNEL=2.6.32
		else
			# default to the lesser version
			SUPPORTED_KERNEL=2.6.8
		fi
	elif [ "${DIST}" = "SuSE" ]
	then
		# check if it is SuSE
		cat /etc/SuSE-release | grep "SuSE Linux" &> /dev/null
		SUSE=$?

		# check if it is SLES
		cat /etc/SuSE-release | grep "SUSE LINUX Enterprise Server" &> /dev/null
		SLES=$?

		if [ ${SUSE} -eq 0 ]
		then
			SUPPORTED_KERNEL=2.6.8
		elif [ ${SLES} -eq 0 ]
		then
			SUPPORTED_KERNEL=2.6.5
		else
			# default to the lesser version
			SUPPORTED_KERNEL=2.6.5
		fi
	elif [ "${DIST}" = "Turbolinux" ]
	then
		grep 10 /etc/turbolinux-release &> /dev/null
		if [ $? -eq 0 ]
		then
			SUPPORTED_KERNEL=2.6.9
		fi
	else
		myecho
		myecho WARNING: Unable to determine supported kernel version.
		myecho WARNING: The currently installed version is ${VERSION}.
		myecho
	fi
	
	SUP_K_MAJOR=${SUPPORTED_KERNEL%%.*}
	SUP_K_MINTMP=${SUPPORTED_KERNEL##$SUP_K_MAJOR.}
	SUP_K_MINOR=${SUP_K_MINTMP%%.[0-9]*}
	SUP_K_SUB1TMP=${SUPPORTED_KERNEL##$SUP_K_MAJOR.$SUP_K_MINOR.}
	SUP_K_SUB1=${SUP_K_SUB1TMP%%.*}

	# Check the kernel version
	if [ ${K_MAJOR} -lt ${SUP_K_MAJOR} ] ||
		[ ${K_MAJOR} -le ${SUP_K_MAJOR} -a ${K_MINOR} -lt ${SUP_K_MINOR} ] || 
		[ ${K_MAJOR} -le ${SUP_K_MAJOR} -a ${K_MINOR} -le ${SUP_K_MINOR} -a ${K_SUB1} -lt ${SUP_K_SUB1} ]
	then
		# The current kernel does not meet the minimum requirements
		myecho
		myecho "This software requires a kernel upgrade."
		myecho "The current kernel version is ${VERSION}."
		myecho "Please upgrade the kernel to at least version ${SUPPORTED_KERNEL}."
		myecho "After upgrading, rerun $0."
		myecho
		myexit 1
	elif [ ${K_MAJOR} -eq ${SUP_K_MAJOR} -a ${K_MINOR} -eq ${SUP_K_MINOR} -a ${K_SUB1} -eq ${SUP_K_SUB1} ]
	then
		# The current kernel version is officially supported, so do nothing
		myecho
	else
		# The current kernel version exceeds the minimum requirements
		myecho
		myecho "The current kernel version (${VERSION}) is later than"
		myecho "the version currently supported by this software (${SUPPORTED_KERNEL})."
		myecho
	fi
}


######################################################################
# Print the error from the installation return code
#     $1 - return code
######################################################################
disp_install_return_code()
{
	case "$1" in
		0)
			myecho "Installation successful."
			;;
		*)
			if [ ${UPGRADE_INSTALL} -eq 0 ]
			then
				NUMRPMS=`echo ${2} | wc -w`

				# check if the return code is equal to the number of RPMs
				if [ $NUMRPMS -ne $1  ]
				then
					myecho
					myecho "Fatal error during installation. Installation terminated."
					myexit 1
				else
					# validate that the RPMs got installed
					for RPM in $2
					do
						# get the 4 part RPM name
						PKG=${RPM%%-[0-9].*}

						# verify that the package was actually installed
						${QUERY_PKG} $PKG &> /dev/null
						RET=$?

						if [ $RET -ne 0  ]
						then
							RPM_FAIL=1
							myecho
							myecho "Fatal error during installation. Installation terminated."
							myexit 1
						fi
					done
				fi
			fi
			;;
	esac
}

######################################################################
# Print the error from the deletion return code
#     $1 - return code
######################################################################
disp_erase_return_code()
{
	case "$1" in
		0)
			echo "successfully removed $PKGNM"
			;;
		*)
			echo "Error removing ${PKGNM}.  Installation terminated."
			myexit
			;; 
	esac
}

######################################################################
# Set the compiler-specific package suffix if appropriate
#	$1 - package base name (e.g. DLGCcom)
######################################################################
set_pkg_suffix()
{
	if [ $# -ne 1 ]
	then
		echo "FATAL ERROR:"
		echo "FATAL ERROR: Usage: set_pkg_suffix <pkgname>"
		echo "FATAL ERROR:"
		myexit 1
	fi

	# reset the suffix in case this is a noarch RPM
	PKG_SUFFIX=

	# if we find a matching compiler-specific
	# RPM file then set the suffix accordingly
	/bin/ls ${1}${PKG_SEP}*_gcc${GCCVER}*${PKG_EXT} &>/dev/null && PKG_SUFFIX=_gcc${GCCVER}
}

######################################################################
# Check if all packages are available, if not, remove from string
######################################################################
check_pkg_avail()
{
	FOUNDLST=

	# this may take a while so give the user a progress indicator
	myecho
	myecho -ne "Initializing install, please wait "
	for PKGNM in $ORDER
	do
		myecho -ne "."

		set_pkg_suffix ${PKGNM}

		# abort if more than one version of a given RPM is in the build
		if [ `echo ${PKGNM}${PKGSEP}*${PKG_SUFFIX}*${PKG_EXT} | wc -w` -gt 1 ]
		then
			myecho
			myecho "FATAL ERROR:"
			myecho "FATAL ERROR: Multiple versions of ${PKGNM} are contained"
			myecho "FATAL ERROR: in the build directory"
			for FILE in `/bin/ls ${PKGNM}${PKG_SEP}*${PKG_SUFFIX}*${PKG_EXT}`
			do
				myecho "FATAL ERROR: ${FILE}"
			done
			myecho "FATAL ERROR:"
			myecho
			myexit 1
		fi

		if [ -f ${PKGNM}${PKGSEP}*${PKG_SUFFIX}*${PKG_EXT} ]
		then
			FOUNDLST="$FOUNDLST $PKGNM"
			# store uncompressed RPM content size in a variable
			eval ${PKGNM##$LSBPKGNAME}SIZE=`get_rpm_content_size ${PKGNM}${PKGSEP}*${PKG_SUFFIX}*${PKG_EXT}`
		else
			# set to zero if it's not found
			eval ${PKGNM##$LSBPKGNAME}SIZE=0
		fi
	done

	myecho
}


######################################################################
# Function getpkgbyindex
# Usage: getpkgbyindex <index>
# Descr: Searches for package in ORDER variable at <index> location
######################################################################
getpkgbyindex()
{
	if [ $# -lt 1 ]
	then
		echo "USAGE: getpkgbyindex <index>"
	fi

	# Test for invalid entries greater than 1 character
	# NOTE: This puts a limitation on the script such that there can be no more than 9 entries on the package selection menu.  The reason for this is because the following ascii and integer checks cannot resolve entries with more than one character properly for some reason.  

	if [ ${#1} -gt 1  ]
	then
		return 1
	fi

   # remove the spaces from NUMPKGS
   NUMPKGS=`myecho "${NUMPKGS}" | sed 's/ //g'`

	# test for invalid ascii values
	if [ "${1}" \< "1" ] || [ "${1}" \> "${NUMPKGS}" ]
	then
		return 2
	fi

	# test for invalid integer values
	if [ ${1} -lt 1  ] || [ ${1} -gt ${NUMPKGS} ]
	then
		return 3
	fi

	RES=`echo $DISPLAY_ORDER | cut -s -f$1 -d" "`
	echo $RES
}


######################################################################
# Function to prompt and ask the package numbers to be installed
# And complete the dependency checking work also
# Put the final list in TARGETLIST
######################################################################
asklist()
{
	FLAG=1
	RETURN_CODE=0

	while [ $FLAG -eq 1 ]
	do
		let COUNTER=${COUNTER}+1
                if [ ${SUB_PROCESS} -eq 0 ]
                then
		        echo -e "\nEnter the packages you want to install, separated by a space, or [A,a,Q,q]: \c"
		else 
                        echo -e "\nEnter the packages you want to install, separated by a space, or [Q,q]: \c"
                fi
		read PLIST

		# Check for null selection
		if [ -z "$PLIST" ]
		then
			echo Invalid Response.
			FLAG=1
			continue
		fi

		PLIST=" `echo $PLIST | tr '[a-z]' '[A-Z]'` "
		PNAMELST=" "
		TLIST=" "
		PLISTTST=" `echo $PLIST | tr -d '[B-P,R-Z]'` "

		if [ "$PLIST" != "$PLISTTST" ]
		then
			echo Invalid Response.
			FLAG=1
			continue
		fi

		if echo "$PLIST" | grep " Q " &> /dev/null
		then
			if [ ${COUNTER} -eq 1 ]
			then
				RETURN_CODE=2
			fi
			
			QUITFLAG="1"
			break
		fi

		# Check if the user has selected ALL packages
		if echo "$PLIST" | grep " A " &> /dev/null
		then
			PNAMELST=" $ORDER "
			TLIST=" $ORDER "
			FLAG=0
			ALL=1
			break
		fi

		# Check if the first  Package Number is valid.
		PNAME=`getpkgbyindex $PLIST`
		PLISTTST=" `echo $PLIST | tr -d '[B-P,R-Z]'` "
		if [ "$PLIST" != "$PLISTTST" ]
		then
			echo " "
			echo -e "\nYou have selected an invalid package." 
			myexit 1
		fi

		RQSTLIST=

		# Check for dependencies
		for PL in $PLIST
		do
			PNAME=`getpkgbyindex $PL`
			if [ $? -eq 2 -o $? -eq 1 ]
			then
				echo Invalid Package Number \"$PL\"
				FLAG=1
				break
			fi

			# Build the list of requested pkgs (not dependencies)
			RQSTLIST="$RQSTLIST $PNAME"

			# Append dependency to SelectedPackages List
			PNAMELST="$PNAMELST $PNAME"

			# Check for dependency for $PNAME
			if [ ${INSTALL_32BIT_ONLY} -eq 0 ]
			then
				DEPLIST=`grep '^#>#'$PNAME $0 | cut -f2 -d:`
			else
				DEPLIST=`grep '^#^#'$PNAME $0 | cut -f2 -d:`
			fi
			TLIST="$TLIST $DEPLIST $PNAME"
			FLAG=0
		done

	done
	TARGETLIST=" $TLIST "
	return ${RETURN_CODE}
}


######################################################################
# Function to return the package description and size in megabytes
# Usage: getpkgdesc <pkgname>
######################################################################
getpkgdesc()
{
	if [ ${INSTALL_32BIT_ONLY} -eq 0 ]
	then
		DESC=`grep '^#>#'$1 $0 | cut -f3 -d:`
		DEPS=`grep '^#>#'$1 $0 | cut -f2 -d:`
	else
		DESC=`grep '^#^#'$1 $0 | cut -f3 -d:`
		DEPS=`grep '^#^#'$1 $0 | cut -f2 -d:`
	fi

	BC=`menu_item_byte_count "$1 ${DEPS}"`
	let BCMB=${BC}/1045876
	echo "$DESC (${BCMB} MB)"
}

######################################################################
# Function to display one line of Menu
# Usage: displayline "<n>:<Long Message>"
######################################################################
displayline()
{
	if [ $# -lt 1 ]
	then
		echo
	else
		echo $1 | awk -F: '{ printf("%3s    %-s\n",$1,$2); }'
	fi
}


######################################################################
# Function to Display a list of all packages that can be installed
######################################################################
displaypackages()
{
	tput clear
	indx=1
	echo
	echo
	displayline "Item:Package Description"
	displayline "----:----------------------------------------------------------------------"
	TMLIST=$DISPLAY_ORDER
	for nm in $TMLIST
	do
		
  	    displayline "$indx: `getpkgdesc $nm`"
	    let indx=$indx+1
            # if called from HMP5.0 installation, only display HMP w/0 DNI package
            if [ ${SUB_PROCESS} -eq 1 ]
	    then
                break
            fi
	done
	displayline
	if [ ${SUB_PROCESS} -eq 0 ]
	then
	    ALLSIZE=`menu_item_byte_count "${ORDER}"`
	    let ALLSIZE=${ALLSIZE}/1045876
	    displayline "A: Install All (${ALLSIZE} MB)"
	fi
	displayline "Q: Quit Installation"
}

#########################################################################
#       Function replaces /bin/echo (or shell builtin) but only         #
#       prints the message if ${SILENT}=0.                              #
#########################################################################
myecho()
{
	if [ ${SILENT} -eq 0 ]
	then
		case "$1" in
			-n|-e|-ne)					# handle echo options
				echo ${1} "${2-}"
				;;
			*)
				echo "$*"
				;;
		esac
	fi
}

#########################################################################
#       After the install, if any config files have been backed up      #
#       and we have a conversion utility for the file then use it to    #
#       migrate the parameters from the user's saved file to the new    #
#       file in ${DLGCROOT}.  If we don't have a conversion utility,    #
#       just inform the user that the parameters will need to be        #
#       manually transferred.                                           #
#########################################################################
migrate_config_from_backup()
{

	# get list of backed-up files
	FILE_LIST=`find ${DLGCROOT} -type f -name \*.rpmsave -print`

	# if any backed-up files exist
	if [ -n "${FILE_LIST}" ]
	then
		for file in ${FILE_LIST}
		do

			# isolate file subdir, name, and extension.
			FILE_SUBDIR=`dirname $file | awk -F"/" '{ print $NF }'`
			FILE_NAME=${file##*/}
			FILE_EXT=${FILE_NAME##*.}

			# if we have a migration utility, migrate the user's configuration
			# from the saved copy into the ${DLGCROOT} dir and save a copy.
			# Otherwise just copy the file back into ${DLGCROOT} as <file>.rpmsave
			
			case "${FILE_EXT}" in
				
				scd)
					if [ "${FILE_NAME}" = "pyramid.scd" ] &&
						[ -x ${DLGCROOT}/bin/tbparms ]
					then
						myecho
						${DLGCROOT}/bin/tbparms \
							-f ${DLGCROOT}/${FILE_SUBDIR}/${FILE_NAME} \
							-c ${DLGCROOT}/${FILE_SUBDIR}/${FILE_NAME}.new > /dev/null 
						myecho "The configuration information from your pyramid.scd file was"
						myecho "automatically migrated and the resulting file copied to"
						myecho "${DLGCROOT}/${FILE_SUBDIR}/${FILE_NAME}.new."
						myecho
					fi
					;;

			esac

		done

		if [ ${SILENT} -eq 0 ]
		then
			echo
			echo "Your modified configuration files have been renamed and saved as listed below."
			echo "If you have installed new hardware, or are utilizing new features in this"
			echo "release, may need to migrate and/or modify the parameters in your existing"
			echo "configuration files.  Refer to the ${SHORTRELNAME} ${RELNUM} for Linux Software"
			echo "Installation Guide for additional details.  Otherwise, if you do not intend to"
			echo "make any additional configuration changes, you may choose to rename these files"
			echo "without the .rpmsave extension."
			echo
			find /usr/dialogic -type f -name \*.rpmsave -print |
				awk '{ printf("\t%s\n", $0) }'
		fi

	fi

}

#########################################################################
#       Determine if necessary system utilities are in the path         #
#       ADD ENTRIES AS NECESSARY WHEN THIS SCRIPT IS MODIFIED           #
#########################################################################
check_prerequisite_cmds()
{
	for CMD in awk basename cp cut date df dirname echo egrep eval find \
					grep id md5sum mkdir mv printf rm sed stty tput tr wc
	do
		# type is a bash internal command which can be used
		# to verify if the specified program exists in the path.
		type -a ${CMD} &> /dev/null
		if [ $? != 0 ]
		then
			echo "Error: $0: The command ${CMD} is not in the path."
			myexit 1
		fi
	done
}


#########################################################################
#       Detect previous version of software.                            #
#########################################################################
detect_previous()
{
	if [ ${UPGRADE_INSTALL} -eq 0 ]
	then
		# No previous release - nothing to do (yet)
		UPGRADE=0
	else
		UPGRADE=1
	fi
}


#########################################################################
#       Display a list of valid package selections for this release.
#       Used for silent install error cases
#########################################################################
silent_valid_list()
{
	TMLIST="${SILENT_KEYWORDS}"

	# gotta call this function to initialize package sizes
	check_pkg_avail

	echo
	echo "Valid package selections are:"
	echo

	echo "Package Name  Package Description"
	echo "------------  -------------------"

	for nm in $TMLIST
	do
		printf "%-11s   %s\n" $nm "`getpkgdesc $nm`"
		#echo "$nm	`getpkgdesc $nm`"
	done

	ALLSIZE=`menu_item_byte_count "${ORDER}"`
	let ALLSIZE=${ALLSIZE}/1045876

	#echo "ALL		All available packages (${ALLSIZE} MB)"
	printf "%-11s   %s (%s MB)\n" "ALL" "All available packages" ${ALLSIZE}
	echo
}

#########################################################################
#       Add dependencies to list of packages specified from the
#       command line.  Used for silent install only.       
#########################################################################
silent_depend()
{
	# Append a trailing space to the list of available
	# packages so we can avoid false grep returns
	# for packages with similar names
	TORDER=${SILENT_KEYWORDS}" ."

	for PKG in ${SILENT_LIST}
	do

		# check if the user specified all packages from the command line

		if [ "${PKG}" = "ALL" ]
		then
			ALL=1
			TLIST=${ORDER}
			break
		fi
		
		# check if the user specified an invalid package from the command line.
		# if so, display a list of valid names for this release and exit.

		if [ "`echo ${TORDER} | grep -c ${PKG}\ `" = "0" ] && [ ${ALL} = "0" ]
		then
			echo "$0: ERROR: Invalid package name ${PKG}."

			silent_valid_list
			myexit 1
		fi

		# Check for dependency for $PKG
		if [ ${INSTALL_32BIT_ONLY} -eq 0 ]
		then
			DEPLIST=`grep '^#>#'${PKG} $0 | cut -f2 -d:`
		else
			DEPLIST=`grep '^#^#'${PKG} $0 | cut -f2 -d:`
		fi
		TLIST="${TLIST} ${DEPLIST} ${PKG}"

	done		# for PKG in ${SILENT_LIST}

	TARGETLIST=" ${TLIST} "
}

#########################################################################
#    Process command line. Check for silent install. If requested,
#    set flag and grab list of user-specified packages
#########################################################################
process_cmd_line()
{
	if [ $# -gt 0 ]
	then
                if [ $# -eq 1 ]
                then
                        echo "Usage: $0 [--silent] [--sub_process] <PKG1> <PKG2> ... <PKGn>"
                        silent_valid_list
                        myexit 1
                else
                     SILENT_LIST=`echo $* | cut -d" " -f2-`
                     if [ "$1" = "--remote" ]
                     then
                          SILENT=1
                          REMOTE=1
                          SUB_PROCESS=1
                     elif [ "$1" = "--silent" ]
                     then
                          SILENT=1
                     elif [ "$1" = "--sub_process" ]
                     then
                          SUB_PROCESS=1
                     fi
                fi
        fi

}


#########################################################################
# display the name of RPMs from a previous release which have not
# yet been upgraded.
#########################################################################
warn_old_rpms()
{
	OLDRPMS=
	for rpm in `${QUERY_PKG} DLGC\* ${LSBPKGNAME}\*`
	do
		if [ ! -f ${rpm}${PKGSEP}*${PKG_EXT} ]
		then
 			OLDRPMS="${OLDRPMS} ${rpm}"
		fi
	done

	if [ -n "${OLDRPMS}" ]
	then
		myecho
		myecho "The following RPM's from a previous release have not yet been updated."
		myecho "Please do so by re-running install.sh and selecting the appropriate"
		myecho "menu items."
		for i in ${OLDRPMS}
		do
			myecho "		$i"
		done
		myecho
	fi

}


#########################################################################
# extract the uncompressed size of the content of an RPM
#
# INPUT:    $1 = RPM name
# RETURNS:  uncompressed size of RPM content in bytes on stdout
#########################################################################
get_rpm_content_size()
{
	if [ $# -ne 1 ]
	then
		echo "USAGE: get_rpm_content_size <pathname>"
		echo "$1"
		myexit 1
	fi

	if [ ! -f $1 ]
	then
		echo "ERROR: get_rpm_content_size(): File $1 not found"
		echo "-1"
		myexit 1
	fi

	if [ "${DIST}" = "Debian" ]
	then
		echo `${PKG_INFO} $1 | grep size | sed 's/size //g' | sed 's/ bytes: .*//g'`
	else
		echo `${PKG_INFO} $1 | awk '/^Size[  ]*:/ { print $3 }'`
	fi
}


#########################################################################
# check if the amount of available disk space on the filesystem
# containing ${DLGCROOT} is sufficient for the selected packages.
#
# INPUT: $1 = Number of bytes needed to install the selected packages
#########################################################################
check_diskspace_dlgcroot_filesystem()
{
	if [ $# -ne 1 ]
	then
		echo "Usage: check_diskspace_dlgcroot_filesystem() <number of bytes>"
		myexit 1
	fi

	if [ $1 -le 0 ]
	then
		echo "ERROR: check_diskspace_dlgcroot_filesystem(): number of bytes must be > 0"
		myexit 1
	fi

	myecho
	myecho -ne "Checking for sufficient disk space (need $1 MB): "

	# check ${DLGCROOT} first in case it's a symlink
	SPACE_AVAILABLE=`df -mP ${DLGCROOT} 2>/dev/null | grep -v "^Filesystem" | awk '{ print $4 }'`
	TOTAL_SPACE=`df -mP ${DLGCROOT} 2>/dev/null | grep -v "^Filesystem" | awk '{ print $2 }'`

	# df will fail if ${DLGCROOT} doesn't exist.  If so, try /usr
	# since in this release the software must be installed there.
	if [ -z ${SPACE_AVAILABLE} ]
	then
		# TO-DO: fix hardcoded path
		SPACE_AVAILABLE=`df -mP /usr 2>/dev/null | grep -v "^Filesystem" | awk '{ print $4 }'`
		TOTAL_SPACE=`df -mP /usr 2>/dev/null | grep -v "^Filesystem" | awk '{ print $2 }'`
	fi

	# how much will be used after install is done
	let TMP1=${SPACE_AVAILABLE}+$1
	# how much will be left after install is done
	let TMP2=${TOTAL_SPACE}-${TMP1}
	# percentage used
	let PERCENT_USED=100*${TMP2}/${TOTAL_SPACE}

	if [ ${SPACE_AVAILABLE} -le $1 ]
	then
		myecho
		myecho "Insufficient free disk space is available on the filesystem containing"
		myecho "${DLGCROOT} to install the selected software."
		myecho "Need $1 MB, but only have ${SPACE_AVAILABLE} MB."
		myecho
		myecho "Please free up some disk space and re-run $0 again."
		myecho
		myecho "$0 aborting due to lack of required disk space."
		myexit 3
	else
		# let the user know what % of the disk will
		# be used after installation is complete. 
		myecho "OK"
		myecho "After installation, the filesystem containing the"
		myecho "${FAMILY} Software will be ${PERCENT_USED}% full."
		myecho
	fi

}

#########################################################################
# return the number of bytes associated with a menu selection
#
#  INPUT: $1 = A space-separated list of RPM's associated with a selection
# OUTPUT: Returns size in megabytes for a menu selection on stdout
#########################################################################
menu_item_byte_count()
{
	if [ $# -ne 1 ]
	then
		echo "Usage: menu_item_byte_count <list of RPMs>"
		myexit 1
	fi

	# initialize total byte count for a menu selection
	MS=0

	# loop through all RPM's adding their size to the total
	for rpm in $1
	do
		TMS=`eval echo ${rpm##$LSBPKGNAME}SIZE`
		let MS=${MS}+${TMS}
	done

	echo ${MS}
}


#########################################################################
# Build a list of RPM transactions.  In dependency order, group RPM's   #
# together in a transaction which need the --replacefiles flag and      #
# those which do not in a separate transactions.                        #
#                                                                       #
# NEEDS_REPLACEFILES variable must be set before function is called.    #
# Sets environment variables "TRANSx" where x is the transaction        #
# number, and TRANSNUM which indicates the one greater than the         #
# number of transactions created.  Note that TRANS0 is an empty string  #
# and should be ignored.                                                #
#                                                                       #
# INPUT: $1 = List of RPM's to be installed in dependency order         #
#########################################################################
buildtrans()
{

	# check arguments
	if [ -z "$1" ]
	then
		echo "ERROR: usage: buildtrans <dependency-sorted list of RPM's>"
		myexit 1
	fi

	lasttrans=0		# 1=no --replacefiles, 0=--replacefiles
	TRANSNUM=0

	for PKG in ${1}
	do
		echo " ${NEEDS_REPLACEFILES} " | grep -i "${PKG} " &> /dev/null
		replaceflag=$?

		# is --replacefiles needed for this RPM?
		case "${replaceflag}" in
			0)
				# --replacefiles is needed for this RPM
				case "${lasttrans}" in	
					0)
						# last transaction list was no --replacefiles
						# so start a new transaction list and add this
						# RPM name to it
						let TRANSNUM=${TRANSNUM}+1
						lasttrans=1
						eval TRANS${TRANSNUM}=\"NEEDS_RF \${TRANS${TRANSNUM}} ${PKG}\"
						;;
					1)
						# last transaction list was --replacefiles
						# so append this RPM name to it
						let TRANSNUM=${TRANSNUM}+1
						eval TRANS${TRANSNUM}=\"\${TRANS${TRANSNUM}} ${PKG}\"
						;;
					*)
						# should not occur
						echo "ERROR: $0: unexpected value for lasttrans: ${lasttrans}"
						myexit 1
						;;
				esac
				;;
	
			1)
				# No it isn't.
				case "${lasttrans}" in
					0)
							# last transaction list was no --replacefiles
							# so append this RPM name to it
							eval TRANS${TRANSNUM}=\"\${TRANS${TRANSNUM}} ${PKG}\"
							;;
					1)
							# last transaction list was --replacefiles
							# so start a new transaction list and add this
							# RPM name to it
							let TRANSNUM=${TRANSNUM}+1
							lasttrans=0
							eval TRANS${TRANSNUM}=\"\${TRANS${TRANSNUM}} ${PKG}\"
							;;
					*)
							# should not occur
							echo "ERROR: $0: unexpected value for lasttrans: ${lasttrans}"
							myexit 1
							;;
				esac
				;;
	
			*)
				# should not occur
				echo "ERROR: $0: unexpected value for replaceflag: ${replaceflag}"
				myexit 1
				;;
	
			esac
	done

}

#########################################################################
#                                                                       #
# Check for any obsolete packages on the system.  Put the list of those #
# found into ${DELETE_LIST}.                                            #
#                                                                       #
#########################################################################
check_obsolete_packages()
{
		myecho
		myecho "Checking for obsolete packages:"

		#
		# cycle through the list of packages from previous releases which have been
		# obsoleted and if found, add them to the list of packages to delete.
		#
		DELETE_LIST=

		for PKGNM in ${OBSOLETE_PKGS}
		do
			# assume not found until proven otherwise
			FOUNDOLD=0

			# check for installed version of the obsolete package and set flag if found
			rpm -q --quiet ${PKGNM} && FOUNDOLD=1

			# if found add to list of packages to be deleted and for interactive install
			# let the user know that this package is obsolete and will be removed
			if [ ${FOUNDOLD} -eq 1 ]
			then
				DELETE_LIST="${DELETE_LIST} ${PKGNM}"
				[ ${SILENT} -eq 0 ] &&
					printf "   %-14s   %s\n" $PKGNM... "is obsolete and will be removed"
			fi
		done

		# if the list of packages to be deleted is empty, let the user know
		if [ -z "${DELETE_LIST}" ]
		then
			myecho "   None found."
		fi
}


#########################################################################
#                                                                       #
# Delete the obsolete packages contained in the list ${DELETE_LIST}     #
#                                                                       #
#########################################################################
delete_obsolete_packages()
{
	myecho
	myecho "Removing packages:"
	for PKGNM in ${DELETE_LIST}
	do
		myecho -ne "   $PKGNM\tdue to obsolescence...\t"

		${ERASE_PKG} ${PKGNM} &> /dev/null

		RETVAL=$?
		[ ${RETVAL} -ne 0 ] && ERROR=1
		disp_erase_return_code $RETVAL
	done

	if [ -z "${DELETE_LIST}" ]
	then
		myecho "   None found."
	fi
}

#########################################################################
#                                                                       #
# Ask the user if they want to configure the newly installed software   #
#                                                                       #
#########################################################################
ask_configure()
{
	echo
	echo -ne "Do you wish to run config.sh to configure your system [y/n] ? \c"
	read line
	resp=`echo ${line} | cut -c1 | tr '[A-Z]' '[a-z]'`
	if [ -z "${resp}" ] || [ "${resp}" = "y" ]
	then
		if [ -x ./config.sh ]
		then
			tput clear
			./config.sh

		   if [ -x "${INTEL_DIALOGIC_DIR}/demos/Verification/VerificationDemo/Release/VerificationDemo" ]
			then
				myecho ""
				myecho -ne "Do you wish to run the HMP demo on your system [y/n] ? \c"
				read line
				resp=`echo ${line} | cut -c1 | tr '[A-Z]' '[a-z]'`

				if [ "${resp}" = "y" ]
				then
					tput clear
					dlstart
					ORIG_DIR=`pwd`
					#source Dialogic environment variables
					source /etc/profile.d/ct_intel.sh
					source /etc/profile.d/orbacus.sh
					cd "${INTEL_DIALOGIC_DIR}/demos/Verification/VerificationDemo/Release/"
					./VerificationDemo
					cd "${ORIG_DIR}"
					dlstop
				fi
			fi
		else
			echo
			echo "ERROR: config.sh not found in the current directory".
			echo
		fi
	fi
}


#########################################################################
#                                                                       #
# Install files not included in the RPM's                               #
#                                                                       #
#########################################################################
install_non_rpm_files()
{
	# if there is an existing buildinfo file, rename it.
	if [ -d ${DLGCROOT}/cfg ]
	then
		if [ -f ${DLGCROOT}/cfg/buildinfo ]
		then
			mv ${DLGCROOT}/cfg/buildinfo ${DLGCROOT}/cfg/buildinfo_prev
		fi

		# copy the new buildinfo file into ${DLGCROOT}/cfg
		if [ -f buildinfo ]
		then
			cp -f buildinfo ${DLGCROOT}/cfg
		fi
	fi

	if [ -d ${DLGCROOT}/bin ]
	then
		# copy the uninstall script into ${DLGCROOT}/bin
		[ -f ${DLGCROOT}/bin/dlguninstall.sh ] && rm -f ${DLGCROOT}/bin/dlguninstall.sh
		cp dlguninstall.sh ${DLGCROOT}/bin

		if [ "${tplatform}" == "hmp" ] && [ "${RELNUM}" == "1.5"  ]
		then
			# copy the config.sh script into ${DLGCROOT}/bin
			[ -f ${DLGCROOT}/bin/config.sh ] && rm -f ${DLGCROOT}/bin/config.sh
			cp config.sh ${DLGCROOT}/bin
		fi
	fi

	if [ ${PLATFORM} = "HMP" ] && [ "${RELNUM}" == "4.1"  ] ||
	[ ${PLATFORM} = "TVL2" ] || [ ${PLATFORM} = "OVL" ] || [ ${PLATFORM} = "ZVL" ]
	then
		if [ -x /etc/init.d/tvl2_startup ]
		then
			if [ ${DIST} = "RedHat" ]
			then
				ln -s /etc/init.d/tvl2_startup /etc/rc1.d/S00tvl2_startup &> /dev/null
				ln -s /etc/init.d/tvl2_startup /etc/rc2.d/S00tvl2_startup &> /dev/null
				ln -s /etc/init.d/tvl2_startup /etc/rc3.d/S00tvl2_startup &> /dev/null
				ln -s /etc/init.d/tvl2_startup /etc/rc5.d/S00tvl2_startup &> /dev/null
			elif [ ${DIST} = "SuSE" ]
			then
				ln -s /etc/init.d/tvl2_startup /etc/init.d/rc1.d/S00tvl2_startup &> /dev/null
				ln -s /etc/init.d/tvl2_startup /etc/init.d/rc2.d/S00tvl2_startup &> /dev/null
				ln -s /etc/init.d/tvl2_startup /etc/init.d/rc3.d/S00tvl2_startup &> /dev/null
				ln -s /etc/init.d/tvl2_startup /etc/init.d/rc5.d/S00tvl2_startup &> /dev/null
			fi
		fi
	fi
}

#########################################################################
#                                                                       #
# Post-installation custom actions                                      #
#                                                                       #
# NOTICE: Consumers of individual RPMs: these must be run after         #
#         all RPMs have been installed/upgraded.                        #
#                                                                       #
#########################################################################
post_install_custom_actions()
{
	# Check for file containing our environment variables.
	OURVARS=/etc/profile.d/ct_intel.sh
	if [ -x ${OURVARS} ] 
	then
	        # get our environment variables
	        . ${OURVARS}
	fi

	if [ ${DIST} = "Debian" ]
	then
		# set the environment variables in /etc/profile
		grep '^# Dialogic(r) HMP Environment Variables$' /etc/profile &> /dev/null
		if [ $? -ne 0 ]
		then
			echo "# Dialogic(r) HMP Environment Variables" >> /etc/profile
		fi

		grep '^. /etc/profile.d/ct_intel.sh$' /etc/profile &> /dev/null
		if [ $? -ne 0 ]
		then
			echo . /etc/profile.d/ct_intel.sh >> /etc/profile
		fi

		grep '^. /etc/profile.d/orbacus.sh$' /etc/profile &> /dev/null
		if [ $? -ne 0 ]
		then
			echo . /etc/profile.d/orbacus.sh >> /etc/profile
		fi
 
		# temp for orbacus and ct_intel
		if ! [ -d /var/lock/subsys ]
		then
			mkdir /var/lock/subsys
		fi
	fi

	# install files not delivred as part of the RPM's
	install_non_rpm_files

}

#########################################################################
#                                                                       #
# Check for existing packages on the system                             #
#                                                                       #
#########################################################################
check_existing_packages()
{
	for PKGNM in $SORTLIST
	do
		OLDRPM=`${QUERY_PKG} ${PKGNM} 2>/dev/null`
		# the sed command below needs to be changed
		# once the release includes noarch or src RPM's
		PKG_SUFFIX=
		echo ${OLDRPM} | grep -q "_gcc${GCCVER}" && PKG_SUFFIX=_gcc${GCCVER}
		NEWRPM=`/bin/ls ${PKGNM}${PKGSEP}*${PKG_SUFFIX}*${PKG_EXT} 2>/dev/null | sed 's/.[0-z]*${PKG_EXT}//g'`
		echo ${NEWRPM} | grep -q "${OLDRPM}"
		RET=$?
		
		if [ ${RET} -eq 0 ]
		then
			FOUNDLST="$FOUNDLST $PKGNM"
			[ ${SILENT} -eq 0 ] &&
				printf "   %-14s   %s\n" $PKGNM... "is already installed"

		else
			# package not found on the system so add it to
			# the list of packages to be installed or upgraded.
			PNAMELST="$PNAMELST $PKGNM"
			RPMSIZE=`eval echo \\$${PKGNM##$LSBPKGNAME}SIZE`
			let SPACE_NEEDED=${SPACE_NEEDED}+${RPMSIZE}
			if [ `echo ${OLDRPM} | grep -c is\ not\ installed` -eq 0 ]
			then
				UPFLAG=1
				# Some version-release of ${PKGNM} found on the system.
				[ ${SILENT} -eq 0 ] &&
					printf "   %-14s   %s\n" $PKGNM... "will be upgraded"
			fi
		fi
	done
}

#########################################################################
#                                                                       #
# Install the packages contained in the transaction lists created       #
# when buildtrans() is called.                                          #
#                                                                       #
#########################################################################
run_install_transactions()
{
	RETVAL=0

	if [ ${SILENT} -ne 0 ]
	then
		# complete silence for silent install
		REDIR_OUT="&> /dev/null"
	else
		# redirect stderr to prevent file upgrade messages
		# from corrupting the install progress display for
		# interactive install.
		REDIR_OUT="2>/dev/null"
	fi

	tnum=0
	while [ ${tnum} -le ${TRANSNUM} ]
	do
		CTRANS=`eval echo \\${TRANS${tnum}}`
		# only process non-zero length transactions
		if [ -n "${CTRANS}" ]
		then
			echo ${CTRANS} | grep "^NEEDS_RF" &> /dev/null
			RV=$?

			if [ ${RV} -eq 0 ]
			then
				CTRANS="`echo ${CTRANS} | sed 's/NEEDS_RF //g'`"
				REPLACE="--replacefiles"
			else
				REPLACE=
			fi

			NUMRPMSINTRANS=`echo ${CTRANS} | wc -w`
			CNT=1

			# expand each RPM name in the transaction list to include
			# the appropriate compiler-specific package suffix if needed
			while [ ${CNT} -le ${NUMRPMSINTRANS} ]
			do
				THERPM=`echo ${CTRANS} | cut -d" " -f${CNT}`
				set_pkg_suffix ${THERPM}
				CTRANS=`echo ${CTRANS} | sed 's/'"${THERPM}"'/'"${THERPM}${PKG_SEP}"'*'"${PKG_SUFFIX}"'*'"${PKG_EXT}"'/1'`
				let CNT=${CNT}+1
			done

			# install/upgrade the RPMs in the current transaction list
			myecho
			
			# force replacefiles if 64-bit
			if [ ${INSTALL_32BIT_ONLY} -eq 0 ]
			then
				REPLACE="--replacefiles"
			fi
			
			eval ${UPGRADE_PKG} ${NODEPS} ${REPLACE} ${CTRANS} ${REDIR_OUT}

			RETVAL=$?

			if [ ${RETVAL} -eq 0 ]
			then
				# remember that we installed at least 1 RPM
				PKGFLAG=1
			else
				# error occurred
				ERROR=1
			fi

			# display install result
			disp_install_return_code $RETVAL "${CTRANS}"

			# null out the transaction list
			eval TRANS${tnum}=			
		fi

		let tnum=tnum+1

	done
}

######################################################################
# Get OS release information
######################################################################
get_release_info()
{
	OS=`uname -s`

	if [ "${OS}" = "Linux" ] ; then
		if [ -f /etc/redhat-release ] ; then
			DIST='RedHat'
			PSUEDONAME=`cat /etc/redhat-release`
		elif [ -f /etc/SuSE-release ] ; then
			DIST="SuSE"
			PSUEDONAME=`cat /etc/SuSE-release`
		elif [ -f /etc/debian_version ] ; then
			DIST="Debian"
			PSUEDONAME=`cat /etc/debian_version`
		elif [ -f /etc/turbolinux-release  ] ; then
			DIST="Turbolinux"
			PSUEDONAME=`cat /etc/turbolinux-release`
		else
			DIST="UNKNOWN"
			PSUEDONAME="UNKNOWN"
		fi
	else
		#Unsupported OS
		DIST="UNKNOWN"
		PSUEDONAME="UNKNOWN"
	fi
}

######################################################################
# Check if the OS release is valid
######################################################################
check_release()
{
   # Supported OS Distributions
   case "${DIST}" in
      Debian)
         # Supported OS Distribution
         ;;
      RedHat)
         # Supported OS Distribution
         ;;
      SuSE)
         # Supported OSD
         ;;
      Turbolinux)
         # Supported OS Distribution
         ;;
      UNKNOWN)
         # Unable to determine OS Distribution
         echo
         echo "Unable to determine the specific distribution of Linux you are running."
         echo
         myexit 1
         ;;
      *)
         # Unsupported OSD (i.e., Mandrake)
         echo
         echo "OS Distribution is not supported."
         echo
         myexit 1
         ;;
   esac

   # Unsupported OS Descriptions
   case "${PSUEDONAME}" in
      "Red Hat Enterprise Linux AS release 3 (Taroon)")
         # Unsupported OS version
         echo
         echo "${OS_DESC} is not supported."
         echo
         myexit 1
         ;;
      UNKNOWN)
         # Unable to determine OS Version
         echo
         echo "Unable to determine the OS version"
         echo
         myexit 1
         ;;
      *)
         # Supported OS Version (i.e., Red Hat AS 3.0 Update 1)
         ;;
   esac
}

######################################################################
# set the package commands
######################################################################
set_package_cmds()
{
	# Supported OS Distributions
	case "${DIST}" in
		Debian)
			ERASE_PKG="dpkg --purge"
			INSTALL_PKG="dpkg -i"
			QUERY_PKG="dpkg -l"
			UPGRADE_PKG="dpkg -i"

			PKG_EXT=".deb"

			# determine the lsb package name
			LSBPKGNAME=`ls *${PKG_EXT}` &> /dev/null
			LSBPKGNAME=`echo ${LSBPKGNAME} | sed "s:lsb-${COMPANY_NAME}-::g" | sed 's:-.*::g'`
			LSBPKGNAME=lsb-${COMPANY_NAME}-${LSBPKGNAME}-

			PKGBASE=${PKG%%_*}
			PKG_INFO="dpkg --info"
			PKGNAME="${PKG%_*$PKG_EXT}"
			PKGS=`ls ${LSBPKGNAME}*_*${PKG_EXT}`
			PKGSEP="_"
			;;
		RedHat | SuSE | Turbolinux)
			ERASE_PKG="rpm --erase"
			INSTALL_PKG="rpm --install"
			QUERY_PKG="rpm --query"
			UPGRADE_PKG="rpm --upgrade -hv"
 
			PKG_EXT=".rpm"

			# determine the lsb package name
			LSBPKGNAME=`ls *${PKG_EXT}` &> /dev/null
			LSBPKGNAME=`echo ${LSBPKGNAME} | sed "s:lsb-${COMPANY_NAME}-::g" | sed 's:-.*::g'`
			LSBPKGNAME=lsb-${COMPANY_NAME}-${LSBPKGNAME}-

			PKGBASE=${PKG%%-*}
			PKG_INFO="rpm -qpi"
#			PKG_INFO_CONTENT_SIZE_CMDS=`awk '/^Size[  ]*:/ { print $3 }'`
			PKGNAME=${PKG%.*$PKG_EXT}
			PKGS=`ls ${LSBPKGNAME}*-*${PKG_EXT}`
			PKGSEP="-"
			;;
		*)
			# Unsupported OSD (i.e., Mandrake)
			echo
			echo "OS Distribution is not supported."
			echo
			myexit 1
			;;
	esac
}

######################################################################
# query for installed packages
######################################################################
query_for_package()
{
        # Supported OS Distributions
        case "${DIST}" in
                Debian)
                        # Supported OS Distribution
                        FOUND_VERSION=`${QUERY_PKG} ${PKGBASE} | grep ${LSBPKGNAME}`
                        FOUND_VERSION=${FOUND_VERSION##*$LSBPKGNAME }
                        FOUND_VERSION=${FOUND_VERSION%% *}
                        FOUND_VERSION="${PKGBASE}_${FOUND_VERSION}"
                        ;;
                RedHat)
                        # Supported OS Distribution
                        FOUND_VERSION=`${QUERY_PKG} ${PKGBASE}`
                        ;;
                SuSE)
                        # Supported OSD
                        FOUND_VERSION=`${QUERY_PKG} ${PKGBASE}`
                        ;;
                Turbolinux)
                        # Supported OSD
                        FOUND_VERSION=`${QUERY_PKG} ${PKGBASE}`
                        ;;
                UNKNOWN)
                        # Unable to determine OS Distribution
                        echo
                        echo "Unable to determine the specific distribution of Linux you are running."
                        echo
			myexit 1
                        ;;
                *)
                        # Unsupported OSD (i.e., Mandrake)
                        echo
                        echo "OS Distribution is not supported."
                        echo
                        myexit 1
                        ;;
        esac
}

######################################################################
# Save the current tty settings
#
# Sets the global variable TTYSAVE
######################################################################
save_tty_settings()
{
   [ ${SILENT} -eq 0 ] && TTYSAVE=`stty -g 2>/dev/null`
}

######################################################################
# Save the current tty settings
#
# Sets the global variable TTYSAVE
######################################################################
restore_tty_settings()
{
   # restore the previously saved tty settings if they exist
   [ ${SILENT} -eq 0 -a -n "${TTYSAVE}" ] && stty ${TTYSAVE} &> /dev/null
}

######################################################################
# Set the tty settings
#
######################################################################
set_tty_settings()
{
if [ ${REMOTE} -eq 0 ]
then
   # Disable Ctrl+C, Ctrl+Z, Ctrl+D, & Ctrl+\
   stty intr \000 susp \000 eof \000 quit \000
fi
}


######################################################################
# Exit the program, resetting the prior terminal settings.
#
# $1 = Exit value to return to the shell
######################################################################
myexit()
{
   restore_tty_settings
   exit $1
}

######################################################################
# Check for the lsb dependency
#
# Exits the install if lsb-base is not installed
######################################################################
lsb_check()
{
	if ! [ -f /lib/lsb/init-functions ]
	then
		echo "#################################################################"
		echo "The base LSB package must be installed prior to installing HMP."
		echo "#################################################################"
		myexit 1
	fi
}

######################################################################
# check for dependencies
######################################################################
check_dependencies()
{
	# The T1/E1 RPM require the kernel source files.
	# Otherwise we won't be able to build the drivers.
	echo ${TARGETLIST} | grep t1e1 &> /dev/null
	RET=$?

	# check if the user choose to install the T1/E1 RPM
	if [ ${RET} = 0  ]
	then
		# look for the kernel source files
		if ! [ -d /lib/modules/`uname -r`/build ]
		then
			myecho
			myecho "ERROR: Unable to find kernel source files in"
			myecho "ERROR: \"/lib/modules/`uname -r`/build/\""
			myecho "ERROR: These files are required by the"
			myecho "ERROR: Dialogic(r) DNI Boards."
		
			myexit 1
		fi
	fi

	return
}

######################################################################
######################################################################
### BEGINNING OF SCRIPT
######################################################################
######################################################################

# Process command line arguments
process_cmd_line $*

# Save the current tty settings
TTYSAVE=
save_tty_settings

# Disable Ctrl+C, Ctrl+Z, Ctrl+D, & Ctrl+\
set_tty_settings

# make sure all commands and files needed by this script are available
check_prerequisite_cmds

# Check the OS release
get_release_info
check_release

# Get architecture
ARCH=`uname -m`

# Check if user is root
check_root

# Check if required kernel version is present
check_kernel

# set the the OSD specific package commands (rpm, dpkg, etc...)
set_package_cmds

# Check for dependency
lsb_check

# Check if gcc is available
check_gcc

# Check if force 32bit installation
check_32bit_install

# Validate the system date against the build creation date
check_date

# validate that this is a supported upgrade path
validate_upgrade_path

# Create a list of all the known packages
if [ ${INSTALL_32BIT_ONLY} -eq 0 ]
then
	ORDER=`grep '^#>#'${LSBPKGNAME} $0 | cut -f1 -d: | cut -f3 -d#`
else
	ORDER=`grep '^#^#'${LSBPKGNAME} $0 | cut -f1 -d: | cut -f3 -d#`
fi


# Used for the menu
if	[ "${PLATFORM}" = "HMP" ] && [ ${RELNUM} = "3.1" ] ||
	[ ${RELNUM} = "4.1" ] && [ "${PLATFORM}" = "HMP" ] ||
	[ "${PLATFORM}" = "ZVL" ]
then
    if [ ${SUB_PROCESS} -eq 1 ]
    then
		if [ ${INSTALL_32BIT_ONLY} -eq 0 ]
		then
			DISPLAY_ORDER=`grep '^#>#'${LSBPKGNAME} $0 | cut -f1 -d: | cut -f3 -d# |
			egrep -i "${LSBPKGNAME}hmp"`
		else
			DISPLAY_ORDER=`grep '^#^#'${LSBPKGNAME} $0 | cut -f1 -d: | cut -f3 -d# |
			egrep -i "${LSBPKGNAME}hmp"`
		fi
    else
		if [ ${INSTALL_32BIT_ONLY} -eq 0 ]
		then
			DISPLAY_ORDER=`grep '^#>#'${LSBPKGNAME} $0 | cut -f1 -d: | cut -f3 -d# |
			egrep -i "${LSBPKGNAME}hmp|${LSBPKGNAME}t1e1|${LSBPKGNAME}snmp"`
		else
			DISPLAY_ORDER=`grep '^#^#'${LSBPKGNAME} $0 | cut -f1 -d: | cut -f3 -d# |
			egrep -i "${LSBPKGNAME}hmp|${LSBPKGNAME}t1e1|${LSBPKGNAME}snmp"`
		fi
    fi
else
	if [ ${INSTALL_32BIT_ONLY} -eq 0 ]
	then
		DISPLAY_ORDER=`grep '^#>#'${LSBPKGNAME} $0 | cut -f1 -d: | cut -f3 -d# |
		egrep -i "${LSBPKGNAME}hmp|${LSBPKGNAME}t1e1|${LSBPKGNAME}snmp|${LSBPKGNAME}docs"`
	else
		DISPLAY_ORDER=`grep '^#^#'${LSBPKGNAME} $0 | cut -f1 -d: | cut -f3 -d# |
		egrep -i "${LSBPKGNAME}hmp|${LSBPKGNAME}t1e1|${LSBPKGNAME}snmp|${LSBPKGNAME}docs"`
	fi
fi

# Used for list of silent install packages
#
# WARNING:
#	DO NOT REMOVE ANY ITEM FROM THE SILENT_KEYWORDS LIST UNLESS THE
#	CORRESPONDING FUNCTIONALITY HAS BEEN DROPPED FROM THE RELEASE.
#	THESE ENTRIES MUST BE MAINTAINED BETWEEN RELEASES.
#
SILENT_KEYWORDS=${DISPLAY_ORDER}

# Detect if a previous release is installed, and choose the
# appropriate configuration migration and upgrade scheme
detect_previous


# Remove any packages from the list of install/upgrade
# candidates which are not included in this release
check_pkg_avail

# Count number of available, known packages
NUMPKGS=`echo $DISPLAY_ORDER | wc -w`

# Check if dependencies are met for all available, known packages
# ADD HERE

# For interactive install mode, display initial banner
if [ ${SUB_PROCESS} -eq 1 ] 
then 
    SELECT_ALL_MSG=""
else 
    SELECT_ALL_MSG="A for all packages,"
fi 

if [ ${SILENT} -eq 0 ] && [ ${SUB_PROCESS} -eq 0 ] 
then
	echo
	echo "================================================================================"
	center "${FULLNAME}"
	[ -n "${MAINTREL}" ] && center "${MAINTREL}"
fi
if [ ${SILENT} -eq 0 ] 
then
	center "Redistributable Runtime"
	center "INSTALLATION"
	echo
	echo "You will now have the opportunity to install software packages."
	echo "After the menu is displayed, enter the package number(s) of the desired"
	echo "packages, separated by a space.  Enter ${SELECT_ALL_MSG} Q to quit."
	echo
	echo "Package dependencies will be automatically resolved during installation.  For"
	echo "example, selecting a single package will automatically install all packages"
	echo "required for that selection."
	echo

	echo -e "Press ENTER to display the menu of packages:\c"

	read _ANS
fi

# Loop until user terminates or script aborts
while [ $QUITFLAG -eq 0 ]
do
	# if not a silent install, then prompt the user else
	# process the command line arguments which list the packages
	# to be installed

	if [ ${SILENT} -eq 0 ]
	then
		# interactive install

		# Display a list of all packages that can be installed
		displaypackages

		# Ask for Package Names to be installed
		TARGETLIST=
		asklist
		ERROR=${RETURN_CODE}
	else
		# silent install
		silent_depend
		TARGETLIST=" ${TLIST} "
	fi

	# break out of loop if ${QUITFLAG} was set in either asklist or silent_depend
	[ ${QUITFLAG} -eq 1 ] && break

	check_dependencies

	# Sort the response and make it unique
	SORTLIST=
	for ORDERPKG in $ORDER
	do
		if `echo "$TARGETLIST" | grep "$ORDERPKG"  &> /dev/null`
		then
			SORTLIST="$SORTLIST $ORDERPKG"
		fi
	done

	myecho
	myecho "Package installation order (including dependencies):"
	for NM in $SORTLIST
	do
	    myecho "   $NM"
	done

	# if this is an upgrade, check for any obsolete packages
	if [ ${UPGRADE} -eq 1 ]
	then
		check_obsolete_packages
	fi

	myecho
	myecho "Checking for previously installed packages:"

	FOUNDLST=
	PNAMELST=
	SPACE_NEEDED=0
	SPACE_AVAILABLE=0
	TOTAL_SPACE=0

	check_existing_packages

	if [ -z "$FOUNDLST" ] && [ ${UPFLAG} -ne 1 ]
	then
		myecho "   None found."
	fi

	if [ -z "${PNAMELST}" ]
	then
		myecho
		myecho "All needed packages are installed."
	else
		#
		# one or more requested packages are not currently installed
		#

		# convert space needed to Mbytes
		let SPACE_NEEDED=${SPACE_NEEDED}/1045876

		# check for sufficient free disk space
		check_diskspace_dlgcroot_filesystem ${SPACE_NEEDED}

		# tell the user what will be done
		TCNT=`echo ${PNAMELST} | wc -w | sed 's/[	 ]*//g'`
		# handle plural
		[ ${TCNT} -gt 1 ] && W="s"
		myecho "Installing or upgrading ${TCNT} needed package${W}:"

		# build the list(s) of install transactions
		buildtrans "${PNAMELST}"

		# Install the RPM's associated with the transactions list(s)
		run_install_transactions

		# if this is an upgrade, delete all obsolete packages which were found.
		if [ ${UPGRADE} -eq 1 ] && [ -n "${DELETE_LIST}" ]
		then
			delete_obsolete_packages
		fi
	fi

	myecho

	# post install custom actions
	post_install_custom_actions

	if [ ${SILENT} -eq 0 ]
	then
		# for interactive install, wait for the user to
		# press a key before redisplaying the menu again.
		echo
		echo -e "Press ENTER to continue:\c"
		read TMPFLG
	else
		# for silent install, exit the loop
		QUITFLAG=1
	fi

done # while [ $QUITFLAG -eq 0 ]

#
# if this is an upgrade then migrate any saved configuration
#
#TVL2 filter upgrade only true for HMP 
if [ ${UPGRADE} -eq 1 ]
then            
	if [ ${UPGRADE} -eq 1 ]
	then
		# migrate any saved configuration files
		migrate_config_from_backup
	fi
fi


# if we installed any software during this interactive install session
# then ask the user if they want to run the configuration script
# TVL2: Exclude config.sh from TVL2 platform
if [ ${PKGFLAG} -eq 1 ] && [ ${SILENT} -eq 0 ] 
then
   if [ "${PLATFORM}" = "HMP" ] && [ ${RELNUM} = "1.5" ]
   then
		ask_configure
   fi
else
   echo
fi

myexit ${ERROR}

#End of Script


