#!/bin/bash
#:set ts=3
#
######################################################################
#
# Script to install Dialogic(R) Software Development Kit
#
# Copyright (C) 2001-2008 Dialogic Corporation.  All Rights Reserved.
#
# All names, products, and services mentioned herein are the
# trademarks or registered trademarks of their respective
# organizations and are the sole property of their respective
# owners.
#
######################################################################

### SCRIPT VARIABLES AND CONSTANTS

#Using the buildinfo file determine what type of platform
#i.e hmp or tvl2
tplatform=`awk  '/Major Number/ {print $6}' buildinfo`
trel=`awk  '/Release Minor Number/ {print $6}' buildinfo`
tsubrel=`awk  '/Release SubMinor Number/ {print $6}' buildinfo`

# Release-specific definitions
BUS=""
COMPANY_NAME="dialogic"
FAMILY="Dialogic(R)"
MAINTREL=""
RELNUM=$trel
RELNAME="HMP Software Release ${RELNUM}LIN"
UPDATE=""
PLATFORM="`echo ${tplatform} | tr '[a-z]' '[A-Z]'`"

case "${tplatform}" in
	hmp)
		RELNAME="HMP Software Release ${RELNUM}LIN"
		
		case "$RELNUM" in
			1.5)
				COMPANY_NAME="intel"
				;;
			3.1 | 4.1)
				if [ "$tsubrel" != "-1" ]
				then
					RELNAME="${RELNUM}.${SUBRELNUM}LIN"
					FULLNAME=$RELNAME
				fi
				;;		
		esac
		;;
	ovl | tvl2)
		RELNAME="Dialogic(R) Multimedia Software Release ${RELNUM} for AdvancedTCA"
		FULLNAME="${RELNAME}"
		;;
	zvl)
		RELNAME="Multimedia Software for PCIe"
		FULLNAME="${RELNAME}"
		;;
esac

if [ -z "${UPDATE}" ]
then
        FULLNAME="${FAMILY} ${RELNAME}"
else
        FULLNAME="${FAMILY} ${RELNAME} ${UPDATE}"
fi

BUILDDIR=`pwd`
SDKERROR=0

#Determine the platform either tvl2 or hmp by 
#looking at the Major Number in the top buildinfo file.
PLATFORM=""
SDKSILENT=0

######################################################################
######################################################################
### GENERAL PURPOSE FUNCTIONS
######################################################################
######################################################################


#########################################################################
#    Process command line. Check for silent install. If requested,
#########################################################################
process_cmd_line()
{
        if [ $# -gt 0 ]
        then
                if [ "$1" == "--silent" ]
                then
                        SDKSILENT=1
                fi
        fi
}



######################################################################
# Get OS release information
######################################################################
get_release_info()
{
        OS=`uname -s`
 
        if [ "${OS}" = "Linux" ] ; then
                if [ -f /etc/redhat-release ] ; then
                        DIST='RedHat'
                        PSUEDONAME=`cat /etc/redhat-release`
                elif [ -f /etc/SuSE-release ] ; then
                        DIST="SuSE"
                        PSUEDONAME=`cat /etc/SuSE-release`
                elif [ -f /etc/debian_version ] ; then
                        DIST="Debian"
                        PSUEDONAME=`cat /etc/debian_version`
                elif [ -f /etc/turbolinux-release ] ; then
                        DIST="Turbolinux"
                        PSUEDONAME=`cat /etc/turbolinux-release`
        else
            DIST="UNKNOWN"
            PSUEDONAME="UNKNOWN"
                fi
        else
                #Unsupported OS
                DIST="UNKNOWN"
                PSUEDONAME="UNKNOWN"
        fi
}

######################################################################
# Check if the OS release is valid
######################################################################
check_release()
{
   # Supported OS Distributions
   case "${DIST}" in
      Debian)
         # Supported OS Distribution
         ;;
      RedHat)
         # Supported OS Distribution
         ;;
      SuSE)
         # Supported OSD
         ;;
      Turbolinux)
      # Supported OS Distribution
         ;;
      UNKNOWN)
         # Unable to determine OS Distribution
         echo
         echo "Unable to determine the specific distribution of Linux you are running."
         echo
         exit 1
         ;;
      *)
         # Unsupported OSD (i.e., Mandrake)
         echo
         echo "OS Distribution is not supported."
         echo
         exit 1
         ;;
   esac
 
   # Unsupported OS Descriptions
   case "${PSUEDONAME}" in
      "Red Hat Enterprise Linux AS release 3 (Taroon)")
         # Unsupported OS version
         echo
         echo "${OS_DESC} is not supported."
         echo
         exit 1
         ;;
      UNKNOWN)
         # Unable to determine OS Version
         echo
         echo "Unable to determine the OS version"
         echo
         exit 1
         ;;
      *)
         # Supported OS Version (i.e., Red Hat AS 3.0 Update 1)
         ;;
   esac
}

######################################################################
# set the package commands
######################################################################
set_package_cmds()
{
   # Supported OS Distributions
   case "${DIST}" in
      Debian)
	PKG_EXT=".deb"
	PKGS=`ls ${LSBPKGNAME}*_*${PKG_EXT}`

	ERASE_PKG="dpkg --purge"
	INSTALL_PKG="dpkg -i"
	QUERY_PKG="dpkg -l"
	UPGRADE_PKG="dpkg -i"

        PKGBASE=${PKG%%_*}
        PKGNAME=${PKG%_*$PKG_EXT}
	;;
      RedHat)
        PKG_EXT=".rpm"
        PKGS=`ls ${LSBPKGNAME}*-*${PKG_EXT}`

        ERASE_PKG="rpm --erase"
        INSTALL_PKG="rpm --install"
        QUERY_PKG="rpm --query"
        UPGRADE_PKG="rpm --upgrade -vh"

        PKGBASE=${PKG%%-*}
        PKGNAME=${PKG%.*$PKG_EXT}
	;;
      SuSE)
	PKG_EXT=".rpm"
	PKGS=`ls ${LSBPKGNAME}*-*${PKG_EXT}`

        ERASE_PKG="rpm --erase"
        INSTALL_PKG="rpm --install"
        QUERY_PKG="rpm --query"
        UPGRADE_PKG="rpm --upgrade -vh"

        PKGBASE=${PKG%%-*}
        PKGNAME=${PKG%.*$PKG_EXT}
         ;;
	  Turbolinux)
        PKG_EXT=".rpm"
        PKGS=`ls ${LSBPKGNAME}*-*${PKG_EXT}`

        ERASE_PKG="rpm --erase"
        INSTALL_PKG="rpm --install"
        QUERY_PKG="rpm --query"
        UPGRADE_PKG="rpm --upgrade -vh"

        PKGBASE=${PKG%%-*}
        PKGNAME=${PKG%.*$PKG_EXT}
		;;
      UNKNOWN)
         # Unable to determine OS Distribution
         echo
         echo "Unable to determine the specific distribution of Linux you are running."
         echo
         exit 1
         ;;
      *)
         # Unsupported OSD (i.e., Mandrake)
         echo
         echo "OS Distribution is not supported."
         echo
         exit 1
         ;;
   esac

	# determine the lsb package name
	LSBPKGNAME=`ls *${PKG_EXT}` &> /dev/null
	LSBPKGNAME=`echo ${LSBPKGNAME} | sed "s:lsb-${COMPANY_NAME}-::g" | sed 's:-.*::g'`
	LSBPKGNAME=lsb-${COMPANY_NAME}-${LSBPKGNAME}-
}

######################################################################
# query for installed packages
######################################################################
query_for_package()
{
	# Supported OS Distributions
	case "${DIST}" in
		Debian)
			# Supported OS Distribution
			FOUND_VERSION=`${QUERY_PKG} ${PKGBASE} | grep ${LSBPKGNAME}`
			FOUND_VERSION=${FOUND_VERSION##*$LSBPKGNAME }
			FOUND_VERSION=${FOUND_VERSION%% *}
			FOUND_VERSION="${PKGBASE}_${FOUND_VERSION}"
			;;
		RedHat)
			# Supported OS Distribution
			FOUND_VERSION=`${QUERY_PKG} ${PKGBASE}`
			;;
		SuSE)
			# Supported OSD
			FOUND_VERSION=`${QUERY_PKG} ${PKGBASE}`
			;;
		Turbolinux)
            # Supported OS Distribution
            FOUND_VERSION=`${QUERY_PKG} ${PKGBASE}`
            ;;
      	UNKNOWN)
			# Unable to determine OS Distribution
			echo
			echo "Unable to determine the specific distribution of Linux you are running."
			echo
			myexit 1
			;;
		*)
			# Unsupported OSD (i.e., Mandrake)
			echo
			echo "OS Distribution is not supported."
			echo
			exit 1
			;;
	esac
}

######################################################################
# Print the error from the installation return code
#     $1 - return code
######################################################################
disp_install_return_code()
{
	case "$1" in
		0)
			echo "Installation successful."
			;;
		*)
			NUMRPMS=`echo ${2} | wc -w`

			# check if the return code is equal to the number of RPMs
			if [ $NUMRPMS -ne $1  ]
			then
				echo
				echo "Fatal error during installation. Installation terminated."
				exit 1
			else
				# validate that the RPMs got installed
				for RPM in $2
				do
					# get the 4 part RPM name
					PKG=${RPM%%-[0-9].*}

					# verify that the package was actually installed
					${QUERY_PKG} $PKG &> /dev/null
					RET=$?

					if [ $RET -ne 0  ]
					then
						echo
						echo "Fatal error during installation. Installation terminated."
						exit 1
					fi
				done
			fi
			;;
	esac
}

######################################################################
######################################################################
### BEGINNING OF SCRIPT
######################################################################
######################################################################

process_cmd_line $*
get_release_info
check_release
set_package_cmds

# loop through all RPM's
for PKG in ${PKGS}
do
	set_package_cmds
	query_for_package ${PKGBASE}

	if [ "${FOUND_VERSION}" != "${PKGNAME}" ]
	then
		# what is currently installed does not match that in our package
		RET=0
		if [ "$SDKSILENT" -eq 0 ]
		then
			${UPGRADE_PKG} ${PKG} 2>/dev/null
		else
			${UPGRADE_PKG} ${PKG} &>/dev/null
		fi
		RET=$?
		disp_install_return_code ${RET} ${PKG}
		
		[ ${RET} -ne 0 ] && SDKERROR=1
	else
		echo "${FULLNAME}"
		echo "Software Development Kit for Linux is already installed."
		exit "${SDKERROR}"
	fi
done


exit "${SDKERROR}"




