/**
* @file bridgedcall.cpp
* @brief Definition of BridgedCall class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "logger.h"
#include "endpointmgr.h"
#include "bridgedcall.h"
#include "locker.h"
#include <string.h>

#include "bridgedcallopening_state.h"
#include "appevents.h"

//*****************************************************************************
// Function: BridgedCall::BridgedCall()
// Description: Default constructor
// Return:  BridgedCall*
// Parameters: none 
//*****************************************************************************
BridgedCall::BridgedCall()
{
}

//*****************************************************************************
// Function: BridgedCall::BridgedCall(int index, int m3gIndex, int sipIndex)
// Description: Initializing constructor
// Return:  BridgedCall*
// Parameters: int index 
//             int m3gIndex 
//             int sipIndex 
//*****************************************************************************
BridgedCall::BridgedCall(int index, int m3gIndex, int sipIndex):
   m_pCurrentState(0),
   m_Index(index),
   m_m3gIndex(m3gIndex),
   m_sipIndex(sipIndex)
{
   strncpy(m_CurrentStateStr, "OPENING        ",MAX_CURRENT_STATE_STR);
   strcpy(m_LastMessageBuff, "");

   m_pBridgeCallOpeningState = new BridgedCallOpeningState(this);
   m_pBridgedCallIdleState = new BridgedCallIdleState(this);
   m_pBridgedCallConnectingState = new BridgedCallConnectingState(this);
   m_pBridgeCallConnectedState = new BridgedCallConnectedState(this);
   m_pBridgedCallDisconnectingState = new BridgedCallDisconnectingState(this);
}

//*****************************************************************************
// Function: BridgedCall::~BridgedCall()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedCall::~BridgedCall()
{
   delete m_pBridgeCallOpeningState;
   delete m_pBridgedCallIdleState;
   delete m_pBridgedCallConnectingState;
   delete m_pBridgeCallConnectedState;
   delete m_pBridgedCallDisconnectingState;
}

//*****************************************************************************
// Function: void BridgedCall::ChangeState(BridgedCallState::E_BRIDGEDCALLSTATE e_NewCallState)
// Description: Change the object state
// Return: void 
// Parameters: BridgedCallState::E_BRIDGEDCALLSTATE e_NewCallState 
//*****************************************************************************
void BridgedCall::ChangeState(BridgedCallState::E_BRIDGEDCALLSTATE e_NewCallState)
{
   char oldStateStr[MAX_CURRENT_STATE_STR];
   strcpy(oldStateStr, m_pCurrentState->GetStateStr());
   
   // reset the current state for the next call:
   m_pCurrentState->Reset();

   switch ( e_NewCallState )
   {
      case BridgedCallState::IDLE_STATE:
         {
         SIPEndpoint *pSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx(SipIndex());
	 pSIPEndpoint->ZeroIpmPortConnectCompletionCnt();
         m_pCurrentState = m_pBridgedCallIdleState;
         }
         break;

      case BridgedCallState::CONNECTING_STATE:
         m_pCurrentState = m_pBridgedCallConnectingState;
         break;

      case BridgedCallState::CONNECTED_STATE:
         m_pCurrentState = m_pBridgeCallConnectedState;
         break;

      case BridgedCallState::DISCONNECTING_STATE:
         m_pCurrentState = m_pBridgedCallDisconnectingState;
         break;

      default:
         LOG_ERROR(0,"Unexpected state: %d\n",static_cast<int>(e_NewCallState));
         break;
   }

   // cache the state string in the endpoint
   strcpy(m_CurrentStateStr,m_pCurrentState->GetStateStr());
   LOG_ENTRY(0, "BridgedCall[%d] State transition: %s ---> %s\n",m_Index, oldStateStr,
		m_CurrentStateStr);
}


//*****************************************************************************
// Function: void BridgedCall::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCall::ProcessEvent(METAEVENT metaevent)
{
   m_StateLock.Lock();
   // dispatch event information to appropriate state:
   m_pCurrentState->ProcessEvent(metaevent);
   m_StateLock.Unlock();
}

//*****************************************************************************
// Function: BridgedCallState::E_BRIDGEDCALLSTATE BridgedCall::GetState()
// Description: Return the current state object
// Return: BridgedCallState::E_BRIDGEDCALLSTATE 
// Parameters: none 
//*****************************************************************************
BridgedCallState::E_BRIDGEDCALLSTATE BridgedCall::GetState()
{
   m_StateLock.Lock();
   BridgedCallState::E_BRIDGEDCALLSTATE state = m_pCurrentState->GetState();
   m_StateLock.Unlock();
   return state;
}

//*****************************************************************************
// Function: void BridgedCall::SetLastMessageString(const char *pMessageStr)
// Description: Store as the last messages
// Return: void 
// Parameters: char *pMessageStr 
//*****************************************************************************
void BridgedCall::SetLastMessageString(const char *pMessageStr)
{
   int length = (MAX_LAST_MSG_STRING > strlen(pMessageStr)) ? strlen(pMessageStr) : MAX_LAST_MSG_STRING-1;
   memset(m_LastMessageBuff, 0, sizeof(char)*MAX_LAST_MSG_STRING);
   strncpy(&m_LastMessageBuff[0], pMessageStr, length);
   m_LastMessageBuff[length] = '\0';
}

//*****************************************************************************
// Function: char* BridgedCall::GetStateString()
// Description: Get a formatted string describing the current state
// Return: char* 
// Parameters: none 
//*****************************************************************************
char *BridgedCall::GetStateString()
{
   memset(m_StateBuffer, 0, sizeof(char)*MAX_STATE_STRING);
   sprintf(m_StateBuffer, "BRC%2d: %15s M3G:%d <-> SIP:%d %s",
           m_Index,
           m_CurrentStateStr,
           m_m3gIndex,
           m_sipIndex,
           m_LastMessageBuff);
   return m_StateBuffer;
}

//*****************************************************************************
// Function: void BridgedCall::Open()
// Description: Set the BridgedCall to openming
// Return: void 
// Parameters: none 
//*****************************************************************************
void BridgedCall::Open()
{
   m_pCurrentState = m_pBridgeCallOpeningState;
}
