/**
* @file bridgedmedial.cpp
* @brief Definition of BridgedMedia class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "logger.h"
#include "endpointmgr.h"
#include "bridgedmedia.h"
#include "locker.h"
#include <string.h>

#include "bridgedmediaopening_state.h"
#include "appevents.h"

//*****************************************************************************
// Function: BridgedMedia::BridgedMedia(int index, int m3gIndex, int mmIndex)
// Description: Initializing constructor
// Return:  BridgedMedia*
// Parameters: int index 
//             int m3gIndex 
//             int mmIndex 
//*****************************************************************************
BridgedMedia::BridgedMedia(int index, int m3gIndex, int mmIndex):
   m_pCurrentState(0),
   m_Index(index),
   m_m3gIndex(m3gIndex),
   m_mmIndex(mmIndex)
{
   strncpy(m_CurrentStateStr, "OPENING        ",MAX_CURRENT_STATE_STR);
   strcpy(m_LastMessageBuff, "");

   m_pBridgedMediaOpeningState = new BridgedMediaOpeningState(this);
   m_pBridgedMediaIdleState = new BridgedMediaIdleState(this);
   m_pBridgedMediaPlayingState = new BridgedMediaPlayingState(this);
   m_pBridgedMediaRecordingState = new BridgedMediaRecordingState(this);
}

//*****************************************************************************
// Function: BridgedMedia::~BridgedMedia()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedMedia::~BridgedMedia()
{
   delete m_pBridgedMediaOpeningState;
   delete m_pBridgedMediaIdleState;
   delete m_pBridgedMediaPlayingState;
   delete m_pBridgedMediaRecordingState;
}

//*****************************************************************************
// Function: void BridgedMedia::ChangeState(BridgedMediaState::E_BRIDGEDMEDIASTATE e_NewCallState)
// Description: Change the object state
// Return: void 
// Parameters: BridgedMediaState::E_BRIDGEDMEDIASTATE e_NewCallState 
//*****************************************************************************
void BridgedMedia::ChangeState(BridgedMediaState::E_BRIDGEDMEDIASTATE e_NewCallState)
{
   char oldStateStr[MAX_CURRENT_STATE_STR];
   strcpy(oldStateStr, m_pCurrentState->GetStateStr());
   
   // reset the current state for the next call:
   m_pCurrentState->Reset();

   switch ( e_NewCallState )
   {
      case BridgedMediaState::IDLE_STATE:
         m_pCurrentState = m_pBridgedMediaIdleState;
         break;

      case BridgedMediaState::PLAYING_STATE:
         m_pCurrentState = m_pBridgedMediaPlayingState;
         break;

      case BridgedMediaState::RECORDING_STATE:
         m_pCurrentState = m_pBridgedMediaRecordingState;
         break;

      default:
         LOG_ERROR(0,"Unexpected state: %d\n",static_cast<int>(e_NewCallState));
         break;
   }

   // cache the state string in the endpoint
   strcpy(m_CurrentStateStr,m_pCurrentState->GetStateStr());
   LOG_ENTRY(0, "BridgedMedia[%d] State transition: %s ---> %s\n",m_Index, oldStateStr,
		m_CurrentStateStr);
}


//*****************************************************************************
// Function: void BridgedMedia::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: bool 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedMedia::ProcessEvent(METAEVENT metaevent)
{
   m_StateLock.Lock();
   // dispatch event information to appropriate state:
   m_pCurrentState->ProcessEvent(metaevent);
   m_StateLock.Unlock();
}

//*****************************************************************************
// Function: BridgedMediaState::E_BRIDGEDMEDIASTATE BridgedMedia::GetState()
// Description: Return the current state object
// Return: BridgedMediaState::E_BRIDGEDMEDIASTATE 
// Parameters: none 
//*****************************************************************************
BridgedMediaState::E_BRIDGEDMEDIASTATE BridgedMedia::GetState()
{
   m_StateLock.Lock();
   BridgedMediaState::E_BRIDGEDMEDIASTATE state = m_pCurrentState->GetState();
   m_StateLock.Unlock();
   return state;
}

//*****************************************************************************
// Function: void BridgedMedia::SetLastMessageString(const char *pMessageStr)
// Description: Store as the last messages
// Return: void 
// Parameters: char *pMessageStr 
//*****************************************************************************
void BridgedMedia::SetLastMessageString(const char *pMessageStr)
{
   int length = (MAX_LAST_MSG_STRING > strlen(pMessageStr)) ? strlen(pMessageStr) : MAX_LAST_MSG_STRING-1;
   memset(m_LastMessageBuff, 0, sizeof(char)*MAX_LAST_MSG_STRING);
   strncpy(&m_LastMessageBuff[0], pMessageStr, length);
   m_LastMessageBuff[length] = '\0';
}

//*****************************************************************************
// Function: char* BridgedMedia::GetStateString()
// Description: Get a formatted string describing the current state
// Return: char* 
// Parameters: none 
//*****************************************************************************
char *BridgedMedia::GetStateString()
{
   memset(m_StateBuffer, 0, sizeof(char)*MAX_STATE_STRING);
   sprintf(m_StateBuffer, "BRM%2d: %15s M3G:%d <-> MM:%d %s                                         ",
           m_Index,
           m_CurrentStateStr,
           m_m3gIndex,
           m_mmIndex,
           m_LastMessageBuff);
   return m_StateBuffer;
}

//*****************************************************************************
// Function: void BridgedMedia::Open()
// Description: Set the BridgedMedia to openming
// Return: void 
// Parameters: none 
//*****************************************************************************
void BridgedMedia::Open()
{
   m_pCurrentState = m_pBridgedMediaOpeningState;
}
