/**
* @file bridgedmediaidle_state.cpp
* @brief Definition of BridgedMediaIdleState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "bridgedmediaidle_state.h"
#include "bridgedmedia.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"
#include "epstate.h"

// static constant members:
const int BridgedMediaIdleState::H245_STARTED    = 0x01;
const int BridgedMediaIdleState::M3G_CONNECTED   = 0x02;
const int BridgedMediaIdleState::MEDIA_READY     = 0x03;

//*****************************************************************************
// Function: BridgedMediaIdleState::BridgedMediaIdleState(BridgedMedia *pBridgedMedia)
// Description: Initializing constructor
// Return:  BridgedMediaIdleState*
// Parameters: BridgedMedia *pBridgedMedia 
//*****************************************************************************
BridgedMediaIdleState::BridgedMediaIdleState(BridgedMedia* pBridgedMedia) :
   BridgedMediaState(pBridgedMedia)
{
}


//*****************************************************************************
// Function: BridgedMediaIdleState::~BridgedMediaIdleState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedMediaIdleState::~BridgedMediaIdleState()
{
}

//*****************************************************************************
// Function: void BridgedMediaIdleState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedMediaIdleState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      case APP_H245_START_REQUEST:
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaIdleState rcvd:APP_H245_START_REQUEST\n",m_pBridgedMedia->Index());
         break;
      case APP_MM_ENDPOINT_PLAYING:
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaIdleState rcvd:APP_MM_ENDPOINT_PLAYING\n",m_pBridgedMedia->Index());
	 ProcessMMPlaying(metaevent);
         break;
      case APP_MM_ENDPOINT_RECORDING:
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaIdleState rcvd:APP_MM_ENDPOINT_RECORDING\n",m_pBridgedMedia->Index());
	 ProcessMMRecording(metaevent);
         break;
      case APP_MM_ENDPOINT_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaIdleState rcvd:APP_MM_ENDPOINT_DISCONNECTED\n",m_pBridgedMedia->Index());
         break;
      case APP_H245_STARTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaIdleState rcvd:APP_H245_STARTED\n",m_pBridgedMedia->Index());
         SetStatus(H245_STARTED);
         ProcessEndpointConnected(metaevent);
         break;
      case APP_M3G_ENDPOINT_CONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaIdleState rcvd:APP_M3G_ENDPOINT_CONNECTED\n",m_pBridgedMedia->Index());
         SetStatus(M3G_CONNECTED | H245_STARTED);
         ProcessEndpointConnected(metaevent);
         break;
      case APP_M3G_ENDPOINT_DISCONNECT:
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaIdleState rcvd:APP_M3G_ENDPOINT_DISCONNECT\n",m_pBridgedMedia->Index());
         break;
      case APP_M3G_ENDPOINT_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaIdleState rcvd:APP_M3G_ENDPOINT_DISCONNECTED\n",m_pBridgedMedia->Index());
         break;
      default:
         LOG_ERROR(metaevent.evtdev,"BridgedMedia[%d] BridgedMediaIdleState: Unexpected event type: 0x%x\n",
		   m_pBridgedMedia->Index(),metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void BridgedMediaIdleState::ProcessMMPlaying(METAEVENT metaevent)
// Description: Process an MM busy  event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedMediaIdleState::ProcessMMPlaying(METAEVENT metaevent)
{
      // Advance BridgedMedia's state machine to PLAYING
      m_pBridgedMedia->ChangeState(PLAYING_STATE);
}

void BridgedMediaIdleState::ProcessMMRecording(METAEVENT metaevent)
{
      // Advance BridgedMedia's state machine to RECORDING
      m_pBridgedMedia->ChangeState(RECORDING_STATE);
}

//*****************************************************************************
// Function: void BridgedMediaIdleState::ProcessEndpointConnected(METAEVENT metaevent)
// Description: Process an MM busy  event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedMediaIdleState::ProcessEndpointConnected(METAEVENT metaevent)
{
   if (IsBridgedMediaReady())  {
      MMEndpoint *pMMEndpoint = EndpointMngr::Instance()->GetMMEPFromIdx( m_pBridgedMedia->MMIndex());
      Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedMedia->M3gIndex());
      if ( pMMEndpoint )   {
       
         pMMEndpoint->SetCoderSelection();
         LOG_ENTRY(0, "BridgedMedia[%d] LastDigitRcvd='%c'\n",
                   m_pBridgedMedia->Index(),
                   pMMEndpoint->GetLastDigitRcvd());
         LOG_ENTRY(0, "BridgedMedia[%d] MMSelectedAudioCoder=%s\n",
                   m_pBridgedMedia->Index(),
                   pMMEndpoint->GetSelAudCoderStr());
         LOG_ENTRY(0, "BridgedMedia[%d] MMSelectedVideoCoder=%s, MMSelectedVideoResolution=%s,\
                       MMSelectedVideoFps=%s, MMSelectedVideoBitRate=%dKbps\n",
                       m_pBridgedMedia->Index(), pMMEndpoint->GetSelVidCoderStr(), pMMEndpoint->GetSelVidResStr(), 
                       pMMEndpoint->GetSelVidFpsStr(), pMMEndpoint->GetSelectedVideoBitRate()/1000);
       
         // Start a play. Play terminates with a  single DTMF/UII or a disconnect.
         if (!strcmp(EndpointMngr::Instance()->GetMMOperation(pMMEndpoint->GetLastDigitRcvd()), "PLAY"))  {
            pMMEndpoint->PlayVideoFile(EndpointMngr::Instance()->GetMMPlayAudioFileName(pMMEndpoint->GetLastDigitRcvd()), EndpointMngr::Instance()->GetMMPlayVideoFileName(pMMEndpoint->GetLastDigitRcvd()));
            LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] Starting video play\n",m_pBridgedMedia->Index());
         } 
         else if (!strcmp(EndpointMngr::Instance()->GetMMOperation(pMMEndpoint->GetLastDigitRcvd()), "RECORD"))  {
            // Start a record.  The recording is terminated with a single DTMF/UII or a disconnect.
            // JM - Set the DCI for the Native Record case
            if(!pEndpoint->GetVidTranscodeEnabled()) 
            {
               pMMEndpoint->SetRecDCIStr(pEndpoint->Get3GEPDCI());
               pMMEndpoint->SetRecDCILen(strlen(pMMEndpoint->GetRecDCIStr())/2);
               LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] Rec DCI[size=%d]:%s\n",m_pBridgedMedia->Index(),pMMEndpoint->GetRecDCILen(),pMMEndpoint->GetRecDCIStr());
            }
            else {
               pMMEndpoint->SetRecDCILen(0);
            }

            pMMEndpoint->RecordVideoFile();
            pEndpoint->SendFastVideoUpdate();
            LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] Sending VFU and Starting video record\n",m_pBridgedMedia->Index());
         }
         else {
            LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] No MM action to take\n",m_pBridgedMedia->Index());
         } 
         //Set LastDigitRcvd to NULL to stop continuous play
         pMMEndpoint->SetLastDigitRcvd(char(NULL));
      }
   }
}

