/**
* @file h245inactive_state.cpp
* @brief Definition of H245InactiveState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "h245inactive_state.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"

const int H245InactiveState::AUDIO_PORT_DISCONNECTED       = 0x0001;
const int H245InactiveState::VIDEO_PORT_DISCONNECTED       = 0x0002;

//*****************************************************************************
// Function: H245InactiveState::H245InactiveState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  H245InactiveState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
H245InactiveState::H245InactiveState(Endpoint* pEndpoint) :
   EPState(pEndpoint)
{
   m_PortDiscState = 0;
}


//*****************************************************************************
// Function: H245InactiveState::~H245InactiveState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
H245InactiveState::~H245InactiveState()
{
}


//*****************************************************************************
// Function: void H245InactiveState::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process an event
// Return: none
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void H245InactiveState::ProcessEvent(long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   // Should not receive many events in this state...
  switch (evtType)
    {
      case M3GEV_RESET_CMPLT:
         ProcessResetComplete();
        break;
      case GCEV_SETCONFIGDATA:
         LOG_ENTRY(evtDev, "H245InactiveState: GCEV_SETCONFIGDATA received\n");
        break;
      case DMEV_PORT_DISCONNECT:
         LOG_ENTRY(evtDev, "H245InactiveState: DMEV_PORT_DISCONNECT received\n");
         ProcessPortDisconnect(evtDev);  
        break;
    default:
      LOG_ERROR(evtDev,"H245InactiveState: Unexpected event type: 0x%x\n", evtType);
      break;
    }
}

//*****************************************************************************
// Function: void H245InactiveState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process a request from the user interface
// Return: void 
// Parameters: EPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  H245InactiveState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   if ( EPState::USER_CONNECT_PROMPT == eUserPrompt )
   {
      LOG_ENTRY(0,"Endpoint[%d] Connect prompt received. Starting H245.\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->ChangeState(H245_STARTING_STATE);
      m_pEndpoint->StartH245();
   }
   else
   {
      LOG_ERROR(0,"Unexpected user prompt on endpoint[%d] in H245_INACTIVE state: %d\n",
                m_pEndpoint->GetIndex(), static_cast<int>(eUserPrompt));
   }
}


//*****************************************************************************
// Function: void H245InactiveState::Shutdown()
// Description: Process a shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void H245InactiveState::Shutdown()
{
  LOG_ENTRY(m_pEndpoint->GetControlHandle(), "Shutdown called in %s state\n", GetStateStr());
  if (m_pEndpoint->IsNbup())  {

     // Stop (synchronous) IPM media first
     m_pEndpoint->StopNbupMedia();

     // Now disconnect 3G and Nbup (also done sync)
     if (m_pEndpoint->IsNbupIPMPortConnected())
       m_pEndpoint->DisconnectIPMNbupFrom3GNbup(m_pEndpoint->GetNetworkHandle(),
                                                m_pEndpoint->GetIPMNbupTxPortInfo(),
                                                m_pEndpoint->Get3GNbupRxPortInfo());
     if (m_pEndpoint->IsNbup3GPortConnected())
       m_pEndpoint->DisconnectIPMNbupFrom3GNbup(m_pEndpoint->GetControlHandle(),
                                                m_pEndpoint->Get3GNbupTxPortInfo(),
                                                m_pEndpoint->GetIPMNbupRxPortInfo());
  } else {
     m_pEndpoint->DisconnectFromPeer();
  }
}

//*****************************************************************************
// Function: void H245InactiveState::ProcessResetComplete()
// Description: M3G reset done; set state to inactive to trigger shutdown
// Return: void
// Parameters: none
//*****************************************************************************
void H245InactiveState::ProcessResetComplete()
{
      LOG_ENTRY(0,"Endpoint[%d] M3g reset complete.  Triggering shutdown.\n",
                m_pEndpoint->GetIndex());
      LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECTED)\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECTED);
      m_pEndpoint->ChangeState(H245_INACTIVE_STATE); 
}

//*****************************************************************************
// Function: void H245InactiveState::ProcessPortDisconnect(int evtDev)
// Description: Process M3G PortDisconnects
// Return: void
// Parameters: evtDev
//*****************************************************************************
void H245InactiveState::ProcessPortDisconnect(int evtDev)
{
      if (m_PortDiscState && (AUDIO_PORT_DISCONNECTED | VIDEO_PORT_DISCONNECTED)) {
	 LOG_ENTRY(evtDev," Audio and Video Ports Disconnected\n");
	 m_pEndpoint->CloseSubDevs();
      } else {
       if (evtDev == m_pEndpoint->GetVideoHandle()) {
         m_PortDiscState |= VIDEO_PORT_DISCONNECTED;
	 LOG_ENTRY(evtDev," Video Port Disconnected\n");
       } else if (evtDev == m_pEndpoint->GetAudioHandle()) {
         m_PortDiscState |= AUDIO_PORT_DISCONNECTED;
	 LOG_ENTRY(evtDev," Audio Port Disconnected\n");
       }
      }
}
