/**
* @file isdncalling_state.cpp
* @brief Definition of ISDNIdleState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "isdncalling_state.h"
#include "isdnendpoint.h"
#include "endpointmgr.h"
#include "appevents.h"
#include "logger.h"

//*****************************************************************************
// Function: ISDNCallingState::ISDNCallingState(ISDNEndpoint *pISDNEndpoint)
// Description: Initializing constructor
// Return:  ISDNCallingState*
// Parameters: ISDNEndpoint *pISDNEndpoint 
//*****************************************************************************
ISDNCallingState::ISDNCallingState(ISDNEndpoint* pISDNEndpoint):
   ISDNEPState(pISDNEndpoint)
{
   m_pISDNEndpoint = pISDNEndpoint;
}


//*****************************************************************************
// Function: ISDNCallingState::~ISDNCallingState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
ISDNCallingState::~ISDNCallingState()
{
}

//*****************************************************************************
// Function: void ISDNCallingState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void ISDNCallingState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      // GC/R4 events
      case GCEV_DIALING:
         LOG_ENTRY(metaevent.evtdev, "ISDNCalling state: GCEV_DIALING received\n");
         break;
      case GCEV_PROCEEDING:
         LOG_ENTRY(metaevent.evtdev, "ISDNCalling state: GCEV_PROCEEDING received\n");
         ProcessProceeding(metaevent);
         break;
      case GCEV_ALERTING:
         LOG_ENTRY(metaevent.evtdev, "ISDNCalling state: GCEV_ALERTING received\n");
         break;
      case GCEV_CONNECTED:
         LOG_ENTRY(metaevent.evtdev, "ISDNCalling state: GCEV_CONNECTED received\n");
         ProcessConnected(metaevent);
         break;    
      case GCEV_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "ISDNCalling state: GCEV_DISCONNECTED received\n");
         ProcessDisconnect(metaevent);
         break;    
      case GCEV_BLOCKED:
         LOG_ERROR(metaevent.evtdev, "ISDNCalling state: GCEV_BLOCKED received - channel may not recover\n");
         break;
      case GCEV_UNBLOCKED:
         LOG_ERROR(metaevent.evtdev, "ISDNCalling state: GCEV_UNBLOCKED received - ignore\n");
         break;
      case GCEV_TASKFAIL:
         LOG_ENTRY(metaevent.evtdev, "ISDNCalling state: GCEV_TASKFAIL received\n");
         break;   
      default:
         LOG_ERROR(metaevent.evtdev,"ISDNCalling state: Unexpected event type: 0x%x\n", 
                   metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void ISDNCallingState::Shutdown()
// Description: Process shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void ISDNCallingState::Shutdown()
{
  LOG_ENTRY(m_pISDNEndpoint->GetGCISDNHandle(), "Shutdown called in %s state.  No Op\n", GetStateStr());
}

//*****************************************************************************
// Function: void ISDNCallingState::ProcessProceeding(METAEVENT metaevent)
// Description: Process call proceeding event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void ISDNCallingState::ProcessProceeding(METAEVENT metaevent)
{
   LOG_ENTRY(metaevent.evtdev,"ProcessProceeding\n");

   // Advance to proceeding state
   m_pISDNEndpoint->ChangeState(PROCEEDING_STATE);
}

//*****************************************************************************
// Function: void ISDNCallingState::ProcessConnected(METAEVENT metaevent)
// Description: Process connected event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void ISDNCallingState::ProcessConnected(METAEVENT metaevent)
{
   LOG_ENTRY(metaevent.evtdev,"ProcessConnected\n");

   //  Start H245 now that the call is up
   m_pISDNEndpoint->SendConnected(1);
   m_pISDNEndpoint->ChangeState(CONNECTED_STATE);
}

//*****************************************************************************
// Function: void ISDNCallingState::ProcessDisconnect(METAEVENT metaevent)
// Description: Process disconnect event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void ISDNCallingState::ProcessDisconnect(METAEVENT metaevent)
{
   LOG_ENTRY(metaevent.evtdev,"ProcessDisconnect\n");

   // Advance to dropping state
   m_pISDNEndpoint->ChangeState(DROPPING_STATE);
}


