/**
* @file openinglcs_state.cpp
* @brief Definition of OpeningLCsState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "openinglcs_state.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"


//*****************************************************************************
// Function: OpeningLCsState::OpeningLCsState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  OpeningLCsState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
OpeningLCsState::OpeningLCsState(Endpoint* pEndpoint) :
   EPState(pEndpoint)
{
}

//*****************************************************************************
// Function: OpeningLCsState::~OpeningLCsState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
OpeningLCsState::~OpeningLCsState()
{
}

//*****************************************************************************
// Function: void OpeningLCsState::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process an event
// Return: none
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void OpeningLCsState::ProcessEvent(long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   switch (evtType)
   {
      case M3GEV_SET_PARM_CMPLT:
         break;   // setting inband MPEG-4 DCI completion

      case M3GEV_OPEN_LC_CMPLT:
         {
            M3G_REMOTE_OLCACK_RESP* pOLCAck = reinterpret_cast<M3G_REMOTE_OLCACK_RESP*>(pEvtData);
            M3G_LOGICAL_CHANNEL_NUMBER lcn = pOLCAck->logicalChannelNumber;
            LOG_ENTRY(evtDev,"\t  version:%d\n", pOLCAck->version);
            LOG_ENTRY(evtDev,"\t  LCN:%d\n",lcn);

            if (M3G_E_AUDIO_CAPABILITY == pOLCAck->capabilityType)
            {
               LOG_ENTRY(evtDev,"\t  type:AUDIO\n\n");
               ActivateAudioChannel(M3G_E_TX, lcn);
            }
            else
            {
               LOG_ENTRY(evtDev,"\t  type:VIDEO\n\n");
               ActivateVideoChannel(M3G_E_TX, lcn);
            }
         }
         break;

      case M3GEV_OPEN_LC_FAIL:
         LOG_ERROR(evtDev, "OLCRequest failure\n");
         //Shutdown();
         break;

      case M3GEV_REMOTE_OLC_RCVD:
         {
            M3G_REMOTE_OLC_REQ* pOLCReq = reinterpret_cast<M3G_REMOTE_OLC_REQ*>(pEvtData);
            M3G_LOGICAL_CHANNEL_NUMBER lcn = pOLCReq->logicalChannelNumber;
            LOG_ENTRY(evtDev,"\t  version:%d\n", pOLCReq->version);
            LOG_ENTRY(evtDev,"\t  LCN:%d\n",lcn);
            
            m_pEndpoint->RecordLCMuxParams(evtDev, &pOLCReq->h223MultiplexParams);
            const char* tStr = (M3G_E_AUDIO_CAPABILITY == pOLCReq->capabilityType) ? " AUDIO" : " VIDEO";
            LOG_ENTRY(evtDev,"\t  type:%s\n",tStr);

            if (M3G_E_AUDIO_CAPABILITY == pOLCReq->capabilityType)
            {
               m_pEndpoint->RecordAudioCap(evtDev, &pOLCReq->mediaCapability.audioCapability);
               m_pEndpoint->SetAudioRxLCN(lcn);
            }
            else
            {
               // Get DCI from Remote Open Logical channel
               char tempDci[256], ch[4];
               int ctr;
               if (pOLCReq->mediaCapability.videoCapability.coderType == M3G_E_MPEG4)
               {
                   for(ctr=0;ctr<int(pOLCReq->mediaCapability.videoCapability.options.mpeg4.decoderConfigLength);ctr++)
                   {
                      sprintf(ch,"%.2x",pOLCReq->mediaCapability.videoCapability.options.mpeg4.decoderConfigInfo[ctr]);
                      if(ctr==0){
                        strcpy(tempDci,ch);
                      } else {
                        strcat(tempDci,ch);
                      }
                   }
                   LOG_ENTRY(evtDev,"\t  3GEP DCI size=%d | string=%s\n",ctr,tempDci);
                   m_pEndpoint->Store3GEPDCI(tempDci);
               }
               else if (pOLCReq->mediaCapability.videoCapability.coderType == M3G_E_H264)
               {
                   for(ctr=0;ctr<int(pOLCReq->mediaCapability.videoCapability.options.h264.decoderConfigLength);ctr++)
                   {
                      sprintf(ch,"%.2x",pOLCReq->mediaCapability.videoCapability.options.h264.decoderConfigInfo[ctr]);
                      if(ctr==0){
                        strcpy(tempDci,ch);
                      } else {
                        strcat(tempDci,ch);
                      }
                   }
                   LOG_ENTRY(evtDev,"\t  3GEP DCI size=%d | string=%s\n",ctr,tempDci);
                   m_pEndpoint->Store3GEPDCI(tempDci);
               }

               m_pEndpoint->RecordVideoCap(evtDev, &pOLCReq->mediaCapability.videoCapability);
               m_pEndpoint->SetVideoRxLCN(lcn);
            }
            m_pEndpoint->AckOLCReq(pOLCReq);
         }
         break;

      case M3GEV_RESPOND_TO_LC_CMPLT:
         {
            M3G_LOGICAL_CHANNEL_NUMBER* pLCN = reinterpret_cast<M3G_LOGICAL_CHANNEL_NUMBER*>(pEvtData);
            LOG_ENTRY(evtDev,"\t  LCN:%d\n",*pLCN);
            if (*pLCN == m_pEndpoint->GetAudioRxLCN())
            {
                ActivateAudioChannel(M3G_E_RX, *pLCN);
            }
            else // assume video
            {
                ActivateVideoChannel(M3G_E_RX, *pLCN);
            }
         }
         break;

      case M3GEV_REMOTE_CLOSE_LC_RCVD:
         {
            M3G_REMOTE_CLOSED_LC* pCLCInd = reinterpret_cast<M3G_REMOTE_CLOSED_LC*>(pEvtData);
            M3G_LOGICAL_CHANNEL_NUMBER lcn = pCLCInd->logicalChannelNumber;
            LOG_ENTRY(evtDev,"\t  version:%d\n", pCLCInd->version);
            LOG_ENTRY(evtDev,"\t  LCN:%d\n", lcn);
            LOG_ENTRY(evtDev,"\t  reason:%d\n", pCLCInd->reason);
            if (lcn == m_pEndpoint->GetAudioRxLCN())
            {
               ClearStatus(AUDIO_RX_LCN_OPEN);
               m_pEndpoint->SetAudioRxLCN(0);
            }
            else if (lcn == m_pEndpoint->GetAudioTxLCN())
            {
               ClearStatus(AUDIO_TX_LCN_OPEN);
               m_pEndpoint->SetAudioTxLCN(0);
            }
            else if (lcn == m_pEndpoint->GetVideoRxLCN())
            {
               ClearStatus(VIDEO_RX_LCN_OPEN);
               m_pEndpoint->SetVideoRxLCN(0);
            }
            else if (lcn == m_pEndpoint->GetVideoTxLCN())
            {
               ClearStatus(VIDEO_TX_LCN_OPEN);
               m_pEndpoint->SetVideoTxLCN(0);
            }
            // remain in this state assuming remote will soon send H.245 EndSession (handled below)
         }
         break;

      case M3GEV_RESPOND_TO_LC_FAIL:
         LOG_ERROR(evtDev, "Respond to OLC failure\n");
         //Shutdown();
         break;

      // FDX only media mode:
      case M3GEV_MODIFY_MEDIA_CMPLT:
         {
            const char *dirStr[] = 
            {
               "M3G_E_IDLE",
               "M3G_E_TX",
               "M3G_E_RX",
               "M3G_E_TXRX"
            };

            M3G_E_DIRECTION direction = *(reinterpret_cast<M3G_E_DIRECTION*>(pEvtData));
            LOG_ENTRY(evtDev,"\t  dir:%s\n",dirStr[direction]);

            // if audio transaction
            if (evtDev == m_pEndpoint->GetAudioHandle())
            {
               switch (direction)
               {
                  case M3G_E_TX:
                     SetStatus(AUDIO_TX_ACTIVE);
                     break;

                  case M3G_E_RX:
                     SetStatus(AUDIO_RX_ACTIVE);
                     break;

                  case M3G_E_TXRX:
                     SetStatus(AUDIO_TX_ACTIVE | AUDIO_RX_ACTIVE);
                     break;

                  default:   // should never happen
                     break;
               }
            }
            else // else video transaction
            {
               switch (direction)
               {
                  case M3G_E_TX:
                     SetStatus(VIDEO_TX_ACTIVE);
                     break;

                  case M3G_E_RX:
                     SetStatus(VIDEO_RX_ACTIVE);
                     break;

                  case M3G_E_TXRX:
                     SetStatus(VIDEO_TX_ACTIVE | VIDEO_RX_ACTIVE);
                     break;

                  default:   // should never happen
                     break;
               }
            }

            // if all media is active, transition to active state
            CheckAllChannelsActive();
            break;
         }

      case M3GEV_MODIFY_MEDIA_FAIL:
         LOG_ERROR(evtDev, "Start media failure\n");
         //Shutdown();
         break;

      // FDX only media mode:
      case M3GEV_START_MEDIA_CMPLT:
         // if audio transaction
         if (evtDev == m_pEndpoint->GetAudioHandle())
         {
             SetStatus(AUDIO_TX_ACTIVE | AUDIO_RX_ACTIVE);
         }
         else // else video transaction
         {
             SetStatus(VIDEO_TX_ACTIVE | VIDEO_RX_ACTIVE);
         }
         // if all media is active, transition to active state
         CheckAllChannelsActive();
        break;

      case M3GEV_START_MEDIA_FAIL:
         LOG_ERROR(evtDev, "Start media failure\n");
         //TODO:  Determine error handling, i.e. Shutdown();?
        break;

      case M3GEV_FRAMING_LOST:
         LOG_ERROR(evtDev, "H.223 framing failure\n");
         CloseLCs();
         break;

      case M3GEV_ENDSESSION_RCVD:
         LOG_ERROR(evtDev, "H.245 EndSession rcvd\n");
         CloseLCs();
         break;

      case M3GEV_H245_MES_EVT: 
         { 
            M3G_MES* pMesInd = reinterpret_cast<M3G_MES *>(pEvtData);
            LOG_ENTRY(evtDev,"\t  MESEvtType: %d\n", pMesInd->MESEvtType);
            LOG_ENTRY(evtDev,"\t  MESRejectCause: %d\n", pMesInd->MESRejectCause);
            break; 
         }

      case M3GEV_RESET_CMPLT:
         LOG_ENTRY(evtDev, "M3GEV_RESET_CMPLT received\n");
         ProcessResetComplete();
        break;

#ifdef MONA
      // this may be received late and a second time according to RadVision
      case M3GEV_SEND_MONA_PREF_MSG:
         {
            M3G_MONA_TXRX_MPC_SUPPORT* pMPC = reinterpret_cast<M3G_MONA_TXRX_MPC_SUPPORT*>(pEvtData); 
            LOG_ENTRY(evtDev,"\t rxMPC:0x%x txMPC:0x%x\n", pMPC->rxMPCMask, pMPC->txMPCMask);
            break;
         }

      case M3GEV_CALL_STATISTICS:
         m_pEndpoint->RecordStatistics(evtDev, reinterpret_cast<M3G_CALL_STATISTICS*>(pEvtData));
         break;
#endif

      default:
         LOG_ERROR(evtDev,"OpeningLCsState: Unexpected event type: 0x%x\n", evtType);
         break;
   }
}

//*****************************************************************************
// Function: void OpeningLCsState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process request from user interface
// Return: void 
// Parameters: EPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  OpeningLCsState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   if ( EPState::USER_DISCONNECT_PROMPT == eUserPrompt )
   {
      CloseLCs();
   }
   else
   {
      LOG_ERROR(0,"Unexpected user prompt on endpoint[%d] in OPENING_LCS state: %d\n",
                m_pEndpoint->GetIndex(), static_cast<int>(eUserPrompt));
   }
}


//*****************************************************************************
// Function: void OpeningLCsState::CloseLCs()
// Description: Close logical channels
// Return: void 
// Parameters: none 
//*****************************************************************************
void OpeningLCsState::CloseLCs()
{
   if (IsAudioStarted())
   {
      m_pEndpoint->StopAudio();
   }
   if (IsVideoStarted())
   {
      m_pEndpoint->StopVideo();
   }
   m_pEndpoint->CloseLCs();
   m_pEndpoint->ChangeState(CLOSING_LCS_STATE);
}


//*****************************************************************************
// Function: void OpeningLCsState::Shutdown()
// Description: Process shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void OpeningLCsState::Shutdown()
{
  LOG_ENTRY(m_pEndpoint->GetControlHandle(), "Shutdown called in %s state.  Resetting M3g\n", GetStateStr());
  m_pEndpoint->Reinitialize();
  m_pEndpoint->StopNbupMedia();
}


//*****************************************************************************
// Function: void OpeningiLCsState::ProcessResetComplete()
// Description: M3G reset done; set state to inactive to trigger shutdown 
// Return: void
// Parameters: none
//*****************************************************************************
void OpeningLCsState::ProcessResetComplete()
{
      LOG_ENTRY(0,"Endpoint[%d] M3g reset complete.  Triggering shutdown.\n",
                m_pEndpoint->GetIndex());
      LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECTED)\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECTED);
      m_pEndpoint->ChangeState(H245_INACTIVE_STATE);
}

