/**
* @file mmendpoint.cpp
* @brief Definition of RTSPEndpoint class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include <time.h>
#include "logger.h"
#include "rtspendpoint.h"
#include "endpointmgr.h"
#include "locker.h"
#include "appevents.h"
#include "endpoint.h"

RTSP_URLS RTSPEndpoint::m_rtspUrls;

extern unsigned char AsciiOctetToHex(const char *pOctet);

//*****************************************************************************
// Function: RTSPEndpoint::RTSPEndpoint(int numEP, const char *RTSPName, const char *RTSPName, const char *mapID)
// Description: Initializing constructor
// Return:  RTSPEndpoint*
// Parameters: int numEP 
//             const char *RTSPName 
//             const char *RTSPName 
//             const char *mapID 
//*****************************************************************************
RTSPEndpoint::RTSPEndpoint(int numEP, const char *RTSPName, const char *mapID):
   m_Index(numEP),
   m_ConnectionTime(0),
   m_pCurrentState(NULL),
   m_IPMHandle(-1),
   m_pRTSPClient(NULL)
{
   SetEndpointType(RTSP_ENDPOINT);
   strcpy(m_IPMName, RTSPName);
   strcpy(m_mapID, mapID);
   strncpy(m_CurrentStateStr, "OPENING        ", MAX_CURRENT_STATE_STR);
   strcpy(m_LastMessageBuff, "");

   m_DciOctetStrSize = 0;
   memset(m_DciOctetStr, 0, OCTET_STRING_SIZE);

   m_pRTSPOpeningState = new RTSPOpeningState(this);
   m_pRTSPPortConnectingState = new RTSPPortConnectingState(this);
   m_pRTSPIdleState = new RTSPIdleState(this);
   m_pRTSPPlayingState = new RTSPPlayingState(this);
   m_pRTSPRecordingState = new RTSPRecordingState(this);
}

//*****************************************************************************
// Function: RTSPEndpoint::~RTSPEndpoint()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
RTSPEndpoint::~RTSPEndpoint()
{
  LOG_ENTRY(m_IPMHandle, "RTSPEndpoint Destructor called\n");
  delete m_pRTSPOpeningState;
  delete m_pRTSPPortConnectingState;
  delete m_pRTSPIdleState;
  delete m_pRTSPPlayingState;
  delete m_pRTSPRecordingState;
}

void RTSPEndpoint::QueueNextVideo(char dtmf)
{
   m_qDtmf.push(dtmf);
}

void RTSPEndpoint::RTSPServerPlay()
{
   if (m_pRTSPClient)
   {
      LOG_ENTRY(m_IPMHandle, "RTSPEndpoint::RTSPServerPlay() - Requesting RTSP Server to start streaming\n");
      m_pRTSPClient->Play();
   }
}

void RTSPEndpoint::SetDCI(const char* szDCI)
{
   std::string DCI(szDCI);
   m_DciOctetStrSize = DCI.length()/2;

   memset(m_DciOctetStr, 0, OCTET_STRING_SIZE);
   if (!m_DciOctetStrSize)
      return;

   unsigned char dciOctetStr[OCTET_STRING_SIZE] = {0};
   for (unsigned int i = 0; i < m_DciOctetStrSize; i++)
   {
     dciOctetStr[i] = AsciiOctetToHex(&DCI[2*i]);
   }
   memcpy(m_DciOctetStr, dciOctetStr, m_DciOctetStrSize);
}

void RTSPEndpoint::PlayVideoFile(char dtmf)
{
  std::string url ;
  std::string srvrAddr;

  LOG_ENTRY(m_IPMHandle, "RTSPEndpoint::PlayVideoFile(). dtmf = '%c'\n", dtmf);
  if (m_pRTSPClient == NULL) {
     GetUrl(dtmf, url);
     LOG_ENTRY(m_IPMHandle, "RTSP URL: '%s'\n", url.c_str());
     //char RtspStr[256];
     ////sprintf(RtspStr,"%s",url.c_str());
     //SetLastMessageString(RtspStr);
     SetLastMessageString(url.c_str());
     if (!url.empty()) {
       m_pRTSPClient = new CRTSPClient( m_localAudioRTPPort, m_localVideoRTPPort, this );
       if (m_pRTSPClient->CreateRTSPSession(url.c_str(), false) == -1) {
          LOG_ENTRY(m_IPMHandle, "RTSPEndpoint::PlayVideoFile(). Failed to create RTSP client session\n");
          SetLastMessageString("RTSP URL Error - check DTMF/URL pair in config");
          delete m_pRTSPClient;
          m_pRTSPClient = NULL;
          return;
       }
       m_remoteAudioCoderPayloadType = m_pRTSPClient->GetAudioPayloadType();
       m_remoteVideoCoderPayloadType = m_pRTSPClient->GetVideoPayloadType();
       m_localAudioCoderPayloadType = m_remoteAudioCoderPayloadType;
       m_localVideoCoderPayloadType = m_remoteVideoCoderPayloadType;

       if ( strncmp(m_pRTSPClient->GetAudioCodecName(), "AMR", 3)==0)
       {
          SetAudioToAMR();
       }
       else {
          SetAudioToG711U();
       }
       unsigned profile = m_pRTSPClient->GetRTSPVideoProfile();
       if ( strncmp(m_pRTSPClient->GetVideoCodecName(), "MP4V", 4)==0) {
          //SetDCI(m_pRTSPClient->decoder_config_info());
          m_remoteVideoCoderType = CODER_TYPE_MP4V_ES;
          if (profile == 3) {
             SetVideoToMPEG4(false);
             m_remoteImageProfile = VIDEO_PROFILE_LEVEL_SP3_MPEG4;
             m_remoteImageLevel = VIDEO_LEVEL_DEFAULT;
             m_remoteImageWidth = VIDEO_IMAGE_WIDTH_352;
             m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_288;
	     m_remoteBitRate = 384000;
	     m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
          } else {
             SetVideoToMPEG4(true);
             m_remoteImageProfile = VIDEO_PROFILE_LEVEL_SP0_MPEG4;
             m_remoteImageLevel = VIDEO_LEVEL_DEFAULT;
             m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
             m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
	     m_remoteBitRate = 384000;
	     m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
          }
       } else if (strncmp(m_pRTSPClient->GetVideoCodecName(), "H264", 4)==0) {
	          m_remoteVideoCoderType = CODER_TYPE_H264;
             unsigned level = profile & 0xFF; // if level >= 1.2, we may assume CIF
             LOG_ENTRY(m_IPMHandle, "Remote H264 level = %d\n", level);
             if (level < 12)  
             {
                SetVideoToH264(true);
                m_remoteImageProfile = VIDEO_PROFILE_BASELINE_H264;
                m_remoteImageLevel = EMM_VIDEO_LEVEL_DEFAULT;
                m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
                m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
	             m_remoteBitRate = 384000;
	             m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
             }
             else
             {
                SetVideoToH264(false);
                m_remoteImageProfile = VIDEO_PROFILE_BASELINE_H264;
                m_remoteImageLevel = EMM_VIDEO_LEVEL_DEFAULT;
                m_remoteImageWidth = VIDEO_IMAGE_WIDTH_352;
                m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_288;
	             m_remoteBitRate = 384000;
	             m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
             }
       } else if (strncmp(m_pRTSPClient->GetVideoCodecName(), "H263-1998", 9)==0) {
	 m_remoteVideoCoderType = CODER_TYPE_H263_1998;
         if (profile == 3) {
             SetVideoToH263_1998(false);
             m_remoteImageProfile = VIDEO_PROFILE_0_H263;
             m_remoteImageLevel = EMM_VIDEO_LEVEL_DEFAULT;
             m_remoteImageWidth = VIDEO_IMAGE_WIDTH_352;
             m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_288;
	     m_remoteBitRate = 384000;
	     m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
          } else {
             SetVideoToH263_1998(true);
             m_remoteImageProfile = VIDEO_PROFILE_0_H263;
             m_remoteImageLevel = EMM_VIDEO_LEVEL_DEFAULT;
             m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
             m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
	     m_remoteBitRate = 384000;
	     m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
          }
       } else if (strncmp(m_pRTSPClient->GetVideoCodecName(), "H263-2000", 9)==0) {
	 m_remoteVideoCoderType = CODER_TYPE_H263_1998;
         if (profile == 3) {
             SetVideoToH263_1998(false);
             m_remoteImageProfile = VIDEO_PROFILE_0_H263;
             m_remoteImageLevel = EMM_VIDEO_LEVEL_DEFAULT;
             m_remoteImageWidth = VIDEO_IMAGE_WIDTH_352;
             m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_288;
	     m_remoteBitRate = 384000;
	     m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
          } else {
             SetVideoToH263_1998(true);
             m_remoteImageProfile = VIDEO_PROFILE_0_H263;
             m_remoteImageLevel = EMM_VIDEO_LEVEL_DEFAULT;
             m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
             m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
	     m_remoteBitRate = 384000;
	     m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
          }
       } else if (strncmp(m_pRTSPClient->GetVideoCodecName(), "H263", 4)==0) {
	 m_remoteVideoCoderType = CODER_TYPE_H263_1998;
         if (profile == 3) {
             SetVideoToH263(false);
             m_remoteImageProfile = VIDEO_PROFILE_0_H263;
             m_remoteImageLevel = EMM_VIDEO_LEVEL_DEFAULT;
             m_remoteImageWidth = VIDEO_IMAGE_WIDTH_352;
             m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_288;
	     m_remoteBitRate = 384000;
	     m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
          } else {
             SetVideoToH263(true);
             m_remoteImageProfile = VIDEO_PROFILE_0_H263;
             m_remoteImageLevel = EMM_VIDEO_LEVEL_DEFAULT;
             m_remoteImageWidth = VIDEO_IMAGE_WIDTH_176;
             m_remoteImageHeight = VIDEO_IMAGE_HEIGHT_144;
	     m_remoteBitRate = 384000;
	     m_remoteSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
          }
       }

       else {
          LOG_ENTRY(m_IPMHandle, "RTSPEndpoint::PlayVideoFile(). Video is not MPEG4 or H263 or H264. Not playing.\n", dtmf);
          return;
       }
       m_remoteAudioRTPPort = m_pRTSPClient->GetRTSPAudioPort();
       m_remoteVideoRTPPort = m_pRTSPClient->GetRTSPVideoPort();
       strcpy(m_remoteVideoIPAddress, m_pRTSPClient->GetRTSPVideoIP());
       strcpy(m_remoteAudioIPAddress, m_pRTSPClient->GetRTSPAudioIP());
       if (StartMedia() != -1) {
         //m_pRTSPClient->Play(); // Actualy RTSP streaming started on IPMEV_STARTMEDIA event
       }
     }
     else {
       LOG_ERROR(m_IPMHandle, "RTSPEndpoint::PlayVideoFile(). dtmf = '%c', no URL in config for that DTMF\n", dtmf);
       SetLastMessageString("No URL in config for that DTMF");
     }
  }
}

int RTSPEndpoint::StartMedia()
{
#define IPM_RX_TX  1

   IPM_VIDEO_CODER_INFO_EX l_ExtraVideoCoderInfoEx;

   // Set up and get audio and video RTP streams going via ipm_startmedia
   LOG_ENTRY(m_IPMHandle, "Priming StartMedia:RTP streams starting\n");
   LOG_ENTRY(m_IPMHandle, 
             "Remote RTP audio port = %d  Remote RTP video port = %d Remote Audio IP address = %s Remote Video IP address = %s\n",
             m_remoteAudioRTPPort, m_remoteVideoRTPPort, m_remoteAudioIPAddress,
             m_remoteVideoIPAddress);

   // clear the MediaInfo structures
   memset(&m_mediaInfo, 0, sizeof(IPM_MEDIA_INFO));

   int miCnt = 0;

   // remote audio RTP port
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_REMOTE_RTP_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.unPortId = m_remoteAudioRTPPort;
   strcpy(m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.cIPAddress, m_remoteAudioIPAddress);
   miCnt++;

   // remote audio RTCP port
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_REMOTE_RTCP_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.unPortId = m_remoteAudioRTPPort + 1;
   strcpy(m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.cIPAddress, m_remoteAudioIPAddress);
   miCnt++;

   // remote video RTP port
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_VIDEO_REMOTE_RTP_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.unPortId = m_remoteVideoRTPPort;
   strcpy(m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.cIPAddress, m_remoteVideoIPAddress);
   miCnt++;

   // remote video RTCP port
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_VIDEO_REMOTE_RTCP_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.unPortId = m_remoteVideoRTPPort + 1; 
   strcpy(m_mediaInfo.MediaData[miCnt].mediaInfo.PortInfo.cIPAddress, m_remoteVideoIPAddress);
   miCnt++;

#if IPM_RX_TX
   // remote audio codec
   LOG_ENTRY(m_IPMHandle, "AUDIO_REMOTE_CODER_INFO[%d]:\n",miCnt+1);
   LOG_ENTRY(m_IPMHandle, "\tremCoderType=%d, remCoderPayload=%d, remRedPayload=%d\n", 
                m_remoteAudioCoderType, m_remoteAudioCoderPayloadType, m_remoteAudioRedPayloadType);
   LOG_ENTRY(m_IPMHandle, "\tremFrameSize=%d, remFPP=%d, remVadEnable=%d\n", 
                m_remoteAudioCoderFramesize, m_remoteAudioFramesPerPkt, m_remoteAudioVadEnable);
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_REMOTE_CODER_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.CoderInfo.eCoderType = m_remoteAudioCoderType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.eFrameSize = m_remoteAudioCoderFramesize;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unFramesPerPkt = m_remoteAudioFramesPerPkt;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.eVadEnable = m_remoteAudioVadEnable;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unCoderPayloadType = m_remoteAudioCoderPayloadType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unRedPayloadType = m_remoteAudioRedPayloadType;   
   miCnt++;

   if((m_remoteAudioCoderType&0xFF) == CODER_TYPE_AMRNB_12_2k){
      LOG_ENTRY(m_IPMHandle, "AUDIO_REMOTE_CODER_OPTIONS[%d]:\n",miCnt+1);
      LOG_ENTRY(m_IPMHandle, "\tlocCoderType=CODER_TYPE_AMRNB_12_2k, locCoderOptions=CODER_OPT_AMR_OCTET\n");
      m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_REMOTE_CODER_OPTIONS_INFO;
      m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderOptionsInfo.unVersion =
        IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderOptionsInfo.unCoderOptions=
        CODER_OPT_AMR_CMR_TRACK | CODER_OPT_AMR_OCTET; //CODER_OPT_AMR_EFFICIENT;
      miCnt++;
   }
#endif

   // local audio codec
   LOG_ENTRY(m_IPMHandle, "AUDIO_LOCAL_CODER_INFO[%d]:\n",miCnt+1);
   LOG_ENTRY(m_IPMHandle, "\tlocCoderType=%d, locCoderPayload=%d, locRedPayload=%d\n", 
                m_localAudioCoderType, m_localAudioCoderPayloadType, m_localAudioRedPayloadType);
   LOG_ENTRY(m_IPMHandle, "\tlocFrameSize=%d, locFPP=%d, locVadEnable=%d\n", 
                m_localAudioCoderFramesize, m_localAudioFramesPerPkt, m_localAudioVadEnable);
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_INFO;
   m_mediaInfo.MediaData[miCnt].mediaInfo.CoderInfo.eCoderType = m_localAudioCoderType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.eFrameSize = m_localAudioCoderFramesize;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unFramesPerPkt = m_localAudioFramesPerPkt;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.eVadEnable = m_localAudioVadEnable;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unCoderPayloadType = m_localAudioCoderPayloadType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderInfo.unRedPayloadType = m_localAudioRedPayloadType;   
   miCnt++;

   if((m_localAudioCoderType&0xFF) == CODER_TYPE_AMRNB_12_2k){
      LOG_ENTRY(m_IPMHandle, "AUDIO_LOCAL_CODER_OPTIONS[%d]:\n",miCnt+1);
      LOG_ENTRY(m_IPMHandle, "\tlocCoderType=CODER_TYPE_AMRNB_12_2k, locCoderOptions=CODER_OPT_AMR_OCTET\n");
      m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_OPTIONS_INFO;
      m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderOptionsInfo.unVersion =
        IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_mediaInfo.MediaData[miCnt].mediaInfo.AudioCoderOptionsInfo.unCoderOptions=
        CODER_OPT_AMR_OCTET;
      miCnt++;
   }

#if IPM_RX_TX
   // remote video codec
   LOG_ENTRY(m_IPMHandle, "VIDEO_REMOTE_CODER_INFO[%d]:\n",miCnt+1);
   LOG_ENTRY(m_IPMHandle, "\tremCoderType=0x%x, remCoderPayload=0x%x, remImageProfile=0x%x\n", 
                m_remoteVideoCoderType, m_remoteVideoCoderPayloadType, m_remoteImageProfile);
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_VIDEO_REMOTE_CODER_INFO;
   INIT_IPM_VIDEO_CODER_INFO(&m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo);
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.eCoderType = m_remoteVideoCoderType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.unCoderPayloadType = m_remoteVideoCoderPayloadType;
   
   INIT_IPM_VIDEO_CODER_INFO_EX(&l_ExtraVideoCoderInfoEx);
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &l_ExtraVideoCoderInfoEx;
   l_ExtraVideoCoderInfoEx.eProfile = m_remoteImageProfile;
   l_ExtraVideoCoderInfoEx.eLevel = m_remoteImageLevel;
   l_ExtraVideoCoderInfoEx.eImageWidth = m_remoteImageWidth;
   l_ExtraVideoCoderInfoEx.eImageHeight = m_remoteImageHeight;
   l_ExtraVideoCoderInfoEx.eFramesPerSec = m_remoteFramesPerSec;
   l_ExtraVideoCoderInfoEx.unBitRate = m_remoteBitRate;
   l_ExtraVideoCoderInfoEx.eSamplingRate = m_remoteSamplingRate;
   l_ExtraVideoCoderInfoEx.unVisualConfigSize = 0;
   l_ExtraVideoCoderInfoEx.szVisualConfiguration = NULL;
   miCnt++;
#endif

   // local video codec
   LOG_ENTRY(m_IPMHandle, "VIDEO_LOCAL_CODER_INFO[%d]:\n",miCnt+1);
   LOG_ENTRY(m_IPMHandle, "\tlocCoderType=0x%x, locCoderPayload=0x%x, locImageProfile=0x%x\n", 
                m_localVideoCoderType, m_localVideoCoderPayloadType, m_localImageProfile);
   m_mediaInfo.MediaData[miCnt].eMediaType = MEDIATYPE_VIDEO_LOCAL_CODER_INFO;
   INIT_IPM_VIDEO_CODER_INFO(&m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo);
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.eCoderType = m_localVideoCoderType;
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.unCoderPayloadType = m_localVideoCoderPayloadType;
   
   INIT_IPM_VIDEO_CODER_INFO_EX(&l_ExtraVideoCoderInfoEx);
   m_mediaInfo.MediaData[miCnt].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &l_ExtraVideoCoderInfoEx;
   l_ExtraVideoCoderInfoEx.eProfile = m_localImageProfile;
   l_ExtraVideoCoderInfoEx.eLevel = m_localImageLevel;
   l_ExtraVideoCoderInfoEx.eImageWidth = m_localImageWidth;
   l_ExtraVideoCoderInfoEx.eImageHeight = m_localImageHeight;
   l_ExtraVideoCoderInfoEx.eFramesPerSec = m_localFramesPerSec;
   l_ExtraVideoCoderInfoEx.unBitRate = m_localBitRate;
   l_ExtraVideoCoderInfoEx.eSamplingRate = m_localSamplingRate;
   if (m_localVideoCoderType == CODER_TYPE_MP4V_ES) {
      l_ExtraVideoCoderInfoEx.unVisualConfigSize = m_DciOctetStrSize;
      l_ExtraVideoCoderInfoEx.szVisualConfiguration = m_DciOctetStr;
   } 
   else {
      l_ExtraVideoCoderInfoEx.unVisualConfigSize = 0;
      l_ExtraVideoCoderInfoEx.szVisualConfiguration = NULL;
   }
   miCnt++;

   // Must correspond to number of elements set
   m_mediaInfo.unCount = miCnt;

   LOG_ENTRY(m_IPMHandle, "IP_MEDIA_INFO contains %d elements\n", miCnt);

#if IPM_RX_TX
   if ( ipm_StartMedia(m_IPMHandle, &m_mediaInfo, DATA_IP_TDM_BIDIRECTIONAL, EV_ASYNC) != 0 )
#else
   if ( ipm_StartMedia(m_IPMHandle, &m_mediaInfo, DATA_IP_RECEIVEONLY, EV_ASYNC) != 0 )
#endif
   {
      LOG_ERROR(m_IPMHandle, "ipm_StartMedia() failed - %s\n", ATDV_ERRMSGP(m_IPMHandle));
      return -1;
   }
   else
   {
      LOG_ENTRY(m_IPMHandle, "RTP streams started\n");
   }
   return 0;
}

//*****************************************************************************
// Function: void RTSPEndpoint::StopMedia()
// Description: Stop the media stream
// Return: void 
// Parameters: none 
//*****************************************************************************
void RTSPEndpoint::StopMedia()
{
   LOG_ENTRY(m_IPMHandle, "Stoping RTP streams for %s\n", m_IPMName);
   if ( ipm_Stop(m_IPMHandle, STOP_ALL, EV_ASYNC) != 0 )
   {
      LOG_ERROR(m_IPMHandle, "ipm_Stop() failed - %s\n", ATDV_ERRMSGP(m_IPMHandle));
   }
}

void RTSPEndpoint::CheckOperationQueue()
{
   if (!m_qDtmf.empty()) {
      char dtmf = m_qDtmf.front();
      m_qDtmf.pop();
      PlayVideoFile(dtmf);
   }
}

void RTSPEndpoint::RecordVideoFile()
{
  LOG_ENTRY(m_IPMHandle, "RTSPEndpoint::RecordVideoFile(). TBD\n");
}

void RTSPEndpoint::ProcessH245UII(char UII)
{
  // A single DTMF/UII digit has arrive on the peer 3G endpoint.  Insert it into a fake
  // metaevent and dispatch it.
  METAEVENT uiiMetaEvent;
  static char charBuffer;

  charBuffer = UII;
  memset (&uiiMetaEvent, 0, sizeof(METAEVENT));
  uiiMetaEvent.evtdev = m_IPMHandle;
  uiiMetaEvent.evttype = M3GEV_H245_UII_RCVD;
  uiiMetaEvent.evtdatap = &charBuffer;

  ProcessEvent(uiiMetaEvent);
}

void RTSPEndpoint::StopRTSPPlay()
{
    LOG_ENTRY(m_IPMHandle, "Stopping RTSP Play. \n");
    if (m_pRTSPClient) {
        m_pRTSPClient->Teardown();
        delete m_pRTSPClient;
        m_pRTSPClient = NULL;
    }
    StopMedia();
}

void RTSPEndpoint::StopRTSPRecord()
{
    LOG_ENTRY(m_IPMHandle, "Stopping RTSP Record. TBD\n");
}

//*****************************************************************************
// Function: void RTSPEndpoint::ChangeState(RTSPEPState::E_RTSPEPSTATE e_NewState)
// Description: Change the object state
// Return: void 
// Parameters: RTSPEPState::E_RTSPEPSTATE e_NewState 
//*****************************************************************************
void RTSPEndpoint::ChangeState(RTSPEPState::E_RTSPEPSTATE e_NewState)
{
   char oldStateStr[MAX_CURRENT_STATE_STR];
   strcpy(oldStateStr, m_pCurrentState->GetStateStr());

   // reset the current state for the next call:
   m_pCurrentState->Reset();

   switch ( e_NewState )
   {
      case RTSPEPState::OPENING_STATE:
         m_pCurrentState = m_pRTSPOpeningState;
         break;

      case RTSPEPState::PORTCONNECTING_STATE:
         m_pCurrentState = m_pRTSPPortConnectingState;
         break;

      case RTSPEPState::IDLE_STATE:
         CheckOperationQueue();
         m_pCurrentState = m_pRTSPIdleState;
         break;

      case RTSPEPState::PLAYING_STATE:
         m_pCurrentState = m_pRTSPPlayingState;
         break;

      case RTSPEPState::RECORDING_STATE:
         m_pCurrentState = m_pRTSPRecordingState;
         break;

      case RTSPEPState::CLOSED_STATE:
         LOG_ENTRY(m_IPMHandle, "RTSPEndpoint[%d] ready for device closure\n", GetIndex());
         break;
      default:
         LOG_ERROR(0,"RTSPEndpoint: Unexpected state: %d\n",static_cast<int>(e_NewState));
         break;
   }

   // cache the state string in the endpoint
   strcpy(m_CurrentStateStr,m_pCurrentState->GetStateStr());
   LOG_ENTRY(m_IPMHandle, "RTSPEndpoint[%d] State transition: %s ---> %s\n",
		GetIndex(), oldStateStr, m_CurrentStateStr);
}

//*****************************************************************************
// Function: void RTSPEndpoint::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void RTSPEndpoint::ProcessEvent(METAEVENT metaevent)
{
   m_StateLock.Lock();
   // dispatch event information to appropriate state:
   m_pCurrentState->ProcessEvent(metaevent);
   m_StateLock.Unlock();
}

//*****************************************************************************
// Function: void RTSPEndpoint::OpenSubDevs()
// Description: Open all associated devices
// Return: void 
// Parameters: none 
//*****************************************************************************
void RTSPEndpoint::OpenSubDevs()
{
  // initialize the endpoint state to opening
  m_pCurrentState = m_pRTSPOpeningState;
  
  // Open the RTSP device
  // Note that opening is async only - sync not possible.
  LOG_ENTRY(0, "Opening %s\n", m_IPMName);
  if ((m_IPMHandle = ipm_Open(m_IPMName, NULL, EV_ASYNC)) < 0) {
    LOG_ERROR(0, "ipm_Open(devicename=%s) Failed", m_IPMName);
  } else {
    LOG_ENTRY(m_IPMHandle, "%s successfully opened\n", m_IPMName);
  }
}

//*****************************************************************************
// Function: void RTSPEndpoint::GetIPMPortInfo()
// Description: Get RTSP port information
// Return: void 
// Parameters: none 
//*****************************************************************************
void RTSPEndpoint::GetIPMPortInfo ()
{
   // Info arrives with successful completion event on async call
   if ( dev_GetTransmitPortInfo(m_IPMHandle, this) != DEV_SUCCESS )
   {
      LOG_ERROR(m_IPMHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_IPMHandle,"Successful dev_GetTransmitPortInfo() call\n");
   }

   if ( dev_GetReceivePortInfo(m_IPMHandle, this) != DEV_SUCCESS )
   {
      LOG_ERROR(m_IPMHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_IPMHandle,"Successful dev_GetReceivePortInfo() call\n");
   }
   GetLocalMediaInfo();
}

void RTSPEndpoint::GetLocalMediaInfo()
{
   IPM_MEDIA_INFO LocalMediaInfo;

   LOG_ENTRY(m_IPMHandle, "Getting Local Media info\n");

   LocalMediaInfo.unCount = 2;
   LocalMediaInfo.MediaData[0].eMediaType = MEDIATYPE_VIDEO_LOCAL_RTP_INFO;
   LocalMediaInfo.MediaData[1].eMediaType = MEDIATYPE_AUDIO_LOCAL_RTP_INFO;

   if ( ipm_GetLocalMediaInfo(m_IPMHandle, &LocalMediaInfo, EV_ASYNC) == -1 )
      LOG_ENTRY(m_IPMHandle, "ipm_GetLocalLocalMediaInfo failed - %s with error = %ld\n",
                ATDV_NAMEP(m_IPMHandle), ATDV_LASTERR(m_IPMHandle));
   else
      LOG_ENTRY(m_IPMHandle, "ipm_GetLocalLocalMediaInfo started.\n");
}

void RTSPEndpoint::RecordLocalMediaInfo(METAEVENT metaevent, IPM_MEDIA_INFO* pLocalMediaInfo)
{
      for ( unsigned int i=0; i<pLocalMediaInfo->unCount; i++ )
      {
         switch ( pLocalMediaInfo->MediaData[i].eMediaType )
         {
            case MEDIATYPE_VIDEO_LOCAL_RTP_INFO:
               LOG_ENTRY(m_IPMHandle, "MediaType=MEDIATYPE_VIDEO_LOCAL_RTP_INFO\n");
               LOG_ENTRY(m_IPMHandle, "PortId=%d\n", pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId);
               LOG_ENTRY(m_IPMHandle, "IP=%s\n", pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               m_localVideoRTPPort = pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId;
               strcpy(m_localVideoRTPIPAddress, pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               break;
            case MEDIATYPE_VIDEO_LOCAL_RTCP_INFO:
               LOG_ENTRY(m_IPMHandle, "MediaType=MEDIATYPE_VIDEO_LOCAL_RTCP_INFO\n");
               LOG_ENTRY(m_IPMHandle, "PortId=%d\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId);
               LOG_ENTRY(m_IPMHandle, "IP=%s\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               m_localVideoRTCPPort = pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId;
               strcpy(m_localVideoRTCPIPAddress, pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               break;
            case MEDIATYPE_AUDIO_LOCAL_RTP_INFO:
               LOG_ENTRY(m_IPMHandle, "MediaType=MEDIATYPE_AUDIO_LOCAL_RTP_INFO\n");
               LOG_ENTRY(m_IPMHandle, "PortId=%d\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId);
               LOG_ENTRY(m_IPMHandle, "IP=%s\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               m_localAudioRTPPort = pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId;
               strcpy(m_localAudioRTPIPAddress, pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               break;
            case MEDIATYPE_AUDIO_LOCAL_RTCP_INFO:
               LOG_ENTRY(m_IPMHandle, "MediaType=MEDIATYPE_AUDIO_LOCAL_RTCP_INFO\n");
               LOG_ENTRY(m_IPMHandle, "PortId=%d\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId);
               LOG_ENTRY(m_IPMHandle, "IP=%s\n",pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               m_localAudioRTCPPort = pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.unPortId;
               strcpy(m_localAudioRTCPIPAddress, pLocalMediaInfo->MediaData[i].mediaInfo.PortInfo.cIPAddress);
               break;
            default:
               LOG_ENTRY(m_IPMHandle, "Uexpected value returned from ipm_GetLocalMedia - %d\n",
                         pLocalMediaInfo->MediaData[i].eMediaType);
               break;
         }
      }


}

//*****************************************************************************
// Function: void RTSPEndpoint::RecordPortInfoList(METAEVENT metaevent, DM_PORT_INFO_LIST *pPortInfoList)
// Description: Save the RTSP port information
// Return: void 
// Parameters: METAEVENT metaevent 
//             DM_PORT_INFO_LIST *pPortInfoList 
//*****************************************************************************
void RTSPEndpoint::RecordPortInfoList(METAEVENT metaevent, DM_PORT_INFO_LIST* pPortInfoList)
{
   LOG_ENTRY(metaevent.evtdev,"RTSPEndpoint::RecordPortInfoList: DM_PORT_INFO_LIST:\n");
   LOG_ENTRY(metaevent.evtdev,"\tunVersion:0x%X\n", pPortInfoList->unVersion);
   LOG_ENTRY(metaevent.evtdev,"\tunCount:%d\n", pPortInfoList->unCount);
   for ( unsigned int i=0; i< pPortInfoList->unCount; i++ )
   {
      LOG_ENTRY(metaevent.evtdev,"\tport_info[%d]:\n",i);
      LOG_ENTRY(metaevent.evtdev,"\t\tunVersion: 0x%X\n",pPortInfoList->port_info[i].unVersion);

      char tmpBuff[128] = {0};
      char tokenBuff[10] = {0};
      for ( int j=0; j<16; j++ )
      {
         sprintf(tokenBuff, "0x%x ", int(pPortInfoList->port_info[i].device_ID[j]));
         strcat(tmpBuff, tokenBuff);
      }
      LOG_ENTRY(metaevent.evtdev,"\t\tdevice_ID: %s\n", tmpBuff);
      LOG_ENTRY(metaevent.evtdev,"\t\tport_ID: %s\n",reinterpret_cast<char *>(pPortInfoList->port_info[i].port_ID));
      LOG_ENTRY(metaevent.evtdev,"\t\tport_media_type: %d\n", pPortInfoList->port_info[i].port_media_type);

      switch ( pPortInfoList->port_info[i].port_media_type )
      {
         case DM_PORT_MEDIA_TYPE_AUDIO:
            if ( metaevent.evttype == DMEV_GET_TX_PORT_INFO )
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Tx audio port info to %p\n", &m_audioPortTxInfo);
               memcpy(&m_audioPortTxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            else
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Rx audio port info to %p\n", &m_audioPortRxInfo);
               memcpy(&m_audioPortRxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            break;
         case DM_PORT_MEDIA_TYPE_VIDEO:
            if ( metaevent.evttype == DMEV_GET_TX_PORT_INFO )
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Tx video port info to %p\n", &m_videoPortTxInfo);
               memcpy(&m_videoPortTxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            else
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Rx video port info to %p\n", &m_videoPortRxInfo);
               memcpy(&m_videoPortRxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            break;
         case DM_PORT_MEDIA_TYPE_AUDIO_LINEAR:
            LOG_ENTRY(metaevent.evtdev,"\t\tDetected (but not saving) audio linear port info\n");
            break;
         case DM_PORT_MEDIA_TYPE_NBUP:
            LOG_ENTRY(metaevent.evtdev,"\t\tDetected (but not saving) H223(NBUP) port info\n");
            break;
         case DM_PORT_MEDIA_TYPE_NONE:
            LOG_ENTRY(metaevent.evtdev,"\t\tDetected (but not saving) MEDIA_TYPE_NONE port info\n");
            break;
         default:
            LOG_ENTRY(metaevent.evtdev,"\t\tUnknown DM_PORT_MEDIA_TYPE port info: 0x%X\n", pPortInfoList->port_info[i].port_media_type);
            break;
      }
   }
}

//*****************************************************************************
// Function: void RTSPEndpoint::ConnectToPeer(DM_PORT_INFO &a_audioPortRxInfo, DM_PORT_INFO &a_videoPortRxInfo)
// Description: Connect media streams to peer endpoint
// Return: void 
// Parameters: DM_PORT_INFO &a_audioPortRxInfo 
//             DM_PORT_INFO &a_videoPortRxInfo 
//*****************************************************************************
void RTSPEndpoint::ConnectToPeer(DM_PORT_INFO& a_audioPortRxInfo, DM_PORT_INFO& a_videoPortRxInfo)
{
   // Connect RTSP ports to 3G device's corresponding streams

   /************************************************************************

    Looking for this:

                    RTSP                 3G

    Audio
          Tx --------------->  >----------------- Rx
          Rx ---------------<  <----------------- Tx

    Video
          Tx --------------->  >----------------- Rx
          Rx ---------------<  <----------------- Tx

   ********************************************************************/        
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

   // Have TX and RX port info lists (one each) from RTSP
   // Each list has an audio, video and linear audio node
   // 
   // Have TX and RX port info lists from 3G.  Audio device and video device both have
   // a TX and RX list.  Should only be a video PORT_INFO in the video device's, and
   // an audio PORT_INFO in the audio device's.
   // 

   // Important info here!!!!!
   ///////////////////////////
   // "Tx" device is the device whose handle is used in the dev_PortConnect call.
   // port_info_tx gets the transmit port of the device whose handle is used in the dev_PortConnect call.
   // port_info_rx gets the receive port of the other device.
   // BUT - port_info_rx must contain the device_ID of the device whose handle is used in the dev_PortConnect
   // call, or a "multiple device" error will be found in the RTF log

   Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( GetBridgedRtsp()->M3gIndex());

   // This call connects the Tx of RTSP device to Rx of 3G device (Audio)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList,  EndpointMngr::AUDIO_PORT, GetAudTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx),
          &(GetAudioTxPortInfo()),
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx),
          &(pEndpoint->GetAudioRxPortInfo()),
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortConnect(m_IPMHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(m_IPMHandle,
                "dev_PortConnect() failure in RTSP endpoint - %s\n",
                ATDV_ERRMSGP(m_IPMHandle));
   }
   else
   {
      LOG_ENTRY(m_IPMHandle,"Successful dev_PortConnect() call in RTSP endpoint\n");
   }

   // This call connects the Tx of RTSPdevice to Rx of 3G device (Video)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::VIDEO_PORT, GetVidTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
          &(GetVideoTxPortInfo()),
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
          &(pEndpoint->GetVideoRxPortInfo()),
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortConnect(m_IPMHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(m_IPMHandle, 
                "dev_PortConnect() failure in RTSP endpoint - %s\n",
                ATDV_ERRMSGP(m_IPMHandle));
   }
   else
   {
      LOG_ENTRY(m_IPMHandle,"Successful dev_PortConnect() call in RTSP endpoint\n");
   }
}

//*****************************************************************************
// Function: void RTSPEndpoint::DisconnectFromPeer()
// Description: Disconnect media streams from peer endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void RTSPEndpoint::DisconnectFromPeer()
{
   Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( GetBridgedRtsp()->M3gIndex());
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

   if (m_IPMHandle == -1) {
    LOG_ENTRY(m_IPMHandle, "PortDisconnect (audio) not needed on device %s\n", m_IPMName);
   }
   else {
     // This call disconnects the Tx of RTSPdevice to Rx of 3G device (Audio)
     EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::AUDIO_PORT,GetAudTranscodeEnabled());
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx),
	    &(GetAudioTxPortInfo()),
	    sizeof(DM_PORT_INFO));
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx),
	    &(pEndpoint->GetAudioRxPortInfo()),
	    sizeof(DM_PORT_INFO));
     
     if ( DEV_SUCCESS != dev_PortDisconnect(m_IPMHandle, &portConnectInfoList, this) )
       {
	 LOG_ERROR(m_IPMHandle,
		   "dev_PortDisconnect() failure in RTSP endpoint - %s\n",
		   ATDV_ERRMSGP(m_IPMHandle));
       }
     else
       {
	 LOG_ENTRY(m_IPMHandle,"Successful dev_PortDisconnect() call in RTSP endpoint\n");
       }
   }
   if (m_IPMHandle == -1) {
    LOG_ENTRY(m_IPMHandle, "PortDisconnect (video) not needed on device %s\n", m_IPMName);
   }
   else {
     // This call disconnects the Tx of RTSPdevice to Rx of 3G device (Video)
     EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::VIDEO_PORT,GetVidTranscodeEnabled());
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
	    &(GetVideoTxPortInfo()),
	    sizeof(DM_PORT_INFO));
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
	    &(pEndpoint->GetVideoRxPortInfo()),
	    sizeof(DM_PORT_INFO));
     
     if ( DEV_SUCCESS != dev_PortDisconnect(m_IPMHandle, &portConnectInfoList, this) )
       {
	 LOG_ERROR(m_IPMHandle, 
		   "dev_PortDisconnect() failure in RTSP endpoint - %s\n",
		   ATDV_ERRMSGP(m_IPMHandle));
       }
     else
       {
	 LOG_ENTRY(m_IPMHandle,"Successful dev_PortDisconnect() call in RTSP endpoint\n");
       }
   }
}

//*****************************************************************************
// Function: void RTSPEndpoint::LogDevError()
// Description: Lof dev API error
// Return: void 
// Parameters: none 
//*****************************************************************************
void RTSPEndpoint::LogDevError()
{
   DEV_ERRINFO errorInfo;
   if ( dev_ErrorInfo(&errorInfo) != -1 )
   {
      LOG_ERROR (m_IPMHandle, "Error during dev_* call: %d - %s\n", errorInfo.dev_ErrValue, errorInfo.dev_Msg);
   }
}


//*****************************************************************************
// Function: void RTSPEndpoint::CloseSubDevs()
// Description: Close devices assoicated with this endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void RTSPEndpoint::CloseSubDevs()
{
  
  // TO DO: Close devs only after waiting for Disconnect events
  // App should wait fo PortDisconnect events
  // before closing devices.  This is a temporary workaround :(
  usleep(500000);

  if (m_IPMHandle == -1) {
    LOG_ENTRY(0, "RTSP close not needed on device %s\n", m_IPMName);
  }
  else {
    LOG_ENTRY(m_IPMHandle, "Closing RTSP device\n");
    if ( ipm_Close(m_IPMHandle, NULL) < 0 )
      {
	LOG_ERROR(m_IPMHandle, "ipm_Close(%s) failure\n", m_IPMName);
      }
    else {
      m_IPMHandle = -1;
    }
  }
}

//*****************************************************************************
// Function: void RTSPEndpoint::SetLastMessageString(char *pMessageStr)
// Description: Save the last message string
// Return: void 
// Parameters: char *pMessageStr 
//*****************************************************************************
void RTSPEndpoint::SetLastMessageString(const char *pMessageStr)
{
   int length = (MAX_LAST_MSG_STRING > strlen(pMessageStr)) ? strlen(pMessageStr) : MAX_LAST_MSG_STRING-1;
   memset(m_LastMessageBuff, 0, sizeof(char)*MAX_LAST_MSG_STRING);
   strncpy(&m_LastMessageBuff[0], pMessageStr, length);
   m_LastMessageBuff[length] = '\0';
}

//*****************************************************************************
// Function: char* RTSPEndpoint::GetStateString()
// Description: Format and return a status string
// Return: char* 
// Parameters: none 
//*****************************************************************************
char * RTSPEndpoint::GetStateString()
{
//   cout << "EP:    STATE:    AUDIO:         Tx    VIDEO:        Tx        MISC." << endl;
//   cout << "                 TxLCN: RxLCN:  CAP:  TxLCN: RxLCN: CAP:    CMD/IND:" << endl;
   memset(m_StateBuffer, 0, sizeof(char)*MAX_STATE_STRING);
   sprintf(m_StateBuffer, "RTSP%2d: %15s AUD:%5s%1s  VID:%5s%1s URL:%s                                                              ",
           m_Index,
           m_CurrentStateStr,
           GetSelAudCoderStr(),
           (GetAudTranscodeEnabled()?"*":" "),
           GetSelVidCoderStr(),
           (GetVidTranscodeEnabled()?"*":" "),
           m_LastMessageBuff);
   return m_StateBuffer;
}

//*****************************************************************************
// Function: void RTSPIdleState::Shutdown()
// Description: Process shutdown request
// Return: void
// Parameters: none
//*****************************************************************************
void RTSPEndpoint::Shutdown()
{
  // May need to have state-dependent shutdown on RTSP endpoint.  If so, dispatch
  // shutdown request to states here.

  LOG_ENTRY(GetIPMHandle(), "Shutdown called in %s state\n", m_pCurrentState->GetStateStr());
  DisconnectFromPeer();
  //CloseSubDevs();
  //Notify(APP_RTSP_ENDPOINT_DISCONNECTED);
}


//*****************************************************************************
// Function: void RTSPEndpoint::SetVideoToMPEG4()
// Description: Set the video configuration to MPEG4
// Return: void
// Parameters: none
//*****************************************************************************
void RTSPEndpoint::SetVideoToMPEG4(bool bQcif)
{
  
  SetSelectedVideoCoder(VID_CODER_MPEG4);
  m_localVideoCoderType = CODER_TYPE_MP4V_ES;
  //m_localVideoCoderPayloadType = 102;
  if (bQcif) {
     LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetVideoToMPEG4() - QCIF\n");
     m_localFramesPerSec = 		VIDEO_FRAMESPERSEC_10; 
     m_localImageHeight = 	VIDEO_IMAGE_HEIGHT_144;
     m_localImageWidth = 	VIDEO_IMAGE_WIDTH_176;
     m_localImageProfile = 	VIDEO_PROFILE_LEVEL_SP0_MPEG4;
     m_localBitRate = 	VIDEO_BITRATE_40K;
  } else {
     LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetVideoToMPEG4() - CIF\n");
     m_localFramesPerSec = 		VIDEO_FRAMESPERSEC_10; 
     m_localImageHeight = 	VIDEO_IMAGE_HEIGHT_288;
     m_localImageWidth = 	VIDEO_IMAGE_WIDTH_352;
     m_localImageProfile = 	VIDEO_PROFILE_LEVEL_SP3_MPEG4;
     m_localBitRate = 	VIDEO_BITRATE_384K;
  }
  m_localImageLevel = 	EMM_VIDEO_LEVEL_UNDEFINED;
  m_localSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
}

//*****************************************************************************
// Function: void RTSPEndpoint::SetVideoToH264()
// Description: Set the video configuration to H264
// Return: void
// Parameters: none
//*****************************************************************************
void RTSPEndpoint::SetVideoToH264(bool bQcif)
{

  SetSelectedVideoCoder(VID_CODER_H264);
  m_localVideoCoderType = CODER_TYPE_H264;
  //m_localVideoCoderPayloadType = 102;
  if (bQcif) {
     LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetVideoToH264() - QCIF\n");
     m_localFramesPerSec = 		VIDEO_FRAMESPERSEC_10; 
     m_localImageHeight = 	VIDEO_IMAGE_HEIGHT_144;
     m_localImageWidth = 	VIDEO_IMAGE_WIDTH_176;
     m_localImageProfile = 	VIDEO_PROFILE_BASELINE_H264;
     m_localImageLevel = 	VIDEO_LEVEL_1_H264;
     m_localBitRate = 	VIDEO_BITRATE_40K;
  } else {
     LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetVideoToH264() - CIF\n");
     m_localFramesPerSec = 		VIDEO_FRAMESPERSEC_10; 
     m_localImageHeight = 	VIDEO_IMAGE_HEIGHT_288;
     m_localImageWidth = 	VIDEO_IMAGE_WIDTH_352;
     m_localImageProfile = 	VIDEO_PROFILE_BASELINE_H264;
     m_localImageLevel = 	VIDEO_LEVEL_1_2_H264;
     m_localBitRate = 	VIDEO_BITRATE_384K;
  }
  
  m_localSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
}


//*****************************************************************************
// Function: void RTSPEndpoint::SetVideoToH263()
// Description: Set the video configuration to H263
// Return: void
// Parameters: none
//*****************************************************************************
void RTSPEndpoint::SetVideoToH263(bool bQcif)
{
   
  SetSelectedVideoCoder(VID_CODER_H263);
  m_localVideoCoderType = CODER_TYPE_H263;
  //m_localVideoCoderPayloadType = 34;
  m_localImageProfile = VIDEO_PROFILE_0_H263;
  m_localImageLevel = VIDEO_LEVEL_10_H263;
  m_localSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;

  if (bQcif) {
     LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetVideoToH263() - QCIF \n");
     m_localFramesPerSec =             VIDEO_FRAMESPERSEC_10;
     m_localImageHeight =     VIDEO_IMAGE_HEIGHT_144; 
     m_localImageWidth =      VIDEO_IMAGE_WIDTH_176;
     m_localBitRate =         VIDEO_BITRATE_40K;
  } else {
     LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetVideoToH263() - CIF \n");
     m_localFramesPerSec =             VIDEO_FRAMESPERSEC_10;
     m_localImageHeight =     VIDEO_IMAGE_HEIGHT_288; 
     m_localImageWidth =      VIDEO_IMAGE_WIDTH_352;
     m_localImageProfile =         EMM_VIDEO_PROFILE_UNDEFINED; 
     m_localBitRate =         VIDEO_BITRATE_384K;
  }
}

//*****************************************************************************
// Function: void RTSPEndpoint::SetVideoToH263_1998()
// Description: Set the video configuration to H263
// Return: void
// Parameters: none
//*****************************************************************************
void RTSPEndpoint::SetVideoToH263_1998(bool bQcif)
{
   
  SetSelectedVideoCoder(VID_CODER_H263_1998);
  m_localVideoCoderType = CODER_TYPE_H263_1998;
  //m_localVideoCoderPayloadType = 34;
  m_localImageProfile = VIDEO_PROFILE_0_H263;
  m_localImageLevel = VIDEO_LEVEL_10_H263;
  m_localSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;

  if (bQcif) {
     LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetVideoToH263_1998() - QCIF \n");
     m_localFramesPerSec =             VIDEO_FRAMESPERSEC_10;
     m_localImageHeight =     VIDEO_IMAGE_HEIGHT_144; 
     m_localImageWidth =      VIDEO_IMAGE_WIDTH_176;
     m_localBitRate =         VIDEO_BITRATE_40K;
  } else {
     LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetVideoToH263_1998() - CIF \n");
     m_localFramesPerSec =             VIDEO_FRAMESPERSEC_10;
     m_localImageHeight =     VIDEO_IMAGE_HEIGHT_288; 
     m_localImageWidth =      VIDEO_IMAGE_WIDTH_352;
     m_localImageProfile =         EMM_VIDEO_PROFILE_UNDEFINED; 
     m_localBitRate =         VIDEO_BITRATE_384K;
  }
}


//*****************************************************************************
// Function: void RTSPEndpoint::SetAudioToAMR()
// Description: Set the Audio configuration to AMR
// Return: void
// Parameters: none
//*****************************************************************************
void RTSPEndpoint::SetAudioToAMR()
{
   
  LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetAudioToAMR() \n");
  SetSelectedAudioCoder(AUD_CODER_AMR);
  m_remoteAudioCoderType = CODER_TYPE_AMRNB_12_2k;
  m_localAudioCoderType = CODER_TYPE_AMRNB_12_2k;

   m_remoteAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_remoteAudioFramesPerPkt = 10;
   m_remoteAudioVadEnable = CODER_VAD_ENABLE;
   m_remoteAudioRedPayloadType = 97;

   m_localAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_localAudioFramesPerPkt = 10;
   m_localAudioVadEnable = CODER_VAD_ENABLE;
   m_localAudioRedPayloadType = 97;

   m_ipmAudioCoderResource = RESOURCE_IPM_AMR_NB;
  
}

//*****************************************************************************
// Function: void RTSPEndpoint::SetAudioToPCM()
// Description: Set the Audio configuration to PCM
// Return: void
// Parameters: none
//*****************************************************************************
void RTSPEndpoint::SetAudioToG711U()
{
   
  LOG_ENTRY(GetIPMHandle(), "RTSPEndpoint - SetAudioToG711U() \n");
   m_remoteAudioCoderType = CODER_TYPE_G711ULAW64K;
   m_remoteAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_remoteAudioFramesPerPkt = 1;
   m_remoteAudioVadEnable = CODER_VAD_ENABLE;
   m_remoteAudioRedPayloadType = 0;

   m_localAudioCoderType = CODER_TYPE_G711ULAW64K;
   m_localAudioCoderFramesize = CODER_FRAMESIZE_20;
   m_localAudioFramesPerPkt = 1;
   m_localAudioVadEnable = CODER_VAD_ENABLE;
   m_localAudioRedPayloadType = 0;

   m_ipmAudioCoderResource = RESOURCE_IPM_G711_20MS;

}


