/**
* @file sipendpoint.h
* @brief Declaration of SIPEndpoint class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#ifndef  _SIPENDPOINT_H
#define  _SIPENDPOINT_H

#include "port_connect.h"
#include "gclib.h"
#include "ipmlib.h"
#include "baseendpoint.h"

#include "sdpAPI.h"
#include "sipopening_state.h"
#include "sipreservingmedia_state.h"
#include "sipportconnecting_state.h"
#include "sipidle_state.h"
#include "sipcalling_state.h"
#include "sipconnected_state.h"
#include "sipdropping_state.h"
#include "sipreleasing_state.h"
#include "sipoffered_state.h"
#include "sipclosed_state.h"

#include <time.h>
#include "locker.h"
#include "port_connect.h"

// DTMF options
#define INBAND 0
#define RFC2833 1

// Info to be passed through to dev_PortConnect completion events
#define IPM_AUDIO_PORTCONNECT  0
#define IPM_VIDEO_PORTCONNECT  1

class Endpoint;

//*****************************************************************************
// Class: ::SIPEndpoint
// Description: This is the SIP endpoint class
//*****************************************************************************
class SIPEndpoint : public BaseEndpoint
{
public:
   SIPEndpoint(int numEP, const char *SIPName, const char *IPMName, const char *mapID);
   ~SIPEndpoint();

   void ProcessEvent(METAEVENT metavent);
   SIPEPState::E_SIPEPSTATE GetState();
   char *GetStateString();
   void OpenSubDevs();
   void MakeCall(METAEVENT metaevent);
   void WaitForCall();
   void GetLocalMediaInfo ();
   void ReserveResources();
   void ReleaseResources();
   void ProcessFarEndConnected(METAEVENT metaevent);
   void ProcessCallAnswered(METAEVENT metaevent);
   void ProcessDisconnect(METAEVENT metaevent);
   void ProcessDropCall(METAEVENT metaevent);
   void LogGlobalCallError();
   void Shutdown() { if ( m_pCurrentState ) m_pCurrentState->Shutdown();}
   void CloseSubDevs();
   LINEDEV GetGCSIPHandle() { return m_GCSIPHandle;}
   SRL_DEVICE_HANDLE GetIPMHandle() { return m_IPMHandle;}

   // Port connect related.  IPM devices do not have a separate
   // audio and video component, as do 3G devices.
   // We save only a Tx and Rx list ptr.
   void SetRxPortInfoList(DM_PORT_INFO_LIST *pRxPortInfoList) { m_RxPortList = *pRxPortInfoList;}
   void SetTxPortInfoList(DM_PORT_INFO_LIST *pTxPortInfoList) { m_TxPortList = *pTxPortInfoList;}

   // Get first element of Tx and Rx lists.
   // Not sure if this is needed for IPM
   DM_PORT_INFO& GetRxPortInfo() { return m_RxPortList.port_info[0];}
   DM_PORT_INFO& GetTxPortInfo() { return m_TxPortList.port_info[0];}

   DM_PORT_INFO& GetAudioRxPortInfo() { return m_audioPortRxInfo;}
   DM_PORT_INFO& GetAudioTxPortInfo() { return m_audioPortTxInfo;}
   DM_PORT_INFO& GetVideoRxPortInfo() { return m_videoPortRxInfo;}
   DM_PORT_INFO& GetVideoTxPortInfo() { return m_videoPortTxInfo;}

   void ConnectToPeer(DM_PORT_INFO& a_audioPortRxInfo, 
                      DM_PORT_INFO& a_videoPortRxInfo); 
   void DisconnectFromPeer();
   void ChangeState(SIPEPState::E_SIPEPSTATE e_NewState);
   void StartMedia();
   void StopMedia();
   void SetLastMessageString(const char *pMessageStr);
   void SetDtmfMode(int dtmfMode);
   void LogDevError();
   void GetIPMPortInfo ();
   void RecordPortInfoList(METAEVENT metaevent, DM_PORT_INFO_LIST* pPortInfoList);
   void DropCall(METAEVENT metaevent);
   bool IsSIPIFrameUpdate(METAEVENT metaevent);
   void SendSIPOK ();
   void AnswerCall(METAEVENT metaevent);
   void SaveRemoteMediaInfo(METAEVENT metaevent);

	void Set3GEPDCI(const char *pDciStr)
	{
       strcpy(m_3GEPDciStr, pDciStr);
	}
	char *Get3GEPDCI()
	{
		return m_3GEPDciStr;
	}

   unsigned short GetIpmPortConnectCompletionCnt() { return m_ipmPortConnectCompletionCnt;};
   void ZeroIpmPortConnectCompletionCnt() { m_ipmPortConnectCompletionCnt = 0;};
   void IncrementIpmPortConnectCompletionCnt() { m_ipmPortConnectCompletionCnt++;};

   char *MapIP() { return m_mapID;}
   int GetIndex() { return m_Index;}

   void SendSipInfoDtmf(char dtmf);
   bool isSIPInboundDevice(); 
   
private:
   bool SendIFrameRequest();
   void BuildAnswerSDP(sdpSessionDescription *sdp);
   void SetVideoToH263(bool isInboundCall);
   void SetVideoToMPEG4();
   void SetVideoToH264();
   void SetAudioToAMR();
   void SetAudioToG723();
   void SetAudioToG729();
   void SetAudioToG711U();
   void SetAudioToG711A();
   void SetupSipCoders();
   void UpdateH263Attributes(char*, int, int);
   void UpdateMpeg4Attributes(int profileAndLevel, int bitrate, char dciStr[MAX_STATE_STRING]);
   void UpdateH264Attributes(int profileAndLevel, int bitrate, char dciStr[MAX_STATE_STRING]);
   void ParseSdp(sdpSessionDescription& sdp, char* dataBuff);
  
   int                        m_Index;
   SIPOpeningState           *m_pSIPOpeningState;
   SIPReservingMediaState    *m_pSIPReservingMediaState;
   SIPPortConnectingState    *m_pSIPPortConnectingState;
   SIPIdleState              *m_pSIPIdleState;
   SIPCallingState           *m_pSIPCallingState;
   SIPConnectedState         *m_pSIPConnectedState;
   SIPDroppingState          *m_pSIPDroppingState;
   SIPReleasingState         *m_pSIPReleasingState;
   SIPOfferedState           *m_pSIPOfferedState;
   SIPClosedState            *m_pSIPClosedState;
   SIPEPState                *m_pCurrentState;
   char                       m_SIPName[DEV_NAME_SIZE];
   char                       m_IPMName[DEV_NAME_SIZE];
   char                       m_mapID[DEV_NAME_SIZE];
   LINEDEV                    m_GCSIPHandle;
   SRL_DEVICE_HANDLE          m_IPMHandle;
   char                       m_StateBuffer[MAX_STATE_STRING];
   char                       m_LastMessageBuff[MAX_LAST_MSG_STRING];
   CRN                        m_curCallCRN;
   IPM_MEDIA_INFO             m_mediaInfo;
   short                      m_sdpCoderType;
   DEV_RESOURCE_LIST          m_resourceList;

   // Port connect related

   // Ptrs to PORT_INFO_LISTs returned
   // Can likely eliminate, if PORT_INFO below is used
   DM_PORT_INFO_LIST          m_RxPortList;
   DM_PORT_INFO_LIST          m_TxPortList;

   unsigned short             m_ipmPortConnectCompletionCnt;

   // PORT_INFO structs themselves
   DM_PORT_INFO               m_audioPortTxInfo;
   DM_PORT_INFO               m_audioPortRxInfo;
   DM_PORT_INFO               m_videoPortTxInfo;
   DM_PORT_INFO               m_videoPortRxInfo;

   // Values for local media interfaces.  Needed for Oceanville, as media
   // NICS may change
   unsigned int               m_localVideoRTPPort;
   char                       m_localVideoRTPIPAddress[IP_ADDR_SIZE];
   unsigned int               m_localVideoRTCPPort;
   char                       m_localVideoRTCPIPAddress[IP_ADDR_SIZE];
   unsigned int               m_localAudioRTPPort;
   char                       m_localAudioRTPIPAddress[IP_ADDR_SIZE];
   unsigned int               m_localAudioRTCPPort;
   char                       m_localAudioRTCPIPAddress[IP_ADDR_SIZE];

   // Holders for remote IP addresses and RTP ports parsed out of SDPs
   unsigned int               m_remoteAudioRTPPort;
   unsigned int               m_remoteVideoRTPPort;
   char                       m_remoteAudioIPAddress[IP_ADDR_SIZE];
   char                       m_remoteVideoIPAddress[IP_ADDR_SIZE];
   
   //*****************************************************************************
   // Description: Audio and video format and coder configurations
   //*****************************************************************************
   char                       m_sdpRFC2833Format[30];
   char                       m_sdpRFC2833Rtpmap[30];

   char                       m_sdpAudioFormat[30];
   char                       m_sdpAudioRtpmap[30];
   char                       m_sdpAudioFmtp[30];

   eIPM_CODER_TYPE            m_remoteAudioCoderType;
   eIPM_CODER_FRAMESIZE       m_remoteAudioCoderFramesize;
   int                        m_remoteAudioFramesPerPkt; 
   eIPM_CODER_VAD             m_remoteAudioVadEnable;
   int                        m_remoteAudioCoderPayloadType;
   int                        m_remoteAudioRedPayloadType;

   eIPM_CODER_TYPE            m_localAudioCoderType;
   eIPM_CODER_FRAMESIZE       m_localAudioCoderFramesize;
   int                        m_localAudioFramesPerPkt; 
   eIPM_CODER_VAD             m_localAudioVadEnable;
   int                        m_localAudioCoderPayloadType;
   int                        m_localAudioRedPayloadType;

   eDEV_RESOURCE_TYPE         m_ipmAudioCoderResource;

   char                       m_sdpVideoFormat[30];
   char                       m_sdpVideoRtpmap[30];
   char                       m_sdpVideoFmtp[MAX_STATE_STRING];

   eIPM_CODER_TYPE            m_localVideoCoderType;
   int                        m_localVideoCoderPayloadType;
   eVIDEO_PROFILE             m_localImageProfile;
   eVIDEO_LEVEL               m_localImageLevel;

   eVIDEO_IMAGE_WIDTH         m_localImageWidth;
   eVIDEO_IMAGE_HEIGHT        m_localImageHeight;
   eVIDEO_FRAMESPERSEC        m_localFramesPerSec;
   unsigned int               m_localBitRate;
   eVIDEO_SAMPLING_RATE       m_localSamplingRate;

   char                       m_3GEPDciStr[128];	
   unsigned char              m_dciOctetStrSize;
   unsigned char              m_dciOctetStr[128];
   char                       m_SdpDciStr[128];

   eIPM_CODER_TYPE            m_remoteVideoCoderType;
   int                        m_remoteVideoCoderPayloadType;
   eVIDEO_PROFILE             m_remoteImageProfile;
   eVIDEO_LEVEL               m_remoteImageLevel;

   eVIDEO_IMAGE_WIDTH         m_remoteImageWidth;
   eVIDEO_IMAGE_HEIGHT        m_remoteImageHeight;
   eVIDEO_FRAMESPERSEC        m_remoteFramesPerSec;
   unsigned int               m_remoteBitRate;
   eVIDEO_SAMPLING_RATE       m_remoteSamplingRate;

   int                        m_nAMR_PT;
   int                        m_nH263_PT;
   int                        m_nMPEG4_PT;
   int                        m_nH264_PT;
   int                        m_nRFC2833_PT;
   
   char                       m_CurrentStateStr[MAX_CURRENT_STATE_STR];
   CLocker                    m_StateLock;
};

#endif


