/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* DIALOGIC CONFIDENTIAL
*
* Copyright (C) 2006-2007 Dialogic Corporation. All Rights Reserved.
* The source code contained or described herein and all documents related
* to the source code ("Material") are owned by Dialogic Corporation or its
* suppliers or licensors. Title to the Material remains with Dialogic Corporation
* or its suppliers and licensors. The Material contains trade secrets and
* proprietary and confidential information of Dialogic or its suppliers and
* licensors. The Material is protected by worldwide copyright and trade secret
* laws and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted, distributed,
* or disclosed in any way without Dialogic's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
//***********************************************************************
//***********************************************************************
// CnfBoard.cpp: implementation of the CnfBoard class.
//
//////////////////////////////////////////////////////////////////////

#include "pdl.h"

static const char * CNF_MODULE_NAME = "SdpParser";


/////////////////////////////////////////////////////////////////////////////
//      State machine definition
/////////////////////////////////////////////////////////////////////////////
// States (in addition to pre_defined  S_BEGIN, S_FINAL etc.)
enum {
   S_SETATTR = 0,       // Set Attributes 
   S_SETATTR_EXIT,      // Set Attributes (exiting)
   S_ENABLEEVT,         // Enable Events
   S_ENABLEEVT_EXIT,    // Enable Events ( exiting)
   S_DEVCNT,            // Count sub-devices 
   S_DEVCNT_EXIT,       // Count sub-devices  (exiting)
   S_DISABLEEVT,        // Disable Events
};

static NAME_TABLE state_tbl[] = {
    {"S_SETATTR",         S_SETATTR},
    {"S_SETATTR_EXIT",    S_SETATTR_EXIT},
    {"S_ENABLEEVT",       S_ENABLEEVT},
    {"S_ENABLEEVT_EXIT",  S_ENABLEEVT_EXIT},
    {"S_DEVCNT",          S_DEVCNT},
    {"S_DEVCNT_EXIT",     S_DEVCNT_EXIT},
    {"S_DISABLEEVT",      S_DISABLEEVT},
    {0, 0}
}; // End of state_tbl[]
 
//*****************************************************************************
// Purpose	: 
//    return state name for given value
// Parameters:	
//    [in] state
// Returns:	
//    name 
//*****************************************************************************
const char * CnfBoard::state_name(int state){
const char *name;
   if (! str_findname(state,&name,state_tbl) ){
         name = common_state_name(state);
   }
 return name;
} // End of state_name()

//==========================================================
// Action codes 
enum {
    A_SETATTR,       // set parameters 
    A_ENABLEEVT,     // Enable events
    A_DEVCNT,        // Get cnf device count
    A_CREATECONF,    // Create Conferences
    A_TRYCLOSE,      // Count Conferences and close if number is 0
};

static NAME_TABLE ac_tbl[] = {
    { "A_SETATTR",    A_SETATTR    }, 
    { "A_ENABLEEVT",  A_ENABLEEVT  }, 
    { "A_DEVCNT",     A_DEVCNT     }, 
    { "A_CREATECONF", A_CREATECONF }, 
    { "A_TRYCLOSE",   A_TRYCLOSE   },     
    { 0,0 }     
}; // End of ac_tbl[]

//*****************************************************************************
// Purpose	: 
//    return action name for given value
// Parameters:	
//    [in] action
// Returns:	
//    name 
//*****************************************************************************
const char * CnfBoard::action_name(int action){
const char *name;
  if (! str_findname(action, &name, ac_tbl) ){
        name = common_action_name(action);
  }
  return name;
}// End of action_name()


//
//   Action table
// { current_state     action   state_if_pass,  action_if_fail, state_if_fail}
//
//====================================================================================
// OPEN ->  SET_ATTR -> ENABLEEVENTS -> DEVCNT -> CREATE CONFERENCES and RELAX
// S_OPEN ->S_SETATTR-> S_ENABLEEVT -> S_DEVCNT -> S_RELAX
static ACTION_STRU board_action_stru[] = {
//  current_state     action          state_if_pass   action(fail)

    { S_ANY,          A_CLOSE,        S_FINAL,          A_GIVEUP  },
    { S_ANY,          A_GIVEUP,       S_FINAL,          A_NONE    },
 
    { S_OPEN,         A_SETATTR,      S_SETATTR,        A_CLOSE },
    { S_OPEN,         A_EXIT,         S_OPEN_EXIT,      A_CLOSE },

    { S_SETATTR,      A_ENABLEEVT,    S_ENABLEEVT,      A_CLOSE   },
    { S_SETATTR,      A_EXIT,         S_SETATTR_EXIT,   A_CLOSE   },

    { S_ENABLEEVT,    A_DEVCNT,       S_DEVCNT,         A_CLOSE   },
    { S_ENABLEEVT,    A_EXIT,         S_ENABLEEVT_EXIT, A_CLOSE   },

    { S_DEVCNT,       A_CREATECONF,   S_RELAX,          A_CLOSE   },
    { S_DEVCNT,       A_EXIT,         S_DEVCNT_EXIT,    A_CLOSE   },

    { S_RELAX,        A_TRYCLOSE,     S_DISABLEEVT,     A_NONE    },   
    { S_WAIT_SUBDEV,  A_TRYCLOSE,     S_DISABLEEVT,     A_NONE    },   

    { S_DISABLEEVT,   A_CLOSE,        S_FINAL,          A_NONE    },

// No operation
    { S_ANY,          A_NONE,         S_SAME,           A_NONE     },

// State machine error (missing case to handle specified action)
    { S_RELAX,        A_ERROR,        S_SAME,           A_CLOSE },
    { S_ANY,          A_ERROR,        S_SAME,           A_CLOSE },

// Not part of state machine, indicates last line
    { S_END,   0,    0,     0  }   
}; // End of board_action_stru[]

//------------------------------------------------------------------------------------
static EVENT_STRU board_event_stru[] = {

//  current_state     event                      action     

// After Open
    { S_OPEN,         CNFEV_OPEN,                A_SETATTR   },
    { S_OPEN,         CNFEV_OPEN_FAIL,           A_CLOSE     },
    { S_OPEN,         USREV_EXIT_REQUEST,        A_EXIT      },
    { S_OPEN,         USREV_TIMEOUT,             A_CLOSE     },
    
    { S_OPEN_EXIT,    CNFEV_OPEN,                A_CLOSE,    },
    { S_OPEN_EXIT,    CNFEV_OPEN_FAIL,           A_CLOSE,    },
    { S_OPEN_EXIT,    USREV_TIMEOUT,             A_CLOSE,    },

// After Set Attributes
    { S_SETATTR,      CNFEV_SET_ATTRIBUTE,       A_ENABLEEVT },
    { S_SETATTR,      CNFEV_SET_ATTRIBUTE_FAIL,  A_CLOSE     },
    { S_SETATTR,      USREV_EXIT_REQUEST,        A_EXIT      },
    { S_SETATTR,      USREV_TIMEOUT,             A_CLOSE     },

    { S_SETATTR_EXIT, CNFEV_SET_ATTRIBUTE,       A_CLOSE     },
    { S_SETATTR_EXIT, CNFEV_SET_ATTRIBUTE_FAIL,  A_CLOSE     },
    { S_SETATTR_EXIT, USREV_TIMEOUT,             A_CLOSE     },

// After Enable events
    { S_ENABLEEVT,    CNFEV_ENABLE_EVENT,         A_DEVCNT    },
    { S_ENABLEEVT,    CNFEV_ENABLE_EVENT_FAIL,    A_CLOSE     },
    { S_ENABLEEVT,    USREV_EXIT_REQUEST,         A_EXIT      },
    { S_ENABLEEVT,    USREV_TIMEOUT,              A_CLOSE     },

    { S_ENABLEEVT_EXIT, CNFEV_ENABLE_EVENT,       A_CLOSE     },
    { S_ENABLEEVT_EXIT, CNFEV_ENABLE_EVENT_FAIL,  A_CLOSE     },
    { S_ENABLEEVT_EXIT, USREV_TIMEOUT,            A_CLOSE     },

// After GetDeviceCount
    { S_DEVCNT,  CNFEV_GET_DEVICE_COUNT,          A_CREATECONF},
    { S_DEVCNT,  CNFEV_GET_DEVICE_COUNT_FAIL,     A_CLOSE     },
    { S_DEVCNT,  USREV_EXIT_REQUEST,              A_EXIT      },
    { S_DEVCNT,  USREV_TIMEOUT,                   A_CLOSE     },

    { S_DEVCNT_EXIT, CNFEV_GET_DEVICE_COUNT,      A_CLOSE     },
    { S_DEVCNT_EXIT, CNFEV_GET_DEVICE_COUNT_FAIL, A_CLOSE     },
    { S_DEVCNT_EXIT, USREV_TIMEOUT,               A_CLOSE     },


// Exit Request
    { S_RELAX,        USREV_EXIT_REQUEST,        A_TRYCLOSE    },

    { S_WAIT_SUBDEV,  USREV_EXIT_REQUEST,        A_TRYCLOSE    },  
    { S_WAIT_SUBDEV,  CNFEV_CONF_CLOSED,         A_TRYCLOSE    },  
    { S_WAIT_SUBDEV,  CNFEV_PARTY_REMOVED,       A_TRYCLOSE    },  
    
    { S_DISABLEEVT,   CNFEV_DISABLE_EVENT,       A_CLOSE    },  
    { S_DISABLEEVT,   CNFEV_DISABLE_EVENT_FAIL,  A_CLOSE    },  

    { S_WAIT_SUBDEV,  USREV_TIMEOUT,             A_CLOSE      },  


// No timeouts in Relax state
    { S_RELAX,        USREV_TIMEOUT,             A_NONE       },


// USREV_CNF_OPEN and CLOSE
    { S_ANY,        USREV_CNF_OPEN,              A_NONE       },
    { S_ANY,        CNFEV_CONF_CLOSED,           A_NONE       },
    { S_ANY,        CNFEV_CONF_OPENED,           A_NONE       },

    { S_ANY,        CNFEV_PARTY_ADDED,           A_NONE       },
    { S_ANY,        CNFEV_PARTY_REMOVED,         A_NONE       },

    
    //Anything else falls here
    { S_ANY,          EVENT_ANY,                 A_ERROR      },
    { S_END,   0,    0  } 
}; // End of board_state_stru[]

//*****************************************************************************
// Purpose	: 
//    Execute an action
// Parameters:	
//    [in] action
// Returns:	
//    true  = success
//    false = error
//*****************************************************************************
bool CnfBoard::Execute(int action){
bool brc = true;
 switch(action) {

     case A_ERROR:
          // Indicate error
          StateMachineError();
          break;

     case A_GIVEUP:  // do nothing but advance state machine
          SetSrlState(SRLSTATE_ALL, RESET_SRL_STATE);
          SetCurrentState(S_FINAL);
          break;      

     case A_NONE:          // do nothing
     case A_EXIT:          // do nothing but advance state machine
          break;
          
     case A_CLOSE:
          brc = Close();
          break;

     case A_SETATTR:
          brc = SetAttributes();
          break;

     case A_ENABLEEVT:
          brc = EnableEvents();
          break;

     case A_DEVCNT:
          brc = GetDeviceCount();
          break;

     case A_CREATECONF:
          brc = CreateConferences();
          // And relax
          SetSrlState(SRLSTATE_INIT_COMPLETE  );
          LOG(LOG_APP, GetName(), "Initialization complete");
          break;  

     case A_TRYCLOSE:
          brc = TryClose();
          break;  

     default:
          LOG(LOG_ERR1, GetName(),
              "CnfBoard: App error. Missing case to handle action #%d %s",
               action, action_name(action) );
          TryClose();
          break;

 }  // switch action
 return brc;
} // End of Execute()

//*****************************************************************************
// Purpose	: 
//    Handle events
// Parameters:	
//   [in] event
//   [in] event data
//   [in] data length
//   [in] METAEVENT (not used)
// Returns:	
//    none
//*****************************************************************************
void CnfBoard::HandleEvent(int event,
                           void *evtdata, int evtlen,
                           METAEVENT *metaeventp){

    switch (event){
        case CNFEV_OPEN_FAIL:
             SetSrlState(SRLSTATE_FAILED_FATAL);
             break;

        case CNFEV_GET_DEVICE_COUNT:
            {
             CPCNF_DEVICE_COUNT_INFO pInfo = (CPCNF_DEVICE_COUNT_INFO) evtdata;
             cnf_dump(pInfo);
             m_max_conf_count = pInfo->unFreeConfCount;
            }
             break;

        case CNFEV_OPEN:
             break;

        case CNFEV_PARTY_REMOVED:
             LOG( LOG_DBG, GetName(), 
                  "Number of parties: %d",
                  --m_number_of_parties);
             break;

        case CNFEV_PARTY_ADDED:
             LOG( LOG_DBG, GetName(), 
                  "Number of parties: %d",
                  ++m_number_of_parties);
             break;

        case CNFEV_CONF_CLOSED:
             LOG( LOG_DBG, GetName(), 
                  "Number of conferences: %d",
                  --m_number_of_conferences);
             break;

        case USREV_CNF_OPEN:
             LOG(LOG_DBG, GetName(), 
                 "Number of conferences: %d",
                 ++m_number_of_conferences);
             break;

        default:
             break;

    } // switch event

    CSrlDevice::HandleEvent(event, evtdata, evtlen, metaeventp);
    
 return;
} // End of HandleEvent()

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
//*****************************************************************************
// Purpose	: 
//    Constructor
// Parameters:	
//    Configuration parameters ( from configuration file )
// Returns:	
//    none
//*****************************************************************************
CnfBoard::CnfBoard(PCommonParams pCommonParams,
                   PBrdParams pBrdParams,
                   PCnfPrmContainer pCnfPrmContainer)
                  : CSrlDeviceContainer(DEV_CNFBOARD, pCommonParams, 0) {

    brd_ActiveTalker = pBrdParams->m_brd_active_talker?ECNF_ATTR_STATE_ENABLED : ECNF_ATTR_STATE_DISABLED ;
    brd_ATInterval   = pBrdParams->m_brd_at_interval;
	brd_DTMFClamping = pBrdParams->m_brd_dtmf_clamping?ECNF_ATTR_STATE_ENABLED : ECNF_ATTR_STATE_DISABLED ;

    m_pCnfPrmContainer = pCnfPrmContainer;  // get list of conference parameters 
    m_pBrdParams       = pBrdParams; 

    m_max_conf_count = 0;
    m_number_of_conferences = 0;
    m_number_of_parties = 0;
    if (IsSrlState(SRLSTATE_RESERVED, true) ){
        OpenPrivateLog(pBrdParams->m_log_level);
        glb_pConferencePool->push_back(this);
        SetDfltTimer(BOARD_TMO);
        pBrdParams->Dump(GetLog());
    // State Machine
        if ( SetStateMachine(board_event_stru, board_action_stru) ) {
           // Open this board
           Open();
        }
    } else {
        LOG(LOG_ERR2, CNF_MODULE_NAME, "Cannot reserve CNF board ");
        SetExitCode(EXIT_INIT);
    }
 return;
} // End of Constructor()

//*****************************************************************************
// Purpose	: 
//    Destructor
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
CnfBoard::~CnfBoard() {

 // If any conference exist (as a result of error)
 // clean references to this board
   list<PSrlDevice>::iterator pos;
   PCnfConference pConf;
   for ( pos = begin(); pos != end(); ++pos) {
       pConf = static_cast<PCnfConference>(*pos);
       pConf->SetCnfBoard(0);
   } 

 // Close Board
    Close();
 return;
}  // End of Destructor()

//*****************************************************************************
// Purpose	: 
//    Find conference by pass code
// Parameters:	
//    [in]  pass code
//    [out] Holder to return conference
// Returns:	
//    true = found
//    false = not found
//*****************************************************************************
bool CnfBoard::FindConferenceByPassCode(const char *pass_code, PCnfConference *ppConf){
   list<PSrlDevice>::iterator pos;
   PCnfConference pConf;
   for ( pos = begin(); pos != end(); ++pos) {
       pConf = static_cast<PCnfConference>(*pos);
       if ( pConf->IsPassCode(pass_code) ){
           LOG(LOG_DBG, GetName(), "CheckConference: passCode=%s, Cnf=%s, passed", pass_code, pConf->GetName());
           *ppConf = pConf; 
           return true;
       }
       LOG(LOG_DBG, GetName(), "CheckConference: passCode=%s, Cnf=%s, failed", pass_code, pConf->GetName());
   } 
  return false;
} // End of FindConferenceByPassCode()

//*****************************************************************************
// Purpose	: 
//    Find conference by Id
// Parameters:	
//    [in] id
//    [out] Holder to return conference
// Returns:	
//    true = found
//    false = not found
//*****************************************************************************
bool CnfBoard::FindConferenceById(int id, PCnfConference *ppConf){
   list<PSrlDevice>::iterator pos;
   PCnfConference pConf;
   for ( pos = begin(); pos != end(); ++pos) {
       pConf = static_cast<PCnfConference>(*pos);
       if ( pConf->IsId(id) ){
           *ppConf = pConf; 
           return true;
       }
   } 
  return false;
} // End of FindConferenceById()


//*****************************************************************************
// Purpose	: 
//    Open Conference bard
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CnfBoard::Open(){
 bool brc = false;
 int rc;
    if ( IsSrlState(SRLSTATE_RESERVED, true) ) { 
        rc = cnf_Open(GetName(), 0, this);
        LOG(RCHANDLE(rc), GetName(),
                 "0x%x = cnf_Open(%s,0,0x%x(this))",
                 rc,GetName(),this);
        if (rc == CNF_ERROR) {
            process_cnf_error();
            SetCurrentState(S_FINAL);
            SetSrlState(SRLSTATE_FAILED_FATAL);
        }else {
            brc = true;
            m_srl_handle = rc;
            SetCurrentState(S_OPEN);
            SetSrlState(SRLSTATE_OPENED);
        }
    }else {
       LOG( LOG_ERR1, GetName(),
            "cnf_Open(): (board) - device not reserved");
       SetCurrentState(S_FINAL);
       SetSrlState(SRLSTATE_FAILED_FATAL);
    }
 return brc;
} // End of Open()

//*****************************************************************************
// Purpose	: 
//    Close conference board
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CnfBoard::Close(){
 bool brc = true;
 int rc;

    if( IsSrlState(SRLSTATE_OPENED, true) ){
        if ( IsSrlState(SRLSTATE_WAITEVENT, true) ){
               LOG( LOG_WARNING, GetName(),
                    "cnf_Close() while waiting for event, state = %d %s",
                    GetCurrentState(), state_name(GetCurrentState()));
                    DumpMissingEvents();
        }

        if (m_number_of_parties > 0 ){
               LOG( LOG_WARNING, GetName(),
                    "cnf_Close() while %d parties are still open",
                    m_number_of_parties);
        }
        
        if (m_number_of_conferences > 0 ){
               LOG( LOG_WARNING, GetName(),
                    "cnf_Close() while %d conferences are still open",
                    m_number_of_conferences );
        }

        CNF_CLOSE_INFO CloseInfo;
        rc = cnf_Close(m_srl_handle, &CloseInfo);

//      This is RFU
//      cnf_dump(&CloseInfo); 

        LOG( RC(rc), GetName(),
                     "%d = cnf_close(0x%x)",
                     rc, m_srl_handle);

        brc = (rc == 0);
        if (!brc){
            process_cnf_error();
        } else {
            m_srl_handle = INV_SRL_HANDLE;
        }

        SetCurrentState(S_FINAL);
        SetSrlState(SRLSTATE_ALL, RESET_SRL_STATE);
    }

 return brc;
} // End of Close()

//*****************************************************************************
// Purpose	: 
//    Enable events
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CnfBoard::EnableEvents(){

    CNF_EVENT_INFO CnfEventInfo;
    unsigned int EventList[] = { 
         ECNF_BRD_EVT_CONF_OPENED,
         ECNF_BRD_EVT_CONF_CLOSED,
         ECNF_BRD_EVT_PARTY_ADDED,
         ECNF_BRD_EVT_PARTY_REMOVED,
//         ECNF_BRD_EVT_ACTIVE_TALKER
    };
    CnfEventInfo.unVersion = CNF_EVENT_INFO_VERSION_0;  ///< Structure version
    CnfEventInfo.unEventCount = sizeof(EventList)/sizeof(unsigned int);  ///< Number of events in list
    CnfEventInfo.punEventList = EventList;              ///< Pointer to event list

    bool brc = false;
    int rc;

    rc = cnf_EnableEvents(m_srl_handle, &CnfEventInfo, this);

    cnf_dump(&CnfEventInfo);

    LOG(RC(rc), GetName(), "%d = cnf_EnableEvents(0x%x, CNF_EVENT_INFO, 0x%x(this))",
                            rc,m_srl_handle,this);

    if ( rc == CNF_ERROR) {
         process_cnf_error();
    }else {
         brc = true;
    }

   return brc;
} // End of EnableEvents()
//*****************************************************************************
// Purpose	: 
//    Disable events
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CnfBoard::DisableEvents(){

    CNF_EVENT_INFO CnfEventInfo;
    unsigned int EventList[] = { 
         ECNF_BRD_EVT_CONF_OPENED,
         ECNF_BRD_EVT_CONF_CLOSED,
         ECNF_BRD_EVT_PARTY_ADDED,
         ECNF_BRD_EVT_PARTY_REMOVED,
//         ECNF_BRD_EVT_ACTIVE_TALKER
    };
    CnfEventInfo.unVersion = CNF_EVENT_INFO_VERSION_0;  ///< Structure version
    CnfEventInfo.unEventCount = sizeof(EventList)/sizeof(unsigned int);  ///< Number of events in list
    CnfEventInfo.punEventList = EventList;              ///< Pointer to event list

    bool brc = false;
    int rc;

    rc = cnf_DisableEvents(m_srl_handle, &CnfEventInfo, this);

    cnf_dump(&CnfEventInfo);

    LOG(RC(rc), GetName(), "%d = cnf_DisableEvents(0x%x, CNF_EVENT_INFO, 0x%x(this))",
                            rc,m_srl_handle,this);

    if ( rc == CNF_ERROR) {
         process_cnf_error();
    }else {
         brc = true;
    }

   return brc;
} // End of DisableEvents()

//*****************************************************************************
// Purpose	: 
//    Request device count
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CnfBoard::GetDeviceCount(){

   bool brc = false;
   int rc;

   rc = cnf_GetDeviceCount(m_srl_handle,  this);

   LOG(RC(rc), GetName(), "%d = cnf_GetDeviceCount(0x%x, 0x%x(this))",
                           rc,m_srl_handle,this);

        if (rc == CNF_ERROR) {
            process_cnf_error();
        }else {
            brc = true;
        }

   return brc;
} // End of GetDeviceCount()

//*****************************************************************************
// Purpose	: 
//    Set attributes
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CnfBoard::SetAttributes(){
    CNF_ATTR CnfAttr[3];
        CnfAttr[0].unVersion   = CNF_ATTR_VERSION_0 ;           ///< Structure version
        CnfAttr[0].unAttribute = ECNF_BRD_ATTR_ACTIVE_TALKER;   ///< Attribute type
        CnfAttr[0].unValue     = brd_ActiveTalker;              ///< Attribute value

        CnfAttr[1].unVersion   = CNF_ATTR_VERSION_0 ;           ///< Structure version
        CnfAttr[1].unAttribute = ECNF_BRD_ATTR_TONE_CLAMPING;   ///< Attribute type
        CnfAttr[1].unValue     = brd_DTMFClamping;              ///< Attribute value

        CnfAttr[2].unVersion   = CNF_ATTR_VERSION_0 ;           ///< Structure version
        CnfAttr[2].unAttribute = ECNF_BRD_ATTR_NOTIFY_INTERVAL; ///< Attribute type
        CnfAttr[2].unValue     = brd_ATInterval;                ///< Attribute value


   CNF_ATTR_INFO CnfAttrInfo;
    CnfAttrInfo.unVersion = CNF_ATTR_INFO_VERSION_0; ///< Structure version
    CnfAttrInfo.unAttrCount = 3;                     ///< Number of attribute structures in list
    CnfAttrInfo.pAttrList = CnfAttr;                 ///< Pointer to attribute structure list

   bool brc = false;
   int rc;

   rc = cnf_SetAttributes(m_srl_handle, &CnfAttrInfo, this);

   cnf_dump(&CnfAttrInfo);
   LOG(RC(rc), GetName(), "%d = cnf_SetAttributes(0x%x,attr,0x%x(this))",
                           rc,m_srl_handle,this);

 
        if (rc == CNF_ERROR) {
            process_cnf_error();
        }else {
            brc = true;
        }

   return brc;
} // End of SetAttributes()

//*****************************************************************************
// Purpose	: 
//    Create conferences
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CnfBoard::CreateConferences(){
bool brc = true;
  if ( m_max_conf_count < (signed)m_pCnfPrmContainer->GetNumberOfConferences() ){
       LOG( LOG_ERR2, GetName(), 
           "Requested: %d conferences: Available = %d",
            m_pCnfPrmContainer->GetNumberOfConferences(), m_max_conf_count);
  }else {
      // Create Conferences
      unsigned int inx = 0;
      PCnfParams pCnfParams;
      CnfConference *pConf;
      while ( m_pCnfPrmContainer->GetConferencePrm(inx++, &pCnfParams) ){
              pConf = new CnfConference(this, m_pCommonParams, pCnfParams);
              push_back(pConf);
      }
  }
  return brc;
} // End of CreateConferences()

//*****************************************************************************
// Purpose	: 
//    Close this board if there are no more conferences
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CnfBoard::TryClose(){
  if (    ( m_number_of_conferences > 0 ) 
       || ( m_number_of_parties > 0 ) ) {
       SetCurrentState(S_WAIT_SUBDEV);
       LOG( LOG_DBG, GetName(),
            "false = TryClose(conferences = %d, parties = %d)",
            m_number_of_conferences, m_number_of_parties );
       return false;
  }

   LOG( LOG_DBG, GetName(),
        "true = TryClose(conferences = %d)",
        m_number_of_conferences);
   DisableEvents();
  return true;
} // End of TryClose()

//*****************************************************************************
// Purpose	: 
//    Dump members
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
void CnfBoard::Dump(){
    LOG(LOG_APP, GetName(), "--CNF Board parameters");
    LOG(LOG_APP, GetName(), "  brd_ActiveTalker   = '%s'",    YESNO(brd_ActiveTalker));
    LOG(LOG_APP, GetName(), "  brd_ATInterval     = %d",      brd_ATInterval);
    LOG(LOG_APP, GetName(), "  brd_DTMFClamping   = '%s'",    YESNO(brd_DTMFClamping));
    LOG(LOG_APP, GetName(), "  brd_PrivateLog     = %d '%s'", brd_PrivateLog,log_get_msgtype_name(brd_PrivateLog) );
 return;
} // End of Dump()

//*****************************************************************************
// Purpose	: 
//    create new cnf party
// Parameters:	
//    SrlDevice that requests this action
// Returns:	
//    bool true = success
//*****************************************************************************
SRL_DEVICE_HANDLE CnfBoard::OpenParty(PSrlDevice pDev){
  SRL_DEVICE_HANDLE rc;
  rc = cnf_OpenParty(m_srl_handle,0,0, pDev);
   LOG(RCHANDLE(rc), GetName(), "%d = cnf_OpenParty(0x%x,0, 0, &pDev(%s))",
                           rc,m_srl_handle,pDev->GetName());
 
    if (rc == CNF_ERROR) {
        process_cnf_error();
    }

 return rc;
} // End of OpenParty()

//*****************************************************************************
// Purpose	: 
//    close cnf party
// Parameters:	
//    SrlDevice that requests this action
// Returns:	
//    bool true = success
//*****************************************************************************
bool CnfBoard::CloseParty(SRL_DEVICE_HANDLE srl_handle){
bool brc = true;
int rc = cnf_CloseParty(srl_handle, 0);
   LOG(RC(rc), GetName(), "%d = cnf_CloseParty(0x%x,0)",
                           rc,m_srl_handle);
 
    if (rc == CNF_ERROR) {
        process_cnf_error();
        brc = false;
    }

 return brc;
} // End of CloseParty()

