//***********************************************************************
// Copyright  2005, Intel Corporation. 
// All rights reserved All names, products, and services mentioned  
// herein are the trademarks or registered trademarks of their respective 
// organizations and are the sole property of their respective owners
//***********************************************************************

// Conference.cpp: implementation of the CConference class.
//
//////////////////////////////////////////////////////////////////////
#include "Conference.h"
#include "AppLog.h"
#include "newscrn.h"
#include "utils.h"
#include "pdl.h"

#include <srllib.h>
#include <dtilib.h>
#include <msilib.h>
#include <dcblib.h>

using namespace std;


////////////////////////////////////////////////////////////////////n
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CConference::CConference()
	: m_hDsp(-1)
	, m_confId(-1)
	, m_isEchoCancelEnabled(false)
	, m_isDtmfEnabled(false)
	, m_bridgeNumber(-1)
	, m_monitorObject(NULL)
	, m_isMonitorInProgress(false)
	, m_monitorTxTimeslot(-1)
{
	m_partyList.clear();
	strncpy(m_muteDigits, "*6", 2);
}

CConference::~CConference()
{
	if(m_partyList.size() != 0) {
		deleteConference();
	}
}


//////////////////////////////////////////////////////////////////////
// Public methods
//////////////////////////////////////////////////////////////////////
// Called from CConfManger class

//*****************************************************************************
// Function	: 
//    CConference::init()
// Purpose	: 
//    Initializing conference variables and assignes dsp resource
// Parameters:	
//    Parm1 <int devH><direction>
// Return type:	
//    int
//*****************************************************************************
int CConference::init(int devH)
{
	m_hDsp = devH;
	m_confId = -1;
	return 0;
}


//*****************************************************************************
// 
// Function	: 
//    CConference::setConferenceFeatures()
// Purpose	: 
//    Just passes some Conf level parameters. Needed for run-time 
//    update of conf parameters 
// Parameters:	
//    Parm1 <bool echoCanceller><input>
//    Parm2 <bool dtmfEnabled><input>
//    Parm3 <const char* muteDig><input>
//    Parm4 <const char* unMuteDig><input>
// Return type:	
//    none
//*****************************************************************************
void CConference::setConferenceFeatures(bool echoCanceller, bool dtmfEnabled,const char* muteDig)
{
	if(m_confId != -1) { // the conference in progress
		updateConfFeatures(echoCanceller, dtmfEnabled);
	}
	else
	{
		m_isEchoCancelEnabled = echoCanceller;
		m_isDtmfEnabled = dtmfEnabled;
		strncpy(m_muteDigits, muteDig, 2);
	}
}


//*****************************************************************************
// Function	: 
//    CConference::addToConf()
// Purpose	: 
//    Adding the participant with that timeslot to the conference.
// Parameters:	
//    Parm1 <int channelNumber><input> - number of IP channel to be added to te conf
//    Parm2 <int timeslot_in><input> - channels CTBus Tx timeslot
//    Parm3 <int * timeslot_out><output> - Tx timeslot of this conference for this channel
// Return type:	
//    int
//*****************************************************************************
int CConference::addToConf(int channelNumber, int timeslot_in, int * timeslot_out)
{
	MS_CDT cdt;

	confParty * tmpParty = new confParty;
	tmpParty->cp_partId = m_partyList.size() + 1; // just an order number, 
												  //you may implement any other ID method 

	cdt.chan_num = timeslot_in; /* scts is the time slot... */
	cdt.chan_sel = MSPN_TS ; /* ...returned from getxmitslot() */
	cdt.chan_attr = m_isEchoCancelEnabled? MSPA_ECHOXCLEN|MSPA_MODEFULLDUPLX:
										   MSPA_MODEFULLDUPLX; 

	if(m_partyList.size() == 0) // first conferee
	{
		if (dcb_estconf(m_hDsp, &cdt, 1, MSCA_ND, &m_confId) == -1)
		{
			checkError("dcb_estconf");
			delete tmpParty;
			return -1;
		}
		sprintf(m_txt, "Created conf 0x%x on device %d", m_confId, m_hDsp);
		logOut(INFO, m_txt, 1);
		if(m_isDtmfEnabled)
		{
			if(dcb_setdigitmsk(m_hDsp, m_confId, CBMM_ALL, CBA_SETMSK) == -1)
				checkError("dcb_setdigitmsk()");
			else
				logOut(INFO, "dcb_setdigitmsk() = OK", 0);
		}
	}
	else
	{
		if(dcb_addtoconf(m_hDsp ,m_confId, &cdt) == -1)
		{
			checkError("dcb_addtoconf()");
			delete tmpParty;
			return -1;
		}
		sprintf(m_txt, "Add party #%d to conf 0x%x",m_partyList.size(), m_confId); 
		logOut(INFO, m_txt, 1);
	}

	tmpParty->cp_txTimeSlot = timeslot_in;
	tmpParty->cp_partChannelNumber = channelNumber;
	tmpParty->cp_digitCount = 0;
	tmpParty->cp_digBuf[0] = 0;
	tmpParty->cp_isRcvOnly = false;
	m_partyList.push_back(tmpParty);

	*timeslot_out = cdt.chan_lts;
	return 0;
}


//*****************************************************************************
// Function	: 
//    CConference::addMonitor()
// Purpose	: 
//    Adding the monitor (recorder)with that timeslot to the conference.
// Parameters:	
//    Parm1 <void * monitorObj><input> - pointer to IP device which voice will record the conf
//    Parm2 <int timeslot_in><input> - channels CTBus Tx timeslot
//    Parm3 <int * timeslot_out><output> - Tx timeslot of this conference for this resource
// Return type:	
//    int
//*****************************************************************************
int CConference::addMonitor(void * monitorObj, int timeslot_in, int * timeslot_out)
{
	if(m_isMonitorInProgress == true)
	{
		logOut(WARN, "The conference is already being monitored", 1);
		return -1;
	}
	if(m_partyList.size() == 0) // no conferees
	{
		logOut(WARN, "No participant in the conference to monitor", 1);
		return -1;
	}
	MS_CDT cdt;
	cdt.chan_num = timeslot_in; /* scts is the time slot... */
	cdt.chan_sel = MSPN_TS ; /* ...returned from getxmitslot() */
	cdt.chan_attr = MSPA_MODERECVONLY; 

	if(dcb_addtoconf(m_hDsp ,m_confId, &cdt) == -1)
	{
		checkError("dcb_addtoconf()");
		return -1;
	}
	sprintf(m_txt, "Add Monitor to conf 0x%x", m_confId); 
	logOut(INFO, m_txt, 1);
	m_monitorObject = monitorObj;
	m_isMonitorInProgress = true;
	m_monitorTxTimeslot = timeslot_in;
	*timeslot_out = cdt.chan_lts;
	return 0;
}

//*****************************************************************************
// Function	: 
//    CConference::removeFromConf()
// Purpose	: 
//    Removing the participant with that timeslot from the conference
// Parameters:	
//    Parm1 <int timeslot><direction>
// Return type:	
//    int
//*****************************************************************************
int CConference::removeFromConf(long txTimeslot)
{

	MS_CDT cdt;
	cdt.chan_num = txTimeslot; /* scts is the time slot... */
	cdt.chan_sel = MSPN_TS ; /* ...returned from getxmitslot() */
	
	if(dcb_remfromconf(m_hDsp ,m_confId, &cdt) == -1)
	{
		checkError("dcb_remfromcon");
		return -1;
	}
	else
	{
		sprintf(m_txt, "Removed ts #%ld from conf %#x",txTimeslot, m_confId);
		logOut(INFO, m_txt,1);
	}
// now remove the participant from the list:
	confParty * partyToRemove = findParticipant(txTimeslot, 0);
	if(partyToRemove)
	{
		m_partyList.remove(partyToRemove);
		delete partyToRemove;
	}
	else
		logOut(WARN, "Cannot remove party, not found", 1);
	return 0;
}

//*****************************************************************************
// Function	: 
//    CConference::removeMonitor()
// Purpose	: 
//    Removes the monitor from the conference
// Parameters:	
// Return type:	
//    void *, returns refernece to the monitor object
//*****************************************************************************
void * CConference::removeMonitor()
{
	if (!m_isMonitorInProgress) {
		return NULL;
	}
	else {
		if( m_monitorObject == NULL)
		{
			logOut(WARN, "Cannot stop the monitor, not found",1);
			return NULL;
		}
	}

	MS_CDT cdt;
	cdt.chan_num = m_monitorTxTimeslot; /* scts is the time slot... */
	cdt.chan_sel = MSPN_TS; 
	if(dcb_remfromconf(m_hDsp ,m_confId, &cdt) == -1) {
		checkError("removeMonitor()");
	}
	else
	{
		m_isMonitorInProgress = false;
		m_monitorTxTimeslot = -1;
		logOut(APP, "Monitor stopped on the conference",1);
	}
	return m_monitorObject;
}


//*****************************************************************************
// Function	: 
//    CConference::deleteConference()

// Purpose	: 
//    Deleting the conference with the conference ID of this object.
// Parameters:	
//    none
// Return type:	
//    int
//*****************************************************************************
int CConference::deleteConference()
{
	sprintf(m_txt,"deleting conf 0x%x  on device %d", m_confId, m_hDsp); 
	logOut(APP, m_txt, 1);
	if(dcb_delconf(m_hDsp, m_confId)) {
		checkError("dcb_delconf");
		return -1;
	}
	m_hDsp = -1;
	m_confId = -1;
	m_monitorObject = NULL;
	m_isMonitorInProgress = false;
// now clean up the list:
	if(!m_partyList.empty())
	{
		list<confParty *>::iterator it;
		confParty * tmp;
		for(it = m_partyList.begin(); it != m_partyList.end(); it++)
		{
			tmp = *it;
			delete tmp;
		}
		m_partyList.clear();
	}
	return 0;
}

//*****************************************************************************
// Function	: 
//    CConference::getResourceCount()
// Purpose	: 
//    Obtaining how many free resources we have on the specified dsp.
// Parameters:
//    none	
// Return type:	
//    int
//*****************************************************************************
int	CConference::getResourceCount()
{
	int res_cnt = -1;
	if(dcb_dsprescount(m_hDsp, &res_cnt) == 0) {
		return res_cnt;
	}
	checkError("dcb_dsprescount()");
	return -1;
}


//*****************************************************************************
// Function	: 
//    CConference::findParticipant()
// 
// Purpose	: 
//    retrieves a list element with cp_txTimeslot == txTimeslot or cp_partyId == partyId 
// 
// Parameters:	
//    Parm1 <long txTimeSlot><input>
//    Parm2 <int partyId><input>
// Return type:	
//    confParty * reference to party list element
//*****************************************************************************
confParty * CConference::findParticipant(long txTimeSlot, int partyId)
{
	list<confParty *>::iterator it;
	for(it = m_partyList.begin(); it != m_partyList.end(); it++)
	{
		if( ((*it))->cp_txTimeSlot == txTimeSlot) {
			return *it; // found
		}
	}
	return NULL; // not found
}

//*****************************************************************************
// 
// Function	: 
//    CConference::processDigit()
// Purpose	: 
//    Invoked when conferee presses any DTMF
// Parameters:	
//    Parm1 <int txTimeSlot><input> - defines the party
//    Parm2 <char digit><input> - DTMF pressed
// Return type:	
//    int
//*****************************************************************************
int CConference::processDigit(int txTimeSlot, char digit)
{
	sprintf(m_txt, "Received DTMF '%c' from TxSlot %d", digit, txTimeSlot);
	logOut(INFO, m_txt, 1);
	confParty * tmpParty = findParticipant(txTimeSlot, 0);
	if(tmpParty == 0)
	{
		sprintf(m_txt, "ProcDig:Cannot find party with txTs=%d", txTimeSlot);
		logOut(WARN, m_txt, 1); 
		return -1;
	}
	tmpParty->cp_digBuf[tmpParty->cp_digitCount] = digit;
	if(tmpParty->cp_digitCount == 0)
	{
		tmpParty->cp_digitCount++;
		return 0;  // go back to wait for the second digit
	}
	else  // this is the second digit:
	{
		if(strncmp(m_muteDigits, tmpParty->cp_digBuf, 2) == 0) // mute command
			setReceiveOnlyMode(tmpParty);
		tmpParty->cp_digitCount = 0;
	}

	return 0;
}


//////////////////////////////////////////////////////////////////////
// Private methods
//////////////////////////////////////////////////////////////////////

//*****************************************************************************
// Function	: 
//    CConference::logOut()
// Purpose	: 
//    Adding information over to the log file
// Parameters:	
//    Parm1 <int type><direction>
//    Parm2 <char * info><direction>
//    Parm3 <int windNumber><direction>
// Return type:	
//    none
//*****************************************************************************
void CConference::logOut(int type, const char * info, int printConsoleNum)
{
	char source[32];
	sprintf(source, "CONF_0x%x", m_confId);
	appLog->addToLog(type, source, info);
	printInfo(type, source, info, brdStatusWindow); 
}

//*****************************************************************************
// Function	: 
//    CConference::checkError()
// Purpose	: 
//    Checking error and obtaining the error message.
// Parameters:	
//    Parm1 <char * funcName><direction>
// Return type:	
//    none
//*****************************************************************************
void CConference::checkError(const char * funcName)
{
	char errMsg[128];
	sprintf(errMsg, "%s - %s", funcName, ATDV_ERRMSGP(m_hDsp));
	logOut(ERR1, errMsg, 1);
}

//*****************************************************************************
// 
// Function	: 
//    CConference::updateConfFeatures()
// Purpose	: 
//    Currently, updates only DTMF detection feature. May be extended 
// Parameters:	
//    Parm1 <bool echoCanceller><input>
//    Parm2 <bool dtmfEnabled><input>
// Return type:	
//    none
//*****************************************************************************
void CConference::updateConfFeatures(bool echoCanceller, bool dtmfEnabled)
{
//  The commeneted code below is working, but creates a quality problem, when
// Echo Canceler is enabled / disabled on a conference while the conference in progress
/*	
	MS_CDT cdt;
	cdt.chan_sel = MSPN_TS ; 
	list<confParty *>::iterator it;
	if(m_isEchoCancelEnabled != echoCanceller) // new value is different
	{    // for all participants set a new EC 
		for(it = m_partyList.begin(); it != m_partyList.end(); it++)	
		{
			cdt.chan_num = ((confParty *) (*it))->cp_txTimeSlot;
			if(dcb_getcde(m_hDsp, m_confId, &cdt) == -1) // retreive current setting
				checkError("dcb_getcde");
			else  // set a new one
			{
				if(echoCanceller)
					cdt.chan_attr = cdt.chan_attr|MSPA_ECHOXCLEN;
				else
					cdt.chan_attr = cdt.chan_attr^MSPA_ECHOXCLEN;
				if(dcb_setcde(m_hDsp, m_confId, &cdt) == -1)
					checkError("dcb_setcde()");
				else
					logOut(INFO, "EchoCanceler - feature updated", 1);
			}
		}
		m_isEchoCancelEnabled = echoCanceller;
	}
*/
	if(m_isDtmfEnabled != dtmfEnabled)
	{
		m_isDtmfEnabled = dtmfEnabled;
		int retVal;
		if(m_isDtmfEnabled)
			retVal = dcb_setdigitmsk(m_hDsp, m_confId, CBMM_ALL, CBA_SETMSK);
		else
			retVal = dcb_setdigitmsk(m_hDsp, m_confId, 0, CBA_SETMSK);
		if(retVal == -1)
			checkError("dcb_setdigitmsk()");
		else
			logOut(INFO, "dcb_setdigitmsk() = OK", 0);
	}
}


//*****************************************************************************
// 
// Function	: 
//    CConference::setReceiveOnlyMode()
// Purpose	: 
//    Mute/UnMute a party
// Parameters:	
//    Parm1 <confParty *party><input>
// Return type:	
//    none
//*****************************************************************************
void CConference::setReceiveOnlyMode(confParty *party)
{

	MS_CDT cdt;
	cdt.chan_sel = MSPN_TS ; 
	cdt.chan_num = party->cp_txTimeSlot;
//now saving current attributes:
	if(dcb_getcde(m_hDsp, m_confId, &cdt) == -1) { // retreive current setting
		checkError("dcb_getcde");
	}
	else  // set a new one
	{
		if(!party->cp_isRcvOnly) // mute this party
		{
			cdt.chan_attr = cdt.chan_attr^MSPA_MODEFULLDUPLX;
			cdt.chan_attr = cdt.chan_attr|MSPA_MODERECVONLY;
			sprintf(m_txt, "Party %d on Chn %d muted", party->cp_partId, party->cp_partChannelNumber);  
			party->cp_isRcvOnly = true;
		}
		else
		{
			cdt.chan_attr = cdt.chan_attr^MSPA_MODERECVONLY;
			cdt.chan_attr = cdt.chan_attr|MSPA_MODEFULLDUPLX;
			sprintf(m_txt, "Party %d on Chn %d un-muted", party->cp_partId, party->cp_partChannelNumber);  
			party->cp_isRcvOnly = false;
		}
//		set new RcvOnly mode:
		if(dcb_setcde(m_hDsp, m_confId, &cdt) == -1)
			checkError("dcb_setcde()");
		else
			logOut(INFO, m_txt, 1);
	}
}
//////////////////////////////////////////////////////////////////////
// End of Conference.cpp
//////////////////////////////////////////////////////////////////////
