/**
* @file config.cpp
* @brief Multimedia demo config file utility
* @date Feb 16, 2007
*
* DIALOGIC CONFIDENTIAL 
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to 
* the source code ("Material") are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its 
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced, 
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual 
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/


// System Level Include Files
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

// Application Level Include Files
#include "config.h"
#include "miscellaneous.h"

#include <iostream>
#include <sstream>

#ifdef WIN32
#define strncasecmp _strnicmp
#define strcasecmp strcmpi
#endif

// Global Variables
static const int max_line_size  = 256;
static const int max_token_size = 132;

extern char g_cAVFileDir[256];
extern int g_nPlatformNameRelease;

// Funcition Descriptions
unsigned char AsciiOctetToHex(const char *pOctet)
{
   unsigned char hexDigit = 0;
   unsigned char value = 0;

   for (unsigned int numDigits = 0; numDigits < 2; numDigits++, pOctet++)
   {
      if (*pOctet >= '0' && *pOctet <= '9')
         hexDigit = *pOctet - '0';
      // lineBuffer was converted to upper case, but for flexibility in reuse:
      else if (*pOctet >= 'a' && *pOctet <= 'f')
         hexDigit = (*pOctet - 'a') + 10;
      else if (*pOctet >= 'A' && *pOctet <= 'F')
         hexDigit = (*pOctet - 'A') + 10;

      value =  (value << 4) + hexDigit;
   }

   return value;
}


// Funcition Descriptions

#ifdef USE_RTSP
int ParseRTSPInfo(LP_CONFIGFILEOPTIONS lpConfigFileOpts, std::string& lineStr)
{
   char dtmf = '0';
   char command[255] = {'\0'};
   char rtspUrl[255] = {'\0'};

   std::istringstream istrLine(lineStr);

   if (istrLine >> command >> dtmf >> rtspUrl)
   {
      lpConfigFileOpts->m_rtspUrls[dtmf] = rtspUrl;
      std::cerr << "RTSP URL, dtmf '"<< dtmf << "', URL = '"<< rtspUrl <<"'" << std::endl;
   }
   else
      return -1;

   return 0;
}
#endif

//*****************************************************************************
// NAME : void LogReportLine(line)
// DESCRIPTION : a line
// INPUT : line - pointer to line being logged, terminated with NULL
// RETURNS : true if successful, else false
//*****************************************************************************
void LogReportLine(int lineNumber, char *line)
{

   //Print each line to the log file.
    IGNORE_WHITE_SPACE(line)                // ignores leading white space
    if ((*line != '#') && (*line != ';') && (*line != '!') && (*line != '$') && (*line != '%')) {
        mmReport(INFO_MSG, NONE, "[%d] %s", lineNumber, line);
        return;                            
    }
    return;
}
//*****************************************************************************
// 		  NAME : bool ProcessConfigFile(lpConfigFileOpts, configFile)
// DESCRIPTION : Parses config file
// 		 INPUT : NA
// 	    OUTPUT : lpConfigFileOpts - pointr to options to fill out
// 			     configFile - config file to use
// 	   RETURNS : true if successful, else false
// 	  CAUTIONS : None
//*****************************************************************************
bool ProcessConfigFile(LP_CONFIGFILEOPTIONS lpConfigFileOpts, const char *configFile)
{
   int     lineNumber                  = 0;       
   bool    retCode                         = true; 
   FILE    *configFp;
   char    line[max_line_size];
   char    optionToken[max_token_size];
   char    valueToken[max_token_size];
   char    *cPtr;

   // Set up defaults for command line options.
   lpConfigFileOpts->unConnections     = 1;
   lpConfigFileOpts->nNumberOfCalls    = 1;
   lpConfigFileOpts->bTraceFlag        = false;
   lpConfigFileOpts->nTraceLevel        = 0;
   lpConfigFileOpts->bOutboundFlag     = false;    // App is always inbound
   lpConfigFileOpts->bUseAliasTable    = true;
   lpConfigFileOpts->interCallDelay    = 5;        // Time in seconds
   lpConfigFileOpts->callDuration      = 5;        // Time in seconds
   lpConfigFileOpts->nStartSpan        = 1;
   lpConfigFileOpts->nStartIpmChan     = 1;
   lpConfigFileOpts->nCallType         = CALL_TYPE_H323;
   lpConfigFileOpts->bRunPstnOnly      = false;
   lpConfigFileOpts->bWaitOnSIPAck     = false; //Wait for SIP ACK to Start Media
   lpConfigFileOpts->cAudioTranscodeDir[0] = '\0';
   lpConfigFileOpts->cVideoTranscodeDir[0] = '\0';
   lpConfigFileOpts->bUseSipInfoDtmf   = false;
   lpConfigFileOpts->nDtmfDetectMode   = dtmfMode_inband;  //inband will detect RFC2833 & inband
   lpConfigFileOpts->bRepeatMenus      = false;
   lpConfigFileOpts->rxdciCifSize           = 0;
   memset(lpConfigFileOpts->rxdciCifStrValue,0,sizeof(unsigned char));
   memset(lpConfigFileOpts->rxdciCifValue,0,sizeof(unsigned char));
   lpConfigFileOpts->rxdciQcifSize           = 0;
   memset(lpConfigFileOpts->rxdciQcifStrValue,0,sizeof(unsigned char));
   memset(lpConfigFileOpts->rxdciQcifValue,0,sizeof(unsigned char));
   lpConfigFileOpts->txdciQcifSize           = 0;
   memset(lpConfigFileOpts->txdciQcifStrValue,0,sizeof(unsigned char));
   memset(lpConfigFileOpts->txdciQcifValue,0,sizeof(unsigned char));
   lpConfigFileOpts->txdciCifSize           = 0;
   memset(lpConfigFileOpts->txdciCifStrValue,0,sizeof(unsigned char));
   memset(lpConfigFileOpts->txdciCifValue,0,sizeof(unsigned char));
   strcpy(lpConfigFileOpts->cAudioFileFormat,"dmf");
   strcpy(lpConfigFileOpts->cVideoFileFormat,"dmf");
   strcpy(lpConfigFileOpts->cVideoTranscodeDir,"native");
   strcpy(lpConfigFileOpts->cAudioTranscodeDir,"native");
   strcpy(lpConfigFileOpts->cAVFileDir,"../avfiles");
   lpConfigFileOpts->IFrameInterval        = 0;   // 0 = off (seconds)

   lpConfigFileOpts->bUseRecordMsg       = false; //Play Msg while Recording
   lpConfigFileOpts->bRecordAudioOnly    = false; //Video call record only audio
   lpConfigFileOpts->nRecordTimeout      = 0;     // 0 = off (seconds)
   lpConfigFileOpts->nMsgRepeatInterval  = 0;     // 0 = off (seconds)

#ifdef TDM_AUDIO
   strcpy(lpConfigFileOpts->cISDNGatewayIP, "127.0.0.1");
   lpConfigFileOpts->nISDNGatewayPort = 40121;
#endif
   strcpy(lpConfigFileOpts->cDestIp, "127.0.0.1");
   strcpy(lpConfigFileOpts->cProxyIp, "0.0.0.0");  // Indicates no Gatekeeper
   strcpy(lpConfigFileOpts->cPhoneNumber, "18005551001");


   lpConfigFileOpts->recordControl     = RECORDING_DEFAULT;
   lpConfigFileOpts->h264SdpPacketizationMode      = 1;
   lpConfigFileOpts->h264SdpProfileLevel      = -1;

   // Open the config file for reading
   configFp = fopen(configFile, "r");
   if (!configFp) {
      printf("Unable to open %s for read\n", configFile);
      perror("due to");
      return(false);       
   }

   // Read and parse the config file
   while (fgets(line, max_line_size, configFp) != NULL) {
      // Go to the next line
      lineNumber++;        

      // Verify the line is not too long - i.e. it has a new line code
      // And if so, change it to a NUL
      for (cPtr = line; *cPtr; cPtr++) {
         if ((*cPtr == 015) || (*cPtr == 012)) {
            break;
         }
      }
      if ((*cPtr != 015) && (*cPtr != 012)) {
         printf("Long line encountered at line number %d, returning\n", lineNumber);
         return false;                       
      }
      *cPtr = '\0';

      //Print active cfg file lines to log file
      LogReportLine(lineNumber,line);

      if (TokenizeLine(line, optionToken, valueToken) == false) {
         printf("Line number %d has a field which is too long\n", lineNumber);
         retCode = false;
         continue;
      }
      if (optionToken[0] == '\0') {                // A comment line
         continue;
      }

      // Process the specific options
      if (strncasecmp(optionToken, "numberofcalls", strlen("numberofcalls")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->nNumberOfCalls = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 
     
      else if (strncasecmp(optionToken, "startingspan", strlen("startingspan")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->nStartSpan = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 
      
      else if (strncasecmp(optionToken, "startingipmchan", strlen("startingipmchan")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->nStartIpmChan = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 

      else if (strncasecmp(optionToken, "intercalldelay", strlen("intercalldelay")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->interCallDelay = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 
      
      else if (strncasecmp(optionToken, "destip", strlen("destip")) == 0) {
         if (VerifyTokenIsIPAddress(valueToken) == true) {
            strcpy(lpConfigFileOpts->cDestIp, valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value to be an IP4 IP address\n", lineNumber);
         }
      } 
      
      else if (strncasecmp(optionToken, "proxyip", strlen("proxyip")) == 0) {
         if (VerifyTokenIsIPAddress(valueToken) == true) {
            strcpy(lpConfigFileOpts->cProxyIp, valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value to be an IP4 IP address\n", lineNumber);
         }
      } 
    
      else if (strncasecmp(optionToken, "outboundcalls", strlen("outboundcalls")) == 0) {
         lpConfigFileOpts->bOutboundFlag = true;
      } 

      else if (strncasecmp(optionToken, "phonenumber", strlen("phonenumber")) == 0) {
         if (VerifyTokenIsPhoneNumber(valueToken) == true) {
            strcpy(lpConfigFileOpts->cPhoneNumber, valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be a valid phone number\n", lineNumber);
         }
      } 
      
      else if (strncasecmp(optionToken, "runpstnonly", strlen("runpstnonly")) == 0) {
         lpConfigFileOpts->bRunPstnOnly = true;
      } 

      else if (strncasecmp(optionToken, "usealiastable", strlen("usealiastable")) == 0) {
         if (strcasecmp(valueToken, "false") == 0) {
            lpConfigFileOpts->bUseAliasTable = false;
         } else {
            lpConfigFileOpts->bUseAliasTable = true;
         }
      } 

      else if (strncasecmp(optionToken, "connections", strlen("connections")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->unConnections = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 
      
      else if (strncasecmp(optionToken, "sipudpport", strlen("sipudpport")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->nSipUdpPort = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 
      
      else if (strncasecmp(optionToken, "callduration", strlen("callduration")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->callDuration = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 
      
      else if (strncasecmp(optionToken, "h264sdppacketizationmode", strlen("h264packetizationmode")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->h264SdpPacketizationMode = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 
      
      else if (strncasecmp(optionToken, "h264sdpprofilelevel", strlen("h264sdpprofilelevel")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->h264SdpProfileLevel = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 

      else if (strncasecmp(optionToken, "recordcontrol", strlen("recordcontrol")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->recordControl = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      } 
      
      else if(strncasecmp(optionToken, "maxbandwidth",strlen("maxbandwidth"))==0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->maxBandwidth = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an long integer\n", lineNumber);
         }
      } 
      
      else if(strncasecmp(optionToken, "maxFrameRate",strlen("maxFrameRate"))==0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->maxFrameRate = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an long integer\n", lineNumber);
         }
      } 
      
      else if (strncasecmp(optionToken, "videoResolution", strlen("videoResolution")) == 0) {
         strcpy(lpConfigFileOpts->videoRes,valueToken);
      } 

      else if (strncasecmp(optionToken, "videoCodec", strlen("videoCodec")) == 0) {
         strcpy(lpConfigFileOpts->videoCodec,valueToken);
         if(( (strncasecmp(valueToken,"MPEG4",5)==0) || (strncasecmp(valueToken,"H264",4)==0) ) && 
              (g_nPlatformNameRelease==HMP_3_1 || g_nPlatformNameRelease==HMP_1_5 || g_nPlatformNameRelease==OVL_1_0) ) {
            printf("ERROR: Can't continue  Codec %s is not supported on current platform\n", valueToken);
            exit(-1);
         }
      } 

      else if (strncasecmp(optionToken, "audioCodec", strlen("audioCodec")) == 0) {
         strcpy(lpConfigFileOpts->audioCodec,valueToken);
      } 

      else if (strncasecmp(optionToken, "rxdciqcif", strlen("rxdciqcif")) == 0) {
         unsigned char size=(char)strlen(valueToken)/2;
         lpConfigFileOpts->rxdciQcifSize = size;
         strcpy(lpConfigFileOpts->rxdciQcifStrValue,valueToken);

         // convert RxDCI from ASCII to binary:
         for (unsigned int i = 0; i <size; i++)
         {
            lpConfigFileOpts->rxdciQcifValue[i] = AsciiOctetToHex(&valueToken[2*i]);
         }  
      } 

      else if (strncasecmp(optionToken, "rxdciCif", strlen("rxdciCif")) == 0) {
         unsigned char size=(char)strlen(valueToken)/2;
         lpConfigFileOpts->rxdciCifSize = size;
         strcpy(lpConfigFileOpts->rxdciCifStrValue,valueToken);

         // convert TxDCI from ASCII to binary:
         for (unsigned int i = 0; i <size; i++)
         {
            lpConfigFileOpts->rxdciCifValue[i] = AsciiOctetToHex(&valueToken[2*i]);
         }
      }
 
      else if (strncasecmp(optionToken, "txdciqcif", strlen("txdciqcif")) == 0) {
         unsigned char size=(char)strlen(valueToken)/2;
         //printf("Config TxDCI value = %s and size=%d\n",valueToken,size);
          
         lpConfigFileOpts->txdciQcifSize = size;
         strcpy(lpConfigFileOpts->txdciQcifStrValue,valueToken);

         // convert TxDCI from ASCII to binary:
         for (unsigned int i = 0; i <size; i++)
         {
            lpConfigFileOpts->txdciQcifValue[i] = AsciiOctetToHex(&valueToken[2*i]);
         }
      }
  
      else if (strncasecmp(optionToken, "txdciCif", strlen("txdciCif")) == 0) {
         unsigned char size=(char)strlen(valueToken)/2;
         //printf("Config TxDCI value = %s and size=%d\n",valueToken,size);

         lpConfigFileOpts->txdciCifSize = size;
         strcpy(lpConfigFileOpts->txdciCifStrValue,valueToken);

         // convert TxDCI from ASCII to binary:
         for (unsigned int i = 0; i <size; i++)
         {
            lpConfigFileOpts->txdciCifValue[i] = AsciiOctetToHex(&valueToken[2*i]);
         }     
      } 
      
      else if (strncasecmp(optionToken, "baseavfilepath", strlen("baseavfilepath")) == 0) {
         strcpy(lpConfigFileOpts->cAVFileDir, valueToken);
      } 
      
      else if (strncasecmp(optionToken, "traceflag", strlen("traceflag")) == 0) {
         if (strcasecmp(valueToken, "false") == 0) {
            lpConfigFileOpts->bTraceFlag = false;
         } else {
            lpConfigFileOpts->bTraceFlag = true;
         } 
      } 
      
      else if (strncasecmp(optionToken, "tracelevel", strlen("tracelevel")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->nTraceLevel = atoi(valueToken);
         }
      } 
       
      else if (strncasecmp(optionToken, "calltype", strlen("calltype")) == 0) {
#ifdef WIN32
         if (stricmp(valueToken, "sip") == 0)
#else
         if (strcasecmp(valueToken, "sip") == 0)
#endif
         {
            lpConfigFileOpts->nCallType = CALL_TYPE_SIP;
         }

#ifdef WIN32
         else if (stricmp(valueToken, "h323") == 0)
#else
         else if (strcasecmp(valueToken, "h323") == 0)
#endif
         {
            lpConfigFileOpts->nCallType = CALL_TYPE_H323;
         }
      } 
      
      else if (strncasecmp(optionToken, "waitonsipack", strlen("waitonsipack")) == 0) {
         if (strcasecmp(valueToken, "false") == 0) {
            lpConfigFileOpts->bWaitOnSIPAck = false;
         } else {
            lpConfigFileOpts->bWaitOnSIPAck = true;
         }
      }

      else if (strncasecmp(optionToken, "iframeinterval", strlen("iframeinterval")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->IFrameInterval = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      }

#ifdef TDM_AUDIO
      else if (strncasecmp(optionToken, "isdngatewayip", strlen("isdngatewayip")) == 0) {
         strcpy(lpConfigFileOpts->cISDNGatewayIP, valueToken);
      } 

      else if (strncasecmp(optionToken, "isdngatewayport", strlen("isdngatewayport")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
	    lpConfigFileOpts->nISDNGatewayPort = atoi(valueToken);
	 } else {
	    retCode = false;
	    printf("Line number %d expects value field to be an integer\n", lineNumber);
	 }
      }
#else
      else if (strncasecmp(optionToken, "isdngatewayip", strlen("isdngatewayip")) == 0) {
          // Do nothing, but allow this option in the config file
      }

      else if (strncasecmp(optionToken, "isdngatewayport", strlen("isdngatewayport")) == 0) {
	   // Do nothing, but allow this option in the config file
      }
#endif
      else if (strncasecmp(optionToken, "usesipinfodtmf", strlen("usesipinfodtmf")) == 0) {
         if (strcasecmp(valueToken, "false") == 0) {
            lpConfigFileOpts->bUseSipInfoDtmf = false;
         } else {
            lpConfigFileOpts->bUseSipInfoDtmf = true;
         }
      }
 
      else if (strncasecmp(optionToken, "dtmfdetectmode", strlen("dtmfdetectmode")) == 0) {
         if(strcasecmp(valueToken, "inband") == 0) {
            lpConfigFileOpts->nDtmfDetectMode = dtmfMode_inband;
         } 
         else if(strcasecmp(valueToken, "rfc2833") == 0) {
            lpConfigFileOpts->nDtmfDetectMode = dtmfMode_rfc2833;
         }
         else if(strcasecmp(valueToken, "sipinfo") == 0) {
            lpConfigFileOpts->nDtmfDetectMode = dtmfMode_sipinfo;
         } 
         else if(strcasecmp(valueToken, "rtc") == 0) {
            lpConfigFileOpts->nDtmfDetectMode = dtmfMode_rtc;
            printf("Unsupported Demo value [%s] in config file\n",lpConfigFileOpts->nDtmfDetectMode);
            exit(-1);
         } 
      }
        
      else if (strncasecmp(optionToken, "repeatmenus", strlen("repeatmenus")) == 0) {
         if (strcasecmp(valueToken, "false") == 0) {
            lpConfigFileOpts->bRepeatMenus = false;
         } else {
            lpConfigFileOpts->bRepeatMenus = true;
         }
      }
        
      else if(strncasecmp(optionToken, "audiofileformat", strlen("audiofileformat")) == 0) {
         strcpy(lpConfigFileOpts->cAudioFileFormat, valueToken);
      }
 
      else if(strncasecmp(optionToken, "videofileformat", strlen("videofileformat")) == 0) {
         strcpy(lpConfigFileOpts->cVideoFileFormat, valueToken);
      }
 
      else if (strncasecmp(optionToken, "audioxcodedir", strlen("audioxcodedir")) == 0) {
         strcpy(lpConfigFileOpts->cAudioTranscodeDir,valueToken);
         if( (strcasecmp(valueToken, "native")!=0) && (strcasecmp(valueToken, "tx")!=0) && (strcasecmp(valueToken, "rx")!=0) && (strcasecmp(valueToken, "txrx")!=0) && (strcasecmp(valueToken, "rxtx")!=0)) {
            printf("Invalid value [%s] for the field audioxcodedir in config file\n",lpConfigFileOpts->cAudioTranscodeDir);
            exit(-1);
         }
      }

      else if (strncasecmp(optionToken, "videoxcodedir", strlen("videoxcodedir")) == 0) {
         strcpy(lpConfigFileOpts->cVideoTranscodeDir,valueToken);
         if (strcasecmp(valueToken, "native") != 0) {
            if( g_nPlatformNameRelease==HMP_3_1 || g_nPlatformNameRelease==HMP_1_5 || g_nPlatformNameRelease==OVL_1_0 ) {
               printf("ERROR: Can't continue transcoding is not supported on current platform\n");
               exit(-1);
            }
//          printf("Transcode enabled direction=%s\n",lpConfigFileOpts->cVideoTranscodeDir);
         }
         if( (strcasecmp(valueToken, "native")!=0) && (strcasecmp(valueToken, "tx")!=0) && (strcasecmp(valueToken, "rx")!=0) && (strcasecmp(valueToken, "txrx")!=0) && (strcasecmp(valueToken, "rxtx")!=0) ) {
            printf("Invalid value [%s] for the field videoxcodedir in config file\n",lpConfigFileOpts->cVideoTranscodeDir);
            exit(-1);
         }
      }


      else if (strncasecmp(optionToken, "rtspinfo", strlen("rtspinfo")) == 0) {
#ifdef USE_RTSP
         std::string inputLine(line);
         ParseRTSPInfo(lpConfigFileOpts,inputLine);
#endif
      }

      else if (strncasecmp(optionToken, "userecordmsg", strlen("userecordmsg")) == 0) {
         if (strcasecmp(valueToken, "false") == 0) {
            lpConfigFileOpts->bUseRecordMsg = false;
         } else {
            lpConfigFileOpts->bUseRecordMsg = true;
         }
      }

      else if (strncasecmp(optionToken, "recordaudioonly", strlen("recordaudioonly")) == 0) {
         if (strcasecmp(valueToken, "false") == 0) {
            lpConfigFileOpts->bRecordAudioOnly = false;
         } else {
            lpConfigFileOpts->bRecordAudioOnly = true;
         }
      }
      else if (strncasecmp(optionToken, "recordtimeout", strlen("recordtimeout")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->nRecordTimeout = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      }
      else if (strncasecmp(optionToken, "msgrepeatinterval", strlen("msgrepeatinterval")) == 0) {
         if (VerifyTokenIsInteger(valueToken) == true) {
            lpConfigFileOpts->nMsgRepeatInterval = atoi(valueToken);
         } else {
            retCode = false;
            printf("Line number %d expects value field to be an integer\n", lineNumber);
         }
      }

      else {
         retCode = false;
         printf("Line number %d is not a valid option\n", lineNumber);
      }
   }   
   // close the input file
   fclose(configFp);
   return(retCode);
}

//*****************************************************************************
// 		  NAME : bool TokenizeLine(line, token1, token2)
// DESCRIPTION : Tokenizes a line
// 		 INPUT : line - pointer to line being tokenized, terminated with NULL
// 	    OUTPUT : token1 - destination for token 1 
// 				 token2 - destination for token 2
// 	   RETURNS : true if successful, else false(token is too long)
// 				 If true and token1 == "" then this is a comment line
// 	  CAUTIONS : Assumes token1, token2 have at least max_token_size bytes
//*****************************************************************************
bool TokenizeLine(char *line, char *token1, char *token2)
{
    int     tokenSize;

    memset(token1, '\0', max_token_size);
    memset(token2, '\0', max_token_size);

    IGNORE_WHITE_SPACE(line)                // ignores leading white space
    if ((*line == '#') || (*line == ';') || (*line == '!')
        || (*line == '$') || (*line == '%')) {
        return true;                            // Comment line return code
    }

    // get the first token
    for (tokenSize = 0;
        (*line > ' ') && (tokenSize < max_token_size) && (*line != '=');
        tokenSize++) {
        *token1++ = tolower(*line);
        *line++;
    }
    if (tokenSize == max_token_size) {
        return false;                           // Invalid token
    }

    IGNORE_WHITE_SPACE(line)                // ignores white space between the 1st & 2nd tokens
    // Get the second token
    for (tokenSize = 0;
        (*line > ' ') && (tokenSize < max_token_size);
        tokenSize++) {
        *token2++ = *line++;
    }
    if (tokenSize == max_token_size) {
        return false;                           // Invalid token
    }
    return true;
}

//*****************************************************************************
// 		  NAME : bool VerifyTokenIsInteger(token)
// DESCRIPTION : Verifies if the token is an integer string
// 		 INPUT : token - pointer to token to check
// 	    OUTPUT : None
// 	   RETURNS : True if yes, Else false
// 	  CAUTIONS : None
//*****************************************************************************
bool VerifyTokenIsInteger(char *token)
{
    if (*token == '\0') {           // Verify at least one integer
        return false;                   // ERRROR RETURN POINT
    }

    while (*token) {
        if (!isdigit(*token)) {
            return false;               // ERRROR RETURN POINT
        }
        token++;                        // bump to the next char
    }
    return true;
}

//*****************************************************************************
// 		  NAME : bool VerifyTokenIsIPAddress(token)
// DESCRIPTION : Verifies token is an IP Address
// 		 INPUT : token - pointer to token to check
// 	    OUTPUT : NA
// 	   RETURNS : True if yes, else False
// 	  CAUTIONS : None
//*****************************************************************************
bool VerifyTokenIsIPAddress(char *token)
{
    char    field1[max_token_size];
    char    field2[max_token_size];
    char    field3[max_token_size];
    char    field4[max_token_size];
    char    *cPtr;

    // verify not too long
    if (strlen(token) >= max_ip_size - 1) {
        return false;
    }

    // Note: if unexpectedly we are missing a . then the next field will be empty
    cPtr = field1;
    while ((*token != '.') && *token) {
        *cPtr++ = *token++;
    }
    *cPtr = '\0';
    if (VerifyTokenIsInteger(field1) == false) {
        return false;
    }

    // bump by the .
    if (*token == '.') {
        token++;
    }

    cPtr = field2;
    while ((*token != '.') && *token) {
        *cPtr++ = *token++;
    }
    *cPtr = '\0';
    if (VerifyTokenIsInteger(field2) == false) {
        return false;
    }

    // bump by the .
    if (*token == '.') {
        token++;
    }
    cPtr = field3;
    while ((*token != '.') && *token) {
        *cPtr++ = *token++;
    }
    *cPtr = '\0';
    if (VerifyTokenIsInteger(field3) == false) {
        return false;
    }

    // bump by the .
    if (*token == '.') {
        token++;
    }

    cPtr = field4;
    while (*token) {
        *cPtr++ = *token++;
    }
    *cPtr = '\0';
    if (VerifyTokenIsInteger(field4) == false) {
        return false;
    }
    return true;
}

/*****************************************************************************
// 		  NAME : bool VerifyTokenIsPhoneNumber(token)
// DESCRIPTION : Verifies token is a PSTN phone number
// 		 INPUT : token - pointer to token to check
// 	    OUTPUT : NA
// 	   RETURNS : True if yes, else False
// 	  CAUTIONS : 
/****************************************************************************/
bool VerifyTokenIsPhoneNumber(char *token)
{
    // verify not too long
    if (strlen(token) >= max_phone_number_size - 1) {
        return false;
    }

    // verify combination of digits & '-'
    while (*token) {
        if (!isdigit(*token) && (*token != '-')) {
            return false;                               
        }
        token++;
    }
    return true;
}

/*****************************************************************************
// 		  NAME : bool TokenCmp(token, str)
// DESCRIPTION : Verifies token with  string passed in
// 		 INPUT : token - pointer to token to check
//               str - string to verify token with
// 	    OUTPUT : NA
// 	   RETURNS : true if yes, else false
// 	  CAUTIONS : 
/****************************************************************************/
bool TokenCmp(char* token, char* str)
{
    return(strncasecmp(token, str, strlen(str)) == 0 ? false : true);
}


