/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* Copyright (C) 2001-2011 Dialogic Corporation. All Rights Reserved.
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
* 1.    Redistributions of source code must retain the above copyright
* notice, this list of conditions and the following disclaimer.
*
* 2.    Redistributions in binary form must reproduce the above copyright
* notice, this list of conditions and the following disclaimer in
* the documentation and/or other materials provided with the
* distribution.
*
* 3.    Neither the name Dialogic nor the names of its
* contributors may be used to endorse or promote products derived from this
* software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
/**
 * @file hsi_interface.h
 * @brief Defines Host Streaming Interface (HSI) for Windows Drivers.
 * @date  April 17, 2006
 *
 */


#ifndef __HSI_INTERFACE_H__
#define __HSI_INTERFACE_H__




/* Magic Numbers */
#define HSI_MAX_BRIDGES             ( 32 )
#define HSI_MAX_BUFFERS             ( 32 )
#define HSI_FLAGS_DIRECTION_MASK    ( 0x1 )
#define HSI_FLAGS_ROUTING_MASK      ( 0x2 )
#define HSI_FLAGS_HOST_TO_BOARD     ( 0x0 << 0 )
#define HSI_FLAGS_BOARD_TO_HOST     ( 0x1 << 0 )
#define HSI_FLAGS_ROUTE             ( 0x1 << 1 )
#define HSI_FLAGS_UNROUTE           ( 0x0 << 1 )




/* IOCTL Interface Definitions */
#if defined( _WIN32 ) || defined( WIN32 ) || defined( __WINDOWS__ )

    #define HSI_IOCTL_PACK_CODE( Dev, Code )       ( CTL_CODE( Dev | 0xE000, Code | 0x800, METHOD_BUFFERED, FILE_ANY_ACCESS ))

#else

	#if defined( KMBC_USER_MODE )
		#include <sys/ioctl.h>
	#else
		#include <linux/ioctl.h>
	#endif
    
	#define HSI_IOCTL_PACK_CODE( Dev, Code )       ( _IO( Dev, Code ))

#endif  // #if defined( KMBC_TARGET_WIN32 )


#define HSI_FILE_DEVICE						0x53
#define HSI_BRIDGE_INIT_IOCTL				HSI_IOCTL_PACK_CODE( HSI_FILE_DEVICE, 0x1 )
#define HSI_BRIDGE_STOP_IOCTL				HSI_IOCTL_PACK_CODE( HSI_FILE_DEVICE, 0x2 )
#define HSI_BRIDGE_ROUTE_IOCTL			HSI_IOCTL_PACK_CODE( HSI_FILE_DEVICE, 0x3 )
#define HSI_INTR_RATE_ENABLE_IOCTL		HSI_IOCTL_PACK_CODE( HSI_FILE_DEVICE, 0x4 )
#define HSI_INTR_RATE_DISABLE_IOCTL		HSI_IOCTL_PACK_CODE( HSI_FILE_DEVICE, 0x5 )




/* Define Callback Function data types */
typedef void (*HSI_ISR_CALLBACK)( int bridgeId, int hsiBufferIndex );
typedef void (*HSI_SYNC_CALLBACK)( int bridgeId );




/**
 * @brief Enumerates available start modes for a Bridge Device.
 *
 * When initializing a Bridge Device, the KMBC can specify one of two possible
 * start modes.  The first mode, HSI_START_ON_INTR_ENABLE, tells the Bridge
 * Device to begin accumulating data for transfer when the KMBC sends a subsequent
 * request to initialize the rate interrupt on that Bridge Device.  The other
 * mode, HSI_START_ON_SYNC, is sent by the KMBC to tell the Bridge Device that
 * it should begin accumulating data for transfer when the it detects a
 * synchronization pattern on the global CT bus.
 */
typedef enum _tagHSI_START_TYPE
{
	HSI_START_ON_INTR_ENABLE = 0,
	HSI_START_ON_SYNC        = 1
}
HSI_START_TYPE, *PHSI_START_TYPE;




/**
 * @brief Enumerates available converstion modes for the bus.
 *
 * There are three modes that can be used to configure the bus law conversion
 * on a Bridge Device.  The bus can be configured to operate in transparent
 * mode, a-law mode, or mu-law mode.
 */
typedef enum _tagHSI_BUS_TYPE
{
	HSI_BUS_TYPE_TRANSPARENT = 0,
	HSI_BUS_TYPE_ALAW        = 1,
	HSI_BUS_TYPE_MULAW       = 2
}
HSI_BUS_TYPE, *PHSI_BUS_TYPE;




/**
 * @brief Enumerates available interrupt rates for the Bridge Device.
 *
 * The KMBC can configure a Bridge Device to operate at one of two frequencies,
 * either 2 ms or 4 ms.
 */
typedef enum _tagHSI_TICK_RATE
{
	HSI_RATE_2MS = 2,
	HSI_RATE_4MS = 4
}
HSI_TICK_RATE, *PHSI_TICK_RATE;




/**
 * @brief Enumerates message response types.
 *
 * These response types are only used on Linux.  They are not necessary on Windows.
 */
typedef enum _tagHSI_RESPONSE_TYPE
{
	HSI_BRIDGE_INIT_COMPLETION = 0,
	HSI_BRIDGE_ROUTE_COMPLETION,
	HSI_BRIDGE_STOP_COMPLETION,
	HSI_BRIDGE_INTR_ENABLE_COMPLETION,
	HSI_BRIDGE_INTR_DISABLE_COMPLETION
}
HSI_RESPONSE_TYPE, *PHSI_RESPONSE_TYPE;




/**
 * @brief Enumerates Stream Allocation Methods.
 *
 * The interface now supports stream grouping to allow a mixed-mode of operation where
 * some channels are configured to operate in transparent mode (data), and others are
 * configured to run in voice mode (will use law conversion).  Multiple methods are
 * required because some types of boards (namely the Totowa-1) use a chip that has a
 * bug that will cause corruption on certain channels in mixed mode.  Boards which use
 * the ASPEN chip to manage the HSI interface should use the Totowa-1 port allocation
 * method.  This method places an unused channel in between each group.  The alternative
 * method leaves no gaps between stream groups.  In both cases, Stream 0 is reserve
 */
typedef enum _tagHSI_PORT_ALLOCATION_METHOD
{
	HSI_ALLOCATION_METHOD_0 = 0,  ///< Leave unused channel between each stream group
	HSI_ALLOCATION_METHOD_1,      ///< No unused channels between each stream group
	HSI_ALLOCATION_METHOD_DYNAMIC ///< Channels can be configured on a per channel basis at runtime - not supported yet
}
HSI_PORT_ALLOCATION_METHOD, *PHSI_PORT_ALLOCATION_METHOD;




/**
 * @brief Specifies parameters for a single route request.
 *
 * The KMBC is capable of sending multiple route requests to a Bridge Device
 * as a single package.  The KMBC does this by sending a HSI_BRIDGE_ROUTE_IOCTL
 * request with an array of these descriptors, which each contains the details
 * of an individual route request.
 */
typedef struct _tagHSI_BRIDGE_ROUTE_DESC
{
	unsigned int bridgePort;	///< Specifies the bridge port to use for (un)route
	unsigned int hardBusTS;		///< Specifies timeslot on CT bus to use for (un)route
	unsigned int flags;			///< Controls direction and type of route request
}
HSI_BRIDGE_ROUTE_DESC, *PHSI_BRIDGE_ROUTE_DESC;




/**
 * @brief Specifies parameters for a bridge initialization request.
 *
 * This structure contains all of the parameters that are sent to a Bridge Device
 * by the KMBC as part of a bridge initialization request.  It is expected that
 * immediately upon successful completion of the bridge initialization request, the
 * Bridge Device is capable of handling route requests.
 */
typedef struct _tagHSI_BRIDGE_INIT_INPUT
{
	unsigned int               logicalId;       ///< Logical ID that Bridge Driver uses to identify bridge
	unsigned int               bridgeId;        ///< Unique ID assigned to Bridge Device by KMBC
	unsigned int               numTotalPorts;   ///< Specifies the total number of bridge ports available
	unsigned int               numVoicePorts;   ///< Specifies number of bridge ports to use for voice
	unsigned int               numDataPorts;    ///< Specifies number of bridge ports to use for data (transparent)
	HSI_PORT_ALLOCATION_METHOD portAllocCode;   ///< Specifies the port allocation code to use for configuring stream groups
	unsigned int               numHsiBuffers;   ///< Specifies number of HSI buffers to use
	HSI_TICK_RATE              tickRate;        ///< Specifies the rate interrupt frequency
	HSI_START_TYPE             startType;       ///< Specifies start mode for DMA ops
	HSI_BUS_TYPE               busType;         ///< Specifies the law conversion to use
	unsigned int               lawIdlePattern;  ///< Idle pattern to use for the law conversion
	unsigned int               syncTimeslot;    ///< Timeslot on CT bus that will contain sync pattern
	char                       syncPattern[4];  ///< 4 byte sync pattern used to sync DMA ops across bridges
	HSI_ISR_CALLBACK           hmpIsrFunc;      ///< Callback function to indicate rate interrupt
	HSI_SYNC_CALLBACK          hmpSyncComplete;	///< Callback function to indicate sync completion
	void*                      hostToBoardBuffers[HSI_MAX_BUFFERS]; ///< Array of transmit HSI buffers
	void*                      boardToHostBuffers[HSI_MAX_BUFFERS];	///< Array of receive HSI buffers
}
HSI_BRIDGE_INIT_INPUT, *PHSI_BRIDGE_INIT_INPUT;

typedef struct _tagHSI_BRIDGE_INIT_INPUT32
{
	unsigned int               logicalId;       ///< Logical ID that Bridge Driver uses to identify bridge
	unsigned int               bridgeId;        ///< Unique ID assigned to Bridge Device by KMBC
	unsigned int               numTotalPorts;   ///< Specifies the total number of bridge ports available
	unsigned int               numVoicePorts;   ///< Specifies number of bridge ports to use for voice
	unsigned int               numDataPorts;    ///< Specifies number of bridge ports to use for data (transparent)
	HSI_PORT_ALLOCATION_METHOD portAllocCode;   ///< Specifies the port allocation code to use for configuring stream groups
	unsigned int               numHsiBuffers;   ///< Specifies number of HSI buffers to use
	HSI_TICK_RATE              tickRate;        ///< Specifies the rate interrupt frequency
	HSI_START_TYPE             startType;       ///< Specifies start mode for DMA ops
	HSI_BUS_TYPE               busType;         ///< Specifies the law conversion to use
	unsigned int               lawIdlePattern;  ///< Idle pattern to use for the law conversion
	unsigned int               syncTimeslot;    ///< Timeslot on CT bus that will contain sync pattern
	char                       syncPattern[4];  ///< 4 byte sync pattern used to sync DMA ops across bridges
	unsigned int               hmpIsrFunc;      ///< Callback function to indicate rate interrupt
	unsigned int               hmpSyncComplete;	///< Callback function to indicate sync completion
	unsigned int               hostToBoardBuffers[HSI_MAX_BUFFERS]; ///< Array of transmit HSI buffers
	unsigned int               boardToHostBuffers[HSI_MAX_BUFFERS];	///< Array of receive HSI buffers
}
HSI_BRIDGE_INIT_INPUT32, *PHSI_BRIDGE_INIT_INPUT32;



/**
 * @brief Specifies data required by KMBC in response to init request.
 *
 * If the Bridge Device successfully completes an initialization request, the
 * KMBC needs to receive the following parameters to continue operating with
 * the Bridge Device.  These parameters are used by the KMBC to properly 
 * synchronize the DMA operations of the Bridge in a multi-bridge environment,
 * and by HMP to determine how to compensate for echo delay on the Bridge.
 */
typedef struct _tagHSI_BRIDGE_INIT_OUTPUT
{
	HSI_RESPONSE_TYPE type;
	unsigned int      bridgeId;
	unsigned int      txDelay;		///< Board transmit echo delay (in 125 us samples)
	unsigned int      rxDelay;		///< Board receive echo delay (in 125 us samples)
	unsigned int      txSyncDelay;	///< Board transmit sync pattern delay (in 125 us samples)
	unsigned int      rxSyncDelay;	///< Board receive sync pattern delay (in 125 us samples)
}
HSI_BRIDGE_INIT_OUTPUT, *PHSI_BRIDGE_INIT_OUTPUT;




/**
 * @brief Specifies a collection of route requests.
 *
 * The UMBC and KMBC are capable of aggregating route requests for a particular
 * Bridge Device.  When allocating this structure, the KMBC is responsible for
 * allocating enough memory to contain the whole request.  The length of the
 * buffer is calculated by adding an additional (numRoutes - 1) bridge route
 * descriptors to the size of this structure.  This allows us to create a
 * variably sized structure, while keeping the descriptor array packed with
 * the header information.
 */
typedef struct _tagHSI_BRIDGE_ROUTE_INPUT
{
	unsigned int          logicalId;	///< Logical ID that Bridge Driver uses to identify bridge
	unsigned int          numRoutes;	///< Number of route descriptors included in package
	HSI_BRIDGE_ROUTE_DESC routes[1];	///< Array of 'numRoutes' route descriptors
}
HSI_BRIDGE_ROUTE_INPUT, *PHSI_BRIDGE_ROUTE_INPUT;




/**
 * @brief Returns any failed route connections to the KMBC.
 *
 * Since the KMBC can send down large packages of route requests to a Bridge
 * Device, the Bridge Device needs to be able to respond with a list of any
 * route requests that failed so that the KMBC can update HMP with the failed
 * requests.
 */
typedef struct _tagHSI_BRIDGE_ROUTE_OUTPUT
{
	HSI_RESPONSE_TYPE     type;
	unsigned int          bridgeId;	
	unsigned int          numFailures;		///< Number of failed routes included in package
	HSI_BRIDGE_ROUTE_DESC failedRoutes[1];	///< Array of 'numFailures' route descriptors
}
HSI_BRIDGE_ROUTE_OUTPUT, *PHSI_BRIDGE_ROUTE_OUTPUT;




typedef struct _tagHSI_BRIDGE_STOP_OUTPUT
{
	HSI_RESPONSE_TYPE type;
	unsigned int      bridgeId;
}
HSI_BRIDGE_STOP_OUTPUT, *PHSI_BRIDGE_STOP_OUTPUT;




typedef struct _tagHSI_BRIDGE_INTR_ENABLE_OUTPUT
{
	HSI_RESPONSE_TYPE type;
	unsigned int      bridgeId;
}
HSI_BRIDGE_INTR_ENABLE_OUTPUT, *PHSI_BRIDGE_INTR_ENABLE_OUTPUT;




typedef struct _tagHSI_BRIDGE_INTR_DISABLE_OUTPUT
{
	HSI_RESPONSE_TYPE type;
	unsigned int      bridgeId;
}
HSI_BRIDGE_INTR_DISABLE_OUTPUT, *PHSI_BRIDGE_INTR_DISABLE_OUTPUT;


#endif	// #ifndef __HSI_INTERFACE_H__

