/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* Copyright (C) 2001-2013 Dialogic Corporation. All Rights Reserved.
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
* 1.    Redistributions of source code must retain the above copyright
* notice, this list of conditions and the following disclaimer.
*
* 2.    Redistributions in binary form must reproduce the above copyright
* notice, this list of conditions and the following disclaimer in
* the documentation and/or other materials provided with the
* distribution.
*
* 3.    Neither the name Dialogic nor the names of its
* contributors may be used to endorse or promote products derived from this
* software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
/**********************************************************************
 * File Name                    : mercd_native.c
 * Description                  : Native Driver file entry points
 *
 *
 **********************************************************************/

#ifndef LiS
#include <linux/module.h>
#include <linux/version.h>
#include <linux/ioctl.h>
#include "hsi_interface.h"
#include "heb_interface.h"

#include "msd.h"
#define _MERCD_C
#include "msdextern.h"
#undef _MERCD_C

#include "mercd_hsi.h"

merc_uint_t HMPBindHandle = 0;
extern int HMP_Enabled;

/**********************************************************************
 *  FunctionName:  mercd_open()
 *        Inputs:  inode - information structure for device node
 *                 file  - file structrure for device node.
 *       Outputs:  none.
 *       Returns:  ErrorCode. 0=Success. Negative=Error
 *                 -ENFILE = Could not allocate another Queue.
 *                           Reached Max Queues allowed.
 *                 -ENOMEM = Ran out of memory trying to allocate Queue.
 *                 -ENODEV = Wrong minor number for device node.
 *   Description:  Called upon file open on the device node.
 *      Comments:
 **********************************************************************/
int mercd_open(struct inode *inode, struct file *file){

   //unsigned int 	minor = MINOR(inode->i_rdev);
   static merc_uint_t   dev = 0;
   pmercd_open_block_sT MsdOpenBlock;
   int found = 0;

   MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();

   if (MsdControlBlock->open_block_count >= MsdControlBlock->maxopen) {
       MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
       MSD_LEVEL2_DBGPRINT( "mercd_open: too many Opens %d \n", MsdControlBlock->open_block_count);
       return (-ENODEV);
   }

   /* find an empty block for this open */
   for (dev %= MsdControlBlock->maxopen; dev < MsdControlBlock->maxopen; dev++) {
        if (MsdControlBlock->popen_block_queue[ dev ] == NULL) {
            found = 1;
            break;
        }
   }

   /* Can't find an empty block, start at the beginning */
   if(found == 0) {
       for (dev %= MsdControlBlock->maxopen; dev < MsdControlBlock->maxopen; dev++) {
           if (MsdControlBlock->popen_block_queue[ dev ] == NULL) {
                found = 1;
                break;
           }
       }
   }

   /* Allocate MsdOpenBlock */
   mercd_zalloc( MsdOpenBlock, pmercd_open_block_sT, sizeof(mercd_open_block_sT) );


   if (MsdOpenBlock == NULL) {
       MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
       printk("mercd_open: memory allocation failed\n");
       return(-ENOMEM);
   }


   MsdOpenBlock->kernelBuf = MSD_ALLOCATE_KERNEL_MEMORY(MERCURY_HOST_IF_BLK_SIZE + sizeof(USER_HEADER));
   if (MsdOpenBlock->kernelBuf == NULL) {
       MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
       printk("mercd_open: memory allocation of MsdOpenBlock->kernelBuf failed\n");
	   mercd_free(MsdOpenBlock, sizeof(mercd_open_block_sT), MERCD_FORCE);
       return(-ENOMEM);
   }

   // Array Implementation
   if (MsdControlBlock->arraySz) {
       mercd_zalloc(MsdOpenBlock->readersArray.array, Uint32 *, (sizeof(Uint32 *)*MsdControlBlock->arraySz));
   } else {
       // Allocate for no board usage
       MsdControlBlock->arraySz = MSD_ARRAY_SIZE;
       mercd_zalloc(MsdOpenBlock->readersArray.array, Uint32 *, (sizeof(Uint32 *)*MsdControlBlock->arraySz));
   }

   if (MsdOpenBlock->readersArray.array == NULL) {
       printk("mercd_open: memory allocation failed for array\n");
       mercd_free(MsdOpenBlock->kernelBuf, MERCURY_HOST_IF_BLK_SIZE + sizeof(USER_HEADER), MERCD_FORCE);
       mercd_free(MsdOpenBlock, sizeof(mercd_open_block_sT), MERCD_FORCE);
       MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
       return(-ENOMEM);
   }

   MsdControlBlock->popen_block_queue[dev] = MsdOpenBlock;
   MsdControlBlock->open_block_count++;

   //MERCD_GET_MAJOR_NODE( dev );

   MSD_INIT_MUTEX(&MsdOpenBlock->open_block_mutex, NULL, NULL);

   MSD_INIT_MUTEX(&MsdOpenBlock->readers_queue_mutex, NULL, NULL);

   MsdOpenBlock->clone_device_id = dev;
   MsdOpenBlock->state = MERCD_OPEN_STATE_OPENED;
   MsdOpenBlock->SavedMsg = NULL;
   MsdOpenBlock->readersArray.head = 0;
   MsdOpenBlock->readersArray.tail = 0;
   MsdOpenBlock->magic = MERCD_MAGIC;
   MsdOpenBlock->BoardNumber = -1; //This is set to a real board number in trace_enable and set back to -1 in trace_disable

   file->private_data = (void *)MsdOpenBlock;

   MsdOpenBlock->up_stream_write_q  = (pmercd_strm_queue_sT) MsdOpenBlock;

   MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();

#ifdef LINUX24
   MOD_INC_USE_COUNT;
#endif

   INIT_WAITQUEUE(MsdOpenBlock->readers);

   return(0);
}

/**********************************************************************
 *  FunctionName:  mercd_close()
 *        Inputs:  inode - information structure for device node
 *                 file  - file structrure for device node.
 *       Outputs:  none.
 *       Returns:  ErrorCode. 0=Success. Negative=Error
 *                 -ENODEV = Wrong minor number for device node.
 *   Description:  Called upon file close on the device node.
 *      Comments:
 **********************************************************************/
int mercd_close(struct inode *inode, struct file *file){

   //unsigned int 	minor = MINOR(inode->i_rdev);
   pmercd_open_block_sT MsdOpenBlock =  (pmercd_open_block_sT)file->private_data;
   pmercd_bind_block_sT BindBlock;
   int i,index, sleep_count = 0, all_nulled = TRUE;
   PSTRM_MSG       Msg;

   if (MsdOpenBlock == NULL) {
       printk("mercd_close: q->q_ptr null\n");
       return(-ENODEV);
   }

   MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();

   MsdOpenBlock->state = MERCD_OPEN_STATE_CLOSE_PEND;

   i = 0 ;
   while (BindBlock = (pmercd_bind_block_sT) MsdOpenBlock->bind_block_q.QueueHead) {
	i++;
	  if (BindBlock->bindhandle == 0)
              break;

	  supp_do_unbind(BindBlock);

	  if(i>40000) {
		printk("Breaking the loop here ..., BindBlock is %x \n",BindBlock);
		break;
	   }
   }
   
   if(MsdOpenBlock->BoardNumber != -1) {
       //Trace has not been disabled, trace program did not disable trace. Disable trace internally.
       supp_do_trace_disable(MsdOpenBlock);
   }
   
#if 0
   index = MsdControlBlock->OpenFreeIndex;

   if (MsdControlBlock->popen_free_block_queue[index])
       MSD_FREE_KERNEL_MEMORY((pmerc_char_t)MsdControlBlock->popen_free_block_queue[index], sizeof(mercd_open_block_sT));
#endif
    MSD_ENTER_MUTEX(&MsdOpenBlock->readers_queue_mutex);

    while((Msg = queue_get_msg_Array(&MsdOpenBlock->readersArray, MSD_READER)) != NULL) {
	   mercd_streams_free(Msg, MERCD_STREAMS_BUFFER, MERCD_FORCE);
           queue_commit_msg_Array(&MsdOpenBlock->readersArray, MSD_READER);
    }
	
    MsdOpenBlock->readersArray.head = 0;
    MsdOpenBlock->readersArray.tail = 0;

    MSD_FREE_KERNEL_MEMORY((Uint32 *)MsdOpenBlock->readersArray.array, sizeof(Uint32 *) *MsdControlBlock->arraySz);
	
	if(MsdOpenBlock->BoardNumber != -1) { //This OpenBlock is not for trace
		MSD_FREE_KERNEL_MEMORY(MsdOpenBlock->kernelBuf, MERCURY_HOST_IF_BLK_SIZE + sizeof(USER_HEADER));
	} else { //This OpenBlock is for trace
		MSD_FREE_KERNEL_MEMORY(MsdOpenBlock->kernelBuf, MSD_MAX_TRACE_BUF_SIZE);	
	}
	
	MsdOpenBlock->BoardNumber = -1;

    if(MsdOpenBlock->SavedMsg)
      mercd_streams_free(MsdOpenBlock->SavedMsg,MERCD_STREAMS_BUFFER, MERCD_FORCE);

    MSD_EXIT_MUTEX(&MsdOpenBlock->readers_queue_mutex);
#if 0
    MsdControlBlock->popen_free_block_queue[index] = MsdOpenBlock;

    MsdControlBlock->OpenFreeIndex++;

    MsdControlBlock->OpenFreeIndex %= MSD_MAX_OPENTOBE_FREED_INDEX;
#endif
    MsdControlBlock->popen_block_queue[MsdOpenBlock->clone_device_id] = NULL;

    MsdControlBlock->open_block_count--;

    MSD_DESTROY_MUTEX(&MsdOpenBlock->open_block_mutex);

    MSD_DESTROY_MUTEX(&MsdOpenBlock->readers_queue_mutex);
	
    //Check for all non-nulled OpenBlock references in padapter_block_list
    //These are set in the set functions (buffer init, buffer stop, license, bridge route) and get cleared in the replies from the DNI boards. 
    //If these are not NULL by now, that means the replies never came and we have to clear these references here.
    while((sleep_count < 3) ) { //loop 3 times waiting for these to be NULLed. In the last loop, if they are not NULLed, null them out
	
        for (i=0;i<MSD_MAX_BOARD_ID_COUNT;i++) {
		
            if(MsdControlBlock->padapter_block_list[i] == NULL) {
               continue;
            }

            if(sleep_count < 2) { //sleep 2 times of 250ms (total 500ms), before setting these to NULL
			
                if(MsdControlBlock->padapter_block_list[i]->hsiDongleMsdOpenBlock == MsdOpenBlock) {
                    all_nulled = FALSE;
                    //printk("mercd_close: warning!!! hsiDongleMsdOpenBlock not NULL\n");
                    break;
                }

                if(MsdControlBlock->padapter_block_list[i]->hsiMsdOpenBlock == MsdOpenBlock) {
                    all_nulled = FALSE;
                    //printk("mercd_close: warning!!! hsiMsdOpenBlock not NULL\n");
                   break;
                } 

                if(MsdControlBlock->padapter_block_list[i]->hsiRouteMsdOpenBlock == MsdOpenBlock) {
                    all_nulled = FALSE;
                    //printk("mercd_close: warning!!! hsiRouteMsdOpenBlock not NULL\n");
                    break;
                }
				
            } else { //time out, set everything to NULL, if not NULLed yet
		
                //Almost  half of second has passed, NULL everything and then get out
                if(MsdControlBlock->padapter_block_list[i]->hsiDongleMsdOpenBlock == MsdOpenBlock) {
                    MsdControlBlock->padapter_block_list[i]->hsiDongleMsdOpenBlock = NULL;
                }

                if(MsdControlBlock->padapter_block_list[i]->hsiMsdOpenBlock == MsdOpenBlock) {
                    MsdControlBlock->padapter_block_list[i]->hsiMsdOpenBlock = NULL;
                }

                if(MsdControlBlock->padapter_block_list[i]->hsiRouteMsdOpenBlock == MsdOpenBlock) {
                    MsdControlBlock->padapter_block_list[i]->hsiRouteMsdOpenBlock = NULL;
                }
				
            }
        }

        if(all_nulled == FALSE) {
            MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
            msleep_interruptible(250);
            MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();
            all_nulled = TRUE;
        } else {
            break;
        }
	
        sleep_count++;
    }

    memset(MsdOpenBlock, 0, sizeof(mercd_open_block_sT));
    mercd_free( MsdOpenBlock, sizeof(mercd_open_block_sT), MERCD_FORCE );
#ifdef LINUX24
    MOD_DEC_USE_COUNT;
#endif

    MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();

    return(0);
}

/**********************************************************************
 *  FunctionName:  mercd_read()
 *        Inputs:  file   - file structure for device
 *                 buffer - user buffer to fill in read device data.
 *                 size   - size of the buffer.
 *                 ppos   - offset into device.
 *       Outputs:  buffer - buffer is filled during the function call.
 *       Returns:  ssize_t >= 0 bytes copied from queue (should be size)
 *                 ssize_t <  0 error condition
 *                 -EAGAIN = Queue is empty and the function was asked
 *                 	      not to block.
 *                 -EINTR  = While blocked waiting for a message the
 *                 	     function was interrupted by a signal.
 *                 	     No message will be returned.
 *                 -EFAULT = Memory access error when trying to copy msg
 *              	     into user buffer.
 *                 -ENOSYS = System call not  supported for that device.
 *                 	     Queues are the only ones that support read.
 *                 -ENODEV = Invalid Minor number device. Not supported.
 *                 -ESPIPE = Pipe operation  called,  but not supported.
 *
 *   Description:  Read data from the mercd device.  Queues are the only
 * 		    ones that support read. Read message from the queue.
 *
 *      Comments: The readv implementation of linux is broken because it
 *      	   calls read underneath.  To fix   readv implementation
 *      	   we will  not block  if  the sent in size is 1.  Since
 *      	   the size field is not  used, because all messages are
 *      	   of uniform size. If we don't do this we will possibly
 *      	   block readv  in the  middle  of  retrieving messages.
 *      	   readv  should return if it had recieved any messages.
 *      	   We  have no way to tell if calle  by  readv  or read.
 **********************************************************************/
ssize_t mercd_read(struct file *file, char *buffer,
		   size_t size, loff_t *ppos)
{
   //unsigned int minor = MINOR(file->f_dentry->d_inode->i_rdev);
   pmercd_open_block_sT    MsdOpenBlock =  (pmercd_open_block_sT)file->private_data;

   ssize_t   rc;
   char        tmp_Mdrv_Msg[sizeof(MDRV_MSG)];
   int convert = FALSE;

#ifdef LINUX24
   // We don't support pread() or pwrite()
   if (ppos != &file->f_pos)
      return -ESPIPE;
#endif

#if defined(__x86_64__)
   /* check to see if 32/64bit task */
   if(is_compat_task()) {
      convert = TRUE;
   }
#endif


   if (MsdOpenBlock->flags & MERCD_OPEN_FLAG_PAYLOAD_ONLY) {
      // Provide only the payload and not the driver header.
      rc = mercd_readQueue(MsdOpenBlock, sizeof(MDRV_MSG), tmp_Mdrv_Msg,
               ((file->f_flags & O_NONBLOCK) ? TRUE : FALSE), FALSE);
      rc = mercd_readQueue(MsdOpenBlock, size, buffer,
               ((file->f_flags & O_NONBLOCK) ? TRUE : FALSE), FALSE);
      return(rc);
   } else {
         return(mercd_readQueue(MsdOpenBlock, size, buffer,
            ((file->f_flags & O_NONBLOCK) ? TRUE : FALSE), convert));
   } // end if (MsdOpenBlock->flags & MERCD_OPEN_FLAG_PAYLOAD_ONLY)
}

/**********************************************************************
 *  FunctionName:  mercd_write()
 *
 *        Inputs:  file   - file structure for device
 *                 buffer - buffer containing data to be written
 *                 size   - size of data to be copied into the driver
 *                 ppos   - offset within the device.
 *
 *       Outputs:  none.
 *
 *       Returns:  ssize_t >= 0 bytes copied from queue (should be size)
 *                 ssize_t <  0 error condition
 *                 -EINVAL = boardNumber out of range or invalid.
 *                 		No board at that Id.
 *                 -ENOTTY = board not in the Running State.
 *                 		Messages can only be sent when the board
 *                 		is in the Running State.
 *                 -EFAULT = Memory access error when trying to copy
 *                 		buffer to board queue.
 *                 -ESPIPE = Pipe operation called, but not supported.
 *                 -ENOSYS = System call not supported for that device.
 *                 		Queues are the only ones that support write.
 *                 -ENODEV = Invalid Minor numbered device. Not supported.
 *
 *   Description:  Copies the driver message buffer to the inbound board
 *   		   queue of the board with the BoardNumber boardId.
 *
 *      Comments:
 **********************************************************************/
ssize_t mercd_write(struct file *file, const char *buffer,
		    size_t size, loff_t *ppos)
{
	//unsigned int  minor = MINOR(file->f_dentry->d_inode->i_rdev);
	PMDRV_MSG32   MdMsg32;
	PMDRV_MSG     MdMsg;
	PSTRM_MSG 	 Msg;
	unsigned char *drvrbuffer=NULL;
	unsigned char *drvrbuffer32=NULL;
	unsigned char *savedbuffer=NULL;
	msgb_t        *tempMsg;
	size_t         adj_size = size;
	pmercd_open_block_sT MsdOpenBlock = (pmercd_open_block_sT)file->private_data;
	int convert = FALSE;

#ifdef LINUX24
	// We don't support pread() or pwrite()
	if (ppos != &file->f_pos)
	return -ESPIPE;
#endif

#if defined(__x86_64__)
	/* check to see if 32/64bit task */
	if(is_compat_task()) {
		convert = TRUE;
	}
#endif

	// allocate kernel memory since we should not touch user memory in kernel mode
	drvrbuffer = MSD_ALLOCATE_KERNEL_MEMORY(size);
	if (!drvrbuffer) {
		printk("mercd_write: unable to allocate memory %d for drvrbuffer\n", size);
		return -ENOMEM;
	}

	// copy user memory into kernel memory
	if (copy_from_user(drvrbuffer, buffer, size)) {
		printk("mercd_write: unable to copy user memory to kernel\n");
		MSD_FREE_KERNEL_MEMORY(drvrbuffer, size);
		return -EFAULT;
	}

	MSD_ENTER_MUTEX(&MsdOpenBlock->open_block_mutex);
	
	MdMsg = (PMDRV_MSG)drvrbuffer;

	if (!MsdOpenBlock->SavedMsg) {
		if(convert) {
			adj_size = size + MD_ADJUST_LEN;
			drvrbuffer32 = drvrbuffer;
			drvrbuffer = MSD_ALLOCATE_KERNEL_MEMORY(adj_size);
			if (!drvrbuffer) {
				printk("mercd_write: unable to allocate memory %d for drvrbuffer\n", adj_size);
				MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
				MSD_EXIT_MUTEX(&MsdOpenBlock->open_block_mutex);
				return -ENOMEM;
			}

			MdMsg32 = (PMDRV_MSG32)drvrbuffer32;
			MdMsg = (PMDRV_MSG)drvrbuffer;

			/* copy mercd header */
			MD_CONVERT_MDMSG32_TO_MDMSG(MdMsg32, MdMsg);
			/* copy mercd body */
			memcpy(((void *)(MdMsg+1)), ((void*)(MdMsg32+1)), size - sizeof(MDRV_MSG32));
		}

		if ((MD_GET_MDMSG_ID(MdMsg) > MSD_MAX_DRV_INTER_MESSAGES) ||
									(MD_GET_MDMSG_ID(MdMsg) <= 0)) {
			printk("mercd_write: Invalid Msg Id %x of size %d \n", MD_GET_MDMSG_ID(MdMsg), size);
			MsdOpenBlock->SavedMsg = 0;
			MSD_FREE_KERNEL_MEMORY(drvrbuffer, adj_size);
			if(convert) {
				MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
			}
			MSD_EXIT_MUTEX(&MsdOpenBlock->open_block_mutex);
			return -EINVAL;
		}
	} else {
		int class;

		class = MD_GET_MDMSG_CLASS(MD_EXTRACT_MDMSG_FROM_STRMMSG(MsdOpenBlock->SavedMsg));

		if (class != size) {
			printk("mercd_write: Unmatching Message Class %d and  %d  %x \n", class, size,
			MD_GET_MDMSG_ID((PMDRV_MSG)MD_EXTRACT_MDMSG_FROM_STRMMSG(MsdOpenBlock->SavedMsg)));
			printk("mercd_write: Current Message Class %#x Id %#x\n", MdMsg->MessageClass, MdMsg->MessageId);
			MSD_FREE_KERNEL_MEMORY(drvrbuffer, size);
			MSD_EXIT_MUTEX(&MsdOpenBlock->open_block_mutex);
			return -EINVAL;
		}
	}

	if (MsdOpenBlock->SavedMsg) {
		MdMsg = (PMDRV_MSG)MD_EXTRACT_MDMSG_FROM_STRMMSG(MsdOpenBlock->SavedMsg);
		tempMsg = MsdOpenBlock->SavedMsg;
		MsdOpenBlock->SavedMsg = 0;
		MSD_EXIT_MUTEX(&MsdOpenBlock->open_block_mutex);

		savedbuffer = mercd_allocator(size);

		if (savedbuffer) {
			// Check for a multi block messages as we can split it into
			// multiple buffers instead of using the one buffer to allocate
			if (MD_GET_MDMSG_ID(MdMsg) == MID_STREAM_MBLK_SEND) {
				memcpy(savedbuffer, drvrbuffer, size);
				MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();
				if (mid_strmmgr_mblk_split_send(MsdOpenBlock, tempMsg,
										savedbuffer, size) != MD_SUCCESS) {
					mercd_free(savedbuffer, size, MERCD_FORCE);
					MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
					MSD_FREE_KERNEL_MEMORY(drvrbuffer, size);
					return -EINVAL;
				}
				mercd_free(savedbuffer, size, MERCD_FORCE);
				MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
				MSD_FREE_KERNEL_MEMORY(drvrbuffer, size);
				return(size);
			}

			MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();
			Msg = native_alloc_msg_desr(size, (char *)savedbuffer);
			if (!Msg) {
				printk("mercd_write: memory allocation failed %d\n", size);
				mercd_free(savedbuffer, size, MERCD_FORCE);
				MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
				MSD_FREE_KERNEL_MEMORY(drvrbuffer, size);
				return -EINVAL;
			}
			MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();

			memcpy(savedbuffer, drvrbuffer, size);
			// Check is there is any contination in the buffer....
			tempMsg->b_cont = Msg;
		} else {
			printk("mercd_write: memory allocation failed %d\n", size);
			MSD_FREE_KERNEL_MEMORY(drvrbuffer, size);
			return -EINVAL;
		}
	} else {

		if ((merc_ushort_t)MdMsg->MessageClass) {

			savedbuffer = mercd_allocator(adj_size);

			if (savedbuffer) {
				MSD_EXIT_MUTEX(&MsdOpenBlock->open_block_mutex);
				MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();
				Msg = native_alloc_msg_desr(adj_size, (char *)savedbuffer);
				if (!Msg) {
					printk("mercd_write: memory allocation failed %d (2)\n", adj_size);
					mercd_free(savedbuffer, adj_size, MERCD_FORCE);
					MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
					MSD_FREE_KERNEL_MEMORY(drvrbuffer, adj_size);
					if(convert) {
						MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
					}
					return -EINVAL;
				}
				MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
				memcpy(savedbuffer, drvrbuffer, adj_size);
				MdMsg = (PMDRV_MSG)MD_EXTRACT_MDMSG_FROM_STRMMSG(Msg);

				if ((MD_GET_MDMSG_ID(MdMsg) > MSD_MAX_DRV_INTER_MESSAGES) ||
											(MD_GET_MDMSG_ID(MdMsg) <= 0)) {
					MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();
					mercd_free(savedbuffer, adj_size, MERCD_FORCE);
					MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
					MSD_FREE_KERNEL_MEMORY(drvrbuffer, adj_size);
					if(convert) {
						MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
					}
					return -EINVAL;
				}

				MsdOpenBlock->SavedMsg = Msg;
			} else {
				MsdOpenBlock->SavedMsg = 0;
				MSD_FREE_KERNEL_MEMORY(drvrbuffer, adj_size);
				if(convert) {
					MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
				}
				MSD_EXIT_MUTEX(&MsdOpenBlock->open_block_mutex);
				return -EINVAL;
			}

			MSD_FREE_KERNEL_MEMORY(drvrbuffer, adj_size);
			if(convert) {
				MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
			}
			return(size);
		} else {
			savedbuffer = mercd_allocator(adj_size);
			if (savedbuffer) {
				MSD_EXIT_MUTEX(&MsdOpenBlock->open_block_mutex);
				MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();
				Msg = native_alloc_msg_desr(adj_size, (char *)savedbuffer);
				if (!Msg) {
					printk("mercd_write: memory allocation failed %d (3)\n", adj_size);
					mercd_free(savedbuffer, adj_size, MERCD_FORCE);
					MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
					MSD_FREE_KERNEL_MEMORY(drvrbuffer, adj_size);
					if(convert) {
						MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
					}
					return -EINVAL;
				}
				MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
				memcpy(savedbuffer, drvrbuffer, adj_size);
				tempMsg = Msg;
			} else {
				MsdOpenBlock->SavedMsg = 0;
				MSD_FREE_KERNEL_MEMORY(drvrbuffer, adj_size);
				if(convert) {
					MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
				}
				MSD_EXIT_MUTEX(&MsdOpenBlock->open_block_mutex);
				return -EINVAL;
			}
		}
	}

	MdMsg = (PMDRV_MSG)MD_EXTRACT_MDMSG_FROM_STRMMSG(tempMsg);
	MdMsg->MessagePtr = 0;
	MdMsg->MessageFlags = MD_MSG_FLAG_ASYNC;

	if ((MD_GET_MDMSG_ID(MdMsg) > MSD_MAX_DRV_INTER_MESSAGES) ||
								(MD_GET_MDMSG_ID(MdMsg) <= 0)) {
		MSD_FREE_KERNEL_MEMORY(drvrbuffer, adj_size);
		if(convert) {
			MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
		}
		return -EINVAL;
	}

	MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();
	(*mercd_osal_mid_func[MD_GET_MDMSG_ID(MdMsg)])(MsdOpenBlock, tempMsg);
	MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();

	MSD_FREE_KERNEL_MEMORY(drvrbuffer, adj_size);
	if(convert) {
		MSD_FREE_KERNEL_MEMORY(drvrbuffer32, size);
	}
	return(size);
}

/**********************************************************************
 *  FunctionName:  mercd_poll()
 *
 *        Inputs:  file - file structure for device
 *                 wait - poll table structure.
 *
 *       Outputs:  none.
 *
 *       Returns:  mask - MASK of poll event flags.
 *
 *   Description:  Poll responds with what conditions exist in the device.
 *                 Both streams and queues support poll.
 *
 *      Comments:  Queues support READ  and WRITE  Polling. Write will
 *      	   always return true.  It will always queue a message
 *      	   inbound to a board.  Read will  only return true if
 *      	   there is a message on the outbound queue.
 *
 *                 Streams support Write and Error reporting  polliing.
 *                 Write will be returned true if there are no commands
 *                 active on the stream.  Errors will be returned if an
 *                 error code exists on the stream. This marks that the
 *                 last stream command failed.
 **********************************************************************/
unsigned int mercd_poll(struct file *file, struct poll_table_struct *wait)
{
   //unsigned int 	minor = MINOR(file->f_dentry->d_inode->i_rdev);
   unsigned int 	mask =  0;
   pmercd_open_block_sT MsdOpenBlock =  (pmercd_open_block_sT)file->private_data;

   poll_wait(file, &(MsdOpenBlock->readers), wait);


   if (!MSD_ARRAY_EMPTY(MsdOpenBlock->readersArray)) {
       mask |=  POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;  // Readable.
   } else {
       mask = 0;
   }

   return mask;
}

/**********************************************************************
 *  FunctionName:  mercd_mmap()
 *
 *        Inputs:  file - device file structure.
 *                 vma - virtual memory region description of mmap region.
 *
 *       Outputs:  none.
 *
 *       Returns:  ErrorCode. 0=Success. Negative=Error
 *                 -EINVAL = VM address not aligned, or VM region not same
 *                 	     size as stream buffer.
 *                 -ENOTTY = Stream has not been attached to the file handle.
 *                 	     This must be done first. (Attach Stream then
 *                 	     Mmap stream buffers the board is in the Running State.)
 *                 -ENXIO  = remapping of memory region failed.
 *                 -ENOSYS = System call not supported for that device.  Streams
 *                 	     are the only devices that support mmap.
 *
 *   Description:  Maps kernel buffers into a users virtual address space.
 *
 *      Comments:  Only stream devices support mmap.  The stream buffers are
 *                 mmapped into the user address space.
 **********************************************************************/
int mercd_mmap(struct file *file, struct vm_area_struct *vma){
#if (LINUX_VERSION_CODE < 0x020400)
   unsigned long offset = vma->vm_offset;
#else
   unsigned long offset = vma->vm_pgoff<<PAGE_SHIFT;
#endif
   //unsigned int  minor = MINOR(file->f_dentry->d_inode->i_rdev);
   merc_uchar_t  sem;
   pmercd_adapter_block_sT padapter;

   // Verify that stream has been attached to the file handle.
   if (offset > 0 ) {
       printk("mercd_mmap: offset is %d \n", offset);
       return -EINVAL;
   }

   if (mercd_adapter_map[MsdControlBlock->sramDumpCfgId] == 0xFF) {
       printk("mercd_mmap: invaild config id\n");
       return -EINVAL;
   }

   padapter = MsdControlBlock->padapter_block_list[mercd_adapter_map[MsdControlBlock->sramDumpCfgId]];

   if (!padapter) {
       printk("mercd_mmap: no adapter number %d found\n", MsdControlBlock->sramDumpCfgId);
       return -EINVAL;
   }

   if ((padapter->phw_info->pciSubSysId != PCI_SUBDEVICE_ID_DM3) &&
       (padapter->phw_info->pciSubSysId != PCI_SUBDEVICE_ID_DISI)) { 
       printk("mercd_mmap: non-Sram board\n");
       return -EINVAL;
   }

   if ((padapter->state == MERCD_ADAPTER_STATE_SUSPENDED) ||
          (padapter->state == MERCD_ADAPTER_STATE_MAPPED)) {
       printk("mercd_mmap: Invalid Board State\n");
       return -EINVAL;
   }

   // Request the host ram without asking for interrupt
   padapter->phost_info->reg_block.SetHostRamRequest =
        padapter->phw_info->virt_map_q[MERCD_PCI_SRAM_MAP]->virt_addr + MERC_SET_HOST_RAM_REQUEST_REG;
   MERC_GET_SRAM_NO_INTR_LOCK((&padapter->phost_info->reg_block), sem);

   if (!(sem & MERC_HOST_RAM_GRANT_R) || (sem == 0xFF)) {
       printk("mercd_mmap: unable to acquire sram\n");
       return -EINVAL;
   }

   if (remap_page_range(vma, vma->vm_start,
			padapter->phw_info->virt_map_q[MERCD_PCI_SRAM_MAP]->phys_addr,
			MsdControlBlock->sramDumpSz, vma->vm_page_prot)) {
       printk("mercd_mmap: remap_page_range failed\n");
       return -EINVAL;
   }

   return 0;
}

#if defined(__x86_64__)
/**********************************************************************
 *  FunctionName:  mercd_ioctl32()
 *
 *        Inputs:  inode - information structure for device node
 *                 file  - file structrure for device node.
 *                 functionId - Ioctl command identifier
 *                 arg - argument that can be used by the ioctl call
 *
 *       Outputs:  none.
 *
 *       Returns:  ErrorCode. 0=Success. Negative=Error
 *
 *   Description:  Kernel driver IOCTL entry point.  Allows for extension
 *   		   functions to be added to the driver.
 *
 *      Comments:  Each IOCTL has its own set of error values.  Each
 *      	   should be documented in the library.
 **********************************************************************/
int mercd_ioctl32(struct inode *inode, struct file *file,
		unsigned int functionId, unsigned long arg){

#ifdef LINUX24
   void *argp = (void *)arg;
#else
#ifdef CONFIG_COMPAT
   void __user *argp = compat_ptr(arg);
#else
   void __user *argp = (void __user *)arg;
#endif
#endif

   return mercd_ioctl_common(inode, file, functionId, argp);
}
#endif

/**********************************************************************
 *  FunctionName:  mercd_ioctl()
 *
 *        Inputs:  inode - information structure for device node
 *                 file  - file structrure for device node.
 *                 functionId - Ioctl command identifier
 *                 arg - argument that can be used by the ioctl call
 *
 *       Outputs:  none.
 *
 *       Returns:  ErrorCode. 0=Success. Negative=Error
 *
 *   Description:  Kernel driver IOCTL entry point.  Allows for extension
 *   		   functions to be added to the driver.
 *
 *      Comments:  Each IOCTL has its own set of error values.  Each
 *      	   should be documented in the library.
 **********************************************************************/
int mercd_ioctl(struct inode *inode, struct file *file,
		unsigned int functionId, unsigned long arg){
		
#ifdef LINUX24
   void *argp = (void *)arg;
#else
   void __user *argp = (void __user *)arg;
#endif

   return mercd_ioctl_common(inode, file, functionId, argp);
}

int mercd_ioctl_common(struct inode *inode, struct file *file, unsigned int functionId, void __user *argp)
{
   pmercd_open_block_sT MsdOpenBlock =  (pmercd_open_block_sT)file->private_data;
   pmercd_adapter_block_sT padapter = NULL;
   int            timeout_val,len,adj_len,command,ret,cnt;
   unsigned int   *gem_brd;
   PMDRV_MSG      MdMsg, tmpMdMsg;
   PMDRV_MSG32    MdMsg32;
   PSTRM_MSG 	  Msg;
   PSTRM_MSG      StrmMsg1;
   pmercd_hmp_x_block_sT     HmpBlock;
   merc_uint_t baru, barl;
   mercd_ioc_bar_rw_sT *iocbar;
   mercd_ioc_arr_rw_sT *iocarr;
   mercd_ioc_btype_sT *iocbty;
   int convert = FALSE;

#if defined(__x86_64__)
   /* check to see if 32/64bit task */
   if(is_compat_task()) {
      convert = TRUE;
   }
#endif

   MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();
#ifndef BIT32
	command = 0x00000000FFFFFFFF & functionId;
#endif
   switch (functionId) {
       case 0x100:
	    //printk("Get command %x \n", command);
	    ctimod_debug_on();
            ret = 0;
            goto out;

       case 0x101:
	    //printk("Get command %x \n", command);
	    ctimod_debug_off();
            ret = 0;
            goto out;

       case 0x102:
	    //printk("Get command %x \n", command);
	    ctimod_debug_clear();
            ret = 0;
            goto out;
       case 0x103:
	    //printk("Get command %x \n", command);
	    ctimod_debug_print();
            ret = 0;
            goto out;

 	case 0x104:
	    //printk("Get command %x \n", command);
	    ctimod_debug_toggle();
            ret = 0;
            goto out;

 	case 0x105:
            if (!MsdControlBlock->adapter_count) {
                ret = 0;
                goto out;
            }

            // Command from host kernel loader
            gem_brd = 0xFF;
            if (argp) {
                // Single board
                gem_brd = (unsigned int*) argp;
                padapter = MsdControlBlock->padapter_block_list[mercd_adapter_log_to_phy_map_table[(unsigned int)gem_brd]];
                if (padapter) {
	            switch (padapter->phw_info->pciDeviceId) {
	               case PCI_DEVICE_ID_GEM:
	               case PCI_DEVICE_ID_GEMMPC:
	               case PCI_DEVICE_ID_GEMRTM:
	               {
                           MSD_ENTER_MUTEX_IRQ(&padapter->phw_info->intr_info->intr_mutex);
                           reset_gemini(padapter);
                           MSD_EXIT_MUTEX_IRQ(&padapter->phw_info->intr_info->intr_mutex);
                           /* Find the 64bit FPGA BAR0 */
                           pci_read_config_dword(padapter->pdevi_gem, 0x10, &barl);
                           pci_read_config_dword(padapter->pdevi_gem, 0x14, &baru);

                           /* Set the CPU OB0 window to FPGA BAR0, 1M, memory access, disabled */
                           writel(0x00100004, padapter->phw_info->bar[0].virt_addr + 0x9cb0);
                           writel(0x40000000, padapter->phw_info->bar[0].virt_addr + 0x9cb4);
                           writel(baru, padapter->phw_info->bar[0].virt_addr + 0x9cbc);
                           writel(barl & 0xfffff000, padapter->phw_info->bar[0].virt_addr + 0x9cb8);
                           break;
                       }
	               default:
	                   break;
                    }
                } 
                ret = 0;
                goto out;
            }

            // All boards
            for (cnt=0;cnt<MSD_MAX_BOARD_ID_COUNT;cnt++) {
                 padapter = MsdControlBlock->padapter_block_list[cnt];

                 if (padapter) {
	             switch (padapter->phw_info->pciDeviceId) {
	                case PCI_DEVICE_ID_GEM:
	                case PCI_DEVICE_ID_GEMMPC:
	                case PCI_DEVICE_ID_GEMRTM:
	                {
                             MSD_ENTER_MUTEX_IRQ(&padapter->phw_info->intr_info->intr_mutex);
                             reset_gemini(padapter);
                             MSD_EXIT_MUTEX_IRQ(&padapter->phw_info->intr_info->intr_mutex);
	                     continue;
	                }
	                default:
	                     continue;
                     }
                 }
            }
            ret = 0;
            goto out;

  case 0x106:
            iocbar = (mercd_ioc_bar_rw_sT *)argp;
            padapter = MsdControlBlock->padapter_block_list[mercd_adapter_log_to_phy_map_table[iocbar->log_id]];
            ret = -EINVAL;
            if (padapter) {
              switch (padapter->phw_info->pciDeviceId) {
                case PCI_DEVICE_ID_GEM:
                case PCI_DEVICE_ID_GEMMPC:
                case PCI_DEVICE_ID_GEMRTM:
                  if((iocbar->bar < 6) && (padapter->phw_info->bar[iocbar->bar].virt_addr != NULL)) {
                    switch(iocbar->len) {
                      case 4:
                        if(((iocbar->off & 3) == 0) && (iocbar->off < padapter->phw_info->bar[iocbar->bar].byte_count)) {
                          iowrite32(iocbar->val, padapter->phw_info->bar[iocbar->bar].virt_addr + iocbar->off);
                          ret = 0;
                        }
                        break;
                    }
                    break;
                  }
                  break;
              }
            }
            goto out;

  case 0x107:
            iocbar = (mercd_ioc_bar_rw_sT *)argp;
            padapter = MsdControlBlock->padapter_block_list[mercd_adapter_log_to_phy_map_table[iocbar->log_id]];
            iocbar->val = 0xdeaddead;
            ret = -EINVAL;
            if (padapter) {
              switch (padapter->phw_info->pciDeviceId) {
                case PCI_DEVICE_ID_GEM:
                case PCI_DEVICE_ID_GEMMPC:
                case PCI_DEVICE_ID_GEMRTM:
                  if((iocbar->bar < 6) && (padapter->phw_info->bar[iocbar->bar].virt_addr != NULL)) {
                    switch(iocbar->len) {
                      case 4:
                        if(((iocbar->off & 3) == 0) && (iocbar->off < padapter->phw_info->bar[iocbar->bar].byte_count)) {
                          iocbar->val = ioread32(padapter->phw_info->bar[iocbar->bar].virt_addr + iocbar->off);
                          ret = 0;
                        }
                        break;
                    }
                    break;
                  }
              }
            }
            goto out;

  case 0x108:
            iocarr = (mercd_ioc_arr_rw_sT *)argp;
            padapter = MsdControlBlock->padapter_block_list[mercd_adapter_log_to_phy_map_table[iocarr->log_id]];
            ret = -EINVAL;
            if (padapter) {
              switch (padapter->phw_info->pciDeviceId) {
                case PCI_DEVICE_ID_GEM:
                case PCI_DEVICE_ID_GEMMPC:
                case PCI_DEVICE_ID_GEMRTM:
                  if((iocarr->bar < 6) &&
                     (padapter->phw_info->bar[iocarr->bar].virt_addr != NULL) &&
                     ((iocarr->off + iocarr->len) <= padapter->phw_info->bar[iocarr->bar].byte_count)) {
                    cnt = 0;
                    while(cnt < iocarr->len) {
                      iowrite32(*(uint32_t *)(iocarr->val + cnt),
                        padapter->phw_info->bar[iocarr->bar].virt_addr + iocarr->off + cnt);
                      cnt += 4;
                    }
                    ret = 0;
                  }
                  break;
              }
            }
            goto out;

  case 0x109:
            iocarr = (mercd_ioc_arr_rw_sT *)argp;
            padapter = MsdControlBlock->padapter_block_list[mercd_adapter_log_to_phy_map_table[iocarr->log_id]];
            ret = -EINVAL;
            if (padapter) {
              switch (padapter->phw_info->pciDeviceId) {
                case PCI_DEVICE_ID_GEM:
                case PCI_DEVICE_ID_GEMMPC:
                case PCI_DEVICE_ID_GEMRTM:
                  if((iocarr->bar < 6) &&
                     (padapter->phw_info->bar[iocarr->bar].virt_addr != NULL) &&
                     ((iocarr->off + iocarr->len) <= padapter->phw_info->bar[iocarr->bar].byte_count)) {
                    cnt = 0;
                    while(cnt < iocarr->len) {
                      *(uint32_t *)(iocarr->val + cnt) = ioread32(padapter->phw_info->bar[iocarr->bar].virt_addr + iocarr->off + cnt);
                      cnt += 4;
                    }
                    ret = 0;
                  }
                  break;
              }
            }
            goto out;

  case 0x10a:
            iocbty = (mercd_ioc_btype_sT *)argp;
            padapter = MsdControlBlock->padapter_block_list[mercd_adapter_log_to_phy_map_table[iocbty->log_id]];
            ret = -EINVAL;
            if (padapter) {
              // Make sure we only operate on Gemini (PCIe / AMC / Opal) adapters
              switch (padapter->phw_info->pciDeviceId) {
                case PCI_DEVICE_ID_GEM:
                case PCI_DEVICE_ID_GEMMPC:
                case PCI_DEVICE_ID_GEMRTM:
                  // Retrieve the overall adapter types
                  iocbty->pciDeviceId = padapter->phw_info->pciDeviceId;
                  iocbty->pciSubSysId = padapter->phw_info->pciSubSysId;
                  if(padapter->pdevi_gem) {
                    // If there is a FPGA bridge device attached get the information
                    iocbty->briVendorId = padapter->pdevi_gem->vendor;
                    iocbty->briDeviceId = padapter->pdevi_gem->device;
                    iocbty->briSubVenId = padapter->pdevi_gem->subsystem_vendor;
                    iocbty->briSubSysId = padapter->pdevi_gem->subsystem_device;
                  } else {
                    // Mark the bridge as NOT present
                    iocbty->briVendorId = 0xffff;
                    iocbty->briDeviceId = 0xffff;
                    iocbty->briSubVenId = 0xffff;
                    iocbty->briSubSysId = 0xffff;
                  }
                  if(padapter->pdevi) {
                    // If there is a CPU device attached get the information
                    iocbty->cpuVendorId = padapter->pdevi->vendor;
                    iocbty->cpuDeviceId = padapter->pdevi->device;
                    iocbty->cpuSubVenId = padapter->pdevi->subsystem_vendor;
                    iocbty->cpuSubSysId = padapter->pdevi->subsystem_device;
                  } else {
                    // Mark the CPU as NOT present
                    iocbty->cpuVendorId = 0xffff;
                    iocbty->cpuDeviceId = 0xffff;
                    iocbty->cpuSubVenId = 0xffff;
                    iocbty->cpuSubSysId = 0xffff;
                  }
                  ret = 0;
                  break;
              }
            }
            goto out;

 	case 0x10b:
            iocbar = (mercd_ioc_bar_rw_sT *)argp;
            padapter = MsdControlBlock->padapter_block_list[mercd_adapter_log_to_phy_map_table[iocbar->log_id]];
            iocbar->val = 0xdeaddead;
            ret = -EINVAL;
            if (padapter) {
              switch (padapter->phw_info->pciDeviceId) {
                case PCI_DEVICE_ID_GEM:
                case PCI_DEVICE_ID_GEMMPC:
                case PCI_DEVICE_ID_GEMRTM:
                  padapter->force_gpio_reset = 1;
                  ret = 0;
                  break;
              }
            }
            goto out;

        case 0x126:
            // IOCTL_READ_MEMORY
            {
               PGEM_RDWR_STRUCT pRdWrStruct = (PGEM_RDWR_STRUCT)argp;
               unsigned long ulTmp;
               unsigned char *BufPtr;
               ret = 0;
              
               // Validate system buffer for incoming and outgoing data
               if (!argp) {
                  goto out;
               }
               if ((MAX_RW_SZ < pRdWrStruct->Length) || (!pRdWrStruct->Length)) {
                   printk("IOCTL_READ_MEMORY Error %d<%d !%d..\n", MAX_RW_SZ, pRdWrStruct->Length, pRdWrStruct->Length);
                   goto out;
               }
               if ((!pRdWrStruct->DataType) || (pRdWrStruct->Length%pRdWrStruct->DataType)) {
                   printk("IOCTL_READ_MEMORY Error2 !%d || %d mod %d\n", pRdWrStruct->DataType, pRdWrStruct->Length, pRdWrStruct->DataType);
                   goto out;
               }
               ulTmp = pRdWrStruct->Length/pRdWrStruct->DataType;
               BufPtr = pRdWrStruct->Data;
               if (pRdWrStruct->AddrType == PHYS_ADDR) {
                   long long pa;

                   MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
                   pa = pRdWrStruct->pAddr;
                   pRdWrStruct->vAddr = (long long)ioremap((merc_uint_t)pa, pRdWrStruct->Length);
                   if (pRdWrStruct->vAddr == 0) {
                       printk("READ: Error unable to map address %#x of len %#x\n", pRdWrStruct->pAddr, pRdWrStruct->Length);
                       return(ret);     // Lock exited
                   }
               }
               if (pRdWrStruct->Length > 4) {
                   MERCD_READ_FROM_SRAM(BufPtr, (void*)pRdWrStruct->vAddr, pRdWrStruct->Length);
               } else if (pRdWrStruct->vAddr) {
                   switch (pRdWrStruct->DataType) {
                      default: *((unsigned char*) BufPtr) = ioread8 ((void*)pRdWrStruct->vAddr); break;
                      case 2: *((unsigned short*)BufPtr) = ioread16((void*)pRdWrStruct->vAddr); break;
                      case 4: *((unsigned long*) BufPtr) = ioread32((void*)pRdWrStruct->vAddr); break;
                   }
               } else {
                   printk("READ: Invalid Address \n");
			   }

               if (pRdWrStruct->AddrType == PHYS_ADDR) {
                   iounmap((void*)pRdWrStruct->vAddr);
                   return(ret);         // Lock exited
               }
               goto out;
            }
        case 0x127:
            // IOCTL_WRITE_MEMORY
            {
               PGEM_RDWR_STRUCT pRdWrStruct = (PGEM_RDWR_STRUCT)argp;
               unsigned long ulTmp;
               unsigned char *BufPtr;
               ret = 0;

               // Validate system buffer for incoming and outgoing data
               if (!argp) {
                  goto out;
               }
               if ((MAX_RW_SZ < pRdWrStruct->Length) || (!pRdWrStruct->Length)) {
                   printk("IOCTL_WRITE_MEMORY Error %d<%d !%d..\n", MAX_RW_SZ, pRdWrStruct->Length, pRdWrStruct->Length);
                   goto out;
               }
               if ((!pRdWrStruct->DataType) || (pRdWrStruct->Length%pRdWrStruct->DataType)) {
                   printk("IOCTL_WRITE_MEMORY Error2 !%d || %d mod %d\n", pRdWrStruct->DataType, pRdWrStruct->Length, pRdWrStruct->DataType);
                   goto out;
               }
               ulTmp = pRdWrStruct->Length/pRdWrStruct->DataType;
               BufPtr = pRdWrStruct->Data;
               if (pRdWrStruct->AddrType == PHYS_ADDR) {
                   long long pa;

                   MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
                   pa = pRdWrStruct->pAddr;
                   pRdWrStruct->vAddr = (long long)ioremap((merc_uint_t)pa, pRdWrStruct->Length);
                   if (pRdWrStruct->vAddr == 0) {
                       printk("WRITE: Error unable to map address %#x of len %#x\n", pRdWrStruct->pAddr, pRdWrStruct->Length);
                       return(ret);     // Lock exited
                   }
               }
               if (pRdWrStruct->Length > 4) {
                   MERCD_WRITE_TO_SRAM((void*)pRdWrStruct->vAddr, BufPtr, pRdWrStruct->Length);
               } else if (pRdWrStruct->vAddr) {
                   switch (pRdWrStruct->DataType) {
                      default: iowrite8 (*(unsigned char*)BufPtr, (void*)pRdWrStruct->vAddr); break;
                      case 2:  iowrite16(*(unsigned short*)BufPtr, (void*)pRdWrStruct->vAddr); break;
                      case 4:  iowrite32(*(unsigned long*)BufPtr, (void*)pRdWrStruct->vAddr); break;
                   }
               } else {
                   printk("WRITE: Invalid Address \n");
			   }

               if (pRdWrStruct->AddrType == PHYS_ADDR) {
                   iounmap((void*)pRdWrStruct->vAddr);
                   return(ret);         // Lock exited
               }
               goto out;
            }
        case 0x128:
            // IOCTL_GET_GEMINI
            {
               PGEM_INFO_STRUCT pInfoStruct = (PGEM_INFO_STRUCT)argp;
               merc_uint_t i, ulTmp = 0;
               // Validate system buffer for incoming and outgoing data
               if (!argp) {
                  goto out;
               }
            
               pInfoStruct->numGemBrds = 0;

               // All boards
               for (cnt=0;cnt<MSD_MAX_BOARD_ID_COUNT;cnt++) {
                    padapter = MsdControlBlock->padapter_block_list[cnt];

                     if (!padapter) {
                         continue;
                     }
	              
                     switch (padapter->phw_info->pciDeviceId) {
	                case PCI_DEVICE_ID_GEM:
	                case PCI_DEVICE_ID_GEMMPC:
	                case PCI_DEVICE_ID_GEMRTM:
                             pInfoStruct->brdInfo[ulTmp].logicalId = (unsigned char)padapter->adapternumber;
                             if (padapter->state != MERCD_ADAPTER_STATE_INIT) {
                                 for (i = 0; i<= MSD_MAX_BOARD_ID_COUNT; i++) {
                                      if (mercd_adapter_log_to_phy_map_table[i] == mercd_adapter_map[padapter->adapternumber]) {
                                          pInfoStruct->brdInfo[ulTmp].logicalId = i;
                                          break;
                                      }
                                 }
                             }
                             pInfoStruct->brdInfo[ulTmp].IMMRpAddr = (long long)padapter->phw_info->bar[0].phys_addr;
                             pInfoStruct->brdInfo[ulTmp].IMMRvAddr = (long long)padapter->phw_info->bar[0].virt_addr;
                             pInfoStruct->brdInfo[ulTmp].IMMRlength = padapter->phw_info->bar[0].byte_count;
                             pInfoStruct->brdInfo[ulTmp].SRAMpAddr = (long long)padapter->phw_info->bar[2].phys_addr; 
                             pInfoStruct->brdInfo[ulTmp].SRAMvAddr = (long long)padapter->phw_info->bar[2].virt_addr-0x200000;
                             pInfoStruct->brdInfo[ulTmp].SRAMlength =  padapter->phw_info->bar[2].byte_count;
                             pInfoStruct->brdInfo[ulTmp].FPGApAddr = (long long)padapter->phw_info->bar[3].phys_addr;
                             pInfoStruct->brdInfo[ulTmp].FPGAvAddr = (long long)padapter->phw_info->bar[3].virt_addr;
                             pInfoStruct->brdInfo[ulTmp].FPGAlength = padapter->phw_info->bar[3].byte_count;
                             /*printk("gem[%d]: brdIndx=%#x,  immrV=%llx (%#x), sramV=%llx (%#x), fpgaV=%llx (%#x)\n",
                               ulTmp, pInfoStruct->brdInfo[ulTmp].logicalId,
                               pInfoStruct->brdInfo[ulTmp].IMMRvAddr, pInfoStruct->brdInfo[ulTmp].IMMRlength,
                               pInfoStruct->brdInfo[ulTmp].SRAMvAddr, pInfoStruct->brdInfo[ulTmp].SRAMlength,
                               pInfoStruct->brdInfo[ulTmp].FPGAvAddr, pInfoStruct->brdInfo[ulTmp].FPGAlength);
                             printk(" immrP=%llx \n", pInfoStruct->brdInfo[ulTmp].IMMRpAddr);
                             printk(" sramP=%llx \n", pInfoStruct->brdInfo[ulTmp].SRAMpAddr);
                             printk(" fpgaP=%llx \n", pInfoStruct->brdInfo[ulTmp].FPGApAddr);*/
                             ulTmp++;
                             pInfoStruct->numGemBrds = ulTmp;
                             break;
                     }
               }
               
               goto out;
            }
  case 0x200: 
 	    /* Sync up HMP parameters with driver */

	    HmpBlock = (pmercd_hmp_x_block_sT)argp;
	    MSD_LEVEL2_DBGPRINT("HMPBindHandle = %d, HMPBoardID = %d, HMPCfgId = %d, HMPSlotId =%d\n", HmpBlock->HMPBindHandle, HmpBlock->HMPBoardId, HmpBlock->HMPCfgId, HmpBlock->HMPSlotId);
	    HMPBindHandle=HmpBlock->HMPBindHandle;
	    HMP_Enabled = 1; 
            MSD_LEVEL2_DBGPRINT("XMSG Enabled for HMP\n");
            ret = 0;
            goto out;

   case 0x201: // Get HSI ISR Callback Function Pointer
            {
            HSI_ISR_CALLBACK *phsiIsrCallback;

            // Get HSI ISR Callback Pointer.
            printk("mercd_ioctl(): Received a get hsi isr callback function IOCTL.\n");
            MsdOpenBlock->flags |= MERCD_OPEN_FLAG_PAYLOAD_ONLY;
            g_stub_hsi_isr_callback_MsdOpenBlock = MsdOpenBlock;
            printk("mercd_ioctl(0x201): MsdOpenBlock: 0x%08x\n", MsdOpenBlock);

            StrmMsg1 = supp_alloc_buf(sizeof(MDRV_MSG)+sizeof(char*), GFP_ATOMIC);
            if(StrmMsg1 == NULL) {
               printk("mercd_ioctl(): Cannot alloc Msg buffer\n");
               ret = -EFAULT;
               goto out;
            }
            MD_SET_MSG_TYPE(StrmMsg1, M_DATA);

            MdMsg = MD_EXTRACT_MDMSG_FROM_STRMMSG(StrmMsg1);
            MD_SET_MSG_WRITE_PTR(StrmMsg1, (MD_GET_MSG_READ_PTR(StrmMsg1)+sizeof(MDRV_MSG)+sizeof(char*)));

            phsiIsrCallback = (HSI_ISR_CALLBACK) MD_GET_MDMSG_PAYLOAD(MdMsg);
            *phsiIsrCallback = (HSI_ISR_CALLBACK) g_hsi_isr_callback;
            ////////////////////////////////////////////////
            supp_process_receive(MsdOpenBlock, StrmMsg1); //
            ////////////////////////////////////////////////
            ret = 0;
            goto out;
            }
   case 0x202: // Get Hsi Buffer
            {
            char **ppHsiBuffer       = NULL;

            printk("mercd_ioctl() received a get hsi buffer IOCTL.\n");
            MsdOpenBlock->flags |= MERCD_OPEN_FLAG_PAYLOAD_ONLY;
            printk("mercd_ioctl(0x202): MsdOpenBlock: 0x%08x\n", MsdOpenBlock);

            // The following memory is deallocated in mid_strmmgr_hsi_buffer_stop().
            g_pAllocatedMemory[g_pAllocatedMemoryIndex] = (char*)abstract_alloc_dma(K_HSI_BLOCK_SIZE);
            printk("mercd_native.c::Allocated g_pAllocatedMemory: 0x%08x\n", g_pAllocatedMemory[g_pAllocatedMemoryIndex]);

            StrmMsg1 = supp_alloc_buf(sizeof(MDRV_MSG)+sizeof(char*), GFP_ATOMIC);
            if(StrmMsg1 == NULL) {
               printk("mercd_ioctl(): Cannot alloc Msg buffer\n");
               ret = -EFAULT;
               goto out;
            }
            MD_SET_MSG_TYPE(StrmMsg1, M_DATA);

            MdMsg = MD_EXTRACT_MDMSG_FROM_STRMMSG(StrmMsg1);
            MD_SET_MSG_WRITE_PTR(StrmMsg1, (MD_GET_MSG_READ_PTR(StrmMsg1)+sizeof(MDRV_MSG)+sizeof(char*)));

            ppHsiBuffer = (char**) MD_GET_MDMSG_PAYLOAD(MdMsg);
            if (g_pAllocatedMemory[g_pAllocatedMemoryIndex]) {
               *ppHsiBuffer = (char*) virt_to_phys((void*)g_pAllocatedMemory[g_pAllocatedMemoryIndex]);
               printk("mercd_native.c::Allocated virt_to_phys(g_pAllocatedMemory): 0x%08x\n", *ppHsiBuffer);
            } else {
               *ppHsiBuffer = NULL;
            }
            if (g_pAllocatedMemoryIndex < K_MAX_HSI_BLOCKS-1)
               g_pAllocatedMemoryIndex++;

            ////////////////////////////////////////////////
            supp_process_receive(MsdOpenBlock, StrmMsg1); //
            ////////////////////////////////////////////////
            ret = 0;
            goto out;
            }
   case 0x203: // Dump Memory
            {
            long          *pLong           = (long *)argp;
            unsigned char *pVirtualAddress = (unsigned char *)pLong[0]; // Virtual Address
            long           dataSize        = pLong[1];                  // Size (1=8-bits, 2=16-bits, 3= 32-bits)
            long           dataCount       = pLong[2];                  // Count (Range: 1-1024)
            long          *poutputBuffer   = NULL   ;
            long           nbrOfBytes      = 0      ;

            printk("mercd_ioctl() received a Get Dump Memory IOCTL.\n");
            printk("mercd_ioctl(): pVirtualAddress: 0x%08x\n", pVirtualAddress);
            printk("mercd_ioctl(): dataSize: 0x%08x\n"       , dataSize       );
            printk("mercd_ioctl(): dataCount: 0x%08x\n"      , dataCount      );

            MsdOpenBlock->flags |= MERCD_OPEN_FLAG_PAYLOAD_ONLY;
            printk("mercd_ioctl(0x203): MsdOpenBlock: 0x%08x\n", MsdOpenBlock);

            if ( (dataSize != 1) && (dataSize != 2) && (dataSize != 4) ) {
               dataSize = 4;
            }

            nbrOfBytes = dataCount * dataSize;
            if  (nbrOfBytes > 1024 ) {
               nbrOfBytes = 1024;
            }

            StrmMsg1 = supp_alloc_buf(sizeof(MDRV_MSG)+nbrOfBytes, GFP_ATOMIC);
            if(StrmMsg1 == NULL) {
               printk("mercd_ioctl(): Cannot alloc Msg buffer\n");
               ret = -EFAULT;
               goto out;
            }
            MD_SET_MSG_TYPE(StrmMsg1, M_DATA);

            MdMsg = MD_EXTRACT_MDMSG_FROM_STRMMSG(StrmMsg1);
            MD_SET_MSG_WRITE_PTR(StrmMsg1, (MD_GET_MSG_READ_PTR(StrmMsg1)+sizeof(MDRV_MSG)+nbrOfBytes));

            poutputBuffer = (long*) MD_GET_MDMSG_PAYLOAD(MdMsg);

            memcpy((void*)poutputBuffer, pVirtualAddress, nbrOfBytes);
            ////////////////////////////////////////////////
            supp_process_receive(MsdOpenBlock, StrmMsg1); //
            ////////////////////////////////////////////////
            ret = 0;
            goto out;
            }
   case 0x204: // Get Base Addresses from Logical Id
            {
            long                   *pLong           = (long *)argp    ;
            long                    logicalId       = pLong[0]        ; // Logical Id
            long                   *poutputBuffer   = NULL            ;
            long                    nbrOfBytes      = 8 * sizeof(long);
            long                    physicalId      = 0xFF            ;

            printk("mercd_ioctl(): received a Get Base Addresses IOCTL.\n");
            MsdOpenBlock->flags |= MERCD_OPEN_FLAG_PAYLOAD_ONLY;
            printk("mercd_ioctl(0x204): MsdOpenBlock: 0x%08x\n", MsdOpenBlock);

            physicalId = mercd_adapter_log_to_phy_map_table[logicalId];
            if (physicalId == 0xFF) {
               printk("mercd_ioctl(): logicalId is not valid.\n");
               ret = -EFAULT;
               goto out;
            }

            padapter = MsdControlBlock->padapter_block_list[mercd_adapter_map[physicalId]];

            if ( (padapter == NULL) || (padapter->phw_info == NULL) || (padapter->phw_info->virt_map_q == NULL) ) {
               printk("mercd_ioctl(): Adapter is not valid.  Either logicalID is bad or board not started.\n");
               ret = -EFAULT;
               goto out;
            }

            StrmMsg1 = supp_alloc_buf(sizeof(MDRV_MSG)+nbrOfBytes, GFP_ATOMIC);
            if(StrmMsg1 == NULL) {
               printk("mercd_ioctl(): Cannot alloc Msg buffer\n");
               ret = -EFAULT;
               goto out;
            }
            MD_SET_MSG_TYPE(StrmMsg1, M_DATA);

            MdMsg = MD_EXTRACT_MDMSG_FROM_STRMMSG(StrmMsg1);
            MD_SET_MSG_WRITE_PTR(StrmMsg1, (MD_GET_MSG_READ_PTR(StrmMsg1)+sizeof(MDRV_MSG)+nbrOfBytes));

            poutputBuffer = (long *) MD_GET_MDMSG_PAYLOAD(MdMsg);
            poutputBuffer[0] = (long) padapter->phw_info->virt_map_q   [MERCD_PCI_SRAM_MAP  ]->virt_addr;
            poutputBuffer[1] = padapter->phw_info->virt_map_q   [MERCD_PCI_SRAM_MAP  ]->phys_addr;
            poutputBuffer[2] = (long) padapter->phw_info->virt_map_q   [MERCD_PCI_BRIDGE_MAP]->virt_addr;
            poutputBuffer[3] = padapter->phw_info->virt_map_q   [MERCD_PCI_BRIDGE_MAP]->phys_addr;
            poutputBuffer[4] = (long) padapter->phw_info->virt_ww_map_q[MERCD_PCI_SRAM_MAP  ]->virt_addr;
            poutputBuffer[5] = padapter->phw_info->virt_ww_map_q[MERCD_PCI_SRAM_MAP  ]->phys_addr;
            poutputBuffer[6] = (long) padapter->phw_info->virt_ww_map_q[MERCD_PCI_BRIDGE_MAP]->virt_addr;
            poutputBuffer[7] = padapter->phw_info->virt_ww_map_q[MERCD_PCI_BRIDGE_MAP]->phys_addr;

            ////////////////////////////////////////////////
            supp_process_receive(MsdOpenBlock, StrmMsg1); //
            ////////////////////////////////////////////////
            ret = 0;
            goto out;
            }
   case 0x205: // Enter Data into Memory
            {
            long          *pLong           = (long *)argp;
            unsigned char *pVirtualAddress = (unsigned char *)pLong[0]; // Virtual Address
            long           dataSize        = pLong[1];                  // Size (1=8-bits, 2=16-bits, 3= 32-bits)
            long           dataCount       = pLong[2];                  // Count (Range: 1-1024)
            long          *poutputBuffer   = NULL    ;
            long           nbrOfBytes      = 0       ;
            int            i               = 0       ;

            printk("mercd_ioctl(): received an Enter Data into Memory IOCTL.\n");
            printk("mercd_ioctl(): pVirtualAddress: 0x%08x\n", pVirtualAddress);
            printk("mercd_ioctl(): dataSize: 0x%08x\n"       , dataSize       );
            printk("mercd_ioctl(): dataCount: 0x%08x\n"      , dataCount      );
            printk("mercd_ioctl(): value[0]: 0x%08x\n"       , pLong[3]       );

            MsdOpenBlock->flags |= MERCD_OPEN_FLAG_PAYLOAD_ONLY;
            printk("mercd_ioctl(0x205): MsdOpenBlock: 0x%08x\n", MsdOpenBlock);

            if ( (dataSize != 1) && (dataSize != 2) && (dataSize != 4) ) {
               dataSize = 1;
            }

            nbrOfBytes = dataCount * dataSize;
            if  (nbrOfBytes > 128 ) {
               dataCount  = 128/dataSize;
               nbrOfBytes = 128;
            }

            if (dataSize == 1) {
                        unsigned char *pchar               = (         unsigned char *)&(pLong[3]);
               volatile unsigned char *pcharVirtualAddress = (volatile unsigned char *) pVirtualAddress;

               for (i=0; i < dataCount; i++)
               {
                  printk("Writing Byte: 0x%02x to 0x%08x\n", pchar[i], &(pcharVirtualAddress[i]));
                  pcharVirtualAddress[i] = pchar[i];
               }
            } else if (dataSize == 2) {
                        unsigned short *pshort               = (         unsigned short *)&(pLong[3]);
               volatile unsigned short *pshortVirtualAddress = (volatile unsigned short *) pVirtualAddress;

               for (i=0; i < dataCount; i++)
               {
                  printk("Writing Short int: 0x%04x to 0x%08x\n", pshort[i], &(pshortVirtualAddress[i]));
                  pshortVirtualAddress[i] = pshort[i];
               }
            } else {
                        unsigned long *plong               = (unsigned long *)&(pLong[3]);
               volatile unsigned long *plongVirtualAddress = (volatile unsigned long *) pVirtualAddress;

               for (i=0; i < dataCount; i++)
               {
                  printk("Writing long int: 0x%08x to 0x%08x\n", plong[i], &(plongVirtualAddress[i]));
                  plongVirtualAddress[i] = plong[i];
               }
            }

            StrmMsg1 = supp_alloc_buf(sizeof(MDRV_MSG)+sizeof(long), GFP_ATOMIC);
            if(StrmMsg1 == NULL) {
               printk("mercd_ioctl(): Cannot alloc Msg buffer\n");
               ret = -EFAULT;
               goto out;
            }
            MD_SET_MSG_TYPE(StrmMsg1, M_DATA);

            MdMsg = MD_EXTRACT_MDMSG_FROM_STRMMSG(StrmMsg1);
            MD_SET_MSG_WRITE_PTR(StrmMsg1, (MD_GET_MSG_READ_PTR(StrmMsg1)+sizeof(MDRV_MSG)+sizeof(long)));

            poutputBuffer = (long*) MD_GET_MDMSG_PAYLOAD(MdMsg);

            poutputBuffer[0] = nbrOfBytes;
            ////////////////////////////////////////////////
            supp_process_receive(MsdOpenBlock, StrmMsg1); //
            ////////////////////////////////////////////////
            ret = 0;
            goto out;
            }
   case HSI_BRIDGE_INIT_IOCTL:
            {
               if (g_hsi_stub_enabled) {
                  ret = stub_mid_strmmgr_hsi_buffer_init(MsdOpenBlock, argp);
                  goto out;
               } else {
                  ret = mid_strmmgr_hsi_buffer_init(MsdOpenBlock, argp);
                  goto out;
               }
            } // end case HSI_BRIDGE_INIT_IOCTL:
   case HSI_BRIDGE_STOP_IOCTL:
            {
               g_stub_hsi_isr_callback_MsdOpenBlock = NULL; // GCR
               if (g_hsi_stub_enabled) {
                  ret = stub_mid_strmmgr_hsi_buffer_stop(MsdOpenBlock, argp);
                  goto out;
               } else {
                  ret = mid_strmmgr_hsi_buffer_stop(MsdOpenBlock, argp);
                  goto out;
               }

            } // end case HSI_BRIDGE_STOP_IOCTL:
   case HSI_BRIDGE_ROUTE_IOCTL:
            {
               if (g_hsi_stub_enabled) {
                  ret = stub_mid_strmmgr_hsi_bridge_connect(MsdOpenBlock, argp);
                  goto out;
               } else {
                  ret = mid_strmmgr_hsi_bridge_connect(MsdOpenBlock, argp);
                  goto out;
               }
            } // end case HSI_BRIDGE_ROUTE_IOCTL:
   case HSI_INTR_RATE_ENABLE_IOCTL:
            {
               if (g_hsi_stub_enabled) {
                  ret = stub_mid_strmmgr_hsi_rate_interrupt_enable(MsdOpenBlock, argp);
                  goto out;
               } else {
                  ret = mid_strmmgr_hsi_rate_interrupt_enable(MsdOpenBlock, argp);
                  goto out;
               }
            } // end case HSI_INTR_RATE_ENABLE_IOCTL:
   case HSI_INTR_RATE_DISABLE_IOCTL:
            {
               if (g_hsi_stub_enabled) {
                  ret = stub_mid_strmmgr_hsi_rate_interrupt_disable(MsdOpenBlock, argp);
                  goto out;
               } else {
                  ret = mid_strmmgr_hsi_rate_interrupt_disable(MsdOpenBlock, argp);
                  goto out;
               }
            } // end case HSI_INTR_RATE_DISABLE_IOCTL:
   case HEB_GET_DONGLE_ID_IOCTL:
            {
               if (g_hsi_stub_enabled) {
                  ret = stub_mid_strmmgr_heb_get_dongle_id(MsdOpenBlock, argp);
                  goto out;
               } else {
                  ret = mid_strmmgr_heb_get_dongle_id(MsdOpenBlock, argp);
                  goto out;
               }
            }

      default:
          MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
	        break;

      out:
          MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
          return(ret);
   }

   command = functionId >> 24;
   len = (functionId & 0x00FFFF00) >> 8;
   timeout_val = functionId & 0x000000FF;
   
   adj_len = len;

   // allocate kernel memory since we should not touch user memory in kernel mode
   MdMsg = MSD_ALLOCATE_KERNEL_MEMORY(len);
   if (!MdMsg) {
       printk("mercd_ioctl: unable to allocate memory %d\n", len);
       return -ENOMEM;
   }

   // copy user memory into kernel memory
   if (copy_from_user(MdMsg, argp, len)) {
       printk("mercd_ioctl: unable to copy user memory to kernel\n");
       MSD_FREE_KERNEL_MEMORY(MdMsg, len);
       return -EFAULT;
   }

   if(convert)
   {
      adj_len = len + MD_ADJUST_LEN;
      MdMsg32 = (PMDRV_MSG32)MdMsg;
      // allocate kernel memory since we should not touch user memory in kernel mode
      MdMsg = MSD_ALLOCATE_KERNEL_MEMORY(adj_len);
      if (!MdMsg) {
         printk("mercd_ioctl: unable to allocate memory for MdMsg with len %d\n", adj_len);
         MSD_FREE_KERNEL_MEMORY(MdMsg32, len);
         return -ENOMEM;
      }
      /* copy mercd header */
      MD_CONVERT_MDMSG32_TO_MDMSG(MdMsg32, MdMsg);
      /* copy mercd body */
      memcpy(((void *)(MdMsg+1)), ((void*)(MdMsg32+1)), len - sizeof(MDRV_MSG32));
   }

   MSD_ENTER_CONTROL_BLOCK_MUTEX_BH();

   switch (command) {
      case MD_COMMAND_IOCTL:
        // We need to cover the message with the mblk and
        // put the size around it That way  we do not have
        // to change anything around the code.
        Msg = native_alloc_msg_desr(adj_len, (char *)MdMsg);

        // Protect the driver from any harm user can make
        if ((MD_GET_MDMSG_ID(MdMsg) > MSD_MAX_DRV_INTER_MESSAGES) ||
                                   (MD_GET_MDMSG_ID(MdMsg) <= 0))  {
           MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
           MSD_FREE_KERNEL_MEMORY(MdMsg, adj_len);
           if(convert)
           {
              MSD_FREE_KERNEL_MEMORY(MdMsg32, len);		   
           }
           return -EINVAL;
	    }

        MdMsg->MessageFlags = MD_MSG_FLAG_SYNC;

        if (Msg) {
           (*mercd_osal_mid_func[MD_GET_MDMSG_ID(MdMsg)])(MsdOpenBlock,Msg);
        } else {
           MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
           MSD_FREE_KERNEL_MEMORY(MdMsg, adj_len);
           if(convert)
           {
              MSD_FREE_KERNEL_MEMORY(MdMsg32, len);
           }
           return -EINVAL;
        }

	    break;

      default:
        MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
        MSD_FREE_KERNEL_MEMORY(MdMsg, adj_len);
        if(convert)
        {
           MSD_FREE_KERNEL_MEMORY(MdMsg32, len);
        }
	    return -EINVAL;
   }

   MSD_EXIT_CONTROL_BLOCK_MUTEX_BH();
   
   if(convert)
   {
      /* copy mercd header */
      MD_CONVERT_MDMSG_TO_MDMSG32(MdMsg, MdMsg32);
      /* copy mercd body */
      memcpy(((void *)(MdMsg32+1)), ((void*)(MdMsg+1)), adj_len - sizeof(MDRV_MSG));
      tmpMdMsg = MdMsg;
      MdMsg = MdMsg32;
   }
   
   copy_to_user(argp, MdMsg, len);
   MSD_FREE_KERNEL_MEMORY(MdMsg, len);
 
   if(convert)
   {
      MSD_FREE_KERNEL_MEMORY(tmpMdMsg, adj_len);
   }
   
   return(0);
}

#endif

