/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* Copyright (C) 2001-2010 Dialogic Corporation. All Rights Reserved.
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
* 1.    Redistributions of source code must retain the above copyright
* notice, this list of conditions and the following disclaimer.
*
* 2.    Redistributions in binary form must reproduce the above copyright
* notice, this list of conditions and the following disclaimer in
* the documentation and/or other materials provided with the
* distribution.
*
* 3.    Neither the name Dialogic nor the names of its
* contributors may be used to endorse or promote products derived from this
* software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
/**********************************************************************
 * File Name                    : mercd_ossdl_linux.c
 * Description                  : OS specific driver layer
 *
 *
 **********************************************************************/

#include <linux/version.h>

#include "msd.h"
#define MERCD_OSSDL_LINUX_C
#include "msdextern.h"
#undef MERCD_OSSDL_LINUX_C

#include "msdpciif.h"

#ifndef IRQF_SHARED
#define IRQF_SHARED SA_SHIRQ
#endif
/* 
 * This file defines the OSSDL interfaces functions for Linux 2.x
 * The following functions are present in this file 
 * 
 * linux_register_dpc
 * linux_deregister_dpc
 * linux_intr_register_intr_srvc
 * linux_intr_deregister_intr_srvc
 * linux_phys2virt
 * linux_free_virt_addr
 * linux_timeout
 * linux_untimeout
 * linux_intr_srvc
 */

extern int DPC_Disabled;

/***************************************************************************
 * Function Name                : linux_phys2virt
 * Function Type                : OSSDL and OSAL funtion
 * Inputs                       : rcvPtr
 * Outputs                      : Virtual Address 
 * Calling functions            : MERCD_OSAL_MEM_PHYS_VIRT Interfaces
 * Description                  : Solaris Implementation of Streams Driver
 *				  PhysicaToVirtual Routine. This routine maps
 *				  a physical address to Kernel virtual address.
 * Additional comments          :
 ****************************************************************************/
void linux_phys2virt(void *rcvPtr)
{
	pmercd_osal_mem_phys_virt_sT pmem = (pmercd_osal_mem_phys_virt_sT)rcvPtr;
	mercd_dhal_drvr_specific_sT drvr_specific = { 0 };
	pmerc_void_t kva = 0;

	MSD_FUNCTION_TRACE( "linux_phys2virt", ONE_PARAMETER, (size_t) rcvPtr);

        	
#if LINUX_VERSION_CODE < 0x020100               /* 2.0 kernel */
	kva = vremap((size_t)pmem->PhysAddr, (size_t)pmem->ByteCount);
#else
	kva = ioremap((size_t)pmem->PhysAddr, (size_t)pmem->ByteCount);
#endif

	if ( ! kva ) {
		printk("PhysToVirt: failed PhysAddr=%x, Count=%x\n", pmem->PhysAddr, pmem->ByteCount);
		pmem->ret = NULL;
		return;
	}

	MSD_LEVEL2_DBGPRINT("PhysToVirt:Virt=%x, Phys=%x, Count=%x\n", 
	       kva, pmem->PhysAddr, pmem->ByteCount);

	/* 
         * Platform Specific function 
	 */
	drvr_specific.ConfigId = pmem->ConfigId;

	(*mercd_dhal_func[MERCD_DHAL_DRVR_SPECIFIC])((void *)&drvr_specific);

	pmem->ret = kva;

	return;
}

/***************************************************************************
 * Function Name                : linux_free_virt_addr
 * Function Type                : OSSDL and OSAL function
 * Inputs                       : rcvPtr
 * Outputs                      : MD_STATUS
 * Calling functions            : MERCD_OSAL_MEM_VIRT_FREE Interfaces
 * Description                  : Streams Driver FreeVirtualAddress Routine.
 *                                This routine frees the virtual address
 *                                mapped to the physical address.
 * Additional comments          :
 ****************************************************************************/
void linux_free_virt_addr(void *rcvPtr)
{
	pmercd_osal_mem_virt_free_sT pmem = (pmercd_osal_mem_virt_free_sT)rcvPtr;
        
	MSD_FUNCTION_TRACE("linux_free_virt_addr", ONE_PARAMETER, (size_t)rcvPtr);

#if LINUX_VERSION_CODE < 0x020100               /* 2.0 kernel */
	vfree((void *)pmem->VirtAddr);
#else
	iounmap((void *)pmem->VirtAddr);

#endif
	pmem->ret = MD_SUCCESS;

	return;
}


/***************************************************************************
 * Function Name                : linux_timeout
 * Function Type                : OSSDL and OSAL function
 * Inputs                       : rcvPtr
 * Outputs                      : MD_STATUS
 * Calling functions            : MERCD_OSAL_TIMEOUT_START Interfaces
 * Description                  : Streams Driver Timeout Routine. This routine
 *                                sets a Timeout interval for a board
 * Additional comments          :
 ****************************************************************************/
void linux_timeout(void *rcvPtr)
{
	pmercd_osal_timeout_start_sT ptimeout = (pmercd_osal_timeout_start_sT) rcvPtr;
	size_t hz_wait= (ptimeout->Interval * HZ)/1000;

	//MSD_FUNCTION_TRACE("linux_timeout", ONE_PARAMETER, (size_t)rcvPtr);

#ifdef LiS
	*ptimeout->Handle = (size_t)timeout((void (*)())ptimeout->Function, 
	                                    (caddr_t)ptimeout->Context, 
	                                    hz_wait);
#else
	do {
	    struct timer_list *adapter_timeout;

	    adapter_timeout = ptimeout->Handle;
	    init_timer(adapter_timeout);
	    adapter_timeout->function = ptimeout->Function;
	    adapter_timeout->expires = abstract_jiffies() + hz_wait;
	    adapter_timeout->data = (unsigned long)ptimeout->Context;
	    add_timer(adapter_timeout);
	} while(0);
#endif /* LiS */

	ptimeout->ret = MD_SUCCESS;

	return;
}


/***************************************************************************
 * Function Name                : linux_untimeout
 * Function Type                : OSSDL and OSAL function
 * Inputs                       : rcvPtr
 * Outputs                      : MD_STATUS
 * Calling functions            : MERCD_OSAL_TIMEOUT_STOP Interfaces
 * Description                  : Streams Driver Untimeout Routine. This
 *                                routine removes previously installed Timeout
 * Additional comments          :
 ****************************************************************************/
void linux_untimeout(void *rcvPtr )
{
	pmercd_osal_timeout_stop_sT ptimeout = (pmercd_osal_timeout_stop_sT) rcvPtr;

	MSD_FUNCTION_TRACE("linux_untimeout", ONE_PARAMETER, (size_t)rcvPtr);

#ifdef LiS
	(void) untimeout((int)ptimeout->Handle);
#else
	del_timer(ptimeout->Handle);
#endif /* LiS */
	
	ptimeout->ret = MD_SUCCESS;

	return;
}

/***************************************************************************
 * Function Name                : linux_intr_register_intr_srvc
 * Function Type                : OSSDL and OSAL Interface function
 * Inputs                       : rcvPtr
 * Outputs                      : MD_STATUS
 * Calling functions            : MERCD_OSAL_INTR_REGISTER Interfaces
 * Description                  : Streams Driver intr_register_intr_srvc
 *                                Routine. This routine registers Mercury
 *                                Driver Interrupt service routine with the
 *                                Solaris Kernel
 * Additional comments          :
 ****************************************************************************/

void linux_intr_register_intr_srvc(void *rcvPtr)
{
	pmercd_osal_intr_register_sT pintr = (pmercd_osal_intr_register_sT)rcvPtr;
	merc_int_t rc;
	merc_ulong_t flag = IRQF_SHARED;

	MSD_FUNCTION_TRACE("linux_intr_register_intr_srvc", 
	                   ONE_PARAMETER,
	                   (size_t)rcvPtr);

	if (pintr->ConfigId >= MSD_MAX_BOARD_ID_COUNT ) {
		printk("mercd: IntrReg: Failed invalid ConfigId: config id(%d), Max id(%d)\n", pintr->ConfigId, MSD_MAX_BOARD_ID_COUNT ) ;
		pintr->ret = MD_FAILURE;
		return;
	}


	//WW support: request_irq can register linux_intr_srvc or linux_ww_intr_srvc or linux_gem_intr_srvc
	if (pintr->phw_info->pciSubSysId == PCI_SUBDEVICE_ID_GEM) {
	    // Gemini uses MSI - so they are not shared
	    flag &= ~IRQF_SHARED;
	}

	if ((rc = request_irq(pintr->dip->irq, (void *)pintr->Isr, flag, "mercdintr", (pmerc_void_t)pintr->dip))) {
		printk("mercd: IntrReg: request_irq() failed with return code %d\n", rc);
		pintr->ret = MD_FAILURE;
		return;
	}

	MSD_INIT_MUTEX(&pintr->phw_info->intr_info->intr_mutex, "hwintr mutex", NULL);

	// return the ISR Handle to caller
	*pintr->IsrHandle = pintr->ConfigId;

	pintr->ret = MD_SUCCESS;

	return;
}


/***************************************************************************
 * Function Name                : linux_intr_deregister_intr_srvc
 * Function Type                : OSSDL and OSAL Interface function
 * Inputs                       : rcvPtr
 * Outputs                      : MD_STATUS
 * Calling functions            : MERCD_OSAL_INTR_DEREGISTER Interfaces
 * Description                  : Streams Driver intr_deregister_intr_srvc Routine.
 *                                This routine is used to deregister the
 *                                interrupt with Solaris.
 * Additional comments          :
 ****************************************************************************/
void linux_intr_deregister_intr_srvc(void *rcvPtr)
{
  pmercd_osal_intr_deregister_sT pintr = (pmercd_osal_intr_deregister_sT)rcvPtr;

  MSD_FUNCTION_TRACE("linux_intr_deregister_intr_srvc", ONE_PARAMETER, (ULONG) rcvPtr);

  if (pintr->ConfigId >= MSD_MAX_BOARD_ID_COUNT) {
      printk("MsdDeregIntr: Illegal config id (%d) >= %d\n",
             			pintr->ConfigId, MSD_MAX_BOARD_ID_COUNT);
      pintr->ret = MD_FAILURE;
      return;
  }

  MSD_LEVEL2_DBGPRINT("MsdDeregIntr:free_irq @ IRQ %d,CfgId %d\n",
	                        pintr->dip->irq, pintr->ConfigId);

  // Must deregister interrutp from kernel
  free_irq(pintr->dip->irq, (void *)pintr->dip);

  pintr->ret = MD_SUCCESS;
  return;
}


/***************************************************************************
 * Function Name                : linux_intr_srvc/mercdintr
 * Function Type                : Interrupt functions
 * Inputs                       :
 * Outputs                      :
 * Calling functions            :
 * Description                  : Streams Driver mercd_intr_srvc Routine.
 *                                This is the interrupt service routine
 *                                registered and called on interrupt by Solaris
 * Additional comments          : DMA changes ..... 
 ****************************************************************************/
irqreturn_t linux_intr_srvc(int InterruptVector, void *dev_id, struct pt_regs *regs)
{
  merc_uchar_t    IntrReason;
  merc_uchar_t    HostRamReq;
  merc_uchar_t    need_int = 0;
  merc_ulong_t    PlxIntrReg; 
  mercd_hs_t*     hsd = NULL;
  pmercd_adapter_block_sT   padapter;
  merc_int_t i;

  hsd = (mercd_hs_t*)pci_get_drvdata(dev_id);
  if (!(hsd) || !(padapter = hsd->padapter)) {
      printk("linux_intr_srvc: Invalid dev_id\n");
      return IRQ_NONE;
  }

  padapter->phw_info->ret = MD_SUCCESS;

  if((padapter->state == MERCD_ADAPTER_STATE_MAPPED) ||
     (padapter->state == MERCD_ADAPTER_STATE_READY) ||
     (padapter->state == MERCD_ADAPTER_STATE_SUSPENDED) ||
     (padapter->state == MERCD_ADAPTER_STATE_INIT)) {
      return IRQ_NONE;
  }

  (*mercd_dhal_func[MERCD_DHAL_INTR_CHECK])( (void *)padapter) ;

  if ( padapter->phw_info->ret != MD_SUCCESS ) {
      return IRQ_NONE;
  }

  MSD_LEVEL2_DBGPRINT("mercd: interrupt check ... ok\n");

  if (!MSD_SPIN_TRYLOCK(&padapter->phw_info->intr_info->intr_mutex)) {
	return IRQ_NONE;
  }

  // get the interrupt reason
  IntrReason = MsdRegReadUchar(padapter->phost_info->reg_block.ClrMsgReadyIntr);

  MERC_CLEAR_INTERRUPT((&padapter->phost_info->reg_block), IntrReason);

  if (IntrReason & MERC_ERROR_INTR_R) {
      if (!(padapter->flags.SramOurs & MERC_BOARD_FLAG_SRAM_IS_OURS)) {
          // Get the Semaphore
          MERC_GET_SRAM_LOCK((&padapter->phost_info->reg_block), HostRamReq);

    	  if (!(HostRamReq & MERC_HOST_RAM_GRANT_FOR_CLR_R)){
              padapter->state = MERCD_ADAPTER_STATE_OUT_OF_SERVICE;
              printk("linux_intr_srvc: Adapter Out Of service\n");
              MSD_EXIT_MUTEX(&padapter->phw_info->intr_info->intr_mutex);
              return IRQ_NONE;
          }
      }
      
      if (padapter->panicDump == NULL)
          mercd_zalloc( padapter->panicDump, pmerc_uchar_t, MD_PANIC_DUMP_MAX_SIZE);
 
      if (!padapter->panicDump) {
          printk("linux_intr_srvc: unable to allocate memory\n");
          MSD_EXIT_MUTEX(&padapter->phw_info->intr_info->intr_mutex);
          return IRQ_NONE;
      }

 
      (void)supp_read_brd_panic_buf(padapter, (pmerc_uchar_t)padapter->panicDump);

      printk("mercd: Got Error Interrupt from board %d. Sram Crash dump...\n", padapter->adapternumber);
      for (i=0; i<MD_PANIC_DUMP_MAX_SIZE; i++) {
	   printk("%c ", padapter->panicDump[i]);
	   if (!(i%8))
		printk("\n");
      }
      printk("\n");

      // This is a Board DEATH intr.
      // Just set the adapter state to OOS for now
      // expect user to send shutboard to clean up
      padapter->state = MERCD_ADAPTER_STATE_OUT_OF_SERVICE;
      printk("linux_intr_srvc: Adapter Out Of service(1)\n");
      MSD_EXIT_MUTEX(&padapter->phw_info->intr_info->intr_mutex);

      return IRQ_NONE;
  }

  PlxIntrReg = MsdPlxGetIntrReg(mercd_adapter_map[padapter->adapternumber]);
	 
  ////////////////////////SRAMDMA BEGIN///////////////////////////
  // check for correct intr types
  if (!(IntrReason&(MERC_HOST_RAM_PEND_INTR_R|MERC_MSG_READY_INTR_R))) {
      if (PlxIntrReg & MSD_PCI_PLX_INT_CSR_DMA1_ACTIVE_R) {
          padapter->flags.DMAIntrReceived = (MERC_ADAPTER_FLAG_RCV_DMA_INTR | 
                                                 MERC_ADAPTER_FLAG_DMA_INTR);

          MsdPlxDmaClrInt_Chn1(mercd_adapter_map[padapter->adapternumber]);

	  // *** Free the SRAM LOCK  ***********************************
	  // We do not need this anymore, board can continue processing
	  //************************************************************

	  if (padapter->flags.LaunchIntr & MERC_ADAPTER_FLAG_LAUNCH_INTR) {
              //sending intr automatically releases sram lock
              padapter->flags.LaunchIntr &= ~MERC_ADAPTER_FLAG_LAUNCH_INTR;

              MERC_FREE_SRAM_LOCK((&padapter->phost_info->reg_block));
              MERC_INTR_ADAPTER((&padapter->phost_info->reg_block));
              // indicate we no longer have semaphore in adapter block
              padapter->flags.SramOurs  &= ~MERC_ADAPTER_FLAG_SRAM_IS_OURS;
          } else {
              // must release sram lock so fw can access it
              MERC_FREE_SRAM_LOCK((&padapter->phost_info->reg_block));
              padapter->flags.SramOurs  &= ~MERC_ADAPTER_FLAG_SRAM_IS_OURS;
          }
      } else  // Channel 0 is for Transmit interrupts. To be supported in the future 
          if (PlxIntrReg & MSD_PCI_PLX_INT_CSR_DMA0_ACTIVE_R) {
              padapter->flags.DMAIntrReceived =  (MERC_ADAPTER_FLAG_SND_DMA_INTR |
                                                         MERC_ADAPTER_FLAG_DMA_INTR);
              MsdPlxDmaClrInt_Chn0(mercd_adapter_map[padapter->adapternumber]);
      } else {
	      //This is not possible.. But just exit ..... 
              MSD_EXIT_MUTEX(&padapter->phw_info->intr_info->intr_mutex);
              return IRQ_NONE;
      }
  } else {
      if (PlxIntrReg & MSD_PCI_PLX_INT_CSR_DMA1_ACTIVE_R) {
          MsdPlxDmaClrInt_Chn1(mercd_adapter_map[padapter->adapternumber]);

	  padapter->flags.DMAIntrReceived = (MERC_ADAPTER_FLAG_RCV_DMA_INTR |
                                                 MERC_ADAPTER_FLAG_DMA_INTR);

	  // Release the SRAM lock here .... , No more SRAM access needed.
	  if (padapter->flags.LaunchIntr & MERC_ADAPTER_FLAG_LAUNCH_INTR) {
              //sending intr automatically releases sram lock
              padapter->flags.LaunchIntr &= ~MERC_ADAPTER_FLAG_LAUNCH_INTR;

              MERC_FREE_SRAM_LOCK((&padapter->phost_info->reg_block));
              MERC_INTR_ADAPTER((&padapter->phost_info->reg_block));
              // indicate we no longer have semaphore in adapter block
              padapter->flags.SramOurs  &= ~MERC_ADAPTER_FLAG_SRAM_IS_OURS;
          } else {
              // must release sram lock so fw can access it
              MERC_FREE_SRAM_LOCK((&padapter->phost_info->reg_block));
              padapter->flags.SramOurs  &= ~MERC_ADAPTER_FLAG_SRAM_IS_OURS;
          }

      }
  }
  
  // Check for non-DMA Complete Interrupt during DMA - do not service it
  if (padapter->phw_info->pdma_info->rx_dma_count > 0) {
      if (padapter->flags.DPCDmaPending == 0) {
          MSD_LEVEL1_DBGPRINT("mercd: Bad Case (rx_dma_count=%d and DPCDmaPending=%d)\n", padapter->phw_info->pdma_info->rx_dma_count, padapter->flags.DPCDmaPending);
          MSD_EXIT_MUTEX(&padapter->phw_info->intr_info->intr_mutex);
          return IRQ_NONE;
      } else {
          if (padapter->flags.DMAIntrReceived == 0) {
              MSD_LEVEL1_DBGPRINT("mercd: Received a non-DMA Complete Interrupt after DMA Started\n");
              MSD_EXIT_MUTEX(&padapter->phw_info->intr_info->intr_mutex);
              return IRQ_NONE;
          }
      }
  } else {
      if (padapter->flags.DPCDmaPending > 0) {
          MSD_LEVEL1_DBGPRINT("mercd: Bad Case (rx_dma_count=%d and DPCDmaPending=%d)\n", padapter->phw_info->pdma_info->rx_dma_count, padapter->flags.DPCDmaPending);
          MSD_EXIT_MUTEX(&padapter->phw_info->intr_info->intr_mutex);
          return IRQ_NONE;
      }
  }

  need_int = 0;

  // DPC is protected by Mutex ...
  if (padapter->phw_info->un_flag != UNIT_BUSY) {

      need_int = 1;

      // LA: Task# 14456 ...
      // before this was set regardless with or without an interrupt need_int ( 1 or 0 )
      padapter->flags.RecvPending |= MERC_ADAPTER_FLAG_RECEIVE_PEND;

      MSD_LEVEL2_DBGPRINT("dpc data addr = %x, slotn = %d, content = %d\n",
          padapter->phw_info->intr_info->dpc_task.data, padapter->phw_info->slot_number,
          *(merc_uchar_t *) padapter->phw_info->intr_info->dpc_task.data );


  }

  MSD_EXIT_MUTEX(&padapter->phw_info->intr_info->intr_mutex);

  if (need_int) {
      //Scheduling can be done in different ways as shown below
      if (DPC_Disabled) {
          mercd_generic_intr_processing( (void *) &padapter->phw_info->slot_number);
      } else {
          tasklet_hi_schedule(&(padapter->phw_info->intr_info->dpc_task));
      }
  }

  return IRQ_HANDLED;
}


/***************************************************************************
 * Function Name                : linux_gem_intr_srvc
 * Function Type                : Interrupt functions
 * Inputs                       :
 * Outputs                      :
 * Calling functions            :
 * Description                  : Driver Interrupt Routine for Gemini.
 *                                This is the interrupt service routine
 *                                registered and called on interrupt by Linux
 ****************************************************************************/
irqreturn_t linux_gem_intr_srvc(int InterruptVector, void *dev_id, struct pt_regs *regs)
{
  pmercd_adapter_block_sT padapter;
  pmerc_uchar_t BaseAddr = 0;
  pmerc_uchar_t MemAddr = 0;
  mercd_hs_t*   hsd = NULL;
  merc_uint_t   ulTmp = 0;
#if 0
  merc_ulong_t  brdCnt = 0;
  static merc_ulong_t isr=0;
  static merc_ulong_t initCnt=0;
#endif

  hsd = (mercd_hs_t*)pci_get_drvdata(dev_id);
  if (!(hsd) || !(padapter = hsd->padapter)) {
      printk("linux_gem_intr_srvc: Invalid dev_id\n");
      return IRQ_NONE;
  }

  BaseAddr =(pmerc_uchar_t)(padapter->phw_info->bar[3].virt_addr);
  MemAddr = BaseAddr + MSD_GEM_RATE_CNT;
  ulTmp = readl(MemAddr);
  padapter->hsiRateIntrData = ulTmp;
 
  // Callback
  if (padapter->hsi_isr_callback) {
      padapter->hsi_isr_callback(padapter->hsiBridgeId, padapter->hsiRateIntrData);
  }

#if 0
  // Interrupt tracing
  brdCnt = ulTmp&0x00FFFFFF;

  // initialize to board count
  if (!isr)
      initCnt = brdCnt;
  else
      initCnt++;	 
  
  if (brdCnt != initCnt) {
      if (initCnt < 0xFFFFFF) {
	  padapter->hsiDrift++;
          if (brdCnt > initCnt)
              padapter->hsiDelta = brdCnt - initCnt;
          else
              padapter->hsiDelta = initCnt - brdCnt;
          //printk("GEM_FPGA_ISR: DRIFT Detected\n");
      }
      initCnt = brdCnt;
  }

  padapter->hsiBoardCount = ulTmp;
  padapter->hsiDriverCount = initCnt;

  if (!(isr++ % 150000)) { 
      printk("GEM_FPGA_ISR: %d %d ==> %d\n", brdCnt, initCnt, isr);
  }
#endif

  return IRQ_HANDLED; 
}


/***************************************************************************
 * Function Name                : linux_gemmpc_intr_srvc
 * Function Type                : Interrupt functions
 * Inputs                       :
 * Outputs                      :
 * Calling functions            :
 * Description                  : Driver Interrupt Routine for Gemini MPC
 *                                This is the interrupt service routine
 *                                registered and called on interrupt by Linux
 ****************************************************************************/
irqreturn_t linux_gemmpc_intr_srvc(int InterruptVector, void *dev_id, struct pt_regs *regs)
{
  pmercd_adapter_block_sT padapter;
  merc_uint_t   ulTmp = 0;
  merc_uint_t   mpcInt = 0;
  mercd_hs_t*   hsd = NULL;

  hsd = (mercd_hs_t*)pci_get_drvdata(dev_id);
  if (!(hsd) || !(padapter = hsd->padapter)) {
      printk("linux_gemmpc_intr_srvc: Invalid dev_id on InterruptVec %#x\n", InterruptVector);
      return IRQ_NONE;
  }

  // Read the Outbound Mailbox Data Register
  mpcInt = readl(padapter->phw_info->virt_map_q[MERCD_PCI_BRIDGE_MAP]->virt_addr + MSD_MPC_PEX_OMBDR);

  // Clear Outbound Mailbox Control & Data Register
  writel(ulTmp, padapter->phw_info->virt_map_q[MERCD_PCI_BRIDGE_MAP]->virt_addr + MSD_MPC_PEX_OMBCR);
  writel(ulTmp, padapter->phw_info->virt_map_q[MERCD_PCI_BRIDGE_MAP]->virt_addr + MSD_MPC_PEX_OMBDR);

  switch (mpcInt) {
     case MSD_GEM_DO_RESET:
          padapter->flags.SramOurs = MERC_ADAPTER_FLAG_PERFORM_RESET;
          break;
     default:
          padapter->flags.SramOurs = MERC_ADAPTER_FLAG_SRAM_IS_OURS;
          break; 
  }
          
  // Schedule the DPC
  tasklet_hi_schedule(&(padapter->phw_info->intr_info->dpc_task));

  return IRQ_HANDLED; 
}
