/**
* @file bridgedcallconnecting_state.cpp
* @brief Definition of BridgedCallConnectingState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "bridgedcallconnecting_state.h"
#include "bridgedcall.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"

// static constant members:
const int BridgedCallConnectingState::H245_STARTED    = 0x01;
const int BridgedCallConnectingState::SIP_CONNECTED   = 0x02;
const int BridgedCallConnectingState::M3G_CONNECTED   = 0x04;
const int BridgedCallConnectingState::FULLY_CONNECTED = 0x07;

//*****************************************************************************
// Function: BridgedCallConnectingState::BridgedCallConnectingState(BridgedCall *pBridgedCall)
// Description: Initializing constructor
// Return:  BridgedCallConnectingState*
// Parameters: BridgedCall *pBridgedCall 
//*****************************************************************************
BridgedCallConnectingState::BridgedCallConnectingState(BridgedCall* pBridgedCall) :
   BridgedCallState(pBridgedCall)
{
}


//*****************************************************************************
// Function: BridgedCallConnectingState::~BridgedCallConnectingState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedCallConnectingState::~BridgedCallConnectingState()
{
}

//*****************************************************************************
// Function: void BridgedCallConnectingState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallConnectingState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      case APP_H245_STARTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgeCallConnectingState rcvd:APP_H245_STARTED\n",m_pBridgedCall->Index());
         SetStatus(H245_STARTED);
         break;
      case APP_SIP_ENDPOINT_CONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgeCallConnectingState rcvd:APP_SIP_ENDPOINT_CONNECTED\n",m_pBridgedCall->Index());
         SetStatus(SIP_CONNECTED);
         ProcessEndpointConnected(metaevent);
         break;
      case APP_SIP_ENDPOINT_ANSWERED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgeCallConnectingState rcvd:APP_SIP_ENDPOINT_ANSWERED\n",m_pBridgedCall->Index());
         SetStatus(SIP_CONNECTED);
         ProcessEndpointConnected(metaevent);
         break;
      case APP_M3G_ENDPOINT_CONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgeCallConnectingState rcvd:APP_M3G_ENDPOINT_CONNECTED\n",m_pBridgedCall->Index());
         SetStatus(M3G_CONNECTED);
         ProcessEndpointConnected(metaevent);
         break;
      case APP_M3G_ENDPOINT_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgeCallConnectingState rcvd:APP_M3G_ENDPOINT_DISCONNECTED\n", m_pBridgedCall->Index());
         ProcessEndpointDisconnected(metaevent);
         break;
      case APP_SIP_ENDPOINT_DISCONNECT:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgeCallConnectingState rcvd:APP_SIP_ENDPOINT_DISCONNECT\n",m_pBridgedCall->Index());
         break;
      case APP_SIP_ENDPOINT_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgeCallConnectingState rcvd:APP_SIP_ENDPOINT_DISCONNECTED\n",m_pBridgedCall->Index());
         ProcessSipEndpointDisconnected(metaevent);
         break;
      case APP_H245_START_REQUEST:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgeCallConnectingState rcvd:APP_H245_START_REQUEST\n",m_pBridgedCall->Index());
      break;
      default:
         LOG_ERROR(metaevent.evtdev,"BridgedCall[%d] BridgeCallConnectingState: Unexpected event type: 0x%x\n",
		   m_pBridgedCall->Index(),metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void BridgedCallConnectingState::ProcessSipEndpointDisconnected(METAEVENT metaevent)
// Description: Process an SIP disconnected event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallConnectingState::ProcessSipEndpointDisconnected(METAEVENT metaevent)
{
   m_pBridgedCall->ChangeState(IDLE_STATE);
}

//*****************************************************************************
// Function: void BridgedCallConnectingState::ProcessEndpointDisconnected(METAEVENT metaevent)
// Description: Process an 3G disconnect event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallConnectingState::ProcessEndpointDisconnected(METAEVENT metaevent)
{
  SIPEndpoint *pSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx( m_pBridgedCall->SipIndex());
  if ( pSIPEndpoint )
  {
     LOG_ENTRY(metaevent.evtdev, "Preparing fake GCEV_DISCONNECT to react to APP_M3G_ENDPOINT_DISCONNECT\n");
     METAEVENT tmpMetaevent;
     tmpMetaevent.evttype =  GCEV_DISCONNECTED;
     tmpMetaevent.evtdev = 0;
     pSIPEndpoint->ProcessEvent(tmpMetaevent);
  }
  m_pBridgedCall->ChangeState(IDLE_STATE);
}

//*****************************************************************************
// Function: void BridgedCallConnectingState::ProcessEndpointConnected(METAEVENT metaevent)
// Description: Process an endpoint connected event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallConnectingState::ProcessEndpointConnected(METAEVENT metaevent)
{
   if ( APP_M3G_ENDPOINT_CONNECTED == metaevent.evttype )
   {
      SIPEndpoint *pSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx( m_pBridgedCall->SipIndex());
      if ( pSIPEndpoint )
      {

	if (pSIPEndpoint->GetState() == SIPEPState::IDLE_STATE)
	  // Only do outbound call if the SIP endpoint associated with this 3G is in an idle state -
	  // i.e. it's not in the process of handling the inbound call.
          // Note that both 3G call legs will generate the APP_M3G_ENDPOINT_CONNECTED event
	  {
	    LOG_ENTRY(0, "BridgedCall[%d] - initiating outbound SIP call\n", m_pBridgedCall->Index());
	    Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedCall->M3gIndex());
	    if ( pEndpoint )
	      {
                if(pEndpoint->GetAudTranscodeEnabled())
                {
	           LOG_ENTRY(0, "BridgedCall[%d] - Audio TranscodeMode Enabled(TRANSCODE_ON)\n", m_pBridgedCall->Index());
                   // For now, Set the 1st Preferred coder for SIP call 
                   pSIPEndpoint->SetSelectedAudioCoder(EndpointMngr::Instance()->GetPreferredSipAudioCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] PreferredSipAudioCoder=%d:\n",
                          m_pBridgedCall->Index(),
                          EndpointMngr::Instance()->GetPreferredSipAudioCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] 3GSelectedAudioCoder=%s[%d]:\n",
                          m_pBridgedCall->Index(),
                          pEndpoint->GetSelAudCoderStr(), pEndpoint->GetSelectedAudioCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] IPSelectedAudioCoder=%s[%d]:\n",
                          m_pBridgedCall->Index(),
                          pSIPEndpoint->GetSelAudCoderStr(), pSIPEndpoint->GetSelectedAudioCoder());

                }
                else
                {
                   // Set the selected 3G coders for SIP call
	           LOG_ENTRY(0, "BridgedCall[%d] - Audio TranscodeMode Disabled(NATIVE)\n", m_pBridgedCall->Index());
                   pSIPEndpoint->SetSelectedAudioCoder(pEndpoint->GetSelectedAudioCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] 3GSelectedAudioCoder=%s[%d]:\n",
                          m_pBridgedCall->Index(),
                          pEndpoint->GetSelAudCoderStr(), pEndpoint->GetSelectedAudioCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] IPSelectedAudioCoder=%s[%d]:\n",
                          m_pBridgedCall->Index(),
                          pSIPEndpoint->GetSelAudCoderStr(), pSIPEndpoint->GetSelectedAudioCoder());
                }
 
                if(pEndpoint->GetVidTranscodeEnabled())
                {
	           LOG_ENTRY(0, "BridgedCall[%d] - Video TranscodeMode Enabled(TRANSCODE_ON)\n", m_pBridgedCall->Index());
                   // For now, Set the 1st Preferred coder for SIP call 
                   pSIPEndpoint->SetSelectedVideoCoder(EndpointMngr::Instance()->GetPreferredSipVideoCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] PreferredSipVideoCoder=%d\n",
                          m_pBridgedCall->Index(),
                          EndpointMngr::Instance()->GetPreferredSipVideoCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] 3GSelectedVidCoder=%s[%d]:\n",
                          m_pBridgedCall->Index(),
                          pEndpoint->GetSelVidCoderStr(), pEndpoint->GetSelectedVideoCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] IPSelectedVidCoder=%s[%d]:\n",
                          m_pBridgedCall->Index(),
                          pSIPEndpoint->GetSelVidCoderStr(), pSIPEndpoint->GetSelectedVideoCoder());
                   // JM - If transcode, pass Dialogic QCIF DCI value to SIP endpoint here
                   LOG_ENTRY(0, "BridgedCall[%d] Use Dialogic DCI:%s\n",m_pBridgedCall->Index(),DIALOGIC_QCIF_DCI_STR);
		   pSIPEndpoint->Set3GEPDCI(DIALOGIC_QCIF_DCI_STR);

                }
                else
                {
                   // Set the selected 3G coders for SIP call
	           LOG_ENTRY(0, "BridgedCall[%d] - Video TranscodeMode Disabled(NATIVE)\n", m_pBridgedCall->Index());
                   pSIPEndpoint->SetSelectedVideoCoder(pEndpoint->GetSelectedVideoCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] 3GSelectedVidCoder=%s[%d]:\n", m_pBridgedCall->Index(),
                          pEndpoint->GetSelVidCoderStr(), pEndpoint->GetSelectedVideoCoder());
                   LOG_ENTRY(0, "BridgedCall[%d] IPSelectedVidCoder=%s[%d]:\n", m_pBridgedCall->Index(),
                          pSIPEndpoint->GetSelVidCoderStr(), pSIPEndpoint->GetSelectedVideoCoder());
                   // JM If native, pass handphone DCI value to SIP endpoint here
                   LOG_ENTRY(0, "BridgedCall[%d] 3GEP DCI:%s\n",m_pBridgedCall->Index(), pEndpoint->Get3GEPDCI());
                   pSIPEndpoint->Set3GEPDCI(pEndpoint->Get3GEPDCI());
                }

        LOG_ENTRY(0, "BridgedCall[%d] CallInfo:%s\n",m_pBridgedCall->Index(), pEndpoint->GetCallInfo());
		METAEVENT tmpMetaevent;
		tmpMetaevent.evttype = (long) APP_CALLREQ;
		tmpMetaevent.evtdev = pSIPEndpoint->GetGCSIPHandle();
		tmpMetaevent.evtdatap = const_cast<char *>(pEndpoint->GetCallInfo());
		tmpMetaevent.evtlen = strlen(pEndpoint->GetCallInfo());
		pSIPEndpoint->ProcessEvent(tmpMetaevent);
	      }
	  }
      }
   }  // End if 3G endpoint connected

/****** JH - outbound call done in IDLE state
   if (APP_SIP_ENDPOINT_ANSWERED == metaevent.evttype)
   {
     Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedCall->M3gIndex());
     if ( pEndpoint )  {
       ISDNEndpoint *pSigEndpoint = pEndpoint->GetSigProtocolEndpoint();
       if (pSigEndpoint) {
         LOG_ENTRY(0, "BridgedCall[%d] - initiating outbound ISDN call\n", m_pBridgedCall->Index());
	 METAEVENT tmpMetaevent;
	 tmpMetaevent.evttype = (long) APP_CALLREQ;
	 tmpMetaevent.evtdev = pSigEndpoint->GetGCISDNHandle();
	 pSigEndpoint->ProcessEvent(tmpMetaevent);
       }
       else  {
         LOG_ERROR(0,"BridgedCall[%d] Expected SS7/ISDN device, found none\n", m_pBridgedCall->Index());
       }
     }
   }  // End if SIP endpoint asnwered
********/

   if ( IsBridgedCallFullyConnected() )
   {
      m_pBridgedCall->ChangeState(CONNECTED_STATE);
   }
}

