/**
* @file bridgedcallopening_state.cpp
* @brief Definition of BridgedCallOpeningState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "bridgedcallopening_state.h"
#include "bridgedcall.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"

// static constant members:
const int BridgedCallOpeningState::SIP_OPENED           = 0x01;
const int BridgedCallOpeningState::M3G_OPENED           = 0x02;
const int BridgedCallOpeningState::SIP_PORT_CONNECTED   = 0x04;
const int BridgedCallOpeningState::M3G_PORT_CONNECTED   = 0x08;
const int BridgedCallOpeningState::FULLY_OPENED         = 0x03;
const int BridgedCallOpeningState::FULLY_PORT_CONNECTED = 0x0F;

//*****************************************************************************
// Function: BridgedCallOpeningState::BridgedCallOpeningState(BridgedCall *pBridgedCall)
// Description: Initializing constructor
// Return:  BridgedCallOpeningState*
// Parameters: BridgedCall *pBridgedCall 
//*****************************************************************************
BridgedCallOpeningState::BridgedCallOpeningState(BridgedCall* pBridgedCall)
   : BridgedCallState(pBridgedCall)
{
}

//*****************************************************************************
// Function: BridgedCallOpeningState::~BridgedCallOpeningState()
// Description: Descructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedCallOpeningState::~BridgedCallOpeningState()
{
}

//*****************************************************************************
// Function: void BridgedCallOpeningState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallOpeningState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      case APP_SIP_ENDPOINT_OPENED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallOpeningState rcvd:APP_SIP_ENDPOINT_OPENED\n", m_pBridgedCall->Index());
         SetStatus(SIP_OPENED);
         ProcessEndpointOpened(metaevent);
         break;
      case APP_M3G_ENDPOINT_OPENED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallOpeningState rcvd:APP_M3G_ENDPOINT_OPENED\n", m_pBridgedCall->Index());
         SetStatus(M3G_OPENED);
         ProcessEndpointOpened(metaevent);
         break;
      case APP_SIP_ENDPOINT_PORT_CONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallOpeningState rcvd:APP_SIP_ENDPOINT_PORT_CONNECTED\n", m_pBridgedCall->Index());
         SetStatus(SIP_PORT_CONNECTED);
         ProcessEndpointPortConnected(metaevent);
         break;
      case APP_M3G_ENDPOINT_PORT_CONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallOpeningState rcvd:APP_M3G_ENDPOINT_PORT_CONNECTED\n", m_pBridgedCall->Index());
         SetStatus(M3G_PORT_CONNECTED);
         ProcessEndpointPortConnected(metaevent);
         break;
      default:
         LOG_ERROR(metaevent.evtdev,"BridgedCall[%d] BridgedCallOpeningState: Unexpected event type: 0x%x\n", 
		   m_pBridgedCall->Index(),metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void BridgedCallOpeningState::ProcessEndpointOpened(METAEVENT metaevent)
// Description: Process endpoint opened event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallOpeningState::ProcessEndpointOpened(METAEVENT metaevent)
{
   if ( IsBridgedCallFullyOpened() )
   {
      // send commands to SIP and M3G endpoints to connect to each other
      SIPEndpoint *pSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx( m_pBridgedCall->SipIndex());
      Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedCall->M3gIndex());

      if ( pSIPEndpoint && pEndpoint )
      {
         pEndpoint->ConnectToPeer(pSIPEndpoint->GetAudioRxPortInfo(), pSIPEndpoint->GetVideoRxPortInfo());
         pSIPEndpoint->ConnectToPeer(pEndpoint->GetAudioRxPortInfo(), pEndpoint->GetVideoRxPortInfo());
      }
      else
      {
         LOG_ENTRY(0,"BridgedCall[%d] Endpoint(s) not found\n",m_pBridgedCall->Index());
      }
   }
}

//*****************************************************************************
// Function: void BridgedCallOpeningState::ProcessEndpointPortConnected(METAEVENT metaevent)
// Description: Process endpoint port connected event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallOpeningState::ProcessEndpointPortConnected(METAEVENT metaevent)
{
   if ( IsBridgedCallFullyPortConnected() )
   {
      m_pBridgedCall->ChangeState(IDLE_STATE);
   }
}

