/**
* @file bridgedmediaplaying_state.cpp
* @brief Definition of BridgedMediaPlayingState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "bridgedmediaplaying_state.h"
#include "bridgedmedia.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"
#include "epstate.h"

//*****************************************************************************
// Function: BridgedMediaPlayingState::BridgedMediaPlayingState(BridgedMedia *pBridgedMedia)
// Description: Initializing constructor
// Return:  BridgedMediaPlayingState*
// Parameters: BridgedMedia *pBridgedMedia 
//*****************************************************************************
BridgedMediaPlayingState::BridgedMediaPlayingState(BridgedMedia* pBridgedMedia) :
   BridgedMediaState(pBridgedMedia)
{
}


//*****************************************************************************
// Function: BridgedMediaPlayingState::~BridgedMediaPlayingState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedMediaPlayingState::~BridgedMediaPlayingState()
{
}

//*****************************************************************************
// Function: void BridgedMediaPlayingState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedMediaPlayingState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      case APP_M3G_ENDPOINT_DISCONNECT:
      LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaPlayingState rcvd:APP_M3G_ENDPOINT_DISCONNECT\n",m_pBridgedMedia->Index());
      break;
      case APP_M3G_ENDPOINT_DISCONNECTED:
      LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaPlayingState rcvd:APP_M3G_ENDPOINT_DISCONNECTED\n",m_pBridgedMedia->Index());
      Process3GDisconnected(metaevent);
      break;
      case APP_MM_ENDPOINT_DISCONNECT: 
            LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaPlayingState rcvd:APP_MM_ENDPOINT_DISCONNECT\n",m_pBridgedMedia->Index());
            Isdn3GDisconnect(metaevent);
      		break;
      case APP_MM_ENDPOINT_DISCONNECTED: 
      LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaPlayingState rcvd:APP_MM_ENDPOINT_DISCONNECTED\n",m_pBridgedMedia->Index());
      break;
      case APP_MM_ENDPOINT_IDLE:
      LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] BridgedMediaPlayingState rcvd:APP_MM_ENDPOINT_IDLE\n",m_pBridgedMedia->Index());
      // MM play has terminated
      ProcessMediaDone(metaevent);
      break;
      default:
         LOG_ERROR(metaevent.evtdev,"BridgedMedia[%d] BridgedMediaPlayingState: Unexpected event type: 0x%x\n",
		   m_pBridgedMedia->Index(),metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void BridgedMediaPlayingState::Isdn3GDisconnect(METAEVENT metaevent)
// Description: Disconnect the 3G call
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedMediaPlayingState::Isdn3GDisconnect(METAEVENT metaevent)
{
      // Disconnect call now
      Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedMedia->M3gIndex());
      if ( pEndpoint )
      {
            //pEndpoint->StopMedia();
            // Try ISDN/SS7 disconnect
            if (pEndpoint->IsSigProtocolManaged())  {
                 LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] Preparing fake GCEV_DISCONNECT for ISDN/SS7 connection\n",
                           m_pBridgedMedia->Index());
                 METAEVENT tmpMetaevent;
                 tmpMetaevent.evttype =  GCEV_DISCONNECTED;
                 tmpMetaevent.evtdev = 0;
                 ISDNEndpoint *pISDNEndpoint = pEndpoint->GetSigProtocolEndpoint();
                 if (pISDNEndpoint && (pISDNEndpoint->GetState() != ISDNEPState::WAITCALL_STATE))
                   pISDNEndpoint->ProcessEvent(tmpMetaevent);
            }	
      }
      // And if there is a loopback peer, do the same
      Endpoint *pLoopbackEndpoint = pEndpoint->GetLoopbackPeer();
      if (pLoopbackEndpoint && (pLoopbackEndpoint->GetState() != EPState::H245_INACTIVE_STATE))
      {
            LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] Stopping loopback peer\n",m_pBridgedMedia->Index());
            pLoopbackEndpoint->StopMedia();
      }

}
//*****************************************************************************
// Function: void BridgedMediaPlayingState::ProcessMMPlaying(METAEVENT metaevent)
// Description: Process an MM playing  event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedMediaPlayingState::ProcessMediaDone(METAEVENT metaevent)
{
   MMEndpoint *pMMEndpoint = EndpointMngr::Instance()->GetMMEPFromIdx( m_pBridgedMedia->MMIndex());
   Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedMedia->M3gIndex());  
   
   // Media Done playing; advance BridgedMedia's state machine to idle 
   m_pBridgedMedia->ChangeState(IDLE_STATE);
   
     
   // Check for valid digit operation
   if ( pMMEndpoint && pMMEndpoint->GetLastDigitRcvd() )   {
      pMMEndpoint->SetCoderSelection();
      LOG_ENTRY(0, "BridgedMedia[%d] LastDigitRcvd='%c'\n",
                   m_pBridgedMedia->Index(),
                   pMMEndpoint->GetLastDigitRcvd());
      LOG_ENTRY(0, "BridgedMedia[%d] MMSelectedAudioCoder=%s\n",
                    m_pBridgedMedia->Index(),
                    pMMEndpoint->GetSelAudCoderStr());
      LOG_ENTRY(0, "BridgedMedia[%d] MMSelectedVideoCoder=%s, MMSelectedVideoResolution=%s,MMSelectedVideoFps=%s, MMSelectedVideoBitRate=%dKbps\n",
                    m_pBridgedMedia->Index(), pMMEndpoint->GetSelVidCoderStr(), pMMEndpoint->GetSelVidResStr(),
                    pMMEndpoint->GetSelVidFpsStr(), pMMEndpoint->GetSelectedVideoBitRate()/1000);
       
      // Start a play. Play terminates with a  single DTMF/UII or a disconnect.
      if (!strcmp(EndpointMngr::Instance()->GetMMOperation(pMMEndpoint->GetLastDigitRcvd()), "PLAY"))  {
         pMMEndpoint->PlayVideoFile(EndpointMngr::Instance()->GetMMPlayAudioFileName(pMMEndpoint->GetLastDigitRcvd()), EndpointMngr::Instance()->GetMMPlayVideoFileName(pMMEndpoint->GetLastDigitRcvd()));
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] Starting video play\n",m_pBridgedMedia->Index());
         
         //Set LastDigitRcvd to NULL to stop continuous play
         pMMEndpoint->SetLastDigitRcvd(char(NULL));
      } 
      else if (!strcmp(EndpointMngr::Instance()->GetMMOperation(pMMEndpoint->GetLastDigitRcvd()), "RECORD"))  {
         // Start a record.  The recording is terminated with a single DTMF/UII or a disconnect.
         // JM - Set the DCI for the Native Record case
            if(!pEndpoint->GetVidTranscodeEnabled())
            {
               pMMEndpoint->SetRecDCIStr(pEndpoint->Get3GEPDCI());
               pMMEndpoint->SetRecDCILen(strlen(pMMEndpoint->GetRecDCIStr())/2);
               LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] Rec DCI[size=%d]:%s\n",m_pBridgedMedia->Index(),pMMEndpoint->GetRecDCILen(),pMMEndpoint->GetRecDCIStr());
            }
            else {
               pMMEndpoint->SetRecDCILen(0);
            }
         pMMEndpoint->RecordVideoFile();
         pEndpoint->SendFastVideoUpdate();
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] Sending VFU and Starting video record\n",m_pBridgedMedia->Index());
      }
      else if (!strcmp(EndpointMngr::Instance()->GetMMOperation(pMMEndpoint->GetLastDigitRcvd()), "DISC")){
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] digit - Disconnect \n",m_pBridgedMedia->Index());
         // Call this to disconnect call 
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] generating APP_MM_ENDPOINT_DISCONNECT event in IDLE_STATE\n", m_pBridgedMedia->Index());
         METAEVENT tmpMetaevent;
         tmpMetaevent.evttype =  APP_MM_ENDPOINT_DISCONNECT;
         tmpMetaevent.evtdev = 0;
         ProcessEvent(tmpMetaevent);
      }
      else {
         LOG_ENTRY(metaevent.evtdev, "BridgedMedia[%d] No MM action to take - go to IDLE \n",m_pBridgedMedia->Index());
         pMMEndpoint->SetLastMessageString("No MM Operation for dtmf");
      }
   }
}

//*****************************************************************************
// Function: void BridgedMediaPlayingState::Process3GDisconnected(METAEVENT metaevent)
// Description: Process a 3G Disconnect event	
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedMediaPlayingState::Process3GDisconnected(METAEVENT metaevent)
{
  // Want to stop the play in progress
  MMEndpoint *pMMEndpoint = EndpointMngr::Instance()->GetMMEPFromIdx( m_pBridgedMedia->MMIndex());
  if (pMMEndpoint)  {
    pMMEndpoint->StopMMPlay();
  } 
}

