/**
* @file bridgedrtspl.cpp
* @brief Definition of BridgedRtsp class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "logger.h"
#include "endpointmgr.h"
#include "bridgedrtsp.h"
#include "locker.h"
#include <string.h>

#include "bridgedrtspopening_state.h"
#include "appevents.h"

//*****************************************************************************
// Function: BridgedRtsp::BridgedRtsp(int index, int m3gIndex, int rtspIndex)
// Description: Initializing constructor
// Return:  BridgedRtsp*
// Parameters: int index 
//             int m3gIndex 
//             int rtspIndex 
//*****************************************************************************
BridgedRtsp::BridgedRtsp(int index, int m3gIndex, int rtspIndex):
   m_pCurrentState(0),
   m_Index(index),
   m_m3gIndex(m3gIndex),
   m_rtspIndex(rtspIndex)
{
   strncpy(m_CurrentStateStr, "OPENING        ",MAX_CURRENT_STATE_STR);
   strcpy(m_LastMessageBuff, "");

   m_pBridgedRtspOpeningState = new BridgedRtspOpeningState(this);
   m_pBridgedRtspIdleState = new BridgedRtspIdleState(this);
   m_pBridgedRtspPlayingState = new BridgedRtspPlayingState(this);
   m_pBridgedRtspRecordingState = new BridgedRtspRecordingState(this);
}

//*****************************************************************************
// Function: BridgedRtsp::~BridgedRtsp()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedRtsp::~BridgedRtsp()
{
   delete m_pBridgedRtspOpeningState;
   delete m_pBridgedRtspIdleState;
   delete m_pBridgedRtspPlayingState;
   delete m_pBridgedRtspRecordingState;
}

//*****************************************************************************
// Function: void BridgedRtsp::ChangeState(BridgedRtspState::E_BRIDGEDRTSPSTATE e_NewCallState)
// Description: Change the object state
// Return: void 
// Parameters: BridgedRtspState::E_BRIDGEDRTSPSTATE e_NewCallState 
//*****************************************************************************
void BridgedRtsp::ChangeState(BridgedRtspState::E_BRIDGEDRTSPSTATE e_NewCallState)
{
   char oldStateStr[MAX_CURRENT_STATE_STR];
   strcpy(oldStateStr, m_pCurrentState->GetStateStr());
   
   // reset the current state for the next call:
   m_pCurrentState->Reset();

   switch ( e_NewCallState )
   {
      case BridgedRtspState::IDLE_STATE:
         m_pCurrentState = m_pBridgedRtspIdleState;
         break;

      case BridgedRtspState::PLAYING_STATE:
         m_pCurrentState = m_pBridgedRtspPlayingState;
         break;

      case BridgedRtspState::RECORDING_STATE:
         m_pCurrentState = m_pBridgedRtspRecordingState;
         break;

      default:
         LOG_ERROR(0,"BridgedRtsp: Unexpected state: %d\n",static_cast<int>(e_NewCallState));
         break;
   }

   // cache the state string in the endpoint
   strcpy(m_CurrentStateStr,m_pCurrentState->GetStateStr());
   LOG_ENTRY(0, "BridgedRtsp[%d] State transition: %s ---> %s\n",m_Index, oldStateStr,
		m_CurrentStateStr);
}


//*****************************************************************************
// Function: void BridgedRtsp::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: bool 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedRtsp::ProcessEvent(METAEVENT metaevent)
{
   m_StateLock.Lock();
   // dispatch event information to appropriate state:
   m_pCurrentState->ProcessEvent(metaevent);
   m_StateLock.Unlock();
}

//*****************************************************************************
// Function: BridgedRtspState::E_BRIDGEDRTSPSTATE BridgedRtsp::GetState()
// Description: Return the current state object
// Return: BridgedRtspState::E_BRIDGEDRTSPSTATE 
// Parameters: none 
//*****************************************************************************
BridgedRtspState::E_BRIDGEDRTSPSTATE BridgedRtsp::GetState()
{
   m_StateLock.Lock();
   BridgedRtspState::E_BRIDGEDRTSPSTATE state = m_pCurrentState->GetState();
   m_StateLock.Unlock();
   return state;
}

//*****************************************************************************
// Function: void BridgedRtsp::SetLastMessageString(char *pMessageStr)
// Description: Store as the last messages
// Return: void 
// Parameters: char *pMessageStr 
//*****************************************************************************
void BridgedRtsp::SetLastMessageString(char *pMessageStr)
{
   int length = (MAX_LAST_MSG_STRING > strlen(pMessageStr)) ? strlen(pMessageStr) : MAX_LAST_MSG_STRING-1;
   memset(m_LastMessageBuff, 0, sizeof(char)*MAX_LAST_MSG_STRING);
   strncpy(&m_LastMessageBuff[0], pMessageStr, length);
   m_LastMessageBuff[length] = '\0';
}

//*****************************************************************************
// Function: char* BridgedRtsp::GetStateString()
// Description: Get a formatted string describing the current state
// Return: char* 
// Parameters: none 
//*****************************************************************************
char *BridgedRtsp::GetStateString()
{
   memset(m_StateBuffer, 0, sizeof(char)*MAX_STATE_STRING);
   sprintf(m_StateBuffer, "BRR%2d: %15s M3G:%d <-> RTSP:%d %s",
           m_Index,
           m_CurrentStateStr,
           m_m3gIndex,
           m_rtspIndex,
           m_LastMessageBuff);
   return m_StateBuffer;
}

//*****************************************************************************
// Function: void BridgedRtsp::Open()
// Description: Set the BridgedRtsp to openming
// Return: void 
// Parameters: none 
//*****************************************************************************
void BridgedRtsp::Open()
{
   m_pCurrentState = m_pBridgedRtspOpeningState;
}
