/**
* @file endpoint.h
* @brief Declaration of the Endpoint class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#ifndef  _ENDPOINT_H
#define  _ENDPOINT_H

#include "port_connect.h"
#include "m3glib.h"
#include "ipmlib.h"
#include "opening_state.h"
#include "connecting_state.h"
#include "h245inactive_state.h"
#include "h245starting_state.h"
#include "openinglcs_state.h"
#include "mediaactive_state.h"
#include "rxmediastopped_state.h"
#include "closinglcs_state.h"
#include "baseendpoint.h"
#include <time.h>
#include "locker.h"
#include "config.h"

typedef struct
{
   int  AL;
   int  AL2Size;
   int  AL3Size;
   bool isAnnexA;
   bool isADblFlag;
   bool isAnnexB;
   bool isBHdr;
   bool isRSCode;
   bool isMblOpXmt;
   int  bitRate;
} LOCAL_H223_CAPS;


typedef enum 
{
   NONE,
   G723,
   AMR,
   H263,
   MPEG4, 
   H264,
} MEDIA_CAP;


typedef struct
{
   int  audioAL;
   int  videoAL;
   bool isAudioSegmentable;
   bool isVideoSegmentable;
   int  AL3ControlFieldSize;
   int  AL3SendBufferSize;
} LOCAL_H223_LC_PARAMS;


typedef struct
{
   bool             acceptRedundantSlices;
   unsigned int     profileIOP;
   bool             nalAlignedMode;
} LOCAL_H264_PARAMS;

class ISDNEndpoint;

#define MAX_MEDIA_CAPS        5


//*****************************************************************************
// Class: ::Endpoint
// Description: This is the 3G endpoint class
//*****************************************************************************
class Endpoint : public BaseEndpoint
{
public:
   Endpoint(int numEP, int numPeer, int numLoopback, const char *dtiName, const char *m3gName, const char *mapID);
   ~Endpoint();

   void SetPreferredH223Defaults(M3G_H223_SESSION *pPreferredH223Defaults)
   {
      m_PreferredH223Defaults = *pPreferredH223Defaults;
   }

   void SetPreferredH223Caps(LOCAL_H223_CAPS *pLocalH223Caps)
        { m_PreferredH223Caps = *pLocalH223Caps;}

   void SetH223LCParams(LOCAL_H223_LC_PARAMS *pLocalH223LCParams)
        { m_PreferredH223LCParams = *pLocalH223LCParams;}

   void SetH264Params(LOCAL_H264_PARAMS *pLocalH264Params)
        { m_PreferredH264Params = *pLocalH264Params;}

   void SetTracing(unsigned int traceLevel, const char *fileName)
   {
       m_TraceLevel = traceLevel;
       strncpy(m_TraceFileName, fileName, MAX_TRACE_FILE_NAME-1);
   }

   void SetDCI(unsigned char *pDciOctetStr, unsigned char octetStrSize, unsigned int dciMode)
   {
       memcpy(m_DciOctetStr, pDciOctetStr, octetStrSize);
       m_DciOctetStrSize = octetStrSize;
       m_DciMode = dciMode;
   }

// SKS
   void Store3GEPDCI(char pDciStr[128]) 
   {
       strcpy(m_3GEPDciStr,pDciStr);
   }
   char *Get3GEPDCI() 
   {
       return m_3GEPDciStr;
   }
   unsigned char Get3GEPDCIOctStrSize(){ return m_DciOctetStrSize;}

   void SetPreferredMediaCaps(int numCaps, MEDIA_CAP *pPreferredMediaList);

   void ProcessEvent(long evtType, void *pEvtData, long  evtLen, long  evtDev);
   void ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt);
   void ProcessSetNetworkHndEvent(SRL_DEVICE_HANDLE hnd);
   EPState::E_EPSTATE GetState();
   void OpenSubDevs();
   void CloseSubDevs();

   M3G_LOGICAL_CHANNEL_NUMBER GetAudioTxLCN() { return m_TxAudioLCN;}
   M3G_LOGICAL_CHANNEL_NUMBER GetAudioRxLCN() { return m_RxAudioLCN;}
   M3G_LOGICAL_CHANNEL_NUMBER GetVideoTxLCN() { return m_TxVideoLCN;}
   M3G_LOGICAL_CHANNEL_NUMBER GetVideoRxLCN() { return m_RxVideoLCN;}
   char *GetStateString();

   SRL_DEVICE_HANDLE GetNetworkHandle() { return m_NetworkHandle;}
   int GetNetworkDeviceHandle() { return m_NetworkDeviceHandle;}
   SRL_DEVICE_HANDLE GetControlHandle() {return m_ControlHandle;}
   SRL_DEVICE_HANDLE GetAudioHandle() {return m_AudioHandle; }
   SRL_DEVICE_HANDLE GetVideoHandle() {return m_VideoHandle; }

   int GetIndex() { return m_Index;}
   int GetPeerIndex() { return m_PeerIndex;}
   Endpoint * GetPeer() { return m_pPeerEndpoint;}
   Endpoint * GetLoopback() { return m_pLoopbackEndpoint;}
   char * GetTransportName() { return m_transportName;}

   void ChangeState(EPState::E_EPSTATE e_NewState);

   void RecordPortInfoList(long evtDev, long evtType, DM_PORT_INFO_LIST* pPortInfoList);

   void SetAudioRxPortInfoList(DM_PORT_INFO_LIST *pRxPortInfoList)
   { m_AudioRxPortList = *pRxPortInfoList;}

   void SetVideoRxPortInfoList(DM_PORT_INFO_LIST *pRxPortInfoList)
   { m_VideoRxPortList = *pRxPortInfoList;}

   void SetAudioTxPortInfoList(DM_PORT_INFO_LIST *pTxPortInfoList)
   { m_AudioTxPortList = *pTxPortInfoList;}

   void SetVideoTxPortInfoList(DM_PORT_INFO_LIST *pTxPortInfoList)
   { m_VideoTxPortList = *pTxPortInfoList;}

   void Set3GNbupTxPortInfoList(DM_PORT_INFO_LIST *pTxPortInfoList)
   { m_3GNbupTxPortList = *pTxPortInfoList;}

   void Set3GNbupRxPortInfoList(DM_PORT_INFO_LIST *pRxPortInfoList)
   { m_3GNbupRxPortList = *pRxPortInfoList;}

   bool IsClearChannel() { return m_transportName[0] == 'd' && m_transportName[1] == 't';}
   bool IsSigProtocolManaged();

   DM_PORT_INFO& GetAudioRxPortInfo() { return m_AudioRxPortList.port_info[0];}
   DM_PORT_INFO& GetAudioTxPortInfo() { return m_AudioTxPortList.port_info[0];}
   DM_PORT_INFO& GetVideoRxPortInfo() { return m_VideoRxPortList.port_info[0];}
   DM_PORT_INFO& GetVideoTxPortInfo() { return m_VideoTxPortList.port_info[0];}
   DM_PORT_INFO& Get3GNbupRxPortInfo() { return m_3GNbupRxPortList.port_info[0];}
   DM_PORT_INFO& Get3GNbupTxPortInfo() { return m_3GNbupTxPortList.port_info[0];}

   bool IsEndpointAllPortsRcvd();
   void ConnectToLoopback();
   void ConnectToPeer(DM_PORT_INFO& a_audioPortRxInfo, DM_PORT_INFO& a_videoPortRxInfo);
   void DisconnectFromPeer();

   void RecordCapsList(long  evtDev, M3G_CAPS_LIST *pCapsList);
   void RecordH223Cap(long  evtDev, M3G_H223_CAPABILITY *pH223Cap);
   void RecordAudioCap(long  evtDev, M3G_AUDIO_CAPABILITY *pAudioCapList);
   void RecordVideoCap(long  evtDev, M3G_VIDEO_CAPABILITY *pVideoCapList);
   void RecordLCMuxParams(long  evtDev, M3G_H223_LC_PARAMS* pLCMux);
#ifdef MONA
   void RecordStatistics(long evtDev, M3G_CALL_STATISTICS* pStats);
#endif

   void SetLocalH223Caps(M3G_CAPS_LIST *pLocalH223Caps)
   { m_LocalH223Caps = *pLocalH223Caps;}

   void SetLocalAudioCaps(M3G_CAPS_LIST *pLocalAudioCaps)
   { m_LocalAudioCaps = *pLocalAudioCaps;}

   void SetLocalVideoCaps(M3G_CAPS_LIST *pLocalVideoCaps)
   { m_LocalVideoCaps = *pLocalVideoCaps;}
   void ConnectTSAndGetPorts();
   void DisconnectTS();
   void GetLocalCaps();
   void SetTCS();
   void StartH245();
   void SendUII();
   void SendFastVideoUpdate();
   void GetMatchedCaps();
   void OpenLCs(M3G_AUDIO_CAPABILITY* pAudioCap, M3G_E_ADAPTATION_LYR_TYPE audioAL, 
                M3G_VIDEO_CAPABILITY* pVideoCap, M3G_E_ADAPTATION_LYR_TYPE videoAL);
   void AckOLCReq(M3G_REMOTE_OLC_REQ *pOLCReq);
   void Reinitialize();

   void SetAudioTxLCN(M3G_LOGICAL_CHANNEL_NUMBER lcn) { m_TxAudioLCN = lcn;}
   void SetAudioRxLCN(M3G_LOGICAL_CHANNEL_NUMBER lcn) { m_RxAudioLCN = lcn;}
   void SetVideoTxLCN(M3G_LOGICAL_CHANNEL_NUMBER lcn) { m_TxVideoLCN = lcn;}
   void SetVideoRxLCN(M3G_LOGICAL_CHANNEL_NUMBER lcn) { m_RxVideoLCN = lcn;}
   void SetTxVideoCapType(M3G_E_VIDEO_TYPE txVidCapType);
   void SetTxAudioCapType(M3G_E_AUDIO_TYPE txAudCapType);

   M3G_E_VIDEO_TYPE GetTxVideoCapType(){ return m_TxVideoCapType;}
   void StartAudio(M3G_E_DIRECTION direction);
   void StartVideo(M3G_E_DIRECTION direction);
   void StopMedia();
   void StopRxMedia();
   void RestartFDXMedia();
   void StopAudio();
   void StopVideo();
   void CloseLCs();
   void Stop_H245();
   void SetLastMessageString(const char *pMessageStr);
   void Shutdown();  
   void Abort();  
   void SetDigitToSend(char digit) { m_DigitToSend = digit;}

   void SetNetworkConfig();
   void ResetNetworkConfig();
   void ReSetNetworkConfig();
   void LogGlobalCallError();
   void SetParameters();

   char *MapID() { return m_mapID;}

   void SetCallInfo( const char *callInfo) { strcpy( m_callInfo, callInfo);}
   const char *GetCallInfo() { return m_callInfo;}
   Endpoint *GetLoopbackPeer() {return m_pLoopbackEndpoint;}
   void Notify(int event);
   EPState   *GetCurState() { return m_pCurrentState;}
   bool ReinitDone() {return m_ReinitDone;}
   void SetSigProtocolEndpoint(ISDNEndpoint *sigEP)  { m_pSigProtocolEndpoint = sigEP; }
   ISDNEndpoint *GetSigProtocolEndpoint()  { return m_pSigProtocolEndpoint; }
  
   // Nbup related
   void SetNbupType(char nbupType) {m_nbupType = nbupType;}
   void SetRemoteNbupIPAddr(const char *remoteNbupIPAddr) { strcpy(m_remoteNbupIPAddr, remoteNbupIPAddr);}
   void SetRemoteNbupPort(unsigned short remoteNbupPort) { m_remoteNbupPort = remoteNbupPort; }
   SRL_DEVICE_HANDLE GetNbupHandle() {return m_NetworkHandle; }
   bool IsNbup() { return m_transportName[0] == 'i' && m_transportName[1] == 'p';}
   bool GetIsLoopbackNotNetwork() { return m_transportName[0] == 'N';}
   void EnableNbupEvents ();
   void StartNbupMedia ();
   void StopNbupMedia ();
   bool IsNbupSend () { return m_nbupType == 'S';}
   bool IsNbupReceive () { return m_nbupType == 'R';}
   void NbupInitSend();
   void NbupInitResponseSend();
   void GetNbupPortInfo ();
   void StopNbupDevice();
   void ConnectIPMNbupTo3GNbup(SRL_DEVICE_HANDLE deviceHandle, DM_PORT_INFO &TxPortInfo,
                               DM_PORT_INFO &RxPortInfo);
   void DisconnectIPMNbupFrom3GNbup(SRL_DEVICE_HANDLE deviceHandle, DM_PORT_INFO &TxPortInfo,
                                    DM_PORT_INFO &RxPortInfo);
   void RecordIPMNbupPortInfo(long evtDev, long evtType, DM_PORT_INFO *portInfo);
   DM_PORT_INFO& GetIPMNbupRxPortInfo() { return m_IPMNbupRxPortInfo;}
   DM_PORT_INFO& GetIPMNbupTxPortInfo() { return m_IPMNbupTxPortInfo;}
   bool IsNbupIPMPortConnected() { return m_NbupIPMPortConnected;}
   bool IsNbup3GPortConnected() { return m_Nbup3GPortConnected;}
   void StartTracing();
   void StopTracing();
   void SetInbandDCITxTo3G();
   void SetInbandH264DCITxTo3G();
   void SetInbandDCIRvdFrom3G();
   bool IsVidCoderPreferred(E_SEL_VID_CODER VidCoder);
   bool IsAudCoderPreferred(E_SEL_AUD_CODER AudCoder);
#ifdef ISDN_CALL_OUT
   void InitiateISDNCall();
#endif

   // JS 1/8/09
   const char* GetMobileDCIRxdStr() { return m_MobileDCIRxdStr.c_str();}
   unsigned GetMobileDCIRxdStrLen() { return m_MobileDCIRxdStr.length();}
   // JS 1/8/09
private:
   M3G_E_ADAPTATION_LYR_TYPE GetMatchingAudioAL(M3G_CAPS_LIST *pMatchedH223CapsList);
   M3G_E_ADAPTATION_LYR_TYPE GetMatchingVideoAL(M3G_CAPS_LIST *pMatchedH223CapsList);

   static const unsigned int  DCI_MODE_DEFAULT;
   static const unsigned int  DCI_MODE_INBAND;
   static const unsigned int  DCI_MODE_H245_OLC;
   int                        m_Index;
   int                        m_PeerIndex;
   Endpoint                  *m_pPeerEndpoint;
   int                        m_LoopbackIndex;
   Endpoint                  *m_pLoopbackEndpoint;
   bool                       m_ProcessingPeerPortConnect;

   SRL_DEVICE_HANDLE          m_NetworkHandle;
   int                        m_NetworkDeviceHandle;
   SRL_DEVICE_HANDLE          m_ControlHandle;
   SRL_DEVICE_HANDLE          m_AudioHandle;
   SRL_DEVICE_HANDLE          m_VideoHandle;
   bool                       m_IsLoopback;
   bool                       m_IsManagedBySigProtocol;
   M3G_H223_SESSION           m_PreferredH223Defaults;
   LOCAL_H223_CAPS            m_PreferredH223Caps;

   LOCAL_H223_LC_PARAMS       m_PreferredH223LCParams;
   LOCAL_H264_PARAMS          m_PreferredH264Params;
   MEDIA_CAP                  m_PreferredMediaCapsList[MAX_MEDIA_CAPS];
   int                        m_NumPreferredMediaCaps;

   int                        m_numParmsSet;

   M3G_CAPS_LIST              m_LocalH223Caps;
   M3G_CAPS_LIST              m_LocalAudioCaps;
   M3G_CAPS_LIST              m_LocalVideoCaps;

   DM_PORT_INFO_LIST          m_AudioRxPortList;
   DM_PORT_INFO_LIST          m_AudioTxPortList;
   DM_PORT_INFO_LIST          m_VideoRxPortList;
   DM_PORT_INFO_LIST          m_VideoTxPortList;
   DM_PORT_INFO_LIST          m_3GNbupRxPortList;
   DM_PORT_INFO_LIST          m_3GNbupTxPortList;

   M3G_LOGICAL_CHANNEL_NUMBER m_RxAudioLCN;
   M3G_LOGICAL_CHANNEL_NUMBER m_TxAudioLCN;
   M3G_LOGICAL_CHANNEL_NUMBER m_RxVideoLCN;
   M3G_LOGICAL_CHANNEL_NUMBER m_TxVideoLCN;
   char                       m_AudTxCapStr[MAX_MEDIA_STR];
   char                       m_VidTxCapStr[MAX_MEDIA_STR];
   char                       m_CurrentStateStr[MAX_CURRENT_STATE_STR];
   char                       m_DigitToSend;
   OpeningState              *m_pOpeningState;
   ConnectingState           *m_pConnectingState;
   H245InactiveState         *m_pH245InactiveState;
   H245StartingState         *m_pH245StartingState;
   OpeningLCsState           *m_pOpeningLCsState;
   MediaActiveState          *m_pMediaActiveState;
   ClosingLCsState           *m_pClosingLCsState;
   EPState                   *m_pCurrentState;
   RxMediaStoppedState       *m_pRxMediaStoppedState;
   CLocker                    m_StateLock;
   char                       m_transportName[DEV_NAME_SIZE];
   bool                       m_ReinitDone;

   char                       m_m3gName[DEV_NAME_SIZE];
   char                       m_mapID[DEV_NAME_SIZE];
   char                       m_StateBuffer[MAX_STATE_STRING];
   char                       m_LastMessageBuff[MAX_LAST_MSG_STRING];
   char                       m_callInfo[LINE_BUFFER_SIZE];
   ISDNEndpoint	 	         *m_pSigProtocolEndpoint;
   char                       m_TraceFileName[MAX_TRACE_FILE_NAME];
   unsigned int               m_TraceLevel;

   unsigned char              m_DciOctetStr[OCTET_STRING_SIZE];
   unsigned char              m_DciOctetStrSize;
   unsigned int               m_DciMode;
   M3G_E_VIDEO_TYPE           m_TxVideoCapType;

   // JS 1/8/09
   unsigned char              m_MobileDCIRxdOctetStr[OCTET_STRING_SIZE];
   unsigned char              m_MobileDCIRxdOctetStrLen;
   string                     m_MobileDCIRxdStr;
   // JS 1/8/09
   char              			m_3GEPDciStr[128];
   // Nbup-related
   char                       m_nbupType;
   char                       m_remoteNbupIPAddr[SIP_ADDR_SIZE];
   unsigned short             m_remoteNbupPort;
   IPM_MEDIA_INFO             m_NbupLocalMediaInfo;
   bool                       m_NbupMediaStarted;
   bool                       m_NbupIPMPortConnected;
   bool                       m_Nbup3GPortConnected;
   DM_PORT_INFO               m_IPMNbupRxPortInfo;
   DM_PORT_INFO               m_IPMNbupTxPortInfo;
};

#endif


