/**
* @file endpointmgr.cpp
* @brief Definition of EndpointMngr class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/


#include "m3glib.h"
#include "config.h"
#include "endpointmgr.h"
#include "endpoint.h"

#include "main.h"
#include "logger.h"
#include <iostream>
#include <map>
#include "terminal.h"


using namespace std;

// initialize singleton instance
EndpointMngr* EndpointMngr::m_pInstance = NULL;

// Global terminal for exit cleanup
extern Terminal *g_pTerminal;

const int EndpointMngr::AUDIO_PORT = (1<<1);
const int EndpointMngr::VIDEO_PORT = (1<<2);
const int EndpointMngr::NBUP_PORT  = (1<<3);

//*****************************************************************************
// Function: EndpointMngr::EndpointMngr()
// Description: Constructor
// Return:  EndpointMngr*
// Parameters: none 
//*****************************************************************************
EndpointMngr::EndpointMngr() :
   m_pLogger(NULL),
   m_IsLoggingEnabled(false),
   m_NumParameters(0),
   m_EnableEventMask(0),
   m_DisableEventMask(0),
   m_BoardHandle(0),
   m_GcSipBoardHandle(0),
   m_NumEPs(0),
   m_NumSIPEPs(0),
   m_NumMMEPs(0),
   m_NumCalls(0),
   m_NumMedias(0),
   m_NumHairps(0),
   m_NumISDNEPs(0),
   m_NumRTSPEPs(0),
   m_NumBridgedRtsp(0),
   m_maxSipTimeSlot(0),
   m_mmAudCoder(AUD_CODER_NONE),
   m_mmVidCoder(VID_CODER_NONE),
   m_mmVidRes(VID_RES_NONE),
   m_BrdTraceLevel(0)
{
   strcpy(m_RtspOperation, "PLAY");
   memset(&m_ParameterTbl, 0, MAX_PARAMETERS * sizeof(M3G_PARM_INFO));
#if USE_HARDCODED_SIZE
   memset(&m_EPIndexTbl, 0, MAX_DEVICES * sizeof(int));
   memset(&m_SIPEPIndexTbl, 0, MAX_DEVICES * sizeof(int));
   memset(&m_MMEPIndexTbl, 0, MAX_DEVICES * sizeof(int));
   memset(&m_RTSPEPIndexTbl, 0, MAX_DEVICES * sizeof(int));
   memset(&m_CallIndexTbl, 0, MAX_DEVICES * sizeof(int));
   memset(&m_MediaIndexTbl, 0, MAX_DEVICES * sizeof(int));
   memset(&m_BridgedRtspIndexTbl, 0, MAX_DEVICES * sizeof(int));
#endif
   strcpy(m_BrdTraceFileName, "");
   memset(&m_VendorId, 0, sizeof(M3G_VENDORID_INFO));
   memset(&m_PreferredSipVideoCoderList, 0, (VID_CODER_LAST * sizeof(E_SEL_VID_CODER)));
   memset(&m_PreferredSipAudioCoderList, 0, (AUD_CODER_LAST * sizeof(E_SEL_AUD_CODER)));
}


//*****************************************************************************
// Function: EndpointMngr::~EndpointMngr()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
EndpointMngr::~EndpointMngr()
{
}


//*****************************************************************************
// Function: EndpointMngr* EndpointMngr::Instance()
// Description: Return the Singleton instance
// Return: EndpointMngr* 
// Parameters: none 
//*****************************************************************************
EndpointMngr *EndpointMngr::Instance()
{
   // this will be instantiated first from main thread during intitialization
   // thus no need for lociking:
   if ( NULL == m_pInstance )
   {
      m_pInstance = new EndpointMngr();
   }
   return m_pInstance;
}


//*****************************************************************************
// Function: void EndpointMngr::Terminate()
// Description: Terminate and delete the Singleton instance
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::Terminate()
{
   if ( NULL != m_pInstance )
   {
      m_pInstance->Cleanup();
      delete m_pInstance;
      m_pInstance = NULL;
   }
}

//*****************************************************************************
// Function: void EndpointMngr::Cleanup()
// Description: Cleanup all resources managed by this manager
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::Cleanup()
{
   // delete all elements of map:
   map<int, Endpoint*>::iterator itr;
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      Endpoint *pEndpoint = (*itr).second;
      delete pEndpoint;
   }
   m_EPContainer.clear();

   // delete all elements of map:
   map<int, SIPEndpoint*>::iterator sipitr;
   for ( sipitr = m_SIPEPContainer.begin(); sipitr != m_SIPEPContainer.end(); sipitr++ )
   {
      SIPEndpoint *pSIPEndpoint = (*sipitr).second;
      delete pSIPEndpoint;
   }
   m_SIPEPContainer.clear();

   // delete all elements of map:
   map<int, MMEndpoint*>::iterator mmitr;
   for ( mmitr = m_MMEPContainer.begin(); mmitr != m_MMEPContainer.end(); mmitr++ )
   {
      MMEndpoint *pMMEndpoint = (*mmitr).second;
      delete pMMEndpoint;
   }
   // delete all elements of map:
   map<int, ISDNEndpoint*>::iterator isdnitr;
   for ( isdnitr = m_ISDNEPContainer.begin(); isdnitr != m_ISDNEPContainer.end(); isdnitr++ )
   {
      ISDNEndpoint *pISDNEndpoint = (*isdnitr).second;
      delete pISDNEndpoint;
   }
   m_ISDNEPContainer.clear();
   m_MMEPContainer.clear();
#ifdef USE_RTSP
   map<int, RTSPEndpoint*>::iterator rtspitr;
   for ( rtspitr = m_RTSPEPContainer.begin(); rtspitr != m_RTSPEPContainer.end(); rtspitr++ )
   {
      RTSPEndpoint *pRTSPEndpoint = (*rtspitr).second;
      delete pRTSPEndpoint;
   }

   m_RTSPEPContainer.clear();
#endif

   LOG_ENTRY(0, "About to close board and stop gc\n");
   CloseM3GBoard();
   StopGlobalCall();
   StopM3G();
}

//*****************************************************************************
// Function: int EndpointMngr::AddEndpoint(int numEP, const char *epType, const char *controlName, const char *streamName, int numPeer, int numLoopbackPeer, const char *mapID)
// Description: Add a 3G endpoint
// Return: int 
// Parameters: int numEP 
//             const char *epType 
//             const char *controlName 
//             const char *streamName 
//             int numPeer 
//             int numLoopbackPeer 
//             const char *mapID 
//*****************************************************************************
int EndpointMngr::AddEndpoint(int numEP, 
                             const char *epType,
                             const char *controlName,
                             const char *streamName, 
                             int numPeer,
                             int numLoopbackPeer,
#ifdef ISDN_CALL_OUT
                             const char *mapID,
                             bool isAudXcodeEnabled,
                             bool isVidXcodeEnabled,
                             const char *dialString)
#else
                             const char *mapID,
                             bool isAudXcodeEnabled,
                             bool isVidXcodeEnabled)
#endif
{
   map<int, Endpoint*>::iterator itr = m_EPContainer.find(numEP);
   if ( itr == m_EPContainer.end() )
   {
      // index not found, so add a new endpoint.
      if ( 'm' == epType[0] )
      {
#if USE_HARDCODED_SIZE
         if (m_EPContainer.size() >= MAX_DEVICES){
            LOG_ERROR(0,"Maximum number of EP devices already configured(EP:%d,size=%d,MAX=%d)\n",
                      numEP, m_EPContainer.size(),MAX_DEVICES);
            cout << "ERROR: Maxmimum number of EP devices already configured."  << endl;
            return B3G_ERROR;
         }
#endif
         Endpoint *pEndpoint = new Endpoint(numEP, numPeer, numLoopbackPeer, streamName, controlName, mapID);
         if ( !m_EPContainer.insert(make_pair(numEP, pEndpoint)).second )
         {
            return B3G_ERROR;
         }
         pEndpoint->SetAudTranscodeEnabled(isAudXcodeEnabled);
         pEndpoint->SetVidTranscodeEnabled(isVidXcodeEnabled);
         ISDNEndpoint *pISDNEndpoint;
         switch (streamName[0])
         {
           case 'I':
#if USE_HARDCODED_SIZE
             if (m_ISDNEPContainer.size() >= MAX_DEVICES){
                LOG_ERROR(0,"Maximum number of ISDN EP devices already configured(EP:%d,size=%d,MAX=%d)\n",
                          numEP,m_ISDNEPContainer.size(),MAX_DEVICES);
                cout << "ERROR: Maxmimum number of ISDN EP devices already configured."  << endl;
               return B3G_ERROR;
             }
#endif

#ifdef ISDN_CALL_OUT
             pISDNEndpoint = new ISDNEndpoint(numEP + 1000, streamName + 1, pEndpoint, ISDN_ENDPOINT, dialString);
#else
             pISDNEndpoint = new ISDNEndpoint(numEP + 1000, streamName + 1, pEndpoint, ISDN_ENDPOINT);
#endif
             if ( !m_ISDNEPContainer.insert(make_pair(numEP + 1000, pISDNEndpoint)).second )
             {
                return B3G_ERROR;
             }
             pEndpoint->SetSigProtocolEndpoint(pISDNEndpoint);
             break;

           case 'S':
#if USE_HARDCODED_SIZE
             if (m_ISDNEPContainer.size() >= MAX_DEVICES){
                LOG_ERROR(0,"Maximum number of SS7 EP devices already configured(EP:%d,size=%d,MAX=%d)\n",
                          numEP,m_ISDNEPContainer.size(),MAX_DEVICES);
                cout << "ERROR: Maxmimum number of SS7 EP devices already configured."  << endl;
               return B3G_ERROR;
             }
#endif
#ifdef ISDN_CALL_OUT
             pISDNEndpoint = new ISDNEndpoint(numEP + 1000, streamName + 1, pEndpoint, SS7_ENDPOINT, dialString);
#else
             pISDNEndpoint = new ISDNEndpoint(numEP + 1000, streamName + 1, pEndpoint, SS7_ENDPOINT);
#endif
             if ( !m_ISDNEPContainer.insert(make_pair(numEP + 1000, pISDNEndpoint)).second )
             {
                return B3G_ERROR;
             }
             pEndpoint->SetSigProtocolEndpoint(pISDNEndpoint);
             break;
         }
      }
      else if ( 's' == epType[0] )
      {
#if USE_HARDCODED_SIZE
         if (m_SIPEPContainer.size() >= MAX_DEVICES){
            LOG_ERROR(0,"Maximum number of SIP EP devices already configured(EP:%d,size=%d,MAX=%d)\n",
                      numEP,m_SIPEPContainer.size(),MAX_DEVICES);
            cout << "ERROR: Maxmimum number of SIP EP devices already configured."  << endl;
            return B3G_ERROR;
         }
#endif
         // SKS
         char *tempStr = 0;
         int tempTimeSlot;
         tempStr=strstr(controlName,"iptB1T");
         if(tempStr!=NULL) {
           sscanf(controlName,"iptB1T%d",&tempTimeSlot);
         }
         if(tempTimeSlot>m_maxSipTimeSlot) {
           m_maxSipTimeSlot=tempTimeSlot;
         }
         // SKS calculation ends here


         SIPEndpoint *pSIPEndpoint = new SIPEndpoint(numEP,controlName, streamName, mapID);
         if ( !m_SIPEPContainer.insert(make_pair(numEP, pSIPEndpoint)).second )
         {
            return B3G_ERROR;
         }
         pSIPEndpoint->SetAudTranscodeEnabled(isAudXcodeEnabled);
         pSIPEndpoint->SetVidTranscodeEnabled(isVidXcodeEnabled);
      }
      else if ( 'v' == epType[0] )
      {
#if USE_HARDCODED_SIZE
         if (m_MMEPContainer.size() >= MAX_DEVICES){
            LOG_ERROR(0,"Maximum number of MM EP devices already configured(EP:%d,size=%d,MAX=%d)\n",
                      numEP,m_MMEPContainer.size(),MAX_DEVICES);
            cout << "ERROR: Maxmimum number of MM EP devices already configured."  << endl;
            return B3G_ERROR;
         }
#endif
         MMEndpoint *pMMEndpoint = new MMEndpoint(numEP, streamName, mapID);
         if ( !m_MMEPContainer.insert(make_pair(numEP, pMMEndpoint)).second )
         {
            return B3G_ERROR;
         }
         pMMEndpoint->SetAudTranscodeEnabled(isAudXcodeEnabled);
         pMMEndpoint->SetVidTranscodeEnabled(isVidXcodeEnabled);
      }
#ifdef USE_RTSP
      else if ( 'r' == epType[0] )
      {
         RTSPEndpoint *pRTSPEndPoint = new RTSPEndpoint(numEP, streamName, mapID);
         if ( !m_RTSPEPContainer.insert(make_pair(numEP, pRTSPEndPoint)).second)
         {
            LOG_ERROR(0, "Could not add RTSP index %d object\n", numEP);
            return B3G_ERROR;
         }
         else
            LOG_ENTRY(0, "Added RTSP index %d object\n", numEP);

         pRTSPEndPoint->SetAudTranscodeEnabled(isAudXcodeEnabled);
         pRTSPEndPoint->SetVidTranscodeEnabled(isVidXcodeEnabled);
      }
#endif
   }
   else
   {
      cout << "ERROR: EP:" << numEP << " already configured."  << endl;
      return B3G_ERROR;
   }
   return B3G_SUCCESS;
}


//*****************************************************************************
// Function: int EndpointMngr::AddParameter(int parmId, int intVal)
// Description: Add a 3G parameter
// Return: int 
// Parameters: int parmId 
//             int intVal 
//*****************************************************************************
int EndpointMngr::AddParameter(unsigned int parmId, int intVal)
{
   if (m_NumParameters >= MAX_PARAMETERS)
   {
      cout << "ERROR: Maxmimum number of parameters already configured."  << endl;
      return B3G_ERROR;
   }

   // parmID value already validated in DemoConfig
   m_ParameterTbl[m_NumParameters].version = M3G_LIBRARY_VERSION;
   m_ParameterTbl[m_NumParameters].parameterType = static_cast<M3G_E_PRM_TYPE>(parmId);
   m_ParameterTbl[m_NumParameters].parmValue.bitmask = intVal;
   m_NumParameters++;

   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: void EndpointMngr::GetParameters(int *numParameters, M3G_PARM_INFO **parameterTbl)
// Description: Provide access to the 3G poarameters
// Return: void 
// Parameters: int *numParameters 
//             M3G_PARM_INFO **parameterTbl 
//*****************************************************************************
void EndpointMngr::GetParameters( int* numParameters, M3G_PARM_INFO** parameterTbl)
{
   *numParameters = m_NumParameters;
   *parameterTbl = m_ParameterTbl;
}

//*****************************************************************************
// Function: int EndpointMngr::AddEvent(int evtId, bool isEnableEvt)
// Description: Enable or disable a 3G event
// Return: int 
// Parameters: int evtId 
//             bool isEnableEvt 
//*****************************************************************************
int EndpointMngr::AddEvent(int evtId, bool isEnableEvt)
{
   // evtId value already validated in DemoConfig
   if (true == isEnableEvt)
   {
      m_EnableEventMask |= evtId;
   }
   else
   {
      m_DisableEventMask |= evtId;
   }

   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: int EndpointMngr::InitH223Session(int startEP, int endEP, M3G_H223_SESSION *pH223Session)
// Description: Initialize an H223 session
// Return: int 
// Parameters: int startEP 
//             int endEP 
//             M3G_H223_SESSION *pH223Session 
//*****************************************************************************
int EndpointMngr::InitH223Session(unsigned int startEP, unsigned int endEP, M3G_H223_SESSION *pH223Session)
{
   Endpoint* pEndpoint = NULL;
   for (unsigned int epNum = startEP; epNum <= endEP; epNum++)
   {
      map<int, Endpoint*>::iterator itr = m_EPContainer.find(epNum);
      if (itr == m_EPContainer.end())
      {
         cout << "ERROR: EP:" << epNum << " unspecified."  << endl;
         return B3G_ERROR;
      }
      pEndpoint = (*itr).second;
      pEndpoint->SetPreferredH223Defaults(pH223Session);
   }
   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: int EndpointMngr::InitH223Caps(int startEP, int endEP, LOCAL_H223_CAPS *pLocalH223Caps)
// Description: Initialize H223 capabilities
// Return: int 
// Parameters: int startEP 
//             int endEP 
//             LOCAL_H223_CAPS *pLocalH223Caps 
//*****************************************************************************
int EndpointMngr::InitH223Caps(unsigned int startEP, unsigned int endEP, LOCAL_H223_CAPS *pLocalH223Caps)
{
   Endpoint* pEndpoint = NULL;
   for (unsigned int epNum = startEP; epNum <= endEP; epNum++)
   {
      map<int, Endpoint*>::iterator itr = m_EPContainer.find(epNum);
      if (itr == m_EPContainer.end())
      {
         cout << "ERROR: EP:" << epNum << " unspecified."  << endl;
         return B3G_ERROR;
      }
      pEndpoint = (*itr).second;
      pEndpoint->SetPreferredH223Caps(pLocalH223Caps);
   }
   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: int EndpointMngr::InitMediaCaps(int startEP, int endEP, int numCaps, MEDIA_CAP *pPreferredMediaList)
// Description: Initialize media capabilities
// Return: int 
// Parameters: int startEP 
//             int endEP 
//             int numCaps 
//             MEDIA_CAP *pPreferredMediaList 
//*****************************************************************************
int EndpointMngr::InitMediaCaps(unsigned int startEP, unsigned int endEP, int numCaps, MEDIA_CAP *pPreferredMediaList)
{
   Endpoint* pEndpoint = NULL;
   for (unsigned int epNum = startEP; epNum <= endEP; epNum++)
   {
      map<int, Endpoint*>::iterator itr = m_EPContainer.find(epNum);
      if (itr == m_EPContainer.end())
      {
         cout << "ERROR: EP:" << epNum << " unspecified."  << endl;
         return B3G_ERROR;
      }

      pEndpoint = (*itr).second;
      pEndpoint->SetPreferredMediaCaps(numCaps, pPreferredMediaList);
   }
   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: int EndpointMngr::InitH223LCParams(int startEP, int endEP, LOCAL_H223_LC_PARAMS *pLocalH223LCParams)
// Description: Initial;ize H223 logical channel parameters
// Return: int 
// Parameters: int startEP 
//             int endEP 
//             LOCAL_H223_LC_PARAMS *pLocalH223LCParams 
//*****************************************************************************
int EndpointMngr::InitH223LCParams(unsigned int startEP, unsigned int endEP, LOCAL_H223_LC_PARAMS *pLocalH223LCParams)
{
   Endpoint* pEndpoint = NULL;
   for (unsigned int epNum = startEP; epNum <= endEP; epNum++)
   {
      map<int, Endpoint*>::iterator itr = m_EPContainer.find(epNum);
      if (itr == m_EPContainer.end())
      {
         cout << "ERROR: EP:" << epNum << " unspecified."  << endl;
         return B3G_ERROR;
      }
      pEndpoint = (*itr).second;
      pEndpoint->SetH223LCParams(pLocalH223LCParams);
   }
   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: int EndpointMngr::InitH264Params(int startEP, int endEP, LOCAL_H264_PARAMS *pLocalH264Params)
// Description: Initial;ize H223 logical channel parameters
// Return: int 
// Parameters: int startEP 
//             int endEP 
//             LOCAL_H264_PARAMS *pLocalH264Params 
//*****************************************************************************
int EndpointMngr::InitH264Params(unsigned int startEP, unsigned int endEP, LOCAL_H264_PARAMS *pLocalH264Params)
{
   Endpoint* pEndpoint = NULL;
   for (unsigned int epNum = startEP; epNum <= endEP; epNum++)
   {
      map<int, Endpoint*>::iterator itr = m_EPContainer.find(epNum);
      if (itr == m_EPContainer.end())
      {
         cout << "ERROR: EP:" << epNum << " unspecified."  << endl;
         return B3G_ERROR;
      }
      pEndpoint = (*itr).second;
      pEndpoint->SetH264Params(pLocalH264Params);
   }
   return B3G_SUCCESS;
}

int EndpointMngr::InitEPTracing(unsigned int startEP, unsigned int endEP, 
                                unsigned int traceLevel, const char *fileName)
{
   Endpoint* pEndpoint = NULL;
   for (unsigned int epNum = startEP; epNum <= endEP; epNum++)
   {
      map<int, Endpoint*>::iterator itr = m_EPContainer.find(epNum);
      if (itr == m_EPContainer.end())
      {
         cout << "ERROR: EP:" << epNum << " unspecified."  << endl;
         return B3G_ERROR;
      }
      pEndpoint = (*itr).second;
      pEndpoint->SetTracing(traceLevel, fileName);
   }
   return B3G_SUCCESS;
}


int EndpointMngr::InitBoardTracing(unsigned int traceLevel, const char *fileName)
{
    m_BrdTraceLevel = traceLevel;
    strncpy(m_BrdTraceFileName, fileName, MAX_TRACE_FILE_NAME-1);
    return B3G_SUCCESS;
}


int EndpointMngr::InitEPDCI(unsigned int startEP, unsigned int endEP, 
                            unsigned char *pDciOctetStr, unsigned char octetStrSize, 
                            unsigned int dciMode)
{
   Endpoint* pEndpoint = NULL;
   for (unsigned int epNum = startEP; epNum <= endEP; epNum++)
   {
      map<int, Endpoint*>::iterator itr = m_EPContainer.find(epNum);
      if (itr == m_EPContainer.end())
      {
         cout << "ERROR: EP:" << epNum << " unspecified."  << endl;
         return B3G_ERROR;
      }
      pEndpoint = (*itr).second;
      pEndpoint->SetDCI(pDciOctetStr, octetStrSize, dciMode);
   }
   return B3G_SUCCESS;
}


//*****************************************************************************
// Function: void EndpointMngr::InitEPs()
// Description: Initialize M3G and SIP board devices
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::InitEPs()
{
   int nSRLMode = SR_POLLMODE;
   if ( sr_setparm(SRL_DEVICE, SR_MODEID, &nSRLMode) == -1 )
   {
      LOG_ERROR(0,"Error sr_setparm(SRL_DEVICE, SR_MODEID, POLL) failure\n");
      ResetTermExit();
   }

   M3G_START_STRUCT startParams;
   INIT_M3G_START_STRUCT(&startParams);
   // note this assumes config file has no holes in EP numbers, e.g. EP1...EP2...EP4, etc.
   startParams.numEndpoints = m_EPContainer.size();

   if (M3G_ERROR == m3g_Start(&startParams))
   {
      LOG_ERROR(0,"m3g_Start() failure");
      ResetTermExit();
   }
   LOG_ENTRY(0,"Successful m3g_Start completion\n");

   // start and allocate resources for the M3G library session:
   // Initialize the specified board defaults:
   m_BoardHandle = m3g_OpenEx("m3gB1", NULL, this, EV_SYNC);
   if ( 0 > m_BoardHandle )
   {
      LOG_ERROR(0, "m3g_Open() failure\n");
      ResetTermExit();
   }
   LOG_ENTRY(m_BoardHandle, "Successful m3g_OpenEx() completion\n");
   OnBoardOpenComplete();   // calls m3g_Reset() on board handle
   //  JM-trial, skip board reset (ie, OnBoardOpenComplete) if using two instances of 3GDemo
   // replace with OnBoardResetComplete to move along state machine 
   //OnBoardResetComplete();
#ifndef ISDN_CALL_OUT
   if ( gc_OpenEx( &m_GcSipBoardHandle, ":P_IP:N_iptB1", EV_ASYNC,(void *) this) != GC_SUCCESS )
   {
      LOG_ERROR(0, "gc_OpenEx(iptB1) failure\n");
      exit(1);
   }
   LOG_ENTRY(m_GcSipBoardHandle, "Successful gc_OpenEx(iptB1) completion\n");
#endif
}


//*****************************************************************************
// Function: Endpoint* EndpointMngr::GetEPFromIdx(int idx)
// Description: Return a 3G endpoint given it's index
// Return: Endpoint* 
// Parameters: int idx 
//*****************************************************************************
Endpoint* EndpointMngr::GetEPFromIdx(int idx)
{
   map<int, Endpoint*>::iterator itr = m_EPContainer.find(idx);
   return((itr != m_EPContainer.end()) ? (*itr).second : NULL);
}

//*****************************************************************************
// Function: Endpoint* EndpointMngr::GetEPFromM3GHandle(SRL_DEVICE_HANDLE devH)
// Description: Return a 3G endpoint given it's m3g device handle
// Return: Endpoint* 
// Parameters: SRL_DEVICE_HANDLE devH 
//*****************************************************************************
Endpoint* EndpointMngr::GetEPFromM3GHandle(SRL_DEVICE_HANDLE devH)
{
   Endpoint *pEndpoint = NULL; 
   map<int, Endpoint*>::iterator itr;
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      pEndpoint = (*itr).second;
      if ( devH == pEndpoint->GetControlHandle() )
      {
         return pEndpoint; 
      }
      if ( devH == pEndpoint->GetAudioHandle() )
      {
         return pEndpoint; 
      }
      if ( devH == pEndpoint->GetVideoHandle() )
      {
         return pEndpoint; 
      }
   }
   // not found, return NULL:
   return NULL;
}

//*****************************************************************************
// Function: Endpoint* EndpointMngr::GetEPFromNetworkHandle(SRL_DEVICE_HANDLE devH)
// Description: Return a 3G endpoint given it's TDM device handle
// Return: Endpoint* 
// Parameters: SRL_DEVICE_HANDLE devH 
//*****************************************************************************
Endpoint* EndpointMngr::GetEPFromNetworkHandle(SRL_DEVICE_HANDLE devH)
{
   Endpoint *pEndpoint = NULL; 
   map<int, Endpoint*>::iterator itr;
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      pEndpoint = (*itr).second;
      if ( devH == pEndpoint->GetNetworkHandle() )
      {
         return pEndpoint; 
      }
      if ( devH == pEndpoint->GetNetworkDeviceHandle() )
      {
         return pEndpoint; 
      }
   }
   // not found, return NULL:
   return NULL;
}

//*****************************************************************************
// Function: int EndpointMngr::GetEPIndexFromMapID(char *mapID)
// Description: Return a 3G endpoint given it's configuration Map ID
// Return: int 
// Parameters: char *mapID 
//*****************************************************************************
int EndpointMngr::GetEPIndexFromMapID(char *mapID)
{
   LOG_ENTRY(0, "GetEPIndexFromMapID(%s)\n",mapID);
   Endpoint *pEndpoint = NULL; 
   map<int, Endpoint*>::iterator itr;
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      pEndpoint = (*itr).second;
      if ( 0 == strcmp(mapID, pEndpoint->MapID()) )
      {
         return pEndpoint->GetIndex(); 
      }
   }
   // not found, return 0: 
   return 0;
}

//*****************************************************************************
// Function: SIPEndpoint* EndpointMngr::GetSIPEPFromIdx(int idx)
// Description: Return a SIP endpoint given it's index
// Return: SIPEndpoint* 
// Parameters: int idx 
//*****************************************************************************
SIPEndpoint* EndpointMngr::GetSIPEPFromIdx(int idx)
{
   map<int, SIPEndpoint*>::iterator itr = m_SIPEPContainer.find(idx);
   return((itr != m_SIPEPContainer.end()) ? (*itr).second : NULL);
}

//*****************************************************************************
// Function: SIPEndpoint* EndpointMngr::GetMMEPFromIdx(int idx)
// Description: Return an MM endpoint given it's index
// Return: MMEndpoint* 
// Parameters: int idx 
//*****************************************************************************
MMEndpoint* EndpointMngr::GetMMEPFromIdx(int idx)
{
   map<int, MMEndpoint*>::iterator itr = m_MMEPContainer.find(idx);
   return((itr != m_MMEPContainer.end()) ? (*itr).second : NULL);
}

#ifdef USE_RTSP
//*****************************************************************************
// Function: RTSPEndpoint* EndpointMngr::GetRTSPEPFromIdx(int idx)
// Description: Return an RTSP endpoint given it's index
// Return: RTSPEndpoint* 
// Parameters: int idx 
//*****************************************************************************
RTSPEndpoint* EndpointMngr::GetRTSPEPFromIdx(int idx)
{
   map<int, RTSPEndpoint*>::iterator itr = m_RTSPEPContainer.find(idx);
   return((itr != m_RTSPEPContainer.end()) ? (*itr).second : NULL);
}
#endif

//*****************************************************************************
// Function: int EndpointMngr::SetSIPInfo(const char *originIPAddr, const char *destIPAddr, 
//                                         const char *originSIPAddr, const char *destSIPAddr,
//                                         const char *maxSIPChans)
// Description: Save the SIP addresses and max channel setting
// Return: int 
// Parameters: const char *originIPAddr 
//             const char *destIPAddr 
//             const char *originSIPAddr 
//             const char *destSIPAddr 
//             const char *maxSIPChans 
//*****************************************************************************
int EndpointMngr::SetSIPInfo (unsigned short localSIPPort, const char *originSIPAddr, 
                              const char *destSIPAddr)
{
   // Set SIP-related IP addresses parsed from config file
   m_localSIPPort = localSIPPort;
   strcpy(m_originSIPADDR, originSIPAddr);
   strcpy(m_destSIPADDR, destSIPAddr);

   LOG_ENTRY(0, "SIPINFO: Origin addr = '%s', origin port = '%u', dest addr = '%s'\n",
           m_originSIPADDR, m_localSIPPort, m_destSIPADDR);
   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: EndpointMngr::InitSipVideoCoderList(E_SEL_VID_CODER *preferredSipVideoList)
// Description: Save the preferred SIP Video Coder List
// Return: int
// Parameters: E_SEL_VID_CODER *preferredSipVideoList
//*****************************************************************************
int EndpointMngr::InitSipVideoCoderList(E_SEL_VID_CODER *preferredSipVideoList)
{
   memcpy(&m_PreferredSipVideoCoderList, preferredSipVideoList, (VID_CODER_LAST * sizeof(E_SEL_VID_CODER)));
   return B3G_SUCCESS;
}


//*****************************************************************************
// Function: EndpointMngr::InitSipAudioCoderList(E_SEL_AUD_CODER *preferredSipAudioList)
// Description: Save the preferred SIP Audio Coder List
// Return: int
// Parameters: E_SEL_AUD_CODER *preferredSipAudioList
//*****************************************************************************
int EndpointMngr::InitSipAudioCoderList(E_SEL_AUD_CODER *preferredSipAudioList)
{
   memcpy(&m_PreferredSipAudioCoderList, preferredSipAudioList, (AUD_CODER_LAST * sizeof(E_SEL_AUD_CODER)));
   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: int EndpointMngr::SetCmdInfo
// Description: Set UDP socket info for connection to ISDN gateway
// Return: int 
// Parameters: const char *connectCmdtIPAddr 
//             unsigned short connectCmdRcvPort 
//             unsigned short connectCmdSndPort 
//*****************************************************************************
int EndpointMngr::SetCmdInfo (const char *connectCmdIPAddr, unsigned short connectCmdRcvPort, 
				unsigned short connectCmdSndPort)
{
   strcpy(m_connectCmdIPAddr, connectCmdIPAddr);
   m_connectCmdRcvPort = connectCmdRcvPort;
   m_connectCmdSndPort = connectCmdSndPort;

   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: int EndpointMngr::SetMMInfo
// Description: Set MM Device info for connection to MM
// Return: int 
// Parameters:  char dtfm
//              const char *mmOperation
//              E_SEL_MM_FILE_TYPE mmFileType 
//              E_SEL_AUD_CODER mmAudCoder
//              E_SEL_VID_CODER mmVidCoder
//              E_SEL_VID_RES mmVidRes
//              eVIDEO_FRAMESPERSEC mmFramesPerSec
//              eVIDEO_BITRATE mmBitRate
//              const char *mmPlayAudioFileName 
//              const char *mmPlayVideoFileName
//              const char *mmRecordAudioFileName 
//              const char *mmRecordVideoFileName 
//*****************************************************************************
int EndpointMngr::SetMMInfo (char dtmf, const char *mmOperation, E_SEL_MM_FILE_TYPE mmFileType,
                             E_SEL_AUD_CODER mmAudCoder,
                             E_SEL_VID_CODER mmVidCoder,E_SEL_VID_RES mmVidRes, 
                             eVIDEO_FRAMESPERSEC mmFramesPerSec,eVIDEO_BITRATE mmBitRate,
                             const char *mmPlayAudioFileName, const char *mmPlayVideoFileName, 
                             const char *mmRecordAudioFileName, const char *mmRecordVideoFileName)
{
   CMMConfig *pNewConfig = NULL;
   if ( m_mmOperations.find(dtmf) == m_mmOperations.end()) {
     pNewConfig = new CMMConfig(
         dtmf, mmOperation, mmFileType, mmAudCoder,
	 mmVidCoder,mmVidRes,
	 mmFramesPerSec,mmBitRate,
	 mmPlayAudioFileName, mmPlayVideoFileName,
	 mmRecordAudioFileName, mmRecordVideoFileName);
     m_mmOperations[dtmf] = pNewConfig;

     LOG_ENTRY(0, "SetMMinfo: DTMF='%c', Operation=%s, MMFileType=%d, AudCoder=%d, VidCoder=%d, VidRes=%d, VidFps=%d, VidBitRate=%d\n",
                   dtmf, GetMMOperation(dtmf), GetMMFileType(dtmf), GetMMAudioCoder(dtmf), GetMMVideoCoder(dtmf),
                   GetMMVideoResolution(dtmf), GetMMVideoFramesPerSec(dtmf), GetMMVideoBitRate(dtmf));
     LOG_ENTRY(0, "SetMMinfo: DTMF='%c', PlayAudfile=%s, PlayVidFile=%s, RecAudFile=%s, RecVidFile=%s\n",
                   dtmf, GetMMPlayAudioFileName(dtmf),GetMMPlayVideoFileName(dtmf), 
                   GetMMRecordAudioFileName(dtmf), GetMMRecordVideoFileName(dtmf)); 
   }
   else {
     LOG_ERROR(0, "Duplicate MMINFO for DTMF '%c'\n", dtmf);
   } 
   return B3G_SUCCESS;
}

//*****************************************************************************
// Function: int EndpointMngr::SetNbupInfo
// Description: Set Nbup Info
// Return: int
// Parameters:  unsigned short associated3GEndpoint
//              char nbupType
//              const char *remoteIPAddr
//              unsigned short remoteNbupPort 
//*****************************************************************************
int EndpointMngr::SetNbupInfo (unsigned short associated3GEndpoint, char nbupType, const char *remoteIPAddr,
                               unsigned short remoteNbupPort)
{
   // Use 3G endpoint index to find desired endpoint, and save nbup data in member vars in that endpoint
   Endpoint *pEndpoint = GetEPFromIdx(associated3GEndpoint);
   if (pEndpoint)  {
     pEndpoint->SetNbupType(nbupType);
     pEndpoint->SetRemoteNbupIPAddr(remoteIPAddr);
     pEndpoint->SetRemoteNbupPort(remoteNbupPort);
     return B3G_SUCCESS;
   }
   return B3G_ERROR;
}

//*****************************************************************************
// Function: SIPEndpoint* EndpointMngr::GetSIPEPFromIPMHandle(SRL_DEVICE_HANDLE devH)
// Description: Return a SIP endpoint given it's ipm device handle
// Return: SIPEndpoint* 
// Parameters: SRL_DEVICE_HANDLE devH 
//*****************************************************************************
SIPEndpoint* EndpointMngr::GetSIPEPFromIPMHandle(SRL_DEVICE_HANDLE devH)
{
   SIPEndpoint *pSIPEndpoint = NULL; 
   map<int, SIPEndpoint*>::iterator itr;
   for ( itr = m_SIPEPContainer.begin(); itr != m_SIPEPContainer.end(); itr++ )
   {
      pSIPEndpoint = (*itr).second;
      if ( devH == pSIPEndpoint->GetIPMHandle() )
      {
         return pSIPEndpoint; 
      }
   }
   // not found, return NULL:
   return NULL;
}

#ifdef USE_RTSP
//*****************************************************************************
// Function: SIPEndpoint* EndpointMngr::GetRTSPEPFromHandle(SRL_DEVICE_HANDLE devH)
// Description: Return a SIP endpoint given it's ipm device handle
// Return: SIPEndpoint* 
// Parameters: SRL_DEVICE_HANDLE devH 
//*****************************************************************************
RTSPEndpoint* EndpointMngr::GetRTSPEPFromHandle(SRL_DEVICE_HANDLE devH)
{
   RTSPEndpoint *pRTSPEndpoint = NULL; 
   map<int, RTSPEndpoint*>::iterator itr;
   for ( itr = m_RTSPEPContainer.begin(); itr != m_RTSPEPContainer.end(); itr++ )
   {
      pRTSPEndpoint = (*itr).second;
      if ( devH == pRTSPEndpoint->GetIPMHandle() )
      {
         return pRTSPEndpoint; 
      }
   }
   // not found, return NULL:
   return NULL;
}
#endif

//*****************************************************************************
// Function: SIPEndpoint* EndpointMngr::GetSIPEPFromGCHandle(LINEDEV devH)
// Description: Return a SIP endpoint given it's GC device handle
// Return: SIPEndpoint* 
// Parameters: LINEDEV devH 
//*****************************************************************************
SIPEndpoint* EndpointMngr::GetSIPEPFromGCHandle(LINEDEV devH)
{
   SIPEndpoint *pSIPEndpoint = NULL; 
   map<int, SIPEndpoint*>::iterator itr;
   for ( itr = m_SIPEPContainer.begin(); itr != m_SIPEPContainer.end(); itr++ )
   {
      pSIPEndpoint = (*itr).second;
      if ( devH == pSIPEndpoint->GetGCSIPHandle() )
      {
         return pSIPEndpoint; 
      }
   }
   // not found, return NULL:
   return NULL;
}

//*****************************************************************************
// Function: MMEndpoint* EndpointMngr::GetMMEPFromHandle(SRL_DEVICE_HANDLE devH)
// Description: Return a MM endpoint given its device handle
// Return: MMEndpoint* 
// Parameters: SRL_DEVICE_HANDLE devH 
//*****************************************************************************
MMEndpoint* EndpointMngr::GetMMEPFromHandle(SRL_DEVICE_HANDLE devH)
{
   MMEndpoint *pMMEndpoint = NULL; 
   map<int, MMEndpoint*>::iterator itr;
   for ( itr = m_MMEPContainer.begin(); itr != m_MMEPContainer.end(); itr++ )
   {
      pMMEndpoint = (*itr).second;
      if ( devH == pMMEndpoint->GetMMHandle() )
      {
         return pMMEndpoint; 
      }
   }
   // not found, return NULL:
   return NULL;
}

//****************************************************************
// Function: BridgedCall* EndpointMngr::GetCallFromIdx(int idx)
// Description: Return a BridgedCall given it's index
// Return: BridgedCall* 
// Parameters: int idx 
//*****************************************************************************
BridgedCall* EndpointMngr::GetCallFromIdx(int idx)
{
   map<int, BridgedCall*>::iterator itr = m_CallContainer.find(idx);
   return((itr != m_CallContainer.end()) ? (*itr).second : NULL);
}

//****************************************************************
// Function: BridgedMedia* EndpointMngr::GetMediaFromIdx(int idx)
// Description: Return a BridgedMedial given it's index
// Return: BridgedMedia* 
// Parameters: int idx 
//*****************************************************************************
BridgedMedia* EndpointMngr::GetMediaFromIdx(int idx)
{
   map<int, BridgedMedia*>::iterator itr = m_MediaContainer.find(idx);
   return((itr != m_MediaContainer.end()) ? (*itr).second : NULL);
}

//****************************************************************
// Function: BridgedHairp* EndpointMngr::GetHairpFromIdx(int idx)
// Description: Return a BridgedHairp given it's index
// Return: BridgedHairp* 
// Parameters: int idx 
//*****************************************************************************
BridgedHairp* EndpointMngr::GetHairpFromIdx(int idx)
{
   map<int, BridgedHairp*>::iterator itr = m_HairpContainer.find(idx);
   return((itr != m_HairpContainer.end()) ? (*itr).second : NULL);
}

#ifdef USE_RTSP
//****************************************************************
// Function: BridgedRtsp* EndpointMngr::GetBridgedRtspFromIdx(int idx)
// Description: Return a BridgedRtsp given it's index
// Return: BridgedRtsp* 
// Parameters: int idx 
//*****************************************************************************
BridgedRtsp* EndpointMngr::GetBridgedRtspFromIdx(int idx)
{
   map<int, BridgedRtsp*>::iterator itr = m_BridgedRtspContainer.find(idx);
   return((itr != m_BridgedRtspContainer.end()) ? (*itr).second : NULL);
}
#endif

//*****************************************************************************
// Function: void EndpointMngr::ProcessEvent(void *pUsrHdl, long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process an event
// Return: none
// Parameters: void *pUsrHdl 
//             long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void EndpointMngr::ProcessEvent(void *pUsrHdl, long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   METAEVENT metaevent;
   if ( gc_GetMetaEvent(&metaevent) == -1 )
   {
      LOG_ERROR(evtDev, "error from gc_GetMetaEvent\n");
   }

   if ( metaevent.flags == GCME_GC_EVENT )
   {
      LOG_ENTRY(evtDev, "GC usrattr = 0x%x\n", metaevent.usrattr);

      if ( evtDev == m_GcSipBoardHandle )
      {
         switch ( evtType )
         {
            case GCEV_OPENEX:
               OnGcSipBoardOpenComplete();
               break;

            case GCEV_UNBLOCKED:
               LOG_ENTRY(evtDev,"GCEV_UNBLOCKED\n"); 
               break; 

            case GCEV_SETCONFIGDATA:
               LOG_ENTRY(evtDev,"GCEV_SETCONFIGDATA\n");
               break;

            default:
               LOG_ERROR(evtDev,"Unexpected GC event type:%d\n",evtType);
               break;
         }
      }
      else
      {

         BaseEndpoint* pBaseEndpoint = reinterpret_cast<BaseEndpoint*>(metaevent.usrattr);
         if ( pBaseEndpoint )
         {
            if ( pBaseEndpoint->GetEndpointType() == M3G_ENDPOINT )
            {
               Endpoint* pEndpoint = reinterpret_cast<Endpoint*>(metaevent.usrattr);
               if ( pEndpoint )
               {
                  pEndpoint->ProcessEvent(metaevent.evttype,
                                          metaevent.evtdatap, 
                                          metaevent.evtlen, 
                                          metaevent.evtdev);
               }
            }
            else if ( pBaseEndpoint->GetEndpointType() == SIP_ENDPOINT )
            {
               SIPEndpoint *pSIPEndpoint = reinterpret_cast<SIPEndpoint*>(metaevent.usrattr);
               if ( pSIPEndpoint )
               {
                  pSIPEndpoint->ProcessEvent(metaevent);
               }
            }
            else if ( pBaseEndpoint->GetEndpointType() == ISDN_ENDPOINT || 
                      pBaseEndpoint->GetEndpointType() == SS7_ENDPOINT )
            {
               ISDNEndpoint *pISDNEndpoint = reinterpret_cast<ISDNEndpoint*>(metaevent.usrattr);
               if ( pISDNEndpoint )
               {
                  pISDNEndpoint->ProcessEvent(metaevent);
               }
            }
#ifdef USE_RTSP
            else if ( pBaseEndpoint->GetEndpointType() == RTSP_ENDPOINT ) 
            {
               RTSPEndpoint *pRTSPEndpoint = reinterpret_cast<RTSPEndpoint*>(metaevent.usrattr);
               if ( pRTSPEndpoint )
               {
                  pRTSPEndpoint->ProcessEvent(metaevent);
               }
            }
#endif
            else

            {
               LOG_ERROR(evtDev, "error finding Endpoint from gc_GetMetaEvent:%d\n",pBaseEndpoint->GetEndpointType());
            } 
         }
      }
   }
   else
   {
      //
      // non-GC event
      if ( true == IsForBoard(pUsrHdl) )
      {
         // no action taken on board device termination events (other than validation):
         switch ( evtType )
         {
            case M3GEV_ENABLE_EVENTS_CMPLT:  // fall-thru
            case M3GEV_DISABLE_EVENTS_CMPLT: // fall-thru
            case M3GEV_START_TRACE_CMPLT:    // fall-thru
            case M3GEV_STOP_TRACE_CMPLT:     // fall-thru
            case M3GEV_SET_VENDORID_CMPLT:
               break;

            case M3GEV_RESET_CMPLT: 
               OnBoardResetComplete();
               break;

            default:
               LOG_ERROR(m_BoardHandle,"Unexpected event type:%d\n",evtType);
               break;
         }
      }
      // handle DMEV_CONNECT specially as it does not suppport use context handles:
      else if ( (DMEV_CONNECT == evtType) || (DMEV_CONNECT_FAIL == evtType) )
      {
         Endpoint* pEndpoint = NULL; 
         if ( NULL != (pEndpoint = GetEPFromM3GHandle(evtDev)) )
         {
            pEndpoint->ProcessEvent(evtType, pEvtData, evtLen, evtDev);
         }
         else if ( NULL != (pEndpoint = GetEPFromNetworkHandle(evtDev)) )
         {
            pEndpoint->ProcessEvent(evtType, pEvtData, evtLen, evtDev);
         }
         else
         {
            LOG_ERROR(m_BoardHandle,"Unexpected event type:%04X on device: %d\n",evtType, evtDev);
         }
      }
      else
      {
         BaseEndpoint* pBaseEndpoint = reinterpret_cast<BaseEndpoint*>(pUsrHdl);
         if ( pBaseEndpoint )
         {
            if ( pBaseEndpoint->GetEndpointType() == M3G_ENDPOINT )
            {
               Endpoint* pEndpoint = reinterpret_cast<Endpoint*>(pUsrHdl);
               if ( pEndpoint )
               {
                  pEndpoint->ProcessEvent(evtType, pEvtData, evtLen, evtDev);
               }
            }
            else if ( pBaseEndpoint->GetEndpointType() == SIP_ENDPOINT )
            {
               // IPML event, identified by its user attribute (dev API calls)
               SIPEndpoint *pSIPEndpoint = GetSIPEPFromIPMHandle(metaevent.evtdev);
               if ( pSIPEndpoint )
               {
                  pSIPEndpoint->ProcessEvent(metaevent);
               }
            }
            else if ( pBaseEndpoint->GetEndpointType() == MM_ENDPOINT )
            {
               // MM event, identified by its user attribute (dev API calls)
               MMEndpoint *pMMEndpoint = GetMMEPFromHandle(metaevent.evtdev);
               if ( pMMEndpoint )
               {
                  pMMEndpoint->ProcessEvent(metaevent);
               }
            }
#ifdef USE_RTSP
            else if ( pBaseEndpoint->GetEndpointType() == RTSP_ENDPOINT )
            {
               RTSPEndpoint *pRTSPEndpoint = GetRTSPEPFromHandle(metaevent.evtdev);
               if ( pRTSPEndpoint )
               {
                  pRTSPEndpoint->ProcessEvent(metaevent);
               }
            }
            else
            {
               LOG_ERROR(evtDev, "error finding Endpoint from gc_GetMetaEvent:%d\n",pBaseEndpoint->GetEndpointType());
            }  
#endif
         }
         else
         {
            // IPML event, identified only by its device handle (IPM API calls)
            // Will be a SIP endpoint.  
            SIPEndpoint *pSIPEndpoint = GetSIPEPFromIPMHandle(metaevent.evtdev);
#ifdef USE_RTSP
	    RTSPEndpoint *pRTSPEndpoint = GetRTSPEPFromHandle(metaevent.evtdev);
#endif
            if ( pSIPEndpoint )
            {
               pSIPEndpoint->ProcessEvent(metaevent);
            }
#ifdef USE_RTSP
            else if ( pRTSPEndpoint )
            {
               pRTSPEndpoint->ProcessEvent(metaevent);
            }
#endif
            else  // might be a dti as user_attr isn't working
            {
               Endpoint *pEndpoint = GetEPFromNetworkHandle(metaevent.evtdev);
               if ( pEndpoint )
               {
                  pEndpoint->ProcessEvent(evtType, pEvtData, evtLen, evtDev); 
               }
            }
         }
      }  // end if non-GC event
   }
}

//*****************************************************************************
// Function: void EndpointMngr::ProcessUserConnectRequest(int endpointIdx, const char *callInfo)
// Description: Process a connect request from the user interface
// Return: void 
// Parameters: int endpointIdx 
//             const char *callInfo 
//*****************************************************************************
void EndpointMngr::ProcessUserConnectRequest(int endpointIdx, const char *callInfo)
{
   Endpoint* pEndpoint = GetEPFromIdx(endpointIdx);
   if ( NULL != pEndpoint )
   {
      pEndpoint->SetCallInfo(callInfo);
      LOG_ENTRY(pEndpoint->GetControlHandle(),"ProcessUserConnectRequest(%s)\n", callInfo);
      pEndpoint->ProcessUserPrompt(EPState::USER_CONNECT_PROMPT);
   }
}

//*****************************************************************************
// Function: void EndpointMngr::ProcessUserDisconnectRequest(int endpointIdx)
// Description: Process a disconnect request from the user interface
// Return: void 
// Parameters: int endpointIdx 
//*****************************************************************************
void EndpointMngr::ProcessUserDisconnectRequest(int endpointIdx)
{
   Endpoint* pEndpoint = GetEPFromIdx(endpointIdx);
   if ( NULL != pEndpoint )
   {
      LOG_ENTRY(pEndpoint->GetControlHandle(),"ProcessUserDisconnectRequest()\n");
      pEndpoint->ProcessUserPrompt(EPState::USER_DISCONNECT_PROMPT);
   }
}

//*****************************************************************************
// Function: void EndpointMngr::CloseM3GBoard
// Description: Close the M3G board-level device
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::CloseM3GBoard()
{
   // stop board level tracing if active
   StopBoardTracing();

  if (M3G_ERROR == m3g_Close(m_BoardHandle))
    {
      LOG_ERROR(m_BoardHandle, "Error m3g_Close() on board\n");
      return;
    }
  LOG_ENTRY(m_BoardHandle, "m3g_Close() on board success\n");
  m_BoardHandle = 0;
}

//*****************************************************************************
// Function: bool EndpointMngr::AllEndpointsAreIdle
// Description: See if all 3G and SIP endpoints are in an idle state
// Return: bool (T/F)
// Parameters: none 
//*****************************************************************************
bool EndpointMngr::AllEndpointsAreIdle()
{
   // Check 3G first
   map<int, Endpoint*>::iterator itr;
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      Endpoint *pEndpoint = (*itr).second;
      if (pEndpoint->GetState() != EPState::H245_INACTIVE_STATE)
	return false;
   }

   // Have all SIP endpoints gracefully shutdown, or at least are they idle?
   map<int, SIPEndpoint*>::iterator sipitr;
   for ( sipitr = m_SIPEPContainer.begin(); sipitr != m_SIPEPContainer.end(); sipitr++ )
   {
      SIPEndpoint *pSIPEndpoint = (*sipitr).second;
      if (!(pSIPEndpoint->GetState() == SIPEPState::IDLE_STATE ||  pSIPEndpoint->GetState() == SIPEPState::CLOSED_STATE))
	return false;
   }
   LOG_ENTRY(0, "Endpointmgr reports all 3G and SIP endpoints are idle\n");
   return true;
}

//*****************************************************************************
// Function: void EndpointMngr::Shutdown()
// Description: Process a shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::Shutdown()
{
   // have all 3G endpoints gracefully shutdown
   map<int, Endpoint*>::iterator itr;
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      Endpoint *pEndpoint = (*itr).second;
      pEndpoint->Shutdown();
   }

   // have all SIP endpoints gracefully shutdown
   map<int, SIPEndpoint*>::iterator sipitr;
   for ( sipitr = m_SIPEPContainer.begin(); sipitr != m_SIPEPContainer.end(); sipitr++ )
   {
      SIPEndpoint *pSIPEndpoint = (*sipitr).second;
      pSIPEndpoint->Shutdown();
   }

   // have all MM endpoints gracefully shutdown
   map<int, MMEndpoint*>::iterator mmitr;
   for ( mmitr = m_MMEPContainer.begin(); mmitr != m_MMEPContainer.end(); mmitr++ )
   {
      MMEndpoint *pMMEndpoint = (*mmitr).second;
      pMMEndpoint->Shutdown();
   }
  // have all ISDN endpoints gracefully shutdown
   map<int, ISDNEndpoint*>::iterator isdnitr;
   for ( isdnitr = m_ISDNEPContainer.begin(); isdnitr != m_ISDNEPContainer.end(); isdnitr++ )
   {
      ISDNEndpoint *pISDNEndpoint = (*isdnitr).second;
      pISDNEndpoint->Shutdown();
   }
#ifdef USE_RTSP
  // have all RTSP endpoints gracefully shutdown
   map<int, RTSPEndpoint*>::iterator rtspitr;
   for ( rtspitr = m_RTSPEPContainer.begin(); rtspitr != m_RTSPEPContainer.end(); rtspitr++ )
   {
      RTSPEndpoint *pRTSPEndpoint = (*rtspitr).second;
      pRTSPEndpoint->Shutdown();
   }
#endif
}

//*****************************************************************************
// Function: void EndpointMngr::Abort()
// Description: Process a m3g_Reset triggered shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::Abort()
{
   // have all 3G endpoints do a reset
   map<int, Endpoint*>::iterator itr;
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      Endpoint *pEndpoint = (*itr).second;
      pEndpoint->Abort();
   }
}

//*****************************************************************************
// Function: void EndpointMngr::ResetTermExit()
// Description: Restore ncurses terminal changes and exit.  Intended to be used
//              only around device open time.
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::ResetTermExit()
{
  // Reset curses terminal and do a hasty exit
  LOG_ENTRY(0, "Endpointmngr - Restoring terminal to original (non-curses) state\n");
  delete g_pTerminal;
  exit(1);
}


//*****************************************************************************
// Function: int EndpointMngr::InitGlobalCall()
// Description: Start GlobaloCall
// Return: int 
// Parameters: none 
//*****************************************************************************
int EndpointMngr::InitGlobalCall ()
{
   int numSIPEPs;

   INIT_IP_VIRTBOARD(&m_virtBoard);

   // Enable TCP (UDP is the default)
   //m_virtBoard.E_SIP_tcpenabled = ENUM_Enabled;
   //m_virtBoard.E_SIP_DefaultTransport = ENUM_TCP;
   // TLS supports a max of 125 calls.  Don't exceed this
   // numSIPEPs = m_SIPEPContainer.size();
   numSIPEPs=m_maxSipTimeSlot; // for random or sequential device naming and in any order 
   if (!numSIPEPs)
       numSIPEPs++;
   m_virtBoard.sip_max_calls = numSIPEPs;
   m_virtBoard.total_max_calls = numSIPEPs;
   m_libData.version          = 0x100;
   m_libData.num_boards       = 1;
   m_virtBoard.version        = IP_VIRTBOARD_VERSION;
   m_virtBoard.h323_max_calls    = IP_CFG_NO_CALLS;
   m_virtBoard.sip_msginfo_mask  = IP_SIP_MSGINFO_ENABLE | IP_SIP_MIME_ENABLE;

   m_virtBoard.sip_signaling_port   = m_localSIPPort;

   m_libData.board_list = &m_virtBoard;
   INIT_IPCCLIB_START_DATA(&m_libData, 1,&m_virtBoard);

   // Enable 3PCC mode
   m_libData.max_parm_data_size     = IP_CFG_PARM_DATA_MAXLEN;
   m_libData.media_operational_mode = MEDIA_OPERATIONAL_MODE_3PCC;

   memset(&m_gcLibStart, 0, sizeof(GC_START_STRUCT));

#ifdef HMP
   CCLIB_START_STRUCT ccstart[] = 
   {
      {"GC_IPM_LIB",NULL},
      {"GC_DM3CC_LIB",NULL},
      {"GC_SS7_LIB",NULL},
      {"GC_H3R_LIB",&m_libData}
   };
   m_gcLibStart.cclib_list = ccstart;
   m_gcLibStart.num_cclibs = 4;

#else // MMP, where currently no GC SS7
  CCLIB_START_STRUCT ccstart[] =
   {
      {"GC_IPM_LIB",NULL},
      {"GC_DM3CC_LIB",NULL},
      {"GC_H3R_LIB",&m_libData}
   };
   m_gcLibStart.cclib_list = ccstart;
   m_gcLibStart.num_cclibs = 3;
#endif

   int rc = gc_Start(&m_gcLibStart);
   if ( rc != GC_SUCCESS )
   {
      LOG_ERROR(0, "gc_Start() Failed");
      return GC_ERROR;
   }
   else
   {
      LOG_ENTRY(0, "GlobalCall (%d SIP Channels) Started Successfully\n", numSIPEPs);
      return GC_SUCCESS;
   }
}

//*****************************************************************************
// Function: void EndpointMngr::StopGlobalCall()
// Description: Stop GlobalCall
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::StopGlobalCall ()
{
   LOG_ENTRY(0, "Global Call Stop called\n");
   if ( gc_Stop() != GC_SUCCESS )
   {
      LOG_ERROR(0, "gc_Stop failed\n");
   }
   else
   {
      LOG_ENTRY(0, "Global Call Stopped Successfully\n");
   }
}

//*****************************************************************************
// Function: void EndpointMngr::StopM3G()
// Description: Stop GlobalCall
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::StopM3G ()
{
   if ( m3g_Stop() == M3G_ERROR )
   {
      LOG_ERROR(0, "m3g_Stop failed\n");
   }
   else
   {
      LOG_ENTRY(0, "M3G Stopped Successfully\n");
   }
}

//*****************************************************************************
// Function: void EndpointMngr::OnBoardOpenComplete()
// Description: Handle On M3G board open completion event
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::OnBoardOpenComplete()
{
   if ( M3G_ERROR == m3g_Reset(m_BoardHandle) )
   {
      LOG_ERROR(0,"m3g_Reset() failure\n");
   }
   else
   {
      LOG_ENTRY(0,"Successful m3g_Reset completion\n");
   }
}

//*****************************************************************************
// Function: void EndpointMngr::OnBoardResetComplete()
// Description: Handle M3G board reset conmpletion event
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::OnBoardResetComplete()
{
   // enable board level tracing if specified
   StartBoardTracing();

   // configure the vendor identification elements if specified
   SetVendorId();

   if (0 != m_EnableEventMask)
   {
      if (M3G_ERROR == m3g_EnableEvents(m_BoardHandle, m_EnableEventMask))
      {
         LOG_ERROR(m_BoardHandle, "m3g_EnableEvents() failure\n");
      }
      else
      {
         LOG_ENTRY(m_BoardHandle, "Successful m3g_EnableEvent(0x%x) completion\n",m_EnableEventMask);
      } 
   }

   if (0 != m_DisableEventMask)
   {
      if (M3G_ERROR == m3g_DisableEvents(m_BoardHandle, m_DisableEventMask))
      {
         LOG_ERROR(m_BoardHandle, "m3g_DisableEvents() failure\n"); 
      }
      else
      {
         LOG_ENTRY(m_BoardHandle, "Successful m3g_DisableEvent(0x%x) completion\n",m_DisableEventMask); 
      } 
   }

   map<int, Endpoint*>::iterator itr; 
   map<int, Endpoint*>::iterator m3gitr; 
   map<int, SIPEndpoint*>::iterator sipitr;
   map<int, MMEndpoint*>::iterator mmitr;
   map<int, ISDNEndpoint*>::iterator isdnitr;
   map<int, BridgedHairp*>::iterator hpitr;
#ifdef USE_RTSP
   map<int, RTSPEndpoint*>::iterator rtspitr;
#endif

   m_MediaIndexTbl = new int [m_MMEPContainer.size()];
   m_CallIndexTbl = new int [m_CallContainer.size()];
   m_HairpIndexTbl = new int [m_HairpContainer.size()];
   m_BridgedRtspIndexTbl = new int [31];
   // Loop over 3G endpoints and set up bridged call/media/hairpin objects
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      Endpoint* pEndpoint = (*itr).second; 
      int peerIndex = pEndpoint->GetPeerIndex();
      int index = pEndpoint->GetIndex();
      
      if ( peerIndex )
      {
         // Peers are either SIP, MM or 3G.  Each gets a different "Bridged" object -
         // BridgedCall for 3G-SIP connection
         // BridgedMedia for 3G-MM
         // BridgedHairp for 3G-3G
         if (GetSIPEPFromIdx(peerIndex) != NULL)
         {
            LOG_ENTRY(pEndpoint->GetControlHandle(), "SIP peer found.  Creating bridged call object\n");
            BridgedCall *pBridgedCall = new BridgedCall( ++m_NumCalls, index, peerIndex);
            if (pBridgedCall)
            {
               // add to index table
               m_CallIndexTbl[m_NumCalls-1] = m_NumCalls;
               // add to container
               if ( !m_CallContainer.insert(make_pair(m_NumCalls, pBridgedCall)).second )
               {
                  LOG_ERROR(m_BoardHandle, "Could not insert BridgedCall object into CallContainer\n");
               }

               // set the pointer for callbacks
               pEndpoint->SetBridgedCall( pBridgedCall); 

               // find the SIP endpopint
               sipitr = m_SIPEPContainer.find(peerIndex);
               SIPEndpoint *pSIPEndpoint = ((sipitr != m_SIPEPContainer.end()) ? (*sipitr).second : NULL);
               if ( (pSIPEndpoint) )
               {
                  // set the pointer for callbacks
                  pSIPEndpoint->SetBridgedCall(pBridgedCall);
                  pBridgedCall->Open();
               }
            }
            else
            {
               LOG_ERROR(m_BoardHandle, "Could not create BridgedCall object\n"); 
            } 
         }
#ifdef USE_RTSP
        else if (GetRTSPEPFromIdx(peerIndex) != NULL) {
	     // Have an RTSP peer, so want a BridgedMedia object
	     LOG_ENTRY(pEndpoint->GetControlHandle(), "RTSP peer found.  Creating bridged RTSP object\n");
	     BridgedRtsp *pBridgedRtsp = new BridgedRtsp( ++m_NumBridgedRtsp, index, peerIndex);
	     if ( pBridgedRtsp )
	       {
		 // add to index table used for displaying calls on terminal
		 m_BridgedRtspIndexTbl[m_NumBridgedRtsp-1] = m_NumBridgedRtsp;
		 // add to container
		 if ( !m_BridgedRtspContainer.insert(make_pair(m_NumBridgedRtsp, pBridgedRtsp)).second )
		  {
		     LOG_ERROR(m_BoardHandle, "Could not insert BridgedRtsp object into m_BridgedRtspContainer\n");
		  }
		 
		 // set the pointer for callbacks
		 pEndpoint->SetBridgedRtsp( pBridgedRtsp); 
		 
		 // find the RTSP endpoint
		 rtspitr = m_RTSPEPContainer.find(peerIndex);
		 RTSPEndpoint *pRTSPEndpoint = ((rtspitr != m_RTSPEPContainer.end()) ? (*rtspitr).second : NULL);
		 if ( (pRTSPEndpoint) )
		   {
		     // set the pointer for callbacks
		     pRTSPEndpoint->SetBridgedRtsp(pBridgedRtsp);
		     pBridgedRtsp->Open();
		   }
	       }
	     else
	       {
		 LOG_ERROR(m_BoardHandle, "Could not create BridgedRtsp object\n"); 
	       } 
        }
#endif
	 else if (GetMMEPFromIdx(peerIndex) != NULL)  {
            // Have an MM peer, so want a BridgedMedia object
            LOG_ENTRY(pEndpoint->GetControlHandle(), "MM peer found.  Creating bridged media object\n");
            BridgedMedia *pBridgedMedia = new BridgedMedia( ++m_NumMedias, index, peerIndex);
            if (pBridgedMedia)
            {
               // add to index table used for displaying calls on terminal
               m_MediaIndexTbl[m_NumMedias-1] = m_NumMedias;
               // add to container
               if ( !m_MediaContainer.insert(make_pair(m_NumMedias, pBridgedMedia)).second )
               {
                  LOG_ERROR(m_BoardHandle, "Could not insert BridgedMedia object into MediaContainer\n");
               }

               // set the pointer for callbacks
               pEndpoint->SetBridgedMedia( pBridgedMedia); 

               // find the MM endpoint
               mmitr = m_MMEPContainer.find(peerIndex);
               MMEndpoint *pMMEndpoint = ((mmitr != m_MMEPContainer.end()) ? (*mmitr).second : NULL);
               if (pMMEndpoint)
               {
                  // set the pointer for callbacks
                  pMMEndpoint->SetBridgedMedia(pBridgedMedia);
                  pBridgedMedia->Open();
               }
            }
            else
            {
               LOG_ERROR(m_BoardHandle, "Could not create BridgedMedia object\n"); 
            }
         }
         else if (GetEPFromIdx(peerIndex) != NULL)  
         {
            // Have a 3G peer, so want a BridgedHairpin object
            // Need to check that we haven't already created a BridgedHairpin object using this 3G
            // endpoint as we are in the middle of a larger loop over all 3G endpoints
            bool gotItAlready = 0;
            for (hpitr = m_HairpContainer.begin(); hpitr != m_HairpContainer.end(); hpitr++ )
            {
               if (((*hpitr).second)->M3gIndex1() == index || ((*hpitr).second)-> M3gIndex2() == index )
                  gotItAlready = 1;
            }

            if (!gotItAlready)  
            {
               LOG_ENTRY(pEndpoint->GetControlHandle(), "3G peer found.  Creating bridged hairpin object\n");
               BridgedHairp *pBridgedHairp = new BridgedHairp( ++m_NumHairps, index, peerIndex);
               if (pBridgedHairp)
               {
                  // add to index table used for displaying calls on terminal
                  m_HairpIndexTbl[m_NumHairps-1] = m_NumHairps;
                  // add to container
                  if (!m_HairpContainer.insert(make_pair(m_NumHairps, pBridgedHairp)).second )
                  {
                     LOG_ERROR(m_BoardHandle, "Could not insert BridgedHairpin object into HairpContainer\n");
                  }

                  // set the pointer for callbacks
                  pEndpoint->SetBridgedHairp( pBridgedHairp); 

                  // find the 3G endpoint
                  m3gitr = m_EPContainer.find(peerIndex);
                  Endpoint *pEndpoint = ((m3gitr != m_EPContainer.end()) ? (*m3gitr).second : NULL);
                  if (pEndpoint)
                  {
                     // set the pointer for callbacks
                     pEndpoint->SetBridgedHairp(pBridgedHairp);
                     pBridgedHairp->Open();
                  }
               }
               else
               {
                  LOG_ERROR(m_BoardHandle, "Could not create BridgedHairp object\n"); 
               }
            }
         }
         else 
         {
            LOG_ERROR(m_BoardHandle, "Neither SIP, MM or 3G peer found\n");
            return;
         }
      }
   }
   
   m_EPIndexTbl = new int [m_EPContainer.size()];
   // Open all types of devices
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      m_EPIndexTbl[m_NumEPs++] = (*itr).first;
      Endpoint* pEndpoint = (*itr).second; 
      pEndpoint->OpenSubDevs();
   } 

   m_SIPEPIndexTbl = new int [m_SIPEPContainer.size()];
   for ( sipitr = m_SIPEPContainer.begin(); sipitr != m_SIPEPContainer.end(); sipitr++ )
   {
      m_SIPEPIndexTbl[m_NumSIPEPs++] = (*sipitr).first;
      SIPEndpoint *pSIPEndpoint = (*sipitr).second;
      pSIPEndpoint->OpenSubDevs();
   }


   m_MMEPIndexTbl = new int [m_MMEPContainer.size()];
   for ( mmitr = m_MMEPContainer.begin(); mmitr != m_MMEPContainer.end(); mmitr++ )
   {
      m_MMEPIndexTbl[m_NumMMEPs++] = (*mmitr).first;
      MMEndpoint *pMMEndpoint = (*mmitr).second;
      pMMEndpoint->OpenSubDevs();
   }


   m_ISDNEPIndexTbl = new int [m_ISDNEPContainer.size()];
   for ( isdnitr = m_ISDNEPContainer.begin(); isdnitr != m_ISDNEPContainer.end(); isdnitr++ )
   {
      m_ISDNEPIndexTbl[m_NumISDNEPs++] = (*isdnitr).first;
      ISDNEndpoint *pISDNEndpoint = (*isdnitr).second;
      pISDNEndpoint->OpenSubDevs();
   }
#ifdef USE_RTSP
   m_RTSPEPIndexTbl = new int [m_RTSPEPContainer.size()];
   for ( rtspitr = m_RTSPEPContainer.begin(); rtspitr != m_RTSPEPContainer.end(); rtspitr++ )
   {
      m_RTSPEPIndexTbl[m_NumRTSPEPs++] = (*rtspitr).first;
      RTSPEndpoint *pRTSPEndpoint = (*rtspitr).second;
      pRTSPEndpoint->OpenSubDevs();
   }
#endif
}

//*****************************************************************************
// Function: void EndpointMngr::InitPortConnectInfoList(DM_PORT_CONNECT_INFO_LIST *pciList)
// Description: Initialize a port info list
// Return: void 
// Parameters: DM_PORT_CONNECT_INFO_LIST *pciList 
//*****************************************************************************
void EndpointMngr::InitPortConnectInfoList(DM_PORT_CONNECT_INFO_LIST *pciList, unsigned int unPortType, bool TranscodeFlag)
{ 
   const char* portType ; 
   LOG_ENTRY(0,">>InitPortConnectInfoList\n");

   // Prep a port connect info list for a dev_PortConnect between IPM and 3G
   memset (pciList, 0, sizeof(DM_PORT_CONNECT_INFO_LIST));
   INIT_DM_PORT_CONNECT_INFO_LIST(pciList); 
   pciList->unCount = 1;
   INIT_DM_PORT_CONNECT_INFO(&pciList->port_connect_info[0]);
   switch (unPortType) {
       case VIDEO_PORT:
         pciList->port_connect_info[0].unFlags = TranscodeFlag?DMFL_TRANSCODE_ON:DMFL_TRANSCODE_NATIVE;
         portType = "Video";
       break;
       case AUDIO_PORT:
         pciList->port_connect_info[0].unFlags = TranscodeFlag?DMFL_TRANSCODE_ON:DMFL_TRANSCODE_NATIVE;
         portType = "Audio";
       break;
       case NBUP_PORT:
         pciList->port_connect_info[0].unFlags = DMFL_TRANSCODE_NATIVE;
         portType = "NbUP";
       break;
   }

   if(DMFL_TRANSCODE_ON == pciList->port_connect_info[0].unFlags)
   {
      LOG_ENTRY(0,  "InitPortConnectInfoList:DMFL_TRANSCODE_ON, port type = %s\n", portType);
   } 
   else
   {   
      LOG_ENTRY(0,  "InitPortConnectInfoList:DMFL_TRANSCODE_NATIVE, port type = %s\n", portType);
   }
}

//*****************************************************************************
// Function: void EndpointMngr::OnGcSipBoardOpenComplete()
// Description: Handle ipt board open comnpletion event
// Return: void 
// Parameters: none 
//*****************************************************************************
void EndpointMngr::OnGcSipBoardOpenComplete()
{
   GC_PARM_BLKP parmblkp = NULL;
   //GC_PARM_BLKP retblkp = NULL;
   gc_util_insert_parm_ref(   &parmblkp,
                              IPSET_CONFIG,
                              IPPARM_REGISTER_SIP_HEADER,
                              (unsigned char)strlen("TextLine1") + 1,
                              (void*)"CallInfo");

   long request_id = 0;
   if ( gc_SetConfigData(   GCTGT_CCLIB_NETIF, 
                            m_GcSipBoardHandle,
                            parmblkp,
                            0,
                            GCUPDATE_IMMEDIATE,
                            &request_id,
                            EV_ASYNC) != GC_SUCCESS )
   {
      LOG_ERROR(m_GcSipBoardHandle, "gc_SetConfigData() failed\n"); 
   }
   else
   {
      LOG_ENTRY(m_GcSipBoardHandle, "gc_SetConfigData() succeded\n"); 
   }

   gc_util_delete_parm_blk( parmblkp);
}


void EndpointMngr::StartBoardTracing()
{
   if (0 < m_BrdTraceLevel)
   {
      M3G_TRACE_INFO traceInfo;
      INIT_M3G_TRACE_INFO(&traceInfo);
      traceInfo.logfile = m_BrdTraceFileName;
      traceInfo.bitmask = m_BrdTraceLevel;

      if (M3G_ERROR == m3g_StartTrace(m_BoardHandle, &traceInfo))
      { 
         LOG_ERROR(m_BoardHandle, "m3g_StartTrace() failure\n");
      }
      else
      {
         LOG_ENTRY(m_BoardHandle, "Successful m3g_StartTrace() completion\n");
      }
   }
}


void EndpointMngr::StopBoardTracing()
{
   if (0 < m_BrdTraceLevel)
   {
      if (M3G_ERROR == m3g_StopTrace(m_BoardHandle))
      { 
         LOG_ERROR(m_BoardHandle, "m3g_StopTrace() failure\n");
      }
      else
      {
         LOG_ENTRY(m_BoardHandle, "Successful m3g_StopTrace() completion\n");
      }
   }
}


void EndpointMngr::SetVendorId()
{
   if (0 < m_VendorId.productNumber.length)
   {
      if (M3G_ERROR == m3g_SetVendorId(m_BoardHandle, &m_VendorId))
      { 
         LOG_ERROR(m_BoardHandle, "SetVendorId() failure\n");
      }
      else
      {
         LOG_ENTRY(m_BoardHandle, "Successful SetVendorId() completion\n");
      }
   } else {

      //adding this will disable sending vendor identification message to remote EP
      M3G_PARM_INFO parmInfo;
      INIT_M3G_PARM_INFO(&parmInfo);

      parmInfo.parameterType = M3G_E_PRM_TX_VENDORID;
      parmInfo.parmValue.boolParam = M3G_FALSE;

      if (M3G_ERROR == m3g_SetParm(m_BoardHandle, &parmInfo))      {
         LOG_ERROR(m_BoardHandle, "m3g_SetParm() failure\n");
      }  else  {
         LOG_ENTRY(m_BoardHandle, "Disable sending vendorID Successful m3g_SetParm(%d) completion\n", parmInfo.parameterType);
      }
   } // if vendor ID is not present into config file, disable sending to 3G EP

}


#ifdef ISDN_CALL_OUT
void EndpointMngr::InitiateISDNCall(unsigned int epIndex)
{
   Endpoint* pEndpoint = GetEPFromIdx(epIndex);
   if (pEndpoint)
   {
      pEndpoint->InitiateISDNCall();
   }
   else
   { 
      LOG_ERROR(0, "Invalid endpoint index [%d] to intiate call\n", epIndex);
   }

}
#endif

//*****************************************************************************
// Function: int EndpointMngr::ValidateConfiguration()
// Description: Validate Configuration is proper
// Return: int
// Parameters: none
//*****************************************************************************
int EndpointMngr::ValidateConfiguration()
{
   int  retVal = B3G_SUCCESS;


   // XCoder Validation
   //LOG_ENTRY(0,"XCoder Configuration:");
   // 1. Check that Coders match for Native peers.

   map<int, Endpoint*>::iterator itr;
   map<int, Endpoint*>::iterator m3gitr;
   map<int, SIPEndpoint*>::iterator sipitr;
   map<int, MMEndpoint*>::iterator mmitr;
   map<int, ISDNEndpoint*>::iterator isdnitr; 
   // Loop over 3G endpoints and validate
   for ( itr = m_EPContainer.begin(); itr != m_EPContainer.end(); itr++ )
   {
      Endpoint* pEndpoint = (*itr).second;
      int peerIndex = pEndpoint->GetPeerIndex();
      int index = pEndpoint->GetIndex();
      if ( peerIndex )
      {
         // Peers are either SIP, MM or 3G.
         //Peer is SIP endpoint
         if (GetSIPEPFromIdx(peerIndex) != NULL)
         {
            LOG_ENTRY(0, "m3g EP:%d <--> SIP Peer index=%d\n", index, peerIndex);
            // find the SIP endpoint
            sipitr = m_SIPEPContainer.find(peerIndex);
            SIPEndpoint *pSIPEndpoint = ((sipitr != m_SIPEPContainer.end()) ? (*sipitr).second : NULL);

            //M3G-SIP Native video
            if(!pEndpoint->GetVidTranscodeEnabled() || !pSIPEndpoint->GetVidTranscodeEnabled())
            {
               //Check that preferred SIP video coder is in preferred MCAP list
               if(EndpointMngr::Instance()->GetPreferredSipVideoCoder() == VID_CODER_NONE)
               {
                  LOG_ENTRY(0,"Warning: SIP Native Video Coder not set, using m3g video coder\n");
               }
               else if(!pEndpoint->IsVidCoderPreferred(EndpointMngr::Instance()->GetPreferredSipVideoCoder()))
               {
                  LOG_ENTRY(0,"Warning:EP:%d  m3g Native Video Coder != SIP Native Video Coder(%d)\n", 
                            index, EndpointMngr::Instance()->GetPreferredSipVideoCoder());
                  //return B3G_ERROR;
               }
            }
            //M3G-SIP Native audio
            if(!pEndpoint->GetAudTranscodeEnabled() || !pSIPEndpoint->GetAudTranscodeEnabled())
            {
               //Check that preferred SIP audio coder is in preferred MCAP list
               if(EndpointMngr::Instance()->GetPreferredSipAudioCoder() == AUD_CODER_NONE)
               {
                  LOG_ENTRY(0,"Warning: SIP Native Audio Coder not set, using m3g video coder\n");
               }
               else if(!pEndpoint->IsAudCoderPreferred(EndpointMngr::Instance()->GetPreferredSipAudioCoder()))
               {
                  LOG_ENTRY(0,"Warning:EP:%d m3g Native Audio Coder != SIP Native Audio Coder(%d)\n",
                          index, EndpointMngr::Instance()->GetPreferredSipAudioCoder());
                  //return B3G_ERROR;
               }
            }

         }
         //Peer is MM endpoint
         else if (GetMMEPFromIdx(peerIndex) != NULL)
         {
            LOG_ENTRY(0, "m3g EP:%d <--> MM Peer index=%d\n", index, peerIndex);
            // find the MM endpoint
            mmitr = m_MMEPContainer.find(peerIndex);
            MMEndpoint *pMMEndpoint = ((mmitr != m_MMEPContainer.end()) ? (*mmitr).second : NULL);            
            
            // Verify MM device has a digit '0' entry
            if (!strcmp(EndpointMngr::Instance()->GetMMOperation('0'), "NO OP"))  {
               printf("EP:%d No MM Operation for digit '0'. Digit '0' operation  must exist!\n", index);
               LOG_ERROR(0,"EP:%d No MM Operation for digit '0'. Digit '0' operation  must exist!\n",index);
            }

            //Check for 3GP Support
            for(int dtmf='0'; dtmf<('0'+10); dtmf++){
              //printf("dtmf=%c, FT=%d, Vid=%d, Aud=%d\n", dtmf,
              //        EndpointMngr::Instance()->GetMMFileType(dtmf),
              //        EndpointMngr::Instance()->GetMMVideoCoder(dtmf),
              //        EndpointMngr::Instance()->GetMMAudioCoder(dtmf));           
              if(EndpointMngr::Instance()->GetMMFileType(dtmf)==MM_FILE_TYPE_3GP)
              {
                if(EndpointMngr::Instance()->GetMMVideoCoder(dtmf)!=VID_CODER_MPEG4)
                {
                  printf("Only MPEG4 Video content currently supported in .3gp file.\n");
                  LOG_ERROR(0,"Only MPEG4 Video content currently supported in .3gp file.\n");
                  return B3G_ERROR;
                }
                if(EndpointMngr::Instance()->GetMMAudioCoder(dtmf)!=AUD_CODER_AMR)
                {
                  printf("Only AMR-NB audio content currently supported in .3gp file.\n");
                  LOG_ERROR(0,"Only AMR-NB audio content currently supported in .3gp file.\n");
                  return B3G_ERROR;
                }
              }
            }

            //M3G-MM Native video
            if(!pEndpoint->GetVidTranscodeEnabled() || !pMMEndpoint->GetVidTranscodeEnabled())
            {
               //Check that the MM video coder is in preferred MCAP
               if(!pEndpoint->IsVidCoderPreferred(EndpointMngr::Instance()->GetMMVideoCoder('0')))
               {
                printf("EP:%d m3g Native Video Coder != MM Native Video Coder\n", index);
                LOG_ERROR(0,"EP:%d m3g Native Video Coder != MM Native Video Coder(%d)\n",
                          index, EndpointMngr::Instance()->GetMMVideoCoder('0'));
                return B3G_ERROR;
               }
               //Check that the resolution is QCIF
               if(EndpointMngr::Instance()->GetMMVideoResolution('0') == VID_RES_CIF)
               {
                 printf("MM Native Resolution cannot be CIF\n"); 
                 LOG_ERROR(0,"MM Native resolution cannot be CIF \n",
                          EndpointMngr::Instance()->GetMMVideoCoder('0'));
                return B3G_ERROR;
               }
            }
            //MM-M3G Native audio
            if(!pEndpoint->GetAudTranscodeEnabled() || !pMMEndpoint->GetAudTranscodeEnabled())
            {
               //Check that the MM audio coder is in preferred MCAP list
               if(!pEndpoint->IsAudCoderPreferred(EndpointMngr::Instance()->GetMMAudioCoder('0')))
               {
                printf("EP:%d m3g Native Audio Coder != MM Native Audio Coder\n", index);
                LOG_ERROR(0,"EP %d m3g Native Audio Coder != MM Native Audio Coder(%d)\n",
                          index, EndpointMngr::Instance()->GetMMAudioCoder('0'));
                return B3G_ERROR;
               }
            }
         } 
         else if (GetEPFromIdx(peerIndex) != NULL)
         {
            // Have a 3G peer, so want a BridgedHairpin object
            LOG_ENTRY(0, "m3g EP:%d <--> 3G Peer index=%d\n", index, peerIndex);
            m3gitr = m_EPContainer.find(peerIndex);
            //Endpoint *pEndpointPeer = ((m3gitr != m_EPContainer.end()) ? (*m3gitr).second : NULL);
         }
      }
        
    }

   return retVal;
}

#define LOOKUPCONFIG(X) \
{ \
 if (m_mmOperations.find(dtmf) != m_mmOperations.end()) \
       pConfig = m_mmOperations[dtmf]; \
    else \
       pConfig = NULL; \
}

const char * EndpointMngr::GetMMPlayAudioFileName(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
     return pConfig->GetMMPlayAudioFileName();
  }
  return NULL;
}
   
const char * EndpointMngr::GetMMPlayVideoFileName(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
     return pConfig->GetMMPlayVideoFileName();
  }
  return NULL;
}

const char * EndpointMngr::GetMMRecordAudioFileName(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
     return pConfig->GetMMRecordAudioFileName();
  }
  return NULL;
}

const char * EndpointMngr::GetMMRecordVideoFileName(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
     return pConfig->GetMMRecordVideoFileName();
  }
  return NULL;
}

E_SEL_AUD_CODER EndpointMngr::GetMMAudioCoder(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
     return pConfig->GetMMAudioCoder();
  }
  return AUD_CODER_NONE;
}

E_SEL_MM_FILE_TYPE EndpointMngr::GetMMFileType(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
     return pConfig->GetMMFileType();
  }
  return MM_FILE_TYPE_DMF;
}

E_SEL_VID_CODER EndpointMngr::GetMMVideoCoder(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
     return pConfig->GetMMVideoCoder();
  }
  return VID_CODER_NONE;
}


E_SEL_VID_RES EndpointMngr::GetMMVideoResolution(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
    return pConfig->GetMMVideoResolution();
  }
  return VID_RES_NONE;
}


eVIDEO_FRAMESPERSEC EndpointMngr::GetMMVideoFramesPerSec(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
    return pConfig->GetMMVideoFramesPerSec();
  }
  return VIDEO_FRAMESPERSEC_DEFAULT;
}


eVIDEO_BITRATE EndpointMngr::GetMMVideoBitRate(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
     return pConfig->GetMMVideoBitRate();
  }
  return VIDEO_BITRATE_DEFAULT;
}

const char * EndpointMngr::GetMMOperation(char dtmf)
{
  CMMConfig* pConfig = NULL;
  LOOKUPCONFIG(dtmf);
  if (pConfig) {
     pConfig->DumpConfig();
     return pConfig->GetMMOperation();
  }
  LOG_ENTRY(0,"No MM Operation for dtmf %c\n",dtmf);
  return "NO OP";
}
#undef LOOKUPCONFIG

void EndpointMngr::RequestVFU(unsigned int m3gEPIndex)
{
   Endpoint* pEndpoint = GetEPFromIdx(m3gEPIndex);
   LOG_ENTRY(0,"RequestVFU(%d)\n",m3gEPIndex);
   if (pEndpoint)
   {
      pEndpoint->SendFastVideoUpdate();
   }
   else
   { 
      LOG_ERROR(0, "Invalid endpoint index [%d] to intiate VFU\n", m3gEPIndex);
   }
}
