/**
* @file epstate.cpp
* @brief Definition of EPState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "epstate.h"
#include "endpoint.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"



// static constants used in derived h245Starting and OpeningLCs states
const int EPState::CHANNEL_STATUS      = 0xFFF00000;
const int EPState::AUDIO_TX_LCN_OPEN   = 0x00100000;
const int EPState::AUDIO_RX_LCN_OPEN   = 0x00200000;
const int EPState::VIDEO_TX_LCN_OPEN   = 0x00400000;
const int EPState::VIDEO_RX_LCN_OPEN   = 0x00800000;
const int EPState::AUDIO_TX_STARTING   = 0x01000000;
const int EPState::AUDIO_RX_STARTING   = 0x02000000;
const int EPState::VIDEO_TX_STARTING   = 0x04000000;
const int EPState::VIDEO_RX_STARTING   = 0x08000000;
const int EPState::AUDIO_TX_ACTIVE     = 0x10000000;
const int EPState::AUDIO_RX_ACTIVE     = 0x20000000;
const int EPState::VIDEO_TX_ACTIVE     = 0x40000000;
const int EPState::VIDEO_RX_ACTIVE     = 0x80000000;
const int EPState::AUDIO_CHANS_STARTED = 0x33000000;
const int EPState::VIDEO_CHANS_STARTED = 0xCC000000;
const int EPState::ALL_CHANNELS_ACTIVE = 0xF0000000;



//*****************************************************************************
// Function: EPState::EPState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  EPState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
EPState::EPState(Endpoint* pEndpoint) :
   m_pEndpoint(pEndpoint)
{
}


//*****************************************************************************
// Function: EPState::~EPState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
EPState::~EPState()
{
}


void EPState::ActivateAudioChannel(M3G_E_DIRECTION direction, M3G_LOGICAL_CHANNEL_NUMBER lcn)
{
   // if the audio is starting or active uni-directional in the opposite direction
   if (true == IsAudioStarted())
   {
      // initiate bi-direction
      m_pEndpoint->StartAudio(M3G_E_TXRX);
   }
   else // else start uni-directional only
   {
      m_pEndpoint->StartAudio(direction);
   }

   // cache channel status and lcn
   switch (direction)
   {
      case M3G_E_TX:
         m_pEndpoint->SetAudioTxLCN(lcn);
         SetStatus(AUDIO_TX_LCN_OPEN);
         break;

      case M3G_E_RX:
         m_pEndpoint->SetAudioRxLCN(lcn);
         SetStatus(AUDIO_RX_LCN_OPEN);
         break;

      default:  // should never happen
         break;
   }
}


void EPState::ActivateVideoChannel(M3G_E_DIRECTION direction, M3G_LOGICAL_CHANNEL_NUMBER lcn)
{
   // if the video is starting or active uni-directional in the opposite direction
   if (true == IsVideoStarted())
   {
      // initiate bi-direction
      m_pEndpoint->StartVideo(M3G_E_TXRX);
   }
   else // else start uni-directional only
   {
      m_pEndpoint->StartVideo(direction);
   }

   // cache channel status and lcn 
   switch (direction)
   {
      case M3G_E_TX:
         m_pEndpoint->SetVideoTxLCN(lcn);
         SetStatus(VIDEO_TX_LCN_OPEN);
         break;

      case M3G_E_RX:
         m_pEndpoint->SetVideoRxLCN(lcn);
         SetStatus(VIDEO_RX_LCN_OPEN);
         break;

      default:  // should never happen
         break;
   }
}


void EPState::CheckAllChannelsActive()
{
   LOG_ENTRY(0,"[%d]:CheckAllChannelsActive() m_StatusBitmask:=0x%0x\n",
                m_pEndpoint->GetIndex(), m_StatusBitmask);

   if (IsStatusSet(ALL_CHANNELS_ACTIVE))
   {
       LOG_ENTRY(0,"Endpoint[%d] audio and video active.  Transitioning to MEDIA_ACTIVE state\n",
                 m_pEndpoint->GetIndex());
       m_pEndpoint->ChangeState(MEDIA_ACTIVE_STATE);
       m_pEndpoint->Notify(APP_M3G_ENDPOINT_CONNECTED);
   }
}
