/**
* @file evtdispatcher.cpp
* @brief Definition of event dispatcher functions
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "evtdispatcher.h"
#include "m3glib.h"
#include "devmgmt.h"
#include "srllib.h"
#include "main.h"
#include "endpoint.h"
#include "endpointmgr.h"
#include "logger.h"


// Extern Variables
extern bool g_UserPromptQuit;

int HandleEvent() ;

//*****************************************************************************
// Function: void* ::PollEvts(void *pContext)
// Description: Poll SRL for events and delegate to the Endpoint manager
// Return: void* 
// Parameters: void *pContext 
//*****************************************************************************
void *PollEvts(void *pContext)
{
   bool issuedShutdown = false;

   // Main event loop, exited only at shutdown
   while ( true )
   {
      // If an exit is desired, see if all endpoints are at an idle state and
      // it is safe to leave the event processing loop.
      // Final shutdown does not require the event processor.
      if (g_UserPromptQuit)
	{
          //  LOG_ENTRY(0, "Check for idle system\n");
	  // Check to see if all endpoints are in an idle state 
	  if (EndpointMngr::Instance()->AllEndpointsAreIdle())  {
	    // Break out of event loop and finish shutdown from idle state
            LOG_ENTRY(0, "System is idle - exiting event dispatcher loop\n");
	    break;
	  }
	  else  {
            if (!issuedShutdown)  {
              LOG_ENTRY(0, "Shutdown from non-idle system\n");
              EndpointMngr::Instance()->Shutdown();
	      issuedShutdown = true;
            }
          }
	}
      if ( 0 > sr_waitevt(50) )  // poll Dialogic events every 50ms
      {
         continue;
      }
      HandleEvent();
   }
   if (!issuedShutdown) {
      LOG_ENTRY(0, "Final shutdown from idle system\n");
      EndpointMngr::Instance()->Shutdown();
   }
   int i = 30;
   while(i--) {
     
     if ( 0 > sr_waitevt(50) )  // poll Dialogic events every 50ms
      {
         continue;
      }
      HandleEvent();
   }

   return 0;
}

int HandleEvent() {
      void *pUsrHdl  = sr_getUserContext();
      long  evtType = sr_getevttype();
      void *pEvtData = sr_getevtdatap();
      long  evtLen  = sr_getevtlen();
      long  evtDev  = sr_getevtdev();

      const char *pEvtStr = GetEvtStr(evtType);

      LOG_ENTRY(evtDev,"Evtdispatcher rcvd:%s\n",pEvtStr);
      LOG_ENTRY(evtDev,"\t            Event: 0x%X\n",evtType);
      LOG_ENTRY(evtDev,"\t       Event Data: 0x%X\n",pEvtData);
      LOG_ENTRY(evtDev,"\tEvent Data Length: %d\n", evtLen);
      LOG_ENTRY(evtDev,"\t     Event Device: %d\n", evtDev);
      LOG_ENTRY(evtDev,"\t  Event User Info: 0x%X\n", pUsrHdl);

      if ( evtDev )
      {
	EndpointMngr::Instance()->ProcessEvent(pUsrHdl, evtType, pEvtData, evtLen, evtDev);
      }
   return 0;
}


//*****************************************************************************
// Function: const char* ::GetEvtStr(long evtType)
// Description: Return a string description for an integral event type
// Return: const char* 
// Parameters: long evtType 
//*****************************************************************************
const char* GetEvtStr(long evtType)
{
   static const char *m3gev_str[] = 
   { 
      "M3GEV_OPEN_CMPLT",
      "M3GEV_OPEN_FAIL",
      "M3GEV_ENABLE_EVENTS_CMPLT",
      "M3GEV_ENABLE_EVENTS_FAIL",
      "M3GEV_DISABLE_EVENTS_CMPLT",
      "M3GEV_DISABLE_EVENTS_FAIL",
      "M3GEV_SET_PARM_CMPLT",
      "M3GEV_SET_PARM_FAIL",
      "M3GEV_GET_PARM_CMPLT",
      "M3GEV_GET_PARM_FAIL",
      "M3GEV_START_H245_CMPLT",
      "M3GEV_START_H245_FAIL",
      "M3GEV_FRAMING_ESTABLISHED",
      "M3GEV_FRAMING_LOST",
      "M3GEV_MSD_ESTABLISHED",
      "M3GEV_MSD_FAILED",
      "M3GEV_STOP_H245_CMPLT",
      "M3GEV_STOP_H245_FAIL",
      "M3GEV_ENDSESSION_RCVD",
      "M3GEV_ENDSESSION_SENT",
      "M3GEV_GET_LOCAL_CAPS_CMPLT",
      "M3GEV_GET_LOCAL_CAPS_FAIL",
      "M3GEV_SET_TCS_CMPLT",
      "M3GEV_SET_TCS_FAIL",
      "M3GEV_REMOTE_TCS_RCVD",
      "M3GEV_LOCAL_TCS_ACKD",
      "M3GEV_OPEN_LC_CMPLT",
      "M3GEV_OPEN_LC_FAIL",
      "M3GEV_REMOTE_OLC_RCVD",
      "M3GEV_RESPOND_TO_LC_CMPLT",
      "M3GEV_RESPOND_TO_LC_FAIL",
      "M3GEV_CLOSE_LC_CMPLT",
      "M3GEV_CLOSE_LC_FAIL",
      "M3GEV_REMOTE_CLOSE_LC_RCVD",
      "M3GEV_START_MEDIA_CMPLT",
      "M3GEV_START_MEDIA_FAIL",
      "M3GEV_MODIFY_MEDIA_CMPLT",
      "M3GEV_MODIFY_MEDIA_FAIL",
      "M3GEV_STOP_MEDIA_CMPLT",
      "M3GEV_STOP_MEDIA_FAIL",
      "M3GEV_SEND_H245_UII_CMPLT",
      "M3GEV_SEND_H245_UII_FAIL",
      "M3GEV_H245_UII_RCVD",
      "M3GEV_SEND_H245_MISC_CMD_CMPLT",
      "M3GEV_SEND_H245_MISC_CMD_FAIL",
      "M3GEV_H245_MISC_CMD_RCVD",
      "M3GEV_H245_REMOTE_PAUSED",
      "M3GEV_H245_REMOTE_RESUMED",
      "M3GEV_H245_MSD_EVT",
      "M3GEV_H245_MES_EVT",
      "M3GEV_H223_SKEW_IND_RCVD",
      "M3GEV_H223_SKEW_IND_SENT",
      "M3GEV_RESET_CMPLT",
    "M3GEV_RESET_FAIL",
    "M3GEV_START_TRACE_CMPLT",
    "M3GEV_START_TRACE_FAIL",
    "M3GEV_STOP_TRACE_CMPLT",
    "M3GEV_STOP_TRACE_FAIL",
    "M3GEV_SET_VENDORID_CMPLT",
    "M3GEV_SET_VENDORID_FAIL",
    "M3GEV_REMOTE_VENDORID_RCVD",
    "M3GEV_MONA_PREF_MSG_RCVD",
    "M3GEV_SEND_MONA_PREF_MSG",
    "M3GEV_TX_MPC_ESTABLISHED",
    "M3GEV_RX_MPC_ESTABLISHED",
    "M3GEV_CALL_STATISTICS"
   };

   static const char *dmlev_str[] = 
   {
      "DMEV_CONNECT",
      "DMEV_CONNECT_FAIL",
      "DMEV_DISCONNECT",
      "DMEV_DISCONNECT_FAIL",
      "DMEV_RESERVE_RESOURCE",
      "DMEV_RESERVE_RESOURCE_FAIL",
      "DMEV_RELEASE_RESOURCE",
      "DMEV_RELEASE_RESOURCE_FAIL",
      "DMEV_GET_RESOURCE_RESERVATIONINFO",
      "DMEV_GET_RESOURCE_RESERVATIONINFO_FAIL",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "UNEXPECTED EVENT",
      "DMEV_GET_TX_PORT_INFO",
      "DMEV_GET_TX_PORT_INFO_FAIL",
      "DMEV_GET_RX_PORT_INFO",
      "DMEV_GET_RX_PORT_INFO_FAIL",
      "DMEV_PORT_CONNECT",
      "DMEV_PORT_CONNECT_FAIL",
      "DMEV_PORT_DISCONNECT",
      "DMEV_PORT_DISCONNECT_FAIL"
   };


   static const char *ipmev_str[] = 
   {
      "IPMEV_OPEN",
      "IPMEV_STARTMEDIA",
      "IPMEV_STOPPED",
      "IPMEV_GET_LOCAL_MEDIA_INFO",
      "IPMEV_GET_SESSION_INFO",
      "IPMEV_LISTEN",
      "IPMEV_UNLISTEN",  
      "IPMEV_GET_XMITTS_INFO",
      "IPMEV_SEND_DIGITS",
      "IPMEV_RECEIVE_DIGITS",
      "IPMEV_DIGITS_RECEIVED",
      "IPMEV_EVENT_ENABLED",
      "IPMEV_EVENT_DISABLED",
      "IPMEV_GET_QOS_ALARM_STATUS",
      "IPMEV_RESET_QOS_ALARM_STATUS",
      "IPMEV_SET_QOS_THRESHOLD_INFO",
      "IPMEV_GET_QOS_THRESHOLD_INFO",
      "IPMEV_QOS_ALARM",
      "IPMEV_SET_PARM",
      "IPMEV_GET_PARM",
      "IPMEV_PING",
      "IPMEV_SEND_SIGNAL_TO_IP",
      "IPMEV_TELEPHONY_EVENT",
      "IPMEV_FAXTONE",
      "IPMEV_T38CALLSTATE",
      "IPMEV_T38CAPFRAME_STATUS",
      "IPMEV_T38INFOFRAME_STATUS",
      "IPMEV_T38HDLCFRAME_STATUS",
      "IPMEV_T38POSTPGCODE_STATUS",
      "IPMEV_ERROR",
      "IPMEV_T38POSTPGRESP_STATUS",
      "IPMEV_GET_CTINFO",
      "IPMEV_MODIFYMEDIA",
      "IPMEV_GET_CAPABILITIES",
      "IPMEV_INIT_SEND",
      "IPMEV_INIT_RESPONSE_SEND",
      "IPMEV_INIT_RECEIVED",
      "IPMEV_INIT_COMPLETE",
      "IPMEV_INIT_FAILED",
      "IPMEV_NACK_SENT",
      "IPMEV_SEC_NOTIFY_EXPIRE_KEY_AUDIO",
      "IPMEV_SEC_NOTIFY_EXPIRE_KEY_VIDEO",
      "IPMEV_GENERATEIFRAME"
   };

   static const char *mmev_str[] = 
   {
	"MMEV_OPEN",
	"MMEV_PLAY_ACK",
	"MMEV_RECORD_ACK",
	"MMEV_STOP_ACK",
	"MMEV_RESET_ACK",
	"MMEV_ENABLEEVENTS",
	"MMEV_DISABLEEVENTS",
	"MMEV_PLAY",
	"MMEV_RECORD",
	"MMEV_RESET",
	"MMEV_GET_CHAN_STATE",
	"MMEV_SETPARM",
	"MMEV_GETPARM",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "UNEXPECTED EVENT",
        "MMEV_VIDEO_RECORD_STARTED"
   };

   static char unexpectedEvt[32];
   sprintf(unexpectedEvt, "UNEXPECTED EVENT - 0x%lx", evtType);

   if ( evtType >= 0x1000L && evtType<= 0x1041L )
      return m3gev_str[evtType-0x1000L];
   else if ( evtType >= 0x9E00L && evtType <= 0x9E28L )
      return dmlev_str[evtType-0X9E01L];
   else if ( evtType >= 0x800L && evtType <= 0x8FFL )
      return gcev_msg[evtType-0x800L];
   else if ( evtType >= 0x9000L && evtType <= 0x902BL )
      return ipmev_str[evtType-0x9001L];
   else if ( evtType >= 0xA000L && evtType <= 0xA07F )
      return mmev_str[evtType-0xA001L];
   else
      return unexpectedEvt;
}

