/**
* @file main.cpp
* @brief Definition of the Main program for m3g-sip_gateway demo
*        application.
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "main.h"
#include "m3glib.h"
#include "m3gevts.h"
#include "m3gerrs.h"
#include "devmgmt.h"
#include "port_connect.h"
#include <iostream>
#include <sstream>
#include "config.h"
#include "endpointmgr.h"
#include "evtdispatcher.h"
#include "terminal.h"
#include <ctype.h>
#include <pthread.h>
#include <time.h>
#include <signal.h>
#include <ncurses.h>
#include <unistd.h>
#include <netdb.h>
#include "logger.h"
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>

extern char* M3G_SIP_GATEWAY_BINARY_VERSION_STRING;

using namespace std;

int GetHostInfo(char *a_cIpAddrInDotNotation, in_addr_t *a_IpAddrInByteOrder);

static int _kbhit();
static int _nethit();
static char * getnetbuff();

typedef enum 
{
   EVT_POLL = 0,
   SCRN_RFSH,
   NUMTHREADS
}  B3G_THREAD_NUM;


// static variables:
static pthread_t  g_PollingThreadId;
static  char netCmdBuffer[2048]; // Server to Mux3G Gateway

Terminal  *g_pTerminal = NULL;

// global shutdown flags
bool       g_UserPromptQuit = false;
int 	   ctrlcCnt = 0;

//*****************************************************************************
// Function: int ::main(int argc, char *argv[])
// Description: entry point for m3g-sip_gateway demo:
// Return: int 
// Parameters: int argc 
//             char *argv[] 
// Notes:
//    In addition to the user interface for connecting and disconnecting 3G
//    sessions and the corresponding SIP endpoints, there is also a socket 
//    interface that accepts command messages of the form:
//       C <map_id> <call_info>
//       D <map_id> 
// 
//       The command is applied to 3G endpoint with the map_id established in 
//       the config file.  The call_info string is included as a custom header
//       CallInfo: in the outbound SIP call.
// 
//       The socket is establised to REMOTE_HOST on port APP_CMD_PORT using the
//       #defines above.
//*****************************************************************************
int main(int argc, char *argv[])
{
   signal(SIGQUIT,  intr_hdlr);
   signal(SIGINT,   intr_hdlr);
   signal(SIGTERM,  intr_hdlr);
   signal(SIGWINCH, intr_ResizeHandler);
   
   char        g_cDestIp[20];
   unsigned int g_localIPinByteOrder;


   DemoConfig    *pConfig    = new DemoConfig;
   Logger        *pLogger    = NULL;

   if (B3G_ERROR == ProcessArguments(argc, argv, pConfig))
   {
#if 0 // JS 1/19/09 - done in ProcessArguments
      OutputHelp();
#endif
      delete pConfig;
      return B3G_ERROR;
   }

   const char *logFileName = pConfig->GetLogFile();
   bool isLoggerError = false;

   pLogger = new Logger(logFileName, isLoggerError);
   if ( true == isLoggerError )
   {
      delete pConfig;
      delete pLogger; 
      return B3G_ERROR;
   }
   EndpointMngr::Instance()->SetLogger(pLogger);
   Log3GDemoInfo();  //uses LOG_ENTRY  
 

   // Retrieve IP address of this host server
   if ( GetHostInfo(g_cDestIp, &g_localIPinByteOrder) < 0 )
      return -1;

   LOG_ENTRY(0, "Application will use IP:%s for Call Control.\n", g_cDestIp);

   LOG_ENTRY(0, "Parsing Config file: %s...\n", pConfig->GetConfigFile());   
   if (B3G_ERROR == pConfig->ParseConfigFile()) 
   {
      LOG_ENTRY(0, "Parsing of Config file: %s failed\n", pConfig->GetConfigFile());   
      cout << "Parsing of configuration file: " << pConfig->GetConfigFile() << " failed\n";
      delete pConfig;
      delete pLogger; 
      return B3G_ERROR;
   }
   LOG_ENTRY(0, "Parsing Config file complete\n");
   LOG_ENTRY(0, "===========================================================================\n");

   LOG_ENTRY(0, "Validating Configuration...\n");   
   if (B3G_ERROR == EndpointMngr::Instance()->ValidateConfiguration())
   {
      LOG_ENTRY(0, "Validation of Configuration failed\n");   
      cout << "ERROR: Validation of Configuration  failed" << endl;
      delete pConfig;
      delete pLogger; 
      return B3G_ERROR;
   }
   LOG_ENTRY(0, "Validating Configuration complete\n");

   if (pConfig->GetLoggingEnabled() == false)
      pLogger->SetLoggingEnabled (false);
     
 
   // Can now create a terminal object (which uses Curses) since we are done with config file
   // parsing (uses stdout to report errors) and have a logfile open now.
   g_pTerminal = new Terminal;

   if ( EndpointMngr::Instance()->InitGlobalCall() != 0 )
   {
      // No GC started, forget it...
      printf("Cannot start GlobalCall.  Exiting...\n");
      sleep(1);
      delete pConfig;
      delete pLogger;
      delete g_pTerminal; 
      return B3G_ERROR;
   }

   CreateEvtPollingThread();
   EndpointMngr::Instance()->InitEPs();

   // permit endpoint to get set
   sleep(1);

   // Runtime options.
   char ch;
   bool isHit = false;
   int endpointIdx = 0;
   char callInfo[LINE_BUFFER_SIZE] = {0};

   // main loop
   while ( !g_UserPromptQuit )
   {
      if ( _kbhit() )
      {
         // Allowed options from keyboard pretty minimal - 'Q" for quit
         // convert to upper case to simplify
         ch = toupper(getch());
         strcpy(callInfo, "");
         isHit = true;
         // for testing remote interface
         // sendCmdToHost(ch);
      }
      else if ( _nethit() )
      {
         strcpy(callInfo, getnetbuff());
         char endpointMapID[LINE_BUFFER_SIZE] = {0};

         istringstream cmd(callInfo);

         cmd >> ch;
         cmd >> endpointMapID;
         endpointIdx = EndpointMngr::Instance()->GetEPIndexFromMapID(endpointMapID);
         isHit = true;
      }
      else
      {
         isHit=false;
      }
      if ( isHit )
      {
         switch ( ch )
         {
            case 'C':
               // Connection request from network
               if ( 0 != endpointIdx )
               {
                 EndpointMngr::Instance()->ProcessUserConnectRequest(endpointIdx, callInfo);
                 endpointIdx = 0;
               }
               break;

            case 'D':
               // Disconnection request from network
               if ( 0 != endpointIdx )
               {
                 EndpointMngr::Instance()->ProcessUserDisconnectRequest(endpointIdx);
                 endpointIdx = 0;
               }
               break;

            case 'Q':
               // Only possibility from the keyboard
               g_UserPromptQuit = true;
               g_pTerminal->UpdatePrompt(Terminal::QUITTING_PROMPT);
               break;

            case 'V':
             // Send Video Fast Update
             {
               // get 3G EP index number from keyboard:
               unsigned int m3gEPIndex = 0;
               if (1 == scanf("%d", &m3gEPIndex))
               {
                  EndpointMngr::Instance()->RequestVFU(m3gEPIndex);
                  endpointIdx = 0;
               }

               break;
            }  
               break;
#ifdef ISDN_CALL_OUT
            case 'I':
            {
               // ISDN makecall
               // get ISDN EP index number from keyboard:
               unsigned int isdnEPIndex;
               if (1 == scanf("%d", &isdnEPIndex))
               {
                  EndpointMngr::Instance()->InitiateISDNCall(isdnEPIndex);
                  endpointIdx = 0;
               }

               break;
            }

            case 'H':
            {
               // terminate 3G session
               // get 3G EP index number from keyboard:
               if (1 == scanf("%d", &endpointIdx))
               {
                 EndpointMngr::Instance()->ProcessUserDisconnectRequest(endpointIdx);
                 endpointIdx = 0;
               }
               break;
            }
#endif
         }
      }
      g_pTerminal->RefreshWindow();
      usleep(30000);   // ~30ms
   }

   // Broken out of main loop; must be quitting

   // Wait for termination of the event polling thread.
   // An orderly shutdown is taken care of there.
   pthread_join(g_PollingThreadId,0);

   // Final shutdown of 3G and GC subsystems
   EndpointMngr::Terminate(); 

   delete pLogger;
   delete pConfig;
   delete g_pTerminal;

   return B3G_SUCCESS;
}


//*****************************************************************************
// Function: void ::ResetPrompt()
// Description: reset prompt line to original prompt
// Return: void 
// Parameters: none 
//*****************************************************************************
void ResetPrompt()
{
  g_pTerminal->UpdatePrompt(Terminal::MAIN_MANUAL_PROMPT);
}

#if 0
//*****************************************************************************
// Function: int ::ProcessArguments(int argc, char *argv[], DemoConfig *pConfig)
// Description: Process command line arguments
// Return: int 
// Parameters: int argc 
//             char *argv[] 
//             DemoConfig *pConfig 
//*****************************************************************************
int ProcessArguments(int argc, char *argv[], DemoConfig  *pConfig)
{
   const int MAX_ARGS  = 5;

   if ( MAX_ARGS < argc )
   {
      cout << "ERROR: Too many arguments."  << endl;
      return B3G_ERROR;
   }

   for ( int argNum = 1; argNum<argc; argNum++ )
   {
      if (('-' == argv[argNum][0]) && ('i' == argv[argNum][1]))
      {  
         if ((0 == argv[argNum][2]) && (argNum+1 < argc))
         {
            pConfig->SetConfigFile(&argv[++argNum][0]);
         }
         else if(B3G_ERROR == pConfig->SetConfigFile(&argv[argNum][2]))
         {
            return B3G_ERROR;
         }
      } 
      else if (('-' == argv[argNum][0]) && ('o' == argv[argNum][1]))
      {
         if ((0 == argv[argNum][2]) && (argNum+1 < argc))
         {
           pConfig->SetLogFile(&argv[++argNum][0]);
         }
         else if( B3G_ERROR == pConfig->SetLogFile(&argv[argNum][2]))
         {
            return B3G_ERROR;
         }
      }
      else
      {
         cout << "Unsupported option:" << argv[argNum] << endl << endl;
         return B3G_ERROR;
      }
   }
   return B3G_SUCCESS;
}
#else   // JS 1/19/09 - Replaced arguments processing function with more general one
        //    and added log file rotation capability
//*****************************************************************************
// Function: int ::ProcessArguments(int argc, char *argv[], DemoConfig *pConfig)
// Description: Process command line arguments
// Return: int 
// Parameters: int argc 
//             char *argv[] 
//             DemoConfig *pConfig 
//*****************************************************************************
int ProcessArguments(int argc, char *argv[], DemoConfig  *pConfig)
{
   int opt;
   bool bRotateLogs = false;
   bool bRotateLogsAndExit = false;
   char sysCmd[100];
   while ( (opt = getopt (argc, argv, "?Rri:o:")) != -1 ) {
       switch (opt) {
           case 'i':
               pConfig->SetConfigFile(optarg);
               break;
           case 'o':
               pConfig->SetLogFile(optarg);
               break;
           case 'r':
               bRotateLogs = true;
               sprintf(sysCmd, "logrotate -f ../logrotate.conf");
               break;
           case 'R':
               bRotateLogs = true;
               bRotateLogsAndExit = true;
               sprintf(sysCmd, "logrotate -f ../logrotate.conf");
               break;
           case '?':
               OutputHelp();
               return B3G_ERROR;
               break;
           default:
               printf("Unknown option: '%c'\n. Use '?' to see usage.\n", opt);
               return B3G_ERROR;
               break;
       }
   }
   if (bRotateLogs) {
       printf("Rotating log files, may take some time\n");
       system (sysCmd);
       if (bRotateLogsAndExit)
           return B3G_ERROR;
   }
   return B3G_SUCCESS;
}
#endif
//*****************************************************************************
// Function: void ::OutputHelp()
// Description: Output program usage
// Return: void 
// Parameters: none 
//*****************************************************************************
void OutputHelp()
{
#if 0 
   cout << "Usage:" << endl << "\tm3g-sip_gateway [-iCONFIGFILE] [-oLOGFILE]" << endl
   << "Defaults: -iconfig.txt -ologfile.txt" << endl;
#else  // JS 1/19/09
   cout << "Usage:" << endl << "\tm3g-sip_gateway [-i <config file name>] [-o <log file name>] [-r] [-R]" << endl;
   cout << "Defaults: -i config.txt -o logfile.txt" << endl;
   cout << "-i <config file>  -- See sample_config_files directory for a list of config files" << endl;
   cout << "-o <log file> " << endl;
   cout << "-r no argument -- Rotate log files based on configuation in logrotate.conf." << endl;
   cout << "      Once logs have been rotated, application continues to initialize and run" << endl;
   cout << "-R no argument -- Rotate log files and exit application." << endl;
   cout << "      -R is equivalent to executing 'logrotate -f logrotate.conf' at the shell" << endl;
   cout << " Log file rotation is performed before any other activity, such as device" << endl;
   cout << "      initialization. Large log files may take some time before" << endl;
   cout << "      any activity is seen on the console" << endl;
#endif
}

//*****************************************************************************
// Function: void ::CreateEvtPollingThread()
// Description: Create SRL event polling thread
// Return: void 
// Parameters: none 
//*****************************************************************************
void  CreateEvtPollingThread()
{
   PTHREAD_START pEvtPollingThread = (PTHREAD_START)PollEvts;
   if ( 0 != pthread_create(&g_PollingThreadId, NULL, pEvtPollingThread, NULL) )
   {
      cout << "ERROR: pthread_create failure" << endl;
      exit(1);
   }
}

//*****************************************************************************
// Function: void ::intr_hdlr(int)
// Description: Interrupt handler to force various levels of shutdown - from
//                 graceful to quick and dirty exit
// Return: void 
// Parameters: int 
//*****************************************************************************
void intr_hdlr(int )
{
  // One Ctrl-C same as hitting 'Q' on command line
   LOG_ENTRY(0, "Ctrl-C recieved\n");
   g_pTerminal->UpdatePrompt(Terminal::QUITTING_PROMPT);

   if (ctrlcCnt >= 1)
   {
     // Two plus, and get a little more anxious and try 3G resets on all 3G devices
     LOG_ENTRY(0, "Aborting gateway...\n");
     EndpointMngr::Instance()->Abort();
   }

   // If user is pounding on Ctrl-C...
   if (ctrlcCnt >= 5)  {
      LOG_ERROR(0, "Emergency exit.  Restoring terminal.\n");
      delete g_pTerminal;
      exit(1);
   }

   g_UserPromptQuit = true;
   ctrlcCnt++;
}

//*****************************************************************************
// Function: void ::intr_ResizeHandler(int sig)
// Description: Handle screen resize
// Return: void 
// Parameters: int sig 
//*****************************************************************************
void intr_ResizeHandler(int sig)
{
   g_pTerminal->ResizeWindow();
}

//*****************************************************************************
// Function: int ::_kbhit()
// Description: Detect keyboard hit
// Return: int 
// Parameters: none 
//*****************************************************************************
int _kbhit() 
{
   static const int STDIN = 0;
   static bool initialized = false;

   if ( ! initialized )
   {
      // Use termios to turn off line buffering
      termios term;
      tcgetattr(STDIN, &term);
      term.c_lflag &= ~ICANON;
      tcsetattr(STDIN, TCSANOW, &term);
      setbuf(stdin, NULL);
      initialized = true;
   }

   int bytesWaiting;
   ioctl(STDIN, FIONREAD, &bytesWaiting);
   return bytesWaiting;
}

//*****************************************************************************
// Function: int ::_nethit()
// Description: Detect command socket message
// Return: int 
// Parameters: none 
//*****************************************************************************
int _nethit()
{
   // static timeval maxWaitTime = {0L,10L}; // 10 usec
   static int sockHand;
   static bool initialized = false;
   struct sockaddr_in server;
   struct sockaddr_in client;
   int    bytesWaiting;
   int    length;
   int    flags;

   if ( ! initialized )
   {
      sockHand = socket(AF_INET ,SOCK_DGRAM , IPPROTO_UDP);
      if ( sockHand < 0 )
      {
         LOG_ERROR(0,"server socket() failure");
         return 0;
      }
      if ( (flags = fcntl(sockHand, F_GETFL, 0)) < 0 )
      {
         LOG_ERROR(0,"server fcntl(F_GETFL) failure");
         return 0;
      }
      flags |= O_NONBLOCK;
      if ( fcntl(sockHand, F_SETFL, flags) < 0 )
      {
         LOG_ERROR(0,"server fcntl(F_SETFL) failure");
         return 0;
      }
      server.sin_family = AF_INET;
      server.sin_addr.s_addr = INADDR_ANY;
      server.sin_port = htons( EndpointMngr::Instance()->GetConnectCmdRcvPort());

      if ( bind(sockHand, (struct sockaddr *)&server, sizeof(server)) < 0 )
      {
         LOG_ERROR(0,"server bind() failure", 0);
         return 0;
      }

      initialized = true;
   }

   length = sizeof(client); 
   bytesWaiting = recvfrom(sockHand, 
                           netCmdBuffer, 
                           2048,
                           0,
                           (sockaddr *)&client,
                           (socklen_t *)&length);
   // if there is no data function will return an error
   // with errno set to EWOULDBLOCK
   if ( bytesWaiting < 0 )
   {
      if ( errno != EWOULDBLOCK )
      {
         cout << "server recvfrom() errno= " << errno;
      }
      bytesWaiting = 0;
   }
   else
   {
      Capitalize(netCmdBuffer);
   }
   return bytesWaiting;
}

//*****************************************************************************
// Function: char* ::getnetbuff()
// Description: Return pointer to the command socket message buffer
// Return: char* 
// Parameters: none 
//*****************************************************************************
char * getnetbuff()
{
   return netCmdBuffer;
}

//*****************************************************************************
// Function: void sendCmdToISDNGW(char * cmd) 
// Description: Send a connect/disconnect command to an ISDN Clear Channel Gateway 
// Return:  
// Parameters: none 
//*****************************************************************************
void sendCmdToISDNGW(char* cmd)
{
   static int initialized = 0;
   static int sockHand;
   static struct sockaddr_in server;
   static int    length;
   int    bufflen;
 
   if (! initialized)
   {
      sockHand = socket(AF_INET ,SOCK_DGRAM , IPPROTO_UDP);
      if (sockHand < 0)
      {
         LOG_ERROR(0, "sendCmdToISDNGW socket() failure");
         return;
      }
      server.sin_family = AF_INET;
      server.sin_addr.s_addr = inet_addr( EndpointMngr::Instance()->GetConnectCmdIPAddr());
      server.sin_port = htons( EndpointMngr::Instance()->GetConnectCmdSndPort());
      length = sizeof(server);
      initialized = 1;
   }
 
   bufflen = strlen(cmd) + 1;
   if (sendto(sockHand,
              cmd,
              bufflen,
              0,
              (struct sockaddr *)&server,
              length) != bufflen)
   {
     LOG_ERROR(0, "sendCmdToISDNGW() sendto() failure", 0);
   }
}

//*****************************************************************************
// Function: void Log3GDemoInfo()
// Description: Logs 3GDemo Banner and information
// Return:
// Parameters: none
//*****************************************************************************
void Log3GDemoInfo()
{
   LOG_ENTRY(0, "===========================================================================\n");
   LOG_ENTRY(0, "         Dialogic(R) 3G-324M Multimedia Gateway Demo                       \n");
   LOG_ENTRY(0, "===========================================================================\n");
   LOG_ENTRY(0, "3GDemo Version = %s\n",M3G_SIP_GATEWAY_BINARY_VERSION_STRING);
   LOG_ENTRY(0, "===========================================================================\n");
}

//*****************************************************************************
// 		  NAME : GetHostInfo()
// DESCRIPTION : Retrieve the systems host information - Ip address etc
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : -1 if the app failed or 0 if the app succeeded
// 	  CAUTIONS : None
//*****************************************************************************
#ifdef WIN32
int GetHostInfo(char *a_cIpAddrInDotNotation, unsigned long *a_IpAddrInByteOrder)
#else
int GetHostInfo(char *a_cIpAddrInDotNotation, in_addr_t *a_IpAddrInByteOrder)
#endif
{
   // retrieve the server IP address to use for Call Control
   // we will use the first enabled NIC
#define GCNAME_MAXSIZE 100
   char hostname[GCNAME_MAXSIZE];
   if ( (gethostname(hostname, GCNAME_MAXSIZE)) < 0 )
   {
      printf("ERROR: Could not read host name information. Exiting...%d\n", errno);
      return -1;
   }

   //printf("Host Name = %s, IP Address: ", hostname);
   struct hostent *hp;
   hp = gethostbyname(hostname);
   struct in_addr in;
   char **p;
   for ( p = hp->h_addr_list; *p != 0; p++ )
   {
      (void) memcpy(&in.s_addr, *p, sizeof (in.s_addr));
//           (void) printf("%s\t%s\n", inet_ntoa(in), hp->h_name);
      strcpy(a_cIpAddrInDotNotation, inet_ntoa(in));
      (void) memcpy(a_IpAddrInByteOrder, *p, sizeof (in.s_addr));
      //for (q = hp->h_aliases; *q != 0; q++)
      //    (void) printf(" %s", *q);
      //(void) putchar('\n');
   }
   return 0;
}


