/**
* @file mediaactive_state.cpp
* @brief Definition of MediaActiveState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "mediaactive_state.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"


// static constant members:
const int MediaActiveState::AUDIO_STOPPED  = 0x01;
const int MediaActiveState::VIDEO_STOPPED  = 0x02;
const int MediaActiveState::MEDIA_STOPPED  = 0x03;
const int MediaActiveState::MEDIA_STOPPING = 0x10;

extern int UnsignedCharHexArrayToString(unsigned char* origStr, int strLen, char* destStr);

//*****************************************************************************
// Function: MediaActiveState::MediaActiveState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  MediaActiveState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
MediaActiveState::MediaActiveState(Endpoint* pEndpoint) :
   EPState(pEndpoint)
{
}

//*****************************************************************************
// Function: MediaActiveState::~MediaActiveState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
MediaActiveState::~MediaActiveState()
{
}

//*****************************************************************************
// Function: void MediaActiveState::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process an event
// Return: none
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void MediaActiveState::ProcessEvent(long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   switch ( evtType )
   {
      case M3GEV_IFRAME_RCVD:
         {
            M3G_IFRAME_DATA* pDciInd = reinterpret_cast<M3G_IFRAME_DATA *>(pEvtData);
            char message[128] ={0};

            switch (pDciInd->direction)
            {
               case M3G_E_RX:
               {
                  LOG_ENTRY(evtDev,"\t\tM3GEV_IFRAME_RCVD from 3G : iframeSize: %d dcilength:%d dciChanged:%d\n",
                      pDciInd->iframeSize, pDciInd->decoderConfigLength, pDciInd->dciChanged);
                  sprintf (message, "R-3G[I.%d-D.%d%c]  ", pDciInd->iframeSize, pDciInd->decoderConfigLength, (pDciInd->dciChanged == true) ? '+' : '-' );

                  break;
               }
            
               case M3G_E_TX:
               {
                  LOG_ENTRY(evtDev,"\t\tM3GEV_IFRAME_RCVD from PSTREAM : iframeSize: %d dcilength:%d dciChanged:%d\n",
                      pDciInd->iframeSize, pDciInd->decoderConfigLength, pDciInd->dciChanged);
                  sprintf (message, "R-PS[I.%d-D.%d%c]  ", pDciInd->iframeSize, pDciInd->decoderConfigLength, (pDciInd->dciChanged == true) ? '+' : '-' );
                  

                  break;
               }

               default:
               {
                  LOG_ERROR(evtDev,"MediaActiveState: M3GEV_IFRAME_RCVD direction: 0%d\n", pDciInd->direction);
                  strcpy (message, "ERROR");
                  break;
               }
            }
            m_pEndpoint->SetLastMessageString(message);

            if (pDciInd->decoderConfigLength > 0) {
               char* temp = new char[2*pDciInd->decoderConfigLength+1];
               UnsignedCharHexArrayToString(pDciInd->decoderConfigInfo, pDciInd->decoderConfigLength, temp);
               temp[2*pDciInd->decoderConfigLength] = '\0';
               LOG_ENTRY(evtDev,"\t\t\tdecoderConfigInfo: '%s'\n", temp);
               delete [] temp;
            }

            break;
         }

      case M3GEV_H245_MISC_CMD_RCVD:
         {
            M3G_H245_MISC_CMD* pMiscCmdInd = reinterpret_cast<M3G_H245_MISC_CMD *>(pEvtData);
            LOG_ENTRY(evtDev,"\tversion: %d LCN:%d\n",
                      pMiscCmdInd->version, pMiscCmdInd->logicalChannelNumber);
            switch (pMiscCmdInd->h245MiscCmdType)
            {
               case M3G_E_FAST_UPDATE_PICTURE:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_FAST_UPDATE_PICTURE (no parms)\n\n");
                  m_pEndpoint->SetLastMessageString("R[FVU]");
                  break;

               case M3G_E_FAST_UPDATE_GOB:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_FAST_UPDATE_GOB: 1stGOB: %d num GOBs:%d\n",
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateGOB.numFirstGOB, 
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateGOB.numGOBs);
                  break;

               case M3G_E_FAST_UPDATE_MB:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_FAST_UPDATE_MB: 1stGOB:%d 1stMB:%d numMBs: %d\n",
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateMB.numFirstMB,
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateMB.numFirstGOB,
                            pMiscCmdInd->h245MiscCmdParams.fastUpdateMB.numMBs);
                  break;

               case M3G_E_TEMP_SPAT_TRDFF:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_TEMP_SPAT_TRDFF: tempSpatialTrdff:%d\n",
                            pMiscCmdInd->h245MiscCmdParams.tempSpatialTrdff);
                  break;

               case M3G_E_VIDEO_FREEZE:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_VIDEO_FREEZE\n");
                  break;

               case M3G_E_SYNC_EVERY_GOB:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_SYNC_EVERY_GOB\n");
                  break;

               case M3G_E_NOSYNC_EVERY_GOB:
                  LOG_ENTRY(evtDev,"\t\tM3G_E_NOSYNC_EVERY_GOB\n");
                  break;

               default:
                  LOG_ERROR(evtDev,"Unknown Miscellaneous Command Indication.\n");
            }
            break;
         }

      case M3GEV_H245_UII_RCVD:
         {
            char pLastActionBuffer[MAX_LAST_MSG_STRING] = {'\0'};
            sprintf(pLastActionBuffer,"R[");

            M3G_H245_UII* pH245UII = reinterpret_cast<M3G_H245_UII *>(pEvtData);
            LOG_ENTRY(evtDev,"\tversion: %d  numDigits:%d|digitBuffer=%s : \n",
                      pH245UII->version, pH245UII->numDigits,pH245UII->digitBuffer);

            if (EndpointMngr::Instance()->GetSipInfoDtmfEnabled())
            {
               if ( pH245UII->numDigits >= 1 )
               {
                  SIPEndpoint *pSipEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx(m_pEndpoint->GetPeerIndex());
                  if ( pSipEndpoint )
                  {
                     pSipEndpoint->SendSipInfoDtmf(pH245UII->digitBuffer[0]);
                  }
               }
            }

            // If a Multimedia endpoint is connected to the 3G, notify it of the DTMF/UII received
            MMEndpoint *pMMEndpoint = EndpointMngr::Instance()->GetMMEPFromIdx(m_pEndpoint->GetPeerIndex());
            if ( pMMEndpoint )
            {
              pMMEndpoint->ProcessH245UII(pH245UII->digitBuffer[0]);
            }

#ifdef USE_RTSP
            // If a RTSP endpoint is connected to the 3G, notify it of the DTMF/UII received
            RTSPEndpoint *pRTSPEndpoint = EndpointMngr::Instance()->GetRTSPEPFromIdx(m_pEndpoint->GetPeerIndex());
            if ( pRTSPEndpoint ) 
              pRTSPEndpoint->ProcessH245UII(pH245UII->digitBuffer[0]);
#endif
            for (int i=0;i<pH245UII->numDigits;i++)
            {
               sprintf(pLastActionBuffer+2,"%c",pH245UII->digitBuffer[i]);
            }
            sprintf(pLastActionBuffer + int(pH245UII->numDigits) + 2,"]");
            LOG_ENTRY(evtDev,"pH245UII->digitBuffer:%s\n",pLastActionBuffer);
            m_pEndpoint->SetLastMessageString(pLastActionBuffer);
            break;
         }

      case M3GEV_STOP_MEDIA_CMPLT:
         {
            if (evtDev == m_pEndpoint->GetAudioHandle())
            {
               SetStatus(AUDIO_STOPPED);
            }
            else
            {
               SetStatus(VIDEO_STOPPED);
            }
            // check for multimedia (a/v) both stopped and if so close LCs and transition:
            CheckMediaStopped();
            break;
         }

      case M3GEV_SEND_H245_UII_CMPLT:
         m_pEndpoint->SetLastMessageString("S[UII]");
         break;

      case M3GEV_SEND_H245_MISC_CMD_CMPLT:
         // only support sending fastVideoUpdate in this demo:
         m_pEndpoint->SetLastMessageString("S[FVU]");
         break;

      case M3GEV_REMOTE_CLOSE_LC_RCVD:
         {
            M3G_REMOTE_CLOSED_LC* pCLCInd = reinterpret_cast<M3G_REMOTE_CLOSED_LC*>(pEvtData);
            M3G_LOGICAL_CHANNEL_NUMBER lcn = pCLCInd->logicalChannelNumber;
            LOG_ENTRY(evtDev,"\t  version:%d\n", pCLCInd->version);
            LOG_ENTRY(evtDev,"\t  LCN:%d\n", lcn);
            LOG_ENTRY(evtDev,"\t  reason:%d\n", pCLCInd->reason);
            if ( lcn == m_pEndpoint->GetAudioRxLCN() )
            {
               m_pEndpoint->SetAudioRxLCN(0);
            }
            else if ( lcn == m_pEndpoint->GetAudioTxLCN() )
            {
               m_pEndpoint->SetAudioTxLCN(0);
            }
            else if ( lcn == m_pEndpoint->GetVideoRxLCN() )
            {
               m_pEndpoint->SetVideoRxLCN(0);
            }
            else if ( lcn == m_pEndpoint->GetVideoTxLCN() )
            {
               m_pEndpoint->SetVideoTxLCN(0);
            }
            break;
            // remain in this state assuming remote will soon send H.245 EndSession (handled below)
         }
         break;

      case M3GEV_REMOTE_VENDORID_RCVD:
         {
            M3G_VENDORID_INFO* pRmtVId = reinterpret_cast<M3G_VENDORID_INFO *>(pEvtData);
            LOG_ENTRY(evtDev,"\t  version:0x%x\n", pRmtVId->version);
            if (M3G_E_OID_TYPE == pRmtVId->vendor.oidType)
            {
               LOG_ENTRY(evtDev,"\t  vendor:\n");
               LOG_ENTRY(evtDev,"\t\t oid.length: %d\n", pRmtVId->vendor.oidValue.oid.length);
               LOG_ENTRY(evtDev,"\t\t oid.objectId[]: %x %x %x...\n", 
                         pRmtVId->vendor.oidValue.oid.objectId[0],
                         pRmtVId->vendor.oidValue.oid.objectId[1],
                         pRmtVId->vendor.oidValue.oid.objectId[2]);
            }
            else if (M3G_E_H221_ID_TYPE == pRmtVId->vendor.oidType)
            {
               LOG_ENTRY(evtDev,"\t  vendor:\n");
               LOG_ENTRY(evtDev,"\t\t t35CountryCode: %x\n", 
                         pRmtVId->vendor.oidValue.h221NonStd.t35CountryCode);
               LOG_ENTRY(evtDev,"\t\t t35Extension: %x\n", 
                         pRmtVId->vendor.oidValue.h221NonStd.t35Extension);
               LOG_ENTRY(evtDev,"\t\t manufacturerCode: %x\n", 
                         pRmtVId->vendor.oidValue.h221NonStd.manufacturerCode);
            }
            LOG_ENTRY(evtDev,"\t productNumber[%d]: %x %x %x...\n",
                      pRmtVId->productNumber.length,
                      pRmtVId->productNumber.octet[0],
                      pRmtVId->productNumber.octet[1],
                      pRmtVId->productNumber.octet[2]);
            LOG_ENTRY(evtDev,"\t versionNumber[%d]: %x %x %x...\n",
                      pRmtVId->versionNumber.length,
                      pRmtVId->versionNumber.octet[0],
                      pRmtVId->versionNumber.octet[1],
                      pRmtVId->versionNumber.octet[2]);
            break;
         }

      case M3GEV_FRAMING_LOST:
         LOG_ERROR(evtDev, "H.223 framing failure\n");
         StopMedia();
         break;

      case M3GEV_ENDSESSION_RCVD:
         LOG_ERROR(evtDev, "H.245 EndSession rcvd\n");
         StopMedia();
         break;

      case M3GEV_RESET_CMPLT:
         ProcessResetComplete();
        break;

      // following H.245 transactions can occur after media is established in MONA:
      case M3GEV_MSD_ESTABLISHED:
         {
            const char* msdStr[] =
            {
               "M3G_E_H245_MASTER",
               "M3G_E_H245_SLAVE",
               "M3G_E_H245_IDENTICAL_MSD_NUMBERS"
            };
            M3G_E_H245_MSD_RESULT msdResult = *(reinterpret_cast<M3G_E_H245_MSD_RESULT*>(pEvtData));
            LOG_ENTRY(evtDev,"\t  MSD result:%s\n", msdStr[msdResult]);
            break;
         }

      case M3GEV_REMOTE_TCS_RCVD:
         break;

      case M3GEV_LOCAL_TCS_ACKD:
         break;

      case M3GEV_H245_MES_EVT:
         {
            M3G_MES* pMesInd = reinterpret_cast<M3G_MES *>(pEvtData);
            LOG_ENTRY(evtDev,"\t  MESEvtType: %d\n", pMesInd->MESEvtType);
            LOG_ENTRY(evtDev,"\t  MESRejectCause: %d\n", pMesInd->MESRejectCause);
            break;
         }

      case M3GEV_H245_MSD_EVT:
         {
            M3G_E_MSD_EVT_TYPE* pMsdInd = reinterpret_cast<M3G_E_MSD_EVT_TYPE *>(pEvtData);
            LOG_ENTRY(evtDev,"\t  M3G_E_MSD_EVT_TYPE: %d\n", int(*pMsdInd));
            break;
         }

      case M3GEV_MSD_FAILED:
         {
            LOG_ERROR(evtDev, "MSD failure.\n");
            break;
         }


#ifdef MONA
      case M3GEV_CALL_STATISTICS:
         m_pEndpoint->RecordStatistics(evtDev, reinterpret_cast<M3G_CALL_STATISTICS*>(pEvtData));
         break;

      // may be notified of this multiple times
      case M3GEV_SEND_MONA_PREF_MSG:
         {
            M3G_MONA_TXRX_MPC_SUPPORT* pMPC = reinterpret_cast<M3G_MONA_TXRX_MPC_SUPPORT*>(pEvtData); 
            LOG_ENTRY(evtDev,"\t rxMPC:0x%x txMPC:0x%x\n", pMPC->rxMPCMask, pMPC->txMPCMask);
            break;
         }
#endif

      default:
         LOG_ERROR(evtDev,"MediaActiveState: Unexpected event type: 0x%x\n", evtType);
         break;
   }
}

//*****************************************************************************
// Function: void MediaActiveState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process request from user interface
// Return: void 
// Parameters: EPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  MediaActiveState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   if ( EPState::USER_DISCONNECT_PROMPT == eUserPrompt )
   {
      LOG_ENTRY(0,"Endpoint[%d] user disconnect from MediaActiveState.  Stopping media\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECT);
      LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECT)\n",
                m_pEndpoint->GetIndex());
      StopMedia();
   }
   else
   {
      LOG_ERROR(0,"Unexpected user prompt on endpoint[%d] in MEDIA_ACTIVE state: %d\n",
                m_pEndpoint->GetIndex(), static_cast<int>(eUserPrompt));
   }
}


//*****************************************************************************
// Function: void MediaActiveState::Shutdown()
// Description: Process shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void MediaActiveState::Shutdown()
{
  LOG_ENTRY(0,"Endpoint[%d] Shutdown ordered in %s  Stopping media\n",
	    m_pEndpoint->GetIndex(), GetStateStr());
   // Stopping M3G media triggers the shutdown process from this state
   StopMedia();
}

//*****************************************************************************
// Function: void MediaActiveState::StopMedia()
// Description: Stop media stream
// Return: void 
// Parameters: none 
//*****************************************************************************
void MediaActiveState::StopMedia()
{
   if (false == IsStatusSet(MEDIA_STOPPING))
   {
      SetStatus(MEDIA_STOPPING);
      m_pEndpoint->StopMedia();
   }
}


//*****************************************************************************
// Function: void MediaActiveState::CheckMediaStopped()
// Description: Change state if media stream is stopped
// Return: void 
// Parameters: none 
//*****************************************************************************
void MediaActiveState::CheckMediaStopped()
{
   if (IsStatusSet(MEDIA_STOPPED))
   {
      m_pEndpoint->ChangeState(CLOSING_LCS_STATE);
      // As of MMP build90, No need to call CloseLC when stopping H245
      // m_pEndpoint->CloseLCs(); 
      // Call m3g_StopH245() because no need to call CloseLC since handset 
      // has not send closelc and already disconnected the call.
      m_pEndpoint->Stop_H245();

   }
}

//*****************************************************************************
// Function: void ClosingiLCsState::ProcessResetComplete()
// Description: M3G reset done; set state to inactive to trigger shutdown
// Return: void
// Parameters: none
//*****************************************************************************
void MediaActiveState::ProcessResetComplete()
{
      LOG_ENTRY(0,"Endpoint[%d] M3g reset complete.  Triggering shutdown.\n",
                m_pEndpoint->GetIndex());
      LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECTED)\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECTED);
      m_pEndpoint->ChangeState(H245_INACTIVE_STATE); 
}
