/**
* @file mmendpoint.cpp
* @brief Definition of MMEndpoint class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include <time.h>
#include "logger.h"
#include "mmendpoint.h"
#include "endpointmgr.h"
#include "locker.h"
#include "appevents.h"
#include "endpoint.h"

#define RTC 0

//*****************************************************************************
// Function: MMEndpoint::MMEndpoint(int numEP, const char *MMName, const char *IPMName, const char *mapID)
// Description: Initializing constructor
// Return:  MMEndpoint*
// Parameters: int numEP 
//             const char *MMName 
//             const char *IPMName 
//             const char *mapID 
//*****************************************************************************
MMEndpoint::MMEndpoint(int numEP, const char *MMName, const char *mapID):
   m_Index(numEP),
   m_ConnectionTime(0),
   m_pCurrentState(NULL),
   m_MMHandle(-1),
   m_LastDigitRcvd('0')
{
   SetEndpointType(MM_ENDPOINT);
   strcpy(m_MMName, MMName);
   strcpy(m_mapID, mapID);
   strncpy(m_CurrentStateStr, "OPENING     ", MAX_CURRENT_STATE_STR);
   strcpy(m_LastMessageBuff, "");

   m_pMMOpeningState = new MMOpeningState(this);
   m_pMMPortConnectingState = new MMPortConnectingState(this);
   m_pMMIdleState = new MMIdleState(this);
   m_pMMPlayingState = new MMPlayingState(this);
   m_pMMRecordingState = new MMRecordingState(this);
   m_LastDigitRcvd = '0';
}

//*****************************************************************************
// Function: MMEndpoint::~MMEndpoint()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
MMEndpoint::~MMEndpoint()
{
  delete m_pMMOpeningState;
  delete m_pMMPortConnectingState;
  delete m_pMMIdleState;
  delete m_pMMPlayingState;
  delete m_pMMRecordingState;
}

//*****************************************************************************
// Function: MMEndpoint::PlayVideoFile(char m_mmPlayAudioFileName[FNAME_SIZE], char m_mmPlayVideoFileName[FNAME_SIZE])
// Description: Play video file stopped using Runtime Control
// Return:  none
// Parameters: char m_mmPlayAudioFileName[FNAME_SIZE]
//             char m_mmPlayVideoFileName[FNAME_SIZE] 
//*****************************************************************************
void MMEndpoint::PlayVideoFile(const char m_mmPlayAudioFileName[FNAME_SIZE], const char m_mmPlayVideoFileName[FNAME_SIZE])
{
  MM_MEDIA_ITEM_LIST audioMediaList;
  MM_MEDIA_ITEM_LIST videoMediaList;
  
  MM_PLAY_RECORD_LIST playRecordList[2];
  MM_PLAY_INFO        playInfo;

  MM_AUDIO_CODEC audioCodec;
  MM_VIDEO_CODEC videoCodec;

  //Runtime control 
  MM_RUNTIME_CONTROL rtc;
  INIT_MM_RUNTIME_CONTROL(&rtc);

#if RTC  
  rtc.unVersion              = MM_RUNTIME_CONTROL_VERSION;
  rtc.Reason                 = EMM_TERM_MAXDTMF;
  rtc.unValue                = 2;
  rtc.Action                 = EMM_TA_AUDIO_VIDEO_STOP;
  rtc.next                   = (PMM_RUNTIME_CONTROL)0;
#endif

  // Do not forget these next 2 INITs !  mm_Play will
  // not work without them
  INIT_MM_MEDIA_AUDIO(&audioMediaList.item.audio);
  INIT_MM_MEDIA_VIDEO(&videoMediaList.item.video);

  INIT_MM_MEDIA_ITEM_LIST(&audioMediaList);
  INIT_MM_MEDIA_ITEM_LIST(&videoMediaList);

  INIT_MM_PLAY_RECORD_LIST(playRecordList);
  INIT_MM_PLAY_INFO(&playInfo);
  INIT_MM_AUDIO_CODEC(&audioCodec);
  INIT_MM_VIDEO_CODEC(&videoCodec);

  LOG_ENTRY(m_MMHandle, "Priming MMPlay Video Coder=%s, Audio Coder=%s\n",GetSelVidCoderStr(),GetSelAudCoderStr());
  
  // Set the Audio FileFormat and Audio Codec
  if(GetSelectedMMFileType()==MM_FILE_TYPE_3GP) {
    audioMediaList.item.audio.eFileFormat           = EMM_FILE_FORMAT_3GP;
    audioCodec.unCoding                             = MM_DATA_FORMAT_AMR_NB_12_20K;
    if(GetSelectedAudioCoder()!=AUD_CODER_AMR) {
       LOG_ERROR(m_MMHandle, "MMEndpoint::PlayVideoFile, Audio Coder not correct (%d) Must be AMR-NB in .3gp file format!\n", GetSelectedAudioCoder());
    } 
  } else { //DMF PROPRIETARY format
    // Set the Audio Codec
    if(GetSelectedAudioCoder()==AUD_CODER_AMR) {
      audioCodec.unCoding                             = MM_DATA_FORMAT_AMR_NB_12_20K;
      audioMediaList.item.audio.eFileFormat           = EMM_AUD_FILEFORMAT_PROPRIETARY;
    }
    else if (GetSelectedAudioCoder()==AUD_CODER_G723) {
      audioCodec.unCoding                             = MM_DATA_FORMAT_G723_1_5_30K;
      audioMediaList.item.audio.eFileFormat           = EMM_AUD_FILEFORMAT_PROPRIETARY;
    }
    else if (GetSelectedAudioCoder()==AUD_CODER_G729) {
      audioCodec.unCoding                             = MM_DATA_FORMAT_G729A;
      audioMediaList.item.audio.eFileFormat           = EMM_AUD_FILEFORMAT_PROPRIETARY;
    }
    else if (GetSelectedAudioCoder()==AUD_CODER_PCM) {
      audioCodec.unCoding                             = MM_DATA_FORMAT_PCM;
      audioMediaList.item.audio.eFileFormat           = EMM_AUD_FILEFORMAT_VOX;
    }
    else {
      // log error
      LOG_ERROR(m_MMHandle, "MMEndpoint::PlayVideoFile, Audio Coder is not a supported type\n");
    }
  }
  audioCodec.unBitsPerSample                      = 16;
  audioCodec.unSampleRate                         = MM_DRT_8KHZ; //was  8000;

  audioMediaList.item.audio.unOffset                = 0;
  audioMediaList.ItemChain                          = EMM_ITEM_EOT;
  audioMediaList.item.audio.codec                   = audioCodec; 
  audioMediaList.item.audio.unMode                  = 0; //MM_MODE_FILE_TYPE_VOX;
  //audioMediaList.item.audio.szFileName = EndpointMngr::Instance()->GetMMPlayAudioFileName(); 
  
  //Playback recorded file after record
  audioMediaList.item.audio.szFileName =m_mmPlayAudioFileName;

  // Now video 

  if (GetSelectedVideoCoder()==VID_CODER_MPEG4) {
    videoCodec.Profile                                = VIDEO_PROFILE_LEVEL_SP0_MPEG4;    // QCIF & maxbitrate 64kpbs
    videoCodec.Coding                                 = VIDEO_CODING_MP4V_ES;
  	videoCodec.Level                                    = VIDEO_LEVEL_DEFAULT;
  }
  else if (GetSelectedVideoCoder()==VID_CODER_H263) {
   videoCodec.Coding                                  = VIDEO_CODING_H263;
   videoCodec.Profile                                 = VIDEO_PROFILE_0_H263;
   videoCodec.Level                                 = VIDEO_LEVEL_30_H263;
  }
  else if (GetSelectedVideoCoder()==VID_CODER_H264) {
   videoCodec.Coding                                  = VIDEO_CODING_H264;
   videoCodec.Profile                                 = VIDEO_PROFILE_BASELINE_H264;
   videoCodec.Level                                 = VIDEO_LEVEL_1_H264;
  }
//  videoCodec.Level                                    = VIDEO_LEVEL_DEFAULT;
  videoCodec.FramesPerSec                           = GetSelectedVideoFramesPerSec();
  if (GetSelectedVideoResolution()==VID_RES_SQCIF) {
     videoCodec.ImageHeight                         = VIDEO_IMAGE_HEIGHT_96;
     videoCodec.ImageWidth                          = VIDEO_IMAGE_WIDTH_128;
  }
  else if (GetSelectedVideoResolution()==VID_RES_CIF) {
     videoCodec.ImageHeight                         = VIDEO_IMAGE_HEIGHT_288;
     videoCodec.ImageWidth                          = VIDEO_IMAGE_WIDTH_352;
  }
  else {
     //Default to QCIF
     videoCodec.ImageHeight                         = VIDEO_IMAGE_HEIGHT_144;
     videoCodec.ImageWidth                          = VIDEO_IMAGE_WIDTH_176;
  }
  videoCodec.BitRate                                = GetSelectedVideoBitRate();

  videoMediaList.item.video.unMode                  = 0; //MM_MODE_NOIFRMBEEPINITIATED;
  
  // Set the Video FileFormat
  if(GetSelectedMMFileType()==MM_FILE_TYPE_3GP) {
    videoMediaList.item.video.eFileFormat           = EMM_FILE_FORMAT_3GP;
  } else {
    videoMediaList.item.video.eFileFormat           = EMM_FILE_FORMAT_PROPRIETARY;
  }  
  
  videoMediaList.ItemChain                          = EMM_ITEM_EOT;
  videoMediaList.item.video.codec                   = videoCodec;

  //videoMediaList.item.video.szFileName = EndpointMngr::Instance()->GetMMPlayVideoFileName();
  videoMediaList.item.video.szFileName =m_mmPlayVideoFileName; //Playback recorded file after record

  playRecordList[0].ItemType                        = EMM_MEDIA_TYPE_AUDIO;
  playRecordList[0].list                            = &audioMediaList;
  playRecordList[0].ItemChain                       = EMM_ITEM_CONT;
  
  playRecordList[1].ItemType                        = EMM_MEDIA_TYPE_VIDEO;
  playRecordList[1].list                            = &videoMediaList;
  playRecordList[1].ItemChain                       = EMM_ITEM_EOT;

  playInfo.eFileFormat    = EMM_FILE_FORMAT_UNDEFINED;
  playInfo.list           = playRecordList;
  
  // start play
  LOG_ENTRY(m_MMHandle, "Playing -> audioF = %s, videoF = %s\n",  
	    audioMediaList.item.audio.szFileName, videoMediaList.item.video.szFileName);
  if(m_mmPlayVideoFileName==NULL || m_mmPlayAudioFileName==NULL)
  {
     SetLastMessageString("MM file not specified");
     return;
  }
  else
     SetLastMessageString(m_mmPlayVideoFileName);
#if RTC
  if (mm_Play(m_MMHandle, &playInfo, &rtc, this) < 0) {
#else
  if (mm_Play(m_MMHandle, &playInfo, NULL, this) < 0) {
#endif
    LOG_ERROR(m_MMHandle, "mm_Play() failed\n");
  }
  else {
    LOG_ENTRY(m_MMHandle, "mm_Play() in progress\n");
  }
}


//*****************************************************************************
// Function: MMEndpoint::RecordVideoFile()
// Description: Record video file stopped using Runtime Control
// Return:  none
// Parameters: 
//*****************************************************************************
void MMEndpoint::RecordVideoFile()

{
  MM_MEDIA_ITEM_LIST audioMediaList;
  MM_MEDIA_ITEM_LIST videoMediaList;
  
  MM_PLAY_RECORD_LIST playRecordList[2];
  MM_RECORD_INFO        recordInfo;

  MM_AUDIO_CODEC audioCodec;
  MM_VIDEO_CODEC videoCodec;
  
  char mRecordAudioFileName[FNAME_SIZE]; 
  char mRecordVideoFileName[FNAME_SIZE];
  
  //Runtime control
  MM_RUNTIME_CONTROL rtc;
  INIT_MM_RUNTIME_CONTROL(&rtc);

#if RTC
  rtc.unVersion              = MM_RUNTIME_CONTROL_VERSION;
  rtc.Reason                 = EMM_TERM_MAXDTMF;
  rtc.unValue                = 1;
  rtc.Action                 = EMM_TA_AUDIO_VIDEO_STOP;
  rtc.next                   = (PMM_RUNTIME_CONTROL)0;
#endif
	
  // Do not forget these next 2 INITs !  mm_Record
  // not work without them
  INIT_MM_MEDIA_AUDIO(&audioMediaList.item.audio);
  INIT_MM_MEDIA_VIDEO(&videoMediaList.item.video);

  INIT_MM_MEDIA_ITEM_LIST(&audioMediaList);
  INIT_MM_MEDIA_ITEM_LIST(&videoMediaList);

  INIT_MM_PLAY_RECORD_LIST(playRecordList);
  INIT_MM_PLAY_INFO(&recordInfo);
  INIT_MM_AUDIO_CODEC(&audioCodec);
  INIT_MM_VIDEO_CODEC(&videoCodec);

  //sprintf(mRecordAudioFileName,"%s_%d.aud",EndpointMngr::Instance()->GetMMRecordAudioFileName(m_LastDigitRcvd),GetIndex());
  //sprintf(mRecordVideoFileName,"%s_%d.vid",EndpointMngr::Instance()->GetMMRecordVideoFileName(m_LastDigitRcvd),GetIndex());
  sprintf(mRecordAudioFileName,"%s",EndpointMngr::Instance()->GetMMRecordAudioFileName(m_LastDigitRcvd));
  sprintf(mRecordVideoFileName,"%s",EndpointMngr::Instance()->GetMMRecordVideoFileName(m_LastDigitRcvd));
  
  if(GetSelectedAudioCoder()==AUD_CODER_AMR) {
    audioCodec.unCoding                             = MM_DATA_FORMAT_AMR_NB_12_20K;
    audioMediaList.item.audio.eFileFormat             = EMM_AUD_FILEFORMAT_PROPRIETARY;
  }
  else if (GetSelectedAudioCoder()==AUD_CODER_G723) {
    audioCodec.unCoding                               = MM_DATA_FORMAT_G723_1_5_30K;
    audioMediaList.item.audio.eFileFormat             = EMM_AUD_FILEFORMAT_PROPRIETARY;
  }
  else if (GetSelectedAudioCoder()==AUD_CODER_G729) {
    audioCodec.unCoding                               = MM_DATA_FORMAT_G729A;
    audioMediaList.item.audio.eFileFormat             = EMM_AUD_FILEFORMAT_PROPRIETARY;
  }
  else if (GetSelectedAudioCoder()==AUD_CODER_PCM) {
    audioCodec.unCoding                             = MM_DATA_FORMAT_PCM;
    audioMediaList.item.audio.eFileFormat           = EMM_AUD_FILEFORMAT_VOX;
  }
  else {
      // log error
      LOG_ERROR(m_MMHandle, "MMEndpoint::RecordVideoFile, Audio Coder is not a supported type\n");
  }
  
  audioCodec.unBitsPerSample                        = 16;
  audioCodec.unSampleRate                           = MM_DRT_8KHZ;
  
  audioMediaList.item.audio.unOffset                = 0;
  audioMediaList.ItemChain                          = EMM_ITEM_EOT;
  audioMediaList.item.audio.codec                   = audioCodec; 
  audioMediaList.item.audio.unMode                  = 0; //MM_MODE_FILE_TYPE_VOX; //MM_MODE_BEEPINITIATED;

  //audioMediaList.item.audio.szFileName = EndpointMngr::Instance()->GetMMRecordAudioFileName();
  audioMediaList.item.audio.szFileName = mRecordAudioFileName;

  // Video
  videoCodec.Level                                  = VIDEO_LEVEL_DEFAULT;
  videoCodec.FramesPerSec                           = GetSelectedVideoFramesPerSec();
  videoCodec.BitRate                                = GetSelectedVideoBitRate();
  
  if (GetSelectedVideoCoder()==VID_CODER_MPEG4) {
      videoCodec.Profile                            = VIDEO_PROFILE_LEVEL_SP0_MPEG4;  // QCIF & maxbitrate 64kpbs
      videoCodec.Coding                             = VIDEO_CODING_MP4V_ES;
  }
  else if (GetSelectedVideoCoder()==VID_CODER_H263) {
      videoCodec.Coding                             = VIDEO_CODING_H263;
      videoCodec.Profile                            = VIDEO_PROFILE_0_H263;
      videoCodec.Level                            = VIDEO_LEVEL_10_H263;
  }
  else if (GetSelectedVideoCoder()==VID_CODER_H264) {
      videoCodec.Coding                             = VIDEO_CODING_H264;
      videoCodec.Profile                            = VIDEO_PROFILE_BASELINE_H264;
      videoCodec.Level                            =  VIDEO_LEVEL_1_H264;
  }
  
  if (GetSelectedVideoResolution()==VID_RES_SQCIF) {
     videoCodec.ImageHeight                         = VIDEO_IMAGE_HEIGHT_96;
     videoCodec.ImageWidth                          = VIDEO_IMAGE_WIDTH_128;
  }
  else if (GetSelectedVideoResolution()==VID_RES_CIF) {
     videoCodec.ImageHeight                         = VIDEO_IMAGE_HEIGHT_288;
     videoCodec.ImageWidth                          = VIDEO_IMAGE_WIDTH_352;
  }
  else {
     //Defualt to QCIF
     videoCodec.ImageHeight                         = VIDEO_IMAGE_HEIGHT_144;
     videoCodec.ImageWidth                          = VIDEO_IMAGE_WIDTH_176;
  }

   //JM - DCI
   // For Native Records, App should set DCI for the Recorded file from OLC in case one is not instream
   // For Transcode Records, Len=0 and app settings should not be used anyway.
   if(m_RecDciLen) {
      LOG_ENTRY(m_MMHandle, "App Set Record DCI[%d]=%s\n",m_RecDciLen, m_RecDciStr);
      for (unsigned int i = 0; i <m_RecDciLen; i++) {
            m_RecDciValue[i] = AsciiOctetToHex(&m_RecDciStr[2*i]);
      }
      videoCodec.VisualConfigSize = m_RecDciLen;
      videoCodec.VisualConfiguration = m_RecDciValue;
   }
   else
      videoCodec.VisualConfigSize = 0;

  videoMediaList.item.video.unMode                  = MM_MODE_BEEPINITIATED;
  videoMediaList.item.video.eFileFormat             = EMM_FILE_FORMAT_PROPRIETARY;

  videoMediaList.ItemChain                          = EMM_ITEM_EOT;
  videoMediaList.item.video.codec                   = videoCodec;

  // videoMediaList.item.video.szFileName = EndpointMngr::Instance()->GetMMRecordVideoFileName();
  videoMediaList.item.video.szFileName = mRecordVideoFileName; 

  playRecordList[0].ItemType = EMM_MEDIA_TYPE_AUDIO;
  playRecordList[0].ItemChain = EMM_ITEM_CONT;
  playRecordList[0].list = &audioMediaList;

  playRecordList[1].ItemType = EMM_MEDIA_TYPE_VIDEO;
  playRecordList[1].ItemChain = EMM_ITEM_EOT;
  playRecordList[1].list = &videoMediaList;

  recordInfo.eFileFormat  = EMM_FILE_FORMAT_UNDEFINED;
  recordInfo.list         = playRecordList;

  LOG_ENTRY(m_MMHandle, "Recording -> audioF(%d) = %s, videoF(%d) = %s\n",  
   audioCodec.unCoding ,audioMediaList.item.audio.szFileName,videoCodec.Coding,videoMediaList.item.video.szFileName);
  SetLastMessageString(mRecordVideoFileName);
  
#if RTC
  // Runtime Control implementation
  if ( mm_Record(m_MMHandle, &recordInfo, &rtc, this) < 0 ) {
#else
  if ( mm_Record(m_MMHandle, &recordInfo, NULL, this) < 0 ) {
#endif
    LOG_ERROR(m_MMHandle, "mm_Record() failed\n");
  }
  else {
    LOG_ENTRY(m_MMHandle, "mm_Record() in progress\n");
  }
}

//*****************************************************************************
// Function: MMEndpoint::StopMMPlay()
// Description: Stop Multimedia Play
// Return:  none
// Parameters: 
//*****************************************************************************
void MMEndpoint::StopMMPlay()
{
    LOG_ENTRY(m_MMHandle, "Stopping MultiMedia Play\n");
    MM_STOP mmStopInfo[2];
    MM_STOP_DETAILS mmStopDetails;
   
    memset(&mmStopDetails, 0, sizeof(MM_STOP_DETAILS));

    mmStopInfo[0].unVersion = 0;
    mmStopInfo[0].ItemChain = EMM_ITEM_CONT;
    mmStopInfo[0].ItemType = EMM_STOP_VIDEO_PLAY;
    mmStopInfo[0].details = mmStopDetails;
    mmStopInfo[0].next = &mmStopInfo[1];

    mmStopInfo[1].unVersion = 0;
    mmStopInfo[1].ItemChain = EMM_ITEM_EOT;
    mmStopInfo[1].ItemType = EMM_STOP_AUDIO_PLAY;
    mmStopInfo[1].details = mmStopDetails;
    mmStopInfo[1].next = NULL;

    if (mm_Stop(m_MMHandle, mmStopInfo, this) < 0) {
        LOG_ERROR(m_MMHandle, "mm_Stop() for Play failed\n");
    }
    else  {
      LOG_ENTRY(m_MMHandle, "mm_Stop issued\n");
    }

}

//*****************************************************************************
// Function: MMEndpoint::StopMMRecord()
// Description: Stop Multimedia Record
// Return:  none
// Parameters: 
//*****************************************************************************
void MMEndpoint::StopMMRecord()
{
    LOG_ENTRY(m_MMHandle, "Stopping MultiMedia Record\n");
    MM_STOP mmStopInfo[2];
    MM_STOP_DETAILS mmStopDetails;

    memset(&mmStopDetails, 0, sizeof(MM_STOP_DETAILS));

    mmStopInfo[0].unVersion = 0;
    mmStopInfo[0].ItemChain = EMM_ITEM_CONT;
    mmStopInfo[0].ItemType = EMM_STOP_VIDEO_RECORD;
    mmStopInfo[0].details = mmStopDetails;
    mmStopInfo[0].next = &mmStopInfo[1];

    mmStopInfo[1].unVersion = 0;
    mmStopInfo[1].ItemChain = EMM_ITEM_EOT;
    mmStopInfo[1].ItemType = EMM_STOP_AUDIO_RECORD;
    mmStopInfo[1].details = mmStopDetails;
    mmStopInfo[1].next = NULL;

    if (mm_Stop(m_MMHandle, mmStopInfo, this) < 0) {
        LOG_ERROR(m_MMHandle, "mm_Stop() for Record failed\n");
    }
    else  {
      LOG_ENTRY(m_MMHandle, "mm_Stop issued\n");
    }
}

//*****************************************************************************
// Function: void MMEndpoint::ChangeState(MMEPState::E_MMEPSTATE e_NewState)
// Description: Change the object state
// Return: void 
// Parameters: MMEPState::E_MMEPSTATE e_NewState 
//*****************************************************************************
void MMEndpoint::ChangeState(MMEPState::E_MMEPSTATE e_NewState)
{
   char oldStateStr[MAX_CURRENT_STATE_STR];
   strcpy(oldStateStr, m_pCurrentState->GetStateStr());

   // reset the current state for the next call:
   m_pCurrentState->Reset();

   switch ( e_NewState )
   {
      case MMEPState::OPENING_STATE:
         m_pCurrentState = m_pMMOpeningState;
         break;

      case MMEPState::PORTCONNECTING_STATE:
         m_pCurrentState = m_pMMPortConnectingState;
         break;

      case MMEPState::IDLE_STATE:
         m_pCurrentState = m_pMMIdleState;
         break;

      case MMEPState::PLAYING_STATE:
         m_pCurrentState = m_pMMPlayingState;
         break;

      case MMEPState::RECORDING_STATE:
         m_pCurrentState = m_pMMRecordingState;
         break;

      default:
         LOG_ERROR(0,"Unexpected state: %d\n",static_cast<int>(e_NewState));
         break;
   }

   // cache the state string in the endpoint
   strcpy(m_CurrentStateStr,m_pCurrentState->GetStateStr());
   LOG_ENTRY(m_MMHandle, "MMEndpoint[%d] State transition: %s ---> %s\n", GetIndex(), oldStateStr, m_CurrentStateStr);
}

//*****************************************************************************
// Function: void MMEndpoint::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void MMEndpoint::ProcessEvent(METAEVENT metaevent)
{
   m_StateLock.Lock();
   // dispatch event information to appropriate state:
   m_pCurrentState->ProcessEvent(metaevent);
   m_StateLock.Unlock();
}

//*****************************************************************************
// Function: void MMEndpoint::ProcessH245UII(char UII)
// Description: Process an H245UII  event
// Return: none
// Parameters: char UII
//*****************************************************************************
void MMEndpoint::ProcessH245UII(char UII)
{
  // A single DTMF/UII digit has arrive on the peer 3G endpoint.  Insert it into a fake
  // metaevent and dispatch it.
  METAEVENT uiiMetaEvent;
  static char charBuffer;

  charBuffer = UII;
  memset (&uiiMetaEvent, 0, sizeof(METAEVENT));
  uiiMetaEvent.evtdev = m_MMHandle;
  uiiMetaEvent.evttype = M3GEV_H245_UII_RCVD;
  uiiMetaEvent.evtdatap = &charBuffer;

  ProcessEvent(uiiMetaEvent);
}

//*****************************************************************************
// Function: void MMEndpoint::OpenSubDevs()
// Description: Open all associated devices
// Return: void 
// Parameters: none 
//*****************************************************************************
void MMEndpoint::OpenSubDevs()
{
  // initialize the endpoint state to opening
  m_pCurrentState = m_pMMOpeningState;
  
  // Open the MM device
  // Note that opening is async only - sync not possible.
  LOG_ENTRY(0, "Opening %s\n", m_MMName);
  if ((m_MMHandle = mm_Open(m_MMName, NULL, this)) < 0) {
    LOG_ERROR(0, "mm_Open(devicename=%s) Failed", m_MMName);
  } else {
    LOG_ENTRY(0, "%s successfully opened\n", m_MMName);
  }
}

//*****************************************************************************
// Function: void MMEndpoint::GetMMPortInfo()
// Description: Get MM port information
// Return: void 
// Parameters: none 
//*****************************************************************************
void MMEndpoint::GetMMPortInfo ()
{
   // Info arrives with successful completion event on async call
   if ( dev_GetTransmitPortInfo(m_MMHandle, this) != DEV_SUCCESS )
   {
      LOG_ERROR(m_MMHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_MMHandle,"Successful dev_GetTransmitPortInfo() call\n");
   }

   if ( dev_GetReceivePortInfo(m_MMHandle, this) != DEV_SUCCESS )
   {
      LOG_ERROR(m_MMHandle,"dev_GetTransmitPortInfo() failure\n");
   }
   else
   {
      LOG_ENTRY(m_MMHandle,"Successful dev_GetReceivePortInfo() call\n");
   }
}

//*****************************************************************************
// Function: void MMEndpoint::RecordPortInfoList(METAEVENT metaevent, DM_PORT_INFO_LIST *pPortInfoList)
// Description: Save the MM port information
// Return: void 
// Parameters: METAEVENT metaevent 
//             DM_PORT_INFO_LIST *pPortInfoList 
//*****************************************************************************
void MMEndpoint::RecordPortInfoList(METAEVENT metaevent, DM_PORT_INFO_LIST* pPortInfoList)
{
   LOG_ENTRY(metaevent.evtdev,"DM_PORT_INFO_LIST:\n");
   LOG_ENTRY(metaevent.evtdev,"\tunVersion:%d\n", pPortInfoList->unVersion);
   LOG_ENTRY(metaevent.evtdev,"\tunCount:%d\n", pPortInfoList->unCount);
   for ( unsigned int i=0; i< pPortInfoList->unCount; i++ )
   {
      LOG_ENTRY(metaevent.evtdev,"\tport_info[%d]:\n",i);
      LOG_ENTRY(metaevent.evtdev,"\t\tunVersion: %d\n",pPortInfoList->port_info[i].unVersion);

      char tmpBuff[128] = {0};
      char tokenBuff[10] = {0};
      for ( int j=0; j<16; j++ )
      {
         sprintf(tokenBuff, "0x%x ", int(pPortInfoList->port_info[i].device_ID[j]));
         strcat(tmpBuff, tokenBuff);
      }
      LOG_ENTRY(metaevent.evtdev,"\t\tdevice_ID: %s\n", tmpBuff);
      LOG_ENTRY(metaevent.evtdev,"\t\tport_ID: %s\n",reinterpret_cast<char *>(pPortInfoList->port_info[i].port_ID));
      LOG_ENTRY(metaevent.evtdev,"\t\tport_media_type: %d\n", pPortInfoList->port_info[i].port_media_type);

      switch ( pPortInfoList->port_info[i].port_media_type )
      {
         case DM_PORT_MEDIA_TYPE_AUDIO:
            if ( metaevent.evttype == DMEV_GET_TX_PORT_INFO )
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Tx audio port info\n");
               memcpy(&m_audioPortTxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            else
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Rx audio port info\n");
               memcpy(&m_audioPortRxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            break;
         case DM_PORT_MEDIA_TYPE_VIDEO:
            if ( metaevent.evttype == DMEV_GET_TX_PORT_INFO )
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Tx video port info\n");
               memcpy(&m_videoPortTxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            else
            {
               LOG_ENTRY(metaevent.evtdev,"\t\tSaving Rx video port info\n");
               memcpy(&m_videoPortRxInfo, &pPortInfoList->port_info[i], sizeof(DM_PORT_INFO));
            }
            break;
         case DM_PORT_MEDIA_TYPE_AUDIO_LINEAR:
            LOG_ENTRY(metaevent.evtdev,"\t\tDetected (but not saving) audio linear port info\n");
            break;
         case DM_PORT_MEDIA_TYPE_NBUP:
            LOG_ENTRY(metaevent.evtdev,"\t\tDetected (but not saving) H223(NBUP) port info\n");
            break;
         case DM_PORT_MEDIA_TYPE_NONE:
            LOG_ENTRY(metaevent.evtdev,"\t\tDetected (but not saving) MEDIA_TYPE_NONE port info\n");
            break;
         default:
            LOG_ENTRY(metaevent.evtdev,"\t\tUnknown DM_PORT_MEDIA_TYPE port info\n");
            break;
      }
   }
}

//*****************************************************************************
// Function: void MMEndpoint::ConnectToPeer(DM_PORT_INFO &a_audioPortRxInfo, DM_PORT_INFO &a_videoPortRxInfo)
// Description: Connect media streams to peer endpoint
// Return: void 
// Parameters: DM_PORT_INFO &a_audioPortRxInfo 
//             DM_PORT_INFO &a_videoPortRxInfo 
//*****************************************************************************
void MMEndpoint::ConnectToPeer(DM_PORT_INFO& a_audioPortRxInfo, DM_PORT_INFO& a_videoPortRxInfo)
{
   // Connect MM ports to 3G device's corresponding streams

   /************************************************************************

    Looking for this:

                    IPM                 3G

    Audio
          Tx --------------->  >----------------- Rx
          Rx ---------------<  <----------------- Tx

    Video
          Tx --------------->  >----------------- Rx
          Rx ---------------<  <----------------- Tx

   ********************************************************************/        
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

   // Have TX and RX port info lists (one each) from IPM
   // Each list has an audio, video and linear audio node
   // 
   // Have TX and RX port info lists from 3G.  Audio device and video device both have
   // a TX and RX list.  Should only be a video PORT_INFO in the video device's, and
   // an audio PORT_INFO in the audio device's.
   // 

   // Important info here!!!!!
   ///////////////////////////
   // "Tx" device is the device whose handle is used in the dev_PortConnect call.
   // port_info_tx gets the transmit port of the device whose handle is used in the dev_PortConnect call.
   // port_info_rx gets the receive port of the other device.
   // BUT - port_info_rx must contain the device_ID of the device whose handle is used in the dev_PortConnect
   // call, or a "multiple device" error will be found in the RTF log

   // This call connects the Tx of MM device to Rx of 3G device (Audio)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, 
                                                     EndpointMngr::AUDIO_PORT,
                                                     GetAudTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx),
          &m_audioPortTxInfo,
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx),
          &a_audioPortRxInfo,
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortConnect(m_MMHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(m_MMHandle,
                "dev_PortConnect() failure in MM endpoint - %s\n",
                ATDV_ERRMSGP(m_MMHandle));
   }
   else
   {
      LOG_ENTRY(m_MMHandle,"Successful dev_PortConnect() call in MM endpoint\n");
   }

   // This call connects the Tx of MMdevice to Rx of 3G device (Video)
   EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList, EndpointMngr::VIDEO_PORT, GetVidTranscodeEnabled());
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
          &m_videoPortTxInfo,
          sizeof(DM_PORT_INFO));
   memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
          &a_videoPortRxInfo,
          sizeof(DM_PORT_INFO));

   if ( DEV_SUCCESS != dev_PortConnect(m_MMHandle, &portConnectInfoList, this) )
   {
      LOG_ERROR(m_MMHandle, 
                "dev_PortConnect() failure in MM endpoint - %s\n",
                ATDV_ERRMSGP(m_MMHandle));
   }
   else
   {
      LOG_ENTRY(m_MMHandle,"Successful dev_PortConnect() call in MM endpoint\n");
   }
}

//*****************************************************************************
// Function: void MMEndpoint::DisconnectFromPeer()
// Description: Disconnect media streams from peer endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void MMEndpoint::DisconnectFromPeer()
{
   Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( GetBridgedMedia()->M3gIndex());
   DM_PORT_CONNECT_INFO_LIST portConnectInfoList;

   if (m_MMHandle == -1) {
    LOG_ENTRY(m_MMHandle, "PortDisconnect (audio) not needed on device %s\n", m_MMName);
   }
   else {
     // This call disconnects the Tx of MMdevice to Rx of 3G device (Audio)
     EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList,  
                                                       EndpointMngr::AUDIO_PORT,
                                                       GetAudTranscodeEnabled());
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx),
	    &(GetAudioTxPortInfo()),
	    sizeof(DM_PORT_INFO));
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx),
	    &(pEndpoint->GetAudioRxPortInfo()),
	    sizeof(DM_PORT_INFO));
     
     if ( DEV_SUCCESS != dev_PortDisconnect(m_MMHandle, &portConnectInfoList, this) )
       {
	 LOG_ERROR(m_MMHandle,
		   "dev_PortDisconnect() failure in MM endpoint - %s\n",
		   ATDV_ERRMSGP(m_MMHandle));
       }
     else
       {
	 LOG_ENTRY(m_MMHandle,"Successful dev_PortDisconnect() call in MM endpoint\n");
       }
   }
   if (m_MMHandle == -1) {
    LOG_ENTRY(m_MMHandle, "PortDisconnect (video) not needed on device %s\n", m_MMName);
   }
   else {
     // This call disconnects the Tx of MMdevice to Rx of 3G device (Video)
     EndpointMngr::Instance()->InitPortConnectInfoList(&portConnectInfoList,
                                                       EndpointMngr::VIDEO_PORT,
                                                       GetVidTranscodeEnabled());
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_tx), 
	    &(GetVideoTxPortInfo()),
	    sizeof(DM_PORT_INFO));
     memcpy(&(portConnectInfoList.port_connect_info[0].port_info_rx), 
	    &(pEndpoint->GetVideoRxPortInfo()),
	    sizeof(DM_PORT_INFO));
     
     if ( DEV_SUCCESS != dev_PortDisconnect(m_MMHandle, &portConnectInfoList, this) )
       {
	 LOG_ERROR(m_MMHandle, 
		   "dev_PortDisconnect() failure in MM endpoint - %s\n",
		   ATDV_ERRMSGP(m_MMHandle));
       }
     else
       {
	 LOG_ENTRY(m_MMHandle,"Successful dev_PortDisconnect() call in MM endpoint\n");
       }
   }
}

//*****************************************************************************
// Function: void MMEndpoint::LogDevError()
// Description: Lof dev API error
// Return: void 
// Parameters: none 
//*****************************************************************************
void MMEndpoint::LogDevError()
{
   DEV_ERRINFO errorInfo;
   if ( dev_ErrorInfo(&errorInfo) != -1 )
   {
      LOG_ERROR (m_MMHandle, "Error during dev_* call: %d - %s\n", errorInfo.dev_ErrValue, errorInfo.dev_Msg);
   }
}


//*****************************************************************************
// Function: void MMEndpoint::CloseSubDevs()
// Description: Close devices assoicated with this endpoint
// Return: void 
// Parameters: none 
//*****************************************************************************
void MMEndpoint::CloseSubDevs()
{
  
  // TO DO: Close devs only after waiting for Disconnect events
  // App should wait fo PortDisconnect events
  // before closing devices.  This is a temporary workaround :(
  //usleep(500000);

  if (m_MMHandle == -1) {
    LOG_ENTRY(0, "MM close not needed on device %s\n", m_MMName);
  }
  else {
    LOG_ENTRY(m_MMHandle, "Closing MM device\n");
    if ( mm_Close(m_MMHandle, NULL) < 0 )
      {
	LOG_ERROR(m_MMHandle, "mm_Close(%s) failure\n", m_MMName);
      }
    else {
      m_MMHandle = -1;
    }
  }
}

//*****************************************************************************
// Function: void MMEndpoint::SetLastMessageString(const char *pMessageStr)
// Description: Save the last message string
// Return: void 
// Parameters: char *pMessageStr 
//*****************************************************************************
void MMEndpoint::SetLastMessageString(const char *pMessageStr)
{
   int length = (MAX_LAST_MSG_STRING > strlen(pMessageStr)) ? strlen(pMessageStr) : MAX_LAST_MSG_STRING-1;
   memset(m_LastMessageBuff, 0, sizeof(char)*MAX_LAST_MSG_STRING);
   strncpy(&m_LastMessageBuff[0], pMessageStr, length);
   m_LastMessageBuff[length] = '\0';
}

//*****************************************************************************
// Function: char* MMEndpoint::GetStateString()
// Description: Format and return a status string
// Return: char* 
// Parameters: none 
//*****************************************************************************
char * MMEndpoint::GetStateString()
{
   memset(m_StateBuffer, 0, sizeof(char)*MAX_STATE_STRING);
   sprintf(m_StateBuffer, "MM %2d: %15s AUD:%5s%1s  VID:%5s%1s %5s %6s   %14s                                  ",
           m_Index,
           m_CurrentStateStr,
           GetSelAudCoderStr(),
           (GetAudTranscodeEnabled()?"*":" "),
           GetSelVidCoderStr(),
           (GetVidTranscodeEnabled()?"*":" "),
           GetSelVidResStr(),
           GetSelVidFpsStr(),
           m_LastMessageBuff);
   return m_StateBuffer;
}

//*****************************************************************************
// Function: void MMIdleState::Shutdown()
// Description: Process shutdown request
// Return: void
// Parameters: none
//*****************************************************************************
void MMEndpoint::Shutdown()
{
  // May need to have state-dependent shutdown on MM endpoint.  If so, dispatch
  // shutdown request to states here.

  LOG_ENTRY(GetMMHandle(), "Shutdown called in %s state\n", m_pCurrentState->GetStateStr());
  DisconnectFromPeer();
  //CloseSubDevs();
  //Notify(APP_MM_ENDPOINT_DISCONNECTED);
}


//*****************************************************************************
// Function: void MMEndpoint::SetVideoToMPEG4()
// Description: Set the video configuration to MPEG4
// Return: void
// Parameters: none
//*****************************************************************************
void MMEndpoint::SetVideoToMPEG4()
{
  
  LOG_ENTRY(GetMMHandle(), "MMEndpoint - SetVideoToMPEG4()\n");
  //SetSelectedVideoCoder(VID_CODER_MPEG4);
  m_VideoCoderFPS = 		VIDEO_FRAMESPERSEC_10; 
  m_VideoCoderImageHeight = 	VIDEO_IMAGE_HEIGHT_144;
  m_VideoCoderImageWidth = 	VIDEO_IMAGE_WIDTH_176;
  m_VideoCoderCoding = 		VIDEO_CODING_MP4V_ES;
  m_VideoCoderLevel = 		EMM_VIDEO_LEVEL_UNDEFINED;
  m_VideoCoderProfile = 	VIDEO_PROFILE_LEVEL_SP0_MPEG4;
  m_VideoCoderBitRate = 	EMM_VIDEO_BITRATE_UNDEFINED;
  m_VideoCoderFileFormat = 	EMM_FILE_FORMAT_PROPRIETARY;
}

//*****************************************************************************
// Function: void MMEndpoint::SetVideoToH263()
// Description: Set the video configuration to H263
// Return: void
// Parameters: none
//*****************************************************************************
void MMEndpoint::SetVideoToH263()
{
   
  LOG_ENTRY(GetMMHandle(), "MMEndpoint - SetVideoToH263() \n");
  //SetSelectedVideoCoder(VID_CODER_H263);
  m_VideoCoderFPS =             EMM_VIDEO_FRAMESPERSEC_UNDEFINED;
  m_VideoCoderImageHeight =     EMM_VIDEO_IMAGE_HEIGHT_UNDEFINED; 
  m_VideoCoderImageWidth =      EMM_VIDEO_IMAGE_WIDTH_UNDEFINED;
  m_VideoCoderCoding =          EMM_VIDEO_CODING_UNDEFINED;
  m_VideoCoderLevel =           EMM_VIDEO_LEVEL_UNDEFINED;
  m_VideoCoderProfile =         EMM_VIDEO_PROFILE_UNDEFINED; 
  m_VideoCoderBitRate =         EMM_VIDEO_BITRATE_UNDEFINED;
  m_VideoCoderFileFormat =      EMM_FILE_FORMAT_PROPRIETARY;
}

#if 0
//*****************************************************************************
// Function: void MMEndpoint::SetVideoToH264()
// Description: Set the video configuration to H264
// Return: void
// Parameters: none
//*****************************************************************************
void MMEndpoint::SetVideoToH264()
{
   
  LOG_ENTRY(GetMMHandle(), "MMEndpoint - SetVideoToH264() \n");
  //SetSelectedVideoCoder(VID_CODER_H264);
  m_VideoCoderFPS = 		VIDEO_FRAMESPERSEC_10; 
  m_VideoCoderImageHeight = 	VIDEO_IMAGE_HEIGHT_144;
  m_VideoCoderImageWidth = 	VIDEO_IMAGE_WIDTH_176;
  m_VideoCoderCoding = 		VIDEO_CODING_H264;
  m_VideoCoderLevel = 		VIDEO_LEVEL_1_H264;
  m_VideoCoderProfile = 	VIDEO_PROFILE_BASELINE_H264;
  m_VideoCoderBitRate = 	EMM_VIDEO_BITRATE_UNDEFINED;
  m_VideoCoderFileFormat = 	EMM_FILE_FORMAT_PROPRIETARY;
}
#endif

//*****************************************************************************
// Function: void MMEndpoint::SetAudioToAMR()
// Description: Set the Audio configuration to AMR
// Return: void
// Parameters: none
//*****************************************************************************
void MMEndpoint::SetAudioToAMR()
{
   
  LOG_ENTRY(GetMMHandle(), "MMEndpoint - SetAudioToAMR() \n");
  //SetSelectedAudioCoder(AUD_CODER_AMR);
  m_AudioCoderCoding = MM_DATA_FORMAT_AMR_NB_12_20K;
  m_AudioCoderSampleRate = MM_DRT_8KHZ;
  //m_AudioCoderMode = MM_MODE_FILE_TYPE_VOX;
  m_AudioCoderFileFormat = EMM_AUD_FILEFORMAT_PROPRIETARY;
}

//*****************************************************************************
// Function: void MMEndpoint::SetAudioToPCM()
// Description: Set the Audio configuration to PCM
// Return: void
// Parameters: none
//*****************************************************************************
void MMEndpoint::SetAudioToPCM()
{
   
  LOG_ENTRY(GetMMHandle(), "MMEndpoint - SetAudioToPCM() \n");
  m_AudioCoderCoding = MM_DATA_FORMAT_PCM;
  m_AudioCoderSampleRate = MM_DRT_8KHZ;
  //m_AudioCoderMode = MM_MODE_FILE_TYPE_VOX;
  m_AudioCoderFileFormat = EMM_AUD_FILEFORMAT_PROPRIETARY;
}

//*****************************************************************************
// Function: void MMEndpoint::SetCoderSelection()
// Description: Set the Coder Selections based on digit
// Return: void
// Parameters: none
//*****************************************************************************
void MMEndpoint::SetCoderSelection()
{
   char dtmf;
   if(!m_LastDigitRcvd)
      m_LastDigitRcvd = '0';
    
   dtmf = GetLastDigitRcvd();
   //Set the mm Selected Audio/Video Coder types
   SetSelectedMMFileType(EndpointMngr::Instance()->GetMMFileType(dtmf));
   SetSelectedAudioCoder(EndpointMngr::Instance()->GetMMAudioCoder(dtmf));
   SetSelectedVideoCoder(EndpointMngr::Instance()->GetMMVideoCoder(dtmf));
   SetSelectedVideoResolution(EndpointMngr::Instance()->GetMMVideoResolution(dtmf));
   SetSelectedVideoFramesPerSec(EndpointMngr::Instance()->GetMMVideoFramesPerSec(dtmf));
   SetSelectedVideoBitRate(EndpointMngr::Instance()->GetMMVideoBitRate(dtmf));
   
   return;    
}

