/**
* @file shutdown_state.cpp
* @brief Definition of ShutdownStateState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "shutdown_state.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"

const int ShutdownState::AUDIO_STOPPED       = 0x0001;
const int ShutdownState::VIDEO_STOPPED       = 0x0002;
const int ShutdownState::MEDIA_STOPPED       = 0x0003;
const int ShutdownState::H245_STOPPED        = 0x0004;
const int ShutdownState::AUDIO_TX_LCN_CLOSED = 0x0010;
const int ShutdownState::VIDEO_TX_LCN_CLOSED = 0x0020;
const int ShutdownState::TX_LCNS_CLOSED      = 0x0030;

//*****************************************************************************
// Function: ShutdownState::ShutdownState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  ShutdownState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
ShutdownState::ShutdownState(Endpoint* pEndpoint) :
   EPState(pEndpoint)
{
}


//*****************************************************************************
// Function: ShutdownState::~ShutdownState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
ShutdownState::~ShutdownState()
{
}


//*****************************************************************************
// Function: bool ShutdownState::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Procecc an event
// Return: bool 
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
bool ShutdownState::ProcessEvent(long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   switch ( evtType )
   {
      case M3GEV_STOP_MEDIA_CMPLT:
         if ( evtDev == m_pEndpoint->GetAudioHandle() )
         {
            SetStatus(AUDIO_STOPPED);
         }
         else
         {
            SetStatus(VIDEO_STOPPED);
         }
         CheckMediaStopped();
         break;

      case M3GEV_CLOSE_LC_CMPLT:
         M3G_LOGICAL_CHANNEL_NUMBER* pLCN = reinterpret_cast<M3G_LOGICAL_CHANNEL_NUMBER*>(pEvtData);
         LOG_ENTRY(evtDev,"\t  LCN:%d\n\n", *pLCN);
         if ( *pLCN == m_pEndpoint->GetAudioTxLCN() )
         {
            SetStatus(AUDIO_TX_LCN_CLOSED);
            m_pEndpoint->SetAudioTxLCN(0);
         }
         else // assume video LCN
         {
            SetStatus(VIDEO_TX_LCN_CLOSED);
            m_pEndpoint->SetVideoTxLCN(0);
         }
         CheckAllLCNsClosed();
         break;

      case M3GEV_STOP_H245_CMPLT:
         SetStatus(H245_STOPPED);
         H245Stopped();
         break;

      case M3GEV_STOP_H245_FAIL:
         LOG_ERROR(evtDev, "Stop H245 failure\n");
         break;

      case DMEV_PORT_CONNECT:
         LOG_ENTRY(evtDev,"DMEV_PORT_CONNECT\n");
         break;

   }
   return false;

}

//*****************************************************************************
// Function: void ShutdownState::CheckMediaStopped()
// Description: Close logical channels if media is stopped
// Return: void 
// Parameters: none 
//*****************************************************************************
void ShutdownState::CheckMediaStopped()
{
   if (IsStatusSet(MEDIA_STOPPED))
   {
     // No need to call closelc, call stoph245 firmware will take care of close now build 90 onwards
     // m_pEndpoint->CloseLCs();
     m_pEndpoint->Stop_H245();
   }
}

//*****************************************************************************
// Function: void ShutdownState::CheckAllLCNsClosed()
// Description: Stop H245 is logical channels are stopped
// Return: void 
// Parameters: none 
//*****************************************************************************
void ShutdownState::CheckAllLCNsClosed()
{
   if (IsStatusSet(TX_LCNS_CLOSED))
   {
      m_pEndpoint->Stop_H245();
   }
}

//*****************************************************************************
// Function: void ShutdownState::H245Stopped()
// Description: Process H245 stopped event
// Return: void 
// Parameters: none 
//*****************************************************************************
void ShutdownState::H245Stopped()
{
   m_pEndpoint->DisconnectFromPeer();
}


