/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* DIALOGIC CONFIDENTIAL
*
* Copyright (C) 2006-2007 Dialogic Corporation. All Rights Reserved.
* The source code contained or described herein and all documents related
* to the source code ("Material") are owned by Dialogic Corporation or its
* suppliers or licensors. Title to the Material remains with Dialogic Corporation
* or its suppliers and licensors. The Material contains trade secrets and
* proprietary and confidential information of Dialogic or its suppliers and
* licensors. The Material is protected by worldwide copyright and trade secret
* laws and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted, distributed,
* or disclosed in any way without Dialogic's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
//***********************************************************************
//***********************************************************************

// DetectCfg.cpp: implementation of the CDetectCfg class.
//
//////////////////////////////////////////////////////////////////////

#include "pdl.h"
# ifdef WIN32 
#   include <WinSock.h> 
#       include <NCMApi.h>
#       include <NCMTypes.h>
# else 
#   include <sys/socket.h>
#   include <netinet/in.h>
#   include <arpa/inet.h>
#   include <netdb.h>
#       include <iostream> 
#       include <dasi.h>
# endif


static const char *CFG_MODULE_NAME = "CDetectCfg";

//*****************************************************************************
// Purpose	: 
//    CDetectInet constructor
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
CDetectInet::CDetectInet(){
    *m_host_name = 0;
    m_host_addr = 0;
    int rc = gethostname(m_host_name, sizeof(m_host_name));
    if (rc != 0){
        GLBLOG(LOG_ERR1,CFG_MODULE_NAME,
               "%d = gethostname('%s', %d)",
                rc, m_host_name, sizeof(m_host_name));
    }else {
         hostent * hent = gethostbyname(m_host_name);
         if (hent == 0) {
             GLBLOG(LOG_ERR1,CFG_MODULE_NAME,
                   "0 = gethostbyname('%s')",
                    m_host_name);

         }else {
             if (hent->h_length == 0){
                 GLBLOG(LOG_ERR1,CFG_MODULE_NAME,
                       "hostent *hent = gethostbyname('%s'); hent->h_length = 0",
                        m_host_name);
             }else {
                 struct in_addr addr;
                 memcpy(&addr,hent->h_addr_list[0],4);

                 str_storestring(&m_host_addr,
                                 inet_ntoa(addr));

                 GLBLOG(LOG_DBG, CFG_MODULE_NAME, "HostName = %s", m_host_name); 
                 GLBLOG(LOG_DBG, CFG_MODULE_NAME, "HostAddr = %s", m_host_addr); 
             } // hent->length
         } // hent != 0
    } // GetHostName
}
//*****************************************************************************
// Purpose	: 
//    Dump names of all detected devices
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
void CDetectCfg::Dump(){
    GLBLOG(LOG_APP, CFG_MODULE_NAME, "=== Detected Configuration ===");
    m_DxxDevices.Dump();
    m_DtiDevices.Dump();
    m_IpmDevices.Dump();
    m_CnfBoards.Dump();
    m_IptDevices.Dump();

 return;
} //	End of Dump()

//*****************************************************************************
// Purpose	: 
//    Dump number of detected devices (by type)
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
void CDetectCfg::DumpSize(){
    GLBLOG(LOG_DBG, CFG_MODULE_NAME, "=== Detected Configuration ===");

    GLBLOG(LOG_DBG, CFG_MODULE_NAME, "  %s: %4u devices",
                    m_DxxDevices.GetName(),
                    m_DxxDevices.ReservedSize() + m_DxxDevices.UnReservedSize());

    GLBLOG(LOG_DBG, CFG_MODULE_NAME, "  %s: %4u timeslots",
                    m_DtiDevices.GetName(),
                    m_DtiDevices.ReservedSize() + m_DtiDevices.UnReservedSize());

    GLBLOG(LOG_DBG, CFG_MODULE_NAME, "  %s: %4u devices",
                    m_IpmDevices.GetName(),
                    m_IpmDevices.ReservedSize() + m_IpmDevices.UnReservedSize());

    GLBLOG(LOG_DBG, CFG_MODULE_NAME, "  %s: %4u devices",
                    m_IptDevices.GetName(),
                    m_IptDevices.ReservedSize() + m_IptDevices.UnReservedSize());

    GLBLOG(LOG_DBG, CFG_MODULE_NAME, "  %s: %4u boards",
                    m_CnfBoards.GetName(),
                                 m_CnfBoards.ReservedSize() + m_CnfBoards.UnReservedSize());


 return;
} //	End of DumpSize()

//*****************************************************************************
// Purpose	: 
//    Get Number of available devices of given type
// Parameters:	
//    [int] type
// Returns:	
//    number of devices
//    0 if not found or invalid type
//*****************************************************************************
size_t CDetectCfg::GetNumberOf(DLG_DEVICE_TYPE type){
size_t ret = 0;
    switch (type){

        case DEV_DX:
             ret = m_DxxDevices.GetSize();
             break;

        case DEV_IPM:
             ret = m_IpmDevices.GetSize();
             break;

        case DEV_DTI:
             ret = m_DtiDevices.GetSize();
             break;            
             
        case DEV_IPT:
             ret = m_IptDevices.GetSize();
             break;

        case DEV_CNFBOARD:
             ret = m_CnfBoards.GetSize();
             break;   

        case DEV_CNFCONFERENCE:
        case DEV_CNFPARTY:
            // Party and Conferenve names are not known before open,
            // Return 0
             ret = 0;
             break;   
             
        case DEV_REST:
             ret = m_OtherDevices.GetSize();
             break;   

        // just for the records and to make g++ happy:
        case DEV_NONE:
             ret = 0;
             break;
    } // switch type
 return ret;
} //	End of GetNumberOf()
//*****************************************************************************
// Purpose	: 
//    Declare device in use (don't allow other object to Reserve)
// Parameters:	
//    [in] device type 
//    [in/out] device name (pass 0 to reserve ANY available )
//             pass 0 means:
//               const char *name = 0;
//               Pass &name;
// Returns:	
//    true success, false if not found
//*****************************************************************************
bool CDetectCfg::Reserve(DLG_DEVICE_TYPE type, const char **name){
bool brc = false;
    switch (type){
        case DEV_NONE:
             brc = true;
             *name = "Container";
             break;

        case DEV_DX:
             brc = m_DxxDevices.Reserve(name);
             break;

        case DEV_IPM:
             brc = m_IpmDevices.Reserve(name);
             break;

        case DEV_DTI:
             brc = m_DtiDevices.Reserve(name);
             break;            
             
        case DEV_IPT:
             brc = m_IptDevices.Reserve(name);
             break;

        case DEV_CNFBOARD:
             brc = m_CnfBoards.Reserve(name);
             break;   

        case DEV_CNFCONFERENCE:
        case DEV_CNFPARTY:
            // Party and Conferenve names are not known before open,
            // Return NULL name
             *name = 0;
             brc = true;
             break;   
             
        case DEV_REST:
             brc = m_OtherDevices.Reserve(name);
             break;   

        default:
             GLBLOG(LOG_ERR1, CFG_MODULE_NAME, "Reserve:: cannot handle device type %d, see DLG_DEVICE_TYPE in DetectCfg.h",  type);
             brc = false;
             break;
    } // switch type
 return brc;
} //	End of Reserve()

//*****************************************************************************
// Purpose	: 
//    Declare device is not in use (allow other object to Reserve)
// Parameters:	
//    [in] device type
//    [in] and device name
// Returns:	
//    true success, false if not reserved
//*****************************************************************************
bool CDetectCfg::UnReserve(DLG_DEVICE_TYPE type, const char *name){
bool brc = false;
    switch (type){
        case DEV_NONE:
             brc = true;
             break;

        case DEV_DX:
             brc = m_DxxDevices.UnReserve(name);
             break;

        case DEV_IPM:
             brc = m_IpmDevices.UnReserve(name);
             break;

        case DEV_IPT:
             brc = m_IptDevices.UnReserve(name);
             break;

        case DEV_CNFBOARD:
             brc = m_CnfBoards.UnReserve(name);
             break;

        case DEV_CNFCONFERENCE:
        case DEV_CNFPARTY:
             brc = true;
             break;

        case DEV_DTI:
             brc = m_DtiDevices.UnReserve(name);
             break;             

        case DEV_REST:
             brc = m_OtherDevices.UnReserve(name);
             break;   

        default:
             GLBLOG(LOG_ERR1, CFG_MODULE_NAME, "Reserve:: cannot handle device type %d, see DLG_DEVICE_TYPE in DetectCfg.h",  type);
             brc = false;
             break;
    } // switch type
 return brc;
} //	End of UnReserve()

//*****************************************************************************
// Purpose	: 
//    Declare all devices from given type are not in use (allow other object to Reserve)
// Parameters:	
//    [in] device type
// Returns:	
//    true success, false if not reserved
//*****************************************************************************
bool CDetectCfg::UnReserve(DLG_DEVICE_TYPE type){
bool brc = false;
    switch (type){
        case DEV_NONE:
             brc = true;
             break;

        case DEV_DX:
             brc = m_DxxDevices.UnReserve();
             break;

        case DEV_IPM:
             brc = m_IpmDevices.UnReserve();
             break;

        case DEV_CNFBOARD:
             brc = m_CnfBoards.UnReserve();
             break;

        case DEV_CNFCONFERENCE:
        case DEV_CNFPARTY:
             brc = true;
             break;

        case DEV_IPT:
             brc = m_IptDevices.UnReserve();
             break;

        case DEV_DTI:
             brc = m_DtiDevices.UnReserve();
             break;             

        case DEV_REST:
             brc = m_OtherDevices.UnReserve();
             break;   

        default:
             GLBLOG(LOG_ERR1, CFG_MODULE_NAME, "Reserve:: cannot handle device type %d, see DLG_DEVICE_TYPE in DetectCfg.h",  type);
             brc = false;
             break;
    } // switch type
 return brc;
} //	End of UnReserve()

//*****************************************************************************
// Purpose	: 
//    Declare all devices from all types are not in use (allow other object to Reserve)
// Parameters:	
//    none
// Returns:	
//    true success, false if not reserved
//*****************************************************************************
bool CDetectCfg::UnReserve(){
bool brc;
    brc = m_DxxDevices.UnReserve();
    brc = m_DtiDevices.UnReserve();
    brc = m_IpmDevices.UnReserve();
    brc = m_CnfBoards.UnReserve();
    brc = m_IptDevices.UnReserve();
    brc = m_OtherDevices.UnReserve();
 return brc;
} //	End of UnReserve()

//*****************************************************************************
// Purpose	: 
//    Declare that device of given type exist and can be reserved 
// Parameters:	
//    [in] device type
//    [in] name
// Returns:	
//    false means duplication
//*****************************************************************************
bool CDetectCfg::Define(DLG_DEVICE_TYPE type, const char *name){
bool brc = false;
    switch (type){
        case DEV_NONE:
             brc = true;
             break;

        case DEV_DX:
             brc = m_DxxDevices.Define(name);
             break;

        case DEV_IPM:
             brc = m_IpmDevices.Define(name);
             break;

        case DEV_IPT:
             brc = m_IptDevices.Define(name);
             break;

        case DEV_CNFBOARD:
             brc = m_CnfBoards.Define(name);
             break;

        case DEV_CNFCONFERENCE:
        case DEV_CNFPARTY:
             brc = true;
             break;

        case DEV_DTI:
             brc = m_DtiDevices.Define(name);
             break;             

        case DEV_REST:
             brc = m_OtherDevices.Define(name);
             break;             

        default:
             GLBLOG(LOG_ERR1, CFG_MODULE_NAME, "Define:: cannot handle device type %d, see DLG_DEVICE_TYPE in DetectCfg.h",  type);
             brc = false;
             break;
    } // switch type
 return brc;
} //	End of Define()

//*****************************************************************************
// Purpose	: 
//    Check if given dxxx device has play/record capabilities
// Parameters:	
//    [in] device name
// Returns:	
//    true if it has play/record capabilities
//    false if it hasn't or device does not exist
//*****************************************************************************
bool CDetectCfg::ConfirmDxx(const char *dxname){
int handle;
bool brc = false;
    handle = dx_open(dxname,0);
      if (IS_VALID_HANDLE(handle) ) {
          // Device exist
          int rc;
          FEATURE_TABLE VoiceFeature;
          GLBLOG(LOG_DBG, CFG_MODULE_NAME, "%d = dx_open('%s',0)",handle, dxname);

          rc = dx_getfeaturelist(handle,&VoiceFeature);
          GLBLOG(INFORC(rc), CFG_MODULE_NAME,
                     "%d = dx_getfeaturelist(%d ft_play:=0x%x ft_record:=0x%x)",
                      handle, VoiceFeature.ft_play, VoiceFeature.ft_record);
          if (rc == 0){
              brc =    ( VoiceFeature.ft_play != 0 ) 
                    && ( VoiceFeature.ft_record != 0);
          }
          rc = dx_close(handle);
          GLBLOG(INFORC(rc), CFG_MODULE_NAME,
                     "%d = dx_close(%d)", rc, handle);
      }else {
          // device does not exist
          GLBLOG(LOG_ERR1, CFG_MODULE_NAME, "%d = dx_open('%s',0)",handle, dxname);
      }
 return brc;
} //	End of ConfirmDxx()

//*****************************************************************************
// Purpose	: 
//    Detect sub devices for given virtual board
// Parameters:	
//    [in] SRLDEVICEINFO - see manual for description
// Returns:	
//    true = success
//    false = error
//*****************************************************************************
bool CDetectCfg::DetectVirtualBoardConfiguration(SRLDEVICEINFO *pBoardInfo){
int rc;
int num_sub_devices = 0;

   rc = SRLGetSubDevicesOnVirtualBoard(pBoardInfo->szDevName, &num_sub_devices, 0);

   GLBLOG( LOG_DBG, CFG_MODULE_NAME, "  ***Detected %s, %d devices", pBoardInfo->szDevName, num_sub_devices);

   if ( ( rc != ESR_INSUFBUF) && ( rc != ESR_NOERR )  ) {
        GLBLOG( LOG_ERR1, CFG_MODULE_NAME,
                "%d = SRLGetSubDevicesOnVirtualBoard(%s, devices := %d, 0)",
                 rc, pBoardInfo->szDevName, num_sub_devices);
        return false;
   }

   if ( num_sub_devices > 0 ) {
        SRLDEVICEINFO *pDevInfo = 0;
        pDevInfo = new SRLDEVICEINFO[num_sub_devices];
            GLBLOG( LOG_DBG, CFG_MODULE_NAME,
                    "%d = SRLGetSubDevicesOnVirtualBoard(%s, devices := %d, pDevInfo)",
                    rc, pBoardInfo->szDevName, num_sub_devices);

            rc = SRLGetSubDevicesOnVirtualBoard(pBoardInfo->szDevName, &num_sub_devices, pDevInfo);
            GLBLOG( INFORC(rc), CFG_MODULE_NAME,
                      "%d = SRLGetVirtualBoardsOnPhysicalBoard(auid = %d, virt_boards := %d)",
                       rc,num_sub_devices);

            int dev_inx;
            for (dev_inx =0; dev_inx < num_sub_devices; dev_inx++) {
			    if (!GLBLOGOBJ->WillLog(LOG_DBG) ){
				     printf("Inspecting: %-15s\r", pDevInfo[dev_inx].szDevName);fflush(stdout);
				}

                switch (pDevInfo[dev_inx].iDevType){
                       case TYPE_R4_CNF_BOARD: // 536 
                            Define(DEV_CNFBOARD, pBoardInfo->szDevName);
                            break;

                       case TYPE_R4_MM_CHANNEL: // 535
                            Define(DEV_REST, pDevInfo[dev_inx].szDevName);
                            break;

                       case TYPE_R4_MM_BOARD: // 534 
                            Define(DEV_REST, pDevInfo[dev_inx].szDevName);
                            break;

                       case TYPE_R4_VOX_CHANNEL:
                            if (m_pAppParams->GetRequestedNumberOfDxDevices() > detected_dx ) { 
                                // only as needed
                                if ( ConfirmDxx(pDevInfo[dev_inx].szDevName) ){
                                     Define(DEV_DX,pDevInfo[dev_inx].szDevName);
                                     ++detected_dx;
                                }else {
                                     Define(DEV_REST,pDevInfo[dev_inx].szDevName);
                                }
                            }
                            break;

                       case TYPE_R4_DTI_TIMESLOT:
                            Define(DEV_DTI,pDevInfo[dev_inx].szDevName);
                            break;

                       case TYPE_R4_IPM_CHANNEL:
                            Define(DEV_IPM,pDevInfo[dev_inx].szDevName);
                            break;

                       default:
                            break;
                } // switch iDevType
            } // for all subdevices

            GLBLOG(LOG_DBG, CFG_MODULE_NAME,
                            "      %s ... %s",
                            pDevInfo[0].szDevName, pDevInfo[num_sub_devices-1].szDevName );
        delete[] pDevInfo;
   }else {
      // 0 subdevs
      // cnf virtual boards will return 0 subdevs
      GLBLOG(LOG_DBG, CFG_MODULE_NAME,
                      "      %s",  pBoardInfo->szDevName);
      if (pBoardInfo->iDevType == 536) {
         Define(DEV_CNFBOARD,pBoardInfo->szDevName);
      }
   } // else (0 subdevs)
  return true;
} //	End of DetectVirtualBoardConfiguration()

//*****************************************************************************
// Purpose	: 
//    Detect virtual boards that belong to given physical board
// Parameters:	
//    [in] AUID - see manual for description
// Returns:	
//    true = success
//    false = error
//*****************************************************************************
bool CDetectCfg::DetectPhysicalBoardConfiguration(AUID auid){
int rc;
bool brc = true;
// Detect board name
   char brd_name[128];

   int name_length = sizeof(brd_name);
   *brd_name = 0;

   rc = SRLGetPhysicalBoardName(auid, &name_length, brd_name);
   GLBLOG( INFORC(rc), CFG_MODULE_NAME,
                "%d = SRLGetVirtualBoardsOnPhysicalBoard(auid=%d, name_len:=%d, brd_name='%s')",
                rc, auid, name_length, brd_name); 

   int virt_boards = 0;
   rc = SRLGetVirtualBoardsOnPhysicalBoard(auid, &virt_boards, 0);

   GLBLOG( LOG_APP, CFG_MODULE_NAME, "***Detected %s, %d virt_boards", brd_name, virt_boards);

   if ( rc == ESR_INSUFBUF){
       // Expecyed return code
        GLBLOG( LOG_DBG, CFG_MODULE_NAME,
                "%d = SRLGetVirtualBoardsOnPhysicalBoard(auid = %d, virt_boards := %d)",
                rc, virt_boards);
        SRLDEVICEINFO *pInfo; 
        pInfo = new SRLDEVICEINFO[virt_boards];
            rc = SRLGetVirtualBoardsOnPhysicalBoard(auid, &virt_boards, pInfo);
            GLBLOG( LOG_DBG, CFG_MODULE_NAME,
                    "%d = SRLGetVirtualBoardsOnPhysicalBoard(auid = %d, virt_boards := %d)",
                    rc,virt_boards);
            int virt_inx;
            for (virt_inx =0; virt_inx < virt_boards; virt_inx++) {
                 DetectVirtualBoardConfiguration(&(pInfo[virt_inx]));     
            }    

			if (!GLBLOGOBJ->WillLog(LOG_DBG) ){
				 printf("\r%-30s\r", "");fflush(stdout);
			}
        delete[] pInfo;
   }  else {
       // Error
       GLBLOG( LOG_ERR1, CFG_MODULE_NAME,
             "%d = SRLGetVirtualBoardsOnPhysicalBoard(auid = %d, virt_boards := %d)",
              rc, virt_boards);
   } 


  return brc;
} //	End of DetectPhysicalBoardConfiguration()

//*****************************************************************************
// Purpose	: 
//    Detect all physical boards, virtual boards and sub devices of interest for this demo
// Parameters:	
//    none
// Returns:	
//    true = success
//    false = error
//*****************************************************************************
bool CDetectCfg::DetectConfiguration(){
bool brc = true;
AUID *pAuid = 0;
int  rc;
int  nPhysicalBoards = 0;

  // -- Get number of physical boards
     rc = SRLGetAllPhysicalBoards(&nPhysicalBoards, NULL);

     // rc should be INSUFBUFF or NOERR 
     // nPhusical boards should be > 0
	 if (    (     ( rc != ESR_INSUFBUF) 
               &&  ( rc != ESR_NOERR ) 
             )
          || (nPhysicalBoards == 0 ) ){ 

           GLBLOG( LOG_ERR1, CFG_MODULE_NAME, 
                "%d = SRLGetAllPhysicalBoards(&nPhysicalBoards := %d, NULL)",
			         rc,nPhysicalBoards);
           GLBLOG( LOG_ERR1, CFG_MODULE_NAME, "Check if service is started");
		   return false;
     } 

     GLBLOG( LOG_DBG, CFG_MODULE_NAME, 
             "%d = SRLGetAllPhysicalBoards(&nPhysicalBoards := %d, NULL)",
 	         rc,nPhysicalBoards);


    // Get information for all physical boards
     pAuid = new AUID[nPhysicalBoards];
	     rc = SRLGetAllPhysicalBoards(&nPhysicalBoards, pAuid);

         GLBLOG( INFORC(rc), CFG_MODULE_NAME,
                      "%d = SRLGetAllPhysicalBoards(&nPhysicalBoards := %d, &pAuid)",
			          rc, nPhysicalBoards);

         if (  rc != ESR_NOERR )  {
                brc = false;
         }else {            
             int inx;
             for (inx = 0; inx < nPhysicalBoards; inx++ ){
                 DetectPhysicalBoardConfiguration(pAuid[inx]);
             }
         } 

      delete[] pAuid;

      // IPT is not present here, but it is set using gc_Start.
      // 'Detect' IPT boards using global parameters

       {
          // define ipt devices based on global parameters
          // 
          char dev_name[46];
          unsigned int inx;
          // 1 based indexes
          for ( inx = 1; inx <= m_pAppParams->GetRequestedNumberOfIptDevices(); inx++){
                snprintf(dev_name, sizeof(dev_name),"iptB1T%d",inx);
                Define(DEV_IPT,dev_name);
          }
       } // block define IPT


 return brc;
} //	End of DetectConfiguration()


