/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* DIALOGIC CONFIDENTIAL
*
* Copyright (C) 2006-2007 Dialogic Corporation. All Rights Reserved.
* The source code contained or described herein and all documents related
* to the source code ("Material") are owned by Dialogic Corporation or its
* suppliers or licensors. Title to the Material remains with Dialogic Corporation
* or its suppliers and licensors. The Material contains trade secrets and
* proprietary and confidential information of Dialogic or its suppliers and
* licensors. The Material is protected by worldwide copyright and trade secret
* laws and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted, distributed,
* or disclosed in any way without Dialogic's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
//***********************************************************************
//***********************************************************************
// IptDevice.cpp: implementation of the CIptDevice class.
//
//////////////////////////////////////////////////////////////////////
#define PRINT_VERSION

#include <time.h>
#include "pdl.h"

/*
  1.The 8kHz PCMU codec (payload type 0)
  2.The 8kHz 1016 codec (payload type 1)
  3.The 8kHz G726-32 codec (payload type 2)
  4.The 8kHz GSM codec (payload type 3)
  5.The 8kHz G723 codec (payload type 4)
  6.The 8kHz DVI4 codec (payload type 5)
  7.The 16kHz DVI4 codec (payload type 6)
  8.The 8kHz LPC codec (payload type 7)

  9.The 8kHz PCMA codec (payload type 8)

 10.The 8kHz G722 codec (payload type 9)
 11.The 44.1kHz stereo L16 codec (payload type 10)
 12.The 44.1kHz mono L16 codec (payload type 11)
 13.The 8kHz QCELP codec (payload type 12)
 14.The 8kHz CN codec (payload type 13)
 15.The MPA codec (payload type 14)
 16.The 8kHz G728 codec (payload type 15)
 17.The 11.025kHz DVI4 codec (payload type 16)
 18.The 22.050kHz DVI4 codec (payload type 17)
 19.The 8kHz G729 codec (payload type 18)
*/

// translate coder to standard payload type
static int standard_payload_type(int coder){
 int pl = 0;
    switch(coder){
        case CODER_TYPE_G711ULAW64K:	
        case CODER_TYPE_G711ULAW56K:
             pl = 0;
             break;

        case CODER_TYPE_G711ALAW64K:
        case CODER_TYPE_G711ALAW56K:
             pl = 8;
             break;
        case CODER_TYPE_G729:
             pl = 18;
             break;
    }    
 return pl;

}


//*****************************************************************************
// Purpose	: 
//    return parmid and setid name 
// Parameters:	
//    setid, parmid and holders for names
// Returns:	
//    true = both are found
//*****************************************************************************
bool ip_get_setid_name(int set_id, int parm_id,
                       const char **setid_name,
                       const char **parmid_name,
                       DATA_TYPE *dtype){
*setid_name = "";
*parmid_name = "";
bool brc = true;
   switch (set_id){
        case IPSET_CALLINFO:
            *setid_name = "IPSET_CALLINFO";
            switch(parm_id){
                case IPPARM_CONNECTIONMETHOD:
                     *dtype = DATA_CHAR;
                     *parmid_name = "IPPARM_CONNECTIONMETHOD";
                     break;

                case IPPARM_CALLID:
                     *dtype = DATA_OCTETS;
                     *parmid_name = "IPPARM_CALLID";
                     break;

                case IPPARM_DISPLAY:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_DISPLAY";
                     break;

                case IPPARM_FASTSTART_MANDATORY_H245CH:
                     *dtype = DATA_CHAR;
                     *parmid_name = "IPPARM_FASTSTART_MANDATORY_H245CH";
                     break;

                case IPPARM_H245TUNNELING:
                     *dtype = DATA_CHAR;
                     *parmid_name = "IPPARM_H245TUNNELING";
                     break;

                case IPPARM_USERUSER_INFO:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_USERUSER_INFO";
                     break;

                case IPPARM_PHONELIST:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_PHONELIST";
                     break;

                case IPPARM_PROGRESS_IND:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_PROGRESS_IND";
                     break;

                case IPPARM_PRESENTATION_IND:
                     *dtype = DATA_CHAR;
                     *parmid_name = "IPPARM_PRESENTATION_IND";
                     break;

                case IPPARM_MEDIAWAITFORCONNECT:
                     *dtype = DATA_CHAR;
                     *parmid_name = "IPPARM_MEDIAWAITFORCONNECT";
                     break;

                case IPPARM_OFFERED_FASTSTART_CODER:
                     *dtype = DATA_STRUCT;
                     *parmid_name = "IPPARM_OFFERED_FASTSTART_CODER";
                     break;
            }
            break;

        case IPSET_SIP_MSGINFO:
            *setid_name = "IPSET_SIP_MSGINFO";
            switch (parm_id) {   
                case IPPARM_REQUEST_URI:
                     *parmid_name = "IPPARM_REQUEST_URI";
                     *dtype = DATA_STRING;
                     break;

                case IPPARM_CONTACT_URI:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_CONTACT_URI";
                     break;

                case IPPARM_FROM_DISPLAY:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_FROM_DISPLAY";
                     break;

                case IPPARM_TO_DISPLAY:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_TO_DISPLAY";
                     break;

                case IPPARM_CONTACT_DISPLAY:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_CONTACT_DISPLAY";
                     break;

                case IPPARM_REFERRED_BY:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_REFERRED_BY";
                     break;

                case IPPARM_REPLACES:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_REPLACES";
                     break;

                case IPPARM_CONTENT_DISPOSITION:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_CONTENT_DISPOSITION";
                     break;

                case IPPARM_CONTENT_ENCODING:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_CONTENT_ENCODING";
                     break;

                case IPPARM_CONTENT_LENGTH:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_CONTENT_LENGTH";
                     break;

                case IPPARM_CONTENT_TYPE:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_CONTENT_TYPE";
                     break;

                case IPPARM_REFER_TO:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_REFER_TO";
                     break;

                case IPPARM_DIVERSION_URI:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_DIVERSION_URI";
                     break;

                case IPPARM_EVENT_HDR:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_EVENT_HDR";
                     break;

                case IPPARM_EXPIRES_HDR:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_EXPIRES_HDR";
                     break;

                case IPPARM_CALLID_HDR:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_CALLID_HDR";
                     break;

                case IPPARM_SIP_HDR:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_SIP_HDR";
                     break;

                case IPPARM_FROM:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_FROM";
                     break;

                case IPPARM_TO:
                     *dtype = DATA_STRING;
                     *parmid_name = "IPPARM_TO";
                     break;

                default:
                    brc = false;
                    break;
            } // switch parm_id
            break;
		case IPSET_SDP:
            *setid_name = "IPSET_SDP";
            switch (parm_id) {   
                case IPPARM_SDP_OFFER:
                     *parmid_name = "IPPARM_SDP_OFFER";
                     *dtype = DATA_STRING;
                     break;

                case IPPARM_SDP_ANSWER:
                     *parmid_name = "IPPARM_SDP_ANSWER";
                     *dtype = DATA_STRING;
                     break;

                case IPPARM_SDP_OPTION_OFFER:
                     *parmid_name = "IPPARM_SDP_OPTION_OFFER";
                     *dtype = DATA_STRING;
                     break;

                case IPPARM_SDP_OPTION_ANSWER:
                     *parmid_name = "IPPARM_SDP_OPTION_ANSWER";
                     *dtype = DATA_STRING;
                     break;

                default:
                    brc = false;
                    break;
			} // IPSET_SIP switch
			break;
        default: 
            brc = false;
            break;
    } // switch set_id
 return brc;
} // End of ip_get_setid_name 


//*****************************************************************************
// Purpose	: 
//    dump parm block
// Parameters:	
//   [in] parm
// Returns:	
//    none
//*****************************************************************************
void CIptDevice::dump_parm(GC_PARM_DATA_EXT *parm){
    // process parameter
  const char *setid_name;
  const char *parmid_name;
  DATA_TYPE dtype;
  switch(parm->data_size){
       default:
            dtype = DATA_NONE;
            break;

       case 1:
            dtype = DATA_CHAR;
            break;

       case 2:
            dtype = DATA_USHORT;
            break;

       case 4:
            dtype = DATA_UINT;
            break;
  }

  ip_get_setid_name( parm->set_ID, parm->parm_ID,
                     &setid_name, &parmid_name,
                     &dtype);
        
  LOG( LOG_DBG, GetName(), 
                "  set_ID = 0x%x, %s",
                parm->set_ID, setid_name);     

  LOG( LOG_DBG, GetName(), 
                "  parm_ID = 0x%x %s",
                 parm->parm_ID, parmid_name);     

  unsigned int data;                                      

      switch(dtype){
           default:
                LOG( LOG_DBG, GetName(), 
                    "  data_size = %d, value_buf[]",
                    parm->data_size);
                break;
           case DATA_CHAR:
           case DATA_UCHAR:
                data = *(unsigned char *)parm->pData;
                LOG( LOG_DBG, GetName(), 
                    "  data_size = %d, value = 0x%x",
                    parm->data_size, data);
                break;
           case DATA_SHORT:
           case DATA_USHORT:
                data = *(unsigned short *)parm->pData;
                LOG( LOG_DBG, GetName(), 
                    "  data_size = %d, value = 0x%x",
                    parm->data_size, data);
                break;
           case DATA_INT:
           case DATA_UINT:
                data = *(unsigned int *)parm->pData;
                LOG( LOG_DBG, GetName(), 
                    "  data_size = %d, value = 0x%x",
                    parm->data_size, data);
                break;

           case DATA_STRING:
               { char buffer[512];
                 unsigned int len = parm->data_size;
                 if (len > sizeof(buffer)-1){
                     len = sizeof(buffer) -2;
                     buffer[len-1] = '.';
                     buffer[len-2] = '.';
                 }
                 if (len) {
                    memmove(buffer, parm->pData, len);
                 }
                 buffer[len]=0;
                 LOG( LOG_DBG, GetName(), 
                    "  data_size = %d, value = \n%s",
                    parm->data_size, buffer);
               }
                break;
      }
 return;
}

//*****************************************************************************
// Class CnfDemoResourceList
//*****************************************************************************
static const char *s_reslist = "ResourceList";
static NAME_TABLE res_table[] = {
    { "RESOURCE_TYPE_NONE",            RESOURCE_TYPE_NONE } ,
    { "RESOURCE_IPM_LBR",              RESOURCE_IPM_LBR },
    { "RESOURCE_IPM_ALL_AUDIO_CODERS", RESOURCE_IPM_ALL_AUDIO_CODERS },
    { "RESOURCE_IPM_G711_40MS",        RESOURCE_IPM_G711_40MS },
    { "RESOURCE_IPM_G711_30MS",        RESOURCE_IPM_G711_30MS },
    { "RESOURCE_IPM_G711_20MS",        RESOURCE_IPM_G711_20MS },
    { "RESOURCE_IPM_G711_10MS",        RESOURCE_IPM_G711_10MS },
    { "RESOURCE_IPM_G723",             RESOURCE_IPM_G723 },
    { "RESOURCE_IPM_G726",             RESOURCE_IPM_G726 },
    { "RESOURCE_IPM_G729",             RESOURCE_IPM_G729 },
    { "RESOURCE_IPM_AMR_NB",           RESOURCE_IPM_AMR_NB },
    { "RESOURCE_IPM_EVRC",             RESOURCE_IPM_EVRC },
    { "RESOURCE_IPM_GSM_EFR",          RESOURCE_IPM_GSM_EFR },
    { "RESOURCE_IPM_ILBC",             RESOURCE_IPM_ILBC },
    { "RESOURCE_IPM_QCELP",            RESOURCE_IPM_QCELP },
    { "RESOURCE_IPM_SMV",              RESOURCE_IPM_SMV },
    { "RESOURCE_IPM_AMR_WB",           RESOURCE_IPM_AMR_WB },
    { "RESOURCE_IPM_G729E",            RESOURCE_IPM_G729E },
    { "RESOURCE_TYPE_MAX",             RESOURCE_TYPE_MAX },
    { 0,0 }
};

CnfDemoResourceList::CnfDemoResourceList(){
	version = DEV_RESOURCE_LIST_VERSION;
    count = 0;
    memset(&rsList,0,sizeof(rsList));
    m_type = RESERVE_RESOURCE_NONE;
};

void CnfDemoResourceList::Dump(CGenLog *pLog){
    const char *name =  get_reserve_resource_name(m_type);
    pLog->Log(LOG_DBG, s_reslist,"version = 0x%x", version);
    pLog->Log(LOG_DBG, s_reslist,"count   = %d", count);
    pLog->Log(LOG_DBG, s_reslist,"type    = %d %s", m_type, name);
    
    int inx;
    for(inx=0; inx < count; inx++){
        str_findname(rsList[inx], &name, res_table);
        pLog->Log(LOG_DBG, s_reslist,"   %d %s", rsList[inx],name);
    }
};

void CnfDemoResourceList::Init(IPM_CODER_INFO *pCoder){
	count = 0;
    switch(pCoder->eCoderType){
        case CODER_TYPE_G711ALAW64K:
        case CODER_TYPE_G711ALAW56K:
        case CODER_TYPE_G711ULAW64K:
        case CODER_TYPE_G711ULAW56K:
        case CODER_TYPE_G721ADPCM:

		default:
             switch (pCoder->eFrameSize){
                default:
                     GLBLOG(LOG_ERR2, "CnfDemoResourceList", "FrameSize %d: not supported",pCoder->eFrameSize);
                     break;
                case CODER_FRAMESIZE_5:
                    // Not supported
                     GLBLOG(LOG_ERR2, "CnfDemoResourceList", "CODER_FRAMESIZE_5: not supported");       
                     break;
                case CODER_FRAMESIZE_10:
                     rsList[count++] = RESOURCE_IPM_G711_10MS;
                     break;
                case CODER_FRAMESIZE_20:
                     rsList[count++] = RESOURCE_IPM_G711_20MS;
                     break;
                case CODER_FRAMESIZE_30:
   			         rsList[count++] = RESOURCE_IPM_G711_30MS;
                     break;
             }  // switch ip_frame_size
			 break;

		case CODER_TYPE_G7231_5_3K:
		case CODER_TYPE_G7231_6_3K:
			 rsList[count++] = RESOURCE_IPM_G723;
			 break;
	}
 return;
};

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
//*****************************************************************************
// Purpose	: 
//    Constructor
// Parameters:	
//    Configuration parameters ( from configuration file )
// Returns:	
//    none
//*****************************************************************************
CIptDevice::CIptDevice(PCommonParams pCommonParams,
                       PIptParams pIptParams ) 
                     : CNtwkDevice(DEV_IPT, pCommonParams, 0) { 

    m_pIptParams = pIptParams;
    SelectCoder();  
    m_ipm_name = 0;


    if (EXIT_OK == GetExitCode()){
       if (! OpenPrivateLog(m_pIptParams->m_private_log) ){
                SetExitCode(EXIT_LOG);
       }

       if ( EXIT_OK == GetExitCode() ){
             // dx_open etc. 
           if (!open_dx()){
               SetExitCode(EXIT_INIT);
           }
       }

       if ( EXIT_OK == GetExitCode() ){
             // gc_open  etc. 
           if (! Open() ){
               SetExitCode(EXIT_INIT);
           }
       }

    }

    pIptParams->Dump(GetLog());

    m_pSdpParser = 0;
	if (Is3PCCMode()){
		if (GetLog()) {
			m_pSdpParser = new CSdpParser(GetLog());	// parcer for 3PCC mode
		}else {
			m_pSdpParser = new CSdpParser(glb_pAppLog);
		}
	}

    memset(&m_LocalRtpPort, 0, sizeof(m_LocalRtpPort));
    memset(&m_LocalRtcpPort, 0, sizeof(m_LocalRtpPort));
    m_ipm_started = false;

 return;
} // End of Constructor() 

//*****************************************************************************
// Purpose	: 
//    Destructor
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
CIptDevice::~CIptDevice() {
    
    delete m_pSdpParser;
   // UnReserve
   glb_pDetectCfg->UnReserve(DEV_IPM,m_ipm_name);

 return;
} // End of Destructor() 

//*****************************************************************************
// Function	: 
//    OnNewCall():    specific cleanup and init for new call
//    OnNewCall:      specific one-time initialization
// Parameters:	
//    none
// Returns:	
//    true = success
//    false = failure
//*****************************************************************************
bool CIptDevice::OnNewCall() {
   LOG(LOG_APP, GetName(), "Ready to answer %s calls", m_pIptParams->ip_protocol);
   return SetDtmfTransfer();

} // End of OnNewCall() 

bool CIptDevice::OnInitComplete() {
  bool brc = SetCoders();
       brc = SetDtmfTransfer() && brc;
       brc = EnableEchoCancel() && brc;
 return brc;
} // End of OnInitComplete() 

//*****************************************************************************
// Purpose	: 
//    Close device
// Parameters:	
//    none
// Returns:	
//    true = success
//    false = failure
//*****************************************************************************
bool CIptDevice::Close(){
int rc;
	if (separate_ntwk_open){
        UnReserveResource();
		rc = ipm_Close(m_ntwk_srl_handle,0);
		m_ntwk_srl_handle = INV_SRL_HANDLE;
        LOG( RC(rc), m_ipm_name,
             "%d = ipm_Close (0x%x, 0)",
             rc, m_ntwk_srl_handle );

	}
  return CNtwkDevice::Close();
}

//*****************************************************************************
// Purpose	: 
//    Open device
// Parameters:	
//    none
// Returns:	
//    true = success
//    false = failure
//*****************************************************************************
bool CIptDevice::Open(){
 bool brc = false;
 int rc;
 if (IsSrlState(SRLSTATE_VOICEOPENED, true) ) {
     if (    IsSrlState(SRLSTATE_RESERVED, true)  ) { 
          if ( glb_pDetectCfg->Reserve(DEV_IPM, &m_ipm_name) ){
               char gc_name[200];
			   if (Is3PCCMode()){
				   snprintf(gc_name, sizeof(gc_name), ":N_%s:P_%s",
                            GetName(), m_pIptParams->ip_protocol);
			   }else {
				   snprintf(gc_name, sizeof(gc_name), ":N_%s:P_%s:M_%s",
                            GetName(), m_pIptParams->ip_protocol, m_ipm_name);
			   }

	   		   OpenIpml(); // 3PCC mode

               rc = gc_OpenEx(&m_srl_handle, gc_name,EV_ASYNC, this);
               LOG( RC(rc), GetName(),
                    "%d = gc_OpenEx(hndl := 0x%x, %s, EV_ASYNC,0x%x(this) )",
                    rc, m_srl_handle, gc_name, this);

               if (rc == GC_SUCCESS){
                    brc = true ;
                    SetCurrentState(S_OPEN);
                    SetSrlState(SRLSTATE_OPENED);
               }else {
                    process_gc_error();
                    SetCurrentState(S_FINAL);
               }

          }
     } // if ipt is reserved
     else {
           LOG( LOG_ERR1, GetName(), "gc_Open(): (board) - device not reserved");
     } 
 } // if voice is opened
 if (!brc){     
     SetSrlState(SRLSTATE_FAILED_FATAL);
 }

 return brc;
} // End of Open() 


//*****************************************************************************
// Purpose	: 
//    Retrieve IP data (SDP)
// Parameters:	
//   [in] METAEVENT 
// Returns:	
//    none
//*****************************************************************************
void CIptDevice::RetrieveIpData(METAEVENT *metaeventp){
  GC_PARM_BLK *pBlock = (GC_PARM_BLK *)(metaeventp->extevtdatap);
  GC_PARM_DATA_EXT parm;
  INIT_GC_PARM_DATA_EXT(&parm);

  while ( GC_SUCCESS == gc_util_next_parm_ex(pBlock, &parm) ){
		  dump_parm(&parm);
          if (Is3PCCMode()){
			  if ((IPSET_SDP == parm.set_ID ) && (IPPARM_SDP_OFFER == parm.parm_ID) ) {
					 m_pSdpParser->SetBuffer((const char *)parm.pData);
					 m_pSdpParser->ParseSdpOffer();
                     // according to offered sdp, select coder in selected_coder
                     SelectCoder();
                     // According to selected_coder, reserve appropriate resource
                     // This may switch between HOST IP and Board IP 
                     ReserveResource();
                     // In Titusville, local media info depends of selected coder
                     // GetLocalMediaInfo is executed after each coder selection
                     GetLocalMediaInfo(MEDIATYPE_AUDIO_LOCAL_RTP_INFO, &m_LocalRtpPort);
			  }
          }
  } // while
  if (Is3PCCMode()) {
	  m_pSdpParser->Dump();
	  if (m_pSdpParser->m_MediaInfo.sdp_k){
		 // encryption not supported
			LOG(LOG_WARNING, GetName(), "Encryprtion not supported, dropping this call");
            DropRequest();
	  }
  }
 return;
}

//*****************************************************************************
// Purpose	: 
//    Handle events
// Parameters:	
//   [in] event
//   [in] event data
//   [in] data length
//   [in] METAEVENT 
// Returns:	
//    none
//*****************************************************************************
void CIptDevice::HandleEvent(int event,
                             void *evtdata, int evtlen,
                             METAEVENT *metaeventp){

    switch (event){
        case GCEV_OPENEX:
            // Permanent setup
            // Get Network handle ( GC_MEDIADEVICE or GC_NETWORKDEVICE
             GetMediaResourceHandle();
             break;

		case GCEV_ANSWERED:
             RetrieveIpData(metaeventp);
             break;

		case GCEV_ACCEPT:
             RetrieveIpData(metaeventp);
             break;

        case GCEV_REQ_MODIFY_CALL:
        case GCEV_OFFERED:
             RetrieveIpData(metaeventp);
             break;

        case IPMEV_QOS_ALARM:
             OnReceiveAlarmEvent((IPM_QOS_ALARM_DATA *)evtdata);
             break;

        default:
             break;
    } // switch  (event)

    CNtwkDevice::HandleEvent(event, evtdata, evtlen, metaeventp);

 return ;
} // End of HandleEvent

static NAME_TABLE qos_name_table[] = {
    { "QOSTYPE_DTMFDISCARDED", QOSTYPE_DTMFDISCARDED },
    { "QOSTYPE_LOSTPACKETS", QOSTYPE_LOSTPACKETS},
    { "QOSTYPE_JITTER", QOSTYPE_JITTER},
    { "QOSTYPE_ROUNDTRIPLATENCY", QOSTYPE_ROUNDTRIPLATENCY},
    { "QOSTYPE_RTCPTIMEOUT", QOSTYPE_RTCPTIMEOUT},
    { "QOSTYPE_RTPTIMEOUT", QOSTYPE_RTPTIMEOUT},
    { "QOSTYPE_SEC_AUTH_FAIL_AUDIO", QOSTYPE_SEC_AUTH_FAIL_AUDIO},
    { "QOSTYPE_SEC_AUTH_FAIL_VIDEO", QOSTYPE_SEC_AUTH_FAIL_VIDEO},
    { "QOSTYPE_SEC_PKT_REPLAY_AUDIO", QOSTYPE_SEC_PKT_REPLAY_AUDIO},
    { "QOSTYPE_SEC_PKT_REPLAY_VIDEO", QOSTYPE_SEC_PKT_REPLAY_VIDEO},
    { "QOSTYPE_SEC_MKI_NOMATCH_AUDIO", QOSTYPE_SEC_MKI_NOMATCH_AUDIO},
    { "QOSTYPE_SEC_MKI_NOMATCH_VIDEO", QOSTYPE_SEC_MKI_NOMATCH_VIDEO},
    { "QOSTYPE_NETWORKFAILURE", QOSTYPE_NETWORKFAILURE},
    { 0, 0}
};

static NAME_TABLE state_name_table[] = {
    { "ALARM_STATE_OFF",  ALARM_STATE_OFF },
    { "ALARM_STATE_ON",  ALARM_STATE_ON },
    { 0,0 }
};

static CNameTable state_table (state_name_table);
static CNameTable qos_table (qos_name_table);

//*****************************************************************************
// Purpose	: 
//    Handle QOS Alarm
// Parameters:	
//    alarm data
// Returns:	
//    true = success
//*****************************************************************************
void CIptDevice::OnReceiveAlarmEvent(IPM_QOS_ALARM_DATA * alarm_data){
    const char *state_name = qos_table.find_name(alarm_data->eQoSType);
    const char *alarm_type = state_table.find_name(alarm_data->eAlarmState);

	LOG(LOG_WARNING, GetName(), "Received QOS Alarm" );
	LOG(LOG_WARNING, GetName(), "    Type  = %d %s",alarm_data->eQoSType,alarm_type );
	LOG(LOG_WARNING, GetName(), "    State = %d %s",alarm_data->eAlarmState,state_name );
 return;
}
//*****************************************************************************
// Purpose	: 
//    Set coders according to configuration file
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::SetCoders(){

if (Is3PCCMode()){
	 LOG( LOG_API, GetName(),
		 "SetCoders(): skip in 3PCC mode");
	 return true;
} 

int rc;
bool brc;
IP_CAPABILITY ipCap;
GC_PARM_BLKP pParmBlk = 0;

 int coder = m_pIptParams->ip_coder;       // CODER_TYPE_G711ALAW64K;
 int vad   = m_pIptParams->ip_vad;         // CODER_VAD_DISABLE;
 int frame = m_pIptParams->ip_frame_size;  // CODER_FRAMESIZE_30;

	memset(&ipCap, 0, sizeof(ipCap));

	ipCap.capability = coder;
	ipCap.type = GCCAPTYPE_AUDIO;
	ipCap.extra.audio.frames_per_pkt = frame;  
	ipCap.extra.audio.VAD = (unsigned short)vad;
	ipCap.payload_type = m_pIptParams->ip_payoad_type;
	ipCap.direction = IP_CAP_DIR_LCLTRANSMIT;
	
    rc =  gc_util_insert_parm_ref( &pParmBlk,
                                   GCSET_CHAN_CAPABILITY, IPPARM_LOCAL_CAPABILITY,
						           sizeof (IP_CAPABILITY), &ipCap);


    brc = (rc == GC_SUCCESS);

	if (brc) {
		ipCap.direction = IP_CAP_DIR_LCLRECEIVE;
        rc =  gc_util_insert_parm_ref( &pParmBlk,
                                   GCSET_CHAN_CAPABILITY, IPPARM_LOCAL_CAPABILITY,
						           sizeof (IP_CAPABILITY), &ipCap);
        brc = (rc == GC_SUCCESS);

	    if (brc) {

			rc = gc_SetUserInfo(GCTGT_GCLIB_CHAN, m_srl_handle, pParmBlk, GC_ALLCALLS);
			brc = (rc == GC_SUCCESS);

		    LOG( LOG_API, GetName(),
                 "SetCoders(): Coder = %d %s GC_ALLCALLS",
                 ipCap.capability, ip_coder_name(ipCap.capability));

		    LOG( LOG_API, GetName(),
                 "             FrameSize = %d %s",
                 ipCap.extra.audio.frames_per_pkt, ip_framesize_name(ipCap.extra.audio.frames_per_pkt));

		    LOG( LOG_API, GetName(),
                 "             Vad = %d %s",
                 ipCap.extra.audio.VAD, ip_vad_name(ipCap.extra.audio.VAD));

		    LOG( RC(rc), GetName(),
                 "%d = gc_SetUserInfo(GCTGT_GCLIB_CHAN, hndl = 0x%x, pParmBlk, GC_ALLCALLS)",
                 rc, m_srl_handle);

            if (rc != GC_SUCCESS){
                process_gc_error();
            }
        } // if brc (success second insert parm)

    	gc_util_delete_parm_blk(pParmBlk);
    } // if brc (success first insert parm)
 	return brc;
} // End of SetCoders()

//*****************************************************************************
// Purpose	: 
//    Set DTMF transfer type according to configuration file
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::SetDtmfTransfer(){
int rc;
bool brc;

   if (Is3PCCMode()){
		int mode = DTMFXFERMODE_INBAND;  // Default
		const char *name = "INBAND";
		switch(m_pIptParams->ip_digits){
			case IP_DTMF_TYPE_INBAND_RTP:
				 mode = DTMFXFERMODE_INBAND;
				 break;
			case IP_DTMF_TYPE_RFC_2833:
				 mode = DTMFXFERMODE_RFC2833;
				 name = "RFC2833";
				 break;
		}

		IPM_PARM_INFO ParmInfo;
		ParmInfo.eParm = PARMCH_DTMFXFERMODE;
		ParmInfo.pvParmValue = &mode;

		rc = ipm_SetParm(m_ntwk_srl_handle, &ParmInfo, EV_SYNC);
		LOG( RC(rc), GetName(),
				 "ipm_SetParm([%s] XFERMODE, %d %s, EV_SYNC)",
				 m_ipm_name, *(int *)ParmInfo.pvParmValue, name);
		brc = ( rc == 0);
		if (!brc){
			process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
		}

        int pltype = 101;
	    ParmInfo.eParm = PARMCH_RFC2833EVT_TX_PLT;
	    ParmInfo.pvParmValue = &pltype;

	    rc = ipm_SetParm(m_ntwk_srl_handle, &ParmInfo, EV_SYNC);
	    LOG( RC(rc), GetName(),
			     "ipm_SetParm([%s] PARMCH_RFC2833EVT_TX_PLT, %d, EV_SYNC)",
			     m_ipm_name, *(int *)ParmInfo.pvParmValue);
	    brc = ( rc == 0);

	    if (!brc){
		    process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
	    }


	    ParmInfo.eParm = PARMCH_RFC2833EVT_RX_PLT;
	    rc = ipm_SetParm(m_ntwk_srl_handle, &ParmInfo, EV_SYNC);
	    LOG( RC(rc), GetName(),
			     "ipm_SetParm([%s] PARMCH_RFC2833EVT_RX_PLT, %d, EV_SYNC)",
			     m_ipm_name, *(int *)ParmInfo.pvParmValue);
	    brc = ( rc == 0);
	    if (!brc){
		    process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
	    }

        int val = 1;
        ParmInfo.eParm = PARMCH_RFC2833GEN_TO_TDM;
		ParmInfo.pvParmValue = &val;
     	rc = ipm_SetParm(m_ntwk_srl_handle, &ParmInfo, EV_SYNC);
     	LOG( RC(rc), GetName(),
			 "ipm_SetParm([%s] PARMCH_RFC2833GEN_TO_TDM, %d,  EV_SYNC)",
              m_ipm_name, *(int *)ParmInfo.pvParmValue);
	    brc = ( rc == 0);
	    if (!brc){
		    process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
	    }


        ParmInfo.eParm = PARMCH_RFC2833GEN_TO_IP;
		ParmInfo.pvParmValue = &val;
     	rc = ipm_SetParm(m_ntwk_srl_handle, &ParmInfo, EV_SYNC);
     	LOG( RC(rc), GetName(),
			 "ipm_SetParm([%s] PARMCH_RFC2833GEN_TO_IP, %d, EV_SYNC)",
             m_ipm_name, *(int *)ParmInfo.pvParmValue);
	    brc = ( rc == 0);
	    if (!brc){
		    process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
	    }

     return brc;
   } 

// 1PCC mode:
int mode = m_pIptParams->ip_digits;
GC_PARM_BLKP pParmBlk = 0;

	rc = gc_util_insert_parm_val(&pParmBlk, IPSET_DTMF,
							     IPPARM_SUPPORT_DTMF_BITMASK,
							     sizeof (int), mode);

    brc = (rc == GC_SUCCESS);

	if (brc) {
		rc = gc_SetUserInfo(GCTGT_GCLIB_CHAN, m_srl_handle, pParmBlk, GC_SINGLECALL);
        LOG( RC(rc), GetName(),
             "SetDtmfTransfer(): mode = %d %s, GC_SINGLECALL",
             mode, ip_digit_type_name(mode)) ;

        LOG( RC(rc), GetName(),
             "%d = gc_SetUserInfo [Set digits](GCTGT_GCLIB_CHAN, hndl = 0x%x, pParmBlk, GC_SINGLECALL)",
             rc, m_srl_handle );
        
     	gc_util_delete_parm_blk(pParmBlk);

        if (rc != GC_SUCCESS){
            process_gc_error();
        }
    }

 	return brc;
}// End of SetDtmfTransfer()

//*****************************************************************************
// Purpose	: 
//    Enable echo cancellation
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::EnableEchoCancel(){
 int rc = 0;
#ifdef IPM_SET_ECHOCANCEL
 bool brc = true;
 int echoCancellation = ECACTIVE_ON;
 IPM_PARM_INFO ipmParmInfo;
 GC_PARM_BLKP gcParmBlk = 0;

   ipmParmInfo.eParm = PARMCH_ECACTIVE;
   ipmParmInfo.pvParmValue = (void *)&echoCancellation;

   if (Is3PCCMode() ) {
#ifdef IPM_SET_ECHOCANCEL
		rc = ipm_SetParm(m_ntwk_srl_handle, &ipmParmInfo, EV_SYNC);
		LOG( RC(rc), GetName(),
				 "ipm_SetParm([%s, EchoCancel] %d ECACTIVE_ON, EV_SYNC)",
				 m_ipm_name, ECACTIVE_ON);
		brc = ( rc == 0);
		if (!brc){
			process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
		}
#endif
   }else {
      gc_util_insert_parm_ref( &gcParmBlk,
                              IPSET_CONFIG,
                              IPPARM_IPMPARM,
                              (unsigned long)sizeof(IPM_PARM_INFO),
                              &ipmParmInfo);
   }

   int echoTail = ECHO_TAIL_64;
   ipmParmInfo.eParm = PARMCH_ECHOTAIL;
   ipmParmInfo.pvParmValue = (void *)&echoTail;

   if (Is3PCCMode() ){
#ifdef IPM_SET_ECHOCANCEL
		rc = ipm_SetParm(m_ntwk_srl_handle, &ipmParmInfo, EV_SYNC);
		LOG( RC(rc), GetName(),
				 "ipm_SetParm([%s, EchoTail] %d ECHO_TAIL_64, EV_SYNC)",
				 m_ipm_name,  ECHO_TAIL_64);
		brc = ( rc == 0);
		if (!brc){
			process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
		}
#endif
   } else {
		gc_util_insert_parm_ref(&gcParmBlk,
                            IPSET_CONFIG,
                            IPPARM_IPMPARM,
                            (unsigned long)sizeof(IPM_PARM_INFO),
                            &ipmParmInfo);

		rc = gc_SetUserInfo(GCTGT_GCLIB_CHAN, m_srl_handle, gcParmBlk, GC_ALLCALLS);

		LOG( RC(rc), GetName(),
			 "%d = gc_SetUserInfo [Echo cancel](GCTGT_GCLIB_CHAN, hndl = 0x%x, pParmBlk, GC_ALLCALLS)",
			rc, m_srl_handle );

		gc_util_delete_parm_blk(gcParmBlk);

		brc = (rc == GC_SUCCESS);

	    if (rc != GC_SUCCESS){
		    process_gc_error();
		}
   }
  return brc;
#else
    LOG( LOG_API, GetName(),
         "%d = gc_SetUserInfo [Echo cancel](is not supported in this release)",rc);
  return true; 
#endif
}

//*****************************************************************************
// Purpose	: 
//     Enable IPMP Events
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::EnableIpmlEvents(){
 // Enable IPML events
 eIPM_EVENT Events[20];
 int inx=0;
     Events[inx++] = EVT_LOSTPACKETS;
	 Events[inx++] = EVT_JITTER;
	 Events[inx++] = EVT_RTCPTIMEOUT;
	 Events[inx++] = EVT_RTPTIMEOUT;
	 Events[inx++] = EVT_RFC2833;
     Events[inx++] = EVT_DTMFDISCARDED;


  int rc  = ipm_EnableEvents(m_ntwk_srl_handle, Events, inx, EV_SYNC);

	LOG( RC(rc), m_ipm_name,
		 "%d = ipm_EnableEvents (handle=0x%x,{LOSTPACKETS,JITTER,RTCPTIMEOUT,RTPTIMEOUT,RFC2833})",
		 rc, m_ntwk_srl_handle );

  bool brc = true; 

  if (rc != 0 ){
	  brc = false;
	  process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
  }
 return brc;
}

static NAME_TABLE info_name[] = {
    {  "MEDIATYPE_AUDIO_REMOTE_RTP_INFO",  MEDIATYPE_AUDIO_REMOTE_RTP_INFO }, 
    {  "MEDIATYPE_AUDIO_LOCAL_RTP_INFO",   MEDIATYPE_AUDIO_LOCAL_RTP_INFO }, 
    {  "MEDIATYPE_AUDIO_REMOTE_RTCP_INFO", MEDIATYPE_AUDIO_REMOTE_RTCP_INFO }, 
    {  "MEDIATYPE_AUDIO_LOCAL_RTCP_INFO",  MEDIATYPE_AUDIO_LOCAL_RTCP_INFO }, 
    {  "MEDIATYPE_AUDIO_REMOTE_CODER_INFO",MEDIATYPE_AUDIO_REMOTE_CODER_INFO }, 
    {  "MEDIATYPE_AUDIO_LOCAL_CODER_INFO", MEDIATYPE_AUDIO_LOCAL_CODER_INFO }, 
	//FAX
    {  "MEDIATYPE_LOCAL_UDPTL_T38_INFO",   MEDIATYPE_LOCAL_UDPTL_T38_INFO }, 
    {  "MEDIATYPE_REMOTE_UDPTL_T38_INFO",  MEDIATYPE_REMOTE_UDPTL_T38_INFO }, 
    {  "MEDIATYPE_FAX_SIGNAL",             MEDIATYPE_FAX_SIGNAL }, 

	//VIDEO
    {  "MEDIATYPE_VIDEO_REMOTE_RTP_INFO",  MEDIATYPE_VIDEO_REMOTE_RTP_INFO }, 
    {  "MEDIATYPE_VIDEO_LOCAL_RTP_INFO",   MEDIATYPE_VIDEO_LOCAL_RTP_INFO }, 
    {  "MEDIATYPE_VIDEO_REMOTE_RTCP_INFO", MEDIATYPE_VIDEO_REMOTE_RTCP_INFO }, 
    {  "MEDIATYPE_VIDEO_LOCAL_RTCP_INFO",  MEDIATYPE_VIDEO_LOCAL_RTCP_INFO }, 
    {  "MEDIATYPE_VIDEO_REMOTE_CODER_INFO",MEDIATYPE_VIDEO_REMOTE_CODER_INFO }, 
    {  "MEDIATYPE_VIDEO_LOCAL_CODER_INFO", MEDIATYPE_VIDEO_LOCAL_CODER_INFO }, 
    { 0, 0}
};

//*****************************************************************************
// Purpose	: 
//     GetLocalMediaInfo
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::GetLocalMediaInfo(eIPM_MEDIA_TYPE type, IPM_PORT_INFO * pInfo) {
 IPM_MEDIA_INFO LocalMediaInfo;

   // Get local media info (RTP)
   LocalMediaInfo.unCount = 1;
   LocalMediaInfo.MediaData[0].eMediaType = type;

   int rc = ipm_GetLocalMediaInfo(m_ntwk_srl_handle, &LocalMediaInfo, EV_SYNC );
   const char *name;
   str_findname(type, &name, info_name);
   LOG( RC(rc), m_ipm_name,
			 "%d = ipm_GetLocalMediaInfo (handle=0x%x,{%s}, EV_SYNC)",
			 rc, m_ntwk_srl_handle, name );

   bool brc = true;
   if (rc != 0){
       brc = false;
       process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
   } else {
       bool found = false;
	   unsigned int inx;
       for (inx=0; inx<LocalMediaInfo.unCount; inx++){
           if (type == LocalMediaInfo.MediaData[inx].eMediaType) {
               *pInfo = LocalMediaInfo.MediaData[inx].mediaInfo.PortInfo;
               found = true;
           }
           const char *name;
           str_findname(LocalMediaInfo.MediaData[inx].eMediaType, &name, info_name);
           DumpPortInfo(pInfo);
           break;
       }
       if (!found){
	    	LOG( LOG_ERR1, m_ipm_name, "Error getting MEDIATYPE_AUDIO_LOCAL_RTP_INFO ( missing in responce)");
           brc = false;
       } 
       // rfc 2327:
       // Rtcp IP = Rtp IP
       // Rtcp port = IP port+ 1        
       m_LocalRtcpPort = m_LocalRtpPort;
       m_LocalRtcpPort.unPortId++;
       LOG(LOG_API, GetName(),"Local RTCP port (next)"); 
       DumpPortInfo(&m_LocalRtcpPort);
   }
  return brc; 
}

//*****************************************************************************
// Purpose	: 
//    Select coder ( 3PCC mode)
//  This procedure should select a coder using remote sdp offer 
//     Currently, only one selection is available - 
//                whatever is specified in config file
//
// Parameters:	
//    none, use received Media Info from SDP message
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::SelectCoder() {
 memset(&selected_coder,0,sizeof(selected_coder));

   selected_coder.eCoderType = m_pIptParams->ip_coder;
   selected_coder.eFrameSize = m_pIptParams->ip_frame_size;
   selected_coder.eVadEnable = m_pIptParams->ip_vad;

   if (m_pIptParams->ip_payoad_type == IP_USE_STANDARD_PAYLOADTYPE){
       selected_coder.unCoderPayloadType = standard_payload_type(m_pIptParams->ip_coder);
   } else {
       selected_coder.unCoderPayloadType = m_pIptParams->ip_payoad_type;
   }
   selected_coder.unRedPayloadType = 0;

 return true;
}

//*****************************************************************************
// Purpose	: 
//     Reserve Resource
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::ReserveResource() {
 int rc;
 bool brc = true;
 UnReserveResource();

 switch ( GetReserveResourceType() ) {
		default:
		case RESERVE_RESOURCE_NONE:
             dev_resource_list.m_type = RESERVE_RESOURCE_NONE;
			 break;

		case RESERVE_RESOURCE_EX:
             dev_resource_list.Init(&selected_coder);
             dev_resource_list.Dump(GetLog());
			 rc = dev_ReserveResourceEx(m_ntwk_srl_handle, &dev_resource_list , EV_SYNC);
			 LOG( RC(rc), m_ipm_name,
			 "%d = dev_ReserveResourceEx (0x%x, <ResourceList>, EV_SYNC)",
			  rc, m_ntwk_srl_handle );

			 if (rc == BRD_FAILURE){
  				 process_dev_error( );
				 brc = false;
			 } else {
				 dev_resource_list.m_type = RESERVE_RESOURCE_EX;
			 }
			 break;

		case RESERVE_RESOURCE:
		     rc = dev_ReserveResource(m_ntwk_srl_handle, RESOURCE_IPM_LBR , EV_SYNC);
				  LOG( RC(rc), m_ipm_name,
					   "%d = dev_ReserveResource(0x%x, RESOURCE_IPM_LBR, EV_SYNC)",
						rc, m_ntwk_srl_handle );

			  if (rc == BRD_FAILURE){
  					process_dev_error( );
					brc = false;
			  } else {
					dev_resource_list.m_type = RESERVE_RESOURCE;
			  }
			  break;
	 } // switch GetReserveResourceType()
 return brc;
}
//*****************************************************************************
// Purpose	: 
//     Reserve Resource
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::UnReserveResource(){
 int rc;
 bool brc = true;
	switch (dev_resource_list.m_type) {
		default:
		case RESERVE_RESOURCE_NONE:
			 break;

		case RESERVE_RESOURCE_EX:
             dev_resource_list.Dump(GetLog());
             rc = dev_ReleaseResourceEx(m_ntwk_srl_handle, &dev_resource_list, EV_SYNC);
		     LOG( RC(rc), m_ipm_name,
			       "%d = dev_ReleaseResourceEx (0x%x, <ResourceList>, EV_SYNC)",
				   rc, m_ntwk_srl_handle );

 			 if (rc == BRD_FAILURE){
  			     process_dev_error( );
                 brc = false;
			 } 
			 break;	

		case RESERVE_RESOURCE:
             rc = dev_ReleaseResource(m_ntwk_srl_handle, RESOURCE_IPM_LBR, EV_SYNC);
		     LOG( RC(rc), m_ipm_name,
			       "%d = dev_ReleaseResource (0x%x, RESOURCE_IPM_LBR, EV_SYNC)",
				   rc, m_ntwk_srl_handle );

 			 if (rc == BRD_FAILURE){
  			     process_dev_error( );
                 brc = false;
			 } 
			 break;	
	}// switch reserved_resource
 dev_resource_list.m_type = RESERVE_RESOURCE_NONE;
 return brc;
}

//*****************************************************************************
// Purpose	: 
//     Open ipml in 3PCC mode
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::OpenIpml(){
bool brc = true;
 if (Is3PCCMode() ){
     m_ntwk_srl_handle = ipm_Open(m_ipm_name, NULL, EV_SYNC);
	 int rc = (m_ntwk_srl_handle > 0)?0:-1;
     LOG( RC(rc), m_ipm_name,
          "0x%x = ipm_Open (name = %s, EV_SYNC)",
          m_ntwk_srl_handle, m_ipm_name );
	 if (rc == -1 ){
	     brc = false;
	     process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
	 } else {
		 separate_ntwk_open = true;
         if (!EnableIpmlEvents()) {
              brc = false;
         } 
	 } // else - open success ends
        
	 if (brc){
         ReserveResource();
	 } // brc
 } // 3PCC mode
 return brc;
}
//*****************************************************************************
// Purpose	: 
//     Get Network handle ( GC_MEDIADEVICE )
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CIptDevice::GetMediaResourceHandle(){
  bool brc = true;
	if (Is3PCCMode() ){
        if ( ! GetLocalMediaInfo(MEDIATYPE_AUDIO_LOCAL_RTP_INFO, &m_LocalRtpPort) ) {
               brc = false;
		}
	}else {
		// 1PCC mode
        int rc =  gc_GetResourceH(m_srl_handle, (int *)&m_ntwk_srl_handle, GC_MEDIADEVICE);
        LOG( RC(rc), GetName(),
             "%d = gc_GetResourceH (hndl = 0x%x, ipml_hndl := 0x%x, GC_MEDIADEVICE)",
             rc, m_srl_handle, m_ntwk_srl_handle );
        
        if (rc != GC_SUCCESS) {
            brc = false;
            process_gc_error();
        }
	}
    return brc;
}// End of GetMediaResourceHandle()

//*****************************************************************************
// Purpose	: 
//     Make SDP message for 3PCC mode
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
# ifdef _WIN32
#   define NL "\n\r"
# else
#   define NL "\n"
# endif

bool CIptDevice::MakeSdpMessage(int sdp_type){
 time_t now;
 time(&now);
 char msg[MAX_SDP_MESSAGE];
 GC_PARM_BLK *gc_parm_blkp = 0;
// \n\r
 const char *rtpmap;
 if (selected_coder.unCoderPayloadType == 0){
     rtpmap = "0 PCMU/8000";
 }else {
     rtpmap = "8 PCMA/8000";
 }

 snprintf(msg, sizeof(msg)-1, 
	       "v=0"NL
		   "o=Dialogic_IPCCLib %d %d IN IP4 %s"NL  // 8000, 8000, IpAddr
		   "s=CnfDemo"NL
		   "c=IN IP4 %s"NL                  // IpAddr
		   "t=0 0"NL
           "a=direction:active"NL
		   "m=audio %d RTP/AVP %d"NL  // Port, Coder 8? (18 2 15)
           "a=rtpmap:%s"NL 
 //          "a=rtpmap:0 PCMU/8000"NL 
 //          "a=rtpmap:8 PCMA/8000"NL 
           "a=rtpmap:101 telephone-event/8000"NL  
           "a=ptime:%d"NL
           "a=sendrecv"NL,

		   40375936, 40375937, m_LocalRtpPort.cIPAddress,
           m_LocalRtpPort.cIPAddress,
           m_LocalRtpPort.unPortId, selected_coder.unCoderPayloadType,
           rtpmap, selected_coder.eFrameSize); 

  int rc = gc_util_insert_parm_ref_ex( &gc_parm_blkp,
									   IPSET_SDP,
								       sdp_type,
			  						   strlen(msg)+1,
									   msg);
  if (rc == GC_SUCCESS) {
	  GC_PARM_DATA_EXT parm;
	  INIT_GC_PARM_DATA_EXT(&parm);

      LOG(LOG_DBG, GetName(), "Sending SDP message:");
	  while ( GC_SUCCESS == gc_util_next_parm_ex(gc_parm_blkp, &parm) ){
			  dump_parm(&parm);
	  } // while

	  rc = gc_SetUserInfo( GCTGT_GCLIB_CRN,
							m_crn,
							gc_parm_blkp, 
							GC_NEXT_OUTBOUND_MSG);

   	  LOG( RC(rc), GetName(),
	  	   "%d = gc_SetUserInfo (GCTHT_GCLIB_CRN, m_crn = 0x%x, gc_parm_blkp, GC_NEXT_OUTBOUND_MSG)",
		   rc, m_crn );

      if (rc != GC_SUCCESS){
            process_gc_error();
	  }
      gc_util_delete_parm_blk(gc_parm_blkp);
  }

  return (rc == GC_SUCCESS);	  

}
//*****************************************************************************
// Purpose	: 
//    Stop IP media
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CIptDevice::StopMedia(){
  if (m_ipm_started ) {
      int rc  = ipm_Stop(m_ntwk_srl_handle, STOP_ALL, EV_SYNC);

      LOG( RC(rc), m_ipm_name,
  		   "%d = ipm_Stop (handle=0x%x,STOP_ALL, EV_SYNC)",
           rc, m_ntwk_srl_handle );

       if (rc != 0){
           process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
           return false;
       } else {
           m_ipm_started = true;
       }

  }
  return true;
}
//*****************************************************************************
// Purpose	: 
//    Dump media info structure
// Parameters:	
//    IPM_MEDIA_INFO structure
// Returns:	
//    bool (true = success)
//*****************************************************************************
void CIptDevice::DumpMediaInfo(IPM_MEDIA_INFO *pMediaInfo){
    LOG(LOG_API, GetName(), "Media info structure ( %d elements )",pMediaInfo->unCount); 
    for (unsigned int inx = 0; inx<pMediaInfo->unCount; inx++){
        DumpMediaData(&(pMediaInfo->MediaData[inx]));
    }
 return;
}
//*****************************************************************************
// Purpose	: 
//    Dump Port info structure
// Parameters:	
//    IPM_PORT_INFO
// Returns:	
//    bool (true = success)
//*****************************************************************************
void CIptDevice::DumpPortInfo(IPM_PORT_INFO *pPort){
     LOG(LOG_API, GetName(), "    IP = %s:%d",pPort->cIPAddress, pPort->unPortId); 
 return;     
}
//*****************************************************************************
// Purpose	: 
//    Dump coder info structire
// Parameters:	
//    IPM_AUDIO_CODER_INFO
// Returns:	
//    bool (true = success)
//*****************************************************************************

void CIptDevice::DumpAudioCoderInfo(IPM_AUDIO_CODER_INFO *pCoder){
    const char *coder_name = ip_coder_name(pCoder->eCoderType);
    const char *framesize_name =  ip_framesize_name(pCoder->eFrameSize);
    const char *vad_name =  ip_vad_name(pCoder->eVadEnable);

     LOG(LOG_API, GetName(), "    eCoderType         = %d %s", pCoder->eCoderType, coder_name); 
     LOG(LOG_API, GetName(), "    eFrameSize         = %d %s", pCoder->eFrameSize, framesize_name); 
     LOG(LOG_API, GetName(), "    unFramesPerPkt     = %d", pCoder->unFramesPerPkt); 
     LOG(LOG_API, GetName(), "    eVadEnable         = %d %s", pCoder->eVadEnable, vad_name); 
     LOG(LOG_API, GetName(), "    unCoderPayloadType = %d", pCoder->unCoderPayloadType); 
     LOG(LOG_API, GetName(), "    unRedPayloadType   = %d", pCoder->unRedPayloadType); 

}

//*****************************************************************************
// Purpose	: 
//    Dump media data structure
// Parameters:	
//    IPM_MEDIA
// Returns:	
//    bool (true = success)
//*****************************************************************************
void CIptDevice::DumpMediaData(IPM_MEDIA *pMedia){
    switch(pMedia->eMediaType){

        case MEDIATYPE_LOCAL_RTP_INFO:
             LOG(LOG_API, GetName(), "  MEDIATYPE_LOCAL_RTP_INFO");
             DumpPortInfo(&pMedia->mediaInfo.PortInfo);
             break;

        case MEDIATYPE_REMOTE_RTP_INFO:
             LOG(LOG_API, GetName(), "  MEDIATYPE_REMOTE_RTP_INFO");
             DumpPortInfo(&pMedia->mediaInfo.PortInfo);
             break;

        case MEDIATYPE_LOCAL_RTCP_INFO:
             LOG(LOG_API, GetName(), "  MEDIATYPE_LOCAL_RTCP_INFO");
             DumpPortInfo(&pMedia->mediaInfo.PortInfo);
             break;

        case MEDIATYPE_REMOTE_RTCP_INFO:
             LOG(LOG_API, GetName(), "  MEDIATYPE_REMOTE_RTCP_INFO");
             DumpPortInfo(&pMedia->mediaInfo.PortInfo);
             break;

        case MEDIATYPE_REMOTE_CODER_INFO:
             LOG(LOG_API, GetName(), "  MEDIATYPE_REMOTE_CODER_INFO");
             DumpAudioCoderInfo(&pMedia->mediaInfo.AudioCoderInfo);
             break;
        case MEDIATYPE_LOCAL_CODER_INFO:
             LOG(LOG_API, GetName(), "  MEDIATYPE_LOCAL_CODER_INFO");
             DumpAudioCoderInfo(&pMedia->mediaInfo.AudioCoderInfo);
             break;
        default:
             LOG(LOG_API, GetName(), "  MEDIATYPE: %d - not supported",pMedia->eMediaType );
             break;
    } // switch

    return;
}

//*****************************************************************************
// Purpose	: 
//    Start IP media
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CIptDevice::StartMedia(){
IPM_MEDIA_INFO MediaInfo;
unsigned int inx = 0;
  if ( Is3PCCMode() ) {
      MediaInfo.MediaData[inx].eMediaType = MEDIATYPE_LOCAL_RTP_INFO;
      MediaInfo.MediaData[inx].mediaInfo.PortInfo = m_LocalRtpPort;
      inx++;

      MediaInfo.MediaData[inx].eMediaType = MEDIATYPE_LOCAL_RTCP_INFO;
      MediaInfo.MediaData[inx].mediaInfo.PortInfo = m_LocalRtcpPort;
      inx++;

      MediaInfo.MediaData[inx].eMediaType = MEDIATYPE_REMOTE_RTP_INFO;
      MediaInfo.MediaData[inx].mediaInfo.PortInfo.unPortId = m_pSdpParser->m_MediaInfo.m_remote_port;
      str_safecopy( MediaInfo.MediaData[inx].mediaInfo.PortInfo.cIPAddress,
                   sizeof(MediaInfo.MediaData[0].mediaInfo.PortInfo.cIPAddress),
                   m_pSdpParser->m_MediaInfo.sdp_remote_ip );
      inx++;

      // This port is next to RTP port
      MediaInfo.MediaData[inx].eMediaType = MEDIATYPE_REMOTE_RTCP_INFO;
      MediaInfo.MediaData[inx].mediaInfo.PortInfo.unPortId = m_pSdpParser->m_MediaInfo.m_remote_port+1;
      str_safecopy( MediaInfo.MediaData[inx].mediaInfo.PortInfo.cIPAddress,
                   sizeof(MediaInfo.MediaData[0].mediaInfo.PortInfo.cIPAddress),
                   m_pSdpParser->m_MediaInfo.sdp_remote_ip );
      inx++;

      IPM_CODER_INFO LocalCoder;
      IPM_CODER_INFO RemoteCoder;

      LocalCoder.eCoderType = selected_coder.eCoderType;  
      LocalCoder.eFrameSize = selected_coder.eFrameSize;  
      LocalCoder.eVadEnable = selected_coder.eVadEnable ;

      LocalCoder.unFramesPerPkt = 1;

      LocalCoder.unCoderPayloadType = selected_coder.unCoderPayloadType;
      LocalCoder.unRedPayloadType = selected_coder.unRedPayloadType;

      RemoteCoder = LocalCoder;
  
      MediaInfo.MediaData[inx].eMediaType = MEDIATYPE_REMOTE_CODER_INFO;
      MediaInfo.MediaData[inx].mediaInfo.CoderInfo = RemoteCoder;
      inx++;

      MediaInfo.MediaData[inx].eMediaType = MEDIATYPE_LOCAL_CODER_INFO;
      MediaInfo.MediaData[inx].mediaInfo.CoderInfo = LocalCoder;
      inx++;

      MediaInfo.unCount = inx;
      DumpMediaInfo(&MediaInfo);

      int rc  = ipm_StartMedia(m_ntwk_srl_handle, &MediaInfo,
                              DATA_IP_TDM_BIDIRECTIONAL, EV_SYNC);


       LOG( RC(rc), m_ipm_name,
  		     "%d = ipm_StartMedia (handle=0x%x,IpmMediaInfo,DATA_IP_TDM_BIDIRECTIONAL, EV_SYNC)",
   		     rc, m_ntwk_srl_handle );

       if (rc != 0){
           process_ipml_error(m_ntwk_srl_handle, m_ipm_name);
           return false;
       } else {
           m_ipm_started = true;
       }
       SetDtmfTransfer();
  }else {
      LOG(LOG_ERR1, GetName(), "Application error: StartMedia in none 3PCC mode");
  }
 return true;
}

//*****************************************************************************
// Purpose	: 
//    Drop current call
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CIptDevice::DropCall(){
  StopMedia();
  return CNtwkDevice::DropCall();
} // End of DropCall()

//*****************************************************************************
// Purpose	: 
//    Reset line device
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CIptDevice::ResetLineDev(){
  StopMedia();
  return CNtwkDevice::ResetLineDev();
} // End of ResetLineDev()


//*****************************************************************************
// Purpose	: 
//    Answer to incoming call
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CIptDevice::AnswerCall(){

  if (Is3PCCMode() ){
      if (! StartMedia()  ){
             DropRequest();
      } else { 
          if ( !DoAcceptCall() ){
	          if (! MakeSdpMessage( IPPARM_SDP_ANSWER) ){
		           LOG( RC(-1), GetName(),
                   "MakeSdpMessage failed)");
			        return false;
              }
          }
      }
  }
  return CNtwkDevice::AnswerCall();

} // End of AnswerCall()

//*****************************************************************************
// Purpose	: 
//    Accept to incoming call
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CIptDevice::AcceptCall(){
  if (DoAcceptCall() && Is3PCCMode() ) {
	  if (! MakeSdpMessage( IPPARM_SDP_ANSWER) ){
		    LOG( RC(-1), GetName(),
            "MakeSdpMessage failed)");
			return false;
	  }
  }
  return CNtwkDevice::AcceptCall();
} // End of AcceptCall()

//*****************************************************************************
// Purpose	: 
//    Handle GCEV_REQ_MODIFY_CALL
// Parameters:	
// Returns:	
//    success flag
//*****************************************************************************
bool CIptDevice::ModifyCall(){
// Implement here responce to GCEV_REQ_MODIFY_CALL
// if ASYNC function is involved, change appropriately state machine
   return true;
}
