/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* DIALOGIC CONFIDENTIAL
*
* Copyright (C) 2006-2007 Dialogic Corporation. All Rights Reserved.
* The source code contained or described herein and all documents related
* to the source code ("Material") are owned by Dialogic Corporation or its
* suppliers or licensors. Title to the Material remains with Dialogic Corporation
* or its suppliers and licensors. The Material contains trade secrets and
* proprietary and confidential information of Dialogic or its suppliers and
* licensors. The Material is protected by worldwide copyright and trade secret
* laws and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted, distributed,
* or disclosed in any way without Dialogic's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
//***********************************************************************
//***********************************************************************
// IptDevice.cpp: implementation of the CIptDevice class.
//
//////////////////////////////////////////////////////////////////////
#include "pdl.h"

#define MAX_RETRY 3         // max retry in case of invalid pass code

        
const char * dx_get_termination(int termno){
    static NAME_TABLE term_table_stru[] = {
	    { 	"TM_NORMTERM",TM_NORMTERM   },
 	    { 	"TM_MAXDTMF", TM_MAXDTMF    },
	    { 	"TM_MAXSIL",  TM_MAXSIL		},
 	    { 	"TM_MAXNOSIL",TM_MAXNOSIL	},
 	    { 	"TM_LCOFF",   TM_LCOFF		},
 	    { 	"TM_IDDTIME", TM_IDDTIME	},
 	    { 	"TM_MAXTIME", TM_MAXTIME	},
 	    {   "TM_DIGIT",   TM_DIGIT		},
 	    { 	"TM_PATTERN", TM_PATTERN	},
 	    { 	"TM_USRSTOP", TM_USRSTOP	},
 	    { 	"TM_EOD",     TM_EOD		},
 	    { 	"TM_TONE",    TM_TONE		},
 	    { 	"TM_BARGEIN", TM_BARGEIN	},
	    { 	"TM_ERROR",   TM_ERROR		},
	    { 	"TM_MAXDATA", TM_MAXDATA	},
        { 0,0}
    };  // End of term_table[]
    static CNameTable term_table(term_table_stru);

  return term_table.find_name(termno);
} // End of dx_get_termination()

//
// Call State machine
//
enum {
// Initialization
   S_WAIT_UNBLOCKED,   // OpenEx received, wait Unblocked  
   S_WAIT_OPENEX,      // Unblocked received, wait OpenEx

   S_RESET,            // ...ResetLineDev
   S_RESET_EXIT,       // ...ResetLineDev, exit request

// S_RELAX

// Call
   S_ROUTE,            // In process of routing dx to ntwk device
   S_ROUTE_EXIT,       // In process of routing dx, exit request

   S_ANSWER,           // In process of Answering the call 
   S_ANSWER_EXIT,      // In process of Answering the call (exit)

   S_ACCEPT,           // In process of Accepting the call 
   S_ACCEPT_EXIT,      // In process of Accepting the call (exit)
                      
   S_DROP,             // ...Drop call
   S_RELEASE,          // ...Release Call

   S_CHECK_DNIS,       // ...check if DNIS is associated with conference
   S_CHECK_DNIS_EXIT,  // ...check if DNIS is associated with conference

// Conversation
   S_PLAYGREET,        // ...Play Greeting
   S_PLAYBYE,          // ...Play bye
   S_PLAYCALLLATER,    // ...Play Call Later 

   S_GETDIGITS,        // ...Get digits

   S_STOP_DX,          // ...Cancel dx (exiting any voice function)
   S_CHECK_CNF,        // ...Check conference password

// UnListen 
   S_UNLISTEN,         // .. UnListen  from dx device

// Conference
   S_OPENPARTY,        // ...cnf_PartyOpen  
   S_OPENPARTY_EXIT,   // ...cnf_PartyOpen  exiting

   S_ADDPARTY,         // ...cnf_addParty
   S_ADDPARTY_EXIT,    // ...cnf_addParty exiting

   S_SETPARTYATTR,     // ... set party attributes
   S_SETPARTYATTR_EXIT,// ... set party attributes - exit

   S_DEVCONNECT1,      // ...dev_Connect ( first event)
   S_DEVCONNECT1_EXIT, // ...dev_Connect ( first event) exit

   S_DEVCONNECT2,      // ...dev_Connect ( second event)
   S_DEVCONNECT2_EXIT, // ...dev_Connect ( second event) exit

// S_RELAXCNF
   
   S_REMOVEPARTY,       // ...cnf_RemoveFromConference
   S_DEVDISCONNECTNTWK, // dev_Disconnect
   S_DEVDISCONNECTCNF,  // dev_Disconnect cnf party
};

//*****************************************************************************
// Purpose	: 
//    return state name for given value
// Parameters:	
//    state
// Returns:	
//    name 
//*****************************************************************************
const char * CNtwkDevice::state_name(int state){
    static NAME_TABLE state_tbl[] ={
        {"S_WAIT_UNBLOCKED",   S_WAIT_UNBLOCKED  },
        {"S_WAIT_OPENEX",      S_WAIT_OPENEX     },

        {"S_RESET",            S_RESET           },
        {"S_RESET_EXIT",       S_RESET_EXIT      },

        {"S_ANSWER",           S_ANSWER          },
        {"S_ANSWER_EXIT",      S_ANSWER_EXIT     },

        {"S_ACCEPT",           S_ACCEPT          },
        {"S_ACCEPT_EXIT",      S_ACCEPT_EXIT     },

        {"S_ROUTE",            S_ROUTE           },
        {"S_ROUTE_EXIT",       S_ROUTE_EXIT      },

        {"S_CHECK_DNIS",       S_CHECK_DNIS      },
        {"S_CHECK_DNIS_EXIT",  S_CHECK_DNIS_EXIT },

        {"S_DROP",             S_DROP            },
        {"S_RELEASE",          S_RELEASE         },

        {"S_PLAYGREET",        S_PLAYGREET       },
        {"S_PLAYBYE",          S_PLAYBYE         },
        {"S_PLAYCALLLATER",    S_PLAYCALLLATER   },

        {"S_GETDIGITS",        S_GETDIGITS       },
        {"S_STOP_DX",          S_STOP_DX         },

        {"S_CHECK_CNF",        S_CHECK_CNF       },
        {"S_UNLISTEN",         S_UNLISTEN        },

        {"S_OPENPARTY",        S_OPENPARTY       },
        {"S_OPENPARTY_EXIT",   S_OPENPARTY_EXIT  },

        {"S_ADDPARTY",         S_ADDPARTY        },
        {"S_ADDPARTY_EXIT",    S_ADDPARTY_EXIT   },

        {"S_SETPARTYATTR",      S_SETPARTYATTR       },
        {"S_SETPARTYATTR_EXIT", S_SETPARTYATTR_EXIT  },

        {"S_DEVCONNECT1",      S_DEVCONNECT1         },
        {"S_DEVCONNECT1_EXIT", S_DEVCONNECT1_EXIT    },

        {"S_DEVCONNECT2",      S_DEVCONNECT2         },
        {"S_DEVCONNECT2_EXIT", S_DEVCONNECT2_EXIT    },

        {"S_REMOVEPARTY",       S_REMOVEPARTY        },
        {"S_DEVDISCONNECTNTWK", S_DEVDISCONNECTNTWK  },
        {"S_DEVDISCONNECTCNF",  S_DEVDISCONNECTCNF   },

   
        {0, 0}
    };   // End of state_tbl[]
    const char *name;
    if (! str_findname(state, &name, state_tbl) ){
          name = common_state_name(state);
    }
  return name;
}   // End of state_name()

// Actions
// Derived:
// A_NONE                 // do nothing, ignore exit request
// A_RELAX                // relax (outside conference, ex. wait a call)
// A_RELAXCNF             // relax (in conference)
// A_CLOSE                // close this device
// A_ERROR                // Indicate state machine error
// A_EXIT                 // Exit - advance to appropriate .._exit state
// A_GIVEUP				  // Error while closing, advance to final state

enum { 
    A_FATAL,                 // Fatal error, close & request exit
    A_MOVE_UNBLOCKED,        // do nothing,  Advance to s_wait_unblocked
    A_MOVE_OPENEX,           // do nothing,  Advance to s_wait_openex
    A_MOVE_OPEN,             // do nothing,  Advance to s_open
    A_MOVE,                  // do nothing,  Advance to next

    A_ROUTE,                 // Route  
    A_RESET,                 // Reset line dev  

// A_RELAX is wait call

    A_ANSWER,                // Answer the call
    A_ACCEPT,                // Accept the call

    A_DROP,                  // Drop Call
    A_RELEASE,               // Release call

    A_PLAYGREET,             // Play greeting
    A_PLAYINVALID,           // Play invalid password 
    A_PLAYBYE,               // Play bye file  

    A_CHECK_DNIS,            // Check DNIS
    A_GETDIGITS,             // Get Digits

    A_STOP_DX,               // Stop dx 

    A_CHECK_CNF,             // Check conference password

    A_UNLISTEN,              // UnListen
    A_OPENPARTY,             // cnf_OpenParty 

    A_ADDPARTY,              // cnf_AddParty 
    A_DEVCONNECT,            // dev_Connect 
    A_MOVE_DEVCONNECT2_EXIT, // do nothing, Advance to s_devdisconnect2_exit

    A_DEVDISCONNECTNTWK,     // dev_DisConnect 
    A_DEVDISCONNECTCNF,      // dev_DisConnect 
    A_REMOVEPARTY,           // cnf_RemoveParty 
    A_CLOSEPARTY,            // cnf_CloseParty 
    A_SETPARTYATTR,
    A_MODIFYCALL,            // ReqModifyCall
};


//*****************************************************************************
// Purpose	: 
//    return action name for given value
// Parameters:	
//    [in] action code
// Returns:	
//    name 
//*****************************************************************************
const char * CNtwkDevice::action_name(int action){
    static NAME_TABLE ac_tbl[] = {

        { "A_FATAL",            A_FATAL           },
        { "A_MOVE_UNBLOCKED",   A_MOVE_UNBLOCKED  },
        { "A_MOVE_OPENEX",      A_MOVE_OPENEX     },
        { "A_MOVE_OPEN",        A_MOVE_OPEN       },
        { "A_ROUTE",            A_ROUTE           }, 
        { "A_RESET",            A_RESET           }, 

        { "A_ANSWER",           A_ANSWER          }, 
        { "A_ACCEPT",           A_ACCEPT          }, 

        { "A_CHECK_DNIS",       A_CHECK_DNIS      }, 
   
        { "A_DROP",             A_DROP            }, 
        { "A_RELEASE",          A_RELEASE         },
    
        { "A_PLAYGREET",        A_PLAYGREET       }, 
        { "A_PLAYINVALID",      A_PLAYINVALID     }, 
        { "A_PLAYBYE",          A_PLAYBYE         }, 

        { "A_GETDIGITS",        A_GETDIGITS       }, 
        { "A_STOP_DX",          A_STOP_DX         }, 

        { "A_CHECK_CNF",        A_CHECK_CNF       }, 
        { "A_UNLISTEN",         A_UNLISTEN       }, 

        { "A_OPENPARTY",        A_OPENPARTY       }, 
        { "A_ADDPARTY",         A_ADDPARTY        }, 
        { "A_DEVCONNECT",       A_DEVCONNECT      },    
        { "A_MOVE_DEVCONNECT2_EXIT",   A_MOVE_DEVCONNECT2_EXIT   },    

        { "A_DEVDISCONNECTNTWK", A_DEVDISCONNECTNTWK }, 
        { "A_DEVDISCONNECTCNF",  A_DEVDISCONNECTCNF  }, 
        { "A_REMOVEPARTY",       A_REMOVEPARTY       }, 
        { "A_CLOSEPARTY",        A_CLOSEPARTY        }, 
        { "A_SETPARTYATTR",      A_SETPARTYATTR      }, 
        { "A_MODIFYCALL",        A_MODIFYCALL        },
       { 0,0 }     
    };   // End of ac_tbl[]
 const char *name;
    if (! str_findname(action, &name, ac_tbl) ){
          name = common_action_name(action);
    }
  return name;
}  // End of action_name()

/////////////////////////////////////////////////////////////
// State machine
//===========================================================
static ACTION_STRU ntwk_action_stru[] = {
// Example:
//  { S_ANSWER,       A_PLAYGREET,    S_PLAYGREET,    A_DROP        },
//  In state S_ANSWER, Action A_PLAYGREET was executed
//     Then advance to state S_PLAYGREET if A_PLAYGREET was successful
//       or execute A_DROP as a recover procedure if A_PLAYGREET failed
//       How to continue after A_DROP in state S_ANSWER, see same table:
//  { S_ANY,          A_DROP,         S_DROP,         S_CLOSE },
//   Note:
//     In any state, there is no recover on recover.
//     if recover procedure fails, close and exit.

//  current_state   action           state_if_pass   action_if_fail 

    { S_OPEN,           A_MOVE_OPENEX,    S_WAIT_OPENEX,    A_NONE    },
    { S_OPEN,           A_EXIT,           S_OPEN_EXIT,      A_NONE    },

    { S_WAIT_UNBLOCKED, A_RESET,          S_RESET,          A_CLOSE   },
    { S_WAIT_OPENEX,    A_RESET,          S_RESET,          A_CLOSE   },
    { S_WAIT_OPENEX,    A_EXIT,           S_OPEN_EXIT,      A_NONE    },
    { S_WAIT_OPENEX,    A_MOVE_OPEN,      S_OPEN,           A_NONE    },

// ResetLineDev
    { S_RESET,          A_EXIT,           S_RESET_EXIT,     A_NONE    },
    { S_RESET,          A_RELAX,          S_RELAX,          A_CLOSE   },

// Incoming call 
/*
// --- This is for ANSWER call only
    { S_RELAX,          A_ANSWER,         S_ANSWER,         A_DROP    },


// --- end This is for ANSWER call  
*/

// --- This is for ACCEPT and ANSWER the call
// Incoming call 
    { S_RELAX,          A_ACCEPT,         S_ACCEPT,         A_DROP    },

    { S_ACCEPT,         A_ANSWER,         S_ANSWER,         A_DROP    },
    { S_ACCEPT,         A_EXIT,           S_ACCEPT_EXIT,    A_NONE    },
// --- End This is for ACCEPT and ANSWER the call


    { S_ANSWER,         A_ROUTE,          S_ROUTE,          A_DROP    },
    { S_ANSWER,         A_EXIT,           S_ANSWER_EXIT,    A_NONE    },

// Route 
    { S_ROUTE,          A_EXIT,           S_ROUTE_EXIT,     A_NONE    },
    { S_ROUTE,          A_CHECK_DNIS,     S_CHECK_DNIS,     A_DROP    },

// Check dnis 
    { S_CHECK_DNIS,     A_PLAYGREET,      S_PLAYGREET,      A_DROP    },
    { S_CHECK_DNIS,     A_OPENPARTY,      S_OPENPARTY,      A_PLAYBYE },

// Play/get
    { S_PLAYGREET,      A_GETDIGITS,      S_GETDIGITS,      A_DROP    },
    { S_PLAYGREET,      A_GETDIGITS,      S_GETDIGITS,      A_DROP    },
    { S_ANY,            A_PLAYBYE,        S_PLAYBYE,        A_DROP    },

    { S_GETDIGITS,      A_GETDIGITS,      S_GETDIGITS,      A_DROP    },
    { S_GETDIGITS,      A_CHECK_CNF,      S_CHECK_CNF,      A_DROP    },

    { S_CHECK_CNF,      A_PLAYGREET,      S_PLAYGREET,      A_DROP    },
    { S_CHECK_CNF,      A_PLAYINVALID,    S_PLAYGREET,      A_DROP    },
//Unlisten
    { S_CHECK_CNF,      A_UNLISTEN,       S_UNLISTEN,       A_DROP    },
     
// Add party
    { S_UNLISTEN,       A_OPENPARTY,      S_OPENPARTY,      A_PLAYBYE },

//Conference
    { S_OPENPARTY,      A_EXIT,           S_OPENPARTY_EXIT, A_NONE    },
  // case 2:
    { S_OPENPARTY,      A_ADDPARTY,       S_ADDPARTY,       A_CLOSEPARTY },
  // case 1:
    { S_OPENPARTY,      A_DEVCONNECT,     S_DEVCONNECT1,    A_CLOSEPARTY }, 

    { S_DEVCONNECT2,    A_ADDPARTY,       S_ADDPARTY,       A_CLOSEPARTY  },

    
// case 1: DevConnect, AddParty, SetPartyAttribute
//---------------------------------------------------------------------
    

// case 2: Add party, SetPartyAttribute, DevConnect
//---------------------------------------------------------------------
// Add party
    { S_ADDPARTY,       A_EXIT,              S_ADDPARTY_EXIT,  A_NONE        },
    { S_ADDPARTY,       A_SETPARTYATTR,      S_SETPARTYATTR,   A_REMOVEPARTY },
    { S_ADDPARTY,       A_REMOVEPARTY,       S_REMOVEPARTY,    A_CLOSEPARTY  },
    { S_ADDPARTY_EXIT,  A_REMOVEPARTY,       S_REMOVEPARTY,    A_CLOSEPARTY  },

// Set party attribute
    { S_SETPARTYATTR,  A_EXIT,            S_SETPARTYATTR_EXIT, A_NONE  },
// case 2
    { S_SETPARTYATTR,  A_DEVCONNECT,      S_DEVCONNECT1,       A_CLOSEPARTY    },

// case 1
    { S_SETPARTYATTR,   A_RELAXCNF,       S_RELAXCNF,          A_NONE  },

    { S_SETPARTYATTR,  A_REMOVEPARTY,     S_REMOVEPARTY,       A_CLOSEPARTY    },
    { S_SETPARTYATTR_EXIT,  A_REMOVEPARTY,S_REMOVEPARTY,       A_CLOSEPARTY    },


// Dev connect
    { S_DEVCONNECT1,    A_EXIT,           S_DEVCONNECT1_EXIT,         A_NONE },
    { S_DEVCONNECT1,    A_MOVE,           S_DEVCONNECT2,              A_NONE  },
    { S_DEVCONNECT1,    A_MOVE_DEVCONNECT2_EXIT, S_DEVCONNECT2_EXIT,  A_NONE  },

    { S_DEVCONNECT1_EXIT,A_MOVE,          S_DEVCONNECT2_EXIT,         A_NONE  },


    { S_DEVCONNECT2,    A_EXIT,           S_DEVCONNECT2_EXIT,         A_NONE },
    { S_DEVCONNECT2,    A_DEVDISCONNECTNTWK, S_DEVDISCONNECTNTWK,     A_REMOVEPARTY  },
    { S_DEVCONNECT2,    A_RELAXCNF,       S_RELAXCNF,                 A_NONE  },
    { S_DEVCONNECT2,    A_REMOVEPARTY,    S_REMOVEPARTY,              A_CLOSEPARTY },

    { S_DEVCONNECT2_EXIT, A_DEVDISCONNECTNTWK, S_DEVDISCONNECTNTWK,    A_REMOVEPARTY  },
    { S_DEVCONNECT2_EXIT, A_REMOVEPARTY,       S_REMOVEPARTY,          A_CLOSEPARTY },
//---------------------------------------------------------------------



// In conference
    { S_RELAXCNF,       A_DEVDISCONNECTNTWK, S_DEVDISCONNECTNTWK, A_DEVDISCONNECTCNF},
    { S_RELAXCNF,       A_DEVDISCONNECTCNF,  S_DEVDISCONNECTCNF,  A_REMOVEPARTY },
    { S_RELAXCNF,       A_REMOVEPARTY,       S_REMOVEPARTY,       A_CLOSEPARTY },
    { S_RELAXCNF,       A_MODIFYCALL,        S_RELAXCNF,          A_NONE },

    { S_DEVDISCONNECTNTWK, A_DEVDISCONNECTCNF, S_DEVDISCONNECTCNF, A_REMOVEPARTY },
    { S_DEVDISCONNECTNTWK, A_REMOVEPARTY,      S_REMOVEPARTY,      A_CLOSEPARTY },
    { S_DEVDISCONNECTCNF,  A_REMOVEPARTY,      S_REMOVEPARTY,      A_CLOSEPARTY },


// Drop & release
    { S_DROP,           A_RELEASE,        S_RELEASE,        A_FATAL   },
    { S_RELEASE,        A_RELAX,          S_RELAX,          A_CLOSE   },

// State machine error (missing case to handle specified action)
    { S_ANY,          A_ERROR,            S_SAME,           A_CLOSE   },

// 'Any state' actions:
    { S_ANY,            A_CLOSE,          S_FINAL,          A_GIVEUP  },
    { S_ANY,            A_FATAL,          S_FINAL,          A_GIVEUP  },
    { S_ANY,            A_GIVEUP,         S_FINAL,          A_NONE    },
    { S_ANY,            A_NONE,           S_SAME,           A_NONE    },

    { S_ANY,            A_MOVE_UNBLOCKED, S_WAIT_UNBLOCKED, A_NONE    },
    { S_ANY,            A_RESET,          S_RESET,          A_FATAL   },

    { S_ANY,            A_DROP,           S_DROP,           A_FATAL   },

    // this is close + drop
    { S_ANY,            A_CLOSEPARTY,     S_DROP,           A_FATAL   },
    
    { S_ANY,            A_STOP_DX,        S_STOP_DX,        A_DROP    },


// Not part of state machine, indicates last line
    { S_END,   0,    0,     0  }   

};   // End of ntwk_action_stru[]

//--------------------------------------------
// The state machine definition
//--------------------------------------------
static EVENT_STRU ntwk_event_stru[] = {
// Example:
// { S_ANSWER,         GCEV_ANSWERED,           A_PLAYGREET },
//  In state S_ANSWER, GCEV_ANSWERED is received.
//     Then proceed with action A_PLAYGREET)
//

//  current_state       event                      action           
// After Open
    { S_OPEN,           GCEV_OPENEX,               A_MOVE_UNBLOCKED     },
    { S_OPEN,           GCEV_OPENEX_FAIL,          A_CLOSE     },
    { S_OPEN,           GCEV_UNBLOCKED,            A_MOVE_OPENEX        },
    { S_OPEN,           GCEV_BLOCKED,              A_NONE      },
    { S_OPEN,           USREV_EXIT_REQUEST,        A_EXIT      },
    { S_OPEN,           USREV_TIMEOUT,             A_CLOSE     },

    { S_OPEN_EXIT,      GCEV_OPENEX,               A_CLOSE,    },
    { S_OPEN_EXIT,      GCEV_OPENEX_FAIL,          A_CLOSE,    },
    { S_OPEN_EXIT,      USREV_TIMEOUT,             A_CLOSE,    },
    { S_OPEN_EXIT,      GCEV_BLOCKED,              A_NONE,     },

    { S_WAIT_UNBLOCKED, USREV_EXIT_REQUEST,        A_CLOSE,    },
    { S_WAIT_UNBLOCKED, GCEV_BLOCKED,              A_NONE,     },
    { S_WAIT_UNBLOCKED, GCEV_UNBLOCKED,            A_RESET     },
    { S_WAIT_UNBLOCKED, USREV_TIMEOUT,             A_CLOSE     },

    { S_WAIT_OPENEX,    GCEV_OPENEX,               A_RESET     },
    { S_WAIT_OPENEX,    GCEV_OPENEX_FAIL,          A_CLOSE     },
    { S_WAIT_OPENEX,    USREV_EXIT_REQUEST,        A_EXIT      },
    { S_WAIT_OPENEX,    GCEV_BLOCKED,              A_MOVE_OPEN },

// After ResetLineDev
    { S_RESET,          GCEV_BLOCKED,              A_MOVE_UNBLOCKED },
    { S_RESET,          USREV_EXIT_REQUEST,        A_EXIT      },
    { S_RESET,          GCEV_TASKFAIL,             A_CLOSE     },
    { S_RESET,          USREV_TIMEOUT,             A_CLOSE     },
    { S_RESET,          GCEV_RESETLINEDEV,         A_RELAX     },

    { S_RESET_EXIT,     GCEV_TASKFAIL,             A_CLOSE     },
    { S_RESET_EXIT,     USREV_TIMEOUT,             A_CLOSE     },
    { S_RESET_EXIT,     GCEV_RESETLINEDEV,         A_CLOSE     },

// Incoming call
/*
// This is for accept only
    { S_RELAX,          GCEV_OFFERED,              A_ANSWER    },
// end this is for accept only
*/

// Accept and Answer
    { S_RELAX,          GCEV_OFFERED,              A_ACCEPT    },
    
    { S_ACCEPT,         GCEV_BLOCKED,              A_EXIT      },
    { S_ACCEPT,         USREV_EXIT_REQUEST,        A_DROP      },
    { S_ACCEPT,         GCEV_DISCONNECTED,         A_DROP      },
    { S_ACCEPT,         GCEV_TASKFAIL,             A_RESET     },
    { S_ACCEPT,         USREV_TIMEOUT,             A_RESET     },
    { S_ACCEPT,         GCEV_ACCEPT,               A_ANSWER    },
    { S_ACCEPT,         USREV_SKIP_ACCEPT,         A_ANSWER    },


    { S_ACCEPT_EXIT,    GCEV_TASKFAIL,             A_DROP     },
    { S_ACCEPT_EXIT,    USREV_TIMEOUT,             A_DROP     },
    { S_ACCEPT_EXIT,    GCEV_ANSWERED,             A_DROP     },

// end Accept and answer

    { S_RELAX,          USREV_EXIT_REQUEST,        A_CLOSE     },
    
    { S_ANSWER,         GCEV_BLOCKED,              A_EXIT      },
    { S_ANSWER,         USREV_EXIT_REQUEST,        A_DROP      },
    { S_ANSWER,         GCEV_DISCONNECTED,         A_DROP      },
    { S_ANSWER,         GCEV_TASKFAIL,             A_RESET     },
    { S_ANSWER,         USREV_TIMEOUT,             A_RESET     },
    { S_ANSWER,         GCEV_ANSWERED,             A_ROUTE     },


    { S_ANSWER_EXIT,    GCEV_TASKFAIL,             A_DROP     },
    { S_ANSWER_EXIT,    USREV_TIMEOUT,             A_DROP     },
    { S_ANSWER_EXIT,    GCEV_ANSWERED,             A_DROP     },


// Route
    { S_ROUTE,          GCEV_BLOCKED,              A_EXIT      },
    { S_ROUTE,          USREV_EXIT_REQUEST,        A_EXIT      },
    { S_ROUTE,          GCEV_DISCONNECTED,         A_EXIT      },

    { S_ROUTE,          USREV_TIMEOUT,             A_DROP      },
    { S_ROUTE,          IPMEV_ERROR,               A_DROP      },
    { S_ROUTE,          GCEV_TASKFAIL,             A_DROP      },

    { S_ROUTE,          GCEV_LISTEN,               A_CHECK_DNIS },
    { S_ROUTE,          IPMEV_LISTEN,              A_CHECK_DNIS },
    { S_ROUTE,          USREV_IPML_LISTEN,         A_CHECK_DNIS },

    { S_ROUTE_EXIT,     GCEV_TASKFAIL,             A_DROP       },
    { S_ROUTE_EXIT,     USREV_TIMEOUT,             A_DROP       },
    { S_ROUTE_EXIT,     GCEV_LISTEN,               A_DROP       },
    { S_ROUTE_EXIT,     IPMEV_LISTEN,              A_DROP       },
    { S_ROUTE_EXIT,     USREV_IPML_LISTEN,         A_DROP       },
    { S_ROUTE_EXIT,     IPMEV_ERROR,               A_DROP       },

// Check dnis
// check pass code
    { S_CHECK_DNIS,     GCEV_BLOCKED,              A_DROP        },
    { S_CHECK_DNIS,     USREV_EXIT_REQUEST,        A_DROP        },
    { S_CHECK_DNIS,     GCEV_DISCONNECTED,         A_DROP        },
    { S_CHECK_DNIS,     USREV_CHECK_FAIL,          A_PLAYGREET   },
    { S_CHECK_DNIS,     USREV_CHECK_PASS,          A_OPENPARTY   },

// Play greeting
    { S_PLAYGREET,      GCEV_BLOCKED,              A_STOP_DX     },
    { S_PLAYGREET,      USREV_EXIT_REQUEST,        A_STOP_DX     },
    { S_PLAYGREET,      GCEV_DISCONNECTED,         A_STOP_DX     },
    { S_PLAYGREET,      TDX_ERROR,                 A_DROP        },
    { S_PLAYGREET,      USREV_TIMEOUT,             A_DROP        },
    { S_PLAYGREET,      TDX_PLAY,                  A_GETDIGITS   },


    { S_PLAYBYE,        GCEV_BLOCKED,              A_STOP_DX     },
    { S_PLAYBYE,        USREV_EXIT_REQUEST,        A_STOP_DX     },
    { S_PLAYBYE,        GCEV_DISCONNECTED,         A_STOP_DX     },
    { S_PLAYBYE,        TDX_ERROR,                 A_DROP        },
    { S_PLAYBYE,        USREV_TIMEOUT,             A_DROP        },
    { S_PLAYBYE,        TDX_PLAY,                  A_DROP        },


// Stop play/getdigits
    { S_STOP_DX,        GCEV_BLOCKED,              A_STOP_DX     },
    { S_STOP_DX,        USREV_EXIT_REQUEST,        A_STOP_DX     },
    { S_STOP_DX,        GCEV_DISCONNECTED,         A_STOP_DX     },
    { S_STOP_DX,        TDX_ERROR,                 A_DROP        },
    { S_STOP_DX,        USREV_TIMEOUT,             A_DROP        },
    { S_STOP_DX,        TDX_PLAY,                  A_DROP        },
    { S_STOP_DX,        TDX_GETDIG,                A_DROP        },


// get digits
    { S_GETDIGITS,      GCEV_BLOCKED,              A_STOP_DX     },
    { S_GETDIGITS,      USREV_EXIT_REQUEST,        A_STOP_DX     },
    { S_GETDIGITS,      GCEV_DISCONNECTED,         A_STOP_DX     },
    { S_GETDIGITS,      TDX_ERROR,                 A_DROP        },
    { S_GETDIGITS,      USREV_TIMEOUT,             A_DROP        },
    { S_GETDIGITS,      TDX_GETDIG,                A_CHECK_CNF   },

// check pass code
    { S_CHECK_CNF,      GCEV_BLOCKED,              A_DROP        },
    { S_CHECK_CNF,      USREV_EXIT_REQUEST,        A_DROP        },
    { S_CHECK_CNF,      GCEV_DISCONNECTED,         A_DROP        },
    { S_CHECK_CNF,      USREV_CHECK_FAIL,          A_PLAYBYE     },
    { S_CHECK_CNF,      USREV_CHECK_RETRY,         A_PLAYINVALID },
    { S_CHECK_CNF,      USREV_CHECK_PASS,          A_UNLISTEN    },


    { S_UNLISTEN,       GCEV_BLOCKED,              A_DROP        },
    { S_UNLISTEN,       USREV_EXIT_REQUEST,        A_DROP        },
    { S_UNLISTEN,       GCEV_DISCONNECTED,         A_DROP        },
    { S_UNLISTEN,       GCEV_UNLISTEN,             A_OPENPARTY    },
    { S_UNLISTEN,       IPMEV_UNLISTEN,            A_OPENPARTY    },

// Conference
// open party
    { S_OPENPARTY,      GCEV_BLOCKED,              A_EXIT        },
    { S_OPENPARTY,      USREV_EXIT_REQUEST,        A_EXIT        },
    { S_OPENPARTY,      GCEV_DISCONNECTED,         A_EXIT        },
    { S_OPENPARTY,      CNFEV_OPEN_PARTY_FAIL,     A_CLOSEPARTY  },
    { S_OPENPARTY,      USREV_TIMEOUT,             A_CLOSEPARTY  },

    // case 2: first add party
//  { S_OPENPARTY,      CNFEV_OPEN_PARTY,          A_ADDPARTY    },

    // case 1 first dev connect
    { S_OPENPARTY,      CNFEV_OPEN_PARTY,          A_DEVCONNECT  },

    { S_OPENPARTY_EXIT, CNFEV_OPEN_PARTY_FAIL,     A_CLOSEPARTY  },
    { S_OPENPARTY_EXIT, USREV_TIMEOUT,             A_CLOSEPARTY  },
    { S_OPENPARTY_EXIT, CNFEV_OPEN_PARTY,          A_CLOSEPARTY  },


// ----------------- DevConnect, AddParty, SetAttribute
// Dev connect
    { S_DEVCONNECT1,    GCEV_BLOCKED,              A_EXIT        },
    { S_DEVCONNECT1,    USREV_EXIT_REQUEST,        A_EXIT        },
    { S_DEVCONNECT1,    GCEV_DISCONNECTED,         A_EXIT        },
    { S_DEVCONNECT1,    DMEV_CONNECT_FAIL,         A_MOVE_DEVCONNECT2_EXIT },
    { S_DEVCONNECT1,    USREV_TIMEOUT,             A_MOVE_DEVCONNECT2_EXIT },
    { S_DEVCONNECT1,    DMEV_CONNECT,              A_MOVE        },

    { S_DEVCONNECT1_EXIT, DMEV_CONNECT,            A_NONE        },
    { S_DEVCONNECT1_EXIT, DMEV_CONNECT_FAIL,       A_MOVE        },
    { S_DEVCONNECT1_EXIT, USREV_TIMEOUT,           A_MOVE        },


// Dev connect
    { S_DEVCONNECT2,    GCEV_BLOCKED,              A_EXIT        },
    { S_DEVCONNECT2,    USREV_EXIT_REQUEST,        A_EXIT        },
    { S_DEVCONNECT2,    GCEV_DISCONNECTED,         A_EXIT        },
    { S_DEVCONNECT2,    DMEV_CONNECT_FAIL,         A_DEVDISCONNECTNTWK },
    { S_DEVCONNECT2,    USREV_TIMEOUT,             A_DEVDISCONNECTNTWK },
//  case 2
//  { S_DEVCONNECT2,    DMEV_CONNECT,              A_RELAXCNF    },
//  case 1
    { S_DEVCONNECT2,    DMEV_CONNECT,              A_ADDPARTY    },


    { S_DEVCONNECT2_EXIT, DMEV_CONNECT,            A_DEVDISCONNECTNTWK        },
    { S_DEVCONNECT2_EXIT, DMEV_CONNECT_FAIL,       A_DEVDISCONNECTNTWK        },
    { S_DEVCONNECT2_EXIT, USREV_TIMEOUT,           A_DEVDISCONNECTNTWK        },

    { S_ADDPARTY,       GCEV_BLOCKED,              A_EXIT        },
    { S_ADDPARTY,       USREV_EXIT_REQUEST,        A_EXIT        },
    { S_ADDPARTY,       GCEV_DISCONNECTED,         A_EXIT        },
    { S_ADDPARTY,       CNFEV_ADD_PARTY_FAIL,      A_CLOSEPARTY  },
    { S_ADDPARTY,       USREV_TIMEOUT,             A_CLOSEPARTY  },
    { S_ADDPARTY,       CNFEV_ADD_PARTY,           A_SETPARTYATTR  },

    { S_ADDPARTY_EXIT,  CNFEV_ADD_PARTY_FAIL,      A_CLOSEPARTY  },
    { S_ADDPARTY_EXIT,  USREV_TIMEOUT,             A_CLOSEPARTY  },
    { S_ADDPARTY_EXIT,  CNFEV_ADD_PARTY,           A_REMOVEPARTY },


// set party attribute
    { S_SETPARTYATTR,      GCEV_BLOCKED,              A_EXIT        },
    { S_SETPARTYATTR,      USREV_EXIT_REQUEST,        A_EXIT        },
    { S_SETPARTYATTR,      GCEV_DISCONNECTED,         A_EXIT        },
    { S_SETPARTYATTR,      CNFEV_SET_ATTRIBUTE_FAIL,  A_CLOSEPARTY  },
    { S_SETPARTYATTR,      USREV_TIMEOUT,             A_CLOSEPARTY  },
    { S_SETPARTYATTR,      CNFEV_SET_ATTRIBUTE,       A_RELAXCNF    },

    { S_SETPARTYATTR_EXIT, CNFEV_SET_ATTRIBUTE_FAIL,     A_CLOSEPARTY  },
    { S_SETPARTYATTR_EXIT, USREV_TIMEOUT,                A_CLOSEPARTY  },
    { S_SETPARTYATTR_EXIT, CNFEV_SET_ATTRIBUTE,          A_CLOSEPARTY  },


    
/*
// ----------------- ADD, SetAttribute, DevConnect
// Add party
    { S_ADDPARTY,       GCEV_BLOCKED,              A_EXIT        },
    { S_ADDPARTY,       USREV_EXIT_REQUEST,        A_EXIT        },
    { S_ADDPARTY,       GCEV_DISCONNECTED,         A_EXIT        },
    { S_ADDPARTY,       CNFEV_ADD_PARTY_FAIL,      A_CLOSEPARTY  },
    { S_ADDPARTY,       USREV_TIMEOUT,             A_CLOSEPARTY  },
    { S_ADDPARTY,       CNFEV_ADD_PARTY,           A_SETPARTYATTR  },

    { S_ADDPARTY_EXIT,  CNFEV_ADD_PARTY_FAIL,      A_CLOSEPARTY  },
    { S_ADDPARTY_EXIT,  USREV_TIMEOUT,             A_CLOSEPARTY  },
    { S_ADDPARTY_EXIT,  CNFEV_ADD_PARTY,           A_REMOVEPARTY },


// set party attribute
    { S_SETPARTYATTR,      GCEV_BLOCKED,              A_EXIT        },
    { S_SETPARTYATTR,      USREV_EXIT_REQUEST,        A_EXIT        },
    { S_SETPARTYATTR,      GCEV_DISCONNECTED,         A_EXIT        },
    { S_SETPARTYATTR,      CNFEV_SET_ATTRIBUTE_FAIL,  A_CLOSEPARTY  },
    { S_SETPARTYATTR,      USREV_TIMEOUT,             A_CLOSEPARTY  },
    { S_SETPARTYATTR,      CNFEV_SET_ATTRIBUTE,       A_DEVCONNECT   },

    { S_SETPARTYATTR_EXIT, CNFEV_SET_ATTRIBUTE_FAIL,     A_CLOSEPARTY  },
    { S_SETPARTYATTR_EXIT, USREV_TIMEOUT,                A_CLOSEPARTY  },
    { S_SETPARTYATTR_EXIT, CNFEV_SET_ATTRIBUTE,          A_CLOSEPARTY  },



// Dev connect
    { S_DEVCONNECT1,    GCEV_BLOCKED,              A_EXIT        },
    { S_DEVCONNECT1,    USREV_EXIT_REQUEST,        A_EXIT        },
    { S_DEVCONNECT1,    GCEV_DISCONNECTED,         A_EXIT        },
    { S_DEVCONNECT1,    DMEV_CONNECT_FAIL,         A_MOVE_DEVCONNECT2_EXIT },
    { S_DEVCONNECT1,    USREV_TIMEOUT,             A_MOVE_DEVCONNECT2_EXIT },
    { S_DEVCONNECT1,    DMEV_CONNECT,              A_MOVE        },

    { S_DEVCONNECT1_EXIT, DMEV_CONNECT,            A_NONE        },
    { S_DEVCONNECT1_EXIT, DMEV_CONNECT_FAIL,       A_MOVE        },
    { S_DEVCONNECT1_EXIT, USREV_TIMEOUT,           A_MOVE        },


// Dev connect
    { S_DEVCONNECT2,    GCEV_BLOCKED,              A_EXIT        },
    { S_DEVCONNECT2,    USREV_EXIT_REQUEST,        A_EXIT        },
    { S_DEVCONNECT2,    GCEV_DISCONNECTED,         A_EXIT        },
    { S_DEVCONNECT2,    DMEV_CONNECT_FAIL,         A_DEVDISCONNECTNTWK },
    { S_DEVCONNECT2,    USREV_TIMEOUT,             A_DEVDISCONNECTNTWK },
    { S_DEVCONNECT2,    DMEV_CONNECT,              A_RELAXCNF    },


    { S_DEVCONNECT2_EXIT, DMEV_CONNECT,            A_DEVDISCONNECTNTWK        },
    { S_DEVCONNECT2_EXIT, DMEV_CONNECT_FAIL,       A_DEVDISCONNECTNTWK        },
    { S_DEVCONNECT2_EXIT, USREV_TIMEOUT,           A_DEVDISCONNECTNTWK        },
*/    



// Relax cnf        

    { S_RELAXCNF,       GCEV_BLOCKED,              A_DEVDISCONNECTNTWK },
    { S_RELAXCNF,       USREV_EXIT_REQUEST,        A_DEVDISCONNECTNTWK },
    { S_RELAXCNF,       GCEV_DISCONNECTED,         A_DEVDISCONNECTNTWK },
    { S_RELAXCNF,       USREV_LEAVE_CNF,           A_DEVDISCONNECTNTWK },

    { S_RELAXCNF,       GCEV_REQ_MODIFY_CALL,      A_MODIFYCALL },

// Dev disconnect    
    { S_DEVDISCONNECTNTWK, DMEV_DISCONNECT,        A_DEVDISCONNECTCNF },
    { S_DEVDISCONNECTNTWK, DMEV_DISCONNECT_FAIL,   A_DEVDISCONNECTCNF },
    { S_DEVDISCONNECTNTWK, USREV_TIMEOUT,          A_DEVDISCONNECTCNF },

    { S_DEVDISCONNECTCNF, DMEV_DISCONNECT,         A_REMOVEPARTY },
    { S_DEVDISCONNECTCNF, DMEV_DISCONNECT_FAIL,    A_REMOVEPARTY },
    { S_DEVDISCONNECTCNF, USREV_TIMEOUT,           A_REMOVEPARTY },


    { S_REMOVEPARTY,     CNFEV_REMOVE_PARTY,       A_CLOSEPARTY },
    { S_REMOVEPARTY,     CNFEV_REMOVE_PARTY_FAIL,  A_CLOSEPARTY },
    { S_REMOVEPARTY,     USREV_TIMEOUT,            A_CLOSEPARTY },

// No timeouts in Relax state
    { S_RELAX,          USREV_TIMEOUT,             A_NONE       },
    { S_RELAXCNF,       USREV_TIMEOUT,             A_NONE       },

// Drop & release
    { S_DROP,           GCEV_DROPCALL,             A_RELEASE    },
    { S_DROP,           USREV_TIMEOUT,             A_RELEASE    },
    { S_DROP,           GCEV_TASKFAIL,             A_FATAL      },
    
    { S_RELEASE,        GCEV_RELEASECALL,          A_RELAX      },
    { S_RELEASE,        USREV_TIMEOUT,             A_FATAL      },
    { S_RELEASE,        GCEV_RELEASECALL_FAIL,     A_RESET      },

// This is a way to ignore events in any state:
    { S_ANY,          IPMEV_QOS_ALARM,              A_NONE       },
    { S_ANY,          TDX_CST,                      A_NONE     },

    //Anything else falls here
    { S_ANY,          EVENT_ANY,                   A_ERROR      },
    { S_END,   0,    0  } 
}; // End of ntwk_event_stru[]


//*****************************************************************************
// Purpose	: 
//    Execute next step from state machine
//     Anything related to srl handle (gc_ or dx_) should happen here (except open and maybe close)
//     To request action from other thread, send user defined event (such as USREV_EXIT_REQUEST in this app)
// Parameters:	
//    [in] action
// Returns:	
//    true = success 
//    false = failure 
//*****************************************************************************
bool CNtwkDevice::Execute(int action){
bool brc = true;

  SetDfltTimer(NTWK_TMO); // reset back default timeout if it has been changed

  switch(action) {
      case A_ERROR:
           // Indicate error
           StateMachineError();
           break;

      case A_GIVEUP:  // do nothing but advance state machine
           SetSrlState(SRLSTATE_ALL, RESET_SRL_STATE);
           SetCurrentState(S_FINAL);
           break;      

      case A_NONE:					// do nothing
      case A_EXIT:					// do nothing but advance state machine
      case A_MOVE:					// .... to next 
      case A_MOVE_UNBLOCKED:		// .... to S_WAIT_UNBLOCKED
      case A_MOVE_OPENEX:			// .... to S_OPENEX
      case A_MOVE_OPEN:				// .... to S_OPEN
      case A_MOVE_DEVCONNECT2_EXIT: // .... to S_DEVDESCONNECT2_EXIT

           break;

      case A_CLOSE:
           brc = Close();
           break;

      case A_FATAL:  
           SetSrlState(SRLSTATE_FAILED_FATAL);
           Close();
           break;

      case A_ROUTE:
           brc = RouteVoice();
           break;

      case A_RESET:
           brc = ResetLineDev();
           break;

      case A_ANSWER:
           brc = AnswerCall();
           break;

      case A_ACCEPT:
           brc = AcceptCall();
           break;

      case A_DROP:
           brc = DropCall();
           break;

      case A_RELEASE:
           brc = ReleaseCall();
           break;

      case A_PLAYGREET:
           brc = PlayFile(m_pCommonParams->m_welcome_file);
           break;

      case A_PLAYINVALID:
           brc = PlayFile(m_pCommonParams->m_bad_passcode_file);
           break;

      case A_PLAYBYE:
           brc = PlayFile(m_pCommonParams->m_goodbye_file);
           break;

      case A_GETDIGITS:
           brc = GetDigits();
           break;

      case A_STOP_DX:
           brc = StopDx();
           break;
    
      case A_CHECK_CNF:
           CheckForValidConference();
           break;

      case A_UNLISTEN:
           UnListen();
           break;

      case A_CHECK_DNIS:
           CheckDnis();
           break;

      case A_RELAXCNF:
           m_pCnfConf->PutEvent(USREV_BEEP_IN, GetName());
           break;

      case A_RELAX:
           SetSrlState(SRLSTATE_INIT_COMPLETE);
           brc = WaitCall();
           break;

      case A_OPENPARTY: 
           brc = OpenParty();
           break;

      case A_ADDPARTY: 
           brc = AddParty();
           break;

      case A_DEVCONNECT: 
           brc = DevConnect(DEV_NTWK_PARTY);
           break;

      case A_DEVDISCONNECTNTWK: 
           m_pCnfConf->PutEvent(USREV_BEEP_OUT, GetName());
           brc = DevDisconnect(DEV_NTWK_PARTY);
           break;

      case A_DEVDISCONNECTCNF: 
           brc = DevDisconnect(DEV_CNF_PARTY);
           break;

      case A_REMOVEPARTY: 
           brc = RemoveParty();
           break;
   
      case A_CLOSEPARTY: 
           CloseParty();
           brc = DropCall();
           break;
        
      case A_SETPARTYATTR: 
           SetPartyAttributes(m_pCnfConf->GetClamping());
           break;

      case A_MODIFYCALL:
           ModifyCall();        // IP Request modify call
           break;

      default:
           LOG( LOG_ERR1, GetName(),
                "NtwkDevice: App error. Missing case to handle action #%d %s",
                 action, action_name(action));
           break;
  } // switch action
 return brc;
} // End of Execute()
//*****************************************************************************
// Purpose	: 
//    Handle GCEV_REQ_MODIFY_CALL
// Parameters:	
// Returns:	
//    success flag
//*****************************************************************************
bool CNtwkDevice::ModifyCall(){
  // Actual implementation is im IptDevice.cpp
    return true;
}
//*****************************************************************************
// Purpose	: 
//    Handle events
// Parameters:	
//   [in] event
//   [in] event data
//   [in] data length
//   [in] METAEVENT
// Returns:	
//    none
//*****************************************************************************
void CNtwkDevice::HandleEvent(int event,
                              void *evtdata, int evtlen,
                              METAEVENT *metaeventp){
int rc;
    switch ( event ){

        case CNFEV_ADD_PARTY:
             LOG(LOG_APP, GetName(), ">> Join conference %d %s",
                          m_pCnfConf->GetId(), m_pCnfConf->GetName());
             break;

        case CNFEV_REMOVE_PARTY:
          // Indicate party was removed
             if (m_pCnfConf){
                 m_pCnfConf->PutEvent(USREV_PARTY_REMOVED,GetName() , sizeof (this),this);
             }
             LOG(LOG_APP, GetName(), "<< Leave conference %d %s",
                          m_pCnfConf->GetId(), m_pCnfConf->GetName());
             break;   

        case GCEV_OPENEX_FAIL:
             SetSrlState(SRLSTATE_FAILED_FATAL);
             break;

        case GCEV_OPENEX:
             break;

        case TDX_RECORD:
        case TDX_PLAY:
             CloseVoiceFile();
             GetTermReason();
             break;

        case TDX_ERROR:
             CloseVoiceFile();
             break;    

        case TDX_GETDIG:
             GetTermReason();
            { // trim trailing # (if exist)
               char *ch = strchr(m_digits.dg_value,'#');
               if (ch) *ch = 0;
            }
             ClearDigits();
             LOG( LOG_APP,GetName(), 
                  "Digits received: %s",m_digits.dg_value);
             break; 
             
        case CNFEV_OPEN_PARTY:
            {
             PCNF_OPEN_PARTY_RESULT pResult = (PCNF_OPEN_PARTY_RESULT) evtdata;
             cnf_dump(pResult);
            }
             break;
        
        case GCEV_OFFERED:
            // retrieve common call data: crn, ani, dnis
             SetSrlState(SRLSTATE_CALL_ACTIVE);
             rc = gc_GetCRN(&m_crn, metaeventp);
             LOG( RC(rc),GetName(),
                  "%d = gc_GetCrn(m_crn :=0x%x)",
                  rc, m_crn);

             rc = gc_GetCallInfo(m_crn, ORIGINATION_ADDRESS, m_ani);
             LOG( RC(rc),GetName(),
                  "%d = gc_GetCallInfo(m_crn=0x%x, ORIGINATION_ADDRESS, m_ani:='%s')",
                  rc, m_crn, m_ani);

             rc = gc_GetCallInfo(m_crn, DESTINATION_ADDRESS, m_dnis );
             LOG( RC(rc),GetName(),
                  "%d = gc_GetCallInfo(m_crn=0x%x, DESTINATION_ADDRESS, m_dnis:='%s')",
                  rc, m_crn, m_dnis);

             LOG( LOG_APP, GetName(),
                  "*** New call from %s calling %s", m_ani, m_dnis);
             break;

        case GCEV_RESETLINEDEV: 
             SetSrlState(SRLSTATE_CALL_ACTIVE,RESET_SRL_STATE);
             OnInitComplete();
             break;

        case GCEV_RELEASECALL:
             SetSrlState(SRLSTATE_CALL_ACTIVE,RESET_SRL_STATE);
             break;

        case GCEV_DISCONNECTED:
             LOG( LOG_APP, GetName(), "*** Disconnected from %s", m_ani);
            // Drop!
        case GCEV_TASKFAIL:
             ProcessEventInfo(metaeventp);
             break;
    } // switch event

    CSrlDevice::HandleEvent(event, evtdata, evtlen, metaeventp);

  return ;
}  // End of HandleEvent()

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
//*****************************************************************************
// Purpose	: 
//    Constructor
// Parameters:	
//    [in] Device type (DEV_IPT or DEV_DTI)
//    [in] Configuration parameters (from configuration file)
//    [in] Optional name. When name is 0, use any available from specified type
// Returns:	
//    none
//*****************************************************************************
CNtwkDevice::CNtwkDevice(DLG_DEVICE_TYPE devtype,
                         PCommonParams pCommonParams, 
                         const char *name)
                        : CSrlDevice(devtype, pCommonParams, name) {
    m_crn = 0;
    *m_dnis=0;
    *m_ani=0;

    m_num_inv_passcode = 0;               // number of invalid pass codes

	memset(&m_digits, 0, sizeof(m_digits));

    m_do_waitcall = true;

    memset(&m_gc_info,0,sizeof(m_gc_info));

    SetExitCode(EXIT_OK);

    if (! SetStateMachine(ntwk_event_stru, ntwk_action_stru) ) {
        SetExitCode(EXIT_VALIDATE);
    }

    SetDfltTimer(NTWK_TMO);

    m_pCnfConf = 0;
 return;
}  // End of Constructor()

//*****************************************************************************
// Purpose	: 
//    Destructor
// Parameters:	
//    none
// Returns:	
//    none
//*****************************************************************************
CNtwkDevice::~CNtwkDevice() {

   // Close dx
   close_dx();
   CloseVoiceFile(); 

   // Close Device
   Close();
 return;
}  // End of Destructor()

// =====================================================
//  DX Functions
// =====================================================

//*****************************************************************************
// Purpose	: 
//    Close device
// Parameters:	
//    none
// Returns:	
//    true = success
//    false = failure
//*****************************************************************************
bool CNtwkDevice::Close(){
 bool brc = true;
 int rc;
    if( IsSrlState(SRLSTATE_OPENED, true) ){
        if ( IsSrlState(SRLSTATE_WAITEVENT, true) ){
             LOG( LOG_WARNING, GetName(),
                  "gc_Close() while waiting for event, state = %d %s",
                  GetCurrentState(), state_name(GetCurrentState()) );
                  DumpMissingEvents();
        }

        rc = gc_Close(m_srl_handle);
        LOG( RC(rc), GetName(),
             "%d = gc_Close(0x%x)",
             rc, m_srl_handle);

        brc = (rc == 0);
        if (!brc){
            process_gc_error();
        } else {
             m_srl_handle = INV_SRL_HANDLE;
        }
        
        close_dx();
	
        SetCurrentState(S_FINAL);
        SetSrlState(SRLSTATE_ALL, RESET_SRL_STATE);        
    }
 return brc;
}  // End of Close()

//*****************************************************************************
// Purpose	: 
//    process latest event info (cause)
// Parameters:	
//    metaevent
// Returns:	
//    bool - false indicates error while processing
//*****************************************************************************
bool CNtwkDevice::ProcessEventInfo(METAEVENT *metaeventp){
  int rc = gc_ResultInfo(metaeventp, &m_gc_info);
  LOG( RC(rc), GetName(),
       "%d = gc_ResultInfo(metaeventp, pInfo)", rc);
  gc_dump(&m_gc_info);

 return rc == GC_SUCCESS;
} // End of ProcessEventInfo()

// =====================================================
//  NTWK Functions
// =====================================================
//*****************************************************************************
// Purpose	: 
//    Reset line device and drop and release calls 
//          ( recover after blocking event)
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CNtwkDevice::ResetLineDev(){

  int rc = gc_ResetLineDev(m_srl_handle,EV_ASYNC);
      LOG( RCFATAL(rc), GetName(),
           "%d = gc_ResetLineDev(0x%x, EV_ASYNC)",
           rc, m_srl_handle);
      if (rc != GC_SUCCESS){
          SetSrlState ( SRLSTATE_FAILED_FATAL );
          process_gc_error();
          return false;
      }
      // next time, redo wait-call
      m_do_waitcall = true;
  return true;
} // End of ResetLineDev()

//*****************************************************************************
// Purpose	: 
//    Answer to incoming call
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CNtwkDevice::AnswerCall(){
int rc = GC_SUCCESS;

  rc = gc_AnswerCall(m_crn,1, EV_ASYNC);
  LOG( RC(rc), GetName(),
       "%d = gc_AnswerCall(crn = 0x%x, rings = 1, EV_ASYNC)",
       rc, m_crn);

  if ( rc != GC_SUCCESS){
       process_gc_error();
       return false;
  }  
  return true;
} // End of AnswerCall()

//*****************************************************************************
// Purpose	: 
//    Accept to incoming call
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CNtwkDevice::AcceptCall(){
  if (DoAcceptCall() ) {
	  int rc;
      rc = gc_AcceptCall(m_crn, 0, EV_ASYNC);
      LOG( RC(rc), GetName(),
             "%d = gc_AcceptCall(crn = 0x%x, rings = 0, EV_ASYNC)",
             rc, m_crn);
	  
      if ( rc != GC_SUCCESS){
           process_gc_error();
           return false;
	  }

  }else {
      PutEvent(USREV_SKIP_ACCEPT, GetName());
  }
  return true;
} // End of AcceptCall()

//*****************************************************************************
// Purpose	: 
//    Drop current call
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CNtwkDevice::DropCall(){
  int rc = gc_DropCall(m_crn, NORMAL_CLEARING, EV_ASYNC);
      LOG( RCFATAL(rc), GetName(),
           "%d = gc_DropCall(0x%x, cause = 0x%x (NORMAL_CLEARING), EV_ASYNC)",
           rc, m_crn, NORMAL_CLEARING);

      if ( rc != GC_SUCCESS){
           process_gc_error();
           SetSrlState ( SRLSTATE_FAILED_FATAL );
           return false;
      }
  return true;
} // End of DropCall()

//*****************************************************************************
// Purpose	: 
//    Release current call
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CNtwkDevice::ReleaseCall(){
  int rc = gc_ReleaseCallEx(m_crn, EV_ASYNC);
      LOG( RCFATAL(rc), GetName(),
           "%d = gc_ReleaseCallEx(0x%x, EV_ASYNC)",
            rc, m_crn);

      if ( rc != GC_SUCCESS){
           process_gc_error();
           SetSrlState ( SRLSTATE_FAILED_FATAL );
           return false;
      }
  return true;
}  // End of ReleaseCall()

//*****************************************************************************
// Purpose	: 
//    Watch for incoming calls
//      this function should be executed once at the beginning
//           and after each ResetLineDev
// Parameters:	
//    none
// Returns:	
//    bool (true = success)
//*****************************************************************************
bool CNtwkDevice::WaitCall(){

  InitForNewCall();

  if ( IsSrlState( SRLSTATE_REQUESTED_STOP,true) ){
       LOG(LOG_API, GetName(), "***Exit requested, leave WaitCall");
       SetCurrentState(S_FINAL);
       Close();
       return true;
  }

  if ( IsSrlState(SRLSTATE_BLOCKED | SRLSTATE_REQUESTED_STOP,true) ){
       LOG(LOG_API, GetName(), "***WaitCall in Blocked state: resume to S_UNBLOCKED");
       // indicate BLOCKED and go back to wait unblocked      
       return true;
  }

  if (m_do_waitcall) {
	  // do waitcall only after ResetLineDev and very first time
      int rc = gc_WaitCall(m_srl_handle, 0, 0, 0, EV_ASYNC);
          LOG( RCFATAL(rc), GetName(),
               "%d = gc_WaitCall(hndl=0x%x, crn:=0x%x, blk=0, tmo=0,  EV_ASYNC)",
               rc, m_srl_handle,m_crn);

          if ( rc != GC_SUCCESS){
               process_gc_error();
               SetSrlState ( SRLSTATE_FAILED_FATAL );
               return false;                             
          }
          m_do_waitcall = false;
      return true;
  }else {
      LOG( LOG_API, GetName(),
           "gc_WaitCall() - skipped");
  }
  return true;
}  // End of WaitCall()

//*****************************************************************************
// Purpose	: 
//    Initialize for next call
//      Call OnNewCall() - derived class hook  
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CNtwkDevice::InitForNewCall(){
     CloseVoiceFile();
     m_num_inv_passcode = 0;
     ClearDigits();
     m_crn = 0;
     *m_ani=0;
     *m_dnis=0;
     m_pCnfConf = 0;
     SetCnfBoard(0);

     if ( !   IsSrlState(SRLSTATE_REQUESTED_STOP, true)
         && ( S_FINAL != GetCurrentState()  ) ) {
         OnNewCall();  // Allow derived classes to initialize for next call
     }
  return true;
} // End of InitForNewCall()

//-----------------------------------------------------------------------
//     C O N F E R E N C E
//-----------------------------------------------------------------------
//*****************************************************************************
// Purpose	: 
//    Find conference party and join conference
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CNtwkDevice::AddParty(){
    CNF_ASSERT(m_pCnfConf, "Missing m_pCnfConf object");

    bool brc = m_pCnfConf->AddParty(m_cnf_party_srl_handle,this);

    LOG(LOG_DBG, GetName(), "%s = AddParty(0x%x)",brc?"true":"false", m_cnf_party_srl_handle);

 return brc;
}  // End of AddParty()

//*****************************************************************************
// Purpose	: 
//    Remove party from conference
// Parameters:	
//    none
// Returns:	
//    true = success
//*****************************************************************************
bool CNtwkDevice::RemoveParty(){
  CNF_ASSERT(m_pCnfConf, "Missing m_pCnfConf object");

  bool brc = m_pCnfConf->RemoveParty(m_cnf_party_srl_handle, this);
  LOG(LOG_DBG, GetName(), "%s = RemoveParty(0x%x)",brc?"true":"false", m_cnf_party_srl_handle);
 return brc;
}  // End of RemoveParty()


//*****************************************************************************
// Purpose	: 
//    The party is in conference, wait until disconnected
// Parameters:	
//    none
// Returns:	
//    true
//*****************************************************************************
bool CNtwkDevice::Relax(){
    LOG( LOG_APP,GetName(),
         "Party joined conference %s (id %d)",
         m_pCnfConf->GetName(), m_pCnfConf->GetId() );
 return true;
}  // End of Relax()


//*****************************************************************************
// Purpose	: 
//    Check if received digits match a conference
// Parameters:	
//    none
// Returns:	
//    true if no failure
//    EVENT: USREV_CHECK_PASS   if conference is found
//           USREV_CHECK_RETRY  if conference is not found, retry
//           USREV_CHECK_FAIL   if conference is not found, drop the call
//*****************************************************************************
bool CNtwkDevice::CheckForValidConference()  {
  PCnfConference pCnf;
  bool brc = true;
  if ( !glb_pConferencePool->FindConferenceByPassCode(m_digits.dg_value, &pCnf) ){
        if ( m_num_inv_passcode++ < MAX_RETRY) {
             LOG(LOG_DBG, GetName(), "Retry %d, cant match pass_code %s",
                 m_num_inv_passcode,m_digits.dg_value);
             brc = PutEvent(USREV_CHECK_RETRY, GetName());
        }else {
             LOG(LOG_DBG, GetName(), "Failed (attempt %d), cant match pass_code %s",m_num_inv_passcode,m_digits.dg_value);
             brc = PutEvent(USREV_CHECK_FAIL, GetName());
        }
  }else {
      brc = PutEvent(USREV_CHECK_PASS, GetName());
      m_pCnfConf = pCnf;
      SetCnfBoard(pCnf->GetCnfBoard());
  }
  return brc;
}  // End of CheckForValidConference()


//*****************************************************************************
// Purpose	: 
//    Check if received ANI matches a conference
//      any should match "C<pass_code>@..."
// Parameters:	
//    none
// Returns:	
//    true if no failure
//    EVENT: USREV_CHECK_PASS   if conference is found
//           USREV_CHECK_FAIL   if conference is not found, play the prompt
//*****************************************************************************
bool CNtwkDevice::CheckDnis()  {
  PCnfConference pCnf;
  bool brc = true;
  if ( tolower(*m_dnis) == 'c') {
       char my_dnis[GC_ADDRSIZE];
       str_safecopy(my_dnis,sizeof(my_dnis),m_dnis+1);
       char *ch = strchr(my_dnis,'@');
       if (ch) *ch = 0;
       if ( glb_pConferencePool->FindConferenceByPassCode(my_dnis, &pCnf) ){
            LOG( LOG_APP, GetName(),
                 "CheckDnis()::Matched to conference %s pass_code %s",
                 pCnf->GetName(), my_dnis);
            brc = PutEvent(USREV_CHECK_PASS, GetName());
            m_pCnfConf = pCnf;
            SetCnfBoard(pCnf->GetCnfBoard());
          return brc;
       }
  }
  LOG( LOG_API, GetName(),
       "CheckDnis():: Not matched to conference: %s",
       m_dnis);
  brc = PutEvent(USREV_CHECK_FAIL, GetName());
 return brc;
}  // End of CheckForValidConference()
