//***********************************************************************
// Copyright  2005, Intel Corporation. 
// All rights reserved All names, products, and services mentioned  
// herein are the trademarks or registered trademarks of their respective 
// organizations and are the sole property of their respective owners
//***********************************************************************

// ConfManager.cpp: implementation of the CConfManager class.
//
//////////////////////////////////////////////////////////////////////

#if defined(_WIN32)
#include <windows.h>
#else
#include <unistd.h>
#endif

#include "ConfManager.h"
#include "Conference.h"
#include "IpDev.h"
#include "AppLog.h"
#include "utils.h"
#include "pdl.h"

#include <srllib.h>
#include <dtilib.h>
#include <msilib.h>
#include <dcblib.h>
#include <time.h>
#include <string.h>
#include <list>

#define PTR30515

using namespace std;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

//*****************************************************************************
// Function : 
//    CConfManager::CConfManager()
// Purpose :
//    Constructor   
// Parameters:  
//    none
// Return type: 
//    none
//*****************************************************************************
CConfManager::CConfManager()
{
	m_hDspArray.clear();
	m_confInfoArray.clear();
}

//*****************************************************************************
// Function : 
//    CConfManager::~CConfManager()
// Purpose  : 
//    Destructor
// Parameters:  
//    none
// Return type: 
//    none
//*****************************************************************************
CConfManager::~CConfManager()
{
}

//////////////////////////////////////////////////////////////////////
// Public methods
//////////////////////////////////////////////////////////////////////


//*****************************************************************************
// Function : 
//    CConfManager::initConfManager()
// Purpose  : 
//    Initializing the conference Manager such as
//    finding total number of resources, setting configuration etc.
// Parameters:  
//    Parm1 <const char * fileBuf><input>
// Return type: 
//    int
//*****************************************************************************
int CConfManager::initConfManager(const char * fileBuf)
{
	char devName[64];
	
	if(setBoardLevelPrm(fileBuf) == -1)
		return -1;
	return 0;	
}



//*****************************************************************************
// Function : 
//    CConfManager::setBoardLevelPrm()
// 
// Purpose  : 
//    Sets DCB board parameters from config file
// 
// Parameters:  
//    Parm1 <char *fileBuf> in - buffer containing the config file
// // Return type:  
//    int
//*****************************************************************************
int CConfManager::setBoardLevelPrm(const char *fileBuf)
{
	int parmVal;
	int h_brd = -1;
	int h_dsp = -1;
#if defined(_WIN32)
	int val;
#else
	unsigned char val;
#endif
	
	if((h_brd = dcb_open("dcbB1", 0)) == -1) // name hardcoded, since only 1 logical boards exists
	{
		logOut(ERR2, "Cannot open dcbB1", 2);
		return -1;
	}
	if ((h_dsp = dcb_open("dcbB1D1",0)) == -1)
	{
		logOut(ERR2, "Cannot open dcbB1D1", 2);
		return -1;
	}
	m_hDspArray.push_back(h_dsp); // put the DSP handle to the array 
	sprintf(m_txt, "dcb_open(dcbB1D1) = %d", h_dsp);
	logOut(INFO, m_txt, 1);
	
	parseBoardParameters(fileBuf);
	
	// Setting board parameters:
	if(m_confFeatures.cf_activeTalker.at_enabled) // active talker enabled
	{
		parmVal = ACTID_ON;
		if (dcb_setbrdparm(h_brd, MSG_ACTID, &parmVal) == -1) 
		{
			sprintf(m_txt, "dcb_setbrdparm(MSG_ACTID) - %s", ATDV_ERRMSGP(h_brd));
			reportError(WARN, m_txt);
		}
		else
		{
			parmVal = m_confFeatures.cf_activeTalker.at_interval;
			if (dcb_setbrdparm(h_brd, MSG_ACTTALKNOTIFYINTERVAL, &parmVal) == -1) 
			{
				sprintf(m_txt, "dcb_setbrdparm(MSG_ACTTALKNOTIFYINTERVAL) - %s", ATDV_ERRMSGP(h_brd));
				reportError(WARN, m_txt);
			}
			
			/* Commented out until feature supported
			parmVal = m_confFeatures.cf_activeTalker.at_algorithm;
			if (dcb_setbrdparm(h_brd, MSG_ALGORITHM, &parmVal) == -1) 
			{
			sprintf(m_txt, "dcb_setbrdparm(MSG_ALGORITHM) - %s", ATDV_ERRMSGP(h_brd));
			reportError(WARN, m_txt); 
			}
			#endif
			*/
		}
	};
	if(m_confFeatures.cf_toneClamping) // tone clamping
		parmVal = TONECLAMP_ON;
	else
		parmVal = TONECLAMP_OFF;
	
	if (dcb_setbrdparm(h_brd, MSG_TONECLAMP, &parmVal) == -1) 
	{
		sprintf(m_txt, "dcb_setbrdparm(MSG_TONECLAMP = %d) - %s",parmVal, ATDV_ERRMSGP(h_brd));
		reportError(WARN, m_txt);
	}
/*      Commented out until feature supported
	if(m_confFeatures.cf_volumeDigit.vol_control) // tone clamping
	{
		if (dcb_setbrdparm(h_brd, MSG_VOLDIG, (void *) &(m_confFeatures.cf_volumeDigit)) == -1) 
		{
			sprintf(m_txt, "dcb_setbrdparm(MSG_VOLDIG, %d %d %d) - %s",
				m_confFeatures.cf_volumeDigit.vol_up,m_confFeatures.cf_volumeDigit.vol_reset,
				m_confFeatures.cf_volumeDigit.vol_down, ATDV_ERRMSGP(h_brd));
			reportError(WARN, m_txt);
		}
	};
#endif
*/

	parmVal = ON; // Enabling DCBEV_CTU event
	val = ON ;
	if (dcb_evtstatus(MSG_RESTBL, SET_EVENT,  &val) == -1) 
		logOut(WARN, "Cannot enable events", 2);

	m_numberOfDsp = ATDV_SUBDEVS(h_brd);
	if(m_numberOfDsp <= 0)
		reportError(ERR2, "No conference resource available");
	else
	{
		logOut(INFO,"Conference resource available" , 2);
	}
	dcb_close(h_brd);
	return (m_numberOfDsp >0)?0: -1;
}

//*****************************************************************************
// Function : 
//    CConfManager::loadConferenceTable()
// 
// Purpose  : calls private CConfManager:parseConfInfo()
//    to load conference passcodes and features
// 
// Parameters:  
//    Parm1 <const char *fileBuf><direction>
// 
// Return type: 
//    none
//*****************************************************************************
int CConfManager::loadConferenceTable(const char *fileBuf)
{
	if(parseConfInfo(fileBuf) == -1)
	{
		logOut(WARN, "Cannot load Conference table from the file", 2); 
		return -1;
	}
	else
		logOut(INFO, "Conference Table loaded", 2); 
	return 0;
}


//*****************************************************************************
// Function : 
//    CConfManager::getConfResources()
// Purpose  : 
//    Returns total count of conferencing resources in system
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CConfManager::getConfResources()
{
	int cnt = 0, total = 0;
	for(int i = 0; i<m_numberOfDsp; i++)
	{
		if(dcb_dsprescount(m_hDspArray[i], &cnt) == -1)
			reportError(WARN, ATDV_ERRMSGP(m_hDspArray[i]));
		else
			total += cnt;
	}
	sprintf(m_txt, "** %d conf resources available", total);	
	logOut(APP, m_txt, 2);
	return total;
}


//*****************************************************************************
// Function	: 
//    CConfManager::addToConference()
// Purpose	: 
//    adding that device to the conference
// Parameters:	
//    Parm1 <CIpDev * ipDev>
// Return type:	
//    int
//*****************************************************************************
int	CConfManager::addToConference(CIpDev * ipDev)
{
	int idx=1 ;
	logOut(INFO, "In addToConference", 1);
	idx = validatePasscode(ipDev->getDigBuffer());
	if(idx < 0) // passcode doesn't exists in a config file
	{
		logOut(APP, "Bad Passcode", 1);
		return BAD_PASSCODE;
	}
	
	int dsp_h, rx_confTimeslot; 
	
	logOut(INFO, "Before findAvailableResource", 1);
	dsp_h = findAvailableResource(); // // try to find at least 1 free...
	logOut(INFO, "findAvailableResource", 1);
	if(dsp_h == -1) // no free resources on the system
	{
		reportError(WARN, "The system is out of DCB resources");
		return NO_SYSTEM_RESOURCE;
	}
	
	CConference * conf = findConference(m_confInfoArray[idx].ci_bridgeNumber);
	logOut(INFO, "findConference", 1);
	if( conf == NULL) // passcode valid, but conference has not been created yet
	{
		
		logOut(INFO, "New Conference", 1);
		conf = initNewConference(dsp_h);
		conf->setConferenceFeatures(m_confInfoArray[idx].ci_echoCancellerEnabled,
			m_confInfoArray[idx].ci_detectDigits, m_confFeatures.cf_muteDigits); 
	}
	logOut(INFO, "addtoConf... ", 1);
	if(conf->addToConf(ipDev->getChannelNumber(), ipDev->getTxSlot(), &rx_confTimeslot) == -1)
	{
		reportError(ERR2, "Cannot add to conference");
		return NO_SYSTEM_RESOURCE;
	}
	
	conf->setBridgeNumber(m_confInfoArray[idx].ci_bridgeNumber);
	ipDev->setConfBridgeNumber(m_confInfoArray[idx].ci_bridgeNumber);
	ipDev->dti_Listen(rx_confTimeslot, EV_ASYNC);
	
	sprintf(m_txt, "Ch. %d -> to conf %d, total %d part's",  ipDev->getChannelNumber(),
		m_confInfoArray[idx].ci_bridgeNumber,conf->getNumberOfParticipants());
	logOut(APP, m_txt, 2);
	
	return 0;
}



//*****************************************************************************
// Function : 
//    CConfManager::removeFromConference()
// Purpose  : 
//    removing object from the conference
// Parameters:  
//    Parm1 <CIpDev * ipDev><direction>
// Return type: 
//    int
//*****************************************************************************
int CConfManager::removeFromConference(CIpDev * ipDev)
{
	CConference * tmpConf = findConference(ipDev->getConfBridgeNumber());
	if(!tmpConf)
	{
		sprintf(m_txt, "Cannot remove channel %d from conf. %d, not found",
			ipDev->getChannelNumber(), ipDev->getConfBridgeNumber() );
		reportError(ERR1, m_txt);
		return -1;
	}
	
	int cnt = tmpConf->getNumberOfParticipants();
	if(cnt == 1) 
	{
		stopMonitor(ipDev->getConfBridgeNumber()); 
		if(tmpConf->deleteConference() == 0)
		{
			removeFromList(tmpConf);
			sprintf(m_txt, "Conf bridge %d deleted", ipDev->getConfBridgeNumber());
			logOut(APP, m_txt,2);
		}
	}
	else
	{
		if(tmpConf->removeFromConf(ipDev->getTxSlot()) != -1)
		{
			sprintf(m_txt, "Conf %d - Channel %d Removed", ipDev->getConfBridgeNumber(),
				ipDev->getChannelNumber());
			logOut(INFO, m_txt, 1);
		}
		sprintf(m_txt, "Bridge %d: Party removed, %d remaining", ipDev->getConfBridgeNumber(),
			tmpConf->getNumberOfParticipants());
		logOut(APP, m_txt, 2);
	};
	return 0;
}


//*****************************************************************************
// Function : 
//    CConfManager::destroyConfManager()
// Purpose  : 
//    Deleting the Conference Manager
// Parameters:
//    none  
// Return type: 
//    none
//*****************************************************************************
void CConfManager::destroyConfManager()
{
	destroyConference(-1); 
#if defined(_WIN32)
	Sleep(2000);
#else
	sleep(2);
#endif
	for(int i = 0; i < m_numberOfDsp; i++) {
		dcb_close(m_hDspArray[i]);
	}
	for(list<CConference *>::iterator it = m_confList.begin(); it != m_confList.end(); it++) {
		delete *it;
	}
	m_confList.clear();
	m_hDspArray.clear(); 
	m_confInfoArray.clear();
}


//////////////////////////////////////////////////////////////////////
// Private methods
//////////////////////////////////////////////////////////////////////

//*****************************************************************************
// Function : 
//    CConfManager::initNewConference()
// Purpose  : 
//    Creating new bridge with the bridge Id that's passed in.
// Parameters:  none
//    
// Return type: 
//    CConference *
//*****************************************************************************
CConference * CConfManager::initNewConference(int hDsp)
{
	CConference * tmpCnf = new CConference;
	tmpCnf->init(hDsp);
	m_confList.push_back(tmpCnf); 
	logOut(APP, "New conference registered", 2);
	return tmpCnf;
}



//*****************************************************************************
// Function : 
//    CConfManager::removeFromList()

// Purpose  : 
//    removing the conference with Id ID = confId from the Linked List
// Parameters:  
//    Parm1 <int confId>
// Return type: 
//    none
//*****************************************************************************
void CConfManager::removeFromList(CConference * confToRemove)
{
	m_confList.remove(confToRemove);
	delete confToRemove;
}



//*****************************************************************************
// Function : 
//    CConfManager::findAvailableResource()
// Purpose  : 
//    Looking for a dsp that has at least 2 free resources
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CConfManager::findAvailableResource()
{
	int i, cnt = 0;
	char buffer[24];
	
	for(i = 0; i < m_numberOfDsp; i++)
	{
		if(dcb_dsprescount(m_hDspArray[i], &cnt) !=-1)
		{
			if(cnt > 0) // at least 1 free resource 
				return m_hDspArray[i];
		}
	}
	
	reportError(WARN, "No Conference resource found");
	return -1; // no DSP with at least 2 resources available 
}

//*****************************************************************************
// Function : 
//    CConfManager::validatePasscode()
// Purpose  : 
//    Checking validity of the passcode for the bridge that was entered by
//    the caller
// Parameters:  
//    Parm1 <const char * passcode><direction>
// Return type: 
//    int - array index of confInfo entry with ci_passcode = passcode
//*****************************************************************************
int CConfManager::validatePasscode(const char * passcode)
{
	if(strlen(passcode) < 2) // at least 1 number before '#'
		return -1;
	for(int i = 0; i < m_confInfoArray.size(); i++) {
		if(strcmp(passcode, m_confInfoArray[i].ci_passcode) == 0) { // matched
			return i;
		}
	}
	return -1;
}

//*****************************************************************************
// Function : 
//    CConfManager::findConference()
// Purpose  : 
//    Searches a conference in the linked either by bridge number (from file) or 
//    by internal conference ID                                         
// Parameters:  
//    Parm1 <int bridgeId, , int confId = 0><both input>
// Return type: 
//    CConference *
//*****************************************************************************
CConference * CConfManager::findConference(int bridgeNumber, int confId)
{
	list<CConference *>::iterator it;
	for(it = m_confList.begin(); it != m_confList.end(); it++)
	{
		if(confId == 0) // searching by bridge number
		{
			if( ( (*it))->getBridgeNumber() == bridgeNumber)  {
				return (*it); // found
			}
		}
		else {
			if( ( (*it))->getConfId() == confId) {
				return (*it); // found
			}
		}
	}
	return NULL; // not found
}


//*****************************************************************************
// Function : 
//    CConfManager::parseBoardParameters()
// Purpose  : 
//    Reading the configuration file to set board parameters
// Parameters:  
//    Parm1 <char * fileBuf><input>
// Return type: void
//*****************************************************************************
void CConfManager::parseBoardParameters(const char * fileBuf)
{
	int i;
// 2 definitions below to help to parse the config file
	const char * boardConfigParams[] = {
		"ActiveTalker", "ATInterval", "DTMFClamping",
		"MuteDigits", "Sip", "VolumeControl",
		"VolumeUp", "VolumeDown", "VolumeReset", NULL
	};

	enum {
		ACTIVE_TALK = 0, AT_INTERVAL, DTMF_CLAMP, MUTE_DIG, SIP_ENABLE,
		VOLUME_CONTROL, VOLUME_UP, VOLUME_DOWN, VOLUME_RESET
	};


// pre-set all parameters to default values 
	m_confFeatures.cf_activeTalker.at_enabled = 0;
	m_confFeatures.cf_activeTalker.at_interval = 3; // 100 ms units
	strcpy(m_confFeatures.cf_muteDigits, "*6");
	m_confFeatures.cf_toneClamping = 0;
	m_confFeatures.cf_volumeDigit.vol_control = OFF;
	m_confFeatures.cf_sipEnable = OFF;

/*
	no support in current beta release
    m_confFeatures->cf_activeTalker.at_algorithm = ALGO_LOUD; //the loudest talker;
*/

	if(fileBuf == NULL) // nothing to parse, leave params in default values
		return;
	char * current = (char *)fileBuf; // ponter for iterations
	current = strstr(current, "arameters]"); //skipped "P" for case insesitivity
	if(!current) // section not found
		return;
	current = GetNextLine(current);
	current = SkipComments(current);
	char stringValue[4] = {0};
	int  intValue = 0;
	while(current && *current != '[')
	{
		for(i = 0;boardConfigParams[i];i++)
		{
			if(strncmp(current, boardConfigParams[i], strlen(boardConfigParams[i])) == 0)
				break;
		}
		switch(i)
		{
		case ACTIVE_TALK: 
			if(sscanf(current,"%*[^=]= %3s",  stringValue) == 1) // assigned
				if(strcmp(stringValue, "Yes") ==0)
					m_confFeatures.cf_activeTalker.at_enabled = 1;
			break;
		case AT_INTERVAL:
			if(sscanf(current,"%*[^=]= %d", &intValue) == 1) // assigned
				m_confFeatures.cf_activeTalker.at_interval = intValue;
			break;
		case DTMF_CLAMP: 
			if(sscanf(current,"%*[^=]= %3s", stringValue) == 1) // assigned
				if(strcmp(stringValue, "Yes") ==0)
					m_confFeatures.cf_toneClamping = 1;

			break;
		case MUTE_DIG:
			sscanf(current,"%*[^=]= %2s", m_confFeatures.cf_muteDigits);
			break;
		case SIP_ENABLE:
			if(sscanf(current,"%*[^=]= %3s", stringValue) == 1) // assigned
				if(strcmp(stringValue, "Yes") ==0)
					m_confFeatures.cf_sipEnable = true;
			break;
		case VOLUME_CONTROL: 
			if(sscanf(current,"%*[^=]= %3s", stringValue) == 1) // assigned
				if(strcmp(stringValue, "Yes") ==0)
					m_confFeatures.cf_volumeDigit.vol_control = ON;
			break;
		case VOLUME_UP:
			if(sscanf(current,"%*[^=]= %d", &intValue) == 1) // assigned
				m_confFeatures.cf_volumeDigit.vol_up = intValue;
			break;
		case VOLUME_DOWN:
			if(sscanf(current,"%*[^=]= %d", &intValue) == 1) // assigned
				m_confFeatures.cf_volumeDigit.vol_up = intValue;
			break;
		case VOLUME_RESET:
			if(sscanf(current,"%*[^=]= %d", &intValue) == 1) // assigned
				m_confFeatures.cf_volumeDigit.vol_up = intValue;
			break;
		default:
			reportError(WARN,"Wrong Syntax in config file, line ignored");
			break;
		}
		current = GetNextLine(current);
		current = SkipComments(current);
	}
}


//*****************************************************************************
// Function	: 
//    CConfManager:parseConfInfo()
// Purpose	: 
//    Parse the configuration file buffer that contains 
// the passcode info and some conf attributes
// Parameters:	
//    Parm1 <char * fileBuf><input>
// Return type:	
//    int -1 if failed, otherwise 0
//*****************************************************************************
int CConfManager::parseConfInfo(const char * fileBuf)
{
	int idx;
	const char * conferenceConfigParams[] = {"Passcode","DetectDigits", "EchoCanceler", NULL};
	enum { PASSCODE = 0, DETECT_DIGITS, ECHO_CANCELER};

	if(fileBuf == NULL) // nothing to parse, leave params in default values
		return -1;

	confInfo * tmpInfo = new confInfo; // temporary storage
	char stringValue[4] = {0};

	char * current = (char *) fileBuf; // ponter for iterations

	m_confInfoArray.clear(); // clean old table

	for(int i = 0; current; i++)
	{
		int wrongSyntax = 0; // just a flag

		tmpInfo->ci_bridgeNumber = 0; 
		tmpInfo->ci_passcode[0] =0;
		tmpInfo->ci_detectDigits = false;
		tmpInfo->ci_echoCancellerEnabled = false;

		while(current)
		{
			current = SkipComments(current);
			if(!current) 
				break;
			if(strncmp(current, "[Conference", 10) == 0)
				break;
			current = GetNextLine(current);
		}
		if(!current) // section not found
			break;
		if(sscanf(current, "[%*s %d", &tmpInfo->ci_bridgeNumber) != 1)
		{
			strcpy(m_txt, "Wrong syntax near ");
			strncat(m_txt, current, 18);
			strcat(m_txt, " - ignored");
			reportError(WARN, m_txt);
			current = GetNextLine(current);
			continue;
		}
		current = GetNextLine(current);
		current = SkipComments(current);

		while(current && *current != '[')
		{
			for(idx = 0; conferenceConfigParams[idx]; idx++)
			{
				if(strncmp(current, conferenceConfigParams[idx], sizeof(conferenceConfigParams[idx])) == 0)
					break;
			}
			switch(idx)
			{
			case PASSCODE:
				if(sscanf(current, "%*[^=]= %[0123456789]",tmpInfo->ci_passcode) != 1)
				{
					reportError(WARN, "Wrong syntax in a [Conference] section, ignored");
					wrongSyntax = 1;
				}
				break;
			case DETECT_DIGITS:
				sscanf(current,"%*[^=]= %3s",stringValue);
				if(strcmp(stringValue, "Yes") == 0)
					tmpInfo->ci_detectDigits = true;
				else
					tmpInfo->ci_detectDigits = false;
				break;
			case ECHO_CANCELER:
				sscanf(current,"%*[^=]= %3s",stringValue);
				if(strcmp(stringValue, "Yes") == 0)
					tmpInfo->ci_echoCancellerEnabled = true;
				else
					tmpInfo->ci_echoCancellerEnabled = false;
				break;
			default:
				wrongSyntax = 1;
				break;
			}
			if(wrongSyntax) // section ignored, go to the next section
				break;
			current = GetNextLine(current); // Passcode accepted, go to the next parameter
			current = SkipComments(current);
		}

		if(!wrongSyntax) // now add the conference params to the array:
			m_confInfoArray.push_back(*tmpInfo);
	}
	delete tmpInfo; 
	return m_confInfoArray.size();
}


//*****************************************************************************
// Function : 
//    CConfManager::loadDefaultConfig()
// Purpose  : 
//    If the configuration file could not be read, the default bridge and 
//    passcodes are being loaded
// Parameters:  
//    none
// Return type: 
//    int
//*****************************************************************************
int CConfManager::loadDefaultConfig()
{
	const char * buf[] = {"12345", "23456", "34567", "45678", "56789", "67890", "78901"};
	m_confInfoArray.clear();
	confInfo * tmpInfo = new confInfo;
	for(int i = 0; i<7; i++)
	{
		strcpy(tmpInfo->ci_passcode, buf[i]);
		tmpInfo->ci_detectDigits = false;
		
		tmpInfo->ci_echoCancellerEnabled = false;
		tmpInfo->ci_bridgeNumber = i+1;
		
		m_confInfoArray.push_back(*tmpInfo);
	}
	delete tmpInfo;
	logOut(INFO, "Default Conference Table loaded", 2); 
	return m_confInfoArray.size(); 
}


//*****************************************************************************
// Function : 
//    CConfManager::reportError()
// Purpose  : 
//    Checking the error and reporting it
// Parameters:  
//    Parm1 <int type><direction>
//    Parm2 <char * err><direction>
// Return type: 
//    none
//*****************************************************************************
void CConfManager::reportError(int errType, const char * err)
{
	logOut(errType, err, 1);
}


//*****************************************************************************
// Function : 
//    CConfManager::logOut()
// Purpose  : 
//    Logging information to the log file
// Parameters:  
//    Parm1 <char * type><direction>
//    Parm2 <char * txt><direction>
//    Parm3 <int printToScreen><direction>
// Return type: 
//    none
//*****************************************************************************
void CConfManager::logOut(int type, const char * info, int printConsoleNum)
{
	appLog->addToLog(type, "CONF_MNG", info);
	printInfo(type, "CONF_MNG", info, brdStatusWindow);
	scrn_wrefresh(brdStatusWindow);
}


//*****************************************************************************
// Function : 
//    CConfManager::processEvenet()
// Purpose  : 
//    Processing the event 
// Parameters:  
//    Parm1 <long evt><direction>
// Return type: 
//    int
//*****************************************************************************
int CConfManager::processEvenet(long evt)
{
	void * data = sr_getevtdatap();
	int dev_h = sr_getevtdev();
	DCB_CT	   * tmpDcbCt; 	
	switch(evt)
	{
	case DCBEV_DIGIT:
	/*
	DCB_DIGITS * tmpDigit;
	tmpDigit = (DCB_DIGITS *) data;
	// We don't make digit processing here, since if a party in ReceiveOnly mode,
	// no digit will be detected by ConfManager
	break;
		*/
	case DCBEV_CTU:
		tmpDcbCt = (DCB_CT*) data;
		sprintf(m_txt,"DCBEV_CTU from conf 0x%x, TxTs = %d", tmpDcbCt->confid,
			tmpDcbCt->chan_num);
		logOut(EVT, m_txt, 1);
		break;
	case DCBEV_DELALLCONF:
		logOut(EVT, "DCBEV_DELALLCONF", 1);
		break;
	case DCBEV_ERREVT:
		logOut(EVT, "DCBEV_ERREVT", 1);
		break;
	default:
		logOut(EVT, "Unknown DCB event", 1);
		break;
	}
	return 0;
}


//*****************************************************************************
// Author   : 
// Created  : 
// Function : 
//    CConfManager::startMonitor()
// Purpose  : 
//    Starts recording (monitoring) the conference using free Ip channel
// Parameters:  
//    Parm1 <CIpDev *monDev><input>
//    Parm2 <int bridgeNumber><input>
// Return type: 
//    none
//*****************************************************************************
void CConfManager::startMonitor(CIpDev *monDev, int bridgeNumber)
{
	CConference * tmpConf = findConference(bridgeNumber);
	if(!tmpConf)
	{
		sprintf(m_txt, "Cannot monitor conf. %d, not found",bridgeNumber);
		reportError(WARN, m_txt);
		return;
	}
	int confTxTimeslot = 0;
	if(tmpConf->addMonitor(monDev, monDev->getVoiceTxSlot(), &confTxTimeslot) == 0)
	{
		monDev->dx_Listen(confTxTimeslot); 
		monDev->enableState(false);
		sprintf(m_txt, "Conf_%d.pcm", tmpConf->getBridgeNumber());
		monDev->recordVoice(m_txt, -1); 
	}
}



//*****************************************************************************
// Author   : 
// Created  : 
// Function : 
//    CConfManager::stopMonitor()
// Purpose  : 
//    Stops recording of conf bridge == bridgeNumber
// Parameters:  
//    Parm1 <int bridgeNumber><input>
// Return type: 
//    none
//*****************************************************************************
void CConfManager::stopMonitor(int bridgeNumber)
{
	CConference * tmpConf = findConference(bridgeNumber);
	if(!tmpConf)
	{
		sprintf(m_txt, "Cannot stop monitor on conf %d, not found",	bridgeNumber );
		logOut(INFO, m_txt, 1);
		return;
	}
	CIpDev * tmpDev = (CIpDev *) tmpConf->removeMonitor();
	if(tmpDev)
	{
		tmpDev->stopVoice();
		sprintf(m_txt, "Monitor stopped on bridge %d", bridgeNumber );
		logOut(INFO, m_txt, 2);
		tmpDev->dx_Listen(tmpDev->getTxSlot());
		tmpDev->enableState(true);
	}
}

//*****************************************************************************
// Author   : 
// Created  : 
// Function : 
//    CConfManager::destroyConference()
// Purpose  : 
//    Deletes 1 or all conferences
// Parameters:  
//    Parm1 <int bridgeNumber><input> if == -1, delete all existing conf's
// Return type: 
//    none
//*****************************************************************************
void CConfManager::destroyConference(int bridgeNumber)
{
	if(bridgeNumber == -1) // means destroy all conferences:
	{
		if (m_confList.size() == 0)
			logOut(INFO,"No Conferences to delete",1);
		else 
		{
			for(list<CConference *>::iterator it = m_confList.begin(); it != m_confList.end(); it++)
			{
				stopMonitor(((CConference *) (*it))->getBridgeNumber()); 
				((CConference *) (*it))->deleteConference(); 
			}	
			m_confList.clear();
			logOut(INFO, "All Conferences deleted", 2);
		}
	}
	else
	{
		CConference * tmpConf = findConference(bridgeNumber);
		if(!tmpConf)
		{
			sprintf(m_txt, "Cannot destroy conf. %d, not found", bridgeNumber);
			logOut(WARN, m_txt, 1);
		}
		else
		{
			stopMonitor(bridgeNumber);
			if(tmpConf->deleteConference() == 0)
			{
				removeFromList(tmpConf);
				sprintf(m_txt, "Conf bridge %d deleted", bridgeNumber);
				logOut(APP, m_txt,2);
			}
		}
	}
}

//*****************************************************************************
// Author   : 
/// Created : 
// Function : 
//    CConfManager::showStatistics()
// Purpose  : 
//    Displays all conferences and resource usage
// Parameters:  
// Return type: 
//    none
//*****************************************************************************
void CConfManager::showStatistics()
{
	logOut(INFO, "", 2);
	getConfResources();
	int numOfConf = m_confList.size();
	sprintf(m_txt, "   %d Conferences are currently in progress", numOfConf);
	logOut(INFO, m_txt, 2);
	int totalParties = 0;
	CConference * tmpConf = NULL;
	
	for(list<CConference *>::iterator it = m_confList.begin(); it != m_confList.end(); it++)
	{
		tmpConf = *it;
		int parties = tmpConf->getNumberOfParticipants();
		sprintf(m_txt, "   Conf. #%d: %d Parties", tmpConf->getBridgeNumber(),parties);
		totalParties += parties;
		logOut(INFO, m_txt, 2);
	}
	sprintf(m_txt, "** Total %d Conferees at this time", totalParties);
	logOut(INFO, m_txt, 2);
	logOut(INFO, "", 2);
}

//*****************************************************************************
// Author   : 
// Created  : 
// Function : 
//    CConfManager::displayActiveTalkers()
// Purpose  : 
//    Detects and displays all active talkers
// Parameters:  
// Return type: 
//    none
//*****************************************************************************
void CConfManager::displayActiveTalkers()
{
	if (m_confList.size() == 0) // no conference in progress:
	{
		logOut(INFO,"**ActiveTalker: No conference in progress", 2);
		return;
	}
	
	// PTR 30515, if more than 1 conf, dcb_getatibitsEx() may cause Access Violation
	// on some machines
#ifdef PTR30515
	if (m_confList.size() > 1) // no conference in progress:
	{
		logOut(WARN,"Can't get AT on more than 1 conf, HMP1.0 limitation", 1);
		return;
	}
#endif
	
	DCB_CT * atDcbCt = new DCB_CT[m_totalResourceCount];
	int numActiveTalkers = 0;
	
	for(int i = 0; i < m_hDspArray.size(); i++)
	{
		if(dcb_getatibitsEx(m_hDspArray[0], &numActiveTalkers, atDcbCt, NULL) != 0)
		{
			sprintf(m_txt, "Cannot get Active Talker info on device %d", m_hDspArray[i]);
			logOut(WARN, m_txt, 2);
			logOut(INFO, "Check if the feature enabled in config file", 2);
			continue;
		}
		
		if(!numActiveTalkers)
		{
			logOut(INFO, "No Active Talkers at this time", 2);
			continue;
		}
		
		logOut(INFO, "** Currently Active Talkers: **", 2); 
		for(int k = 0; k < numActiveTalkers; k++)
		{
			CConference * tmpConf = findConference(0, atDcbCt[k].confid);
			if(!tmpConf)  // not found??
				continue;
			confParty *partyInfo = tmpConf->findParticipant(atDcbCt[k].chan_num, 0);
			if(partyInfo)
			{
				sprintf(m_txt, "Bridge %d: Party Id %d, Chn %d",tmpConf->getBridgeNumber(),
					partyInfo->cp_partId, partyInfo->cp_partChannelNumber);
				logOut(INFO, m_txt, 2);
			}
		}
	}
	delete[] atDcbCt;
}

//*****************************************************************************
// 
// Function : 
//    CConfManager::updateConferenceTable()
// Purpose  : 
//    Goues through a new conf features table and send the
//    new parameters to all conferences in prgress
// Parameters:  
// Return type: 
//    none
//*****************************************************************************
void CConfManager::updateConferenceTable()
{
	CConference * tmpConf;
	for(int i = 0; i < m_confInfoArray.size(); i++)
	{
		if( (tmpConf = findConference(m_confInfoArray[i].ci_bridgeNumber)) != NULL)
			tmpConf->setConferenceFeatures(m_confInfoArray[i].ci_echoCancellerEnabled,
			m_confInfoArray[i].ci_detectDigits,	m_confFeatures.cf_muteDigits);
	}
}

//*****************************************************************************
// 
// Function : 
//    CConfManager::processDigits()
// Purpose  : 
//    Called from Main loop when user hits DTMF while in a conference
//    Currently, controls onmly mute-unmute, but may be extended
//    via additional implementation in CConference class 
// Parameters:  
//    Parm1 <CIpDev * ipDev><input>
// Return type: 
//    none
//*****************************************************************************
void CConfManager::processDigits(CIpDev * ipDev)
{
	CConference *tmpConf = findConference(ipDev->getConfBridgeNumber());
	if(tmpConf)
		tmpConf->processDigit(ipDev->getTxSlot(), *(ipDev->getDigBuffer()));    
	else
		logOut(WARN, "Cannot find conf object to send digit to", 1);
}


// End of ConfManager.cpp
