//***********************************************************************
// Copyright  2005, Intel Corporation. 
// All rights reserved All names, products, and services mentioned  
// herein are the trademarks or registered trademarks of their respective 
// organizations and are the sole property of their respective owners
//***********************************************************************

//////////////////////////////////////////////////////////////////////
// main.cpp: implementation of the application business logics.
//
//////////////////////////////////////////////////////////////////////

#if defined(_WIN32)
#include <windows.h>
#else
#include <unistd.h>
#endif

#include "Conference.h"
#include "ConfManager.h"
#include "IpDev.h"
#include "AppLog.h"
#include "pdl.h"

#define DEFINEHERE 1
#include "utils.h"

#include <srllib.h>
#include <dtilib.h>
#include <ctinfo.h>
#include <msilib.h>
#include <dcblib.h>
#include <ipmlib.h>
#include <signal.h>
#include <gcerr.h>
#include <stdlib.h>
#include <iostream>
#include <time.h>
#include <curses.h>

using namespace std;

// default definitions, in a case a config file is missing or wrong

#define WELLCOME_FILE      "welcome.pcm"
#define BAD_PASSCODE_FILE  "invalid.pcm"
#define GOODBYE_FILE       "bye.pcm"
#define CALL_LATER_FILE    "bye.pcm"

#define CONFIG_FILE        "conf_demo.cfg"
#define LOG_FILE           "HmpDcbDemo.log"

const char demoName[] = "Intel Netstructure Host Media Processing Conferencing Demo";

CAppLog * appLog;
CConfManager * confMgr;
CIpDev * ipDevArray; 

int exitDevice = -1; // to send QUIT message to

bool appRunning = false;

struct commonParameters // holds application-level parameters from config file
{
	int     cp_maxCalls;
	int     cp_maxLogSize;
	int     cp_logLevel;
	char    cp_logFileName[128];
	char    cp_welcomeFile[128];
	char    cp_badPasscodeFile[128];
	char    cp_callLaterFile[128];
	char    cp_goodByeFile[128];
} appLevelPrm;

const char mainMenu[] = "  <F2-Update Config> <F3-Active Talkers> <F4-Exit> <F5-System Stats>              <F6-Record Conf> <F7-Stop Record> <F8-Stop Conf> <F9-Stop All Confs>";


void setDefaultParameters(commonParameters & structPrm);
void setParametersFromFile(char * fileBuf, commonParameters & structPrm);
void displayParameters();	

void userInputHandler(); 
void cleanup();

void intr_handler(int h); // exit handler

CIpDev * getEventObject(long h_dev, CIpDev * devs, int devsNum);
CIpDev * getFreeChannel(CIpDev * devs, int devsNum);

//*****************************************************************************
// Function	: 
//    main()
// Purpose	: 
//    main start of the application
// Parameters:
//    none
// Return type:
//    int
//*****************************************************************************
int main()
{
	int availableIpChannels = 0;
	char m_txt[128];
	int i=0 ;
	static int j=1;
	char tmpTxt[256];
	memset(&tmpTxt, 0, sizeof(tmpTxt));
	
	signal(SIGINT,  intr_handler); // Exit handlers
	signal(SIGTERM, intr_handler);
	signal(SIGILL, intr_handler);
	signal(SIGFPE, intr_handler);
	signal(SIGABRT, intr_handler);
	
	if ( scrn_initscr() == NULL) {
		cerr << "Error initializing ncurses." << endl;
		exit(-1);
	}
	
	scrn_start_color();
	cbreak();
	scrn_refresh();
	if ((COLS < 80) || (LINES < 25)) {
		endwin();
		cerr << "Your terminal must be minimally 80X25" << endl;
		exit(-1);
	}
	scrn_init_pair(10, COLOR_CYAN, COLOR_BLACK);
	
	titleWindow = scrn_makewin(2, 80, 0, 0);
	scrn_wcolor_set(titleWindow, 10, NULL);
	scrn_box(titleWindow);
	scrn_waddstr(titleWindow, 0, (80-strlen(demoName))/2, demoName);
	scrn_wrefresh(titleWindow);
	
	brdStatusWindow = scrn_makewin(13, 80, 2, 0 );
	scrn_wcolor_set(brdStatusWindow, 10, NULL);
	scrn_box(brdStatusWindow);
	scrn_wrefresh(brdStatusWindow);
	
	menuWindow = scrn_makewin(4, 80, 15, 0);
	scrn_wcolor_set(menuWindow, 10, NULL);
	scrn_box(menuWindow);
	scrn_waddstr(menuWindow, 1, 1, mainMenu);
	scrn_wrefresh(menuWindow);
	
	demoWindow = scrn_makewin(6, 80, 19, 0);
	scrn_wcolor_set(demoWindow, 10, NULL);
	scrn_box(demoWindow);
	scrn_waddstr(demoWindow, 1, 1, "Waiting for user input :  ");
	scrn_wrefresh(demoWindow);
	
	char* fileBuf = ReadConfigFile(CONFIG_FILE); 
	setDefaultParameters(appLevelPrm); //preset common params to default
	setParametersFromFile(fileBuf, appLevelPrm); // set params from the file
	
	appLog = new CAppLog;
	appLog->initLog(appLevelPrm.cp_logFileName, appLevelPrm.cp_maxLogSize);
	
	// Read config file 
	displayParameters();
	
	//start
	GC_START_STRUCT	gclib_start;
	IPCCLIB_START_DATA cclibStartData;
	IP_VIRTBOARD virtBoards[1]; // only 1 NIC supported in current release
	
	memset(&cclibStartData,0,sizeof(IPCCLIB_START_DATA));
	memset(virtBoards,0,sizeof(IP_VIRTBOARD));
	
#if (IP_VIRTBOARD_VERSION >= 0x103)
	// use this function to initialize the IPCCLIB_START_DATA structure
	INIT_IPCCLIB_START_DATA(&cclibStartData, 1, virtBoards);
	INIT_IP_VIRTBOARD(virtBoards);
#else
	cclibStartData.version = 0x100;	// must be
	cclibStartData.delimiter =',';
	cclibStartData.num_boards = 1;
	cclibStartData.board_list = virtBoards;
	
	virtBoards[0].version = 0x100;	// must be
	virtBoards[0].reserved = NULL;
#endif
	
	virtBoards[0].total_max_calls = appLevelPrm.cp_maxCalls;
	virtBoards[0].h323_max_calls = appLevelPrm.cp_maxCalls;
	virtBoards[0].sip_max_calls = appLevelPrm.cp_maxCalls;
	virtBoards[0].localIP.ip_ver = IPVER4; // must be set to IPVER4
	virtBoards[0].localIP.u_ipaddr.ipv4 = IP_CFG_DEFAULT; // or specify host NIC IP address
	virtBoards[0].h323_signaling_port = IP_CFG_DEFAULT;  // or application defined port for H.323 
	virtBoards[0].sip_signaling_port = IP_CFG_DEFAULT;     // or application defined port for SIP
	
	CCLIB_START_STRUCT cclib_start[]={
		{"GC_DM3CC_LIB", NULL},
		{"GC_H3R_LIB", &cclibStartData},
		{"GC_IPM_LIB", NULL}
	};
	
	gclib_start.num_cclibs = sizeof(cclib_start) / sizeof(CCLIB_START_STRUCT);
	gclib_start.cclib_list = cclib_start;
	
	if ( gc_Start(NULL) != GC_SUCCESS) {
		appLog->addToLog (ERR2, "GC","Cannot start GlobalCall");
		printInfo (ERR2, "GC", "Cannot start GlobalCall", brdStatusWindow);
		return 0;
	}
	
	appLog->addToLog (INFO, "GC","gc_Start() - SUCCESS");
	
	confMgr = new CConfManager;
	if(confMgr->initConfManager(fileBuf) == -1) {
		appLog->addToLog( ERR2, "CONF_MNG","Error init Conf manager");
		printInfo(ERR2, "GC", "Error init Conf manager", brdStatusWindow);
		delete confMgr;
		delete appLog;
		delete[] fileBuf;
		gc_Stop();
		cleanup();
		exit(1);
	}
	if(confMgr->loadConferenceTable(fileBuf) == -1) {
		confMgr->loadDefaultConfig();
	}
	
	delete[] fileBuf;
	
	ipDevArray = new CIpDev[appLevelPrm.cp_maxCalls];
	CIpDev * evtDev = NULL;  
	if((ipDevArray+0)->openDev(1) == -1) // open first IP channels, waith for completion then continue
	{
		appLog->addToLog(ERR2, "IP_DEV","No IP channels available");
		printInfo (ERR2, "IP_DEV", "No IP channels available", brdStatusWindow);	
		delete confMgr;
		delete appLog;
		gc_Stop();
		cleanup();
		exit(1);
	}
	else
		appLog->addToLog(INFO, "IP_DEV", "Openning IP front end");
	
	long evt, h_dev; 
	int ret_val = -1;
	appRunning = true;
	
	while(appRunning)
	{
		if(sr_waitevt(20) != SR_TMOUT)
		{
			evt = sr_getevttype();
			h_dev = sr_getevtdev();
			// Check if it's a DCB event
			if ( ((evt & DT_DCB) == DT_DCB) || (evt == DCBEV_ERREVT) ) {
				confMgr->processEvenet(evt);
			}
			else
			{
				evtDev = getEventObject(h_dev, ipDevArray, appLevelPrm.cp_maxCalls);
				if (evtDev)
					ret_val = evtDev->processEvent();
				else
					continue; // should never happen! 
				
				printInfo (INFO, "IP_DEV", "Received IP call",brdStatusWindow);
				switch(ret_val)
				{
				case USR_EVENT_OPEN_CMPT:
					{
						availableIpChannels++;
						if(availableIpChannels < appLevelPrm.cp_maxCalls)
						{
							if((ipDevArray + availableIpChannels)->openDev(availableIpChannels+1) == -1)
							{
								exitDevice = ipDevArray->getLineDev(); 
								appLevelPrm.cp_maxCalls = availableIpChannels;
								sprintf(tmpTxt,"** %d IP Channels Opened, Init Complete", appLevelPrm.cp_maxCalls);
								printInfo (INFO, "IP_DEV", tmpTxt,brdStatusWindow);
							}
						}
						else
						{
							exitDevice = ipDevArray->getLineDev(); 
							sprintf(tmpTxt,"** %d IP Channels Opened, Init Complete", appLevelPrm.cp_maxCalls);
							printInfo(INFO, "IP_DEV", tmpTxt, brdStatusWindow);
						}
					}
					break;
				case USR_EVENT_EXIT:
					appRunning = false;
					break;
				case CALL_CONNECTED:
					if(evtDev->collectPasscodeDigits(appLevelPrm.cp_welcomeFile, 3) == -1) {
						evtDev->dropCallGracefully(appLevelPrm.cp_goodByeFile);
					}
					break;
				case DIGITS_RECEIVED:
					switch(confMgr->addToConference(evtDev))
					{
					case BAD_PASSCODE:
						if(evtDev->collectPasscodeDigits(appLevelPrm.cp_badPasscodeFile) == -1)
							evtDev->dropCallGracefully(appLevelPrm.cp_goodByeFile);
						break;
					case NO_SYSTEM_RESOURCE:
						evtDev->dropCallGracefully(appLevelPrm.cp_callLaterFile);
						break;
					default: // successfully added
						evtDev->enableSingleDigit(true); // enable reporting on each digit during a conf
						break;
					}
					break;
					case USER_DROPPED:
						evtDev->enableSingleDigit(false); 
						if(evtDev->getConfBridgeNumber() != -1) // the channel was in conference
						{
							confMgr->removeFromConference(evtDev);
							evtDev->setConfBridgeNumber(-1);
							evtDev->dti_Unlisten();
						}
						break;
					case SINGLE_DIGIT: // party pressed DTMF while in conference
						// pass it to confManager to process, since IpDev doesn't know what to do 
						confMgr->processDigits(evtDev); 
						break;
				}
			}
		}
		else { // no pending events, so let's check a user input
			userInputHandler();
		}
	} // end of app, when beyond this point
	
	confMgr->destroyConfManager();
	delete confMgr;
	for (i = 0; i < availableIpChannels; i++) {
		(ipDevArray + i)->closeDev();
	}
	
	delete[] ipDevArray;
	
	printInfo(INFO, "APP", "** Exiting program, please wait.......", brdStatusWindow);
#if defined(_WIN32)
	Sleep(5000);
#else
	sleep(5);
#endif
	
	delete appLog;
	cleanup();
	gc_Stop();
	
	return 0;
}


//*****************************************************************************
// Function : 
//    intr_handler()
// Purpose  : 
//    invoked upon abnormal termination or when user kills the app
// Parameters:  
//    Parm1 <int h><input>
// Return type: 
//    none
//*****************************************************************************
void intr_handler(int h)
{
	printf("Sending QUIT message to handle %d\n", exitDevice);
	sr_putevt(exitDevice, USR_EVENT_EXIT, 0,0,0);
}


//*****************************************************************************
// Function : 
//    CIpDev * getEventObject()
// Purpose  : 
//    Find the object by its handle 
// Parameters:  
//    Parm 1: <long> h_dev - event handle
//    Parm 2: <CIpDev *> devs  array to lookup
//    Parm 3: <int> devsNum   - array size
// Return type: 
//    CIpDev *   - returnds object if found, otherwise returns NULL
//*****************************************************************************
CIpDev * getEventObject(long h_dev, CIpDev * devs, int devsNum)
{
	for (int i = 0; i < devsNum; i++) {
		if ( ((devs + i)->getLineDev() == h_dev) ||
			((devs + i)->getMediaDev() == h_dev)) {
			return (devs + i);
		}
	}
	return NULL;
}

//*****************************************************************************
// Author   :
// Created  : 
// Function : 
//    getFreeChannel()
// Purpose  : 
//   looks for and returns a free Ip Device, mostly for monitoring
// Parameters:  
//    Parm1 <CIpDev * devs>< all Ip Devs, input>
//    Parm2 <int devsNum><input> - number of open Ip Channels 
// Return type: 
//    CIpDev * reference to free device, or NULL if not found
//*****************************************************************************
CIpDev * getFreeChannel(CIpDev * devs, int devsNum)
{
	for (int i = devsNum; --i >= 0; ) {
		if ( (devs + i)->isIdle()) {
			return (devs + i);
		}
	}
	return NULL;
}

//*****************************************************************************
// 
// Function : 
//    setDefaultParameters()
// 
// Purpose  : 
//    pre-set application-level parameters to default values
// 
// Parameters:  
//    Parm1 <commonParameters & structPrm><input / output>
// Return type: 
//    none
//*****************************************************************************
void setDefaultParameters(commonParameters & appLevelPrm)
{
	appLevelPrm.cp_maxCalls = 64;
	appLevelPrm.cp_maxLogSize = 10000;
	appLevelPrm.cp_logLevel = 0;
	strcpy(appLevelPrm.cp_logFileName, LOG_FILE); 
	strcpy(appLevelPrm.cp_welcomeFile, WELLCOME_FILE);
	strcpy(appLevelPrm.cp_badPasscodeFile, BAD_PASSCODE_FILE);
	strcpy(appLevelPrm.cp_callLaterFile, GOODBYE_FILE);
	strcpy(appLevelPrm.cp_goodByeFile, CALL_LATER_FILE);
}


//*****************************************************************************
// Function : 
//    setParametersFromFile()
// 
// Purpose  : 
//    Tries to parse the config file and reset those application-level parameters 
//    for which the parsing succeded 
// Parameters:  
//    Parm1 <char * fileBuf><input>
//    Parm2 <commonParameters & structPrm><input/output>
// 
// Return type: 
//    none
//*****************************************************************************
void setParametersFromFile(char * fileBuf, commonParameters & appLevelPrm)
{
	static bool firstLoad = true;
	int i;
	const char * commonParams[] = {"NumberOfChannels", "MaxLogSize", "PrintLevel",
		"LogFileName","WelcomeFileName", "BadPasscodeFileName", "CallLaterFileName",
		"GoodByeFileName",	NULL};
	
	enum { NUMOF_CHANNELS = 0, MAX_LOGSIZE, PRINT_LEVEL, LOGFILE_NAME,
		WELCOME_FILE_NAME, BADPASSCODE_FILE_NAME, CALLLATER_FILE_NAME,
		GOODBYE_FILE_NAME};
	
	if(fileBuf == 0) { // nothing to parse, leave params in default values
		return;
	}
	char * current = (char *)fileBuf; // ponter for iterations
	current = strstr(current, "ommon]"); //skipped 'C' for case insesitivity
	if (current == 0) { // section not found
		return;
	}
	current = GetNextLine(current);
	current = SkipComments(current);
	char stringValue[128] = {0};
	int  intValue = 0;
	while(current != 0 && *current != '[')
	{
		for(i = 0;commonParams[i];i++) {
			if(strncmp(current, commonParams[i], strlen(commonParams[i])) == 0) {
				break;
			}
		}
		
		switch(i) {
		case NUMOF_CHANNELS: 
			if(!firstLoad)
				break;	// cannot change max channels at run time
			if(sscanf(current,"%*[^=]= %d",  &intValue) == 1) // assigned
				appLevelPrm.cp_maxCalls = intValue;
			break;
		case MAX_LOGSIZE:
			if(sscanf(current,"%*[^=]= %d", &intValue) == 1) // assigned
				appLevelPrm.cp_maxLogSize = intValue;
			break;
		case PRINT_LEVEL: 
			if(sscanf(current,"%*[^=]= %d",  &intValue) == 1) // assigned
				appLevelPrm.cp_logLevel = intValue;
			break;
		case LOGFILE_NAME: 
			if(!firstLoad)
				break;	// cannot change max channels at run time
			if(sscanf(current,"%*[^=]= %127s", appLevelPrm.cp_logFileName ) != 1) // error
				printInfo(WARN, "MAIN", "Syntax error near LogFileName in cfg file", brdStatusWindow); 
			break;
		case WELCOME_FILE_NAME:
			if(sscanf(current,"%*[^=]= %127s", appLevelPrm.cp_welcomeFile ) != 1) // error
				printInfo(WARN, "MAIN", "Syntax error near WelcomeFileName in cfg file", brdStatusWindow);
			break;
		case BADPASSCODE_FILE_NAME:
			if(sscanf(current,"%*[^=]= %127s", appLevelPrm.cp_badPasscodeFile ) != 1) // error
				printInfo(WARN, "MAIN", "Syntax error near BadPasscodeFileName in cfg file",brdStatusWindow); 
			break;
		case CALLLATER_FILE_NAME:
			if(sscanf(current,"%*[^=]= %127s", appLevelPrm.cp_callLaterFile) != 1) // error
				printInfo(WARN, "MAIN", "Syntax error near CallLaterFileName in cfg file",brdStatusWindow);
			break;
		case GOODBYE_FILE_NAME:
			if(sscanf(current,"%*[^=]= %127s", appLevelPrm.cp_goodByeFile) != 1) // error
				printInfo(WARN, "MAIN", "Syntax error near GoodByeFileName in cfg file", brdStatusWindow);	
			break;
		default:
			break;
		}
		current = GetNextLine(current);
		current = SkipComments(current);
	}
	firstLoad = false;
}

//*****************************************************************************
// 
// Function : 
//    displayParameters()
// Purpose  : 
//    Displays application level parameters loaded from the config file 
// Parameters:  
// Return type: 
//    none
//*****************************************************************************
void displayParameters()
{
	char txt[128];

	printInfo(INFO, "APP", "** Begin List of Application Parameters **",brdStatusWindow);
	sprintf(txt, "  AppLog: file: %s, Level: %d",appLevelPrm.cp_logFileName, appLevelPrm.cp_logLevel);
	printInfo(INFO, "APP", txt, brdStatusWindow);
	sprintf(txt, "  Greeting: : %s, Bad Code: %s", appLevelPrm.cp_welcomeFile, appLevelPrm.cp_badPasscodeFile);
	printInfo(INFO, "APP", txt, brdStatusWindow);
	sprintf(txt, "  Call Later: %s, GoodBye: %s", appLevelPrm.cp_callLaterFile, appLevelPrm.cp_goodByeFile);
	printInfo(INFO, "APP", txt, brdStatusWindow);
	printInfo(INFO, "APP", "** End List **", brdStatusWindow);
}

//*****************************************************************************
// Author   : 
// Created  : 
// Function : 
//    userInputHandler()
// Purpose  : 
//    This callback invoked by CConsole class object when user preessed 
//    one of the function keys (F1-F10, menu option), or Enter, indicating
//    end of input 
// Parameters:  
//    Parm1 <int menuOption><input> - contains code of function key, if pressed,
//    otherwise 0 
//    Parm2 <const char * inputString><input> - usert input string, if Enter pushed
// Return type: 
//    none
//*****************************************************************************
void userInputHandler()
{
	int currentMenuChoice = 0;
	int Key;
	int i = 0;	
	char  inchar[10], readch = 0;
	char *fileBuf;
	CIpDev *tmpDev;
	char buffer[54] = "";
	
	scrn_box(demoWindow);
	scrn_refresh();
	
	keypad((WINDOW *)demoWindow,TRUE);
	nodelay((WINDOW *)demoWindow,TRUE);
	noecho();
	
	if ((Key = wgetch((WINDOW *)demoWindow)) == ERR) {
		return;
	}
	currentMenuChoice = Key - KEY_F0;
	keypad((WINDOW *)demoWindow,FALSE);
	echo();
	
	switch(currentMenuChoice)
	{
	case 2: // Update Config
		fileBuf = ReadConfigFile(CONFIG_FILE); 
		setParametersFromFile(fileBuf, appLevelPrm); 
		displayParameters();
		if(confMgr->loadConferenceTable(fileBuf) == 0)
			confMgr->updateConferenceTable(); // update conf's in progress
		delete[] fileBuf;
		break;
		
	case 3: // Show Active Talkers
		confMgr->displayActiveTalkers();
		break;
		
		
	case 4: // Show Active Talkers
		appRunning = false;
		break;
		
	case 5: // Show all conference in progress and resource usage
		confMgr->showStatistics();
		break;
		
	case 6: // Start Record Conference (monitoring)
		printInfo(INFO, "APP","Enter Conf bridge number to monitor: ",demoWindow);
		nodelay((WINDOW *)demoWindow,FALSE);
		wgetstr((WINDOW *)demoWindow, inchar);	
		Key = atoi(inchar);
		tmpDev = getFreeChannel(ipDevArray, appLevelPrm.cp_maxCalls);
		
		nodelay((WINDOW *)demoWindow,TRUE);
		sprintf(buffer, "conf bridge to monitor: %d", Key);
		printInfo(INFO, "APP", buffer, demoWindow);
		if(!tmpDev)
			printInfo(INFO, "WARN", "Cannot monitor, no free resource", demoWindow);
		else
			confMgr->startMonitor(tmpDev, Key);  
		break;
		
	case 7:    // Stop Recording
		printInfo(INFO, "APP", "Enter Conf bridge number to stop monitoring on: ",demoWindow);
		nodelay((WINDOW *)demoWindow, FALSE);
		wgetstr((WINDOW *)demoWindow, inchar);	
		Key = atoi(inchar);
		confMgr->stopMonitor(Key);
		break;
		
	case 8:   // Stop Conference
		printInfo(INFO, "APP","Enter Conf bridge number to terminate: ",demoWindow); 
		nodelay((WINDOW *)demoWindow, FALSE);
		wgetstr((WINDOW *)demoWindow, inchar);	
		Key = atoi(inchar);
		for(i=0; i < appLevelPrm.cp_maxCalls; i++)
		{
			if((ipDevArray + i)->getConfBridgeNumber() == Key)
			{			
				(ipDevArray + i)->drop(GC_NORMAL_CLEARING);
				(ipDevArray + i)->setConfBridgeNumber(-1);
			}
		}

		confMgr->destroyConference(Key);
		break;
		
		
	case 9:   // Stop all conferences
		printInfo(INFO, "APP","Are You sure (Y/N) : ",demoWindow); 
		nodelay((WINDOW *)demoWindow,FALSE);
		readch = wgetch((WINDOW *)demoWindow);
		if ((readch == 'y')||(readch == 'Y'))
		{
			for(i=0; i < appLevelPrm.cp_maxCalls; i++)
			{
				(ipDevArray + i)->drop(GC_NORMAL_CLEARING);
				(ipDevArray + i)->setConfBridgeNumber(-1);
		
			}
			confMgr->destroyConference(-1);
		}
		break;
		
	default: 
		printInfo(INFO, "APP", "Unknown Input", demoWindow);
		break;

	}
	
}

//*****************************************************************************
// 
// Function : 
//    printInfo()
// Purpose  : 
//    Formats and prints app's output to the screen (1st or 2nd scrolling area)
// Parameters:  
//    Parm1 <int type><input>
//    Parm2 <char *source><input>
//    Parm3 <const char *info><input>
//    Parm4 <int consWindow><input>
// Return type: 
//    none
//*****************************************************************************
void printInfo(int type, const char *source, const char *info, myWINDOW consWindow)
{
	unsigned long bytesWritten = 0;
	int flag;
	static int i=1;
	static int j=2;
	char buffer[256] = "";
	
	flag = (consWindow == demoWindow) ? 1 : 0;	// find where to print the text in the screen
	
	unsigned short foreColor = COLOR_CYAN;
	
	const char* sysTime = getSystemTime();
	int currPos = 15; 
	
	switch(type)
	{
	case APP:
		foreColor = COLOR_GREEN;
		break;
	case INFO:
		foreColor = COLOR_CYAN;
		break;
	case EVT:
		foreColor = COLOR_GREEN;
		break;
	case WARN:
		foreColor = COLOR_YELLOW;
		break;
	case ERR1:
		foreColor = COLOR_RED;
		break;
	case ERR2:
		foreColor = COLOR_RED;
		break;
	}
	currPos += 7;
	strncpy(buffer, info, strlen(info));
	if (flag)
	{
		if (j>=5)
		{
			j=2;
			wclrtoeol((WINDOW *) demoWindow);
			scrn_wrefresh(demoWindow);
		}
		
		scrn_box(demoWindow);
		scrn_refresh();
		scrn_waddstr(demoWindow, j++, 1, buffer);
		scrn_wrefresh(demoWindow);
		wclrtoeol((WINDOW *) demoWindow);
	}
	else
	{
		if (i >= 15)
		{
			i=1;
			wclrtoeol((WINDOW *) brdStatusWindow);
			scrn_wrefresh(brdStatusWindow);
		}
		scrn_box(brdStatusWindow);
		scrn_refresh();
		scrn_waddstr(brdStatusWindow, i++, 2, buffer);
		scrn_wrefresh(brdStatusWindow);
		wclrtoeol((WINDOW *) brdStatusWindow);
	}
}

void cleanup()
{
	scrn_wclear(demoWindow);
	scrn_box(demoWindow);
	clear();
	endwin();
	delwin((WINDOW *) demoWindow);

	scrn_wclear(menuWindow);
	scrn_box(menuWindow);
	clear();
	endwin();
	delwin((WINDOW *) menuWindow);

	scrn_wclear(titleWindow);
	scrn_box(titleWindow);
	clear();
	endwin();
	delwin((WINDOW *) titleWindow);

	scrn_wclear(brdStatusWindow);
	scrn_box(brdStatusWindow);
	clear();
	endwin();
}
// end of main.cpp
