/**
* @file mmstream.cpp
* @brief Multimedia demo - media streaming 
* @date Fen 16, 2007
*
* DIALOGIC CONFIDENTIAL 
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to 
* the source code ("Material") are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its 
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced, 
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual 
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include <fcntl.h>
#include <string.h>

// Application Level Include Files
#include "mmstream.h"
#include "config.h"
#include "ipmlib.h"
#include "sprop.h"
#include "appevents.h"
#ifdef WIN32
    #define strncasecmp _strnicmp
    #define strcasecmp strcmpi
#endif

//#define DIALOGIC_CIF_DCI_STR  "000001b003000001b58913000001020000012200c488ba9850b04241443f"
//#define DIALOGIC_QCIF_DCI_STR "000001b008000001b58913000001020000012200c488ba9850584121443f"

#define DIALOGIC_QCIF_DCI_STR  "000001b008000001b58913000001020000012200c488ba9850584121443f"
#define DIALOGIC_CIF_DCI_STR 	 "000001b000000001b58913000001020000012200c488ba9850b04241443f"
#define AMR_MAX_FPP 10

// Extern Variables
extern int  g_nTimeToRecord; 
extern int  g_nReadyChannels;
extern int  g_nNumberOfCalls;
extern char g_cAVFileDir[256];
extern int  g_nAudioFileFormat;
extern int  g_nVideoFileFormat;

extern bool g_bUseSipInfoDtmf;
extern int g_nDtmfDetectMode;
extern bool g_bRepeatMenus;
extern RTSP_URLS   g_rtspUrls;
extern unsigned int g_maxBandwidth;
extern unsigned int g_maxFrameRate;

extern unsigned char g_cRxDciCifSize;
extern unsigned char g_cRxDciCifValue[128];
extern char g_cRxDciCifStrValue[128];

extern unsigned char g_cRxDciQcifValue[128];
extern unsigned char g_cRxDciQcifSize;
extern char g_cRxDciQcifStrValue[128];

extern unsigned char g_cTxDciCifSize;
extern unsigned char g_cTxDciCifValue[128];
extern char g_cTxDciCifStrValue[128];

extern unsigned char g_cTxDciQcifValue[128];
extern unsigned char g_cTxDciQcifSize;
extern char g_cTxDciQcifStrValue[128];

extern int g_nPlatformNameRelease;
extern int g_nVideoCodec;
extern int g_nAudioCodec;
extern int g_nVideoRes;

extern int g_nAudioTranscodeDir;
extern int g_nVideoTranscodeDir;

extern bool g_bUseRecordMsg;
extern bool g_bRecordAudioOnly;
extern int  g_nRecordTimeout;

#ifdef TDM_AUDIO
extern char g_cISDNGatewayIP[256];
extern int  g_nISDNGatewayPort;
#endif

extern unsigned char g_h264SdpPacketizationMode;
extern unsigned char g_recordControl;
extern int g_h264SdpProfileLevel;

#define H264_LEVEL(profile_level)  ((profile_level) & 0xFF)
#define H264_B_CONSTRAINT(profile_level)  ((profile_level >> 12) & 0x01)

// Global Variables
static int      s_videoMlineLength            = 21;
static int      s_videoAlineLength            = 4;

#ifdef TDM_AUDIO
static std::string s_cMainMenuAudio("2g_main_menu.pcm");
static std::string s_cMainMenuVideo("2g_main_menu.vid");
static std::string s_cVmailMenuAudio("2g_vmail_menu.pcm");
static std::string s_cVmailMenuVideo("2g_vmail_menu.vid");
static std::string s_cVportalMenuAudio("2g_vportal_menu.pcm");
static std::string s_cVportalMenuVideo("2g_vportal_menu.vid");
#else
static std::string s_cMainMenuAudio("main_menu");
static std::string s_cMainMenuVideo("main_menu");
static std::string s_cVmailMenuAudio("vmail_menu");
static std::string s_cVmailMenuVideo("vmail_menu");
static std::string s_cVportalMenuAudio("vportal_menu");
static std::string s_cVportalMenuVideo("vportal_menu");

static std::string s_cImageFileName("myImage.yuv");
#endif

static std::string s_cClip1Audio("clip1");
static std::string s_cClip1Video("clip1");

static std::string s_cClip2Audio("clip2");
static std::string s_cClip2Video("clip2");

static std::string s_cClip3Audio("clip3");
static std::string s_cClip3Video("clip3");

static std::string s_cRecordingMsgVideo("NowRecordingMsg");
static std::string s_cRecordingMsgAudio("NowRecordingMsg");

extern unsigned char AsciiOctetToHex(const char *pOctet);
char g_cResolutionStr[4][9]={"SQCIF","QCIF","CIF"};

const char      *c_iFrameRequest            = "<?xml version=\"1.0\" encoding=\"utf-8\" ?><media_control><vc_primitive><to_encoder><picture_fast_update></picture_fast_update></to_encoder></vc_primitive></media_control>";
device_type     CMMStream::s_eType(MMS_DEVICE);

// state table definition for the connections
CMMStream::stateTransition CMMStream::s_initialTable[] = 
{
   { MMEV_OPEN,              &CMMStream::OnMMOpen,         CMMStream::state_initial,    "state_initial"},
   { MMEV_RESET,             &CMMStream::OnMMReset,        CMMStream::state_initial,    "state_initial"},
   { MMEV_RESET_FAIL,        &CMMStream::OnDoNothing,      CMMStream::state_initial,    "state_initial"},
   { MMEV_RESET_ACK,         &CMMStream::OnDoNothing,      CMMStream::state_initial,    "state_initial"},
   { MMEV_RESET_ACK_FAIL,    &CMMStream::OnDoNothing,      CMMStream::state_initial,    "state_initial"},
   { MMEV_ERROR,             &CMMStream::OnMMError,        CMMStream::state_initial,    "state_initial"},
   { MMEV_LAST_EVENT_ERROR,  &CMMStream::OnMMError,        CMMStream::state_initial,    "state_initial"},
   { IPMEV_OPEN,             &CMMStream::OnIpmOpenEx,      CMMStream::state_initial,    "state_initial"},
   { IPMEV_GET_LOCAL_MEDIA_INFO, &CMMStream::OnIpmGetLocalInfo, CMMStream::state_idle,  "state_initial"},
   { IPMEV_TELEPHONY_EVENT,  &CMMStream::OnIpmRfcEvent,    CMMStream::state_initial,    "state_initial"},
   { IPMEV_EVENT_ENABLED,    &CMMStream::OnDoNothing,      CMMStream::state_initial,    "state_initial"},
   { IPMEV_ERROR,            &CMMStream::OnIpmError,       CMMStream::state_initial,    "state_initial"},
   { DMEV_GET_TX_PORT_INFO,  &CMMStream::OnGetTxPortInfo,  CMMStream::state_initial,    "state_initial"},
   { DMEV_GET_RX_PORT_INFO,  &CMMStream::OnGetRxPortInfo,  CMMStream::state_initial,    "state_initial"},
   { DMEV_RESERVE_RESOURCE,  &CMMStream::OnReserveResourceEx, CMMStream::state_initial, "state_initial"},
   { DMEV_CONNECT,           &CMMStream::OnDevConnect,     CMMStream::state_initial,    "state_initial"},
   { DMEV_CONNECT_FAIL,      &CMMStream::OnDoNothing,      CMMStream::state_initial,    "state_initial"},
   { DMEV_PORT_CONNECT,      &CMMStream::OnDevPortConnect, CMMStream::state_initial,    "state_initial"},
   { DMEV_PORT_CONNECT_FAIL, &CMMStream::OnDoNothing,      CMMStream::state_initial,    "state_initial"},
   { APPEV_IFRAME_REQ,       &CMMStream::OnIFrameReqRcvd,  CMMStream::state_initial,    "state_initial"},  
   { APPEV_WAIT_TIMEOUT,     &CMMStream::OnAppWaitTimeoutRcvd, CMMStream::state_initial,"state_initial"},  
   { UNDECLARED,             &CMMStream::OnIpmUndeclared,  CMMStream::state_initial,    "state_initial"}
};

CMMStream::stateTransition CMMStream::s_idleTable[] = 
{
   { TDX_GETDIG,             &CMMStream::OnVoxDigitRcvd,   CMMStream::state_idle,      "state_idle"},
#ifdef TDM_AUDIO 
   { TDX_RECORD,             &CMMStream::OnDXRecordDone,   CMMStream::state_idle,      "state_idle"}, 
#endif 
   { MMEV_PLAY,              &CMMStream::OnDoNothing,      CMMStream::state_idle,      "state_idle"},
   { MMEV_PLAY_FAIL,         &CMMStream::OnMMPlayFail,     CMMStream::state_idle,      "state_idle"}, 
   { MMEV_PLAY_ACK,          &CMMStream::OnMMPlayAck,      CMMStream::state_idle,      "state_idle"},
   { MMEV_PLAY_ACK_FAIL,     &CMMStream::OnMMPlayAckFail,  CMMStream::state_idle,      "state_idle"},
   { MMEV_ERROR,             &CMMStream::OnMMError,        CMMStream::state_idle,      "state_idle"},
   { MMEV_RESET,             &CMMStream::OnDoNothing,      CMMStream::state_idle,      "state_idle"},
   { IPMEV_TELEPHONY_EVENT,  &CMMStream::OnIpmRfcEvent,    CMMStream::state_idle,      "state_idle"},
   { IPMEV_GET_LOCAL_MEDIA_INFO, &CMMStream::OnIpmGetLocalInfo, CMMStream::state_idle, "state_idle"},
   { IPMEV_STARTMEDIA,       &CMMStream::OnIpmStartMedia,  CMMStream::state_connected, "state_idle"},
   { IPMEV_EVENT_ENABLED,    &CMMStream::OnDoNothing,      CMMStream::state_idle,      "state_idle"},
   { IPMEV_STOPPED,          &CMMStream::OnIpmStopped,     CMMStream::state_idle,      "state_idle"},
   { IPMEV_ERROR,            &CMMStream::OnIpmError,       CMMStream::state_idle,      "state_idle"},
   { DMEV_CONNECT,           &CMMStream::OnDevConnect,     CMMStream::state_idle,      "state_idle"},
   { DMEV_PORT_CONNECT,      &CMMStream::OnDevPortConnect, CMMStream::state_idle,      "state_idle"},
   { DMEV_PORT_CONNECT_FAIL, &CMMStream::OnDoNothing,      CMMStream::state_idle,      "state_idle"},
   { APPEV_IFRAME_REQ,       &CMMStream::OnIFrameReqRcvd,  CMMStream::state_idle,      "state_idle"}, 
   { APPEV_WAIT_TIMEOUT,     &CMMStream::OnAppWaitTimeoutRcvd, CMMStream::state_idle,  "state_idle"},  
   { UNDECLARED,             &CMMStream::OnIpmUndeclared,  CMMStream::state_idle,      "state_idle"}
};

CMMStream::stateTransition CMMStream::s_connectedTable[] = 
{
   { TDX_GETDIG,            &CMMStream::OnVoxDigitRcvd,    CMMStream::state_connected,    "state_connected"},
   { TDX_CST,               &CMMStream::OnDoNothing,       CMMStream::state_connected,    "state_connected"},
   { MMEV_PLAY,             &CMMStream::OnDoNothing,       CMMStream::state_connected,    "state_connected"},
   { MMEV_PLAY_FAIL,        &CMMStream::OnMMPlayFail,      CMMStream::state_connected,    "state_connected"},
   { MMEV_PLAY_ACK,         &CMMStream::OnMMPlayAck,       CMMStream::state_playMainMenu, "state_connected"},
   { MMEV_PLAY_ACK_FAIL,    &CMMStream::OnMMPlayAckFail,   CMMStream::state_connected,    "state_connected"},
   { MMEV_RECORD_ACK,       &CMMStream::OnMMRecordAck,     CMMStream::state_recordMsg,    "state_connected"},
   { MMEV_RECORD_ACK_FAIL,  &CMMStream::OnMMRecordAckFail, CMMStream::state_recordMsg,    "state_connected"},
   { MMEV_STOP_ACK,         &CMMStream::OnMMStopAck,       CMMStream::state_connected,    "state_connected"},
   { MMEV_RESET,            &CMMStream::OnDoNothing,       CMMStream::state_connected,    "state_connected"},
   { MMEV_ERROR,            &CMMStream::OnMMError,         CMMStream::state_connected,    "state_connected"},
   { IPMEV_TELEPHONY_EVENT, &CMMStream::OnIpmRfcEvent,     CMMStream::state_connected,    "state_connected"},
   { IPMEV_STARTMEDIA,      &CMMStream::OnIpmStartMedia,   CMMStream::state_connected,    "state_connected"}, // 2nd IPM (RTSP) dev
   { IPMEV_STOPPED,         &CMMStream::OnIpmStopped,      CMMStream::state_idle,         "state_connected"},
   { IPMEV_ERROR,           &CMMStream::OnIpmError,        CMMStream::state_idle,         "state_connected"},
   { DMEV_CONNECT,          &CMMStream::OnDevConnect,      CMMStream::state_connected,    "state_connected"},
   { APPEV_IFRAME_REQ,      &CMMStream::OnIFrameReqRcvd,   CMMStream::state_connected,    "state_connected"},    
   { APPEV_WAIT_TIMEOUT,    &CMMStream::OnAppWaitTimeoutRcvd, CMMStream::state_connected, "state_connected"},  
#ifdef USE_MTK_API
   { SMEV_ADD_OVERLAY,      &CMMStream::OnAddOverlay,      CMMStream::state_connected,    "state_connected"},
   { SMEV_ADD_OVERLAY_FAIL, &CMMStream::OnAddOverlayFail,  CMMStream::state_connected,    "state_connected"},
#endif
   { UNDECLARED,            &CMMStream::OnIpmUndeclared,   CMMStream::state_connected,    "state_connected"}
};

CMMStream::stateTransition CMMStream::s_playMainMenuTable[] = 
{
   { TDX_GETDIG,            &CMMStream::OnVoxDigitRcvd,    CMMStream::state_playMainMenu, "state_playMainMenu"},
   { TDX_CST,               &CMMStream::OnDoNothing,       CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_PLAY,             &CMMStream::OnMMPlayDone,      CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_PLAY_FAIL,        &CMMStream::OnMMPlayFail,      CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_PLAY_ACK,         &CMMStream::OnMMPlayAck,       CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_PLAY_ACK_FAIL,    &CMMStream::OnMMPlayAckFail,   CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_RECORD_ACK,       &CMMStream::OnMMRecordAck,     CMMStream::state_recordMsg,    "state_playMainMenu"},
   { MMEV_RECORD_ACK_FAIL,  &CMMStream::OnMMRecordAckFail, CMMStream::state_recordMsg,    "state_playMainMenu"},
   { MMEV_STOP_ACK,         &CMMStream::OnMMStopAck,       CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_STOP_ACK_FAIL,    &CMMStream::OnDoNothing,       CMMStream::state_playMainMenu, "state_playMainMenu"}, 
   { MMEV_SEEK,             &CMMStream::OnMMSeek,          CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_PAUSE,            &CMMStream::OnMMPause,         CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_RESUME,           &CMMStream::OnMMResume,        CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_CAPTURE,          &CMMStream::OnMMCapture,       CMMStream::state_playMainMenu, "state_playMainMenu"},
   { MMEV_RESET,            &CMMStream::OnDoNothing,       CMMStream::state_disconnected, "state_playMainMenu"},
   { MMEV_ERROR,            &CMMStream::OnMMError,         CMMStream::state_playMainMenu, "state_playMainMenu"},
   { IPMEV_TELEPHONY_EVENT, &CMMStream::OnIpmRfcEvent,     CMMStream::state_playMainMenu, "state_playMainMenu"},
   { IPMEV_ERROR,           &CMMStream::OnIpmError,        CMMStream::state_playMainMenu, "state_playMainMenu"},
   { IPMEV_STOPPED,         &CMMStream::OnIpmStopped,      CMMStream::state_idle,         "state_playMainMenu"},
   { APPEV_IFRAME_REQ,      &CMMStream::OnIFrameReqRcvd,   CMMStream::state_playMainMenu, "state_playMainMenu"}, 
   { APPEV_WAIT_TIMEOUT,    &CMMStream::OnAppWaitTimeoutRcvd, CMMStream::state_playMainMenu, "state_playMainMenu"},  
#ifdef TDM_AUDIO
   { DMEV_CONNECT,          &CMMStream::OnDoNothing,       CMMStream::state_playMainMenu, "state_playMainMenu"},
   { DMEV_DISCONNECT,       &CMMStream::OnDoNothing,       CMMStream::state_playMainMenu, "state_playMainMenu"}, 
   { TDX_RECORD,            &CMMStream::OnDXRecordDone,    CMMStream::state_playMainMenu, "state_playMainMenu"},
#endif
#ifdef USE_MTK_API
   { SMEV_REMOVE_OVERLAY,       &CMMStream::OnRemoveOverlay,      CMMStream::state_playMainMenu, "state_playMainMenu"},
   { SMEV_REMOVE_OVERLAY_FAIL,  &CMMStream::OnRemoveOverlayFail,  CMMStream::state_playMainMenu, "state_playMainMenu"},
   { SMEV_REMOVE_ALL_OVERLAYS,  &CMMStream::OnRemoveAllOverlays,  CMMStream::state_playMainMenu, "state_playMainMenu"},
   { SMEV_REMOVE_ALL_OVERLAYS_FAIL, &CMMStream::OnRemoveAllOverlaysFail, CMMStream::state_playMainMenu, "state_playMainMenu"},
#endif
   { UNDECLARED,            &CMMStream::OnIpmUndeclared,   CMMStream::state_playMainMenu, "state_playMainMenu"}
};

CMMStream::stateTransition CMMStream::s_playVPortalMenuTable[] = 
{
   { TDX_GETDIG,            &CMMStream::OnVoxDigitRcvd,    CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { TDX_CST,               &CMMStream::OnDoNothing,       CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_PLAY,             &CMMStream::OnMMPlayDone,      CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_PLAY_FAIL,        &CMMStream::OnMMPlayFail,      CMMStream::state_playVPortalMenu, "state_playVPortalMenu"}, 
   { MMEV_PLAY_ACK,         &CMMStream::OnMMPlayAck,       CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_PLAY_ACK_FAIL,    &CMMStream::OnMMPlayAckFail,   CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_RECORD_ACK,       &CMMStream::OnMMRecordAck,     CMMStream::state_recordMsg,       "state_playVPortalMenu"},
   { MMEV_RECORD_ACK_FAIL,  &CMMStream::OnMMRecordAckFail, CMMStream::state_recordMsg,       "state_playVPortalMenu"},
   { MMEV_STOP_ACK,         &CMMStream::OnMMStopAck,       CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_STOP_ACK_FAIL,    &CMMStream::OnDoNothing,       CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_SEEK,             &CMMStream::OnMMSeek,          CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_PAUSE,            &CMMStream::OnMMPause,         CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_RESUME,           &CMMStream::OnMMResume,        CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_CAPTURE,          &CMMStream::OnMMCapture,       CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { MMEV_RESET,            &CMMStream::OnDoNothing,       CMMStream::state_disconnected,    "state_playVPortalMenu"},
   { MMEV_ERROR,            &CMMStream::OnMMError,         CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { IPMEV_TELEPHONY_EVENT, &CMMStream::OnIpmRfcEvent,     CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { IPMEV_STOPPED,         &CMMStream::OnIpmStopped,      CMMStream::state_idle,            "state_playVPortalMenu"},
   { IPMEV_ERROR,           &CMMStream::OnIpmError,        CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { APPEV_IFRAME_REQ,      &CMMStream::OnIFrameReqRcvd,   CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},  
   { APPEV_WAIT_TIMEOUT,    &CMMStream::OnAppWaitTimeoutRcvd, CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},  
#ifdef TDM_AUDIO
   { DMEV_CONNECT,          &CMMStream::OnDoNothing,       CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { DMEV_DISCONNECT,       &CMMStream::OnDoNothing,       CMMStream::state_playVportalMenu, "state_playVPortalMenu"}, 
   { TDX_RECORD,            &CMMStream::OnDXRecordDone,    CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
#endif
#ifdef USE_MTK_API
   { SMEV_REMOVE_OVERLAY,   &CMMStream::OnRemoveOverlay,   CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { SMEV_REMOVE_OVERLAY_FAIL, &CMMStream::OnRemoveOverlayFail, CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { SMEV_REMOVE_ALL_OVERLAYS, &CMMStream::OnRemoveAllOverlays, CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
   { SMEV_REMOVE_ALL_OVERLAYS_FAIL, &CMMStream::OnRemoveAllOverlaysFail, CMMStream::state_playVPortalMenu, "state_playVPortalMenu"},
#endif
   { UNDECLARED,           &CMMStream::OnIpmUndeclared,    CMMStream::state_playVPortalMenu, "state_playVPortalMenu"}
};

CMMStream::stateTransition CMMStream::s_playVMailMenuTable[] = 
{
   { TDX_GETDIG,            &CMMStream::OnVoxDigitRcvd,    CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { TDX_CST,               &CMMStream::OnDoNothing,       CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_PLAY,             &CMMStream::OnMMPlayDone,      CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_PLAY_FAIL,        &CMMStream::OnMMPlayFail,      CMMStream::state_playVMailMenu, "state_playVMailMenu"}, 
   { MMEV_PLAY_ACK,         &CMMStream::OnMMPlayAck,       CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_PLAY_ACK_FAIL,    &CMMStream::OnMMPlayAckFail,   CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_RECORD,           &CMMStream::OnMMRecordDone,    CMMStream::state_recordMsg,     "state_playVMailMenu"},
   { MMEV_RECORD_FAIL,      &CMMStream::OnMMRecordFail,    CMMStream::state_recordMsg,     "state_playVMailManu"},
   { MMEV_RECORD_ACK,       &CMMStream::OnMMRecordAck,     CMMStream::state_recordMsg,     "state_playVMailMenu"},
   { MMEV_RECORD_ACK_FAIL,  &CMMStream::OnMMRecordAckFail, CMMStream::state_recordMsg,     "state_playVMailMenu"},
   { MMEV_STOP_ACK,         &CMMStream::OnMMStopAck,       CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_STOP_ACK_FAIL,    &CMMStream::OnDoNothing,       CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_SEEK,             &CMMStream::OnMMSeek,          CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_PAUSE,            &CMMStream::OnMMPause,         CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_RESUME,           &CMMStream::OnMMResume,        CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_CAPTURE,          &CMMStream::OnMMCapture,       CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { MMEV_RESET,            &CMMStream::OnDoNothing,       CMMStream::state_disconnected,  "state_playVMailMenu"},
   { MMEV_ERROR,            &CMMStream::OnMMError,         CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { IPMEV_TELEPHONY_EVENT, &CMMStream::OnIpmRfcEvent,     CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { IPMEV_STOPPED,         &CMMStream::OnIpmStopped,      CMMStream::state_idle,          "state_playVMailMenu"},
   { IPMEV_ERROR,           &CMMStream::OnIpmError,        CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { APPEV_IFRAME_REQ,      &CMMStream::OnIFrameReqRcvd,   CMMStream::state_playVMailMenu, "state_playVMailMenu"},    
   { APPEV_WAIT_TIMEOUT,    &CMMStream::OnAppWaitTimeoutRcvd, CMMStream::state_playVMailMenu, "state_palyVMailMenu"},  
#ifdef TDM_AUDIO
   { DMEV_CONNECT,          &CMMStream::OnDoNothing,       CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { DMEV_DISCONNECT,       &CMMStream::OnDoNothing,       CMMStream::state_playVMailMenu, "state_playVMailMenu"}, 
   { TDX_RECORD,            &CMMStream::OnDXRecordDone,    CMMStream::state_playVMailMenu, "state_playVMailMenu"},
#endif
#ifdef USE_MTK_API
   { SMEV_REMOVE_OVERLAY,   &CMMStream::OnRemoveOverlay,   CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { SMEV_REMOVE_OVERLAY_FAIL, &CMMStream::OnRemoveOverlayFail, CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { SMEV_REMOVE_ALL_OVERLAYS, &CMMStream::OnRemoveAllOverlays, CMMStream::state_playVMailMenu, "state_playVMailMenu"},
   { SMEV_REMOVE_ALL_OVERLAYS_FAIL, &CMMStream::OnRemoveAllOverlaysFail, CMMStream::state_playVMailMenu, "state_playVMailMenu"},
#endif
   { UNDECLARED,            &CMMStream::OnIpmUndeclared,   CMMStream::state_playVMailMenu, "state_playVMailMenu"}
};

CMMStream::stateTransition CMMStream::s_recordMsgTable[] =
{
   { TDX_GETDIG,            &CMMStream::OnVoxDigitRcvd,    CMMStream::state_recordMsg,    "state_recordMsg"},
   { TDX_CST,               &CMMStream::OnDoNothing,       CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_PLAY,             &CMMStream::OnMMPlayDone,      CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_PLAY_FAIL,        &CMMStream::OnMMPlayFail,      CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_PLAY_ACK,         &CMMStream::OnMMPlayAck,       CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_PLAY_ACK_FAIL,    &CMMStream::OnMMPlayAckFail,   CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_RECORD,           &CMMStream::OnMMRecordDone,    CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_RECORD_FAIL,      &CMMStream::OnMMRecordFail,    CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_RECORD_ACK,       &CMMStream::OnMMRecordAck,     CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_RECORD_ACK_FAIL,  &CMMStream::OnMMRecordAckFail, CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_VIDEO_RECORD_STARTED, &CMMStream::OnMMVideoStarted, CMMStream::state_recordMsg, "state_recordMsg"},
   { MMEV_STOP_ACK,         &CMMStream::OnMMStopAck,       CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_STOP_ACK_FAIL,    &CMMStream::OnDoNothing,       CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_SEEK,             &CMMStream::OnDoNothing,       CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_PAUSE,            &CMMStream::OnDoNothing,       CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_RESUME,           &CMMStream::OnDoNothing,       CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_CAPTURE,          &CMMStream::OnDoNothing,       CMMStream::state_recordMsg,    "state_recordMsg"},
   { MMEV_RESET,            &CMMStream::OnDoNothing,       CMMStream::state_disconnected, "state_recordMsg"},
   { MMEV_ERROR,            &CMMStream::OnMMError,         CMMStream::state_recordMsg,    "state_recordMsg"},
   { IPMEV_TELEPHONY_EVENT, &CMMStream::OnIpmRfcEvent,     CMMStream::state_recordMsg,    "state_recordMsg"},
   { IPMEV_STOPPED,         &CMMStream::OnIpmStopped,      CMMStream::state_idle,         "state_recordMsg"},
   { IPMEV_ERROR,           &CMMStream::OnIpmError,        CMMStream::state_recordMsg,    "state_recordMsg"},
   { APPEV_IFRAME_REQ,      &CMMStream::OnIFrameReqRcvd,   CMMStream::state_recordMsg,    "state_recordMsg"},    
   { APPEV_WAIT_TIMEOUT,    &CMMStream::OnAppWaitTimeoutRcvd, CMMStream::state_recordMsg,    "state_recordMsg"},  
#ifdef USE_MTK_API
   { SMEV_REMOVE_OVERLAY,   &CMMStream::OnRemoveOverlay,   CMMStream::state_recordMsg,               "state_recordMsg"},
   { SMEV_REMOVE_OVERLAY_FAIL, &CMMStream::OnRemoveOverlayFail, CMMStream::state_recordMsg,          "state_recordMsg"},
   { SMEV_REMOVE_ALL_OVERLAYS, &CMMStream::OnRemoveAllOverlays, CMMStream::state_recordMsg,          "state_recordMsg"},
   { SMEV_REMOVE_ALL_OVERLAYS_FAIL, &CMMStream::OnRemoveAllOverlaysFail, CMMStream::state_recordMsg, "state_recordMsg"},
#endif
   { UNDECLARED,            &CMMStream::OnIpmUndeclared,   CMMStream::state_recordMsg,    "state_recordMsg"}
};

CMMStream::stateTransition CMMStream::s_disconnectedTable[] = 
{
   { TDX_GETDIG,            &CMMStream::OnVoxDigitRcvd,    CMMStream::state_disconnected, "state_disconnected"},
   { TDX_CST,               &CMMStream::OnDoNothing,       CMMStream::state_disconnected, "state_disconnected"},
   { MMEV_PLAY,             &CMMStream::OnDoNothing,       CMMStream::state_disconnected, "state_disconnected"},
   { MMEV_PLAY_FAIL,        &CMMStream::OnMMPlayFail,      CMMStream::state_disconnected, "state_disconnected"}, 
   { MMEV_PLAY_ACK,         &CMMStream::OnMMPlayAck,       CMMStream::state_disconnected, "state_disconnected"},
   { MMEV_PLAY_ACK_FAIL,    &CMMStream::OnMMPlayAckFail,   CMMStream::state_disconnected, "state_disconnected"},
   { MMEV_RECORD,           &CMMStream::OnMMRecordDone,    CMMStream::state_disconnected, "state_disconnected"},
   { MMEV_RECORD_ACK,       &CMMStream::OnMMRecordAck,     CMMStream::state_disconnected, "state_disconnected"},
   { MMEV_RECORD_ACK_FAIL,  &CMMStream::OnMMRecordAckFail, CMMStream::state_disconnected, "state_disconnected"},
   { MMEV_STOP_ACK,         &CMMStream::OnMMStopAck,       CMMStream::state_idle,         "state_disconnected"},
   { MMEV_STOP_ACK_FAIL,    &CMMStream::OnDoNothing,       CMMStream::state_disconnected, "state_disconnected"},
   { MMEV_RESET,            &CMMStream::OnDoNothing,       CMMStream::state_disconnected, "state_disconnected"},
   { MMEV_ERROR,            &CMMStream::OnMMError,         CMMStream::state_disconnected, "state_disconnected"},
   { IPMEV_TELEPHONY_EVENT, &CMMStream::OnIpmRfcEvent,     CMMStream::state_disconnected, "state_disconnected"},
   { IPMEV_STOPPED,         &CMMStream::OnIpmStopped,      CMMStream::state_idle,         "state_disconnected"},
   { IPMEV_ERROR,           &CMMStream::OnIpmError,        CMMStream::state_disconnected, "state_disconnected"},
   { APPEV_IFRAME_REQ,      &CMMStream::OnIFrameReqRcvd,   CMMStream::state_disconnected, "state_disconnected"}, 
   { APPEV_WAIT_TIMEOUT,    &CMMStream::OnAppWaitTimeoutRcvd, CMMStream::state_disconnected, "state_disconnected"},  
   { DMEV_DISCONNECT,       &CMMStream::OnDoNothing,       CMMStream::state_disconnected, "state_disconnected"},
   { DMEV_DISCONNECT_FAIL,  &CMMStream::OnDoNothing,       CMMStream::state_disconnected, "state_disconnected"},
   { DMEV_PORT_DISCONNECT,  &CMMStream::OnDoNothing,       CMMStream::state_disconnected, "state_disconnected"},
   { DMEV_PORT_DISCONNECT_FAIL, &CMMStream::OnDoNothing,   CMMStream::state_disconnected, "state_disconnected"},
   { UNDECLARED,            &CMMStream::OnIpmUndeclared,   CMMStream::state_disconnected, "state_disconnected"}
};

CMMStream::stateTransition CMMStream::s_appStoppingTable[] =
{
   { MMEV_ERROR,            &CMMStream::OnMMError,       CMMStream::state_appStopping,   "state_appStopping"},
   { IPMEV_UNLISTEN,        &CMMStream::StopStream,      CMMStream::state_appStopping,   "state_appStopping"},
   { APPEV_IFRAME_REQ,      &CMMStream::OnIFrameReqRcvd, CMMStream::state_appStopping,   "state_appStopping"},  
   { APPEV_WAIT_TIMEOUT,    &CMMStream::OnAppWaitTimeoutRcvd, CMMStream::state_appStopping, "state_appStopping"},  
   { IPMEV_ERROR,           &CMMStream::OnIpmError,      CMMStream::state_appStopping,   "state_appStopping"},
   { UNDECLARED,            &CMMStream::OnIpmUndeclared, CMMStream::state_appStopping,   "state_appStopping"}
};

CMMStream::stateTransition *CMMStream::s_ipCallTable[] = 
{
   CMMStream::s_initialTable,
   CMMStream::s_idleTable,
   CMMStream::s_connectedTable,
   CMMStream::s_playMainMenuTable,
   CMMStream::s_playVPortalMenuTable,
   CMMStream::s_playVMailMenuTable,
   CMMStream::s_recordMsgTable,
   CMMStream::s_disconnectedTable,
   CMMStream::s_appStoppingTable,
};

// Function Definitions
//*****************************************************************************
// 		  NAME : void CMMStream::ProcessEvent(METAEVENT &metaevt, CMMStream *pChan)
// DESCRIPTION : Handles all events
// 		 INPUT : metaevt - Object of the type metaevent containing stream event data
//				 *pChan - Pointer to CMMStream object on which the event occurred
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::ProcessEvent(METAEVENT &metaevt, CMMStream *pChan)
{
   // Local Variable Declaration
   CMMStream::stateTransition* pTable = 0;
   CMMStream::APPHANDLER ipCallAppHdlr = 0;

   // get the table that contains this state.
   pTable = (s_ipCallTable)[pChan->GetState()];

   // find the entry that matches the current state event.
   for ( ; ; )
   {
      //mmReport(INFO_OK, s_eType, "Looking for event %s in state %s", mmEvt2Str(metaevt.evttype), pTable->st_name);
      if ( pTable->st_event == metaevt.evttype )
      {
         //mmReport(INFO_OK, s_eType, "****** FOUND Event %s in table %s", mmEvt2Str(metaevt.evttype), pTable->st_name);
         break;
      }
      else if ( pTable->st_event == UNDECLARED )
      {
         //mmReport(INFO_OK, s_eType, "+++++++ UNDECLARED for Event %s in table %s", mmEvt2Str(metaevt.evttype), pTable->st_name);
         break;
      }
      else
      {
         // move on
         pTable++;
         //mmReport(INFO_OK, s_eType, "=========== Moving on for Event %s in table %s", mmEvt2Str(metaevt.evttype), pTable->st_name);
      }
   }
   pChan->m_state_name = pTable->st_name;
   mmReport(INFO_DEBUG, s_eType, "--- Got event %s for MMS in state = %s ----", mmEvt2Str(metaevt.evttype), pChan->m_state_name);
   // call the function now.
   ipCallAppHdlr = pTable->st_action;
   //mmReport(INFO_OK, "Executing AppHandler for %s", pTable->st_name);
   if ( ipCallAppHdlr )
   {
      (pChan->*ipCallAppHdlr)(metaevt.evtdatap);
   }
   else
   {
      mmReport(INFO_ERROR, s_eType, "Invalid APP_HANDLER.......");
   }

   if ( pChan->m_bSetState == true )
   {
      pChan->SetState(pTable->st_next);
   }
   return;
}

//*****************************************************************************
// 		  NAME : CMMStream::CMMStream(const char *pMMDevName, 
//							const char *pIpmDevName, const char *pVoxDevName)
// DESCRIPTION : CMMStream Constructor
// 		 INPUT : pMMDevName - Pointer to string containing mmDevice name
//				 pIpmDevName - Pointer to string containing IPMDevice name
//				 pVoxDevName - Pointer to string containing VoxDevice name
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
#ifdef TDM_AUDIO
CMMStream::CMMStream(const char *pMMDevName, 
                     const char *pIpmDevName, 
                     const char *pVoxDevName,
                     const char *pDtiDevName)
#else
CMMStream::CMMStream(const char *pMMDevName, 
		     const char *pIpmCCDevName,  // JS
		     const char *pIpmRTSPDevName,  // JS
                     const char *pVoxDevName)
#endif
{
   //memset(&m_audioMediaList,   0, sizeof(MM_MEDIA_ITEM_LIST));
   //memset(&m_videoMediaList,   0, sizeof(MM_MEDIA_ITEM_LIST));
   //memset(&m_playInfo,         0, sizeof(MM_PLAY_INFO));
   //memset(&m_playRecordList,   0, sizeof(MM_PLAY_RECORD_LIST)*2);
   //memset(&m_ipmLocalMedia,    0, sizeof(IPM_MEDIA_INFO));
   //memset(&m_ipmRemoteMedia,   0, sizeof(IPM_MEDIA_INFO));

   //INIT_MM_MEDIA_ITEM_LIST(&m_audioMediaList);
   //INIT_MM_MEDIA_ITEM_LIST(&m_videoMediaList);
   //INIT_MM_PLAY_INFO(&m_playInfo);
   //INIT_MM_RECORD_INFO(&m_recordInfo);
   //INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[0]);
   //INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[1]);

   memset(&m_CCipmLocalMedia,    0, sizeof(IPM_MEDIA_INFO));
   memset(&m_ccIpmMediaInfo,   0, sizeof(IPM_MEDIA_INFO));
   memset(&m_RTSPipmLocalMedia,    0, sizeof(IPM_MEDIA_INFO)); // JS
   memset(&m_RTSPipmRemoteMedia,   0, sizeof(IPM_MEDIA_INFO)); // JS

   // Initialize Class Member Variables
   m_bSetState                             = true; // let ProcessEvent() set the state.
   m_state_name                            = "state_initial";
   m_eState                                    = state_initial;
   m_eCurrentState                     = state_initial;
   m_bPlayDone                             = true;
   m_bRecordDone                           = true;
   m_bWaiting                           = false;
   m_bMMStopping                           = false;
   m_bMMPlayStarting                       = false;
   m_bMMRecordStarting                     = false;
   m_bCCStreamingRtp                     = false;
   m_bRTSPStreamingRtp                   = false;
   m_bIsGetDigitPending                    = false;
#ifdef TDM_AUDIO
   m_b2gIsBridged                   = false;
#endif
   m_bUserStop                             = false;
   m_unLocalCCIPMAudioRtpPortIndex          = 0;
   m_unLocalCCIPMVideoRtpPortIndex          = 0;
   m_nAudioRtpPort                     = 0;
   m_nVideoRtpPort                     = 0;
   m_nRfc2833PayloadType               = RFCDTMF;
   m_bVideoCall                            = true;
   m_CCipmLocalMedia.unCount             = 2;
   m_CCipmLocalMedia.MediaData[0].eMediaType = MEDIATYPE_VIDEO_LOCAL_RTP_INFO;
   m_CCipmLocalMedia.MediaData[1].eMediaType = MEDIATYPE_AUDIO_LOCAL_RTP_INFO;
   m_RTSPipmLocalMedia.unCount             = 2;
   m_RTSPipmLocalMedia.MediaData[0].eMediaType = MEDIATYPE_VIDEO_LOCAL_RTP_INFO;
   m_RTSPipmLocalMedia.MediaData[1].eMediaType = MEDIATYPE_AUDIO_LOCAL_RTP_INFO;
   m_pRemoteSDP = NULL;

   m_SdpDciValue=(unsigned char*)calloc(128,sizeof(unsigned char));
   m_stRemoteEPVideoCodec.m_SdpDciValue=(unsigned char*)calloc(128,sizeof(unsigned char));
// SKS
   INIT_DEV_RESOURCE_LIST(&m_ResourceList);

   if(g_nPlatformNameRelease<HMP_4_1) {
      m_ResourceList.count                = 6;
      m_ResourceList.rsList[0]          = RESOURCE_IPM_AMR_NB;
      m_ResourceList.rsList[1]          = RESOURCE_IPM_G729;
      m_ResourceList.rsList[2]          = RESOURCE_IPM_G723;
      m_ResourceList.rsList[3]          = RESOURCE_IPM_G711_10MS;
      m_ResourceList.rsList[4]          = RESOURCE_IPM_G711_20MS;
      m_ResourceList.rsList[5]          = RESOURCE_IPM_G711_30MS;
      //m_ResourceList.rsList[6]          = RESOURCE_IPM_G722;
      //m_ResourceList.rsList[7]          = RESOURCE_IPM_AMR_WB;
   }
   else //Only HMP4.1 supports WB codecs. 
   {
      m_ResourceList.count                = 8;
      m_ResourceList.rsList[0]          = RESOURCE_IPM_AMR_NB;
      m_ResourceList.rsList[1]          = RESOURCE_IPM_G729;
      m_ResourceList.rsList[2]          = RESOURCE_IPM_G723;
      m_ResourceList.rsList[3]          = RESOURCE_IPM_G711_10MS;
      m_ResourceList.rsList[4]          = RESOURCE_IPM_G711_20MS;
      m_ResourceList.rsList[5]          = RESOURCE_IPM_G711_30MS;
      m_ResourceList.rsList[6]          = RESOURCE_IPM_G722;
      m_ResourceList.rsList[7]          = RESOURCE_IPM_AMR_WB;
   }	
   m_ResourceList.version              = DEV_RESOURCE_LIST_VERSION;

   m_LocalSDP.clear();

   // clear the menu/clip play list
   m_playList.clear();
   strcpy(m_mmDevName, pMMDevName);
   m_bIsRecordPending                         = false;
   strcpy(m_ipmCCDevName, pIpmCCDevName);  // JS
   strcpy(m_ipmRTSPDevName, pIpmRTSPDevName);  // JS
   strcpy(m_voxDevName, pVoxDevName);

   m_hDMFAudFile=MTK_ERROR;
   m_hDMFVidFile=MTK_ERROR;
   m_h3GPFile=MTK_ERROR;

#ifdef TDM_AUDIO
   strcpy(m_dtiDevName, pDtiDevName);

   // building file names must go somewhere after a call offered!
   // No called number from ISDN at this point

   // Voice/video mail recordings are named with the 
   // called number,calling number and a timestamp
   // Should end up with a filename like "1112223333-4445556666-10-11-12-123"

   // this is just to initialize the file names
   struct  timeb timebuffer;
   ftime(&timebuffer);
   char timestamp[32];
   struct tm *timeParts = localtime (&(timebuffer.time));
   sprintf (timestamp, "-%d-%d-%d-%d\n", timeParts->tm_hour, timeParts->tm_min, timeParts->tm_sec, timebuffer.millitm);
   std::string timeStamp(timestamp);
   std::string cAudioRecordFileName = "x" + timeStamp;
   std::string cVideoRecordFileName = "x" + timeStamp;
#endif

#ifdef USE_RTSP
   m_pRTSPClient = NULL;
#endif

#ifdef USE_MTK_API
   // SKS Image Overlay
   if (g_nVideoTranscodeDir==RXTX || g_nVideoTranscodeDir==TX) {
      BuildOverlay();
   }
#endif
}


#ifdef USE_MTK_API
void CMMStream::BuildOverlay()
{
   MTK_ERROR_INFO *mtkErrorInfo;
   char str[256];
   char imageFileName[20];
   sprintf(imageFileName,"%s.jpg",m_mmDevName);
   eMTK_POSITION_TYPE ePositionType = eMTK_POSITION_TYPE_JUSTIFICATION;
   float x = MTK_JUSTIFY_CENTER;
   float y = MTK_JUSTIFY_BOTTOM;

   // Create Image Template Handle of type JPEG or YUV
   hMtkImageTemplate = mtk_CreateImageTemplate(eMTK_IMAGE_FORMAT_JPEG);
   if(hMtkImageTemplate==MTK_ERROR) {
      mtk_GetErrorInfo(mtkErrorInfo);
      sprintf(str,"mtk_CreateImageTemplate() failed with error: ErrCode=%d|ErrStr=%s|ExErrInfoStr=%s",
                   mtkErrorInfo->unErrorCode,mtkErrorInfo->szErrorString,mtkErrorInfo->szAdditionalInfo);
      mmReport(INFO_DEBUG, s_eType, str);
   }

   // Create Media Template File Handle of xyz.jpg file using Iamge Template handle of JPEG or YUV
   hMtkMediaFileTemplate = mtk_CreateMediaFileTemplate(hMtkImageTemplate, imageFileName);
   if(hMtkMediaFileTemplate==MTK_ERROR) {
      mtk_GetErrorInfo(mtkErrorInfo);
      sprintf(str,"hMtkMediaFileTemplate() failed with error: ErrCode=%d|ErrStr=%s|ExErrInfoStr=%s",
                   mtkErrorInfo->unErrorCode,mtkErrorInfo->szErrorString,mtkErrorInfo->szAdditionalInfo);
      mmReport(INFO_DEBUG, s_eType, str);
   }

   // Create Image Overlay Template handle from Media Template File Handle
   hObImageOverlayTemplate = ob_CreateImageOverlayTemplate(hMtkMediaFileTemplate);
   if(hObImageOverlayTemplate==MTK_ERROR) {
      mtk_GetErrorInfo(mtkErrorInfo);
      sprintf(str,"hObImageOverlayTemplate() failed with error: ErrCode=%d|ErrStr=%s|ExErrInfoStr=%s",
                   mtkErrorInfo->unErrorCode,mtkErrorInfo->szErrorString,mtkErrorInfo->szAdditionalInfo);
      mmReport(INFO_DEBUG, s_eType, str);
   }
	
   // Create Frame Template Handle for Image which represents a rectagular region on the video screen
   hMtkFrameTemplate = mtk_CreateFrameTemplate();
   if(hMtkFrameTemplate==MTK_ERROR) {
      mtk_GetErrorInfo(mtkErrorInfo);
      sprintf(str,"mtk_CreateFrameTemplate() failed with error: ErrCode=%d|ErrStr=%s|ExErrInfoStr=%s",                                     mtkErrorInfo->unErrorCode,mtkErrorInfo->szErrorString,mtkErrorInfo->szAdditionalInfo);
      mmReport(INFO_DEBUG, s_eType, str);
   }

   mtk_SetFramePosition(hMtkFrameTemplate, x, y, ePositionType);
   mtk_SetFrameSize(hMtkFrameTemplate, 40, 30,eMTK_SIZE_TYPE_PIXEL);
   //ob_SetOverlayFillStyle(hMtkFrameTemplate,eOB_FILL_STYLE_MAINTAIN_ASPECT_RATIO);
   ob_SetOverlayFillStyle(hMtkFrameTemplate,eOB_FILL_STYLE_RESIZE_TO_FIT);


   // Bound the Image Overlay Template handle into Frame Template Handle
   hObOverlayBoundingFrame = ob_SetOverlayBoundingFrame(hObImageOverlayTemplate, hMtkFrameTemplate);
   if(hObOverlayBoundingFrame==MTK_ERROR) {
      mtk_GetErrorInfo(mtkErrorInfo);
      sprintf(str,"ob_SetOverlayBoundingFrame() failed with error: ErrCode=%d|ErrStr=%s|ExErrInfoStr=%s",
                   mtkErrorInfo->unErrorCode,mtkErrorInfo->szErrorString,mtkErrorInfo->szAdditionalInfo);
      mmReport(INFO_DEBUG, s_eType, str);
   }

}

void CMMStream::ApplyOverlay()
{
   SM_ADD_OVERLAY_LIST overlayList;
   INIT_SM_ADD_OVERLAY_LIST(&overlayList);

   overlayList.unCount = 1;
   overlayList.addOverlays[0].eDirection = eSM_OVERLAY_DIRECTION_DEVICE;
   overlayList.addOverlays[0].hOverlay = hObImageOverlayTemplate;

   if (sm_AddOverlays(m_mmH, &overlayList, NULL) != MTK_SUCCESS)
   {
      mmReport(INFO_DEBUG, s_eType, "sm_AddOverlays() failed on device %s",m_mmDevName);
   }
}

void CMMStream::RemoveOverlay()
{
   SM_REMOVE_OVERLAY_LIST overlayList;
   INIT_SM_REMOVE_OVERLAY_LIST(&overlayList);
   overlayList.unCount = 1;
   overlayList.overlayHandles[0] =hObImageOverlayTemplate;

   //if (sm_RemoveOverlays(m_mmH, &overlayList, NULL) == MTK_SUCCESS)
   if (sm_RemoveAllOverlays(m_mmH, NULL) != MTK_SUCCESS)
   {
      mmReport(INFO_DEBUG, s_eType, "sm_RemoveAllOverlays() failed on device %s",m_mmDevName);
   }
}

void CMMStream::DestroyOverlay()
{
   if(mtk_DestroyMediaTemplate(hMtkMediaFileTemplate)!=MTK_SUCCESS) {
      mmReport(INFO_DEBUG, s_eType, "mtk_DestroyMediaTemplate() failed on device %s",m_mmDevName);		
   }
   if(ob_DestroyOverlayTemplate(hObImageOverlayTemplate)!=MTK_SUCCESS) {
      mmReport(INFO_DEBUG, s_eType, "ob_DestroyOverlayTemplate() failed on device %s",m_mmDevName);		
   }
   if(mtk_DestroyFrameTemplate(hMtkFrameTemplate)!=MTK_SUCCESS) {
      mmReport(INFO_DEBUG, s_eType, "mtk_DestroyFrameTemplate() failed on device %s",m_mmDevName);			   }
}
#endif

void CMMStream::SetPlayRecordFileName(char extention[12])
{
   // Setup file paths appropriately -
   std::string filePath = g_cAVFileDir;
	
   if(g_nVideoTranscodeDir==RXTX || g_nVideoTranscodeDir==TX || m_bVideoCall==false) {
      if(g_nVideoCodec==H263 || g_nVideoCodec==H263_1998 || g_nVideoCodec==H263_2000) {
         if(g_nVideoRes==QCIF)
            filePath = filePath + "/lores/";
         else
            filePath = filePath + "/hires/";
      } else if(g_nVideoCodec==MPEG4){
         if( g_nVideoRes==QCIF)
            filePath = filePath + "/mpeg4_lores/";
         else
            filePath = filePath + "/mpeg4_hires/";
      } else if(g_nVideoCodec==H264){
         if( g_nVideoRes==QCIF)
            filePath = filePath + "/h264_lores/";
         else
            filePath = filePath + "/h264_hires/";
      }
   } else if(g_nVideoTranscodeDir==NATIVE || g_nVideoTranscodeDir==RX)  {
      if(m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_H263 || m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_H263_1998) {
         if( m_stRemoteEPVideoCodec.m_nResolution==QCIF)
            filePath = filePath + "/lores/";
         else
            filePath = filePath + "/hires/";
      } else if(m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_MP4V_ES) {
         if( m_stRemoteEPVideoCodec.m_nProfileLevelID<=1)
            filePath = filePath + "/mpeg4_lores/";
         else
            filePath = filePath + "/mpeg4_hires/";
      } else if(m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_H264) {
          // Only based on local config, not based on remote signaling
         if( g_nVideoRes==QCIF)
         {
            filePath = filePath + "/h264_lores/";
         }
         else
         {
            filePath = filePath + "/h264_hires/";
         }
      }
   }
   mmReport(INFO_DEBUG, s_eType, "Setting file path to %s\n", filePath.c_str());

   // SKS Record TODO
   m_cVideoRecordFile = filePath + m_mmDevName + ".vid";

   if(g_nAudioFileFormat==MMD_WAV) {
      m_cAudioRecordFile = filePath + m_mmDevName + ".wav";
      m_cClip1Audio = filePath + s_cClip1Audio + ".wav";
      m_cClip2Audio = filePath + s_cClip2Audio + ".wav";
      m_cClip3Audio = filePath + s_cClip3Audio + ".wav";
      m_cMainMenuAudio = filePath + s_cMainMenuAudio + ".wav";
      m_cVmailMenuAudio = filePath + s_cVmailMenuAudio + ".wav";
      m_cVportalMenuAudio = filePath + s_cVportalMenuAudio + ".wav";
      if(g_bUseRecordMsg){
         m_cRecordingMsgAudioFile = filePath + s_cRecordingMsgAudio + ".wav";
      }
   } else if(g_nAudioFileFormat==MMD_3GP) {
      m_cAudioRecordFile = filePath + m_mmDevName + ".amr";
      m_cClip1Audio = filePath + s_cClip1Audio + extention;
      m_cClip2Audio = filePath + s_cClip2Audio + extention;
      m_cClip3Audio = filePath + s_cClip3Audio + extention;
      m_cMainMenuAudio = filePath + s_cMainMenuAudio + extention;
      m_cVmailMenuAudio = filePath + s_cVmailMenuAudio + extention;
      m_cVportalMenuAudio = filePath + s_cVportalMenuAudio + extention;
      if(g_bUseRecordMsg){
         m_cRecordingMsgAudioFile = filePath + s_cRecordingMsgAudio + extention;
      }
   } else { //MMD_DMF
      m_cAudioRecordFile = filePath + m_mmDevName + extention;
      m_cClip1Audio = filePath + s_cClip1Audio + extention;
      m_cClip2Audio = filePath + s_cClip2Audio + extention;
      m_cClip3Audio = filePath + s_cClip3Audio + extention;
      m_cMainMenuAudio = filePath + s_cMainMenuAudio + extention;
      m_cVmailMenuAudio = filePath + s_cVmailMenuAudio + extention;
      m_cVportalMenuAudio = filePath + s_cVportalMenuAudio + extention;
      if(g_bUseRecordMsg){
         m_cRecordingMsgAudioFile = filePath + s_cRecordingMsgAudio + extention;
      }
   }

   if(g_nVideoFileFormat==MMD_3GP) {
      m_cMainMenuVideo = filePath + s_cMainMenuVideo + extention;
      m_cVmailMenuVideo = filePath + s_cVmailMenuVideo + extention;
      m_cVportalMenuVideo = filePath + s_cVportalMenuVideo + extention;
      m_cClip1Video = filePath + s_cClip1Video + extention;
      m_cClip2Video = filePath + s_cClip2Video + extention;
      m_cClip3Video = filePath + s_cClip3Video + extention;
      m_c3GPFileName = filePath + m_mmDevName + extention;
      if(g_bUseRecordMsg){
         m_cRecordingMsgVideoFile = filePath + s_cRecordingMsgVideo + ".vid";
      }
   } else { //MMD_DMF
      m_cMainMenuVideo = filePath + s_cMainMenuVideo + ".vid";  
      m_cVmailMenuVideo = filePath + s_cVmailMenuVideo + ".vid";
      m_cVportalMenuVideo = filePath + s_cVportalMenuVideo + ".vid";
      m_cClip1Video = filePath + s_cClip1Video + ".vid";
      m_cClip2Video = filePath + s_cClip2Video + ".vid";
      m_cClip3Video = filePath + s_cClip3Video + ".vid";
      if(g_bUseRecordMsg){
         m_cRecordingMsgVideoFile = filePath + s_cRecordingMsgVideo + ".vid";
      }
   }

   m_cImageFileName = filePath + s_cImageFileName;

}

void CMMStream::ResetVariables()
{
// reset remote audio codec settings
   m_stRemoteEPAudioCodec.m_nAudioCodec=NO_AUDIO_CODEC;
   m_stRemoteEPAudioCodec.m_nDTMFCodec=-1;
   m_stRemoteEPAudioCodec.m_cCodecName[0]='\0';
   m_stRemoteEPAudioCodec.m_bVadEnabled=false;
   m_stRemoteEPAudioCodec.m_nMinFrameSize=CODER_FRAMESIZE_20;
   m_stRemoteEPAudioCodec.m_nMaxFrameSize=CODER_FRAMESIZE_20;
   m_stRemoteEPAudioCodec.m_nFPP=1;
   m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G711ALAW64K;
   m_stRemoteEPAudioCodec.m_nAudioFrameSize=CODER_FRAMESIZE_20;

   m_stRemoteEPAudioCodec.m_bAmrOctetAlign=true;
   m_stRemoteEPAudioCodec.m_nAmrModeSet=7;
   m_stRemoteEPAudioCodec.m_nAmrOption=(CODER_OPT_AMR_OCTET|CODER_OPT_AMR_CMR_TRACK);

   m_SdpDciLen=0;
   m_SdpDciStrValue[0]='\0';

// reset remote Video codec settings
   m_stRemoteEPVideoCodec.m_nVideoCodec=NO_VIDEO_CODEC;
   m_stRemoteEPVideoCodec.m_cCodecName[0]='\0';
   m_stRemoteEPVideoCodec.m_nProfileLevelID=-1;
   m_stRemoteEPVideoCodec.m_nBitRate=VIDEO_BITRATE_DEFAULT;
   m_stRemoteEPVideoCodec.m_nResolution=-1;
   m_stRemoteEPVideoCodec.m_SdpDciStrValue[0]='\0';
   m_stRemoteEPVideoCodec.m_SdpDciLen=0;
   m_stRemoteEPVideoCodec.m_nCoderType=CODER_TYPE_H263;
   m_stRemoteEPVideoCodec.m_nFrameRate=VIDEO_FRAMESPERSEC_15;
}

int CMMStream::MPIToFPS(int i) {
/*
   double x = 30.0/(1.001 * (double)i);
   int a = (int)x;
   int b = (int)((x-(double)a)*100.0);
   int c = a << 16 | b;
*/
   int c;

   switch (i) {
    case 1: c =  VIDEO_FRAMESPERSEC_2997;   	break;         /* 29.97 fps */
    case 2: c =  VIDEO_FRAMESPERSEC_15;      	break;        /* 15 fps */
    case 3: c =  VIDEO_FRAMESPERSEC_10;     	break;        /* 10 fps */
    case 5: c =  VIDEO_FRAMESPERSEC_6;       	break;       /* 6 fps  */
    default: c =  VIDEO_FRAMESPERSEC_DEFAULT;	break;
    }

   return c;
}

int CMMStream::FPSToMPI(eVIDEO_FRAMESPERSEC fps) 
{
   int nMPI;
  
   if(fps==VIDEO_FRAMESPERSEC_30) {
      nMPI=1;
   }else if(fps==VIDEO_FRAMESPERSEC_2997) {
      nMPI=1;
   }else if(fps==VIDEO_FRAMESPERSEC_15) {
      nMPI=2;
   }else if(fps==VIDEO_FRAMESPERSEC_10) {
      nMPI=3;
   }else if(fps==VIDEO_FRAMESPERSEC_6) {
      nMPI=5;
   } else {
      nMPI=0;
   }
   return nMPI;
}


#ifdef USE_RTSP
//*****************************************************************************
// 		  NAME : void CMMStream::SetupRTSPRemoteMediaInfo2()
// DESCRIPTION : Sets up the Remote Media Info structure
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::SetupRTSPRemoteMediaInfo2()
{
   // Local Variable Declaration
   int mediaCtr = 0;

   mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2()");

   // clear the RemoteMediaInfo structures
   memset(&m_RTSPipmRemoteMedia, 0, sizeof(IPM_MEDIA_INFO));

   // remote audio
   if ( strncmp(m_pRTSPClient->GetAudioCodecName(), "AMR", 3)==0)
   {
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote audio, mediaCtr %d, AMR codec.\n", mediaCtr);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_REMOTE_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = CODER_TYPE_AMRNB_12_2k;

      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.eFrameSize = CODER_FRAMESIZE_20;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unFramesPerPkt = 10;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.eVadEnable = CODER_VAD_ENABLE;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unCoderPayloadType = m_pRTSPClient->GetAudioPayloadType();
      // Jim O says 97
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unRedPayloadType = 97;
      ++mediaCtr;

      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote audio mediaCtr %d, AMR codec options.\n", mediaCtr);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_REMOTE_CODER_OPTIONS_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unVersion =
      IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unCoderOptions= CODER_OPT_AMR_CMR_TRACK | CODER_OPT_AMR_OCTET; //CODER_OPT_AMR_EFFICIENT; 
      ++mediaCtr;
   }
   else
   {
      // G.711
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote audio, mediaCtr %d, g711 codec.\n", mediaCtr);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                              = MEDIATYPE_AUDIO_REMOTE_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = CODER_TYPE_G711ULAW64K;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.eFrameSize = (eIPM_CODER_FRAMESIZE)m_stRemoteEPAudioCodec.m_nAudioFrameSize;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.unFramesPerPkt= 1;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.eVadEnable  = CODER_VAD_DISABLE;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.unCoderPayloadType    = 0;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.unRedPayloadType  = 0;
      ++mediaCtr;
   }

   // local audio
   if ( strncmp(m_pRTSPClient->GetAudioCodecName(), "AMR", 3)==0)
   {
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local audio, mediaCtr %d, AMR codec.\n", mediaCtr);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = CODER_TYPE_AMRNB_12_2k;

      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.eFrameSize = CODER_FRAMESIZE_20;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unFramesPerPkt = 10;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.eVadEnable = CODER_VAD_ENABLE;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unCoderPayloadType = m_pRTSPClient->GetAudioPayloadType();
      // Jim O says 97
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unRedPayloadType = 97;
      //m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unRedPayloadType = 0;
      ++mediaCtr;

      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local audio, mediaCtr %d, AMR codec options.\n", mediaCtr);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_OPTIONS_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unVersion =
      IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unCoderOptions=
      CODER_OPT_AMR_CMR_TRACK | CODER_OPT_AMR_OCTET; //CODER_OPT_AMR_EFFICIENT;
      ++mediaCtr;
   }
   else
   {
      // G.711
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local audio, mediaCtr %d, g711 codec.\n", mediaCtr);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = CODER_TYPE_G711ULAW64K;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.eFrameSize = (eIPM_CODER_FRAMESIZE)m_stRemoteEPAudioCodec.m_nAudioFrameSize; 
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.unFramesPerPkt = 1;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.eVadEnable = CODER_VAD_DISABLE;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.unCoderPayloadType  = 0;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.CoderInfo.unRedPayloadType = 0;
      ++mediaCtr;
   }


   mmReport(INFO_DEBUG, s_eType,"%s\n", m_pRTSPClient->GetVideoCodecName());
   if (strncmp(m_pRTSPClient->GetVideoCodecName(), "MP4V", 4)==0) {
      // remote video
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video, mediaCtr %d, mp4v codec.\n", mediaCtr);
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_REMOTE_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_MP4V_ES;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = m_pRTSPClient->GetVideoPayloadType();
      INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoRemote);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoRemote;
      if (m_pRTSPClient->GetRTSPVideoProfile() == 3) {
         mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video, mp4v profile 3\n");
         m_extraVidInfoRemote.eProfile = VIDEO_PROFILE_LEVEL_SP3_MPEG4;
	 m_extraVidInfoRemote.eImageWidth = VIDEO_IMAGE_WIDTH_352;
	 m_extraVidInfoRemote.eImageHeight = VIDEO_IMAGE_HEIGHT_288;
	 m_extraVidInfoRemote.unBitRate = 384000;
	 m_extraVidInfoRemote.eFramesPerSec = VIDEO_FRAMESPERSEC_30;
	 m_extraVidInfoRemote.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoRemote.unVisualConfigSize=g_cTxDciCifSize;
         m_extraVidInfoRemote.szVisualConfiguration=g_cTxDciCifValue;
      } else {
         mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video, mp4v profile not 3\n");
         m_extraVidInfoRemote.eProfile = VIDEO_PROFILE_LEVEL_SP0_MPEG4;
	 m_extraVidInfoRemote.eImageWidth = VIDEO_IMAGE_WIDTH_176;
	 m_extraVidInfoRemote.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
	 m_extraVidInfoRemote.unBitRate = 40000;
	 m_extraVidInfoRemote.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
	 m_extraVidInfoRemote.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoRemote.unVisualConfigSize=g_cTxDciQcifSize;
         m_extraVidInfoRemote.szVisualConfiguration=g_cTxDciQcifValue;
      }
      ++mediaCtr;
      // local video coder
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local video, mediaCtr %d, mp4v codec.\n", mediaCtr);
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_LOCAL_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_MP4V_ES;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = m_pRTSPClient->GetVideoPayloadType();

      INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoLocal);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoLocal;
      if (m_pRTSPClient->GetRTSPVideoProfile() == 3) {
         mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local video, mp4v profile 3\n");
         m_extraVidInfoLocal.eImageWidth = VIDEO_IMAGE_WIDTH_352;
	 m_extraVidInfoLocal.eImageHeight = VIDEO_IMAGE_HEIGHT_288;
	 m_extraVidInfoLocal.unBitRate = 384000;
	 m_extraVidInfoLocal.eFramesPerSec = VIDEO_FRAMESPERSEC_30;
	 m_extraVidInfoLocal.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoLocal.unVisualConfigSize=g_cRxDciCifSize;
         m_extraVidInfoLocal.szVisualConfiguration=g_cRxDciCifValue;
      } else { // assuming QCIF
         mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local video, mp4v profile not 3\n");
         m_extraVidInfoLocal.eProfile = VIDEO_PROFILE_LEVEL_SP0_MPEG4;
	 m_extraVidInfoLocal.eLevel = VIDEO_LEVEL_10_H263 ;
	 m_extraVidInfoLocal.eImageWidth = VIDEO_IMAGE_WIDTH_176;
	 m_extraVidInfoLocal.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
	 m_extraVidInfoLocal.unBitRate = 40000;
	 m_extraVidInfoLocal.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
	 m_extraVidInfoLocal.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoLocal.unVisualConfigSize=g_cRxDciQcifSize;
         m_extraVidInfoLocal.szVisualConfiguration=g_cRxDciQcifValue;
      }
      ++mediaCtr;

   } else if (strncmp(m_pRTSPClient->GetVideoCodecName(), "H264", 4)==0) {
      // remote video
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video, mediaCtr %d, H264 codec.\n", mediaCtr);
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_REMOTE_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_H264;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = m_pRTSPClient->GetVideoPayloadType();
      INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoRemote);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoRemote;

      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video, H264 profile <%d>\n", m_pRTSPClient->GetRTSPVideoProfile());
      m_extraVidInfoRemote.eProfile = VIDEO_PROFILE_BASELINE_H264;
      m_extraVidInfoRemote.eLevel = VIDEO_LEVEL_1_H264 ;
      m_extraVidInfoRemote.eImageWidth = VIDEO_IMAGE_WIDTH_176;
      m_extraVidInfoRemote.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
      m_extraVidInfoRemote.unBitRate = 40000;
      m_extraVidInfoRemote.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
      m_extraVidInfoRemote.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
      m_extraVidInfoRemote.unVisualConfigSize=g_cTxDciQcifSize;
      m_extraVidInfoRemote.szVisualConfiguration=g_cTxDciQcifValue;
      ++mediaCtr;

      // local video coder
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local video, mediaCtr %d, h264 codec.\n", mediaCtr);
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_LOCAL_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_H264;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = m_pRTSPClient->GetVideoPayloadType();
      INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoLocal);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoLocal;

      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video, H264 profile <%d>\n", m_pRTSPClient->GetRTSPVideoProfile());
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local video, mp4v profile not 3\n");
      m_extraVidInfoLocal.eProfile = VIDEO_PROFILE_BASELINE_H264;
      m_extraVidInfoLocal.eLevel = VIDEO_LEVEL_1_H264 ;
      m_extraVidInfoLocal.eImageWidth = VIDEO_IMAGE_WIDTH_176;
      m_extraVidInfoLocal.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
      m_extraVidInfoLocal.unBitRate = 40000;
      m_extraVidInfoLocal.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
      m_extraVidInfoLocal.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
      m_extraVidInfoLocal.unVisualConfigSize=g_cRxDciQcifSize;
      m_extraVidInfoLocal.szVisualConfiguration=g_cRxDciQcifValue;
      ++mediaCtr;

   } else if (strncmp(m_pRTSPClient->GetVideoCodecName(), "H263-1998", 9)==0) {
      // remote video
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video, mediaCtr %d, h263-1998 codec.\n", mediaCtr);
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_REMOTE_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_H263_1998;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = m_pRTSPClient->GetVideoPayloadType(); 

      if (g_nVideoTranscodeDir==RXTX) {
         INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoLocal);
         m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoLocal;
         m_extraVidInfoRemote.eProfile = VIDEO_PROFILE_0_H263;
         m_extraVidInfoLocal.eLevel = VIDEO_LEVEL_10_H263;
         m_extraVidInfoRemote.eImageWidth = VIDEO_IMAGE_WIDTH_176;
         m_extraVidInfoRemote.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
         m_extraVidInfoRemote.unBitRate = 40000;
         m_extraVidInfoRemote.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
         m_extraVidInfoRemote.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoRemote.unVisualConfigSize = 0;
         m_extraVidInfoRemote.szVisualConfiguration = NULL;
      }
      ++mediaCtr;

      // local video coder
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local video, mediaCtr %d, h263-1998 codec.\n", mediaCtr);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_LOCAL_CODER_INFO;
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_H263_1998;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = m_pRTSPClient->GetVideoPayloadType(); 
      if (g_nVideoTranscodeDir==RXTX) {
         INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoLocal);
         m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoLocal;
         m_extraVidInfoLocal.eProfile = VIDEO_PROFILE_0_H263;
         m_extraVidInfoLocal.eImageWidth = VIDEO_IMAGE_WIDTH_176;
         m_extraVidInfoLocal.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
         m_extraVidInfoLocal.unBitRate = 40000;
         m_extraVidInfoLocal.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
         m_extraVidInfoLocal.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoLocal.unVisualConfigSize = 0;
         m_extraVidInfoLocal.szVisualConfiguration = NULL;
      }
      ++mediaCtr;

   } else if (strncmp(m_pRTSPClient->GetVideoCodecName(), "H263-2000", 9)==0) {
      // remote video
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video, mediaCtr %d, h263-1998 codec.\n", mediaCtr);
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_REMOTE_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_H263_1998;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = m_pRTSPClient->GetVideoPayloadType(); 
      if (g_nVideoTranscodeDir==RXTX) {
         INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoLocal);
         m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoLocal;
         m_extraVidInfoRemote.eProfile = VIDEO_PROFILE_0_H263;
         m_extraVidInfoLocal.eLevel = VIDEO_LEVEL_10_H263;
         m_extraVidInfoRemote.eImageWidth = VIDEO_IMAGE_WIDTH_176;
         m_extraVidInfoRemote.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
         m_extraVidInfoRemote.unBitRate = 40000;
         m_extraVidInfoRemote.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
         m_extraVidInfoRemote.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoRemote.unVisualConfigSize = 0;
         m_extraVidInfoRemote.szVisualConfiguration = NULL;
      }
      ++mediaCtr;

      // local video coder
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local video, mediaCtr %d, h263-1998 codec.\n", mediaCtr);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_LOCAL_CODER_INFO;
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_H263_1998;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = m_pRTSPClient->GetVideoPayloadType(); 
      if (g_nVideoTranscodeDir==RXTX) {
         INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoLocal);
         m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoLocal;
         m_extraVidInfoLocal.eProfile = VIDEO_PROFILE_0_H263;
         m_extraVidInfoLocal.eImageWidth = VIDEO_IMAGE_WIDTH_176;
         m_extraVidInfoLocal.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
         m_extraVidInfoLocal.unBitRate = 40000;
         m_extraVidInfoLocal.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
         m_extraVidInfoLocal.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoLocal.unVisualConfigSize = 0;
         m_extraVidInfoLocal.szVisualConfiguration = NULL;
      }
      ++mediaCtr;

   } else if (strncmp(m_pRTSPClient->GetVideoCodecName(), "H263", 4)==0) {
      // remote video
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video, mediaCtr %d, h263 codec.\n", mediaCtr);
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_REMOTE_CODER_INFO;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_H263;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = m_pRTSPClient->GetVideoPayloadType(); 
      if (g_nVideoTranscodeDir==RXTX) {
         INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoLocal);
         m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoLocal;
         m_extraVidInfoRemote.eProfile = VIDEO_PROFILE_0_H263;
         m_extraVidInfoLocal.eLevel = VIDEO_LEVEL_10_H263;
         m_extraVidInfoRemote.eImageWidth = VIDEO_IMAGE_WIDTH_176;
         m_extraVidInfoRemote.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
         m_extraVidInfoRemote.unBitRate = 40000;
         m_extraVidInfoRemote.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
         m_extraVidInfoRemote.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoRemote.unVisualConfigSize = 0;
         m_extraVidInfoRemote.szVisualConfiguration = NULL;
      }
      ++mediaCtr;

      // local video coder
      mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), local video, mediaCtr %d, h263 codec.\n", mediaCtr);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType                                    = MEDIATYPE_VIDEO_LOCAL_CODER_INFO;
      INIT_IPM_VIDEO_CODER_INFO(&m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           = CODER_TYPE_H263;
      m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   = 34; 
      if (g_nVideoTranscodeDir==RXTX) {
         INIT_IPM_VIDEO_CODER_INFO_EX(&m_extraVidInfoLocal);
         m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = &m_extraVidInfoLocal;
         m_extraVidInfoLocal.eProfile = VIDEO_PROFILE_0_H263;
         m_extraVidInfoLocal.eImageWidth = VIDEO_IMAGE_WIDTH_176;
         m_extraVidInfoLocal.eImageHeight = VIDEO_IMAGE_HEIGHT_144;
         m_extraVidInfoLocal.unBitRate = 40000;
         m_extraVidInfoLocal.eFramesPerSec = VIDEO_FRAMESPERSEC_15;
         m_extraVidInfoLocal.eSamplingRate = VIDEO_SAMPLING_RATE_DEFAULT;
         m_extraVidInfoLocal.unVisualConfigSize = 0;
         m_extraVidInfoLocal.szVisualConfiguration = NULL;
      }
      ++mediaCtr;
   }

   m_nRTSPAudioRtpPort = m_pRTSPClient->GetRTSPAudioPort();
   m_nRTSPVideoRtpPort = m_pRTSPClient->GetRTSPVideoPort();

   m_pRTSPClient->PrintAudioSessionInfo();
   m_pRTSPClient->PrintVideoSessionInfo();
   // remote audio port
   mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote audio rtp port, mediaCtr %d.\n", mediaCtr);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_REMOTE_RTP_INFO;
   strcpy(m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_cRTSPAudioVideoIp);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_nRTSPAudioRtpPort; 

   mmReport(INFO_DEBUG, s_eType, "Audio IpAddress: %s Audio RTP Port: %d", m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId);
   ++mediaCtr;
   
   // Local audio port
   mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), Local audio rtp port, mediaCtr %d.\n", mediaCtr);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_LOCAL_RTP_INFO;
   strcpy(m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress,  m_RTSPipmLocalMedia.MediaData[m_unLocalRTSPIPMAudioRtpPortIndex].mediaInfo.PortInfo.cIPAddress);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_RTSPipmLocalMedia.MediaData[m_unLocalRTSPIPMAudioRtpPortIndex].mediaInfo.PortInfo.unPortId; // m_nRTSPAudioRtpPort; 

   mmReport(INFO_DEBUG, s_eType, "Audio IpAddress: %s Audio Local RTP Port: %d", m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId);
   ++mediaCtr;


   mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote audio rtcp port, mediaCtr %d.\n", mediaCtr);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_REMOTE_RTCP_INFO;
   strcpy(m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_cRTSPAudioVideoIp);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_nRTSPAudioRtpPort+1; 
   ++mediaCtr;

   // remote video port
   mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video rtp port, mediaCtr %d.\n", mediaCtr);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_VIDEO_REMOTE_RTP_INFO;
   strcpy(m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_cRTSPAudioVideoIp);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_nRTSPVideoRtpPort; 

   mmReport(INFO_DEBUG, s_eType, "Video IpAddress: %s Video RTP Port: %d", m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId);
   ++mediaCtr;

   // Local video port
   mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), Local video rtp port, mediaCtr %d.\n", mediaCtr);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_VIDEO_LOCAL_RTP_INFO;
   strcpy(m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress,  m_RTSPipmLocalMedia.MediaData[m_unLocalRTSPIPMVideoRtpPortIndex].mediaInfo.PortInfo.cIPAddress);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_RTSPipmLocalMedia.MediaData[m_unLocalRTSPIPMVideoRtpPortIndex].mediaInfo.PortInfo.unPortId; // JSm_nRTSPVideoRtpPort; 


   mmReport(INFO_DEBUG, s_eType, "Local Video IpAddress: %s Video RTP Port: %d", m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId);
   ++mediaCtr;

   mmReport(INFO_DEBUG, s_eType, "SetupRTSPRemoteMediaInfo2(), remote video rtp port, mediaCtr %d.\n", mediaCtr);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].eMediaType = MEDIATYPE_VIDEO_REMOTE_RTCP_INFO;
   strcpy(m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_cRTSPAudioVideoIp);
   m_RTSPipmRemoteMedia.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_nRTSPVideoRtpPort+1; 
   ++mediaCtr;

   m_RTSPipmRemoteMedia.unCount = mediaCtr ;
   mmReport(INFO_DEBUG, s_eType, "m_RTSPipmRemoteMedia.unCount = %d", m_RTSPipmRemoteMedia.unCount);
}
#endif

//*****************************************************************************
// 		  NAME : void CMMStream::PrintRemoteLocalInfo   
// DESCRIPTION : Prints Remote and LocalIpm Session Info
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::PrintRemoteLocalInfo()
{
   mmReport(INFO_DEBUG, s_eType,"PrintRemoteLocalInfo()");
   unsigned int ctr;
   char *tempStr;
   tempStr=(char *)malloc(1024);
   for(ctr=0;ctr<m_ccIpmMediaInfo.unCount;ctr++) {
      switch (m_ccIpmMediaInfo.MediaData[ctr].eMediaType) 
      {
         case MEDIATYPE_AUDIO_REMOTE_RTP_INFO:
         {
            sprintf(tempStr,"Remote Audio RTP Info: IpAddress = %s | PortID = %u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.cIPAddress,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.unPortId);
            break;
         }
         case MEDIATYPE_AUDIO_LOCAL_RTP_INFO:
         {
            sprintf(tempStr,"Local Audio RTP Info: IpAddress = %s | PortID = %u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.cIPAddress,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.unPortId);
            break;
         }
         case MEDIATYPE_AUDIO_REMOTE_RTCP_INFO:
         {
            sprintf(tempStr,"Remote Audio RTCP Info: IpAddress = %s | PortID = %u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.cIPAddress,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.unPortId);
            break;
         }
         case MEDIATYPE_AUDIO_LOCAL_RTCP_INFO:
         {
            sprintf(tempStr,"Local Audio RTCP Info: IpAddress = %s | PortID = %u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.cIPAddress,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.unPortId);
            break;
         }
         case MEDIATYPE_AUDIO_REMOTE_CODER_INFO:
         {
            sprintf(tempStr,"Remote Audio Info: CoderType=%d | Pt=%u | FrmSz=%d | FPP=%u | Vad=%d",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.CoderInfo.eCoderType,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderInfo.unCoderPayloadType,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderInfo.eFrameSize,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderInfo.unFramesPerPkt,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderInfo.eVadEnable);
            break;
         }
         case MEDIATYPE_AUDIO_LOCAL_CODER_INFO:
         {
            sprintf(tempStr,"Local Audio Info: CoderType=%d | Pt=%u | FrmSz=%d | FPP=%u | Vad=%d",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.CoderInfo.eCoderType,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderInfo.unCoderPayloadType,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderInfo.eFrameSize,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderInfo.unFramesPerPkt,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderInfo.eVadEnable);
            break;
         }
         case MEDIATYPE_AUDIO_REMOTE_CODER_OPTIONS_INFO:
         {
            sprintf(tempStr,"Remote Audio Coder Option Info: Coder Option =%u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderOptionsInfo.unCoderOptions);
            break;
         }
         case MEDIATYPE_AUDIO_LOCAL_CODER_OPTIONS_INFO:
         {
            sprintf(tempStr,"Local Audio Coder Option Info: Coder Option =%u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.AudioCoderOptionsInfo.unCoderOptions);
            break;
         }
         case MEDIATYPE_VIDEO_REMOTE_RTP_INFO:
         {
            sprintf(tempStr,"Remote Video RTP Info: IpAddress = %s | PortID = %u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.cIPAddress,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.unPortId);
            break;
         }
         case MEDIATYPE_VIDEO_LOCAL_RTP_INFO:
         {
            sprintf(tempStr,"Local Video RTP Info: IpAddress = %s | PortID = %u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.cIPAddress,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.unPortId);
            break;
         }
         case MEDIATYPE_VIDEO_REMOTE_RTCP_INFO:
         {
            sprintf(tempStr,"Remote Video RTCP Info: IpAddress = %s | PortID = %u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.cIPAddress,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.unPortId);
            break;
         }
         case MEDIATYPE_VIDEO_LOCAL_RTCP_INFO:
         {
            sprintf(tempStr,"Local Video RTCP Info: IpAddress = %s | PortID = %u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.cIPAddress,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.PortInfo.unPortId);
            break;
         }
         case MEDIATYPE_VIDEO_REMOTE_CODER_INFO:
         {
            sprintf(tempStr,"Remote Video Info: CoderT=%d | Pt=%u | Prof=%d | Lvl=%d | Resln=(%dx%d) | SamplingRate=%d | Rate=%uK | FPS=%d | DciSz=%u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.eCoderType,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.unCoderPayloadType,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eProfile,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageWidth,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageHeight,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eSamplingRate,
                            (m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->unBitRate/1000),
                            (30/FPSToMPI(m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eFramesPerSec)),
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->unVisualConfigSize);
            break;
         }
         case MEDIATYPE_VIDEO_LOCAL_CODER_INFO:
         {
// sprintf(tempStr,"Local Video Info: CoderType=%d | Pt=%u",m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType,m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType);

            sprintf(tempStr,"Local Video Info : CoderT=%d | Pt=%u | Prof=%d | Lvl=%d | Resln=(%dx%d) | SamplingRate=%d | Rate=%uK | FPS=%d | DciSz=%u",
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.eCoderType,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.unCoderPayloadType,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eProfile,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageWidth,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageHeight,
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eSamplingRate,
                            (m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->unBitRate/1000),
                            (30/FPSToMPI(m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eFramesPerSec)),
                             m_ccIpmMediaInfo.MediaData[ctr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->unVisualConfigSize);
            break;
         }
         default:
         break;
      } // end of switch case statement
      mmReport(INFO_DEBUG, s_eType,"%s",tempStr);
      tempStr[0]='\0';
   }	// end of for loop
   free(tempStr);
}

//*****************************************************************************
// 		  NAME : void CMMStream::SetupExtraVideoCoderInfo()
// DESCRIPTION : Sets VideoCoderInfo.pExtraCoderInfo struct
// 		 INPUT : mediaCtr - media index count
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::SetupExtraVideoCoderInfo(int mediaCtr)
{

   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo = new IPM_VIDEO_CODER_INFO_EX;
   INIT_IPM_VIDEO_CODER_INFO_EX( m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo);
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->unVersion = IPM_VIDEO_CODER_INFO_EX_VERSION;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->unVisualConfigSize=0;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->szVisualConfiguration  = NULL;

   // used by decoder
   if(m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_MP4V_ES) {
      if(m_stRemoteEPVideoCodec.m_nBitRate > VIDEO_BITRATE_128K) {
         m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eProfile = VIDEO_PROFILE_LEVEL_SP1_MPEG4;
      } else if(m_stRemoteEPVideoCodec.m_nBitRate >VIDEO_BITRATE_64K) {
         m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eProfile = VIDEO_PROFILE_LEVEL_SP0_MPEG4;
      } else {
         m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eProfile = VIDEO_PROFILE_LEVEL_SP0_MPEG4;
      }
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel = VIDEO_LEVEL_DEFAULT;

   } else if(m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_H264) {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eProfile = VIDEO_PROFILE_BASELINE_H264;
      if(m_stRemoteEPVideoCodec.m_nBitRate <= VIDEO_BITRATE_64K)  {
          m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel = VIDEO_LEVEL_1_H264;
      } else if ( (m_stRemoteEPVideoCodec.m_nBitRate > VIDEO_BITRATE_64K) && (m_stRemoteEPVideoCodec.m_nBitRate <= VIDEO_BITRATE_128K) ) {
          m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel = VIDEO_LEVEL_1_B_H264;
      }else if ( (m_stRemoteEPVideoCodec.m_nBitRate > VIDEO_BITRATE_128K) && (m_stRemoteEPVideoCodec.m_nBitRate <= 192000) ) {
          m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel = VIDEO_LEVEL_1_1_H264;
      } else if ( (m_stRemoteEPVideoCodec.m_nBitRate > 192000) && (m_stRemoteEPVideoCodec.m_nBitRate <= VIDEO_BITRATE_384K) ) {
          m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel = VIDEO_LEVEL_1_2_H264;
      } else {
          m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel = VIDEO_LEVEL_1_3_H264;
      }

   } else {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eProfile = VIDEO_PROFILE_0_H263;
      if(m_stRemoteEPVideoCodec.m_nBitRate > VIDEO_BITRATE_128K) {
         m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel = VIDEO_LEVEL_30_H263;
      } else if(m_stRemoteEPVideoCodec.m_nBitRate >VIDEO_BITRATE_64K) {
         m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel = VIDEO_LEVEL_20_H263;
      } else {
         m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eLevel = VIDEO_LEVEL_10_H263;
      }
   }
      
   if(m_stRemoteEPVideoCodec.m_nResolution==CIF || m_stRemoteEPVideoCodec.m_nProfileLevelID>1) {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageWidth = VIDEO_IMAGE_WIDTH_352;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageHeight = VIDEO_IMAGE_HEIGHT_288;
   } else if(m_stRemoteEPVideoCodec.m_nResolution==QCIF || m_stRemoteEPVideoCodec.m_nProfileLevelID<2)  {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageWidth = VIDEO_IMAGE_WIDTH_176;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageHeight = VIDEO_IMAGE_HEIGHT_144;
   } else if(m_stRemoteEPVideoCodec.m_nResolution==SQCIF) {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageWidth = EMM_VIDEO_IMAGE_WIDTH_128;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eImageHeight = VIDEO_IMAGE_HEIGHT_96;
   }

   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->unBitRate = m_stRemoteEPVideoCodec.m_nBitRate;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eFramesPerSec = m_stRemoteEPVideoCodec.m_nFrameRate;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->eSamplingRate = VIDEO_SAMPLING_RATE_90000;

   if(m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_MP4V_ES && m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType==MEDIATYPE_VIDEO_LOCAL_CODER_INFO) {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->unVisualConfigSize = m_stRemoteEPVideoCodec.m_SdpDciLen;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->szVisualConfiguration = m_stRemoteEPVideoCodec.m_SdpDciValue;
   } else {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->unVisualConfigSize = m_SdpDciLen;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.pExtraCoderInfo->szVisualConfiguration = m_SdpDciValue;
   }
}

//*****************************************************************************
// 		  NAME : void CMMStream::SetupCCRemoteMediaInfo()
// DESCRIPTION : Sets up the Media Info structure
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::SetupCCRemoteMediaInfo()
{
   // Local Variable Declaration
   int mediaCtr = 0;

   mmReport(INFO_DEBUG, s_eType, "SetupCCMediaInfo()");

   memset(&m_ccIpmMediaInfo, 0, sizeof(IPM_MEDIA_INFO));

   // Remote Audio information used by encoder
   m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_REMOTE_CODER_INFO;
#if 1 //JM - Use Native
   // Use Native codec in ipmStart if not transcode so lic resource is not used.
   if (g_nAudioTranscodeDir==NATIVE || g_nAudioTranscodeDir==RX) {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = static_cast<eIPM_CODER_TYPE> (m_stRemoteEPAudioCodec.m_nCoderType | CODER_TYPE_NATIVE);
   } else {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = m_stRemoteEPAudioCodec.m_nCoderType;
   } 
#else
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = m_stRemoteEPAudioCodec.m_nCoderType;
#endif
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.eFrameSize = m_stRemoteEPAudioCodec.m_nAudioFrameSize;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unFramesPerPkt = 1; // m_stRemoteEPAudioCodec.m_nFPP;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.eVadEnable = ((m_stRemoteEPAudioCodec.m_bVadEnabled==true)?CODER_VAD_ENABLE:CODER_VAD_DISABLE);
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unCoderPayloadType = m_stRemoteEPAudioCodec.m_nAudioCodec;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unRedPayloadType = 97;
   // Remote Audio Options information used by encoder if applicable (for AMR or EVRC)
   if(strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr")==0 || strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-nb")==0) {
      mediaCtr+=1;
      m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_REMOTE_CODER_OPTIONS_INFO;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unVersion = IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unCoderOptions= m_stRemoteEPAudioCodec.m_nAmrOption;
   } else if(strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amrwb")==0 || strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-wb")==0) {
      mediaCtr+=1;
      m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_REMOTE_CODER_OPTIONS_INFO;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unVersion = IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unCoderOptions= m_stRemoteEPAudioCodec.m_nAmrOption;
   }

   // Local Audio information used by decoder
   mediaCtr+=1;
   m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_INFO;
#if 1 //JM - Use Native	
   // Use Native codec in ipmStart if not transcode so lic resource is not used.
   if (g_nAudioTranscodeDir==NATIVE || g_nAudioTranscodeDir==TX) {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = static_cast<eIPM_CODER_TYPE> (m_stRemoteEPAudioCodec.m_nCoderType | CODER_TYPE_NATIVE);
   } else {
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = m_stRemoteEPAudioCodec.m_nCoderType;
   } 
#else
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.CoderInfo.eCoderType = m_stRemoteEPAudioCodec.m_nCoderType;
#endif
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.eFrameSize = m_stRemoteEPAudioCodec.m_nAudioFrameSize;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unFramesPerPkt = m_stRemoteEPAudioCodec.m_nFPP;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.eVadEnable = ((m_stRemoteEPAudioCodec.m_bVadEnabled==true)?CODER_VAD_ENABLE:CODER_VAD_DISABLE);
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unCoderPayloadType = m_stRemoteEPAudioCodec.m_nAudioCodec;
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderInfo.unRedPayloadType = 97;
   // Local Audio Options information used by decoder if applicable (for AMR or EVRC)
   if(strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr")==0 || strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-nb")==0) {
      mediaCtr+=1;
      m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_OPTIONS_INFO;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unVersion = IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unCoderOptions= m_stRemoteEPAudioCodec.m_nAmrOption;
   } else if (strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amrwb")==0 || strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-wb")==0) {
      mediaCtr+=1;
      m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_LOCAL_CODER_OPTIONS_INFO;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unVersion = IPM_AUDIO_CODER_OPTIONS_INFO_VERSION;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.AudioCoderOptionsInfo.unCoderOptions= m_stRemoteEPAudioCodec.m_nAmrOption;
   } 

   if (m_bVideoCall){
      // Remote video information used by encoder
      mediaCtr+=1;
      INIT_IPM_VIDEO_CODER_INFO(&m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);

      m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType                                           = MEDIATYPE_VIDEO_REMOTE_CODER_INFO;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType           	= m_stRemoteEPVideoCodec.m_nCoderType;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   	= m_stRemoteEPVideoCodec.m_nVideoCodec;
      SetupExtraVideoCoderInfo(mediaCtr);

      // Local video  information used by decoder
      mediaCtr+=1;
      INIT_IPM_VIDEO_CODER_INFO(&m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo);
      m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType                                    	= MEDIATYPE_VIDEO_LOCAL_CODER_INFO;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.eCoderType          	= m_stRemoteEPVideoCodec.m_nCoderType;
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.VideoCoderInfo.unCoderPayloadType   	= m_stRemoteEPVideoCodec.m_nVideoCodec;
      SetupExtraVideoCoderInfo(mediaCtr);
   }

   // remote audio port
   ++mediaCtr;
   m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_REMOTE_RTP_INFO;
   strcpy(m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_cAudioVideoIp);
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_nAudioRtpPort; 

   ++mediaCtr;
   m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_AUDIO_REMOTE_RTCP_INFO;
   strcpy(m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_cAudioVideoIp);
   m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_nAudioRtpPort+1; 

   if (m_bVideoCall){
      // remote video port
      ++mediaCtr;
      m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_VIDEO_REMOTE_RTP_INFO;
      strcpy(m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_cAudioVideoIp);
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_nVideoRtpPort; 

      ++mediaCtr;
      m_ccIpmMediaInfo.MediaData[mediaCtr].eMediaType = MEDIATYPE_VIDEO_REMOTE_RTCP_INFO;
      strcpy(m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.PortInfo.cIPAddress, m_cAudioVideoIp);
      m_ccIpmMediaInfo.MediaData[mediaCtr].mediaInfo.PortInfo.unPortId = m_nVideoRtpPort+1; 
   }

   m_ccIpmMediaInfo.unCount = mediaCtr + 1;
   mmReport(INFO_DEBUG, s_eType, "m_ccIpmMediaInfo.unCount = %d", m_ccIpmMediaInfo.unCount);
}


//*****************************************************************************
// 		  NAME : void CMMStream::Init()
// DESCRIPTION : Opens stream related devices
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : Bool - True if all the opens succeeded, False otherwise
// 	  CAUTIONS : None
//*****************************************************************************
bool CMMStream::Init()
{
   m_mmH = -1;
   // open the IPM device
   if ( (m_ipmCCH = ipm_Open(m_ipmCCDevName, 0, EV_ASYNC)) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "ipm_Open() failed for %s\n", m_ipmCCDevName);
      return false;
   }
   mmReport(INFO_DEBUG, s_eType, "Opened %s, dev=%d", m_ipmCCDevName, m_ipmCCH);

   CMMStream *pT = this;
   sr_setparm(m_ipmCCH, SR_USERCONTEXT, (void *)&pT);

#ifdef USE_RTSP
   // open the IPM device // JS for RTSP
   if ( (m_ipmRTSPH = ipm_Open(m_ipmRTSPDevName, 0, EV_ASYNC)) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "ipm_Open() failed for %s\n", m_ipmRTSPDevName);
      return false;
   }
   mmReport(INFO_DEBUG, s_eType, "Opened %s, dev=%d", m_ipmRTSPDevName, m_ipmRTSPH);

   sr_setparm(m_ipmRTSPH, SR_USERCONTEXT, (void *)&pT);
#endif

   // open DX device
   if ( (m_voxH = dx_open(m_voxDevName, 0)) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "dx_open() failed for %s\n", m_voxDevName);
      return false;
   }
   mmReport(INFO_DEBUG, s_eType, "Opened %s, dev=%d", m_voxDevName, m_voxH);
   sr_setparm(m_voxH, SR_USERCONTEXT, (void *)&pT);
   // enable single digits
   //dx_setevtmsk(m_voxH, DM_DIGITS);

#ifdef TDM_AUDIO
   // Open a TDM Device.
   // Name it similarly to the ipm device.  I.e. 1:1 channel to channel mapping.  Board
   // on dti device set as needed For example:  ipmB1C1 -> dtiB2T1
   char dtiGcName[DEV_NAME];
   sprintf( dtiGcName, ":N_%s:P_CC", m_dtiDevName);

   if ( (gc_OpenEx(&m_dtiH, dtiGcName, EV_ASYNC, (void*) 0)) < 0 )
   {
      mmReport(ERROR_GCALL, IP_DEVICE, "gc_OpenEx() failed on %s", m_dtiDevName);
      return false;
   }
   mmReport(INFO_DEBUG, IP_DEVICE, "Opened DTI %s, dev=%d", m_dtiDevName, m_dtiH);
#endif

   return true;
}

//*****************************************************************************
// 		  NAME : bool CMMStream::Shutdown()
// DESCRIPTION : Closes all the open devices
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : Bool - True if all the close succeeded
// 	  CAUTIONS : None
//*****************************************************************************
bool CMMStream::Shutdown()
{
   DisconnectMM();
   DisconnectIPM();
   StopStream(0);

#ifdef USE_MTK_API
   // SKS Destroy Built Overlay while exiting
   if (g_nVideoTranscodeDir==RXTX || g_nVideoTranscodeDir==TX) {
      DestroyOverlay();
   }
#endif
   if ( m_ipmCCH > 0 )
   {
      ipm_Close(m_ipmCCH, 0);
      m_ipmCCH = 0;
   }
   if ( m_ipmRTSPH > 0 )
   {
      ipm_Close(m_ipmRTSPH, 0);
      m_ipmRTSPH = 0;
   }
   if ( m_mmH > 0 )
   {
      mm_Close(m_mmH, 0);
      m_mmH = 0;
   }
   if ( m_voxH > 0 )
   {
      dx_unlisten(m_voxH);
      dx_close(m_voxH);
      m_voxH = 0;
   }
#ifdef TDM_AUDIO
   if ( m_dtiH > 0 )
   {
      gc_UnListen(m_dtiH, EV_SYNC);
      gc_Close(m_dtiH);
      m_dtiH = 0;
   }
#endif

   return true;
}

//*****************************************************************************
// 		  NAME : void CMMStream::SetState(state_t nextState)
// DESCRIPTION : Sets the stream state to the state requested
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::SetState(state_t nextState)
{
   m_eState = nextState;
}

//*****************************************************************************
// 		  NAME : int CMMStream::GetState()
// DESCRIPTION : Returns the current stream state
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : State of CMMStream object
// 	  CAUTIONS : None
//*****************************************************************************
int CMMStream::GetState()
{
   m_eCurrentState = m_eState;
   return m_eState;
}

//*****************************************************************************
// 		  NAME : bool CMMStream::SendIFrameRequest()
// DESCRIPTION : Sends the IFrame Request
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : Bool - True if the function succeeded, False otherwise
// 	  CAUTIONS : None
//*****************************************************************************
bool CMMStream::SendIFrameRequest()
{
   // Local Variable Declaration
   GC_PARM_BLKP    gcParmBlk_mime  = 0;
   GC_PARM_BLKP    gcParmBlk_mime1 = 0;
   GC_PARM_BLKP    gcParmBlk_info  = 0;
   bool            bOk             = true;
   const char     *pBodyType       = "Content-Type:application/media_control+xml"; // specify the body type

   mmReport(INFO_DEBUG, s_eType, "SendIFrameRequest()");
   if ( gc_util_insert_parm_ref(&gcParmBlk_mime,
                                IPSET_MIME,
                                IPPARM_MIME_PART_TYPE,
                                (unsigned char)(strlen(pBodyType) + 1),
                                (void*)pBodyType) < 0 )
   {
      mmReport(ERROR_GCALL, s_eType, "SendIFrameRequest() -> gc_util_insert_parm_ref() failed on %s for IPPARM_MIME_PART_TYPE ", m_devName);
      bOk = false;
   }

   // insert the body size
   if ( gc_util_insert_parm_val(&gcParmBlk_mime,
                                IPSET_MIME,
                                IPPARM_MIME_PART_BODY_SIZE,
                                sizeof(unsigned long),
                                strlen(c_iFrameRequest)) < 0 )
   {
      mmReport(ERROR_GCALL, s_eType, "SendIFrameRequest() -> gc_util_insert_parm_val() failed on %s for IPPARM_MIME_PART_BODY_SIZE ", m_devName);
      bOk = false;
   }

   // insert the body
   if ( gc_util_insert_parm_val(&gcParmBlk_mime,
                                IPSET_MIME,
                                IPPARM_MIME_PART_BODY,
                                sizeof(unsigned long),
                                (unsigned long)(c_iFrameRequest)) < 0 )
   {
      mmReport(ERROR_GCALL, s_eType, "SendIFrameRequest() -> gc_util_insert_parm_val() failed on %s for IPPARM_MIME_PART_BODY ", m_devName);
      bOk = false;
   }

   // insert the list of parmBlks into the top level parmBlk
   if ( gc_util_insert_parm_val(&gcParmBlk_mime1,
                                IPSET_MIME,
                                IPPARM_MIME_PART,
                                sizeof(unsigned long),
                                (unsigned long)gcParmBlk_mime) < 0 )
   {
      mmReport(ERROR_GCALL, s_eType, "SendIFrameRequest() -> gc_util_insert_parm_val() failed on %s for IPPARM_MIME_PART", m_devName);
      bOk = false;
   }

   // now set it on the device
   if ( gc_SetUserInfo(GCTGT_GCLIB_CRN,
                       m_gcCurrentCrn,
                       gcParmBlk_mime1,
                       GC_SINGLECALL) < 0 )
   { // for this call only
      mmReport(ERROR_GCALL, s_eType, "gc_SetUserInfo() failed on %s for MIME body in INFO");
      bOk = false;
   }

   // insert the message type
   if ( gc_util_insert_parm_val(&gcParmBlk_info,
                                IPSET_MSG_SIP,
                                IPPARM_MSGTYPE,
                                sizeof(int),
                                IP_MSGTYPE_SIP_INFO) < 0 )
   {
      mmReport(ERROR_GCALL, s_eType, "SendIFrameRequest() -> gc_util_insert_parm_val() failed on %s for SIP INFO", m_devName);
      bOk = false;
   }

   if ( gc_Extension(GCTGT_GCLIB_CRN, m_gcCurrentCrn, IPEXTID_SENDMSG, gcParmBlk_info, NULL, EV_ASYNC) < 0 )
   {
      mmReport(ERROR_GCALL, s_eType, "SendIFrameRequest() -> gc_Extension failed");
      bOk = false;
   }
   gc_util_delete_parm_blk(gcParmBlk_info);
   gc_util_delete_parm_blk(gcParmBlk_mime);

   return bOk;
}

//*****************************************************************************
//         NAME : void CMMStream::OnDoNothing(void *pData)
// DESCRIPTION :  process all events that we don't take action on
//        INPUT : None
//        OUTPUT : None
//       RETURNS : None
//      CAUTIONS : None
//*****************************************************************************
void CMMStream::OnDoNothing(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnDoNothing()");
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnIpmOpenEx(void *pData)
// DESCRIPTION : Opens the mm device
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnIpmOpenEx(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnIpmOpenEx()");

   if (m_mmH == -1) { // JS - in Init(), set m_mmH to -1. Then first ipm_Open() will
                      // lead to MM open
      // open the MM device
      if ( (m_mmH = mm_Open(m_mmDevName, 0, NULL)) < 0 )
      {
         mmReport(INFO_ERROR, s_eType, "mm_Open() failed for %s\n", m_mmDevName);
         return;
      }
      mmReport(INFO_DEBUG, s_eType, "Opened %s, dev=%d", m_mmDevName, m_mmH);
      CMMStream *pT = this;
      sr_setparm(m_mmH, SR_USERCONTEXT, (void *)&pT);
   }

   return;
}

void CMMStream::PrintPortInfo(const char* device, const char* fcn, DM_PORT_INFO_LIST* pPortList) 
{

   printf("%s: %s: \n", device, fcn);
   for ( unsigned int i = 0; i < pPortList->unCount; i++ ) {
      DM_PORT_INFO & p = pPortList->port_info[i];
      printf( "Tx Port %d: \n");
      printf( "   Media Type: %u ", p.port_media_type);
      if ( p.port_media_type == DM_PORT_MEDIA_TYPE_AUDIO ) {
         printf( " (audio)\n");
      } else if ( p.port_media_type == DM_PORT_MEDIA_TYPE_VIDEO ) {
         printf( " (video)\n");
      } else if ( p.port_media_type == DM_PORT_MEDIA_TYPE_NBUP ) {
         printf( " (NBUP)\n");
      } else {
         printf( " demo does not recognize media type " , p.port_media_type );
      }
   }
}

void CMMStream::OnGetTxPortInfo(void* pData)
{
    int nDevice = sr_getevtdev();
    int rc = -1;
    if (nDevice == m_mmH) {
      DM_PORT_INFO_LIST *temp = (DM_PORT_INFO_LIST*)pData;//sr_getevtdatap();
      int evt_len = sr_getevtlen();
      memcpy(&m_Mm_TxPortList, temp, sizeof(DM_PORT_INFO_LIST));
      mmReport(INFO_DEBUG, s_eType, "MM (%s) OnGetTxPortInfo()", m_mmDevName);
      //PrintPortInfo(m_mmDevName, "OnGetTxPortInfo()", &m_Mm_TxPortList);
      m_bHaveMMTxPort = true;
      rc = dev_GetReceivePortInfo(m_mmH, this);
      if ( rc == -1 ) {
              mmReport(INFO_ERROR, s_eType, " dev_GetReceivePortInfo() failed", m_mmDevName);
      }
    }
    else if (nDevice == m_ipmCCH) {
      DM_PORT_INFO_LIST *temp = (DM_PORT_INFO_LIST*)pData;//sr_getevtdata  p();
      int evt_len = sr_getevtlen();
      memcpy(&m_Ipm_CC_TxPortList, temp, sizeof(DM_PORT_INFO_LIST));
      mmReport(INFO_DEBUG, s_eType, "IPM (%s) OnGetTxPortInfo()", m_ipmCCDevName);
      //PrintPortInfo(m_ipmCCDevName, "OnGetTxPortInfo()", &m_Ipm_CC_TxPortList);
      m_bHaveCCIPMTxPort = true;
      rc = dev_GetReceivePortInfo(m_ipmCCH, this);
      if ( rc == -1 ) {
         mmReport(INFO_ERROR, s_eType, " dev_GetReceivePortInfo() failed", m_ipmCCDevName);
      }
    } else if (nDevice == m_ipmRTSPH) {
      memcpy(&m_Ipm_RTSP_TxPortList, sr_getevtdatap(), sizeof(DM_PORT_INFO_LIST));
      mmReport(INFO_DEBUG, s_eType, "IPM_RTSP (%s) OnGetTxPortInfo()", m_ipmRTSPDevName);
      //PrintPortInfo(m_ipmRTSPDevName, "OnGetTxPortInfo()", &m_Ipm_RTSP_TxPortList);
      m_bHaveRTSPIPMTxPort = true;
      rc = dev_GetReceivePortInfo(m_ipmRTSPH, this);
      if ( rc == -1 ) {
         mmReport(INFO_ERROR, s_eType, " dev_GetReceivePortInfo() failed", m_ipmRTSPDevName);
      }
    }
    else {
      mmReport(INFO_ERROR, s_eType, " OnGetTxPortInfo() on non IPM or MM device (nDev=%d)", nDevice);
    }

#ifdef USE_RTSP
    if ( m_bHaveRTSPIPMTxPort && m_bHaveCCIPMTxPort && m_bHaveRTSPIPMRxPort && m_bHaveCCIPMRxPort ) {
        if (g_nVideoTranscodeDir==RXTX)
           RouteIPM_IPM_Ports(DMFL_TRANSCODE_ON);
        else
           RouteIPM_IPM_Ports(DMFL_TRANSCODE_NATIVE);
      }

#else

    if ( m_bHaveMMTxPort && m_bHaveCCIPMTxPort && m_bHaveMMRxPort && m_bHaveCCIPMRxPort ) {
           RouteIPM_MM_Ports();
    }

#endif // USE_RTSP
}


void CMMStream::OnGetRxPortInfo(void* pData)
{
   int nDevice = sr_getevtdev();
   if (nDevice == m_mmH) {
      DM_PORT_INFO_LIST *temp = (DM_PORT_INFO_LIST*)pData;//sr_getevtdatap();
      int evt_len = sr_getevtlen();
      memcpy(&m_Mm_RxPortList, temp, sizeof(DM_PORT_INFO_LIST));
      mmReport(INFO_DEBUG, s_eType, "MM (%s) OnGetRxPortInfo()", m_mmDevName);
      //PrintPortInfo(m_mmDevName, "OnGetRxPortInfo()", &m_Mm_RxPortList);
      m_bHaveMMRxPort = true;
   }
   else if (nDevice == m_ipmCCH) {
      DM_PORT_INFO_LIST *temp = (DM_PORT_INFO_LIST*)pData;//sr_getevtdatap();
      int evt_len = sr_getevtlen();
      memcpy(&m_Ipm_CC_RxPortList, temp, sizeof(DM_PORT_INFO_LIST));
      mmReport(INFO_DEBUG, s_eType, "IPM (%s) OnGetRxPortInfo()", m_ipmCCDevName);
      //PrintPortInfo(m_ipmCCDevName, "OnGetRxPortInfo()", &m_Ipm_CC_RxPortList);
      m_bHaveCCIPMRxPort = true;
   } else if (nDevice == m_ipmRTSPH) {
      memcpy(&m_Ipm_RTSP_RxPortList, sr_getevtdatap(), sizeof(DM_PORT_INFO_LIST));
      mmReport(INFO_DEBUG, s_eType, "IPM_RTSP (%s) OnGetRxPortInfo()", m_ipmRTSPDevName);
      //PrintPortInfo(m_ipmRTSPDevName, "OnGetRxPortInfo()", &m_Ipm_RTSP_RxPortList);
      m_bHaveRTSPIPMRxPort = true;
   }
   else {
      mmReport(INFO_ERROR, s_eType, " OnGetRxPortInfo() on non IPM or MM device (nDev=%d)",nDevice);
   }
#ifdef USE_RTSP
   if ( m_bHaveRTSPIPMTxPort && m_bHaveCCIPMTxPort && m_bHaveRTSPIPMRxPort && m_bHaveCCIPMRxPort ) {
      if (g_nVideoTranscodeDir==RXTX)
         RouteIPM_IPM_Ports(DMFL_TRANSCODE_ON);
      else
         RouteIPM_IPM_Ports(DMFL_TRANSCODE_NATIVE);
   }
#else
   if ( m_bHaveMMTxPort && m_bHaveCCIPMTxPort && m_bHaveMMRxPort && m_bHaveCCIPMRxPort ) {
      RouteIPM_MM_Ports();
   }
#endif // USE_RTSP
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnDevPortConnect(void *pData)
// DESCRIPTION : On OnDevPortConnect do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnDevPortConnect(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnDevPortConnect()");
}

//*****************************************************************************
// Function: int ::getportindex(DM_PORT_INFO_LIST portlist, DM_PORT_MEDIA_TYPE porttype)
// Description: given an array of ports, returns index of desired port type
// Return: int, index of desired port type 
// Parameters: None
// Notes: None
//*****************************************************************************
int CMMStream::getportindex(DM_PORT_INFO_LIST portlist, DM_PORT_MEDIA_TYPE porttype)
{
   for ( unsigned int i = 0; i < portlist.unCount; i++ ) {
      if ( portlist.port_info[i].port_media_type == porttype )
         return i;
   }
   return -1;
}

//*****************************************************************************
// Function: char* ::get_portmediatype_str(DM_PORT_MEDIA_TYPE port_type)
// Description: returns name (string) of port media type
// Return: char*, name of the port 
// Parameters: None
// Notes: None
//*****************************************************************************
const char *CMMStream::get_portmediatype_str(DM_PORT_MEDIA_TYPE port_type)
{
   switch ( port_type ) {
      case DM_PORT_MEDIA_TYPE_NONE:
         return "DM_PORT_MEDIA_TYPE_NONE";
      case DM_PORT_MEDIA_TYPE_AUDIO:
         return "DM_PORT_MEDIA_TYPE_AUDIO";
      case DM_PORT_MEDIA_TYPE_VIDEO:
         return "DM_PORT_MEDIA_TYPE_VIDEO";
      default:
         return "Unknown port media type";
      }
}

//*****************************************************************************
// Function: char* ::get_transcode_str(int transcodeflag)
// Description: returns string equivalent of transcode flag
// Return: char* 
// Parameters: None
// Notes: None
//*****************************************************************************
const char *CMMStream::get_transcode_str(int transcodeflag)
{
        switch ( transcodeflag ) {
        case DMFL_TRANSCODE_ON:
                return "DMFL_TRANSCODE_ON";
        case DMFL_TRANSCODE_NATIVE:
                return "DMFL_TRANSCODE_NATIVE";
        default:
                return "Unknown transcode flag";
        }
}


int CMMStream::RouteIPM_IPM_Ports(int transcodeOption) {
        int ipm_rtsp_arx, ipm_rtsp_atx;
        int ipm_rtsp_vrx, ipm_rtsp_vtx;
        int ipm_cc_arx, ipm_cc_atx;
        int ipm_cc_vrx, ipm_cc_vtx;
        int count = 0;

        mmReport(INFO_DEBUG, s_eType,"RouteIPM_IPM_Ports()->Routing %s and %s.\n", m_ipmRTSPDevName, m_ipmCCDevName);

    // Step 1: Get audio/video rx/tx port indicies of MM and IPM devices.

    // Get index of MM Audio transmit port in the MM device's Transmit
    //    port list. MM device's Transmit port list is initialized
    //    during application startup
        ipm_rtsp_atx =
        getportindex(m_Ipm_RTSP_TxPortList, DM_PORT_MEDIA_TYPE_AUDIO);

    // Get index of MM Video transmit port in the MM device's Transmit
    //    port list. MM device's Transmit port list is initialized
    //    during application startup
        ipm_rtsp_vtx =
        getportindex(m_Ipm_RTSP_TxPortList, DM_PORT_MEDIA_TYPE_VIDEO);

    // Get index of MM Audio receive port in the MM device's Receive
    //    port list. MM device's Receive port list is initialized
    //    during application startup
        ipm_rtsp_arx =
        getportindex(m_Ipm_RTSP_RxPortList, DM_PORT_MEDIA_TYPE_AUDIO);

    // Get index of MM Video receive port in the MM device's Receive
    //    port list. MM device's Receive port list is initialized
    //    during application startup
        ipm_rtsp_vrx =
        getportindex(m_Ipm_RTSP_RxPortList, DM_PORT_MEDIA_TYPE_VIDEO);

    // Get index of IPM Audio transmit port in the IPM device's Transmit
    //    port list. IPM device's Transmit port list is initialized
    //    during application startup
    ipm_cc_atx =
        getportindex(m_Ipm_CC_TxPortList, DM_PORT_MEDIA_TYPE_AUDIO);

    // Get index of IPM Video transmit port in the IPM device's Transmit
    //    port list. IPM device's Transmit port list is initialized
    //    during application startup
        ipm_cc_vtx =
        getportindex(m_Ipm_CC_TxPortList, DM_PORT_MEDIA_TYPE_VIDEO);

    // Get index of IPM Audio receive port in the IPM device's Receive
    //    port list. IPM device's Receive port list is initialized
    //    during application startup
        ipm_cc_arx =
        getportindex(m_Ipm_CC_RxPortList, DM_PORT_MEDIA_TYPE_AUDIO);

    // Get index of IPM Video receive port in the IPM device's Receive
    //    port list. IPM device's Receive port list is initialized
    //    during application startup
    ipm_cc_vrx =
        getportindex(m_Ipm_CC_RxPortList, DM_PORT_MEDIA_TYPE_VIDEO);

                if ( (ipm_rtsp_atx == -1) || (ipm_rtsp_vtx == -1) || (ipm_rtsp_arx == -1)
                         || (ipm_rtsp_vrx == -1) || (ipm_cc_atx == -1) || (ipm_rtsp_vtx == -1)
                         || (ipm_cc_arx == -1) || (ipm_cc_vrx == -1) ) {
 #if 1
        printf("MM Video Tx index = %d\n", ipm_rtsp_atx);
        printf("MM Audio Tx index = %d\n", ipm_rtsp_atx);
        printf("MM Audio Rx index = %d\n", ipm_rtsp_atx);
        printf("MM Video Rx index = %d\n", ipm_rtsp_vrx);
        printf("IPM Audio Tx index = %d\n", ipm_cc_atx);
        printf("IPM Video Tx index = %d\n", ipm_rtsp_vtx);
        printf("IPM Audio Rx index = %d\n", ipm_cc_atx);
        printf("IPM Video Rx index = %d\n", ipm_rtsp_vtx);
#endif

                       printf("ERROR: a port index is -1\n");
                       return -1;
                }

    // Step 2: Create port connection list of MM device

    // set the version of data structure
    //RTSPipmCnctList.unVersion = DM_PORT_CONNECT_INFO_LIST_VERSION_0;
    INIT_DM_PORT_CONNECT_INFO_LIST(&RTSPipmCnctList); 

    // if AUDIO port connection option is requested, set
    //  the MM Tx Audio to IPM Rx Audio
    // Transcode vs. Native connection method is also specified for
    //  this port connection entry in the connection list.
        count = 0;
                RTSPipmCnctList.port_connect_info[count].unFlags = transcodeOption;
                //RTSPipmCnctList.port_connect_info[count].unFlags = DMFL_TRANSCODE_NATIVE; //transcodeOption;
                RTSPipmCnctList.port_connect_info[count].port_info_tx = m_Ipm_RTSP_TxPortList.port_info[ipm_rtsp_atx];
                RTSPipmCnctList.port_connect_info[count].port_info_rx = m_Ipm_CC_RxPortList.port_info[ipm_cc_arx];
                count++;

    // if VIDEO port connection option is requested, set
    //  the MM Tx Video to IPM Rx Video
    // Transcode vs. Native connection method is also specified for
    //  this port connection entry in the connection list.
                RTSPipmCnctList.port_connect_info[count].unFlags = transcodeOption;
                RTSPipmCnctList.port_connect_info[count].port_info_tx = m_Ipm_RTSP_TxPortList.port_info[ipm_rtsp_vtx];
                RTSPipmCnctList.port_connect_info[count].port_info_rx = m_Ipm_CC_RxPortList.port_info[ipm_cc_vrx];
                count++;
        RTSPipmCnctList.unCount = count;

#if 1
        unsigned int i;
        for ( i = 0; i < RTSPipmCnctList.unCount; i++ ) {
                printf
                ("%s(devh=0x%x) #%d: tx=[%s, portID=%s], rx=[%s, portID=%s] [xcode flag=%s]\n",
                 m_ipmRTSPDevName, m_ipmRTSPH, i,
                 get_portmediatype_str(RTSPipmCnctList.port_connect_info[i].
                                                           port_info_tx.port_media_type),
                 RTSPipmCnctList.port_connect_info[i].port_info_tx.port_ID,
                 get_portmediatype_str(RTSPipmCnctList.port_connect_info[i].
                                                           port_info_rx.port_media_type),
                 RTSPipmCnctList.port_connect_info[i].port_info_rx.port_ID,
                 get_transcode_str(RTSPipmCnctList.port_connect_info[i].unFlags));
        }
#endif
    // Step 3: Connection list is built in step 2. Now, route MM's Tx ports
    //    to IPM Rx ports. This will complete connection in one direction.
    //    Steps 4-5 will complete connection in reverse direction
        int rc = dev_PortConnect(m_ipmRTSPH, &RTSPipmCnctList, NULL);
        if ( rc == -1 ) {
                printf
                ("FAILURE: dev_PortConnect(0x%x, RTSPipmCnctList, NULL) transcode flag = %s\n",
                 m_ipmRTSPH, get_transcode_str(transcodeOption));
                return -1;
        }

    // Step 4: Create port connection list of MM device

    // set the version of data structure
    //CCipmCnctList.unVersion = DM_PORT_CONNECT_INFO_LIST_VERSION_0;
    INIT_DM_PORT_CONNECT_INFO_LIST(&CCipmCnctList); 

    // if AUDIO port connection option is requested, set
    //  the IPM Tx Audio to MM Rx Audio
    // Transcode vs. Native connection method is also specified for
    //  this port connection entry in the connection list.
        count = 0;
                CCipmCnctList.port_connect_info[count].unFlags = transcodeOption;
                //CCipmCnctList.port_connect_info[count].unFlags = DMFL_TRANSCODE_NATIVE; //transcodeOption;
                CCipmCnctList.port_connect_info[count].port_info_tx = m_Ipm_CC_TxPortList.port_info[ipm_cc_atx];
                CCipmCnctList.port_connect_info[count].port_info_rx = m_Ipm_RTSP_RxPortList.port_info[ipm_rtsp_arx];
                count++;

    // if VIDEO port connection option is requested, set
    //  the IPM Tx Video to MM Rx Video
    // Transcode vs. Native connection method is also specified for
    //  this port connection entry in the connection list.
                CCipmCnctList.port_connect_info[count].unFlags = transcodeOption;
                CCipmCnctList.port_connect_info[count].port_info_tx = m_Ipm_CC_TxPortList.port_info[ipm_cc_vtx];
                CCipmCnctList.port_connect_info[count].port_info_rx = m_Ipm_RTSP_RxPortList.port_info[ipm_rtsp_vrx];
                count++;

        CCipmCnctList.unCount = count;

#if 1
        for ( i = 0; i < CCipmCnctList.unCount; i++ ) {
                printf
                ("%s(devh=0x%x) #%d: tx=[%s, portID=%s], rx=[%s, portID=%s] [xcode flag=%s]\n",
                 m_ipmCCDevName, m_ipmCCH, i,
                 get_portmediatype_str(CCipmCnctList.port_connect_info[i].
                                                           port_info_tx.port_media_type),
                 CCipmCnctList.port_connect_info[i].port_info_tx.port_ID,
                 get_portmediatype_str(CCipmCnctList.port_connect_info[i].
                                                           port_info_rx.port_media_type),
                 CCipmCnctList.port_connect_info[i].port_info_rx.port_ID,
                 get_transcode_str(CCipmCnctList.port_connect_info[i].unFlags));
        }
#endif
    // Step 5: Connection list is built in step 2. Now, route IPM's Tx ports
    //    to MM Rx ports. This will complete connection in one direction.
    //    Steps 3 and 5 together create a full-duplex connection
        rc = dev_PortConnect(m_ipmCCH, &CCipmCnctList, NULL);
        if ( -1 == rc ) {
                printf
                ("FAILURE:dev_PortConnect(FAILURE: 0x%x, CCipmCnctList, NULL) transcode flag = %s\n",
                 m_ipmCCH, get_transcode_str(transcodeOption));
        }
        return rc;
}

int CMMStream::RouteIPM_MM_Ports() 
{
   int mm_arx, mm_vrx, ipm_arx, ipm_vrx, mm_atx, mm_vtx, ipm_atx, ipm_vtx;
   int count = 0;

   mmReport(INFO_DEBUG, s_eType,"RouteIPM_MM_Ports()->Routing %s and %s.\n", m_ipmCCDevName, m_mmDevName);

   // Step 1: Get audio/video rx/tx port indicies of MM and IPM devices.

   // Get index of MM Audio transmit port in the MM device's Transmit
   //    port list. MM device's Transmit port list is initialized
   //    during application startup
   mm_atx = getportindex(m_Mm_TxPortList, DM_PORT_MEDIA_TYPE_AUDIO);

   // Get index of MM Video transmit port in the MM device's Transmit
   //    port list. MM device's Transmit port list is initialized
   //    during application startup
   mm_vtx = getportindex(m_Mm_TxPortList, DM_PORT_MEDIA_TYPE_VIDEO);

   // Get index of MM Audio receive port in the MM device's Receive
   //    port list. MM device's Receive port list is initialized
   //    during application startup
   mm_arx = getportindex(m_Mm_RxPortList, DM_PORT_MEDIA_TYPE_AUDIO);

   // Get index of MM Video receive port in the MM device's Receive
   //    port list. MM device's Receive port list is initialized
   //    during application startup
   mm_vrx = getportindex(m_Mm_RxPortList, DM_PORT_MEDIA_TYPE_VIDEO);

   // Get index of IPM Audio transmit port in the IPM device's Transmit
   //    port list. IPM device's Transmit port list is initialized
   //    during application startup
   ipm_atx = getportindex(m_Ipm_CC_TxPortList, DM_PORT_MEDIA_TYPE_AUDIO);

   // Get index of IPM Video transmit port in the IPM device's Transmit
   //    port list. IPM device's Transmit port list is initialized
   //    during application startup
   ipm_vtx = getportindex(m_Ipm_CC_TxPortList, DM_PORT_MEDIA_TYPE_VIDEO);

   // Get index of IPM Audio receive port in the IPM device's Receive
   //    port list. IPM device's Receive port list is initialized
   //    during application startup
   ipm_arx = getportindex(m_Ipm_CC_RxPortList, DM_PORT_MEDIA_TYPE_AUDIO);

   // Get index of IPM Video receive port in the IPM device's Receive
   //    port list. IPM device's Receive port list is initialized
   //    during application startup
   ipm_vrx = getportindex(m_Ipm_CC_RxPortList, DM_PORT_MEDIA_TYPE_VIDEO);

   if ( (mm_atx == -1) || (mm_vtx == -1) || (mm_arx == -1)
        || (mm_vrx == -1) || (ipm_atx == -1) || (ipm_vtx == -1)
        || (ipm_arx == -1) || (ipm_vrx == -1) ) {
 #if 1
      printf("MM Video Tx index = %d\n", mm_atx);
      printf("MM Audio Tx index = %d\n", mm_atx);
      printf("MM Audio Rx index = %d\n", mm_atx);
      printf("MM Video Rx index = %d\n", mm_vrx);
      printf("IPM Audio Tx index = %d\n", ipm_atx);
      printf("IPM Video Tx index = %d\n", ipm_vtx);
      printf("IPM Audio Rx index = %d\n", ipm_atx);
      printf("IPM Video Rx index = %d\n", ipm_vtx);
#endif

      printf("ERROR: a port index is -1\n");
      return -1;
   }

   // Step 2: Create port connection list of MM device

   // set the version of data structure
   //mmCnctList.unVersion = DM_PORT_CONNECT_INFO_LIST_VERSION_0;
   INIT_DM_PORT_CONNECT_INFO_LIST(&mmCnctList); 

   // if AUDIO port connection option is requested, set
   //  the MM Tx Audio to IPM Rx Audio
   // Transcode vs. Native connection method is also specified for
   //  this port connection entry in the connection list.
   count = 0;

// SKS commented should influenced by user through config file.
   //mmCnctList.port_connect_info[count].unFlags = DMFL_TRANSCODE_ON;
   //mmCnctList.port_connect_info[count].unFlags = transcodeOption;
   mmCnctList.port_connect_info[count].unFlags = ((g_nAudioTranscodeDir==TX||g_nAudioTranscodeDir==RXTX)?DMFL_TRANSCODE_ON:DMFL_TRANSCODE_NATIVE);
   mmCnctList.port_connect_info[count].port_info_tx = m_Mm_TxPortList.port_info[mm_atx];
   mmCnctList.port_connect_info[count].port_info_rx = m_Ipm_CC_RxPortList.port_info[ipm_arx];
   count++;

   // if VIDEO port connection option is requested, set
   //  the MM Tx Video to IPM Rx Video
   // Transcode vs. Native connection method is also specified for
   //  this port connection entry in the connection list.
   mmCnctList.port_connect_info[count].unFlags = ((g_nVideoTranscodeDir==TX||g_nVideoTranscodeDir==RXTX)?DMFL_TRANSCODE_ON:DMFL_TRANSCODE_NATIVE);;
   mmCnctList.port_connect_info[count].port_info_tx = m_Mm_TxPortList.port_info[mm_vtx];
   mmCnctList.port_connect_info[count].port_info_rx = m_Ipm_CC_RxPortList.port_info[ipm_vrx];
   count++;
   mmCnctList.unCount = count;

#if 1
   unsigned int i;
   for ( i = 0; i < mmCnctList.unCount; i++ ) {
      mmReport(INFO_DEBUG, s_eType, "%s-Tx --> %s-Rx #%d: tx=[%s, portID=%s], rx=[%s, portID=%s] [xcode flag=%s]",
               m_mmDevName,m_ipmCCDevName,i,
               get_portmediatype_str(mmCnctList.port_connect_info[i].port_info_tx.port_media_type),
               mmCnctList.port_connect_info[i].port_info_tx.port_ID,
               get_portmediatype_str(mmCnctList.port_connect_info[i].port_info_rx.port_media_type),
               mmCnctList.port_connect_info[i].port_info_rx.port_ID,
               get_transcode_str(mmCnctList.port_connect_info[i].unFlags));
   }
#endif
   // Step 3: Connection list is built in step 2. Now, route MM's Tx ports
   //    to IPM Rx ports. This will complete connection in one direction.
   //    Steps 4-5 will complete connection in reverse direction
      int rc = dev_PortConnect(m_mmH, &mmCnctList, NULL);
      if ( rc == -1 ) {
         printf("FAILURE: dev_PortConnect(0x%x, mmConnListp, NULL)\n", m_mmH);
            return -1;
      }

   // Step 4: Create port connection list of MM device
   
   // set the version of data structure
   //CCipmCnctList.unVersion = DM_PORT_CONNECT_INFO_LIST_VERSION_0;
   INIT_DM_PORT_CONNECT_INFO_LIST(&CCipmCnctList); 

   // if AUDIO port connection option is requested, set
   //  the IPM Tx Audio to MM Rx Audio
   // Transcode vs. Native connection method is also specified for
   //  this port connection entry in the connection list.
   count = 0;
// SKS commented should influenced by user through config file.
// CCipmCnctList.port_connect_info[count].unFlags = DMFL_TRANSCODE_ON;
   CCipmCnctList.port_connect_info[count].unFlags = ((g_nAudioTranscodeDir==RX || g_nAudioTranscodeDir==RXTX)?DMFL_TRANSCODE_ON:DMFL_TRANSCODE_NATIVE);;
   CCipmCnctList.port_connect_info[count].port_info_tx = m_Ipm_CC_TxPortList.port_info[ipm_atx];
   CCipmCnctList.port_connect_info[count].port_info_rx = m_Mm_RxPortList.port_info[mm_arx];
   count++;

   // if VIDEO port connection option is requested, set
   //  the IPM Tx Video to MM Rx Video
   // Transcode vs. Native connection method is also specified for
   //  this port connection entry in the connection list.
   CCipmCnctList.port_connect_info[count].unFlags = ((g_nVideoTranscodeDir==RX || g_nVideoTranscodeDir==RXTX)?DMFL_TRANSCODE_ON:DMFL_TRANSCODE_NATIVE);
   CCipmCnctList.port_connect_info[count].port_info_tx = m_Ipm_CC_TxPortList.port_info[ipm_vtx];
   CCipmCnctList.port_connect_info[count].port_info_rx = m_Mm_RxPortList.port_info[mm_vrx];
   count++;

   CCipmCnctList.unCount = count;

#if 1
   for ( i = 0; i < CCipmCnctList.unCount; i++ ) {       
      mmReport(INFO_DEBUG, s_eType, "%s-Tx --> %s-Rx #%d: tx=[%s, portID=%s], rx=[%s, portID=%s] [xcode flag=%s]",
               m_ipmCCDevName,m_mmDevName,i,
               get_portmediatype_str(CCipmCnctList.port_connect_info[i].port_info_tx.port_media_type),
               CCipmCnctList.port_connect_info[i].port_info_tx.port_ID,
               get_portmediatype_str(CCipmCnctList.port_connect_info[i].port_info_rx.port_media_type),
               CCipmCnctList.port_connect_info[i].port_info_rx.port_ID,
               get_transcode_str(CCipmCnctList.port_connect_info[i].unFlags));
   }
#endif

   // Step 5: Connection list is built in step 2. Now, route IPM's Tx ports
   //    to MM Rx ports. This will complete connection in one direction.
   //    Steps 3 and 5 together create a full-duplex connection
   rc = dev_PortConnect(m_ipmCCH, &CCipmCnctList, NULL);
   if ( -1 == rc ) {
      printf ("FAILURE:dev_PortConnect(FAILURE: 0x%x, ipmConnListp, NULL)\n", m_ipmCCH);
   }
   return rc;
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnReserveResourceEx(void *pData)
// DESCRIPTION : Reserves Audio Coders required
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnReserveResourceEx(void *pData)
{
   unsigned int     evtDev = sr_getevtdev();
   mmReport(INFO_DEBUG, s_eType, "OnReserveResourceEx()");

   if (evtDev == m_ipmCCH) { // JS
      // call get local media info
      if ( ipm_GetLocalMediaInfo(m_ipmCCH, &m_CCipmLocalMedia, EV_ASYNC) < 0 ) {
         mmReport(INFO_ERROR, s_eType, "ipm_GetLocalMediaInfo() failed on %s", m_ipmCCDevName);
      }
      mmReport(INFO_DEBUG, s_eType, "Reserved Resource for %s, dev=%d", m_ipmCCDevName, m_ipmCCH);

      // enable events for RFC2833 packets
      eIPM_EVENT event[1];
      event[0] = EVT_RFC2833;
      if ( ipm_EnableEvents(m_ipmCCH, event, 1, EV_ASYNC) < 0 ){
         mmReport(INFO_ERROR, s_eType, "ipm_EnableEvent() failed on %s", m_ipmCCDevName);
      }
   }
   else if (evtDev == m_ipmRTSPH) {
      if ( ipm_GetLocalMediaInfo(m_ipmRTSPH, &m_RTSPipmLocalMedia, EV_ASYNC) < 0 ){
         mmReport(INFO_ERROR, s_eType, "ipm_GetLocalMediaInfo() failed on %s", m_ipmRTSPDevName);
      }
      mmReport(INFO_DEBUG, s_eType, "Reserved Resource for %s, dev=%d", m_ipmRTSPDevName, m_ipmRTSPH);
   }
   else {
      mmReport(INFO_ERROR, s_eType, "event on unexpected handle");
   }
   return;
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnIpmGetLocalInfo(void *pData)
// DESCRIPTION : Retrieves the local media information
// 		 INPUT : pData - void *
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnIpmGetLocalInfo(void *pData)
{
   // Local Variable Declaration
   unsigned int    i                   = 0;
   unsigned int    unLocalAudioPort    = 0;
   unsigned int    unLocalVideoPort    = 0;
   char            *pLocalIP           = 0;
   long            ts                  = 0;
   SC_TSINFO       scts;
   unsigned int    evtDev = sr_getevtdev();

   mmReport(INFO_DEBUG, s_eType, "OnGetLocalInfo()");

   if (evtDev == m_ipmCCH) {  // JS: added device handle based processing since there are 2 ipm devs now
      m_CCipmLocalMedia = *(IPM_MEDIA_INFO *)pData;
      for ( i = 0; i < m_CCipmLocalMedia.unCount; i++ )
      {
         switch ( m_CCipmLocalMedia.MediaData[i].eMediaType )
         {
            case MEDIATYPE_REMOTE_RTP_INFO:
               break;
            case MEDIATYPE_AUDIO_LOCAL_RTP_INFO:
               unLocalAudioPort = m_CCipmLocalMedia.MediaData[i].mediaInfo.PortInfo.unPortId;
               pLocalIP = m_CCipmLocalMedia.MediaData[i].mediaInfo.PortInfo.cIPAddress;
               m_unLocalCCIPMAudioRtpPortIndex = i;
               break;
            case MEDIATYPE_VIDEO_LOCAL_RTP_INFO:
               unLocalVideoPort = m_CCipmLocalMedia.MediaData[i].mediaInfo.PortInfo.unPortId;
               // Need to capture video address also!
               m_unLocalCCIPMVideoRtpPortIndex = i;
               break;
            case MEDIATYPE_REMOTE_RTCP_INFO:
               break;
            case MEDIATYPE_LOCAL_RTCP_INFO:
               break;
            case MEDIATYPE_REMOTE_CODER_INFO:
               break;
            case MEDIATYPE_LOCAL_CODER_INFO:
               break;
            default:
               // Anu
               break;
         }
      }
      mmReport(INFO_DEBUG, s_eType, "CC %s: localAudioPort = %d, localVideoPort = %d, localIP = %s", m_ipmCCDevName, unLocalAudioPort, unLocalVideoPort, pLocalIP);

      // get the transmit timeslot
      scts.sc_numts = 1;
      scts.sc_tsarrayp = &ts;
      if ( ipm_GetXmitSlot(m_ipmCCH, &scts, EV_SYNC) < 0 )
      {
         mmReport(INFO_ERROR, s_eType, "ipm_GetXmitSlot() failed on %s", m_ipmCCDevName);
      }
#ifndef TDM_AUDIO 
      // Unless rfc2833 event mode for dtmf, make the Vox device listen to IPM 
      if (g_nDtmfDetectMode == dtmfMode_rfc2833){
         mmReport(INFO_DEBUG, s_eType, "DTMF mode = rfc2833 event, skip dx_listen() on %s", m_voxDevName); 
      }
      else if (g_nDtmfDetectMode == dtmfMode_sipinfo){
         mmReport(INFO_DEBUG, s_eType, "DTMF mode = SIP_INFO event, skip dx_listen() on %s", m_voxDevName); 
      }
      else {
         //make the Vox device listen to IPM
         scts.sc_numts = 1;
         *(scts.sc_tsarrayp) = ts;
         if ( dx_listen(m_voxH, &scts) < 0 )
         {
            mmReport(INFO_ERROR, s_eType, "dx_lisen() failed on %s", m_voxDevName);
         }
         else {
            mmReport(INFO_DEBUG, s_eType, "dx_listen() success on %s", m_voxDevName); 
         }
         mmReport(INFO_DEBUG, s_eType, "Vox channel %s listening to IPM Local TS=%d", m_voxDevName, ts);
      }
#endif
   } else if (evtDev == m_ipmRTSPH) {
      m_RTSPipmLocalMedia = *(IPM_MEDIA_INFO*)pData;
      for ( i = 0; i < m_RTSPipmLocalMedia.unCount; i++ )
      {
         switch ( m_RTSPipmLocalMedia.MediaData[i].eMediaType )
         {
            case MEDIATYPE_REMOTE_RTP_INFO:
               break;
            case MEDIATYPE_AUDIO_LOCAL_RTP_INFO:
               unLocalAudioPort = m_RTSPipmLocalMedia.MediaData[i].mediaInfo.PortInfo.unPortId;
               pLocalIP = m_RTSPipmLocalMedia.MediaData[i].mediaInfo.PortInfo.cIPAddress;
               m_unLocalRTSPIPMAudioRtpPortIndex = i;
               break;
            case MEDIATYPE_VIDEO_LOCAL_RTP_INFO:
               unLocalVideoPort = m_RTSPipmLocalMedia.MediaData[i].mediaInfo.PortInfo.unPortId;
               // Need to capture video address also!
               m_unLocalRTSPIPMVideoRtpPortIndex = i;
               break;
            case MEDIATYPE_REMOTE_RTCP_INFO:
               break;
            case MEDIATYPE_LOCAL_RTCP_INFO:
               break;
            case MEDIATYPE_REMOTE_CODER_INFO:
               break;
            case MEDIATYPE_LOCAL_CODER_INFO:
               break;
            default:
               break;
         }
      }
      mmReport(INFO_DEBUG, s_eType, "RTSP %s: localAudioPort = %d, localVideoPort = %d, localIP = %s", m_ipmRTSPDevName, unLocalAudioPort, unLocalVideoPort, pLocalIP);

   }
   else {
      mmReport(INFO_ERROR, s_eType, "local media for unknown device");
      return ;
   }
   // increment the counter for Ready channels
   if ( (g_nReadyChannels++) >= g_nNumberOfCalls )
   {
      mmReport(INFO_DEBUG, s_eType,"\n---------> Ready to process %d Call(s) <--------", g_nReadyChannels-1);
   }
}

void CMMStream::OnAddOverlay(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnAddOverlay()");
}
void CMMStream::OnAddOverlayFail(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnAddOverlayFail()");
}
void CMMStream::OnRemoveOverlay(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnRemoveOverlay()");
}
void CMMStream::OnRemoveOverlayFail(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnRemoveOverlayFail()");
}
void CMMStream::OnRemoveAllOverlays(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnRemoveAllOverlays()");
}
void CMMStream::OnRemoveAllOverlaysFail(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnRemoveAllOverlaysFail()");
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnIpmStartMedia(void *pData)
// DESCRIPTION : When media has started, retrieve the digits and play the
//				 corresponding menu
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnIpmStartMedia(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnStartMedia()");
#ifdef USE_RTSP
   unsigned int    evtDev = sr_getevtdev();
   if (evtDev == m_ipmRTSPH) {
      m_bRTSPStreamingRtp = true;
      m_pRTSPClient->Play();
      mmReport(INFO_DEBUG, s_eType, "RTSPClient: Play()");
   } else if (evtDev == m_ipmCCH) {
      m_bCCStreamingRtp = true;
      dx_clrdigbuf(m_voxH);
      // start to get digits
      Getdigits();
      PlayVideoFile('0');
   }
#else
   m_bCCStreamingRtp = true;
   dx_clrdigbuf(m_voxH);
   // start to get digits
   Getdigits();
#ifdef USE_MTK_API
   // SKS Apply Overlay if transcoding is on for outgoing video
   if (g_nVideoTranscodeDir==RXTX || g_nVideoTranscodeDir==TX) {
      ApplyOverlay();
   }
#endif
   if (g_recordControl != RECORDING_MANUAL)
   {
      if (g_recordControl == RECORDING_IMMEDIATE){
      Record();
      }
      else {
         Play(menu_greeting);    
      }
   }

#endif //USE_RTSP
}

//*****************************************************************************
//         NAME : void CMMStream::OnIpmRfcEvent(void *pData)
// DESCRIPTION : Got a RFC2833 event via IPM
//        INPUT : None
//        OUTPUT : None
//       RETURNS : None
//      CAUTIONS : None
//*****************************************************************************
void CMMStream::OnIpmRfcEvent(void *pData)
{
    mmReport(INFO_DEBUG, s_eType, "OnIpmRfcEvent()");
    IPM_TELEPHONY_INFO *pDigit = (IPM_TELEPHONY_INFO *)pData;
    switch (pDigit->eTelInfoType)
    {
        case TEL_INFOTYPE_EVENT:
        {
            char rfc_digit;
            int event_dig;
            event_dig = pDigit->TelephonyInfo.TelEvtInfo.eTelephonyEventID;
            mmReport(INFO_DEBUG, s_eType, "OnIpmRfcEvent()::Rfc2833 event [digit = 0x%x]", event_dig);
            // Post a fake TDX_GETDIG event if in rfc2833 event mode b/c dx is not connected
            if (g_nDtmfDetectMode == dtmfMode_rfc2833) {
               if(event_dig < 0xa)
                  rfc_digit = (char)event_dig + '0';
               else if (event_dig == 0xa)
                  rfc_digit = '*';
               else if (event_dig == 0xb)
                  rfc_digit = '#';
               else 
                  rfc_digit = (char)event_dig - 12 + 'a';
               mmReport(INFO_DEBUG, s_eType, "OnIpmRfcEvent()::Forward TDX_GETDIG Event = %c", rfc_digit);
               METAEVENT metaevent;
               metaevent.evttype = TDX_GETDIG;
               metaevent.evtdatap = &rfc_digit;
               metaevent.evtlen = 1;
               ProcessEvent(metaevent, this);
            }
            else if (g_nDtmfDetectMode == dtmfMode_sipinfo) {
                mmReport(INFO_DEBUG, s_eType, "OnIpmRfcEvent()::Rfc2833 detected (ignored) in dtmfMode_sipinfo");
            }
            else {
                mmReport(INFO_DEBUG, s_eType, "OnIpmRfcEvent()::Inband detection of Re-generated tone expected");
            }
        }
        break;
    }
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnIpmStopped(void *pData)
// DESCRIPTION : When streaming stops reset appropriate flags
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnIpmStopped(void *pData)
{
    int nDevice = sr_getevtdev();
    int rc = -1;

    mmReport(INFO_DEBUG, s_eType, "OnIpmStopped()");
    if (nDevice == m_ipmCCH) {
      dx_stopch(m_voxH, EV_SYNC);
      // reset member variables.
      m_bCCStreamingRtp = false;
      m_nRfc2833PayloadType = 0;
      m_bIsGetDigitPending = false;
      m_bVideoCall = true;
      m_bPlayDone = true;
      m_bRecordDone = true;
      m_bWaiting    = false;
      m_bMMStopping = false;
      m_bMMPlayStarting = false;
      m_bMMRecordStarting = false;
      m_bSetState = true;
      m_playList.clear();
      m_bIsRecordPending = false;
    }
#ifdef USE_RTSP
    else if (nDevice == m_ipmRTSPH) {
      m_bSetState = false;
      m_bRTSPStreamingRtp = false;
      if (!m_qDtmf.empty()) {
         char dtmf = m_qDtmf.front();
	 m_qDtmf.pop();
	 PlayVideoFile(dtmf);
      }
    }
#endif
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnIpmGetparm(void *pData)
// DESCRIPTION : Retrieves IPM Parameter
// 		 INPUT : pData - void *
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnIpmGetparm(void *pData)
{
   // Local Variable Declaration
   IPM_PARM_INFO   *pParmInfo = 0;
   int             value;

   mmReport(INFO_DEBUG, s_eType, "OnGetparms()");
   pParmInfo = (IPM_PARM_INFO *)pData;

   value = *(int *)(pParmInfo->pvParmValue);
   mmReport(INFO_DEBUG, s_eType, "OnGetparms() -> Parm = 0x%x, Value = 0x%x", pParmInfo->eParm, value);
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnIpmError(void *pData)
// DESCRIPTION : When an error is received on the IPM Device Do Nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnIpmError(void *pData)
{
   mmReport(INFO_ERROR, s_eType, "OnIpmError()");
   switch ( m_eState )
   {
      case state_appStopping:
         break;
      default:
         mmReport(INFO_DEBUG, s_eType, "[%s] Stopping RTP Streaming ipm_Stop", m_ipmCCDevName);
         if ( ipm_Stop(m_ipmCCH, STOP_MEDIA, EV_ASYNC) < 0 )
         {
            mmReport(INFO_ERROR, s_eType, "ipm_Stop() on %s", m_ipmCCDevName);
         }
   }
}

// JS start

const char* CMMStream::MMErrorStr(unsigned int emm_err) {
    const char* err = NULL;
    switch (emm_err) {
	    case EMMRC_OK:  err = " Successful (no error)."; break;
	    case EMMRC_INVALIDARG:  err = " Invalid argument."; break;
	    case EMMRC_INVALID_FILEFORMAT:  err = " Invalid file format."; break;
	    case EMMRC_A_INVALID_STATE:  err = " Invalid state (audio)."; break;
	    case EMMRC_V_INVALID_STATE:  err = " Invalid state (video)."; break;
	    case EMMRC_AV_INVALID_STATE:  err = " Invalid state (audio/video)."; break;
	    case EMMRC_A_FILE_OPEN_FAILED:  err = " Audio file open failed"; break;
	    case EMMRC_V_FILE_OPEN_FAILED:  err = " Video file open failed."; break;
	    case EMMRC_UNSUPPORTED_MODE:  err = " Unsupported mode."; break;
	    case EMMRC_ALREADYSTOPPED:  err = " Device operations are already stopped."; break;
	    case EMMRC_MEMALLOC_ERROR:  err = " Memory allocation error."; break;
	    case EMMRC_MEMALLOC_POOLNOTFOUND:  err = " Memory allocation pool not found."; break;
	    case EMMRC_INVALIDSTATE_ERROR:  err = " Invalid state."; break;
	    case EMMRC_FILEREAD_FAILED:  err = " File read failed."; break;
	    case EMMRC_NOT_VIDEO_FILE:  err = " Video file format is invalid."; break;
	    case EMMRC_UNKNOWN_ERROR:  err = " Unknown error."; break;
	    case EMMRC_FAILED:  err = " Unspecified failure."; break;
	    case EMMRC_OUT_OF_VIDEO_LICENSE:  err = " Out of Video Licenses"; break;
	    case EMMRC_FILEWRITE_FAILED:  err = " File write failed."; break;
	    case EMMRC_RESOURCE_IN_USE:  err = " Device Resource in use"; break;
	    case EMMRC_I_FILE_OPEN_FAILED:  err = " Image file open failed"; break;
	    case EMMRC_I_INVALID_STATE:  err = " Invalid (Image file) state."; break;
	    case EMMRC_NOT_IMAGE_FILE:  err = " Image file format is imvalid."; break;
	    case EMMRC_I_INVALID_DATA:  err = " Image data not valid."; break;
	    case EMMRC_A_WAVE_FILE_ERROR:  err = " Audio (.wav) file error"; break;
	    case EMMRC_EOF:  err = " End of File"; break;
	    case EMMRC_A_NO_HINT_TRACK:  err = " Audio file - no hint track"; break;
	    case EMMRC_V_NO_HINT_TRACK:  err = " Video file - no hint track"; break;
	    case EMMRC_A_NO_TRACK:  err = " Audio file - no audio track"; break;
	    case EMMRC_V_NO_TRACK:  err = " Video file - no video track"; break;
	    case EMMRC_A_TOO_MANY_TRACKS:  err = " Audio file - too many audio track"; break;
	    case EMMRC_V_TOO_MANY_TRACKS:  err = " Video file - too many video track"; break;
	    case EMMRC_FILE_CORRUPT:  err = " File is corrupt"; break;
	    default:
               err = " Unknown reason";
    }
    return err;
}

const char* CMMStream::MMPlayRecCmpltStr(unsigned int eValue) {
    const char* Str = NULL;
    switch (eValue) {
	    case EMM_CMPLT_VIDEO_PLAY:  Str = " EMM_CMPLT_VIDEO_PLAY"; break;
	    case EMM_CMPLT_VIDEO_RECORD:  Str = " EMM_CMPLT_VIDEO_RECORD"; break;
	    case EMM_CMPLT_AUDIO_PLAY:  Str = " EMM_CMPLT_AUDIO_PLAY"; break;
	    case EMM_CMPLT_AUDIO_RECORD:  Str = " EMM_CMPLT_AUDIO_RECORD"; break;
	    case EMM_CMPLT_TONE:  Str = " EMM_CMPLT_TONE"; break;
	    case EMM_CMPLT_GET_DIGIT:  Str = " EMM_CMPLT_GET_DIGIT"; break;
	    case EMM_CMPLT_IMAGE_PLAY:  Str = " EMM_CMPLT_IMAGE_PLAY"; break;
	    case EMM_CMPLT_VIDEO_RECORD_WITH_UPDATE:  Str = " EMM_CMPLT_VIDEO_RECORD_WITH_UPDATE"; break;
	    case EMM_CMPLT_AUDIO_RECORD_WITH_UPDATE:  Str = " EMM_CMPLT_AUDIO_RECORD_WITH_UPDATE"; break;
	    case EMM_CMPLT_UNDEFINED:  Str = " EMM_CMPLT_UNDEFINED"; break;
	    default:
               Str = " Unknown Cmplt Value";
    }
    return Str;
}

const char* CMMStream::MMPlayRecCmpltReasonStr(unsigned int eValue) {
    const char* Str = NULL;
    switch (eValue) {
	    case EMM_TR_MAX_DTMF_DIGIT:  Str = " EMM_TR_MAX_DTMF_DIGIT"; break;
	    case EMM_TR_CONT_SILON:  Str = " EMM_TR_CONT_SILON"; break;
	    case EMM_TR_CONT_SILOFF:  Str = " EMM_TR_CONT_SILOFF"; break;
	    case EMM_TR_ID_DELAY:  Str = " EMM_TR_ID_DELAY"; break;
	    case EMM_TR_EOF:  Str = " EMM_TR_EOF"; break;
	    case EMM_TR_DIGIT:  Str = " EMM_TR_DIGIT"; break;
	    case EMM_TR_PATTERN_SILON_SILOFF:  Str = " EMM_TR_PATTERN_SILON_SILOFF"; break;
	    case EMM_TR_USERSTOP:  Str = " EMM_TR_USERSTOP"; break;
	    case EMM_TR_TONEID:  Str = " EMM_TR_TONEID"; break;
	    case EMM_TR_ERROR:  Str = " EMM_TR_ERROR"; break;
	    case EMM_TR_DIGMASK:  Str = " EMM_TR_DIGMASK"; break;
	    case EMM_TR_DIGTYPE:  Str = " EMM_TR_DIGTYPE"; break;
	    case EMM_TR_MAXTIME:  Str = " EMM_TR_MAXTIME"; break;
	    case EMM_TR_TIME_FROM_AUDIO_RECORD_STARTED:  Str = " EMM_TR_TIME_FROM_AUDIO_RECORD_STARTED"; break;
	    case EMM_TR_UNDEFINED:  Str = " EMM_TR_UNDEFINED"; break;
	    default:
               Str = " Unknown Reason Value";
    }
    return Str;
}

const char* CMMStream::MMPlayRecCmpltStatusStr(unsigned int eValue) {
    const char* Str = NULL;
    switch (eValue) {
	    case EMM_STATUS_SUCCESS:  Str = " EMM_STATUS_SUCCESS"; break;
	    case EMM_STATUS_RCRD_V_DRPD_FRAME_FULL_ERROR:  Str = " EMM_STATUS_RCRD_V_DRPD_FRAME_FULL_ERROR"; break;
	    case EMM_STATUS_RCRD_V_PKTS_DROPD_FS_GT_MFS:  Str = " EMM_STATUS_RCRD_V_PKTS_DROPD_FS_GT_MFS"; break;
	    case EMM_STATUS_RCRD_A_DRPD_FRAME_FULL_ERROR:  Str = " EMM_STATUS_RCRD_A_DRPD_FRAME_FULL_ERROR"; break;
	    case EMM_STATUS_PLAY_V_ERROR_FS_GT_MFS:  Str = " EMM_STATUS_PLAY_V_ERROR_FS_GT_MFS"; break;
	    case EMM_STATUS_PLAY_V_FILEREAD_ERROR:  Str = " EMM_STATUS_PLAY_V_FILEREAD_ERROR"; break;
	    case EMM_STATUS_PLAY_A_FILEREAD_ERROR:  Str = " EMM_STATUS_PLAY_A_FILEREAD_ERROR"; break;
	    case EMM_STATUS_PLAY_I_FILEREAD_ERROR:  Str = " EMM_STATUS_PLAY_I_FILEREAD_ERROR"; break;
	    case EMM_STATUS_PLAY_I_NULLDATA_ERROR:  Str = " EMM_STATUS_PLAY_I_NULLDATA_ERROR"; break;
	    case EMM_STATUS_RCRD_V_CODEC_MISMATCH_ERROR:  Str = " EMM_STATUS_RCRD_V_CODEC_MISMATCH_ERROR"; break;
	    case EMM_STATUS_RCRD_V_MISSING_MPEG4_VISUALCONFIG_ERROR:  Str = " EMM_STATUS_RCRD_V_MISSING_MPEG4_VISUALCONFIG_ERROR"; break;
	    case EMM_STATUS_PLAY_BAD_PACKET:  Str = " EMM_STATUS_PLAY_BAD_PACKET"; break;
	    case EMM_STATUS_RCRD_CODING_ERROR:  Str = " EMM_STATUS_RCRD_CODING_ERROR"; break;
	    case EMM_STATUS_PLAY_CODING_ERROR:  Str = " EMM_STATUS_PLAY_CODING_ERROR"; break;
	    case EMM_STATUS_RCRD_HEADER_WRITE_ERROR:  Str = " EMM_STATUS_RCRD_HEADER_WRITE_ERROR"; break;
	    case EMM_STATUS_UNDEFINED:  Str = " EMM_STATUS_UNDEFINED"; break;
	    default:
               Str = " Unknown Reason Value";
    }
    return Str;
}

const char* CMMStream::MMStopItemStr(unsigned int eValue) {
    const char* Str = NULL;
    switch (eValue) {
	    case EMM_STOP_VIDEO_PLAY:  Str = " EMM_STOP_VIDEO_PLAY"; break;
	    case EMM_STOP_VIDEO_RECORD:  Str = " EMM_STOP_VIDEO_RECORD"; break;
	    case EMM_STOP_AUDIO_PLAY:  Str = " EMM_STOP_AUDIO_PLAY"; break;
	    case EMM_STOP_AUDIO_RECORD:  Str = " EMM_STOP_AUDIO_RECORD"; break;
	    case EMM_STOP_AUDIO_PLAYTONE:  Str = " EMM_STOP_AUDIO_PLAYTONE"; break;
	    case EMM_STOP_GETDIGIT:  Str = " EMM_STOP_GETDIGIT"; break;
	    case EMM_STOP_IMAGE_PLAY:  Str = " EMM_STOP_IMAGE_PLAY"; break;
	    case EMM_STOP_UNDEFINED:  Str = " EMM_STOP_UNDEFINED"; break;
	    default:
               Str = " Unknown Stop Value";
    }
    return Str;
}

void CMMStream::OnMMPlayAckFail(void *pData)
{
   int nDevice = sr_getevtdev();
   MM_RET_CODE* pRetCode = (MM_RET_CODE*)sr_getevtdatap();
   m_bMMPlayStarting = false;
   mmReport(INFO_DEBUG, s_eType, "OnMMPlayAckFail(hMM=%d)",nDevice);
    
   mmReport(INFO_ERROR, s_eType, " MMPlayAckFail: Ret_Code: Version = 0x%X\n", pRetCode->unVersion);
   mmReport(INFO_ERROR, s_eType, " MMPlayAckFail: Ret_Code: unRetCode = %u\n", pRetCode->unRetCode);
   mmReport(INFO_ERROR, s_eType, " MMPLayAckFail: Error: %s\n", MMErrorStr(pRetCode->unRetCode));
   if(pRetCode->unRetCode==EMMRC_INVALID_FILEFORMAT || pRetCode->unRetCode==EMMRC_INVALIDARG)
      mmReport(INFO_ERROR, s_eType, "   Are you using codec type A files with codec type B RTP and Native Mode?\n");
   // SKS MMEV_PLAY_ACK_FAIL	
   StopPlay(menu_none);
   m_bPlayDone=true; 
}

void CMMStream::OnMMPlayFail(void *pData)
{
   int nDevice = sr_getevtdev();
   MM_PLAY_RECORD_CMPLT* pMmPlayRecCmplt = (MM_PLAY_RECORD_CMPLT*)pData;;

   mmReport(INFO_ERROR, s_eType, "OnMMPlayFail(hMM=%d)\n",nDevice);
 
   for( unsigned int i=0; i<pMmPlayRecCmplt->unCount; i++){
      mmReport(INFO_DEBUG1, s_eType, " MMPlayFail:: Details[%d]: Version = 0x%x", i, pMmPlayRecCmplt->details[i].unVersion);
      mmReport(INFO_DEBUG1, s_eType, " MMPlayFail:: Details[%d]: Complete = %d",i, (eMM_CMPLT_PLAY_RECORD)pMmPlayRecCmplt->details[i].Complete);
      mmReport(INFO_DEBUG1, s_eType, " MMPlayFail:: Details[%d]: CompleteStr = %s",i, MMPlayRecCmpltStr(pMmPlayRecCmplt->details[i].Complete));
      mmReport(INFO_DEBUG1, s_eType, " MMPlayFail:: Details[%d]: Reason = %d",i, (eMM_CMPLT_PLAY_RECORD_REASON)pMmPlayRecCmplt->details[i].Reason);
      mmReport(INFO_DEBUG1, s_eType, " MMPlayFail:: Details[%d]: ReasonStr = %s",i, MMPlayRecCmpltReasonStr(pMmPlayRecCmplt->details[i].Reason));
      mmReport(INFO_DEBUG1, s_eType, " MMPlayFail:: Details[%d]: unDuration= %u",i, pMmPlayRecCmplt->details[i].unDuration);
      mmReport(INFO_DEBUG1, s_eType, " MMPlayFail:: Details[%d]: unNumberOfBytes= %u",i, pMmPlayRecCmplt->details[i].unNumberOfBytes);
      mmReport(INFO_DEBUG1, s_eType, " MMPlayFail:: Details[%d]: Status = %d",i, (eMM_CMPLT_PLAY_RECORD_STATUS)pMmPlayRecCmplt->details[i].Status);
      mmReport(INFO_DEBUG1, s_eType, " MMPlayFail:: Details[%d]: StatusStr = %s",i, MMPlayRecCmpltStatusStr(pMmPlayRecCmplt->details[i].Status));
   } 
   m_bPlayDone=true; 
   
   if(!m_bMMStopping && !m_bMMPlayStarting && !m_bMMRecordStarting){
      if (m_bPlayDone==true){
         PLAY_LIST::iterator pl_Iter;
         pl_Iter = m_playList.begin();
         if ( pl_Iter != m_playList.end() )
         {
            Play(*pl_Iter);
            m_playList.erase(pl_Iter);
         }
      }
      if (!m_bMMPlayStarting && m_bRecordDone==true && m_bIsRecordPending == true){
         mmReport(INFO_DEBUG, s_eType, "OnPlayFail:Now Start Recording...");
         m_bIsRecordPending=false;
         Record();
      }
   }
   
}
// JS end


//*****************************************************************************
// 		  NAME : void CMMStream::OnIpmUndeclared(void *pData)
// DESCRIPTION : When Undeclared Event is received on the IPM Device Do Nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnIpmUndeclared(void *pData)
{
   METAEVENT *pEvt = (METAEVENT *)pData;
   mmReport(INFO_ERROR, s_eType, "OnUndeclared() state = %d, %s (%s)", m_eState, mmEvt2Str(pEvt->evttype), m_state_name);
}

//*****************************************************************************
// 		  NAME : void CMMStream::StartStream(void)
// DESCRIPTION : Start streaming
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::StartStream(void)
{
   mmReport(INFO_DEBUG, s_eType, "[%s] attempting ipm_StartMedia", m_ipmCCDevName);
   if ( m_nRfc2833PayloadType == 0 ) { 
      mmReport(INFO_DEBUG, s_eType, "StartStream()::SetDtmfMode(dtmfMode_inband)",m_ipmCCDevName);
      SetDtmfMode(dtmfMode_inband);
   } else {
      mmReport(INFO_DEBUG, s_eType, "StartStream()::SetDtmfMode(dtmfMode_rfc2833)",m_ipmCCDevName);
      SetDtmfMode(dtmfMode_rfc2833);
   }

   SetupCCRemoteMediaInfo();
   PrintRemoteLocalInfo();	

   if ( ipm_StartMedia(m_ipmCCH, &m_ccIpmMediaInfo, DATA_IP_TDM_BIDIRECTIONAL, EV_ASYNC) < 0 ) {
      mmReport(INFO_ERROR, s_eType, "ipm_StartMedia() error on %s", m_ipmCCDevName);
   }
}

#ifdef USE_RTSP
void CMMStream::PlayVideoFile(char dtmf)
{
   string &url = g_rtspUrls[dtmf];
   if (m_pRTSPClient) {
      m_pRTSPClient->Teardown();
      delete m_pRTSPClient;
   }
   m_pRTSPClient = new CRTSPClient(m_RTSPipmLocalMedia.MediaData[m_unLocalRTSPIPMAudioRtpPortIndex].mediaInfo.PortInfo.unPortId, 
                                     m_RTSPipmLocalMedia.MediaData[m_unLocalRTSPIPMVideoRtpPortIndex].mediaInfo.PortInfo.unPortId, this); // JS
   if (m_pRTSPClient->CreateRTSPSession(url.c_str(), false) != -1) {
      sprintf(m_cRTSPAudioVideoIp, m_pRTSPClient->GetRTSPVideoIP());
      mmReport(INFO_DEBUG, s_eType, "[%s] url is: %s\n", m_ipmRTSPDevName, url.c_str());
      }
      SetupRTSPRemoteMediaInfo2(); 
      mmReport(INFO_DEBUG, s_eType, "[%s] attempting ipm_StartMedia", m_ipmRTSPDevName);
      // call set remote media info.
      if ( ipm_StartMedia(m_ipmRTSPH,
                    &m_RTSPipmRemoteMedia,
                    DATA_IP_TDM_BIDIRECTIONAL,
//                    DATA_IP_RECEIVEONLY,
                    EV_ASYNC) < 0 )
      {
         mmReport(INFO_ERROR, s_eType, "ipm_StartMedia() on %s", m_ipmRTSPDevName);
      }
   }
   else {
      delete m_pRTSPClient;
      m_pRTSPClient = NULL;
      mmReport(INFO_DEBUG, s_eType, "Invalid URL or RTSP digit not configured");
   }
}
#endif

//*****************************************************************************
// 		  NAME : void CMMStream::StopStream(void *pData)
// DESCRIPTION : Stop streaming
// 		 INPUT : pData - void *
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::StopStream(void *pData)
{
#ifdef TDM_AUDIO
   if ( m_b2gIsBridged )
   {
      Disconnect2G();
   }
#endif

#ifdef USE_RTSP
   if (m_pRTSPClient != NULL)
   {
       m_pRTSPClient->Teardown();
       delete m_pRTSPClient;
       m_pRTSPClient = NULL;
   }
#else

#ifdef USE_MTK_API
   // SKS Remove Image Overlay
   if (g_nVideoTranscodeDir==RXTX || g_nVideoTranscodeDir==TX) {
      RemoveOverlay();
   }
#endif
   // Stop play/record here
   StopPlay(menu_none);
   StopRecord();
#endif

   if (m_bCCStreamingRtp) {
      mmReport(INFO_DEBUG, s_eType, "[%s] Stopping RTP Streaming ipm_Stop", m_ipmCCDevName);
      if ( ipm_Stop(m_ipmCCH, STOP_MEDIA, EV_ASYNC) < 0 )
      {
         mmReport(INFO_ERROR, s_eType, "ipm_Stop() on %s", m_ipmCCDevName);
      }
   }
#ifdef USE_RTSP
   if (m_bRTSPStreamingRtp) {
      mmReport(INFO_DEBUG, s_eType, "[%s] Stopping RTP Streaming ipm_Stop", m_ipmRTSPDevName);
      if ( ipm_Stop(m_ipmRTSPH, STOP_MEDIA, EV_ASYNC) < 0 )
      {
         mmReport(INFO_ERROR, s_eType, "ipm_Stop() on %s", m_ipmRTSPDevName);
      }
   }
#endif
}

//*****************************************************************************
// 		  NAME : bool CMMStream::SetDtmfMode(dtmfMode_e eMode)
// DESCRIPTION : Sets the DTMF XFer Mode based on the input
// 		 INPUT : eMode - Object of type dtmfMode containing the input
// 	    OUTPUT : None
// 	   RETURNS : Bool - True if function succeeds, False otherwise
// 	  CAUTIONS : None
//*****************************************************************************
bool CMMStream::SetDtmfMode(dtmfMode_e eMode)
{
   // Local Variable Declaration
   IPM_PARM_INFO       parmInfo;
   eIPM_DTMFXFERMODE   value;
   int                 PLType  = 0;
   switch ( eMode )
   {
      
      case dtmfMode_rfc2833:
         {
            // set the dtmf mode to RFC2833
            value                   = DTMFXFERMODE_RFC2833;
            parmInfo.eParm      = PARMCH_DTMFXFERMODE;
            parmInfo.pvParmValue    = &value;
            mmReport(INFO_DEBUG, s_eType, "[%s] DTMFs will be detected using RFC2833", m_ipmCCDevName);
            if ( ipm_SetParm(m_ipmCCH, &parmInfo, EV_SYNC) < 0 )
            {
               mmReport(INFO_ERROR, s_eType, "ipm_SetParm() on %s", m_ipmCCDevName);
            }

            // set the TX Payload Type
            PLType                  = m_nRfc2833PayloadType;
            parmInfo.eParm          = PARMCH_RFC2833EVT_TX_PLT;
            parmInfo.pvParmValue    = &PLType;
            mmReport(INFO_DEBUG, s_eType, "[%s] Setting Parameter PARMCH_RFC2833EVT_TX_PLT to %d", m_ipmCCDevName, m_nRfc2833PayloadType);
            if ( ipm_SetParm(m_ipmCCH, &parmInfo, EV_SYNC) < 0 )
            {
               mmReport(INFO_ERROR, s_eType, "ipm_SetParm() on %s", m_ipmCCDevName);
            }
            // set the RX Payload Type
            PLType              = m_nRfc2833PayloadType;
            parmInfo.eParm      = PARMCH_RFC2833EVT_RX_PLT;
            parmInfo.pvParmValue = &PLType;
            mmReport(INFO_DEBUG, s_eType, "[%s] Setting Parameter PARMCH_RFC2833EVT_RX_PLT to %d", m_ipmCCDevName, m_nRfc2833PayloadType);
            if ( ipm_SetParm(m_ipmCCH, &parmInfo, EV_SYNC) < 0 )
            {
               mmReport(INFO_ERROR, s_eType, "ipm_SetParm() on %s", m_ipmCCDevName);
            }
         }
         break;

      case dtmfMode_inband:
         {
            eIPM_DTMFXFERMODE value = DTMFXFERMODE_INBAND;
            parmInfo.eParm          = PARMCH_DTMFXFERMODE;
            parmInfo.pvParmValue    = &value;

            mmReport(INFO_DEBUG, s_eType, "[%s] DTMFs will be detected in INBAND Rtp", m_ipmCCDevName);
            if ( ipm_SetParm(m_ipmCCH, &parmInfo, EV_SYNC) < 0 )
            {
               mmReport(INFO_ERROR, s_eType, "ipm_SetParm() on %s", m_ipmCCDevName);
            }
         }
         break;
   }
   return true;
}

//*****************************************************************************
// 		  NAME : void CMMStream::SetIpmParms(void)
// DESCRIPTION : Sets IPM Parameters
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::SetIpmParms(void)
{
}

//*****************************************************************************
// 		  NAME : void CMMStream::GetIpmParms(void)
// DESCRIPTION : Retrieves IPM Parameters
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::GetIpmParms(void)
{
   // Local Variable Declaration
   IPM_PARM_INFO parmInfo;

   parmInfo.eParm = PARMCH_DTMFXFERMODE;
   if ( ipm_GetParm(m_ipmCCH, &parmInfo, EV_ASYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "ipm_GetParm() on %s", m_ipmCCDevName);
   }
}

//*****************************************************************************
// 		  NAME : void CMMStream::ConnectDevices(void)
// DESCRIPTION : Connect the IPM and MM devices 
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::ConnectDevices(void)
{
   mmReport(INFO_DEBUG, s_eType, "ConnectDevices()");
   if ( dev_Connect(m_ipmCCH, m_mmH, DM_FULLDUP, EV_ASYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "dev_Connect() failed on %s and %s", m_ipmCCDevName, m_mmDevName);
   }
}

//*****************************************************************************
// 		  NAME : void CMMStream::DisconnectMM(void)
// DESCRIPTION : Disconnect the MM Device
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::DisconnectMM(void)
{
   mmReport(INFO_DEBUG, s_eType, "DisconnectMM/RTSPIPM()");
#if USE_DEV_PORT_CONNECT // JS
#ifdef USE_RTSP
   if (dev_PortDisconnect(m_ipmRTSPH, &RTSPipmCnctList, this) < 0) 
   {
      mmReport(INFO_ERROR, s_eType, "dev_PortDisconnect() failed on %s", m_ipmRTSPDevName);
   }
   else
      mmReport(INFO_DEBUG, s_eType, "dev_PortDisconnect() called on %s", m_ipmRTSPDevName);
#else
   if (dev_PortDisconnect(m_mmH, &mmCnctList, this) < 0) 
   {
      mmReport(INFO_ERROR, s_eType, "dev_PortDisconnect() failed on %s", m_mmDevName);
   }
   else
      mmReport(INFO_DEBUG, s_eType, "dev_PortDisconnect() called on %s", m_mmDevName);
#endif // USE_RTSP
#else
   if ( dev_Disconnect(m_mmH, EV_ASYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "dev_Disconnect() failed on %s", m_mmDevName);
   }
   else
      mmReport(INFO_DEBUG, s_eType, "dev_Disconnect() called on %s", m_mmDevName);
#endif
}

//*****************************************************************************
// 		  NAME : void CMMStream::DisconnectIPM(void)
// DESCRIPTION : Disconnect the IPM Device
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::DisconnectIPM(void)
{
   mmReport(INFO_DEBUG, s_eType, "DisconnectIPM()");
#if USE_DEV_PORT_CONNECT // JS
   mmReport(INFO_DEBUG, s_eType, "dev_PortDisconnect() called on %s with CCipmCnctList", m_ipmCCDevName);
   if (dev_PortDisconnect(m_ipmCCH, &CCipmCnctList, this) < 0) 
   {
      mmReport(INFO_ERROR, s_eType, "dev_PortDisconnect() failed on %s", m_ipmCCDevName);
   }
#else
   if ( dev_Disconnect(m_ipmCCH, EV_ASYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "dev_Disconnect() failed on %s", m_ipmCCDevName);
   }
#endif
}

//*****************************************************************************
// 		  NAME : void CMMStream::GetMMStreamInfo(void)
// DESCRIPTION : On GetMMStreamInfo do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::GetMMStreamInfo(void)
{
   mmReport(INFO_DEBUG, s_eType, "GetMMStreamInfo()");
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnDevConnect(void *pData)
// DESCRIPTION : On OnDevConnect do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnDevConnect(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnDevConnect()");
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnMMOpen(void *pData)
// DESCRIPTION : When mm open is complete, get the local media information
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMOpen(void *pData)
{
   int nDevice = sr_getevtdev();
   int rc = -1;
   mmReport(INFO_DEBUG, s_eType, "OnMMOpen(h=%d)",nDevice);

   if(nDevice==m_mmH){
      rc = mm_Reset(m_mmH,NULL, this);
         if ( rc == -1 ) {
            mmReport(INFO_ERROR, s_eType, " mm_Reset(%s) failed", m_mmDevName);
         }
   }
}


//*****************************************************************************
// 		  NAME : void CMMStream::OnMMReset(void *pData)
// DESCRIPTION : When mm_Reset is complete, get the local media information
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMReset(void *pData)
{
   int nDevice = sr_getevtdev();
   int rc = -1;
   mmReport(INFO_DEBUG, s_eType, "OnMMReset(h=%d)",nDevice);

#if USE_DEV_PORT_CONNECT
   m_bHaveMMTxPort = false;
   m_bHaveMMRxPort = false;
   m_bHaveCCIPMTxPort = false;
   m_bHaveCCIPMRxPort = false;
   m_bHaveRTSPIPMTxPort = false;
   m_bHaveRTSPIPMRxPort = false;
        rc = dev_GetTransmitPortInfo(m_ipmCCH, this);
        if ( rc == -1 ) {
                mmReport(INFO_ERROR, s_eType, " dev_GetTransmitPortInfo() failed for %s", m_ipmCCDevName);
        }
#ifdef USE_RTSP
        rc = dev_GetTransmitPortInfo(m_ipmRTSPH, this);
        if ( rc == -1 ) {
                mmReport(INFO_ERROR, s_eType, " dev_GetTransmitPortInfo() failed for %s", m_ipmRTSPDevName);
        }
#else
       rc = dev_GetTransmitPortInfo(m_mmH, this);
        if ( rc == -1 ) {
                mmReport(INFO_ERROR, s_eType, " dev_GetTransmitPortInfo() failed for %s", m_mmDevName);
        }
#endif  // USE_RTSP

#else
   ConnectDevices();
#endif

   SetDtmfMode(dtmfMode_rfc2833);

#ifdef NO_RESERVATIONEX
   OnReserveResourceEx(NULL);
#else
   // reserve audio coder(s)
   if ( dev_ReserveResourceEx(m_ipmCCH, &m_ResourceList, EV_ASYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "ReserveResourceEx() failed for %s\n", m_ipmCCDevName);
      return;
   }
// SKS added on 19th March
#ifdef USE_RTSP
   if ( dev_ReserveResourceEx(m_ipmRTSPH, &m_ResourceList, EV_ASYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "ReserveResourceEx() failed for %s\n", m_ipmRTSPDevName);
      return;
   }
   mmReport(INFO_DEBUG, s_eType, "ReserveResourceEx %s, dev=%d", m_ipmRTSPDevName, m_ipmRTSPH);
#endif


   mmReport(INFO_DEBUG, s_eType, "ReserveResourceEx %s, dev=%d", m_ipmCCDevName, m_ipmCCH);
#endif

}

//*****************************************************************************
// 		  NAME : void CMMStream::OnMMGetStreamInfo(void)
// DESCRIPTION : On OnMMGetStreamInfo do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMGetStreamInfo(void)
{
   mmReport(INFO_DEBUG, s_eType, "OnMMGetStreamInfo()");
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnMMPlayAck(void *pData)
// DESCRIPTION : On OnMMPlayAck do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMPlayAck(void *pData)
{
   int nDevice = sr_getevtdev();
   mmReport(INFO_DEBUG, s_eType, "OnMMPlayAck(hMM=%d)",nDevice);
   m_bMMPlayStarting = false;

   if (m_bIsRecordPending == true){
      mmReport(INFO_DEBUG, s_eType, "OnMMPlayAck:Now Start Recording...");
      m_bIsRecordPending=false;
      Record();
   }

}

//*****************************************************************************
// 		  NAME : void CMMStream::OnMMStopAck(void *pData)
// DESCRIPTION : On OnMMStopAck (MMEV_STOP_ACK) do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMStopAck(void *pData)
{
   int nDevice = sr_getevtdev();
   MM_STOP_ACK* pMmStopAck = (MM_STOP_ACK*)pData;;

   mmReport(INFO_DEBUG, s_eType, "OnMMStopAck(hMM=%d)",nDevice);
   m_bMMStopping=false;
   
   for( unsigned int i=0; i<pMmStopAck->unCount; i++){
      mmReport(INFO_DEBUG1, s_eType, "StopAck:: Details[%d]: Version = 0x%x", i, pMmStopAck->details[i].unVersion);
      mmReport(INFO_DEBUG1, s_eType, "StopAck:: Details[%d]: StopItem = %d", i, (eMM_STOP)pMmStopAck->details[i].ItemType);
      mmReport(INFO_DEBUG1, s_eType, "StopAck:: Details[%d]: StopItemStr = %s", i, MMStopItemStr(pMmStopAck->details[i].ItemType));
      mmReport(INFO_DEBUG1, s_eType, "StopAck:: Details[%d]: RetCode = %d", i, pMmStopAck->details[i].unRetCode);
   }

}

//*****************************************************************************
// 		  NAME : void CMMStream::OnMMPlayDone(void *pData)
// DESCRIPTION : On OnMMPlayDone event set the flag for play complete
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMPlayDone(void *pData)
{
   int nDevice = sr_getevtdev();
   // retrive the user context passed when calling mm_Play()
   int menuItem = (int) sr_getUserContext();   
   mmReport(INFO_DEBUG, s_eType, "OnMMPlayDone(hMM=%d) menuItem=%d",nDevice, menuItem);
   m_bPlayDone = true;

   // if we are here due to a specific mm_Stop() call, then we don't have
   // to play anymore prompts or reloop...
   if ( m_bUserStop == true )
   {
      // reset this flag
      m_bUserStop = false;
      return;
   }

   if ( m_playList.size() == 0 )
   {
      switch ( m_eCurrentState )
      {
         // no need to stop and play as there is no pending play anyway.
         case state_playVPortalMenu:
            switch ( menuItem )
            {
               case clip_entertainment:
               case clip_sports:
               case clip_news:
               case menu_image:
                  m_playList.push_back(menu_vportal);
                  break;
               case menu_vportal:
                  // If menu repeat flag is set, start the play of menu again
                  if ( g_bRepeatMenus )
                     m_playList.push_back(menu_vportal);
                  break;
               default:
                  break;
            }
            break;
         case state_playVMailMenu:
            switch ( menuItem )
            {
               case menu_message:
               case menu_image:
                  m_playList.push_back(menu_vmail);
                  break;
               case menu_vmail:
                  if ( g_bRepeatMenus )
                     m_playList.push_back(menu_vmail);
                  break;
               default:
                  break;
            }
            break;
         case state_playMainMenu:
            switch ( menuItem )
            {
               case menu_greeting:
                  if ( g_bRepeatMenus )
                     m_playList.push_back(menu_greeting);
                  break;
               default:
                  break;
            }
            break;
         default:
            break;
      }
   }

   if(!m_bMMStopping && !m_bMMPlayStarting && !m_bMMRecordStarting){
      if (m_bPlayDone==true){
         PLAY_LIST::iterator pl_Iter;
         pl_Iter = m_playList.begin();
         if ( pl_Iter != m_playList.end() )
         {
            Play(*pl_Iter);
            m_playList.erase(pl_Iter);
         }
      }
      if (!m_bMMPlayStarting && m_bRecordDone==true && m_bIsRecordPending == true){
         mmReport(INFO_DEBUG, s_eType, "OnPlayDone:Now Start Recording...");
         m_bIsRecordPending=false;
         Record();
      }
   }


}

//*****************************************************************************
// 		  NAME : void CMMStream::OnMMVideoStarted(void *pData)
// DESCRIPTION : On OnMMVideoStarted do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMVideoStarted(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnMMVideoStarted()");
   mmReport(INFO_DEBUG, s_eType, "Got I-Frame in video stream");
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnMMRecordAck(void *pData)
// DESCRIPTION : On OnMMRecordAck do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMRecordAck(void *pData)
{
   int nDevice = sr_getevtdev();
   mmReport(INFO_DEBUG, s_eType, "OnMMRecordAck(hMM=%d)",nDevice);
   m_bMMRecordStarting = false;
}

//*****************************************************************************
//        NAME : void CMMStream::OnMMRecordAckFail(void *pData)
// DESCRIPTION : On OnMMRecordAckFail no record started
//       INPUT : None
//      OUTPUT : None
//     RETURNS : None
//    CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMRecordAckFail(void *pData)
{
   int nDevice = sr_getevtdev();
   MM_RET_CODE* pRetCode = (MM_RET_CODE*)pData;
   mmReport(INFO_ERROR, s_eType, "OnMMRecordAckFail(hMM=%d)\n",nDevice);
   m_bMMRecordStarting = false;

   mmReport(INFO_ERROR, s_eType, " RecordAckFail: Ret_Code: Version = 0x%x", pRetCode->unVersion);
   mmReport(INFO_ERROR, s_eType, " RecordAckFail: Ret_Code: unRetCode = %u", pRetCode->unRetCode);
   mmReport(INFO_ERROR, s_eType, " RecordAckFail: Error: %s", MMErrorStr(pRetCode->unRetCode));
   if(pRetCode->unRetCode==EMMRC_INVALID_FILEFORMAT || pRetCode->unRetCode==EMMRC_INVALIDARG)
      mmReport(INFO_ERROR, s_eType, "   Are you using codec type A files with codec type B RTP and Native Mode?\n");
   // MMEV_RECORD_ACK_FAIL	
   StopRecord();
   m_bRecordDone=true; 
}

//*****************************************************************************
//        NAME : void CMMStream::OnMMRecordFail(void *pData)
// DESCRIPTION : On OnMMRecordFail no record started
//       INPUT : None
//      OUTPUT : None
//     RETURNS : None
//    CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMRecordFail(void *pData)
{
   int nDevice = sr_getevtdev();
   MM_PLAY_RECORD_CMPLT* pMmPlayRecCmplt = (MM_PLAY_RECORD_CMPLT*)pData;;

   mmReport(INFO_ERROR, s_eType, "OnMMRecordFail(hMM=%d)\n",nDevice);
 
   for( unsigned int i=0; i<pMmPlayRecCmplt->unCount; i++){
      mmReport(INFO_DEBUG1, s_eType, " MMRecordFail:: Details[%d]: Version = 0x%x", i, pMmPlayRecCmplt->details[i].unVersion);
      mmReport(INFO_DEBUG1, s_eType, " MMRecordFail:: Details[%d]: Complete = %d",i, (eMM_CMPLT_PLAY_RECORD)pMmPlayRecCmplt->details[i].Complete);
      mmReport(INFO_DEBUG1, s_eType, " MMRecordFail:: Details[%d]: CompleteStr = %s",i, MMPlayRecCmpltStr(pMmPlayRecCmplt->details[i].Complete));
      mmReport(INFO_DEBUG1, s_eType, " MMRecordFail:: Details[%d]: Reason = %d",i, (eMM_CMPLT_PLAY_RECORD_REASON)pMmPlayRecCmplt->details[i].Reason);
      mmReport(INFO_DEBUG1, s_eType, " MMRecordFail:: Details[%d]: ReasonStr = %s",i, MMPlayRecCmpltReasonStr(pMmPlayRecCmplt->details[i].Reason));
      mmReport(INFO_DEBUG1, s_eType, " MMRecordFail:: Details[%d]: unDuration= %u",i, pMmPlayRecCmplt->details[i].unDuration);
      mmReport(INFO_DEBUG1, s_eType, " MMRecordFail:: Details[%d]: unNumberOfBytes= %u",i, pMmPlayRecCmplt->details[i].unNumberOfBytes);
      mmReport(INFO_DEBUG1, s_eType, " MMRecordFail:: Details[%d]: Status = %d",i, (eMM_CMPLT_PLAY_RECORD_STATUS)pMmPlayRecCmplt->details[i].Status);
      mmReport(INFO_DEBUG1, s_eType, " MMRecordFail:: Details[%d]: StatusStr = %s",i, MMPlayRecCmpltStatusStr(pMmPlayRecCmplt->details[i].Status));
   } 

   if(nDevice==m_mmH)
      m_bRecordDone=true;

   if(!m_bMMStopping && !m_bMMPlayStarting && !m_bMMRecordStarting){
      if (m_bPlayDone==true){
         PLAY_LIST::iterator pl_Iter;
         pl_Iter = m_playList.begin();
         if ( pl_Iter != m_playList.end() )
         {
            Play(*pl_Iter);
            m_playList.erase(pl_Iter);
         }
      }
      if (!m_bMMPlayStarting && m_bIsRecordPending == true){
         mmReport(INFO_DEBUG, s_eType, "OnRecordFail:Now Start Recording...");
         m_bIsRecordPending=false;
         Record();
      }
   }
}
//*****************************************************************************
//*****************************************************************************
// 		  NAME : void CMMStream::OnMMRecordDone(void *pData)
// DESCRIPTION : On OnMMRecordDone event set the flag for record complete
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMRecordDone(void *pData)
{
   int nDevice = sr_getevtdev();
   mmReport(INFO_DEBUG, s_eType, "OnMMRecordDone(hMM=%d)",nDevice);

   if (nDevice==m_mmH){
      m_bRecordDone = true;
   }
   
   if(!m_bMMStopping && !m_bMMPlayStarting && !m_bMMRecordStarting){
      if (m_bPlayDone==true){
         PLAY_LIST::iterator pl_Iter;
         pl_Iter = m_playList.begin();
         if ( pl_Iter != m_playList.end() )
         {
            Play(*pl_Iter);
            m_playList.erase(pl_Iter);
         }
      }
      if (m_bRecordDone==true && m_bIsRecordPending == true){
         mmReport(INFO_DEBUG, s_eType, "OnRecordDone:Now Start Recording...");
         m_bIsRecordPending=false;
         Record();
      }
   }

#if 0 //hmp3gp convert
   if(g_nFileFormat==MMD_3GP) {
   // temporary caling system function to convert using hmp3gp utility subjected hmp3gp utility is in /usr/dialogic/bin folder
      char convertCommand[512];
      sprintf(convertCommand,"hmp3gp -d0 %s %s %s",m_cVideoRecordFile.c_str(),m_cAudioRecordFile.c_str(),m_c3GPFileName.c_str());
      system(convertCommand);

      // SKS uncomment only when mtk api supports conversion from DMF to 3GP
      //convertTo3GP(pData);
   }
#endif
}

// SKS uncomment only when mtk api supports conversion from DMF to 3GP
#if 0
//*****************************************************************************
//      NAME : void CMMStream::outputLastMTKError()
// DESCRIPTION : printf output if any error while converting the file into .3gp
//     INPUT : None
//      OUTPUT : None
//     RETURNS : None
//    CAUTIONS : None
//*****************************************************************************

void CMMStream::outputLastMTKError()
{
   char str[MAX_STRING_SIZE];
   MTK_ERROR_INFO errorInfo;
   INIT_MTK_ERROR_INFO(&errorInfo);

   if(mtk_GetErrorInfo(&errorInfo) == MTK_ERROR)
   {
      sprintf(str,"ERROR getting last mtk error");
   }
   else
   {
      sprintf(str,"MTK Last Error: unErrorCode[%d], szErrorString[%s], szAdditionalInfo[%s]",
               errorInfo.unErrorCode, errorInfo.szErrorString, errorInfo.szAdditionalInfo);
   }
   mmReport(INFO_DEBUG, s_eType, str);
}

//*****************************************************************************
//      NAME : void CMMStream::destroyMTKResources(void *pData)
// DESCRIPTION : destroy mtk objects created to convert into 3gp file
//     INPUT : None
//      OUTPUT : None
//     RETURNS : None
//    CAUTIONS : None
//*****************************************************************************

void CMMStream::destroyMTKResources(void *pData)
{
   char str[MAX_STRING_SIZE];
   if (m_hDMFAudFile != MTK_ERROR)
   {
      if (mtk_DestroyMediaTemplate(m_hDMFAudFile) == MTK_ERROR)
      {
         sprintf(str,"ERROR:: returned from mtk_DestroyMediaTemplate(m_hDMFAudFile)");
         mmReport(INFO_ERROR, s_eType, str);
         outputLastMTKError();
      }
   }

   if (m_hDMFVidFile != MTK_ERROR)
   {
      if (mtk_DestroyMediaTemplate(m_hDMFVidFile) == MTK_ERROR)
      {
         sprintf(str,"ERROR:: returned from mtk_DestroyMediaTemplate(m_hDMFVidFile");
         mmReport(INFO_ERROR, s_eType, str);
         outputLastMTKError();
      }
   }

   if (m_h3GPFile != MTK_ERROR)
   {
      if (mtk_DestroyMediaTemplate(m_h3GPFile) == MTK_ERROR)
      {
         sprintf(str,"ERROR:: returned from mtk_DestroyMediaTemplate(m_h3GPFile\n");
         mmReport(INFO_ERROR, s_eType, str);
         outputLastMTKError();
      }
   }
}

//*****************************************************************************
//         	 NAME : void CMMStream::convertTo3GP(void *pData)
// 	DESCRIPTION : convert to 3gp from DMF
//        INPUT : None
//        OUTPUT : None
//       RETURNS : None
//      CAUTIONS : None
//*****************************************************************************

void CMMStream::convertTo3GP(void *pData)
{
   // SKS TODO conversion to 3gp
   char str[MAX_STRING_SIZE];
   MTK_HANDLE_LIST srcFiles;
   MTK_HANDLE_LIST destFiles;

   INIT_MTK_HANDLE_LIST(&srcFiles);
   INIT_MTK_HANDLE_LIST(&destFiles);

   // create the templates that represent the input aud and vid files
   if ((m_hDMFAudFile = mtk_CreateAudioMediaFileTemplate(eMTK_AUDIO_MEDIA_FILE_TYPE_DMF,
                                                        m_cAudioRecordFile.c_str())) == MTK_ERROR)
   {
      sprintf(str,"MTK_ERROR returned from mtk_CreateAudioMediaFileTemplate() for audio");
      mmReport(INFO_ERROR, s_eType, str);
      outputLastMTKError();
      destroyMTKResources(pData);
   }

   if ((m_hDMFVidFile = mtk_CreateVideoMediaFileTemplate(eMTK_VIDEO_MEDIA_FILE_TYPE_DMF,
                                                        m_cVideoRecordFile.c_str())) == MTK_ERROR)
   {
      sprintf(str,"MTK_ERROR returned from mtk_CreateVideoMediaFileTemplate() for video");
      mmReport(INFO_ERROR, s_eType, str);
      outputLastMTKError();
      destroyMTKResources(pData);
   }

   // create the template that represents the output 3gp file
   if ((m_h3GPFile = mtk_CreateContainerMediaFileTemplate(eMTK_CONTAINER_MEDIA_FILE_TYPE_3GP,
                                                         m_c3GPFileName)) == MTK_ERROR)
   {
      sprintf(str,"MTK_ERROR returned from mtk_CreateContainerMediaFileTemplate()");
      mmReport(INFO_ERROR, s_eType, str);
      outputLastMTKError();
      destroyMTKResources(pData);
   }

   // setup the input file template handle list containing the aud and vid
   // file template handles
   srcFiles.unCount = 2;
   srcFiles.hHandles[0] = m_hDMFAudFile;
   srcFiles.hHandles[1] = m_hDMFVidFile;

   // setup the output file template handle list only containing the 3gp file
   destFiles.unCount = 1;
   destFiles.hHandles[0] = m_h3GPFile;

   if (mc_Convert(&srcFiles, &destFiles) == MTK_ERROR)
   {
      sprintf(str,"ERROR:: mc_Convert() failed");
      mmReport(INFO_ERROR, s_eType, str);
      outputLastMTKError();
   }
   else
   {
      sprintf(str,"mc_Convert() completed succefully");
      mmReport(INFO_DEBUG, s_eType, str);
   }
   destroyMTKResources(pData);
}
#endif
//*****************************************************************************
// 		  NAME : void CMMStream::OnMMUndeclared(void *pData)
// DESCRIPTION : On OnMMUndeclared do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMUndeclared(void *pData)
{
   mmReport(INFO_ERROR, s_eType, "OnMMUndeclared()");
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnMMError(void *pData)
// DESCRIPTION : On MMError do nothing
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnMMError(void *pData)
{
   mmReport(INFO_ERROR, s_eType, "OnMMError()");
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnIFrameReqRcvd(void *pData)
//         DESCRIPTION : Send an IFrame request via a SIP INFO message
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnIFrameReqRcvd(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnIFrameReqRcvd()");

   mmReport(INFO_MSG, s_eType, "Recording active -> Sending periodic SIP INFO to request I-Frame");
   SendIFrameRequest();
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnAppWaitTimeoutRcvd(void *pData)
//         DESCRIPTION : Process app timeout event
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::OnAppWaitTimeoutRcvd(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnAppWaitTimeout()");

   mmReport(INFO_MSG, s_eType, "OnAppWaitTimeout -> Play()...");
   // Play only if there are no outstanding play requests
   if ( !m_bMMStopping && m_bPlayDone == true)
   {
      PLAY_LIST::iterator pl_Iter;
      pl_Iter = m_playList.begin();
      if ( pl_Iter != m_playList.end() )
      {
         Play(*pl_Iter);
         m_playList.erase(pl_Iter);
      }
   }
}

//*****************************************************************************
// 		  NAME : void CMMStream::OnVoxDigitRcvd(void *pData)
// DESCRIPTION : DTMF digit recd
// 		 INPUT : pData - void *
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
#ifdef USE_RTSP
void CMMStream::OnVoxDigitRcvd(void *pData)
{
   // Local Variable Declaration
   long termno = 0;
   char cDigit;

   if ( g_bUseSipInfoDtmf || g_nDtmfDetectMode == dtmfMode_sipinfo )
   {
      termno = TM_MAXDTMF;
      m_digits.dg_value[0] = *((char*)pData);
   }
   else if ( g_nDtmfDetectMode == dtmfMode_rfc2833 )
   {
      termno = TM_MAXDTMF;
      m_digits.dg_value[0] = *((char*)pData);
   }
   else
   {
      m_bIsGetDigitPending = false; // need to call dx_getdig() again
      // reset the flag to indicate that this function will not set the next state
      termno = ATDX_TERMMSK(m_voxH);
   }

   m_bSetState = false;
   mmReport(INFO_DEBUG, s_eType, "OnVoxDigitRcvd() -> Termmask=0x%x", termno);
   if ( termno & TM_MAXDTMF )
   {
      mmReport(INFO_DEBUG, s_eType, "OnVoxDigitRcvd() -> Got DTMF digit %c", m_digits.dg_value[0]);
      cDigit = m_digits.dg_value[0];
      QueueNextVideo(cDigit);
      StopMedia();
   }
    // restart looking for dtmfs
   if ( m_bCCStreamingRtp == true )
   {
      Getdigits();
   }
}

void CMMStream::QueueNextVideo(char dtmf)
{
   m_qDtmf.push(dtmf);
}
#else
void CMMStream::OnVoxDigitRcvd(void *pData)
{
   // Local Variable Declaration
   long termno = 0;
   char cDigit;

   if ( g_bUseSipInfoDtmf || g_nDtmfDetectMode == dtmfMode_sipinfo )
   {
      termno = TM_MAXDTMF;
      m_digits.dg_value[0] = *((char*)pData);
   }
   else if ( g_nDtmfDetectMode == dtmfMode_rfc2833 )
   {
      termno = TM_MAXDTMF;
      m_digits.dg_value[0] = *((char*)pData);
   }
   else
   {
      m_bIsGetDigitPending = false; // need to call dx_getdig() again
      // reset the flag to indicate that this function will not set the next state
      termno = ATDX_TERMMSK(m_voxH);
   }

   m_bSetState = true;
   mmReport(INFO_DEBUG, s_eType, "OnVoxDigitRcvd() -> Termmask = 0x%x", termno);
   if ( termno & TM_MAXDTMF )
   {
      mmReport(INFO_DEBUG, s_eType, "OnVoxDigitRcvd() -> Got DTMF digit %c", m_digits.dg_value[0]);
      cDigit = m_digits.dg_value[0];
      switch ( m_eCurrentState )
      {
         case state_playMainMenu:
            switch ( cDigit )
            {
               case '1':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Playing Video Portal Menu..");
                  // move the current state manually here
                  m_bSetState = false;
                  SetState(state_playVPortalMenu);
                  m_playList.push_back(menu_vportal);
                  if ( m_bPlayDone == false )
                  {
                     StopPlay(menu_none);
                     m_bPlayDone = false;
                  }
                  break;

               case '2':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Playing Video Mail Menu");
                  // move the current state manually here
                  m_bSetState = false;
                  SetState(state_playVMailMenu);
                  m_playList.push_back(menu_vmail);
                  if ( m_bPlayDone == false )
                  {
                     StopPlay(menu_none);
                     m_bPlayDone = false;
                  }
                  break;
#ifdef TDM_AUDIO
               case '4':
                  // Create a connection between this IPM device and its matching
                  // DTI device
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Connecting to 2G DTI channel");
                  Connect2G();
                  break;

               case '5':
                  // Tear down the connection between this IPM device and its matching
                  // DTI device
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Disconnecting 2G DTI channel");
                  Disconnect2G();
                  break;
#endif
               case '#':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Playing Main Menu");
                  // move the current state manually here
                  m_bSetState = false;
                  SetState(state_playMainMenu);
                  m_playList.push_back(menu_greeting);
                  if ( m_bPlayDone == false )
                  {
                     StopPlay(menu_none);
                     m_bPlayDone = false;
                  }
                  break;
            }
            break;

         case state_playVPortalMenu:
            switch ( cDigit )
            {
               case '1':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Play Clip 1");
                  m_playList.push_back(clip_entertainment);
                  if ( m_bPlayDone == false )
                  {
                     StopPlay(menu_none);
                     m_bPlayDone = false;
                  }
                  break;

               case '2':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Play Clip 2");
                  m_playList.push_back(clip_sports);
                  if ( m_bPlayDone == false )
                  {
                     StopPlay(menu_none);
                     m_bPlayDone = false;
                  }
                  break;

               case '3':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Play Clip 3");
                  m_playList.push_back(clip_news);
                  if ( m_bPlayDone == false )
                  {
                     StopPlay(menu_none);
                     m_bPlayDone = false;
                  }
                  break;

#ifdef TDM_AUDIO
               case '4':
                  // Create a connection between this IPM device and its matching
                  // DTI device
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Connecting to 2G DTI channel");
                  Connect2G();
                  // Send socket command to ISDN GW 
                  break;

               case '5':
                  // Tear down the connection between this IPM device and its matching
                  // DTI device
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Disconnecting 2G DTI channel");
                  Disconnect2G();
                  break;
#else
 // SKS 4 to capture image and 5 to play captured image
               case '4':
                  if(g_nVideoTranscodeDir==TX || g_nVideoTranscodeDir==RXTX) {
                     mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Capture Image now upon User Request");
                     PausePlay(true);
                  } else {
                     mmReport(INFO_ERROR, s_eType, "Got DTMF Digit - Can't Capture Image in Native Mode");
                  }
                  break;
               case '5':
                  if(g_nVideoTranscodeDir==TX || g_nVideoTranscodeDir==RXTX) 
                  {
                     mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Playing Captured Image now upon User Request");
                     m_playList.push_back(menu_image);
                     if ( m_bPlayDone == false )
                     {
                        StopPlay(menu_none);
                        m_bPlayDone = false;
                     }
                  } else {
                     mmReport(INFO_ERROR, s_eType, "Got DTMF Digit - Can't Play Image in Native Mode");
                  }
                  break;
#endif
// SKS DVR Control digits
               case '7':
               {
                  PausePlay(false);
		  break;
               }
               case '8':
               {
                  ResumePlay();
                  break;
               }				
               case '9':
               {
                  mmReport(INFO_MSG, s_eType, "DVR Control - Forwading Play on User Request");
                  ForwardRewindPlay(1000);
                  break;
               }
               case '0':
               {
                  mmReport(INFO_MSG, s_eType, "DVR Control - Rewinding Play on User Request");
                  ForwardRewindPlay(-1000);
                  break;
               }
               case '6':
               case '*': 
                  mmReport(INFO_MSG, s_eType, "Got unknown DTMF Digit -> Playing Video Portal Menu..");
                  // move the current state manually here
                  m_bSetState = false;
                  SetState(state_playVPortalMenu);
                  m_playList.push_back(menu_vportal);
                  if ( m_bPlayDone == false )
                  {
                     StopPlay(menu_none);
                     m_bPlayDone = false; 
                  }
                  break; 

               case '#':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Playing Main Menu");
                  // move the current state manually here
                  m_bSetState = false;
                  SetState(state_playMainMenu);
                  m_playList.push_back(menu_greeting);
                  if ( m_bPlayDone == false )
                  {
                     StopPlay(menu_none);
                     m_bPlayDone = false;
                  }
                  break;
            }
            break;

         case state_playVMailMenu:
         case state_recordMsg:
            switch ( cDigit )
            {
               case '1':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Start Recording..");
                  m_playList.clear();
                  m_bWaiting = false;
                  if ( m_bPlayDone == false || m_bRecordDone == false )
                  {
                     StopPlayRecord(menu_none);
                     m_bMMStopping=true;
                  }
                  if (g_bUseRecordMsg){
                     m_playList.push_back(while_recording_message);
                     m_bIsRecordPending = true;
                  }
                  else {
                     m_bIsRecordPending = true;
                  }
                  break;
               case '2':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Stop Recording");
                  m_playList.clear();
                  m_bSetState = false;
                  SetState(state_playVMailMenu);
                  if ( m_bPlayDone == false || m_bRecordDone == false )
                  {
                     StopPlayRecord(menu_none);
                     m_bMMStopping=true;
                  }
                  break;
               case '3':
                  m_playList.clear();
                  m_bWaiting = false;
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Start Playing audio/video file");
                  // move the current state manually here
                  m_bSetState = false;
                  SetState(state_playVMailMenu);
                  m_playList.push_back(menu_message);
                  if ( m_bPlayDone == false || m_bRecordDone == false )
                  {
                     StopPlayRecord(menu_none);
                     m_bMMStopping=true;
                  }
                  break;
#ifdef TDM_AUDIO
               case '4':
                  // Create a connection between this IPM device and its matching
                  // DTI device
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Connecting to 2G DTI channel");
                  Connect2G();
                  break;

               case '5':
                  // Tear down the connection between this IPM device and its matching
                  // DTI device
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Disconnecting 2G DTI channel");
                  Disconnect2G();
                  break;
#else		// SKS 4 to capture image and 5 to play captured image
               case '4':
                  if(g_nVideoTranscodeDir==TX || g_nVideoTranscodeDir==RXTX) {
                     mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Capture Image now upon User Request");
                     PausePlay(true);
                  } else {
                     mmReport(INFO_ERROR, s_eType, "Got DTMF Digit - Can't Capture Image in Native Mode");
                  }
                  break;
               case '5':
                  if(g_nVideoTranscodeDir==TX || g_nVideoTranscodeDir==RXTX) {
                     mmReport(INFO_MSG, s_eType, "Got DTMF Digit - Playing Captured Image now upon User Request");
                     m_bSetState = false;
                     SetState(state_playVMailMenu);
                     m_playList.push_back(menu_image);
                     if ( m_bPlayDone == false )
                     {
                        StopPlay(menu_none);
                        m_bPlayDone = false;
                     }
                  } else {
                     mmReport(INFO_ERROR, s_eType, "Got DTMF Digit - Can't Play Image in Native Mode");
                  }
                  break;
#endif

               case '#':
                  mmReport(INFO_MSG, s_eType, "Got DTMF Digit -> Playing Main Menu");
                  m_playList.clear();
                  m_bSetState = false;
                  SetState(state_playMainMenu);
                  m_playList.push_back(menu_greeting);
                  if ( m_bPlayDone == false || m_bRecordDone == false )
                  {
                     StopPlayRecord(menu_none);
                     m_bMMStopping=true;
                  }
                  break;
            }
            break;

         default:
            break;
      }
   }
   // restart looking for dtmfs
   if ( m_bCCStreamingRtp == true )
   {
      Getdigits();
   }

   // Play only if there are no outstanding play requests
   if ( !m_bMMStopping && m_bPlayDone == true)
   {
      PLAY_LIST::iterator pl_Iter;
      pl_Iter = m_playList.begin();
      if ( pl_Iter != m_playList.end() )
      {
         Play(*pl_Iter);
         m_playList.erase(pl_Iter);
      }
   }
   else {
      mmReport(INFO_MSG, s_eType, "Did not Play (m_bPlayDone=%s)",(m_bPlayDone?"true":"false"));
   }

}
#endif

#ifdef TDM_AUDIO
/*******************************************************************************
 *			NAME: SendCmdToISDNGW
 * Open a socket to the ISDN-PSTN gateway to transmit info on the availability
 * of an audio call on a clear channel
 *		  INPUT: cmd - command as string 
 *					
 *		RETURNS: none		
 ********************************************************************************/
void CMMStream::SendCmdToISDNGW(char* cmd)
{
   static int initialized = 0;
   static int sockHand;
   struct sockaddr_in server;
   int    length;
   int    bufflen;

   if ( ! initialized )
   {
      sockHand = socket(AF_INET ,SOCK_DGRAM , IPPROTO_UDP);
      if ( sockHand < 0 )
      {
         mmReport(INFO_ERROR, NONE, "sendCmdToISDNGW() socket() failure");
         return;
      }
      initialized = 1;
   }

   server.sin_family = AF_INET;   
   server.sin_addr.s_addr = inet_addr(g_cISDNGatewayIP);
   server.sin_port = htons(g_nISDNGatewayPort);
   length = sizeof(server); 

   bufflen = strlen(cmd) + 1;

   if ( sendto(sockHand, 
               cmd, 
               bufflen, 
               0, 
               (struct sockaddr *)&server, 
               length) != bufflen )
   {
      mmReport(INFO_ERROR, NONE, "sendCmdToISDNGW() sendto() failure");
   }
}


void CMMStream::Connect2G()
{
   m_bUserStop = true;
   StopPlay(menu_none);

   DisconnectMM();
   DisconnectIPM();

   // A normal set of connections in the Play mode is assumed.
   // Listen for audio from 3G caller.  DX device also remains connected listening
   // for DTMF tones.

   long         ts = 0;
   SC_TSINFO    scts;

   // Get the transmit timeslot for IPM device
   scts.sc_numts = 1;
   scts.sc_tsarrayp = &ts;
   if ( ipm_GetXmitSlot(m_ipmCCH, &scts, EV_SYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "Connect2G - ipm_GetXmitSlot() failed on %s", m_ipmCCDevName);
   }

   // make the DTI device listen to IPM
   if ( gc_Listen(m_dtiH, &scts, EV_SYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "Connect2g - gc_Listen() failed on %s", m_dtiDevName);
   }

   // Connect IPM to Xmit slot of DTI, so 3G caller can hear 2G caller.  This will override
   // the full duplex dev_Connect done between MM and IPM
   ts = 0;
   scts.sc_numts = 1;
   scts.sc_tsarrayp = &ts;
   if ( gc_GetXmitSlot(m_dtiH, &scts) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "Connect2G - gc_GetXmitSlot() failed on %s", m_dtiDevName);
   }

   // make the IPM device listen to DTI
   if ( ipm_Listen(m_ipmCCH, &scts, EV_SYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "Connect2G - ipm_Listen() failed on %s", m_ipmCCDevName);
   }
   else {
      mmReport(INFO_ERROR, s_eType, "ipm_Listen() success on %s", m_ipmCCDevName); 
   }

   // make voice device listen to DTI
   if ( dx_listen(m_voxH, &scts) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "dx_listen() failed on %s", m_voxDevName); 
   }
   else {
      mmReport(INFO_ERROR, s_eType, "dx_listen() success on %s", m_voxDevName); 
   }

   struct timeb timebuffer;
   ftime(&timebuffer); 
   char recFileName[64];
   struct tm *timeParts = localtime (&(timebuffer.time));
   sprintf( recFileName,"%s-%d-%d-%d-%d.pcm",
            m_voxDevName,
            timeParts->tm_hour,
            timeParts->tm_min, 
            timeParts->tm_sec, 
            timebuffer.millitm);

   m_recFileH = open( recFileName, O_CREAT | O_RDWR | O_TRUNC, 0666);
   if ( m_recFileH == -1 )
   {
      mmReport(ERROR_VOX, s_eType, "open failed:%d", errno);
   }
   else
   {
      DX_IOTT iott;
      iott.io_fhandle = m_recFileH;
      iott.io_type = IO_DEV | IO_EOT;
      iott.io_offset = 0;
      iott.io_length = -1;
      iott.io_nextp = NULL;
      iott.io_prevp = NULL;

      DX_XPB xpb; 
      xpb.wFileFormat = FILE_FORMAT_VOX; 
      xpb.wDataFormat = DATA_FORMAT_ALAW;
      xpb.nSamplesPerSec = DRT_8KHZ;
      xpb.wBitsPerSample = 8;

      if ( dx_reciottdata( m_voxH,  &iott, 0, &xpb, EV_ASYNC) == -1 )
      {
         mmReport(ERROR_VOX, s_eType, "dx_reciottdata failure:%s", ATDV_ERRMSGP(m_voxH));
      }
   } 

   m_b2gIsBridged = true;

   char cmdMsg[256];
   //  sprintf( cmdMsg, "C %s %s:%s", m_dtiDevName, m_ISDNCalledNumber, m_ISDNCallingNumber);
   sprintf( cmdMsg, "C %s %s", m_3GChannelTag, m_dtiDevName);
   SendCmdToISDNGW(cmdMsg);
   mmReport(INFO_MSG, s_eType, "2G Connected [%s]", cmdMsg); 
}


void CMMStream::Disconnect2G()
{

   ConnectDevices();

   // Now have the DTI device listening to the IPM device, and the
   // IPM device listening to the DTI device.
   // Undo these connections, and the dev_Connects still in place with
   // the MM device should still be active.  

   // Note that there is still a DX device listening to incoming IPM stream to
   // detect DTMF

   // Free up the DTI device from the IPM
   if ( gc_UnListen(m_dtiH, EV_SYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "Disconnect2g - gc_UnListen() failed on %s", m_dtiDevName);
   }

   // Do the same with the IPM device
   if ( ipm_UnListen(m_ipmCCH, EV_SYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "Disconnect2G - ipm_UnListen() failed on %s", m_ipmCCDevName);
   }

   if ( m_recFileH != -1 )
   {
      if ( dx_stopch( m_voxH, EV_ASYNC) == -1 )
      {
         mmReport(ERROR_VOX, s_eType, "dx_stopch failure:", ATDV_ERRMSGP(m_voxH));
      }
   }

   /*
   long        ts             = 0;
   SC_TSINFO      scts;
   scts.sc_numts = 1;
   scts.sc_tsarrayp = &ts;
   if ( ipm_GetXmitSlot(m_ipmH, &scts, EV_SYNC) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "ipm_GetXmitSlot() failed on %s", m_ipmDevName);
   }
 
   // make the Vox device listen to IPM
   scts.sc_numts = 1;
   scts.sc_tsarrayp = &ts;
   if ( dx_listen(m_voxH, &scts) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "dx_listen() failed on %s", m_voxDevName);
   }
   else {
      mmReport(INFO_ERROR, s_eType, "dx_listen() success on %s", m_voxDevName); 
   }
   */
   m_b2gIsBridged = false;

   //  char cmdMsg[256]; 
   //  sprintf( cmdMsg, "D %s %s:%s", m_dtiDevName, m_ISDNCalledNumber, m_ISDNCallingNumber); 
   //  SendCmdToISDNGW(cmdMsg);
   mmReport(INFO_MSG, s_eType, "2G Disconnected");
}

void CMMStream::OnDXRecordDone( void *pData)
{
   mmReport(INFO_MSG, s_eType, "OnDXRecordDone");
   close( m_recFileH);
}

#endif 

//*****************************************************************************
// 		  NAME : void CMMStream::Play(menu_e menuItem)
// DESCRIPTION : Plays the menu requested
// 		 INPUT : menuItem - object containing Menu information
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::Play(menu_e menuItem)
{
   MM_RUNTIME_CONTROL mmRtc;
   INIT_MM_RUNTIME_CONTROL(&mmRtc);
   
   mmReport(INFO_DEBUG, s_eType, "Play()");

   if ( m_bVideoCall == false )
   { // audio only call
      mmReport(INFO_DEBUG, s_eType, "Play()::Audio Only");
      //mmRtc.Reason = EMM_TERM_MAXTIME; //MAXTIME
      mmRtc.Reason = EMM_TERM_NORTC; //NO RTC
      mmRtc.Action = EMM_TA_AUDIO_STOP;
      mmRtc.unValue = 15000; // 15 seconds
      
       
      INIT_MM_MEDIA_ITEM_LIST(&m_audioMediaList);
      INIT_MM_MEDIA_AUDIO(&m_audioMediaList.item.audio);
      m_audioMediaList.ItemChain             = EMM_ITEM_EOT;
      m_audioMediaList.item.audio.codec         = m_mmAudioCodecPlay;
      m_audioMediaList.item.audio.unMode = 0;
      m_audioMediaList.item.audio.unOffset = 0;
      m_audioMediaList.item.audio.szFileName       = m_cAudioRecordFile.c_str();
      m_audioMediaList.item.audio.unAccessMode = MM_MEDIA_ACCESS_MODE_FILE;

      if(g_nAudioFileFormat==MMD_3GP) {
          m_audioMediaList.item.audio.eFileFormat       = EMM_FILE_FORMAT_3GP;
      } else if (g_nAudioFileFormat==MMD_WAV) {
          m_audioMediaList.item.audio.eFileFormat       = EMM_AUD_FILEFORMAT_WAVE;
      } else { //MMD_DMF
          if(m_mmAudioCodecPlay.unCoding!=MM_DATA_FORMAT_PCM) {
            m_audioMediaList.item.audio.eFileFormat     = EMM_AUD_FILEFORMAT_PROPRIETARY;
          } else {
            m_audioMediaList.item.audio.eFileFormat     = EMM_AUD_FILEFORMAT_VOX;
          }
      }

      INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[0]);
      INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[1]);
      m_playRecordList[0].ItemType 			= EMM_MEDIA_TYPE_AUDIO;
      m_playRecordList[0].list 				= &m_audioMediaList;
      m_playRecordList[0].ItemChain 			= EMM_ITEM_EOT;

      INIT_MM_PLAY_RECORD_INFO(&m_playInfo);
      m_playInfo.eFileFormat  				= EMM_FILE_FORMAT_PROPRIETARY;
      m_playInfo.list         				= m_playRecordList;
   }
   else
   { // video call
// SKS Image Play section start here
      if(menuItem==menu_image) {
         mmReport(INFO_DEBUG, s_eType, "Play()::Image");
         // SKS Image Play
         mmRtc.Reason = EMM_TERM_MAXTIME;
         mmRtc.Action = EMM_TA_VIDEO_STOP;
         mmRtc.unValue = 10000; // Stop Playing Image after 10 seconds
         
         INIT_MM_MEDIA_ITEM_LIST(&m_imageMediaList);
         INIT_MM_MEDIA_IMAGE(&m_imageMediaList.item.image);
         m_imageMediaList.ItemChain = EMM_ITEM_EOT;
         m_imageMediaList.item.image.szFileName       = m_cImageFileName.c_str();
         m_imageMediaList.item.image.eFormat      = eMTK_IMAGE_FORMAT_YUV;
       	 m_imageMediaList.item.image.unAccessMode = MM_MEDIA_ACCESS_MODE_FILE;
      	 m_imageMediaList.item.image.format.yuv.eFormat = eMTK_YUV_IMAGE_FORMAT_420;
      	 m_imageMediaList.item.image.format.yuv.unWidth = 176;
      	 m_imageMediaList.item.image.format.yuv.unHeight = 144;

      	 INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[0]);
      	 m_playRecordList[0].ItemType        = EMM_MEDIA_TYPE_IMAGE;
      	 m_playRecordList[0].list            = &m_imageMediaList;
      	 m_playRecordList[0].ItemChain          = EMM_ITEM_EOT;

      	 INIT_MM_PLAY_RECORD_INFO(&m_playInfo);
      	 m_playInfo.eFileFormat           = EMM_FILE_FORMAT_PROPRIETARY;
      	 m_playInfo.list                     = m_playRecordList;
      } // SKS Image Play section ends here
      else {
         mmReport(INFO_DEBUG, s_eType, "Play()::Audio+Video");
         mmRtc.Reason = EMM_TERM_NORTC;
         mmRtc.Action = EMM_TA_AUDIO_VIDEO_STOP;
         mmRtc.unValue = 5000; // 5 seconds

         INIT_MM_MEDIA_ITEM_LIST(&m_audioMediaList);
      	 INIT_MM_MEDIA_AUDIO(&m_audioMediaList.item.audio);
     	 m_audioMediaList.ItemChain             = EMM_ITEM_EOT;
         m_audioMediaList.item.audio.codec         = m_mmAudioCodecPlay;
      	 m_audioMediaList.item.audio.unMode = 0;
      	 m_audioMediaList.item.audio.unOffset = 0;
      	 m_audioMediaList.item.audio.szFileName       = m_cAudioRecordFile.c_str();
      	 m_audioMediaList.item.audio.unAccessMode = MM_MEDIA_ACCESS_MODE_FILE;

         if(g_nAudioFileFormat==MMD_3GP) {
            m_audioMediaList.item.audio.eFileFormat      = EMM_FILE_FORMAT_3GP;
         } else if (g_nAudioFileFormat==MMD_WAV) {
             m_audioMediaList.item.audio.eFileFormat       = EMM_AUD_FILEFORMAT_WAVE;
         } else { //MMD_DMF
            if(m_mmAudioCodecPlay.unCoding!=MM_DATA_FORMAT_PCM) {
               m_audioMediaList.item.audio.eFileFormat      = EMM_AUD_FILEFORMAT_PROPRIETARY;
            } else {
               m_audioMediaList.item.audio.eFileFormat   = EMM_AUD_FILEFORMAT_VOX;
            }
         }
      	 // init video
   
      	 INIT_MM_MEDIA_ITEM_LIST(&m_videoMediaList);
      	 INIT_MM_MEDIA_VIDEO(&m_videoMediaList.item.video);
      	 m_videoMediaList.ItemChain = EMM_ITEM_EOT;
      	 m_videoMediaList.item.video.szFileName 		= m_cVideoRecordFile.c_str();
         m_videoMediaList.item.video.codec=m_mmVideoCodecPlay;

         if(g_nVideoFileFormat==MMD_3GP) {
            m_videoMediaList.item.video.eFileFormat      = EMM_FILE_FORMAT_3GP;
         } else { //MMD_DMF
            m_videoMediaList.item.video.eFileFormat      = EMM_FILE_FORMAT_PROPRIETARY;
         }
         m_videoMediaList.item.video.unAccessMode = MM_MEDIA_ACCESS_MODE_FILE;

         INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[0]);
         m_playRecordList[0].ItemType 			= EMM_MEDIA_TYPE_AUDIO;
         m_playRecordList[0].list 			= &m_audioMediaList;
         m_playRecordList[0].ItemChain 			= EMM_ITEM_CONT;

         INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[1]);
         m_playRecordList[1].ItemType 			= EMM_MEDIA_TYPE_VIDEO;
         m_playRecordList[1].list 			= &m_videoMediaList;
         m_playRecordList[1].ItemChain 			= EMM_ITEM_EOT;

         INIT_MM_PLAY_RECORD_INFO(&m_playInfo);
         m_playInfo.eFileFormat  			= EMM_FILE_FORMAT_PROPRIETARY;
         m_playInfo.list         			= m_playRecordList;
      }
   }

   // specify the audio and video file
   switch ( menuItem )
   {
      case menu_greeting:
         m_audioMediaList.item.audio.szFileName = m_cMainMenuAudio.c_str();
         m_videoMediaList.item.video.szFileName = m_cMainMenuVideo.c_str();
         break;
      case menu_message:
         {
          if(g_nAudioFileFormat==MMD_3GP) {
             m_audioMediaList.item.audio.szFileName = m_c3GPFileName.c_str();
          } else if (g_nAudioFileFormat==MMD_DMF) {
             m_audioMediaList.item.audio.szFileName = m_cAudioRecordFile.c_str();
          }
          if(g_nVideoFileFormat==MMD_3GP) {
             m_videoMediaList.item.video.szFileName = m_c3GPFileName.c_str();
          } else if (g_nVideoFileFormat==MMD_DMF) {
             m_videoMediaList.item.video.szFileName = m_cVideoRecordFile.c_str();
          }
          break;
         }
      case while_recording_message:
         if(g_bUseRecordMsg){
            m_audioMediaList.item.audio.szFileName = m_cRecordingMsgAudioFile.c_str();
            m_videoMediaList.item.video.szFileName = m_cRecordingMsgVideoFile.c_str();
         }
         break;
      case menu_vportal:
         m_audioMediaList.item.audio.szFileName = m_cVportalMenuAudio.c_str();
         m_videoMediaList.item.video.szFileName = m_cVportalMenuVideo.c_str();
         break;
      case menu_vmail:
         m_audioMediaList.item.audio.szFileName = m_cVmailMenuAudio.c_str();
         m_videoMediaList.item.video.szFileName = m_cVmailMenuVideo.c_str();
         break;
      case clip_entertainment:
         m_audioMediaList.item.audio.szFileName = m_cClip1Audio.c_str();
         m_videoMediaList.item.video.szFileName = m_cClip1Video.c_str();
         break;
      case clip_sports:
         m_audioMediaList.item.audio.szFileName = m_cClip2Audio.c_str();
         m_videoMediaList.item.video.szFileName = m_cClip2Video.c_str();
         break;
      case clip_news:
         m_audioMediaList.item.audio.szFileName = m_cClip3Audio.c_str();
         m_videoMediaList.item.video.szFileName = m_cClip3Video.c_str();
         break;
   }

   // start play
   m_bPlayDone = false;
   m_bMMPlayStarting = true;
   if(menuItem==menu_image) {
      mmReport(INFO_DEBUG, s_eType, "Playing -> Image file = %s",  m_imageMediaList.item.image.szFileName);
   } else {
      if(m_bVideoCall){
         mmReport(INFO_DEBUG, s_eType, "Playing -> audioF(%d) = %s, videoF(%d) = %s",
                                    m_audioMediaList.item.audio.codec.unCoding, 
                                    m_audioMediaList.item.audio.szFileName, 
                                    m_videoMediaList.item.video.codec.Coding,
                                    m_videoMediaList.item.video.szFileName);
      } else {
         mmReport(INFO_DEBUG, s_eType, "Playing -> audioF(%d) = %s",
                                    m_audioMediaList.item.audio.codec.unCoding, 
                                    m_audioMediaList.item.audio.szFileName); 
      }
   }

//JM - debug     
   mmReport(INFO_DEBUG1, s_eType, "Playing...");
   mmReport(INFO_DEBUG1, s_eType, "\t Audio:");
   mmReport(INFO_DEBUG1, s_eType, "\t :audio File = %s", m_audioMediaList.item.audio.szFileName);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio eFileFormat = %d", m_audioMediaList.item.audio.eFileFormat);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio unMode = %d", m_audioMediaList.item.audio.unMode);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio unOffset = %d", m_audioMediaList.item.audio.unOffset);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio unAccessMode = %d", m_audioMediaList.item.audio.unAccessMode);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio.Codec unCoding = %d", m_audioMediaList.item.audio.codec.unCoding);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio.Codec unSampleRate = 0x%x", m_audioMediaList.item.audio.codec.unSampleRate);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio.Codec unBitsPerSample = %d", m_audioMediaList.item.audio.codec.unBitsPerSample);
   
    
   if(m_bVideoCall){
      mmReport(INFO_DEBUG1, s_eType, "");
      mmReport(INFO_DEBUG1, s_eType, "\t Video:");
      mmReport(INFO_DEBUG1, s_eType, "\t :video File = %s", m_videoMediaList.item.video.szFileName);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec Coding = %d", m_videoMediaList.item.video.codec.Coding);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec Profile = %d", m_videoMediaList.item.video.codec.Profile);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec Level = %d", m_videoMediaList.item.video.codec.Level);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec ImgWidth = %d", m_videoMediaList.item.video.codec.ImageWidth);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec ImgHeight = %d", m_videoMediaList.item.video.codec.ImageHeight);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec BitRate = %d", m_videoMediaList.item.video.codec.BitRate);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec FramesPerSec = 0x%6x", m_videoMediaList.item.video.codec.FramesPerSec);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec SamplingRate = %d", m_videoMediaList.item.video.codec.SamplingRate);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec VisualConfigSize = %d", m_videoMediaList.item.video.codec.VisualConfigSize);
   }


   if ( mm_Play(m_mmH, &m_playInfo, &mmRtc, (void *)menuItem) < 0 ) {
      mmReport(INFO_ERROR, s_eType, "mm_Play() failed on %s", m_mmDevName);
      if(fopen(m_audioMediaList.item.audio.szFileName,"r")==NULL){
         mmReport(INFO_ERROR, s_eType, "\t !ERROR:->File %s does not exist", m_audioMediaList.item.audio.szFileName);
      }
      if (m_bVideoCall){
         if(fopen(m_videoMediaList.item.video.szFileName,"r")==NULL){
            mmReport(INFO_ERROR, s_eType, "\t !ERROR:->File %s does not exist", m_videoMediaList.item.video.szFileName);
         }
      }
      StopPlay(menu_none);
      m_bPlayDone = true;
      m_bMMPlayStarting = false;
   }
}

// SKS Image Play/Capture/DVR Control function defination starts here
void CMMStream::ResumePlay(void)
{
   mmReport(INFO_DEBUG, s_eType, "Resuming Play on device %s",m_mmDevName);
   MM_RESUME_INFO resumeinfo;
   INIT_MM_RESUME_INFO(&resumeinfo);
   resumeinfo.unStreamType = EMM_RESUME_VIDEO_AUDIO;
   resumeinfo.unAttribute = EMM_RESUME_VIDEO_NONE;
   if (mm_Resume(m_mmH, &resumeinfo, NULL) == EMM_ERROR) {
         mmReport(INFO_ERROR, s_eType, "mm_Resume() failed on device %s",m_mmDevName);
         return;
   }
}
void CMMStream::CaptureImage(void)
{
   MM_MEDIA_IMAGE image;
   INIT_MM_MEDIA_IMAGE(&image);
   image.unAccessMode = MM_MEDIA_ACCESS_MODE_FILE;
   image.szFileName = m_cImageFileName.c_str();
   image.eFormat = eMTK_IMAGE_FORMAT_YUV;
   image.format.yuv.eFormat = eMTK_YUV_IMAGE_FORMAT_420;
   image.format.yuv.unWidth = 176;
   image.format.yuv.unHeight = 144;

   mmReport(INFO_DEBUG, s_eType, "Capturing Image on %s",m_mmDevName);
   if (mm_Capture(m_mmH, &image, NULL, NULL) == EMM_ERROR) {
        mmReport(INFO_ERROR, s_eType, "mm_Capture() failed on %s",m_mmDevName);
        return;
   }

}
void CMMStream::OnMMPause(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnMMPause()");
   bool bCaptureImage = (0 != sr_getUserContext());
//	if(m_eCurrentState==state_playVMailMenu || m_eCurrentState==state_playVPortalMenu) 
	if(bCaptureImage==true) {
			CaptureImage();
	}
}
void CMMStream::OnMMResume(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnMMResume() on device %s",m_mmDevName);
}
void CMMStream::OnMMCapture(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnCapture()");
   ResumePlay();
}

void CMMStream::PausePlay(bool bCaptureImage)
{
   MM_PAUSE_INFO pauseinfo;
   INIT_MM_PAUSE_INFO(&pauseinfo);
   pauseinfo.unStreamType = EMM_PAUSE_VIDEO_AUDIO;
   pauseinfo.unAttribute = EMM_PAUSE_AUDIO_SILENCE;
   if (mm_Pause(m_mmH, &pauseinfo, (void *)bCaptureImage) == EMM_ERROR) {
      mmReport(INFO_ERROR, s_eType, "mm_Pause() failed on device %s",m_mmDevName);
      return;
   }
   mmReport(INFO_DEBUG, s_eType, "Pausing Play on device %s",m_mmDevName);
}

void CMMStream::ForwardRewindPlay(int nOffset)
{
   MM_SEEK_INFO seekInfo;
   INIT_MM_SEEK_INFO(&seekInfo);
   seekInfo.unStreamType = EMM_SEEK_VIDEO_AUDIO;
   seekInfo.unOrigin=EMM_SEEK_CUR;
   seekInfo.nOffset=nOffset;
   if (mm_Seek(m_mmH, &seekInfo, NULL) == EMM_ERROR) {
      mmReport(INFO_ERROR, s_eType, "mm_Seek() failed on device %s",m_mmDevName);
      return;
   }
   if(nOffset>0) {
      mmReport(INFO_DEBUG, s_eType, "Forwarding video play to %d ms on device %s",nOffset,m_mmDevName);
   } else {
      mmReport(INFO_DEBUG, s_eType, "Rewinding video play to %d ms on device %s",nOffset,m_mmDevName);
   }
}

void CMMStream::OnMMSeek(void *pData)
{
   mmReport(INFO_DEBUG, s_eType, "OnMMSeek()");
}
// SKS Image Play/Capture/DVR Control function defination ends here

//*****************************************************************************
// 		  NAME : void CMMStream::Record(void)
// DESCRIPTION : Starts Recording
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::Record(void)
{
   MM_RUNTIME_CONTROL mmRtc;
   INIT_MM_RUNTIME_CONTROL(&mmRtc);

   //Audio Only
   if(!m_bVideoCall || g_bRecordAudioOnly){
      mmReport(INFO_DEBUG, s_eType, "Record()::Audio Only");
      
      if(g_nRecordTimeout){
         mmRtc.Reason = EMM_TERM_MAXTIME; //MAXTIME
         mmRtc.Action = EMM_TA_AUDIO_STOP;
         mmRtc.unValue = g_nRecordTimeout * 1000; // in ms
         mmReport(INFO_DEBUG, s_eType, "Record Timeout set to %d sec",g_nRecordTimeout);
      }
      else {
         mmRtc.Reason = EMM_TERM_NORTC; //NO RTC
         mmRtc.Action = EMM_TA_AUDIO_STOP;
         mmReport(INFO_DEBUG, s_eType, "Record Timeout set to %d sec - No RTC Used",g_nRecordTimeout);
      }         

      INIT_MM_MEDIA_ITEM_LIST(&m_audioMediaList);
      INIT_MM_MEDIA_AUDIO(&m_audioMediaList.item.audio);
      m_audioMediaList.ItemChain             = EMM_ITEM_EOT;
      m_audioMediaList.item.audio.codec         = m_mmAudioCodecRecord;
      m_audioMediaList.item.audio.unMode = MM_MODE_AUD_BEEPINITIATED;
      m_audioMediaList.item.audio.unOffset = 0;
      m_audioMediaList.item.audio.szFileName       = m_cAudioRecordFile.c_str();
      m_audioMediaList.item.audio.unAccessMode = MM_MEDIA_ACCESS_MODE_FILE;

      if (g_nAudioFileFormat==MMD_WAV) {
          m_audioMediaList.item.audio.eFileFormat       = EMM_AUD_FILEFORMAT_WAVE;
      } else { //MMD_DMF & MMD_3GP
          if(m_mmAudioCodecRecord.unCoding!=MM_DATA_FORMAT_PCM) {
            m_audioMediaList.item.audio.eFileFormat     = EMM_AUD_FILEFORMAT_PROPRIETARY;
          } else {
            m_audioMediaList.item.audio.eFileFormat     = EMM_AUD_FILEFORMAT_VOX;
          }
      }
      
      INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[0]);
      m_playRecordList[0].ItemType 			= EMM_MEDIA_TYPE_AUDIO;
      m_playRecordList[0].list 				= &m_audioMediaList;
      m_playRecordList[0].ItemChain 			= EMM_ITEM_EOT;

      INIT_MM_PLAY_RECORD_INFO(&m_recordInfo);
      m_recordInfo.eFileFormat  			= EMM_FILE_FORMAT_PROPRIETARY;
      m_recordInfo.list         			= m_playRecordList;
   }
   //Audio+Video Call
   else {
      mmReport(INFO_DEBUG, s_eType, "Record()::Audio+Video");
      
      if(g_nRecordTimeout){
         mmRtc.Reason = EMM_TERM_MAXTIME; //MAXTIME
         mmRtc.Action = EMM_TA_AUDIO_VIDEO_STOP;
         mmRtc.unValue = g_nRecordTimeout * 1000; // in ms
         mmReport(INFO_DEBUG, s_eType, "Record Timeout set to %d sec",g_nRecordTimeout);
      }
      else {
         mmRtc.Reason = EMM_TERM_NORTC; //NO RTC
         mmRtc.Action = EMM_TA_AUDIO_VIDEO_STOP;
         mmReport(INFO_DEBUG, s_eType, "Record Timeout set to %d sec - No RTC Used",g_nRecordTimeout);
      }         

      INIT_MM_MEDIA_ITEM_LIST(&m_audioMediaList);
      INIT_MM_MEDIA_AUDIO(&m_audioMediaList.item.audio);
      m_audioMediaList.ItemChain             = EMM_ITEM_EOT;
      m_audioMediaList.item.audio.codec         = m_mmAudioCodecRecord;
      m_audioMediaList.item.audio.unMode = MM_MODE_AUD_BEEPINITIATED;
      m_audioMediaList.item.audio.unOffset = 0;
      m_audioMediaList.item.audio.szFileName       = m_cAudioRecordFile.c_str();
      m_audioMediaList.item.audio.unAccessMode = MM_MEDIA_ACCESS_MODE_FILE;

      if (g_nAudioFileFormat==MMD_WAV) {
          m_audioMediaList.item.audio.eFileFormat       = EMM_AUD_FILEFORMAT_WAVE;
      } else { //MMD_DMF & MMD_3GP
          if(m_mmAudioCodecRecord.unCoding!=MM_DATA_FORMAT_PCM) {
            m_audioMediaList.item.audio.eFileFormat     = EMM_AUD_FILEFORMAT_PROPRIETARY;
          } else {
            m_audioMediaList.item.audio.eFileFormat     = EMM_AUD_FILEFORMAT_VOX;
          }
      }

      // init video
      INIT_MM_MEDIA_ITEM_LIST(&m_videoMediaList);
      INIT_MM_MEDIA_VIDEO(&m_videoMediaList.item.video);
      m_videoMediaList.ItemChain = EMM_ITEM_EOT;
      m_videoMediaList.item.video.szFileName    = m_cVideoRecordFile.c_str();
      m_videoMediaList.item.video.codec=m_mmVideoCodecRecord;

      m_videoMediaList.item.video.eFileFormat      = EMM_FILE_FORMAT_PROPRIETARY;
      m_videoMediaList.item.video.unAccessMode = MM_MEDIA_ACCESS_MODE_FILE;

      INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[0]);
      m_playRecordList[0].ItemType 			= EMM_MEDIA_TYPE_AUDIO;
      m_playRecordList[0].list 				= &m_audioMediaList;
      m_playRecordList[0].ItemChain 			= EMM_ITEM_CONT;

      INIT_MM_PLAY_RECORD_LIST(&m_playRecordList[1]);
      m_playRecordList[1].ItemType 			= EMM_MEDIA_TYPE_VIDEO;
      m_playRecordList[1].list 				= &m_videoMediaList;
      m_playRecordList[1].ItemChain 			= EMM_ITEM_EOT;

      INIT_MM_PLAY_RECORD_INFO(&m_recordInfo);
      m_recordInfo.eFileFormat  			= EMM_FILE_FORMAT_PROPRIETARY;
      m_recordInfo.list         			= m_playRecordList;

      // send a SIP INFO to request an I-Frame
      mmReport(INFO_MSG, s_eType, "Recording -> Sending SIP INFO to request I-Frame");
      // SKS print DCI value if present
      if(m_videoMediaList.item.video.codec.Coding== VIDEO_CODING_MP4V_ES && m_videoMediaList.item.video.codec.VisualConfigSize>0) {
         unsigned int visCtr;
         char str[256];
         strcpy(str,"VisualConfig=");
         for(visCtr=0;visCtr<m_videoMediaList.item.video.codec.VisualConfigSize;visCtr++)
         {
             sprintf(str,"%s%.2x",str,m_videoMediaList.item.video.codec.VisualConfiguration[visCtr]);
         }
         mmReport(INFO_MSG, s_eType, "Recording -> VisualConfigSize=%d | %s", m_videoMediaList.item.video.codec.VisualConfigSize, str);
      }

      SendIFrameRequest();
   }
      
   // start record
   m_bRecordDone = false;
   m_bMMRecordStarting = true;


   if(m_bVideoCall){
      mmReport(INFO_DEBUG, s_eType, "Recording(A+V) audioF(%d) = %s, videoF(%d) = %s",  
                                     m_audioMediaList.item.audio.codec.unCoding, 
                                     m_audioMediaList.item.audio.szFileName, 
                                     m_videoMediaList.item.video.codec.Coding, 
                                     m_videoMediaList.item.video.szFileName);
   } else {
      mmReport(INFO_DEBUG, s_eType, "Recording(Audio Only) audioF(%d) = %s",
                                    m_audioMediaList.item.audio.codec.unCoding, 
                                    m_audioMediaList.item.audio.szFileName);
   }
 
//JM - debug     
   mmReport(INFO_DEBUG1, s_eType, "Recording...");
   mmReport(INFO_DEBUG1, s_eType, "\t Audio:");
   mmReport(INFO_DEBUG1, s_eType, "\t :audio File = %s", m_audioMediaList.item.audio.szFileName);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio eFileFormat = %d", m_audioMediaList.item.audio.eFileFormat);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio unMode = %d", m_audioMediaList.item.audio.unMode);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio unOffset = %d", m_audioMediaList.item.audio.unOffset);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio unAccessMode = %d", m_audioMediaList.item.audio.unAccessMode);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio.Codec unCoding = %d", m_audioMediaList.item.audio.codec.unCoding);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio.Codec unSampleRate = 0x%x", m_audioMediaList.item.audio.codec.unSampleRate);
   mmReport(INFO_DEBUG1, s_eType, "\t :audio.Codec unBitsPerSample = %d", m_audioMediaList.item.audio.codec.unBitsPerSample);
   
    
   if(m_bVideoCall){
      mmReport(INFO_DEBUG1, s_eType, "");
      mmReport(INFO_DEBUG1, s_eType, "\t Video:");
      mmReport(INFO_DEBUG1, s_eType, "\t :video File = %s", m_videoMediaList.item.video.szFileName);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec Coding = %d", m_videoMediaList.item.video.codec.Coding);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec Profile = %d", m_videoMediaList.item.video.codec.Profile);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec Level = %d", m_videoMediaList.item.video.codec.Level);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec ImgWidth = %d", m_videoMediaList.item.video.codec.ImageWidth);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec ImgHeight = %d", m_videoMediaList.item.video.codec.ImageHeight);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec BitRate = %d", m_videoMediaList.item.video.codec.BitRate);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec FramesPerSec = 0x%6x", m_videoMediaList.item.video.codec.FramesPerSec);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec SamplingRate = %d", m_videoMediaList.item.video.codec.SamplingRate);
      mmReport(INFO_DEBUG1, s_eType, "\t :video.Codec VisualConfigSize = %d", m_videoMediaList.item.video.codec.VisualConfigSize);
   }

   if ( mm_Record(m_mmH, &m_recordInfo, &mmRtc , 0) < 0 )
   {
      mmReport(INFO_ERROR, s_eType, "mm_Record() failed on %s", m_mmDevName);
      m_bRecordDone = true;
      m_bMMRecordStarting = false;
   }

   return;
}

//*****************************************************************************
// 		  NAME : void CMMStream::StopPlay(menu_e menuItem)
// DESCRIPTION : Stops the Play
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::StopPlay(menu_e menuItem)
{
   mmReport(INFO_DEBUG, s_eType, "StopPlay() m_bPlayDone=%s",(m_bPlayDone?"true":"false"));
   if ( m_bPlayDone == false )
   {
      MM_STOP mmStopInfo[2];
      MM_STOP_DETAILS mmStopDetails;
      //memset(&mmStopDetails, 0, sizeof(MM_STOP_DETAILS));
      INIT_MM_STOP_DETAILS(&mmStopDetails);
      INIT_MM_STOP(mmStopInfo);

      mmStopInfo[0].unVersion = 0;
      mmStopInfo[0].ItemChain = EMM_ITEM_CONT;
      mmStopInfo[0].ItemType = EMM_STOP_VIDEO_PLAY;
      mmStopInfo[0].details = mmStopDetails;
      mmStopInfo[0].next = &mmStopInfo[1];

      mmStopInfo[1].unVersion = 0;
      mmStopInfo[1].ItemChain = EMM_ITEM_EOT;
      mmStopInfo[1].ItemType = EMM_STOP_AUDIO_PLAY;
      mmStopInfo[1].details = mmStopDetails;
      mmStopInfo[1].next = NULL;

      //JM - should this be here or just OnMMPlayDone()?
      //m_bPlayDone = true;

      if ( mm_Stop(m_mmH, mmStopInfo, (void *)menuItem) < 0 )
      {
         mmReport(INFO_ERROR, s_eType, "mm_Stop() for Play failed on %s", m_mmDevName);
      }
   }
}

//*****************************************************************************
// 		  NAME : void CMMStream::StopRecord(void)
// DESCRIPTION : Stops recording
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::StopRecord(void)
{
   mmReport(INFO_DEBUG, s_eType, "StopRecord() m_bRecordDone=%s",(m_bRecordDone?"true":"false"));
   if ( m_bRecordDone == false )
   {
      MM_STOP mmStopInfo[2];
      MM_STOP_DETAILS mmStopDetails;
      //memset(&mmStopDetails, 0, sizeof(MM_STOP_DETAILS));
      INIT_MM_STOP_DETAILS(&mmStopDetails);
      INIT_MM_STOP(mmStopInfo);

      mmStopInfo[0].unVersion   = 0;
      mmStopInfo[0].ItemChain   = EMM_ITEM_CONT;
      mmStopInfo[0].ItemType    = EMM_STOP_VIDEO_RECORD;
      mmStopInfo[0].details     = mmStopDetails;
      mmStopInfo[0].next        = &mmStopInfo[1];

      mmStopInfo[1].unVersion   = 0;
      mmStopInfo[1].ItemChain   = EMM_ITEM_EOT;
      mmStopInfo[1].ItemType    = EMM_STOP_AUDIO_RECORD;
      mmStopInfo[1].details     = mmStopDetails;
      mmStopInfo[1].next        = NULL;

      // set the flag to indicate that we have stopped record
      //m_bRecordDone = true;
      if ( mm_Stop(m_mmH, mmStopInfo, 0) < 0 )
      {
         mmReport(INFO_ERROR, s_eType, "mm_Stop() for Record failed on %s", m_mmDevName);
      }
   }
}

//*****************************************************************************

//*****************************************************************************
// 		  NAME : void CMMStream::StopPlayRecord(void)
// DESCRIPTION : Stops play and/or record
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::StopPlayRecord(menu_e menuItem)
{
   int StopCnt = -1;
   MM_STOP mmStopInfo[4];
   MM_STOP_DETAILS mmStopDetails;
   INIT_MM_STOP_DETAILS(&mmStopDetails);
   INIT_MM_STOP(mmStopInfo);
   
   mmReport(INFO_DEBUG, s_eType, "StopPlayRecord() m_bPlayDone=%s, m_bRecordDone=%s",
                                 (m_bPlayDone?"true":"false"), (m_bRecordDone?"true":"false"));
   
   if ( m_bRecordDone == false )
   {
      if (m_bVideoCall){
         StopCnt++;
         mmStopInfo[StopCnt].ItemType    = EMM_STOP_VIDEO_RECORD;
         mmStopInfo[StopCnt].details     = mmStopDetails;
         mmStopInfo[StopCnt].next        = &mmStopInfo[StopCnt+1];
         mmStopInfo[StopCnt].ItemChain   = EMM_ITEM_CONT;      
      }
     
      StopCnt++; 
      mmStopInfo[StopCnt].ItemType    = EMM_STOP_AUDIO_RECORD;
      mmStopInfo[StopCnt].details     = mmStopDetails;
      mmStopInfo[StopCnt].next        = NULL;
      mmStopInfo[StopCnt].ItemChain   = EMM_ITEM_EOT;
      // set the flag to indicate that we have stopped record
      //m_bRecordDone = true;
   }

   if ( m_bPlayDone == false )
   {
      if (StopCnt>=0){
         mmStopInfo[StopCnt].next        = &mmStopInfo[StopCnt+1];
         mmStopInfo[StopCnt].ItemChain   = EMM_ITEM_CONT;
      }
      if (m_bVideoCall){   
         StopCnt++;
         mmStopInfo[StopCnt].ItemType    = EMM_STOP_VIDEO_PLAY;
         mmStopInfo[StopCnt].details     = mmStopDetails;
         mmStopInfo[StopCnt].next        = &mmStopInfo[StopCnt+1];
         mmStopInfo[StopCnt].ItemChain   = EMM_ITEM_CONT;
      }
      StopCnt++;
      mmStopInfo[StopCnt].ItemType    = EMM_STOP_AUDIO_PLAY;
      mmStopInfo[StopCnt].details     = mmStopDetails;
      mmStopInfo[StopCnt].next        = NULL;
      mmStopInfo[StopCnt].ItemChain   = EMM_ITEM_EOT;
      // set the flag to indicate that we have stopped play
      //m_bPlayDone = true;
   }

   if (StopCnt>=0){
      if ( mm_Stop(m_mmH, mmStopInfo, (void*)menuItem) < 0 )
      {
         mmReport(INFO_ERROR, s_eType, "mm_Stop() for Play/Record failed on %s", m_mmDevName);
      }
   }
}

//*****************************************************************************
// 		  NAME : void CMMStream::Getdigits()
// DESCRIPTION : Retrieves the digits
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::Getdigits()
{
   if ( g_bUseSipInfoDtmf || g_nDtmfDetectMode == dtmfMode_sipinfo )
   {
      mmReport(INFO_DEBUG, s_eType, "Getdigits(ignored) - using SIP INFO");
   }
   else if ( g_nDtmfDetectMode == dtmfMode_rfc2833 )
   {
      mmReport(INFO_DEBUG, s_eType, "Getdigits(ignored) - using RFC2833 Event");
   }
   else
   {
      mmReport(INFO_DEBUG, s_eType, "Getdigits()");

      // check if we already have a pending dx_getdig() call
      if ( m_bIsGetDigitPending == true )
         return;

      //dx_stopch(m_voxH, EV_SYNC);
      // use dx_getdig to navigate application
      m_tpt[0].tp_type    = IO_EOT;
      dx_clrtpt(m_tpt, 1);
      m_tpt[0].tp_termno  = DX_MAXDTMF;
      m_tpt[0].tp_length  = 1; // look for 1 digit
      m_tpt[0].tp_flags   = TF_MAXDTMF;

      m_digits.dg_value[0] = 0;

      if ( dx_getdig(m_voxH, m_tpt, &m_digits, EV_ASYNC) < 0 )
      {
         mmReport(INFO_ERROR, s_eType, "dx_getdig() failed on %s err = %s", m_voxDevName, ATDV_ERRMSGP(m_voxH));
      }
      m_bIsGetDigitPending = true;
   }
}
//*****************************************************************************
//      NAME : bool CMMStream::PopulateOfferSdp()
// DESCRIPTION : Populates the SDP information
//     INPUT : None
//      OUTPUT : None
//     RETURNS : Bool - True if function succeeds, False otherwise
//    CAUTIONS : None
//*****************************************************************************
void CMMStream::PopulateOfferSdp(char *sdpOffer)
{
   sprintf(sdpOffer,"v=0\no=IPMediaServer 0 0 IN IP4 %s\ns=Video Demonstration\nc=IN IP4 %s\n t=0 0\nm=video %d RTP/AVP 102 34\nb=AS:%d\na=rtpmap:102 mp4v-es/90000\na=fmtp:102 profile-level-id=0\na=rtpmap:34 h263/90000\na=fmtp:34 QCIF=2\nm=audio %d RTP/AVP 8 0 4 18 96 101\na=rtpmap:8 pcma/8000\na=rtpmap:0 pcmu/8000\na=rtpmap:4 g723/8000\na=fmtp:4 annexa=yes; bitrate=6.3\na=rtpmap:18 g729/8000\na=fmtp:18 annexb=no\na=rtpmap:96 amr/8000\na=fmtp:96 mode-set=7;octet-align=1\na=rtpmap:101 telephone-event/8000",
      m_CCipmLocalMedia.MediaData[m_unLocalCCIPMAudioRtpPortIndex].mediaInfo.PortInfo.cIPAddress,
      m_CCipmLocalMedia.MediaData[m_unLocalCCIPMAudioRtpPortIndex].mediaInfo.PortInfo.cIPAddress,
      m_CCipmLocalMedia.MediaData[m_unLocalCCIPMVideoRtpPortIndex].mediaInfo.PortInfo.unPortId,
      ((g_nVideoTranscodeDir==TX || g_nVideoTranscodeDir==RXTX)?(m_stRemoteEPVideoCodec.m_nBitRate/1000):g_maxBandwidth),
      m_CCipmLocalMedia.MediaData[m_unLocalCCIPMAudioRtpPortIndex].mediaInfo.PortInfo.unPortId);
}


//*****************************************************************************
// 		  NAME : bool CMMStream::PopulateAnswerSdp()
// DESCRIPTION : Populates the SDP information
// 		 INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : Bool - True if function succeeds, False otherwise
// 	  CAUTIONS : None
//*****************************************************************************
sdpSessionDescription *CMMStream::PopulateAnswerSdp()
{
   char temp[100];
   mmReport(INFO_DEBUG, s_eType, "\t\tGenerating SDP Answer");
   time_t l_time;
   time(&l_time);
   char l_cTime[100];
   memset(l_cTime,0,sizeof(l_cTime));
   sprintf(l_cTime,"%d %d",(int)l_time,(int)l_time);

#if 0
   m_LocalSDP.connection()->setNetworkType("IN");
   m_LocalSDP.connection()->setAddressType("IP4");
   m_LocalSDP.connection()->setAddress(m_CCipmLocalMedia.MediaData[m_unLocalCCIPMAudioRtpPortIndex].mediaInfo.PortInfo.cIPAddress);
#endif

   // SKS next line fixed the bug where time description was being added one line for every call in SDP
   m_LocalSDP.clear();

   m_LocalSDP.version()->setVersion("0");
   m_LocalSDP.origin()->setUserName("IPMediaServer");
   m_LocalSDP.origin()->setNetworkType("IN");
   m_LocalSDP.origin()->setAddressType("IP4");
   m_LocalSDP.origin()->setAddress(m_CCipmLocalMedia.MediaData[m_unLocalCCIPMAudioRtpPortIndex].mediaInfo.PortInfo.cIPAddress);
   m_LocalSDP.origin()->setSessionId(l_cTime);
   m_LocalSDP.sessionName()->setName("Video Demonstration");

   sdpTimeDescription* l_pLocalTimeDescription = m_LocalSDP.timeDescriptionList()->addItem();
	
   l_pLocalTimeDescription->time()->setStart(0);
   l_pLocalTimeDescription->time()->setStop(0);

   sdpMediaDescriptionList* pMDList = m_LocalSDP.mediaDescriptionList();
   pMDList->clear();

   sdpMediaDescription* pAudioMD = pMDList->addItem();
   sdpMedia* pAudioMedia = pAudioMD->media();

   pAudioMD->connection()->setNetworkType("IN");
   pAudioMD->connection()->setAddressType("IP4");
   pAudioMD->connection()->setAddress(m_CCipmLocalMedia.MediaData[m_unLocalCCIPMAudioRtpPortIndex].mediaInfo.PortInfo.cIPAddress);
   pAudioMedia->setMedia("audio");

   mmReport(INFO_DEBUG, s_eType, "\t\tSetting SDP media port to %ld",m_CCipmLocalMedia.MediaData[m_unLocalCCIPMAudioRtpPortIndex].mediaInfo.PortInfo.unPortId);
   pAudioMedia->setPort(m_CCipmLocalMedia.MediaData[m_unLocalCCIPMAudioRtpPortIndex].mediaInfo.PortInfo.unPortId);
   pAudioMedia->setTransport("RTP/AVP");

   sprintf(temp, "%d",m_stRemoteEPAudioCodec.m_nAudioCodec);
   pAudioMedia->addFormat(temp);
   if ( m_stRemoteEPAudioCodec.m_nDTMFCodec != -1)  {
      sprintf(temp, "%d",m_stRemoteEPAudioCodec.m_nDTMFCodec);
      pAudioMedia->addFormat(temp);
   }

   // add audio atrributes
   /* obtain a pointer to the Attribute list for this MD */
   sdpAttributeList *pAudioAttrList  = pAudioMD->attributeList();
   /* add an attribute to the list */
   sdpAttribute* pAttribute1 = pAudioAttrList->addItem();
   /* set property and value of attribute */
   pAttribute1 ->setProperty("rtpmap");
   int SdpSampleRate = 8000;
   if( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amrwb")==0 || strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-wb")==0 ) {
      SdpSampleRate = 16000;
   }
   sprintf(temp, "%d %s/%d",m_stRemoteEPAudioCodec.m_nAudioCodec,m_stRemoteEPAudioCodec.m_cCodecName,SdpSampleRate);
   pAttribute1 ->setPropertyValue(temp);

   //AMRNB	
   if( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr")==0 || strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-nb")==0 ) {
      sdpAttribute* pAttributeOctetAlign = pAudioAttrList->addItem();
      pAttributeOctetAlign->setProperty("fmtp");
      //sprintf(temp, "%d mode-set=0,1,2,3,4,5,6,7; octet-align=%d", m_stRemoteEPAudioCodec.m_nAudioCodec,((m_stRemoteEPAudioCodec.m_bAmrOctetAlign==true)?1:0));
      sprintf(temp, "%d mode-set=7; octet-align=%d", m_stRemoteEPAudioCodec.m_nAudioCodec,((m_stRemoteEPAudioCodec.m_bAmrOctetAlign==true)?1:0));
      pAttributeOctetAlign->setPropertyValue(temp);
   }
   //AMRWB
   if( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amrwb")==0 || strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-wb")==0 ) {
      sdpAttribute* pAttributeOctetAlign = pAudioAttrList->addItem();
      pAttributeOctetAlign->setProperty("fmtp");
      //sprintf(temp, "%d mode-set=0,1,2,3,4,5,6,7,8; octet-align=%d", m_stRemoteEPAudioCodec.m_nAudioCodec,((m_stRemoteEPAudioCodec.m_bAmrOctetAlign==true)?1:0));
      sprintf(temp, "%d mode-set=%d; octet-align=%d; robust-sorting=0", m_stRemoteEPAudioCodec.m_nAudioCodec, m_stRemoteEPAudioCodec.m_nAmrModeSet, ((m_stRemoteEPAudioCodec.m_bAmrOctetAlign==true)?1:0));
      pAttributeOctetAlign->setPropertyValue(temp);
   }
   //G723
   if(m_stRemoteEPAudioCodec.m_nAudioCodec==G723) {
      sdpAttribute* pAttributeOctetAlign = pAudioAttrList->addItem();
      pAttributeOctetAlign->setProperty("fmtp");
      sprintf(temp,"%d annexa=%s; bitrate=%s",m_stRemoteEPAudioCodec.m_nAudioCodec,(m_stRemoteEPAudioCodec.m_bVadEnabled==true?"yes":"no"),(m_stRemoteEPAudioCodec.m_nCoderType==CODER_TYPE_G7231_6_3K?"6.3":"5.3"));
      pAttributeOctetAlign->setPropertyValue(temp);
   }
   //G729
   if(m_stRemoteEPAudioCodec.m_nAudioCodec==G729) {
      sdpAttribute* pAttributeOctetAlign = pAudioAttrList->addItem();
      pAttributeOctetAlign->setProperty("fmtp");
      sprintf(temp, "%d annexb=%s", m_stRemoteEPAudioCodec.m_nAudioCodec,((m_stRemoteEPAudioCodec.m_bVadEnabled==true)?"yes":"no"));
      pAttributeOctetAlign->setPropertyValue(temp);
   }
 
   // SKS JACK to force Kapanga to send AMR MFPP
   // sprintf(temp, "%d", m_stRemoteEPAudioCodec.m_nMaxFrameSize);
   sprintf(temp, "%d", m_stRemoteEPAudioCodec.m_nMinFrameSize);
   sdpAttribute* pAttributePTime = pAudioAttrList->addItem();
   pAttributePTime->setProperty("ptime");
   pAttributePTime->setPropertyValue(temp);

   sprintf(temp, "%d", m_stRemoteEPAudioCodec.m_nMaxFrameSize);
   sdpAttribute* pAttributePMaxtime = pAudioAttrList->addItem();
   pAttributePMaxtime->setProperty("maxptime");
   pAttributePMaxtime->setPropertyValue(temp);

   if ( m_stRemoteEPAudioCodec.m_nDTMFCodec != -1)  {
      /* add another attribute to the list */
      sdpAttribute* pAttribute2 = pAudioAttrList->addItem();
      /* set property and value of attribute */
      pAttribute2 ->setProperty("rtpmap");
      sprintf(temp, "%d telephone-event/8000", m_stRemoteEPAudioCodec.m_nDTMFCodec);
      pAttribute2 ->setPropertyValue(temp);
   }

  if(m_bVideoCall){
   sdpMediaDescription* pVideoMD = pMDList->addItem();
   sdpMedia* pVideoMedia = pVideoMD->media();

   pVideoMD->connection()->setNetworkType("IN");
   pVideoMD->connection()->setAddressType("IP4");
   pVideoMD->connection()->setAddress(m_CCipmLocalMedia.MediaData[m_unLocalCCIPMVideoRtpPortIndex].mediaInfo.PortInfo.cIPAddress);
   pVideoMedia->setMedia("video");
   pVideoMedia->setPort(m_CCipmLocalMedia.MediaData[m_unLocalCCIPMVideoRtpPortIndex].mediaInfo.PortInfo.unPortId);

   pVideoMedia->setTransport("RTP/AVP");
   sprintf(temp, "%d", m_stRemoteEPVideoCodec.m_nVideoCodec);
   pVideoMedia->addFormat(temp);
   /* obtain a pointer to the Attribute list for this MD */
   sdpAttributeList *pVideoAttrList  = pVideoMD->attributeList();

   /* add an attribute to the list */
   sdpAttribute* pAttribute3 = pVideoAttrList->addItem();
   /* set property and value of attribute */
   sprintf(temp, "%d %s/90000",m_stRemoteEPVideoCodec.m_nVideoCodec,m_stRemoteEPVideoCodec.m_cCodecName);
   pAttribute3 ->setProperty("rtpmap");
   pAttribute3 ->setPropertyValue(temp);

   /* add an attribute to the list */
   sdpAttribute* pAttribute4 = pVideoAttrList->addItem();
   /* set property and value of attribute */
   pAttribute4 ->setProperty("fmtp");

   if (m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_H264) {
      int profileLevelId;

      if (g_h264SdpProfileLevel == -1)
      {
         // If not forced by h264sdpprofilelevel config parameter, we answer with a symetric profile and level 
         // profileLevelId = m_stRemoteEPVideoCodec.m_nProfileLevelID;
         // Base Profile Level on Outbound level
         int H264Level=H264_LEVEL(m_stRemoteEPVideoCodec.m_nProfileLevelID);
         // HMP max is Level 1_3
         if (H264Level > VIDEO_LEVEL_1_3_H264) 
            H264Level = VIDEO_LEVEL_1_3_H264;
         switch (H264Level)
         {
            case VIDEO_LEVEL_1_3_H264: profileLevelId = 0x42000d; break;
            case VIDEO_LEVEL_1_2_H264: profileLevelId = 0x42000c; break;
            case VIDEO_LEVEL_1_1_H264: 
               if(H264_B_CONSTRAINT(m_stRemoteEPVideoCodec.m_nProfileLevelID))
                  profileLevelId = 0x42100b; //VIDEO_LEVEL_1_B_H264
               else
                  profileLevelId = 0x42000b; //VIDEO_LEVEL_1_1_H264
               break;
            default: profileLevelId = 0x42000a; break;
         }  
      }
      else
         profileLevelId = g_h264SdpProfileLevel;

      if(m_SdpDciLen>0) {
         unsigned char dciHex[B64_MAX_SIZE]={0};
         char sprop[B64_MAX_SIZE]={0};  			
         int dciLen,i, spropLen;

         dciLen=strlen(m_SdpDciStrValue)/2;
			
         for (i = 0; i<dciLen; i++) {
            dciHex[i] = AsciiOctetToHex(&m_SdpDciStrValue[2*i]);
         }

         encodeSprop( dciHex, dciLen, sprop, &spropLen );

         sprintf(temp, "%d profile-level-id=%x; packetization-mode=%d; sprop-parameter-sets=%s", m_stRemoteEPVideoCodec.m_nVideoCodec, profileLevelId, g_h264SdpPacketizationMode,sprop);
      }
      else {   
         sprintf(temp, "%d profile-level-id=%x; packetization-mode=%d", m_stRemoteEPVideoCodec.m_nVideoCodec, profileLevelId, g_h264SdpPacketizationMode);
      }
      pAttribute4 ->setPropertyValue(temp); 
   }
   
   // add video attributes
   else if (m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_MP4V_ES) {
      //printf("sdp=%s|sdplen=%d\n",m_SdpDciStrValue,m_SdpDciLen);
      if(m_SdpDciLen>0) {
         sprintf(temp, "%d profile-level-id=%d; config=%s", m_stRemoteEPVideoCodec.m_nVideoCodec,
                       m_stRemoteEPVideoCodec.m_nProfileLevelID,m_SdpDciStrValue);
      }
      else {   
         sprintf(temp, "%d profile-level-id=%d", m_stRemoteEPVideoCodec.m_nVideoCodec,m_stRemoteEPVideoCodec.m_nProfileLevelID);
      }
      pAttribute4 ->setPropertyValue(temp); 
   } else {
      sprintf(temp,"%d %s=%d", m_stRemoteEPVideoCodec.m_nVideoCodec,
                   g_cResolutionStr[m_stRemoteEPVideoCodec.m_nResolution],
                   ( (g_nVideoTranscodeDir==TX || g_nVideoTranscodeDir==RXTX)?FPSToMPI(m_stRemoteEPVideoCodec.m_nFrameRate):(30/g_maxFrameRate) ));
      pAttribute4 ->setPropertyValue(temp);
   }
   /* add an attribute to the list */
   sdpAttribute* pAttribute5 = pVideoAttrList->addItem();
   /* set property and value of attribute */
   pAttribute5 ->setProperty("sendrecv");
   char maxBandwidth[5];
   sprintf(maxBandwidth,"%d",((g_nVideoTranscodeDir==TX || g_nVideoTranscodeDir==RXTX)?(m_stRemoteEPVideoCodec.m_nBitRate/1000):g_maxBandwidth) );
   
   pVideoMD->bandwidth()->setModifier("AS");
   pVideoMD->bandwidth()->setBandwidthValue(maxBandwidth);
  }//m_bVideoCall
   mmReport(INFO_DEBUG, s_eType, "\t\tdone with sdp");
   return(&m_LocalSDP);
}
//*****************************************************************************
//      NAME : void CMMStream::ParseAndStoreAudioAttributes(sdpMediaDescription *pMD)
//DESCRIPTION : Parse and Store the Incoming Audio information
//     INPUT : pointer to sdpMediaDescription object- containing Remote Media Information
//      OUTPUT : None
//     RETURNS : None
//    CAUTIONS : None
//*****************************************************************************
bool CMMStream::ParseAndStoreAudioAttributes(sdpMediaDescription *pMD)
{
   sdpMedia* pMedia = NULL;
   int 			audioAttribCtr;
   int 			numOfAttributes;
   int 			numOfFormat;
   int 			pt;
   char 			codec_name[20];
   int 			mediaFormatCtr;
   int 			nPayloadType;
   const char 			*tempStr=NULL;
   bool			bAmrFound=false;

   pMedia = pMD->media();
   sdpAttributeList* pAttributeList = pMD->attributeList();

   numOfAttributes = pAttributeList->numItem();
   numOfFormat = pMedia->getNumFormat();
   m_nAudioRtpPort = pMD->media()->getPort();

   for ( mediaFormatCtr=0;mediaFormatCtr<numOfFormat;mediaFormatCtr++ ) {
      nPayloadType=atoi(pMedia->getFormat(mediaFormatCtr));
      //printf("nPayloadType=%d||g_nAudioCodec=%d\n",nPayloadType,g_nAudioCodec);
      //G711U
      if( (nPayloadType==G711U) && (g_nAudioTranscodeDir!=NATIVE || m_stRemoteEPAudioCodec.m_nAudioCodec<nPayloadType) ) {
         m_stRemoteEPAudioCodec.m_nAudioCodec=nPayloadType;
         strcpy(m_stRemoteEPAudioCodec.m_cCodecName,"PCMU");
         m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G711ULAW64K;
         if(g_nAudioTranscodeDir!=NATIVE && g_nAudioCodec==G711U) {
            break;
         }
      //G711A
      }else if( (nPayloadType==G711A) && (g_nAudioTranscodeDir!=NATIVE || m_stRemoteEPAudioCodec.m_nAudioCodec<nPayloadType)){
         if(g_nAudioTranscodeDir==NATIVE && m_stRemoteEPAudioCodec.m_nAudioCodec==G723) {
            continue;
         }
         m_stRemoteEPAudioCodec.m_nAudioCodec=nPayloadType;
         strcpy(m_stRemoteEPAudioCodec.m_cCodecName,"PCMA");
         m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G711ALAW64K;
         if(g_nAudioTranscodeDir!=NATIVE && g_nAudioCodec==G711A) {
            break;
         }
      //G723
      }else if( (nPayloadType==G723) && (m_stRemoteEPAudioCodec.m_nAudioCodec<nPayloadType || m_stRemoteEPAudioCodec.m_nAudioCodec==G711A || g_nAudioTranscodeDir!=NATIVE)){
         m_stRemoteEPAudioCodec.m_nAudioCodec=nPayloadType;
         strcpy(m_stRemoteEPAudioCodec.m_cCodecName,"G723");
         m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G7231_5_3K;
         m_stRemoteEPAudioCodec.m_nMaxFrameSize=m_stRemoteEPAudioCodec.m_nMinFrameSize=CODER_FRAMESIZE_30;
         if(g_nAudioTranscodeDir!=NATIVE && g_nAudioCodec==G723) {
            break;
         }
      //G729
      }else if( (nPayloadType==G729)&& (m_stRemoteEPAudioCodec.m_nAudioCodec<nPayloadType || g_nAudioTranscodeDir!=NATIVE)) {
         m_stRemoteEPAudioCodec.m_nAudioCodec=nPayloadType;
         strcpy(m_stRemoteEPAudioCodec.m_cCodecName,"G729");
         m_stRemoteEPAudioCodec.m_bVadEnabled=false;
         //m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G729ANNEXA;
         m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G729;
         if(g_nAudioTranscodeDir!=NATIVE && g_nAudioCodec==G729) {
            break;
         }
      //G722
      }else if( (nPayloadType==G722)&& (m_stRemoteEPAudioCodec.m_nAudioCodec<nPayloadType || g_nAudioTranscodeDir!=NATIVE)) {
         m_stRemoteEPAudioCodec.m_nAudioCodec=nPayloadType;
         strcpy(m_stRemoteEPAudioCodec.m_cCodecName,"G722");
         m_stRemoteEPAudioCodec.m_bVadEnabled=false;
         m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G722_64K;
         //m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G722_64K_NATIVE;
         if(g_nAudioTranscodeDir!=NATIVE && g_nAudioCodec==G722) {
            break;
         }
      //DynamicPT
      }else if( nPayloadType>=96 && nPayloadType<=127) {
         for ( audioAttribCtr=0;audioAttribCtr<numOfAttributes;audioAttribCtr++ ) {
            sdpAttribute* pAttribute = pAttributeList->getItem(audioAttribCtr);
            pt=-1;
            tempStr=NULL;
            codec_name[0]='\0';
            tempStr=strstr(pAttribute->getValue(),"rtpmap:");
            if(tempStr!=NULL) {
               sscanf(tempStr,"rtpmap:%d %[0-9A-Za-z-]", &pt, codec_name);
            }
            if ( pt!=nPayloadType) { 
               continue;
            }
            //AMRNB
            if( (strcasecmp(codec_name,"amr")==0) || (strcasecmp(codec_name,"amr-nb")==0) ) {
               m_stRemoteEPAudioCodec.m_nAudioCodec=nPayloadType;
               strcpy(m_stRemoteEPAudioCodec.m_cCodecName,"AMR");
               m_stRemoteEPAudioCodec.m_bVadEnabled=false;
               m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_AMRNB_12_2K;
               m_stRemoteEPAudioCodec.m_nMinFrameSize=CODER_FRAMESIZE_20;   // 1 FPP
               m_stRemoteEPAudioCodec.m_nMaxFrameSize=CODER_FRAMESIZE_20;  // default 1 FPP
               //m_stRemoteEPAudioCodec.m_nMaxFrameSize=CODER_FRAMESIZE_20*AMR_MAX_FPP;		// 10 FPP
               if(g_nAudioTranscodeDir==NATIVE ) {
                  bAmrFound=true;
                  break;
               }
               if(g_nAudioTranscodeDir!=NATIVE && g_nAudioCodec==AMR ) {
                  bAmrFound=true;
                  break;
               }
            }
            //AMRWB
            else if( (strcasecmp(codec_name,"amrwb")==0) || (strcasecmp(codec_name,"amr-wb")==0) ) {
               m_stRemoteEPAudioCodec.m_nAudioCodec=nPayloadType;
               strcpy(m_stRemoteEPAudioCodec.m_cCodecName,"amr-wb");
               m_stRemoteEPAudioCodec.m_bVadEnabled=false;
               m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_AMRWB_23_85K;
               //m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_AMRWB_23_85K_NATIVE;
               m_stRemoteEPAudioCodec.m_nMinFrameSize=CODER_FRAMESIZE_20;               // 1 FPP
               m_stRemoteEPAudioCodec.m_nMaxFrameSize=CODER_FRAMESIZE_20;		// default 1 FPP
               //m_stRemoteEPAudioCodec.m_nMaxFrameSize=CODER_FRAMESIZE_20*AMR_MAX_FPP;	// 10 FPP
               if(g_nAudioTranscodeDir==NATIVE ) {
                  bAmrFound=true;
                  break;
               }
               if(g_nAudioTranscodeDir!=NATIVE && g_nAudioCodec==AMRWB ) {
                  bAmrFound=true;
                  break;
               }
            }
         }   // for audio attributes
      } // Dyanmic Audio Payload Type 
      if(bAmrFound==true) {
         break;
      }
      // make sure matching codec will be selected if transcoding is on  for one direction
      if((g_nAudioTranscodeDir==TX  || g_nAudioTranscodeDir==RX ) &&
        (( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr")==0 && g_nAudioCodec==AMR ) ||
         ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"g723")==0 && g_nAudioCodec==G723) ||
         ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"g729")==0 && g_nAudioCodec==G729) ||
         ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"pcmu")==0 && g_nAudioCodec==G711U) ||
         ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"pcma")==0 && g_nAudioCodec==G711A) ||
         ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amrwb")==0 && g_nAudioCodec==AMRWB) ||
         ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-wb")==0 && g_nAudioCodec==AMRWB) ||
         ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"g722")==0 && g_nAudioCodec==G722)
        ) 
        ){
         //printf("mediaFormatCtr=%d|numOfFormat=%d|g_nAudioTranscodeDir=%d|m_stRemoteEPAudioCodec.m_cCodecName=%s|g_nAudioCodec=%d\n", mediaFormatCtr,numOfFormat,g_nAudioTranscodeDir,m_stRemoteEPAudioCodec.m_cCodecName,g_nAudioCodec);
         break;
      }
   }  // for loop for each audio codec offered

   // get the Payload type of the telephone-event
   for( audioAttribCtr=0;audioAttribCtr<numOfAttributes;audioAttribCtr++ ) {
      sdpAttribute* pAttribute = pAttributeList->getItem(audioAttribCtr);
      pt=-1;
      tempStr=NULL;
      codec_name[0]='\0';
      tempStr=strstr(pAttribute->getValue(),"rtpmap:");
      if(tempStr!=NULL) {
         sscanf(tempStr,"rtpmap:%d %[0-9A-Za-z-]", &pt, codec_name);
      }
      if( (strcasecmp(codec_name,"telephone-event")==0) ) {
         m_stRemoteEPAudioCodec.m_nDTMFCodec=pt;
         m_nRfc2833PayloadType = m_stRemoteEPAudioCodec.m_nDTMFCodec;
         break;
      }
   }

   // now validate selected audio codec for one way transcoding 
   if( (g_nAudioTranscodeDir==TX  || g_nAudioTranscodeDir==RX ) && 
      ( ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr")==0 && g_nAudioCodec!=AMR ) ||
          ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"g723")==0 && g_nAudioCodec!=G723) ||
          ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"g729")==0 && g_nAudioCodec!=G729) ||
          ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"pcmu")==0 && g_nAudioCodec!=G711U) ||
          ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"pcma")==0 && g_nAudioCodec!=G711A) ||
          ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amrwb")==0 && g_nAudioCodec!=AMRWB) ||
          ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-wb")==0 && g_nAudioCodec!=AMRWB) ||
          ( strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"g722")==0 && g_nAudioCodec!=G722)
      ) 
     ){
      mmReport(INFO_DEBUG, s_eType,"Codec has to match to enable one way transcoding %s \n dropping call!!!!!!!!!!",m_stRemoteEPAudioCodec.m_cCodecName);
      return false;
   }

   if(g_nAudioFileFormat==MMD_3GP) {
      if( g_nAudioTranscodeDir!=RXTX && !((strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr")==0) || (strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amrwb")==0) || (strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-wb")==0)) ) {
         mmReport(INFO_ERROR, s_eType,"ERROR: XCODE=OFF -> Remote EP should support AMR audio codec to play from 3GP file \n Dropping call!!!!!");
         return false;
      }
   }

   // get the fmtp parameters of audio codecs
   // get the index of the attributes associated with selected audio
   numOfAttributes = pAttributeList->numItem();
   for (audioAttribCtr=0; audioAttribCtr<numOfAttributes;audioAttribCtr++) {
      sdpAttribute* pAttribute = pAttributeList->getItem(audioAttribCtr);
      tempStr=strstr(pAttribute->getValue(),"rtpmap:");
      if(tempStr!=NULL) {
         sscanf(tempStr,"rtpmap:%d",&pt);
         if(m_stRemoteEPAudioCodec.m_nAudioCodec==pt) {
            break;
         }
      }else {
         tempStr=strstr(pAttribute->getValue(),"fmtp:");
         if(tempStr!=NULL) {
            sscanf(tempStr,"fmtp:%d",&pt);
            if(m_stRemoteEPAudioCodec.m_nAudioCodec==pt) {
               break;
            }
         }
      }
   }

   for (;audioAttribCtr<numOfAttributes;audioAttribCtr++) {
      sdpAttribute* pAttribute = pAttributeList->getItem(audioAttribCtr);
      tempStr=strstr(pAttribute->getValue(),"rtpmap:");
      if(tempStr!=NULL) {
         sscanf(tempStr,"rtpmap:%d",&pt);
         if(m_stRemoteEPAudioCodec.m_nAudioCodec!=pt) {
            break;					// parsing completed for selected audio break the attributes loop
         }
      }else {
         tempStr=strstr(pAttribute->getValue(),"fmtp:");
         if(tempStr!=NULL) {
            sscanf(tempStr,"fmtp:%d",&pt);
            if(m_stRemoteEPAudioCodec.m_nAudioCodec!=pt) {
               break;				// parsing completed for selected audio break the attributes loop
            }
         }
      }
      tempStr=strstr(pAttribute->getValue(),"fmtp:");
      if(tempStr!=NULL) {
         if(m_stRemoteEPAudioCodec.m_nAudioCodec==G723) {
            // G723.1 annexa=yes/no
            char cG723BitRate[5];
            char cG723Annex[5];
            tempStr=strstr(pAttribute->getValue(),"annexa=");
            if(tempStr!=NULL){
               sscanf(tempStr,"annexa=%[a-zA-Z]",cG723Annex);
               if(strncmp(cG723Annex,"yes",3)==0) {
                  m_stRemoteEPAudioCodec.m_bVadEnabled=true;
               } else {
                  m_stRemoteEPAudioCodec.m_bVadEnabled=false;
               }
            }
            // G723.1 bitrate
            tempStr=strstr(pAttribute->getValue(),"bitrate=");
            if(tempStr!=NULL){
               sscanf(tempStr,"bitrate=%[.0-9]",cG723BitRate);
               if(strcmp(cG723BitRate,"6.3")==0) {
                  m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G7231_6_3K;
               } else {
                  m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G7231_5_3K;
               }
            } else {
                m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G7231_5_3K;
            }
         } // G723 
         else if(m_stRemoteEPAudioCodec.m_nAudioCodec==G729) {
            // Check annexb=yes/no if G729
            char cG729Annex[5];
            tempStr=strstr(pAttribute->getValue(),"annexb=");
            if(tempStr!=NULL){
               sscanf(tempStr,"annexb=%s",cG729Annex);
	       if(strncmp(cG729Annex,"yes",3)==0) {
                  m_stRemoteEPAudioCodec.m_bVadEnabled=true;
                  m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G729ANNEXAWANNEXB;
               } else {
                  m_stRemoteEPAudioCodec.m_bVadEnabled=false;
                  m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_G729ANNEXA;
               }
            }
         } // G729
         else {
            // AMR octet-align=0-1
            int value=-1;
            tempStr=strstr(pAttribute->getValue(),"octet-align=");
            if(tempStr!=NULL){
               sscanf(tempStr,"octet-align=%d",&value);
               m_stRemoteEPAudioCodec.m_bAmrOctetAlign=((value==1)?true:false);
               if(value==1) {
                  m_stRemoteEPAudioCodec.m_nAmrOption=CODER_OPT_AMR_OCTET|CODER_OPT_AMR_CMR_TRACK;
               } else {
                  m_stRemoteEPAudioCodec.m_nAmrOption=CODER_OPT_AMR_EFFICIENT|CODER_OPT_AMR_CMR_TRACK;
               }
            }
            // AMR mode-set=0-7/08 
            value=-1;
            tempStr=strstr(pAttribute->getValue(),"mode-set=");
            if(tempStr!=NULL) {
               sscanf(tempStr,"mode-set=%d",&value);
               if(value >= 0 && value <= 8) {
                  m_stRemoteEPAudioCodec.m_nAmrModeSet=value;
                  if(m_stRemoteEPAudioCodec.m_nAudioCodec==AMRWB)
                     m_stRemoteEPAudioCodec.m_nCoderType=GetAmrwbCoderType(m_stRemoteEPAudioCodec.m_nAmrModeSet);
               } else {
                  if(m_stRemoteEPAudioCodec.m_nAudioCodec==AMRWB) {
                     m_stRemoteEPAudioCodec.m_nAmrModeSet=8;
                     m_stRemoteEPAudioCodec.m_nCoderType=CODER_TYPE_AMRWB_23_85K;
                  } else {
                     m_stRemoteEPAudioCodec.m_nAmrModeSet=7;
                  }
               }
            }
         }  // AMR (AMRWB & AMRNB)
      }
      tempStr=NULL;
      tempStr=strstr(pAttribute->getValue(),"maxptime:");
      if(tempStr!=NULL) {
         sscanf(tempStr,"maxptime:%d",&m_stRemoteEPAudioCodec.m_nMaxFrameSize);
      } else {
         tempStr=NULL;
         tempStr=strstr(pAttribute->getValue(),"ptime:");
         if(tempStr!=NULL) {
            sscanf(tempStr,"ptime:%d",&m_stRemoteEPAudioCodec.m_nMinFrameSize);
         }
      }
   } // end of attributes for loop for offered audio codec if selected
   
   switch(m_stRemoteEPAudioCodec.m_nAudioCodec) 
   {
      case G711U:
      case G711A:
      case G722:
      {
         if(m_stRemoteEPAudioCodec.m_nMaxFrameSize%30==0) {
            m_stRemoteEPAudioCodec.m_nAudioFrameSize=CODER_FRAMESIZE_30;
         } else if(m_stRemoteEPAudioCodec.m_nMaxFrameSize%20==0) {
            m_stRemoteEPAudioCodec.m_nAudioFrameSize=CODER_FRAMESIZE_20;
         } else if(m_stRemoteEPAudioCodec.m_nMaxFrameSize%10==0){
            m_stRemoteEPAudioCodec.m_nAudioFrameSize=CODER_FRAMESIZE_10;
         } else if(m_stRemoteEPAudioCodec.m_nMaxFrameSize%40==0){
           mmReport(INFO_ERROR, s_eType,"HMP supports FRAMESIZE up to %dms",CODER_FRAMESIZE_30); 
         }
         m_stRemoteEPAudioCodec.m_nFPP=m_stRemoteEPAudioCodec.m_nMaxFrameSize/m_stRemoteEPAudioCodec.m_nAudioFrameSize;
         break;
      }
      case G729:
         m_stRemoteEPAudioCodec.m_nAudioFrameSize=CODER_FRAMESIZE_10;
         m_stRemoteEPAudioCodec.m_nFPP=m_stRemoteEPAudioCodec.m_nMaxFrameSize/m_stRemoteEPAudioCodec.m_nAudioFrameSize;
         break;
      case G723:
         m_stRemoteEPAudioCodec.m_nAudioFrameSize=CODER_FRAMESIZE_30;
         m_stRemoteEPAudioCodec.m_nFPP=m_stRemoteEPAudioCodec.m_nMaxFrameSize/m_stRemoteEPAudioCodec.m_nAudioFrameSize;
         break;
      default:
      {
         m_stRemoteEPAudioCodec.m_nMinFrameSize=CODER_FRAMESIZE_20;
         m_stRemoteEPAudioCodec.m_nAudioFrameSize=CODER_FRAMESIZE_20;
         if(m_stRemoteEPAudioCodec.m_nMaxFrameSize>(CODER_FRAMESIZE_20*AMR_MAX_FPP)) {
            mmReport(INFO_ERROR, s_eType,"HMP v4.1 supports up to %d frames per packet",AMR_MAX_FPP);
            m_stRemoteEPAudioCodec.m_nMaxFrameSize=CODER_FRAMESIZE_20*AMR_MAX_FPP;
         }
         m_stRemoteEPAudioCodec.m_nFPP=m_stRemoteEPAudioCodec.m_nMaxFrameSize/m_stRemoteEPAudioCodec.m_nAudioFrameSize;
         mmReport(INFO_DEBUG, s_eType,"m_stRemoteEPAudioCodec.m_nMaxFrameSize=%d|m_stRemoteEPAudioCodec.m_nMinFrameSize=%d|m_stRemoteEPAudioCodec.m_nFPP=%d",
                                       m_stRemoteEPAudioCodec.m_nMaxFrameSize,m_stRemoteEPAudioCodec.m_nMinFrameSize,m_stRemoteEPAudioCodec.m_nFPP);
         break;
      }
   }
   if(m_stRemoteEPAudioCodec.m_nAudioCodec<G711U) {
      return false;
   } else {
      mmReport(INFO_DEBUG, s_eType,"Selected Remote Audio Codec %s(%d)", m_stRemoteEPAudioCodec.m_cCodecName,m_stRemoteEPAudioCodec.m_nAudioCodec);
      return true;
   }
}

//*****************************************************************************
//      NAME : void CMMStream::ParseAndStoreVideoAttributes(sdpMediaDescription *pMD)
//DESCRIPTION : Parse and Store the Incoming Audio information
//     INPUT : pointer to sdpMediaDescription object- containing Remote Media Information
//      OUTPUT : None
//     RETURNS : None
//    CAUTIONS : None
//*****************************************************************************
bool CMMStream::ParseAndStoreVideoAttributes(sdpMediaDescription *pMD)
{
   sdpMedia* pMedia = NULL;
   int       videoAttribCtr;
   int       numOfAttributes;
   int       numOfFormat;
   int       pt;
   char      codec_name[20];
   int       mediaFormatCtr;
   int       nPayloadType;
   int       profileLevelID=0;
   const char*     tempStr=NULL;
   int       fps=0;
   bool      bMpeg4Found=false;
   int       h264_level = 0;

   pMedia = pMD->media();
   sdpAttributeList* pAttributeList = pMD->attributeList();		// SKS Need Attention

   numOfAttributes = pAttributeList->numItem();
   numOfFormat = pMedia->getNumFormat();
   m_nVideoRtpPort = pMD->media()->getPort();

   // get the bandwidth if available
   sdpBandwidth* l_sessionBandwidth = pMD->bandwidth();
   if (strcmp(l_sessionBandwidth->getModifier(),"AS") == 0)  {
      m_stRemoteEPVideoCodec.m_nBitRate = (eVIDEO_BITRATE)(atoi(l_sessionBandwidth->getBandwidthValue())*1000);
   }

   for ( mediaFormatCtr=0;mediaFormatCtr<numOfFormat;mediaFormatCtr++ ) {
      nPayloadType=atoi(pMedia->getFormat(mediaFormatCtr));
      if( nPayloadType==H263 ) {
         m_stRemoteEPVideoCodec.m_nVideoCodec=nPayloadType;
         strcpy(m_stRemoteEPVideoCodec.m_cCodecName,"H263");
         m_stRemoteEPVideoCodec.m_nCoderType=CODER_TYPE_H263;
         if(g_nVideoTranscodeDir!=NATIVE && g_nVideoCodec==H263) {
            break;
         }
      } else if( nPayloadType>=96 && nPayloadType<=127) {
         for ( videoAttribCtr=0;videoAttribCtr<numOfAttributes;videoAttribCtr++ ) {
            sdpAttribute* pAttribute = pAttributeList->getItem(videoAttribCtr);			// SKS Need Attention
            pt=-1;
            tempStr=NULL;
            codec_name[0]='\0';
            tempStr=strstr(pAttribute->getValue(),"rtpmap:");
            if(tempStr!=NULL) {
                sscanf(tempStr,"rtpmap:%d %[0-9A-Za-z-]", &pt, codec_name);
            }
            if ( pt!=nPayloadType) {
                continue;
            }

            // Set Dynamic Video Codec
            if( strcasecmp(codec_name,"H264")==0 && g_nPlatformNameRelease>HMP_3_1 ) {
              m_stRemoteEPVideoCodec.m_nVideoCodec=nPayloadType;
              strcpy(m_stRemoteEPVideoCodec.m_cCodecName,"H264");
              m_stRemoteEPVideoCodec.m_nCoderType=CODER_TYPE_H264;
            }
            else if( strcasecmp(codec_name,"MP4V-ES")==0 && g_nPlatformNameRelease>HMP_3_1 ) {
               m_stRemoteEPVideoCodec.m_nVideoCodec=nPayloadType;
               strcpy(m_stRemoteEPVideoCodec.m_cCodecName,"MP4V-ES");
               m_stRemoteEPVideoCodec.m_nCoderType=CODER_TYPE_MP4V_ES;
               if(g_nVideoTranscodeDir==NATIVE ) {
                  bMpeg4Found=true;
              	  break;
               }
               if(g_nVideoTranscodeDir!=NATIVE && g_nVideoCodec==MPEG4)  {
                  bMpeg4Found=true;
                  break;
               }
            } 
            else if( strcasecmp(codec_name,"H263-2000")==0 && g_nPlatformNameRelease>HMP_3_1) {
               m_stRemoteEPVideoCodec.m_nVideoCodec=nPayloadType;
               strcpy(m_stRemoteEPVideoCodec.m_cCodecName,"H263-2000");
               m_stRemoteEPVideoCodec.m_nCoderType=CODER_TYPE_H263_1998;
            }
            else if( strcasecmp(codec_name,"H263-1998")==0 && g_nPlatformNameRelease>HMP_3_1) {
              m_stRemoteEPVideoCodec.m_nVideoCodec=nPayloadType;
              strcpy(m_stRemoteEPVideoCodec.m_cCodecName,"H263-1998");
              m_stRemoteEPVideoCodec.m_nCoderType=CODER_TYPE_H263_1998;
            }
            else {
              mmReport(INFO_ERROR, s_eType,"Unsuported Dynamic Codec=%s, pt=%d on this HMP release",codec_name, nPayloadType);
              mmReport(INFO_ERROR, s_eType,"Dropping Call!");
              return false;
            }
            mmReport(INFO_DEBUG1, s_eType,"SDP Codec=%s, pt=%d, codertype=%d",codec_name,nPayloadType,m_stRemoteEPVideoCodec.m_nCoderType);
         
         } // for Video attributes
      } // Dyanmic Video Payload Type
      if(bMpeg4Found==true) {
         break;
      }

      // make sure matching codec will be selected if transcoding is on for one direction
      if( (g_nVideoTranscodeDir==TX  || g_nVideoTranscodeDir==RX ) && (
       ( strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H264")==0 && g_nVideoCodec==H264 ) ||
       ( strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"MP4V-ES")==0 && g_nVideoCodec==MPEG4 ) ||
       ( (strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H263")==0 || strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H263_1998")==0 || strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H263_2000")==0) && (g_nVideoCodec==H263 || g_nVideoCodec==H263_1998 || g_nVideoCodec==H263_2000 ))
       )) {
         //mmReport(INFO_DEBUG1, S_eType, "mediaFormatCtr=%d|numOfFormat=%d|g_nVideoTranscodeDir=%d|m_stRemoteEPVideoCodec.m_cCodecName=%s|g_nVideoCodec=%d\n",mediaFormatCtr,numOfFormat,g_nVideoTranscodeDir,m_stRemoteEPVideoCodec.m_cCodecName,g_nVideoCodec);
          break;
      }
   }	// media format parsing for loop

   if( (g_nVideoTranscodeDir==TX  || g_nVideoTranscodeDir==RX ) && (
       ( strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H264")==0 && g_nVideoCodec!=H264 ) ||
       ( strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"MP4V-ES")==0 && g_nVideoCodec!=MPEG4 ) ||
       ( (strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H263")==0 || strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H263_1998")==0 || strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H263_2000")==0) && (g_nVideoCodec!=H263 || g_nVideoCodec!=H263_1998 || g_nVideoCodec!=H263_2000 ))
     ))  {
      mmReport(INFO_ERROR, s_eType,"ERROR: Codec has to match when one way transcoding %s \n dropping call!!!!!");
      return false;
   }

   // get the fmtp parameters of video codecs
   // get the index of the attributes associated with selected video
   numOfAttributes = pAttributeList->numItem();
   for (videoAttribCtr=0; videoAttribCtr<numOfAttributes;videoAttribCtr++) {
      sdpAttribute* pAttribute = pAttributeList->getItem(videoAttribCtr);
      tempStr=strstr(pAttribute->getValue(),"rtpmap:");
      if(tempStr!=NULL) {
         sscanf(tempStr,"rtpmap:%d",&pt);
         if(m_stRemoteEPVideoCodec.m_nVideoCodec==pt) {
           break;
         }
      }else {
         tempStr=strstr(pAttribute->getValue(),"fmtp:");
         if(tempStr!=NULL) {
            sscanf(tempStr,"fmtp:%d",&pt);
            if(m_stRemoteEPVideoCodec.m_nVideoCodec==pt) {
               break;
            }
         }
      }
   }	// for loop to get exact index

   for (;videoAttribCtr<numOfAttributes;videoAttribCtr++) {
      sdpAttribute* pAttribute = pAttributeList->getItem(videoAttribCtr);
      tempStr=strstr(pAttribute->getValue(),"rtpmap:");
      if(tempStr!=NULL) {
         sscanf(tempStr,"rtpmap:%d",&pt);
         if(m_stRemoteEPVideoCodec.m_nVideoCodec!=pt) {
            break;          // parsing completed for selected video break the attributes loop
         }
      }else {
         tempStr=strstr(pAttribute->getValue(),"fmtp:");
         if(tempStr!=NULL) {
            sscanf(tempStr,"fmtp:%d",&pt);
            if(m_stRemoteEPVideoCodec.m_nVideoCodec!=pt) {
               break;        // parsing completed for selected video break the attributes loop
            }
         }
      }
      tempStr=strstr(pAttribute->getValue(),"fmtp:");
      if (tempStr!=NULL) {
         if(strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"MP4V-ES")==0) {
            //Get profile-level-id and DCI config info
            tempStr=strstr(pAttribute->getValue(),"profile-level-id=");
            if(tempStr!=NULL) {
               sscanf(tempStr,"profile-level-id=%d",&m_stRemoteEPVideoCodec.m_nProfileLevelID);
               if(m_stRemoteEPVideoCodec.m_nProfileLevelID>3) {
                  m_stRemoteEPVideoCodec.m_nProfileLevelID=2;
               }
            }
            fps=2;		// JACK
            tempStr=NULL;
            int i;
            tempStr=strstr(pAttribute->getValue(),"config=");
            if(tempStr!=NULL) {
               sscanf(tempStr,"config=%s",m_stRemoteEPVideoCodec.m_SdpDciStrValue);
               m_stRemoteEPVideoCodec.m_SdpDciLen=(strlen(m_stRemoteEPVideoCodec.m_SdpDciStrValue)/2);
               for (i = 0; i <m_stRemoteEPVideoCodec.m_SdpDciLen; i++) {
                  m_stRemoteEPVideoCodec.m_SdpDciValue[i] = AsciiOctetToHex(&m_stRemoteEPVideoCodec.m_SdpDciStrValue[2*i]);
               }
            }
            break;
         } // end if MPEG4
         else if(strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H264")==0) { 
            //Get profile-level-id and DCI config info
            tempStr=strstr(pAttribute->getValue(),"profile-level-id=");
            if(tempStr!=NULL) {
               sscanf(tempStr,"profile-level-id=%x",&m_stRemoteEPVideoCodec.m_nProfileLevelID);
            }
            fps=2;		// JACK
            tempStr=NULL;

            tempStr=strstr(pAttribute->getValue(),"sprop-parameter-sets=");
            if(tempStr!=NULL) {			
               sscanf(tempStr,"sprop-parameter-sets=%s",m_stRemoteEPVideoCodec.m_SdpDciStrValue);
               decodeSprop( m_stRemoteEPVideoCodec.m_SdpDciStrValue, strlen(m_stRemoteEPVideoCodec.m_SdpDciStrValue), m_stRemoteEPVideoCodec.m_SdpDciValue, &m_stRemoteEPVideoCodec.m_SdpDciLen );
            }
            tempStr=NULL;
            tempStr=strstr(pAttribute->getValue(),"MAXBR=");
            if(tempStr!=NULL) {
               sscanf(tempStr,"MAXBR=%d",&m_stRemoteEPVideoCodec.m_nBitRate);
               m_stRemoteEPVideoCodec.m_nBitRate=(eVIDEO_BITRATE)(m_stRemoteEPVideoCodec.m_nBitRate*1000);
               mmReport(INFO_DEBUG, s_eType,"Offered Bandwidth = %d",m_stRemoteEPVideoCodec.m_nBitRate);
            }
            break;
         } // end if H264
         else {
            tempStr=strstr(pAttribute->getValue(),"MAXBR=");
            if(tempStr!=NULL) {
               sscanf(tempStr,"MAXBR=%d",&m_stRemoteEPVideoCodec.m_nBitRate);
               m_stRemoteEPVideoCodec.m_nBitRate=(eVIDEO_BITRATE)(m_stRemoteEPVideoCodec.m_nBitRate*1000);
               mmReport(INFO_DEBUG, s_eType,"Offered Bandwidth = %d",m_stRemoteEPVideoCodec.m_nBitRate);
            }
            tempStr=strstr(pAttribute->getValue(),"SQCIF=");
            if(tempStr!=NULL) {
               sscanf(tempStr,"SQCIF=%d",&fps);
               m_stRemoteEPVideoCodec.m_nResolution=SQCIF;	
            } else {
               tempStr=strstr(pAttribute->getValue(),"QCIF=");
               if(tempStr!=NULL) {
                  sscanf(tempStr,"QCIF=%d",&fps);
                  //JM: Set CIF if global is set
                  if(g_nVideoRes==CIF){
                     m_stRemoteEPVideoCodec.m_nResolution=CIF;
                  } else
                     m_stRemoteEPVideoCodec.m_nResolution=QCIF;	
               } else {
                  tempStr=strstr(pAttribute->getValue(),"CIF=");
                  if(tempStr!=NULL) {
                     sscanf(tempStr,"CIF=%d",&fps);
                     m_stRemoteEPVideoCodec.m_nResolution=CIF;	 
                  }
               }
            }
            break;
         } // end if H263, H263-1998 or H263-2000
      } // end if fmtp for h263 and mpeg4 and h264 found
   } // end for loop attribute

   // now validate the values
   if(strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"MP4V-ES")==0) {
      if(m_stRemoteEPVideoCodec.m_nProfileLevelID<0) {
         m_stRemoteEPVideoCodec.m_nProfileLevelID=1;
      }
      if(m_stRemoteEPVideoCodec.m_SdpDciLen<=0) {
         if(m_stRemoteEPVideoCodec.m_nProfileLevelID>1 && g_cRxDciCifSize>0) {
            strcpy(m_stRemoteEPVideoCodec.m_SdpDciStrValue,g_cRxDciCifStrValue);
            m_stRemoteEPVideoCodec.m_SdpDciLen=g_cRxDciCifSize;
            m_stRemoteEPVideoCodec.m_SdpDciValue=g_cRxDciCifValue;
         } else if(m_stRemoteEPVideoCodec.m_nProfileLevelID<2 && g_cRxDciQcifSize>0) {
            strcpy(m_stRemoteEPVideoCodec.m_SdpDciStrValue,g_cRxDciQcifStrValue);
            m_stRemoteEPVideoCodec.m_SdpDciLen=g_cRxDciQcifSize;
            m_stRemoteEPVideoCodec.m_SdpDciValue=g_cRxDciQcifValue;
         }
      }
      if(m_stRemoteEPVideoCodec.m_nProfileLevelID>1) {
         m_stRemoteEPVideoCodec.m_nResolution=CIF;
      } else {
         m_stRemoteEPVideoCodec.m_nResolution=QCIF;
      }
   }//end mpeg4 
   else if(strcasecmp(m_stRemoteEPVideoCodec.m_cCodecName,"H264")==0) {
      h264_level=H264_LEVEL(m_stRemoteEPVideoCodec.m_nProfileLevelID);
      mmReport(INFO_DEBUG, s_eType,"Offered H264 Level=%x",h264_level); 
      if(m_stRemoteEPVideoCodec.m_nProfileLevelID<0) {
         m_stRemoteEPVideoCodec.m_nProfileLevelID=0x42000A;
      }
      if(m_stRemoteEPVideoCodec.m_SdpDciLen<=0) {
         if(H264_LEVEL(m_stRemoteEPVideoCodec.m_nProfileLevelID) > VIDEO_LEVEL_1_1_H264 && g_cRxDciCifSize>0) {
            strcpy(m_stRemoteEPVideoCodec.m_SdpDciStrValue,g_cRxDciCifStrValue);
            m_stRemoteEPVideoCodec.m_SdpDciLen=g_cRxDciCifSize;
            m_stRemoteEPVideoCodec.m_SdpDciValue=g_cRxDciCifValue;
          } else if(H264_LEVEL(m_stRemoteEPVideoCodec.m_nProfileLevelID) < VIDEO_LEVEL_1_2_H264 && g_cRxDciQcifSize>0) {
            strcpy(m_stRemoteEPVideoCodec.m_SdpDciStrValue,g_cRxDciQcifStrValue);
            m_stRemoteEPVideoCodec.m_SdpDciLen=g_cRxDciQcifSize;
            m_stRemoteEPVideoCodec.m_SdpDciValue=g_cRxDciQcifValue;
          }
      }
      // assume higher than Level 1_1 is CIF and 384k
      if(h264_level > VIDEO_LEVEL_1_2_H264) { //0x42000d=1_3
         m_stRemoteEPVideoCodec.m_nResolution=CIF;
         if(m_stRemoteEPVideoCodec.m_nBitRate==0)
            m_stRemoteEPVideoCodec.m_nBitRate=(eVIDEO_BITRATE)(VIDEO_BITRATE_384K); //Can be as high as 768k, limit to 384 for now
      }else if(h264_level == VIDEO_LEVEL_1_2_H264) { //0x42000c=1_2
         m_stRemoteEPVideoCodec.m_nResolution=CIF;
         if(m_stRemoteEPVideoCodec.m_nBitRate==0)
            m_stRemoteEPVideoCodec.m_nBitRate=(eVIDEO_BITRATE)(VIDEO_BITRATE_384K);
      } else if (h264_level == VIDEO_LEVEL_1_1_H264) {//0x42000b=1_1, 0x42100b=1_b
         m_stRemoteEPVideoCodec.m_nResolution=QCIF;
         if(m_stRemoteEPVideoCodec.m_nBitRate==0 && H264_B_CONSTRAINT(m_stRemoteEPVideoCodec.m_nProfileLevelID))
            m_stRemoteEPVideoCodec.m_nBitRate=(eVIDEO_BITRATE)(VIDEO_BITRATE_128K);
         else
            m_stRemoteEPVideoCodec.m_nBitRate=(eVIDEO_BITRATE)(192000);
      } else { //0x42000a=1_0
         m_stRemoteEPVideoCodec.m_nResolution=QCIF;
         if(m_stRemoteEPVideoCodec.m_nBitRate==0)
            m_stRemoteEPVideoCodec.m_nBitRate=(eVIDEO_BITRATE)(VIDEO_BITRATE_64K);
      }
   } //end h264
   else { // H263 error handling
      if(m_stRemoteEPVideoCodec.m_nResolution<SQCIF) {
         m_stRemoteEPVideoCodec.m_nResolution=QCIF;
      }
      m_stRemoteEPVideoCodec.m_nBitRate=(eVIDEO_BITRATE)(g_maxBandwidth*1000);
   } //end h263

   // Frame rate
   if(fps>0) {
      m_stRemoteEPVideoCodec.m_nFrameRate=(eVIDEO_FRAMESPERSEC)MPIToFPS((30/fps));
      mmReport(INFO_DEBUG, s_eType,"Offered Frame rate = %d | m_stRemoteEPVideoCodec.m_nFrameRate=%d",(30/fps),m_stRemoteEPVideoCodec.m_nFrameRate);
   }
   else {
      m_stRemoteEPVideoCodec.m_nFrameRate=VIDEO_FRAMESPERSEC_30;
      mmReport(INFO_DEBUG, s_eType,"m_stRemoteEPVideoCodec.m_nFrameRate=%d",m_stRemoteEPVideoCodec.m_nFrameRate);
   }
   if(m_stRemoteEPVideoCodec.m_nVideoCodec<H263) {
      return false;
   }
   else  {
   }
   mmReport(INFO_DEBUG, s_eType,"Remote Video Codec %s(%d)",	m_stRemoteEPVideoCodec.m_cCodecName,m_stRemoteEPVideoCodec.m_nVideoCodec);
   return true;
}

//*****************************************************************************
//            NAME : bool CMMStream::ParseOfferSdp(sdpSessionDescription *a_pSDP)
//     DESCRIPTION : Parse the Incoming SDP information
//           INPUT : remoteSdp - tSdpInfo object containing Remote SDP Information
// 	    OUTPUT : None
// 	   RETURNS : Bool - True if Parsing was successful, False otherwise
// 	  CAUTIONS : None
//*****************************************************************************
int CMMStream::ParseOfferSdp(sdpSessionDescription *a_pSDP)
{
   int i = 0;
   // store the SDP
   m_pRemoteSDP = a_pSDP;
   int md_audio_cnt = 0;
   int md_video_cnt = 0;
   
   mmReport(INFO_DEBUG, s_eType, "ParseOfferSdp()");

   // get the IP addr in the c line
   strcpy(m_cAudioVideoIp, m_pRemoteSDP->connection()->getAddress());
   mmReport(INFO_DEBUG, s_eType, "m_cAudioVideoIp = %s", m_cAudioVideoIp);

   int NumMD = m_pRemoteSDP->mediaDescriptionList()->numItem();
   mmReport(INFO_DEBUG, s_eType, "\t\tSDP contains %d media descriptor(s)",NumMD);

   sdpMediaDescription* pMD = NULL;
   sdpMedia* pMedia = NULL;

   for ( i=0;i<NumMD;i++ )
   {
      pMD = m_pRemoteSDP->mediaDescriptionList()->getItem(i);
      
      if(strlen(m_cAudioVideoIp)==0) {
         // get the IP addr in the c line for each media
         strcpy(m_cAudioVideoIp, pMD->connection()->getAddress());
         mmReport(INFO_DEBUG, s_eType, "Connection Information offered at media level, m_cAudioVideoIp = %s", m_cAudioVideoIp);
      }

      pMedia = pMD->media();
      char strMediaType[11];
      sprintf(strMediaType,"%s",pMedia->getMedia());

      if ( strcasecmp(strMediaType,"audio")==0 ) {
         md_audio_cnt++;
         if(ParseAndStoreAudioAttributes(pMD)!=true) {
            return -1;
         }
      } else if( strcasecmp(strMediaType,"video")==0 ) {
         md_video_cnt++;
         if(ParseAndStoreVideoAttributes(pMD)!=true) {
            return -1;
         }
      } else {
         mmReport(INFO_DEBUG, s_eType, "UnIdentified Media Type %s",strMediaType);
      }
   }
   //Set video Codec default to H263 if audio only call
   if(md_video_cnt==0){
      m_bVideoCall = false;
      m_stRemoteEPVideoCodec.m_nCoderType=CODER_TYPE_H263;
   } 
   SetupPlayRecordSettings();
   return 0;
}

//*****************************************************************************
//            NAME : void CMMStream::SetupPlayRecordSettings()
//     DESCRIPTION : Setup Play and Record Codecs, etc
// 	     INPUT : None
// 	    OUTPUT : None
// 	   RETURNS : None
// 	  CAUTIONS : None
//*****************************************************************************
void CMMStream::SetupPlayRecordSettings()
{
// SKS Local EP codec values will be used by mm_Play and Record 
// 		except sdp variables which will be used by ipm resources as well

   INIT_MM_VIDEO_CODEC(&m_mmVideoCodecPlay);
   INIT_MM_VIDEO_CODEC(&m_mmVideoCodecRecord);
   INIT_MM_AUDIO_CODEC(&m_mmAudioCodecPlay);
   INIT_MM_AUDIO_CODEC(&m_mmAudioCodecRecord);

   if( m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_MP4V_ES || g_nVideoCodec==MPEG4)
   {
      SetDciValue();
   }
   else if (m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_H264 || g_nVideoCodec==H264 )
   {
      SetH264DciValue();
   }

   //Set Audio Codec for Play 
   SetPlayRecordAudioCodec();
   //Set Video Codec for Play 
   SetPlayRecordVideoCodec();
   if(m_bVideoCall){
      mmReport(INFO_DEBUG, s_eType, "Setup mm_Play() for video codec=%s(%d)| audio codec=%s(%d)",
                                 GetVideoCodecStr(g_nVideoCodec),g_nVideoCodec, 
                                 GetAudioCodecStr(g_nAudioCodec), g_nAudioCodec);
   } else {
      mmReport(INFO_DEBUG, s_eType, "No Video in SDP. Setup mm_Play() for audio codec=%s(%d) only",
                                 GetAudioCodecStr(g_nAudioCodec), g_nAudioCodec);
   }

}

unsigned int CMMStream::GetUnCoding(int pt)
{
   unsigned int unCoding;
   switch(pt)
   {
      //case G711U: { unCoding = MM_DATA_FORMAT_MULAW; SetPlayRecordFileName(".g711u"); break; }
      case G711U: { unCoding = MM_DATA_FORMAT_MULAW; SetPlayRecordFileName(".aud"); break; }
      case G711A: { unCoding = MM_DATA_FORMAT_ALAW; SetPlayRecordFileName(".g711a"); break; }
      case G723: 
      {
         if(g_nAudioTranscodeDir==TX || g_nAudioTranscodeDir==RXTX) {
            unCoding=MM_DATA_FORMAT_G723_1_6_30K;
            SetPlayRecordFileName(".g7236k");
         } else {
            if(m_stRemoteEPAudioCodec.m_nCoderType==CODER_TYPE_G7231_6_3K) {
               unCoding=MM_DATA_FORMAT_G723_1_6_30K;
               SetPlayRecordFileName(".g7236k");
            } else {
               unCoding=MM_DATA_FORMAT_G723_1_5_30K;
               SetPlayRecordFileName(".g7235k"); 
            }
         }
         break; 
      }
      case G729: { unCoding=MM_DATA_FORMAT_G729A; SetPlayRecordFileName(".g729"); break; }
      case G726: { unCoding=MM_DATA_FORMAT_G726; SetPlayRecordFileName(".g726"); break; }
      case AMR: { unCoding=MM_DATA_FORMAT_AMR_NB_12_20K; SetPlayRecordFileName(".amr"); break; }
      case PCM: { unCoding=MM_DATA_FORMAT_PCM; SetPlayRecordFileName(".pcm"); break; }
      case PCM16K: { unCoding=MM_DATA_FORMAT_PCM; SetPlayRecordFileName(".pcm16k"); break; }
      case AMRWB: { unCoding=MM_DATA_FORMAT_AMR_WB_23_85K; SetPlayRecordFileName(".amrwb"); break; }
      case G722: { unCoding = MM_DATA_FORMAT_G722_64K; SetPlayRecordFileName(".g722"); break; }
      default: break;
   }
   return unCoding;
}

void CMMStream::SetPlayRecordAudioCodec()
{
   // Get Payload type and options for Dynamic Audio Codecs
   int pt;
   //AMR-NB
   if(strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr")==0 || strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-nb")==0) {
       pt=AMR;
       PMM_AUDIO_CODEC_OPTION_LIST audioOption = new(MM_AUDIO_CODEC_OPTION_LIST);
       INIT_MM_AUDIO_CODEC_OPTION_LIST(audioOption);

       m_mmAudioCodecRecord.options=audioOption;

       audioOption->eParm=EMM_AUDIO_CODEC_OPTION_PAYLOAD_FORMAT;
       if(m_stRemoteEPAudioCodec.m_bAmrOctetAlign==true) {
            audioOption->nValue=EMM_AUDIO_PAYLOAD_FORMAT_OCTET_ALIGNED;
       } else {
            audioOption->nValue=EMM_AUDIO_PAYLOAD_FORMAT_BANDWIDTH_EFFICIENT;
       }
   //AMRWB
   } else if(strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amrwb")==0 || strcasecmp(m_stRemoteEPAudioCodec.m_cCodecName,"amr-wb")==0) {
       pt=AMRWB;
       PMM_AUDIO_CODEC_OPTION_LIST audioOption = new(MM_AUDIO_CODEC_OPTION_LIST);
       INIT_MM_AUDIO_CODEC_OPTION_LIST(audioOption);

       m_mmAudioCodecRecord.options=audioOption;

       audioOption->eParm=EMM_AUDIO_CODEC_OPTION_PAYLOAD_FORMAT;
       if(m_stRemoteEPAudioCodec.m_bAmrOctetAlign==true) {
            audioOption->nValue=EMM_AUDIO_PAYLOAD_FORMAT_OCTET_ALIGNED;
       } else {
            audioOption->nValue=EMM_AUDIO_PAYLOAD_FORMAT_BANDWIDTH_EFFICIENT;
       }
   } else {
       pt=m_stRemoteEPAudioCodec.m_nAudioCodec;
   }

   // Get Coding type for Audio Play/Record
   if(g_nAudioFileFormat==MMD_3GP) {			// set unCoding and file extention right here
      if (g_nAudioCodec==AMRWB) {
         mmReport(INFO_DEBUG, s_eType, "3GP file format selected, changing Play Audio Codec to AMRWB");
         m_mmAudioCodecPlay.unCoding=m_mmAudioCodecRecord.unCoding=MM_DATA_FORMAT_AMR_WB_23_85K;
         SetPlayRecordFileName("_amrwb.3gp");      
      }
      else {
         mmReport(INFO_DEBUG, s_eType, "3GP file format selected, changing Play Audio Codec to AMR");
         m_mmAudioCodecPlay.unCoding=m_mmAudioCodecRecord.unCoding=MM_DATA_FORMAT_AMR_NB_12_20K;
         SetPlayRecordFileName(".3gp");
      }
   } else if (g_nAudioFileFormat==MMD_WAV){
      SetPlayRecordFileName(".wav");
      if(g_nAudioTranscodeDir==RX) {
         m_mmAudioCodecRecord.unCoding=MM_DATA_FORMAT_PCM;
      	 m_mmAudioCodecPlay.unCoding=GetUnCoding(pt);
      } else if(g_nAudioTranscodeDir==TX) {
         m_mmAudioCodecPlay.unCoding=MM_DATA_FORMAT_PCM;
         m_mmAudioCodecRecord.unCoding=GetUnCoding(pt);
      } else if(g_nAudioTranscodeDir==RXTX) {
         m_mmAudioCodecPlay.unCoding=m_mmAudioCodecRecord.unCoding=MM_DATA_FORMAT_PCM;
      } else if(g_nAudioTranscodeDir==NATIVE) {
         m_mmAudioCodecPlay.unCoding=m_mmAudioCodecRecord.unCoding=GetUnCoding(pt);
      }
   } else { //MMD_DMF
      if(g_nAudioTranscodeDir==RX) {
         m_mmAudioCodecRecord.unCoding=GetUnCoding(g_nAudioCodec);
      	 m_mmAudioCodecPlay.unCoding=GetUnCoding(pt);
      } else if(g_nAudioTranscodeDir==TX) {
         m_mmAudioCodecPlay.unCoding=GetUnCoding(g_nAudioCodec);
         m_mmAudioCodecRecord.unCoding=GetUnCoding(pt);
      } else if(g_nAudioTranscodeDir==RXTX) {
         m_mmAudioCodecPlay.unCoding=m_mmAudioCodecRecord.unCoding=GetUnCoding(g_nAudioCodec);
      } else if(g_nAudioTranscodeDir==NATIVE) {
         m_mmAudioCodecPlay.unCoding=m_mmAudioCodecRecord.unCoding=GetUnCoding(pt);
      }
   }
   if (g_nAudioCodec==AMRWB || g_nAudioCodec==G722 || g_nAudioCodec==PCM16K)
   {
      m_mmAudioCodecPlay.unSampleRate=m_mmAudioCodecRecord.unSampleRate=MM_DRT_16KHZ;
      m_mmAudioCodecPlay.unBitsPerSample=m_mmAudioCodecRecord.unBitsPerSample=16;
   } else {
      m_mmAudioCodecPlay.unSampleRate=m_mmAudioCodecRecord.unSampleRate=MM_DRT_8KHZ;
      m_mmAudioCodecPlay.unBitsPerSample=m_mmAudioCodecRecord.unBitsPerSample=16;
   }

}

void CMMStream::SetPlayRecordVideoCodec()
{
   if(g_nVideoTranscodeDir==RX)
   {
      if(g_nVideoCodec==MPEG4) {
         m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_MP4V_ES;
         if(g_maxBandwidth>128) {
            m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP1_MPEG4;
         } else if(g_maxBandwidth>64){
            m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
         } else {
            m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
         }
         if(m_stRemoteEPVideoCodec.m_nBitRate > VIDEO_BITRATE_128K) {
            m_mmVideoCodecPlay.Profile=VIDEO_PROFILE_LEVEL_SP1_MPEG4;
         } else if(m_stRemoteEPVideoCodec.m_nBitRate >VIDEO_BITRATE_64K) {
            m_mmVideoCodecPlay.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
         } else {
            m_mmVideoCodecPlay.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
         }
         m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level = EMM_VIDEO_LEVEL_UNDEFINED;
      } else if(g_nVideoCodec==H264) {
         m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H264;
         m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile = VIDEO_PROFILE_BASELINE_H264;
         m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_H264;
         if (g_nVideoRes==CIF) 
         {
            // at least this
            m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_1_H264;
            // might be higher
            if ( (g_maxBandwidth > 192000) && (g_maxBandwidth <= VIDEO_BITRATE_384K) ) {
               m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_2_H264;
            } else 
               m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level= VIDEO_LEVEL_1_3_H264;
               if ( (m_stRemoteEPVideoCodec.m_nBitRate > 192000) && (m_stRemoteEPVideoCodec.m_nBitRate <= VIDEO_BITRATE_384K) ) {
                  m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_2_H264;
               } else 
                  m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level= VIDEO_LEVEL_1_3_H264;
            }
      } else {
         if (g_nVideoCodec==H263_1998 || g_nVideoCodec==H263_2000)
         {
            m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H263_1998;
         } else {
            m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H263;
         } 
         // All 3gp files should be H263_1998
         if(g_nVideoFileFormat==MMD_3GP) {
            m_mmVideoCodecPlay.Coding=VIDEO_CODING_H263_1998;
         }
         m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile = VIDEO_PROFILE_0_H263;
         if(g_maxBandwidth>128) {
                m_mmVideoCodecRecord.Level=VIDEO_LEVEL_30_H263;
          } else if(g_maxBandwidth>64) {
                m_mmVideoCodecRecord.Level=VIDEO_LEVEL_20_H263;
          } else {
                m_mmVideoCodecRecord.Level=VIDEO_LEVEL_10_H263;
          }
         if(m_stRemoteEPVideoCodec.m_nBitRate > VIDEO_BITRATE_128K) {
                m_mmVideoCodecPlay.Level=VIDEO_LEVEL_30_H263;
          } else if(m_stRemoteEPVideoCodec.m_nBitRate >VIDEO_BITRATE_64K) {
                m_mmVideoCodecPlay.Level=VIDEO_LEVEL_20_H263;
          } else {
                m_mmVideoCodecPlay.Level=VIDEO_LEVEL_10_H263;
          }
      }
      if(g_nVideoRes==CIF) {
         m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_352;
         m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_288;
      } else if(g_nVideoRes==SQCIF) {
         m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_128;
         m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_96;
      } else {
         m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_176;
         m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_144;
      }
      if(m_stRemoteEPVideoCodec.m_nResolution==CIF) {
         m_mmVideoCodecPlay.ImageWidth=VIDEO_IMAGE_WIDTH_352;
         m_mmVideoCodecPlay.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_288;
      } else if(m_stRemoteEPVideoCodec.m_nResolution==SQCIF) {
         m_mmVideoCodecPlay.ImageWidth=VIDEO_IMAGE_WIDTH_128;
         m_mmVideoCodecPlay.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_96;
      } else {
         m_mmVideoCodecPlay.ImageWidth=VIDEO_IMAGE_WIDTH_176;
         m_mmVideoCodecPlay.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_144;
      }
      m_mmVideoCodecPlay.BitRate=m_mmVideoCodecRecord.BitRate=(eVIDEO_BITRATE)(g_maxBandwidth*1000);
      m_mmVideoCodecPlay.FramesPerSec=m_mmVideoCodecRecord.FramesPerSec=(eVIDEO_FRAMESPERSEC)MPIToFPS(30/g_maxFrameRate);
   } else if(g_nVideoTranscodeDir==TX) {
      if(g_nVideoCodec==MPEG4) {
          m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_MP4V_ES;
          if(g_maxBandwidth>128) {
              m_mmVideoCodecPlay.Profile=VIDEO_PROFILE_LEVEL_SP1_MPEG4;
          } else if(g_maxBandwidth>64){
              m_mmVideoCodecPlay.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
          } else {
              m_mmVideoCodecPlay.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
          }
          if(m_stRemoteEPVideoCodec.m_nBitRate > VIDEO_BITRATE_128K) {
              m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP1_MPEG4;
          } else if(m_stRemoteEPVideoCodec.m_nBitRate >VIDEO_BITRATE_64K) {
              m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
          } else {
              m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
          }
          m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level = EMM_VIDEO_LEVEL_UNDEFINED;
    } else if(g_nVideoCodec==H264) {
          m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H264;
          m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile = VIDEO_PROFILE_BASELINE_H264;
            m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_H264;
            if (g_nVideoRes==CIF) 
            {
                // at least this
                m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_1_H264;
                // might be higher
                if ( (g_maxBandwidth > 192000) && (g_maxBandwidth <= VIDEO_BITRATE_384K) ) {
                    m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_2_H264;
                } else 
                    m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level= VIDEO_LEVEL_1_3_H264;

                if ( (m_stRemoteEPVideoCodec.m_nBitRate > 192000) && (m_stRemoteEPVideoCodec.m_nBitRate <= VIDEO_BITRATE_384K) ) {
                    m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_2_H264;
                } else 
                    m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level= VIDEO_LEVEL_1_3_H264;
          }
    } else {
          if (g_nVideoCodec==H263_1998 || g_nVideoCodec==H263_2000)
          {
             m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H263_1998;
          } else {
             m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H263;
          } 
         // All 3gp files should be H263_1998
         if(g_nVideoFileFormat==MMD_3GP) {
            m_mmVideoCodecPlay.Coding=VIDEO_CODING_H263_1998;
         }
          m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile = VIDEO_PROFILE_0_H263;
          if(g_maxBandwidth>128) {
                m_mmVideoCodecPlay.Level=VIDEO_LEVEL_30_H263;
          } else if(g_maxBandwidth>64) {
                m_mmVideoCodecPlay.Level=VIDEO_LEVEL_20_H263;
          } else {
                m_mmVideoCodecPlay.Level=VIDEO_LEVEL_10_H263;
          }
         if(m_stRemoteEPVideoCodec.m_nBitRate > VIDEO_BITRATE_128K) {
                m_mmVideoCodecRecord.Level=VIDEO_LEVEL_30_H263;
          } else if(m_stRemoteEPVideoCodec.m_nBitRate >VIDEO_BITRATE_64K) {
                m_mmVideoCodecRecord.Level=VIDEO_LEVEL_20_H263;
          } else {
                m_mmVideoCodecRecord.Level=VIDEO_LEVEL_10_H263;
          }
    }
    if(g_nVideoRes==CIF) {
      m_mmVideoCodecPlay.ImageWidth=VIDEO_IMAGE_WIDTH_352;
      m_mmVideoCodecPlay.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_288;
    } else if(g_nVideoRes==SQCIF) {
      m_mmVideoCodecPlay.ImageWidth=VIDEO_IMAGE_WIDTH_128;
      m_mmVideoCodecPlay.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_96;
    } else {
      m_mmVideoCodecPlay.ImageWidth=VIDEO_IMAGE_WIDTH_176;
      m_mmVideoCodecPlay.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_144;
    }
    if(m_stRemoteEPVideoCodec.m_nResolution==CIF) {
      m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_352;
      m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_288;
    } else if(m_stRemoteEPVideoCodec.m_nResolution==SQCIF) {
      m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_128;
      m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_96;
    } else {
      m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_176;
      m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_144;
    }
    m_mmVideoCodecPlay.BitRate=m_mmVideoCodecRecord.BitRate=(eVIDEO_BITRATE)(g_maxBandwidth*1000);
    m_mmVideoCodecPlay.FramesPerSec=m_mmVideoCodecRecord.FramesPerSec=(eVIDEO_FRAMESPERSEC)MPIToFPS(30/g_maxFrameRate);
  } else if(g_nVideoTranscodeDir==RXTX) {
    if(g_nVideoCodec==MPEG4) {
          m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_MP4V_ES;
          if( (g_nVideoRes==CIF || g_nVideoRes==QCIF || g_nVideoRes==SQCIF) && g_maxBandwidth>128) {
              m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP1_MPEG4;
          } else if( (g_nVideoRes==CIF || g_nVideoRes==QCIF || g_nVideoRes==SQCIF) && g_maxBandwidth>64){
              m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
          } else if( (g_nVideoRes==QCIF || g_nVideoRes==SQCIF)&& g_maxBandwidth<=64) {
              m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
          }
          m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level = EMM_VIDEO_LEVEL_UNDEFINED;
    } else if(g_nVideoCodec==H264) {
          m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H264;
          m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile = VIDEO_PROFILE_BASELINE_H264;

            // Start at the lowest, let the code below up it to the appopriate one.
            m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_H264;
            if (g_nVideoRes==CIF) 
            {
                // at least this
                m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_1_H264;
                // might be higher
                if ( (g_maxBandwidth > 192000) && (g_maxBandwidth <= VIDEO_BITRATE_384K) ) {
                    m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level = VIDEO_LEVEL_1_2_H264;
                } else 
                    m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level = VIDEO_LEVEL_1_3_H264;
          }
    } else {
          if (g_nVideoCodec==H263_1998 || g_nVideoCodec==H263_2000)
          {
             m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H263_1998;
          } else {
             m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H263;
          } 
          // All 3gp files should be H263_1998
          if(g_nVideoFileFormat==MMD_3GP) {
            m_mmVideoCodecPlay.Coding=VIDEO_CODING_H263_1998;
          }
          m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile = VIDEO_PROFILE_0_H263;
          if( (g_nVideoRes==CIF || g_nVideoRes==QCIF || g_nVideoRes==SQCIF) && g_maxBandwidth>128) {
                m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_30_H263;
          } else if((g_nVideoRes==CIF ||g_nVideoRes==QCIF || g_nVideoRes==SQCIF) && g_maxBandwidth>64) {
                m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_20_H263;
          } else if( (g_nVideoRes==QCIF || g_nVideoRes==SQCIF) && g_maxBandwidth<=64) {
                m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_10_H263;
          }
    }
    if(g_nVideoRes==CIF) {
      m_mmVideoCodecPlay.ImageWidth=m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_352;
      m_mmVideoCodecPlay.ImageHeight=m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_288;
    } else if(g_nVideoRes==SQCIF) {
      m_mmVideoCodecPlay.ImageWidth=m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_128;
      m_mmVideoCodecPlay.ImageHeight=m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_96;
    } else {
      m_mmVideoCodecPlay.ImageWidth=m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_176;
      m_mmVideoCodecPlay.ImageHeight=m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_144;
    }
    m_mmVideoCodecPlay.BitRate=m_mmVideoCodecRecord.BitRate=(eVIDEO_BITRATE)(g_maxBandwidth*1000);
    m_mmVideoCodecPlay.FramesPerSec=m_mmVideoCodecRecord.FramesPerSec=(eVIDEO_FRAMESPERSEC)MPIToFPS(30/g_maxFrameRate);
  } else if(g_nVideoTranscodeDir==NATIVE) {
    if(m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_MP4V_ES) {
      m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_MP4V_ES;
      if( (m_stRemoteEPVideoCodec.m_nResolution==CIF || m_stRemoteEPVideoCodec.m_nResolution==QCIF 
           || m_stRemoteEPVideoCodec.m_nResolution==SQCIF) && g_maxBandwidth>128) {
        m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP1_MPEG4;
      } else if( (m_stRemoteEPVideoCodec.m_nResolution==CIF || m_stRemoteEPVideoCodec.m_nResolution==QCIF
                  || m_stRemoteEPVideoCodec.m_nResolution==SQCIF) && g_maxBandwidth>64) {
        m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
      } else if( (m_stRemoteEPVideoCodec.m_nResolution==QCIF || m_stRemoteEPVideoCodec.m_nResolution==SQCIF) && g_maxBandwidth<=64) {
        m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile=VIDEO_PROFILE_LEVEL_SP0_MPEG4;
      }
      m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level = EMM_VIDEO_LEVEL_UNDEFINED;
    	} else if(m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_H264) {
          m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H264;
          m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile = VIDEO_PROFILE_BASELINE_H264;

            // Start at the lowest, let the code below up it to the appopriate one.
            m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_H264;
            if (m_stRemoteEPVideoCodec.m_nResolution==CIF) 
            {
                // at least this
                m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_1_1_H264;
                // might be higher
                if ((g_maxBandwidth > 192000) && (g_maxBandwidth <= VIDEO_BITRATE_384K)) {
                    m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level = VIDEO_LEVEL_1_2_H264;
                } else 
                    m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level = VIDEO_LEVEL_1_3_H264;
          }
    } else {
      if (g_nVideoCodec==H263_1998 || g_nVideoCodec==H263_2000)
      {
         m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H263_1998;
      } else {
         m_mmVideoCodecPlay.Coding=m_mmVideoCodecRecord.Coding=VIDEO_CODING_H263;
      } 
      // All 3gp files should be H263_1998
      if(g_nVideoFileFormat==MMD_3GP) {
         m_mmVideoCodecPlay.Coding=VIDEO_CODING_H263_1998;
      }
      m_mmVideoCodecPlay.Profile=m_mmVideoCodecRecord.Profile = VIDEO_PROFILE_0_H263;
      if( (m_stRemoteEPVideoCodec.m_nResolution==CIF || m_stRemoteEPVideoCodec.m_nResolution==QCIF
           || m_stRemoteEPVideoCodec.m_nResolution==SQCIF) && g_maxBandwidth>128) {
        m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_30_H263;
      } else if( (m_stRemoteEPVideoCodec.m_nResolution==CIF || m_stRemoteEPVideoCodec.m_nResolution==QCIF
                  || m_stRemoteEPVideoCodec.m_nResolution==SQCIF) && g_maxBandwidth>64) {
        m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_20_H263;
      } else {
        m_mmVideoCodecPlay.Level=m_mmVideoCodecRecord.Level=VIDEO_LEVEL_10_H263;
      } 
    }
    if(m_stRemoteEPVideoCodec.m_nResolution==CIF) {
      m_mmVideoCodecPlay.ImageWidth=m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_352;
      m_mmVideoCodecPlay.ImageHeight=m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_288;
    } else if(m_stRemoteEPVideoCodec.m_nResolution==SQCIF) {
      m_mmVideoCodecPlay.ImageWidth=m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_128;
      m_mmVideoCodecPlay.ImageHeight=m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_96;
    } else {
      m_mmVideoCodecPlay.ImageWidth=m_mmVideoCodecRecord.ImageWidth=VIDEO_IMAGE_WIDTH_176;
      m_mmVideoCodecPlay.ImageHeight=m_mmVideoCodecRecord.ImageHeight=EMM_VIDEO_IMAGE_HEIGHT_144;
    }
    m_mmVideoCodecPlay.BitRate=m_mmVideoCodecRecord.BitRate=(eVIDEO_BITRATE)(g_maxBandwidth*1000);
    m_mmVideoCodecPlay.FramesPerSec=m_mmVideoCodecRecord.FramesPerSec=(eVIDEO_FRAMESPERSEC)MPIToFPS(30/g_maxFrameRate);
  }
}


void CMMStream::SetDciValue()
{
  if( m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_MP4V_ES && g_nVideoCodec==MPEG4) {
    if(g_nVideoTranscodeDir==RX) {
         //Transcode: use Dialogic DCI
        int ctr;
        int dciLen;
        char tempDciStr[128];
        unsigned char tempDciValue[128];
        if(m_stRemoteEPVideoCodec.m_nProfileLevelID>1) {
                strcpy(tempDciStr,DIALOGIC_CIF_DCI_STR);
        } else if(m_stRemoteEPVideoCodec.m_nProfileLevelID<2) {
                strcpy(tempDciStr,DIALOGIC_QCIF_DCI_STR);
        }
        dciLen=(strlen(tempDciStr)/2);
        for (ctr = 0; ctr <dciLen; ctr++) {
            tempDciValue[ctr]=AsciiOctetToHex(&tempDciStr[2*ctr]);
        }
        m_mmVideoCodecRecord.VisualConfigSize=dciLen;
        m_mmVideoCodecRecord.VisualConfiguration=tempDciValue;

      if(m_stRemoteEPVideoCodec.m_nProfileLevelID>1 && g_cTxDciCifSize>0) {
          strcpy(m_SdpDciStrValue,g_cTxDciCifStrValue);
        	m_SdpDciLen=m_mmVideoCodecPlay.VisualConfigSize=g_cTxDciCifSize;
        	m_SdpDciValue=m_mmVideoCodecPlay.VisualConfiguration=g_cTxDciCifValue;
        } else if(m_stRemoteEPVideoCodec.m_nProfileLevelID<=1 && g_cTxDciQcifSize>0) {
          strcpy(m_SdpDciStrValue,g_cTxDciQcifStrValue);
        	m_SdpDciLen=m_mmVideoCodecPlay.VisualConfigSize=g_cTxDciQcifSize;
        	m_SdpDciValue=m_mmVideoCodecPlay.VisualConfiguration=g_cTxDciQcifValue;
        }
        m_mmVideoCodecPlay.VisualConfigSize=m_SdpDciLen;
        m_mmVideoCodecPlay.VisualConfiguration=m_SdpDciValue;

        mmReport(INFO_ERROR, s_eType, "Caution: One way transcoding for MPEG4 codec is not advisable");
        mmReport(INFO_ERROR, s_eType, "Reason: User may not be able to view playback of recorded video correctly");

    } // video xcode direction = RX
    else if(g_nVideoTranscodeDir==TX) {

        m_mmVideoCodecRecord.VisualConfigSize=m_stRemoteEPVideoCodec.m_SdpDciLen;
        m_mmVideoCodecRecord.VisualConfiguration=m_stRemoteEPVideoCodec.m_SdpDciValue;

        if(g_nVideoRes==CIF && g_cTxDciCifSize>0) {
          m_mmVideoCodecPlay.VisualConfigSize=g_cTxDciCifSize;
          m_mmVideoCodecPlay.VisualConfiguration=g_cTxDciCifValue;
        } else if(g_nVideoRes==QCIF && g_cTxDciQcifSize>0) {
          m_mmVideoCodecPlay.VisualConfigSize=g_cTxDciQcifSize;
          m_mmVideoCodecPlay.VisualConfiguration=g_cTxDciQcifValue;
        }
        int ctr;
        if(m_stRemoteEPVideoCodec.m_nProfileLevelID>1) {
                strcpy(m_SdpDciStrValue,DIALOGIC_CIF_DCI_STR);
        } else if(m_stRemoteEPVideoCodec.m_nProfileLevelID<2) {
                strcpy(m_SdpDciStrValue,DIALOGIC_QCIF_DCI_STR);
        }
        m_SdpDciLen=(strlen(m_SdpDciStrValue)/2);
        for (ctr = 0; ctr <m_SdpDciLen; ctr++) {
            m_SdpDciValue[ctr]=AsciiOctetToHex(&m_SdpDciStrValue[2*ctr]);
        }
        mmReport(INFO_ERROR, s_eType, "Caution: One way transcoding for MPEG4 codec is not advisable");
        mmReport(INFO_ERROR, s_eType, "Reason: User may not be able to view playback of recorded video correctly");
    } // video xcode direction = TX
    else if(g_nVideoTranscodeDir==NATIVE) {
       // Assuming profile-level-id upto 1 is QCIF else CIF however every profile supports all 3 resolutions
       // Native: use tx dci from config file if available
       // usually come from stored media files header use [hmp3gp -i] to get that or availabe in offered SDP of RTSP server
       if(m_stRemoteEPVideoCodec.m_nProfileLevelID>1 && g_cTxDciCifSize>0) {
          strcpy(m_SdpDciStrValue,g_cTxDciCifStrValue);
          m_SdpDciLen=g_cTxDciCifSize;
          m_SdpDciValue=g_cTxDciCifValue;
       } else if(m_stRemoteEPVideoCodec.m_nProfileLevelID<2 && g_cTxDciQcifSize>0) {
          strcpy(m_SdpDciStrValue,g_cTxDciQcifStrValue);
          m_SdpDciLen=g_cTxDciQcifSize;
          m_SdpDciValue=g_cTxDciQcifValue;
       }
          m_mmVideoCodecRecord.VisualConfigSize=m_stRemoteEPVideoCodec.m_SdpDciLen;
          m_mmVideoCodecRecord.VisualConfiguration=m_stRemoteEPVideoCodec.m_SdpDciValue;
          m_mmVideoCodecPlay.VisualConfigSize=m_SdpDciLen;
          m_mmVideoCodecPlay.VisualConfiguration=m_SdpDciValue;

    } // video xcode native
    else if(g_nVideoTranscodeDir==RXTX) {
       //Transcode: use Dialogic DCI
       int ctr;
       if(m_stRemoteEPVideoCodec.m_nProfileLevelID>1) {
          strcpy(m_SdpDciStrValue,DIALOGIC_CIF_DCI_STR);
       } else if(m_stRemoteEPVideoCodec.m_nProfileLevelID<2) {
          strcpy(m_SdpDciStrValue,DIALOGIC_QCIF_DCI_STR);
       }
       m_SdpDciLen=(strlen(m_SdpDciStrValue)/2);
       for (ctr = 0; ctr <m_SdpDciLen; ctr++) {
          m_SdpDciValue[ctr]=AsciiOctetToHex(&m_SdpDciStrValue[2*ctr]);
       }
       m_mmVideoCodecRecord.VisualConfigSize=m_SdpDciLen;
       m_mmVideoCodecRecord.VisualConfiguration=m_SdpDciValue;

       if(g_nVideoRes==CIF && g_cTxDciCifSize>0) {
          m_mmVideoCodecPlay.VisualConfigSize=g_cTxDciCifSize;
          m_mmVideoCodecPlay.VisualConfiguration=g_cTxDciCifValue;
        } else if(g_nVideoRes==QCIF && g_cTxDciQcifSize>0) {
          m_mmVideoCodecPlay.VisualConfigSize=g_cTxDciQcifSize;
          m_mmVideoCodecPlay.VisualConfiguration=g_cTxDciQcifValue;
        }
    } // video xcode direction rxtx
  }
  else if( m_stRemoteEPVideoCodec.m_nCoderType!=CODER_TYPE_MP4V_ES && g_nVideoCodec==MPEG4) {
    if(g_nVideoTranscodeDir==RXTX) {
       //Transcode: use Dialogic DCI
       int ctr;
       if(m_stRemoteEPVideoCodec.m_nProfileLevelID>1) {
          strcpy(m_SdpDciStrValue,DIALOGIC_CIF_DCI_STR);
       } else if(m_stRemoteEPVideoCodec.m_nProfileLevelID<2) {
          strcpy(m_SdpDciStrValue,DIALOGIC_QCIF_DCI_STR);
       }
       m_SdpDciLen=(strlen(m_SdpDciStrValue)/2);
       for (ctr = 0; ctr <m_SdpDciLen; ctr++) {
          m_SdpDciValue[ctr]=AsciiOctetToHex(&m_SdpDciStrValue[2*ctr]);
       }
       m_mmVideoCodecRecord.VisualConfigSize=m_SdpDciLen;
       m_mmVideoCodecRecord.VisualConfiguration=m_SdpDciValue;

       if(g_nVideoRes==CIF && g_cTxDciCifSize>0) {
          m_mmVideoCodecPlay.VisualConfigSize=g_cTxDciCifSize;
          m_mmVideoCodecPlay.VisualConfiguration=g_cTxDciCifValue;
        } else if(g_nVideoRes==QCIF && g_cTxDciQcifSize>0) {
          m_mmVideoCodecPlay.VisualConfigSize=g_cTxDciQcifSize;
          m_mmVideoCodecPlay.VisualConfiguration=g_cTxDciQcifValue;
        }
    } // video xcode direction rxtx
  }
}



void CMMStream::SetH264DciValue()
{
   if( m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_H264 && g_nVideoCodec==H264) {
      if(g_nVideoTranscodeDir==NATIVE) {
         // based on the local config
         if(g_nVideoRes==CIF && g_cTxDciCifSize>0) {
            strcpy(m_SdpDciStrValue,g_cTxDciCifStrValue);
            m_SdpDciLen=g_cTxDciCifSize;
            m_SdpDciValue=g_cTxDciCifValue;
         } else if(g_nVideoRes==QCIF && g_cTxDciQcifSize>0) {
            strcpy(m_SdpDciStrValue,g_cTxDciQcifStrValue);
            m_SdpDciLen=g_cTxDciQcifSize;
            m_SdpDciValue=g_cTxDciQcifValue;
         }
         m_mmVideoCodecRecord.VisualConfigSize=m_stRemoteEPVideoCodec.m_SdpDciLen;
         m_mmVideoCodecRecord.VisualConfiguration=m_stRemoteEPVideoCodec.m_SdpDciValue;
         m_mmVideoCodecPlay.VisualConfigSize=m_SdpDciLen;
         m_mmVideoCodecPlay.VisualConfiguration=m_SdpDciValue;
      } // video xcode native
      else {
         mmReport(INFO_ERROR, s_eType, "H264 Transcoding SetH264DciValue()?\n");
      }
   }
}


#if 0
void CMMStream::SetLocalEPCodecValues()
{
  if(m_stRemoteEPVideoCodec.m_nCoderType==CODER_TYPE_MP4V_ES) {
		if (g_nVideoTranscodeDir==RXTX) {
          // Assuming profile-level-id upto 1 is QCIF else CIF however every profile supports all 3 resolutions
          // Native: use tx dci from config file if available
          // usually come from stored media files header use [hmp3gp -i] to get that or availabe in offered SDP of RTSP server
          if(m_stLocalEPVideoCodec.m_nProfileLevelID>1 && g_cTxDciCifSize>0) {
            strcpy(m_stLocalEPVideoCodec.m_SdpDciStrValue,g_cTxDciCifStrValue);
            m_stLocalEPVideoCodec.m_SdpDciLen=g_cTxDciCifSize;
            m_stLocalEPVideoCodec.m_SdpDciValue=g_cTxDciCifValue;
          } else if(m_stLocalEPVideoCodec.m_nProfileLevelID<2 && g_cTxDciQcifSize>0) {
            strcpy(m_stLocalEPVideoCodec.m_SdpDciStrValue,g_cTxDciQcifStrValue);
            m_stLocalEPVideoCodec.m_SdpDciLen=g_cTxDciQcifSize;
            m_stLocalEPVideoCodec.m_SdpDciValue=g_cTxDciQcifValue;
          }
    } else {
         //Transcode: use Dialogic DCI
        int ctr;
        strcpy(m_stLocalEPVideoCodec.m_SdpDciStrValue,(m_stLocalEPVideoCodec.m_nProfileLevelID>1?DIALOGIC_CIF_DCI_STR:DIALOGIC_QCIF_DCI_STR));
        m_stLocalEPVideoCodec.m_SdpDciLen=(strlen(m_stLocalEPVideoCodec.m_SdpDciStrValue)/2);
        for (ctr = 0; ctr <m_stLocalEPVideoCodec.m_SdpDciLen; ctr++) {
            m_stLocalEPVideoCodec.m_SdpDciValue[ctr]=AsciiOctetToHex(&m_stLocalEPVideoCodec.m_SdpDciStrValue[2*ctr]);
        }
    }
  }
}
#endif
//*****************************************************************************
// Function: void CMMStream::StopMedia()
// Description: Stop the media stream
// Return: void
// Parameters: none
//*****************************************************************************
void CMMStream::StopMedia()
{
#ifdef USE_RTSP
   mmReport(INFO_DEBUG, s_eType, "Stoping RTP streams for %s\n", m_ipmRTSPDevName);
   if ( ipm_Stop(m_ipmRTSPH, STOP_ALL, EV_ASYNC) != 0 )
   {
      mmReport(INFO_DEBUG, s_eType, "ipm_Stop() failed - %s\n", ATDV_ERRMSGP(m_ipmRTSPH));
   }
#endif
}

#ifdef TDM_AUDIO
void CMMStream::SaveISDNGwCmd(char * SIPHeader)
{
   // Should look something like this:
   // "CallInfo:C dtiB1T2 1112223333:4445556666"
   // Strip off "CallInfo:", then parse remaining elements

   ParseISDNGWCmd(SIPHeader + 9, m_gwCmd, m_3GChannelTag, m_ISDNCalledNumber, m_ISDNCallingNumber);
   mmReport(INFO_DEBUG, s_eType, "ISDN CalledNumber:%s Calling Number:%s", m_ISDNCalledNumber, m_ISDNCallingNumber);

   struct timeb timebuffer;
   ftime(&timebuffer);
   char timestamp[32];
   struct tm *timeParts = localtime (&(timebuffer.time));
   sprintf (timestamp, "-%d-%d-%d-%d", 
            timeParts->tm_hour,
            timeParts->tm_min,
            timeParts->tm_sec,
            timebuffer.millitm);

   std::string ISDNCalledNumber(m_ISDNCalledNumber);
   std::string ISDNCallingNumber(m_ISDNCallingNumber);
   std::string timeStamp(timestamp);

   std::string cAudioRecordFileName = ISDNCalledNumber + "-" + ISDNCallingNumber + timeStamp;
   std::string cVideoRecordFileName = ISDNCalledNumber + "-" + ISDNCallingNumber + timeStamp;
   std::string filePath = g_cAVFileDir;

   filePath = filePath + "/";
   m_cAudioRecordFile = filePath + cAudioRecordFileName + ".pcm";
   m_cVideoRecordFile = filePath + cVideoRecordFileName + ".vid";

}


void CMMStream::ParseISDNGWCmd(char * cmdString, char *cmd, char *device, char *calledNumber, 
                               char *callingNumber)
{
   // Utility routine to parse ISDN gateway to 3G-SIP gateway and ISDN gateway to MM demo commands
   //
   // Commands are of the form "C dtiB1T1 111222333:4445556666"
   // We need to extract network device name, calling number (last string before colon) 
   // and called number (last string after colon)

   char delim[] = " ";
   char *result = NULL;

   result = strtok( cmdString, delim);
   int tokCnt = 1;
   while ( result != NULL )
   {
      switch ( tokCnt )
      {
         case 1:
            strcpy(cmd, result);
            break;
         case 2:
            strcpy(device, result);
            break;
         case 3:
            {
               // Parse this out into two pieces using a : for a token
               char newDelim[] = ":";
               char *newResult = NULL;
               newResult = strtok( result, newDelim);
               int tokCnt2 = 1;
               while ( newResult != NULL )
               {
                  switch ( tokCnt2 )
                  {
                     case 1:
                        strcpy(calledNumber, newResult);
                        break;
                     case 2:
                        strcpy(callingNumber, newResult);
                        break;
                  }
                  newResult = strtok( NULL, newDelim);
                  tokCnt2++;
               }
            }
            break;
      }
      result = strtok( NULL, delim);
      tokCnt++;
   }  

}
#endif //TDM_AUDIO

eIPM_CODER_TYPE CMMStream::GetAmrwbCoderType(int modeset)
{
   switch (modeset)
   {
      case 0:
        return CODER_TYPE_AMRWB_6_6K;
      case 1:
        return CODER_TYPE_AMRWB_8_85K;
      case 2:
        return CODER_TYPE_AMRWB_12_65K;
      case 3:
        return CODER_TYPE_AMRWB_14_25K;
      case 4:
        return CODER_TYPE_AMRWB_15_85K;
      case 5:
        return CODER_TYPE_AMRWB_18_25K;
      case 6:
        return CODER_TYPE_AMRWB_19_85K;
      case 7:
        return CODER_TYPE_AMRWB_23_05K;
      case 8:
      default:
        return CODER_TYPE_AMRWB_23_85K;
   }
}

const char* CMMStream::GetAudioCodecStr(int audioCodec)
{
   switch (audioCodec)
   {
      case NO_AUDIO_CODEC:
         return "none";
      case G711U:
         return "g711u";
      case G711A:
         return "g711a";
      case G723:
         return "g723";
      case G729:
         return "g729";
      case G726:
         return "g726";
      case RFCDTMF:
         return "dtmf";
      case AMRWB:
         return "amrwb";
      case PCM:
         return "pcm";
      case PCM16K:
         return "pcm16k";
      case G722:
         return "g722";
      case AMR:
         return "amr";
      default:
         return "unknown";
   }
}

const char* CMMStream::GetVideoCodecStr(int videoCodec)
{
   switch (videoCodec)
   {
      case NO_VIDEO_CODEC:
         return "none";
      case H263:
         return "h263";
      case H264:
         return "h264";
      case MPEG4:
         return "mpeg4";
      case H263_1998:
         return "h263+(H263_1998)";
      case H263_2000:
         return "h263++(H263_2000)";
      default:
         return "unknown";
   }
}

//------- end of file --------
