#!/usr/bin/perl -w
##############################################################################
# DIALOGIC CONFIDENTIAL
#
# Copyright (C) [2007] Dialogic Corporation. All Rights Reserved.
# The source code contained or described herein and all documents related
# to the source code ("Material") are owned by Dialogic Corporation or its
# suppliers or licensors. Title to the Material remains with Dialogic Corporation
# or its suppliers and licensors. The Material contains trade secrets and
# proprietary and confidential information of Dialogic or its suppliers and
# licensors. The Material is protected by worldwide copyright and trade secret
# laws and treaty provisions. No part of the Material may be used, copied,
# reproduced, modified, published, uploaded, posted, transmitted, distributed,
# or disclosed in any way without Dialogic's prior express written permission.
#
# No license under any patent, copyright, trade secret or other intellectual
# property right is granted to or conferred upon you by disclosure or delivery
# of the Materials, either expressly, by implication, inducement, estoppel or
# otherwise. Any license under such intellectual property rights must be
# express and approved by Dialogic in writing.
##############################################################################

##############################################################################
#        NAME 		: cte.pl
# DESCRIPTION 		: Automation tool for testing the CLI interface
#    CAUTIONS 		: None
#     HISTORY     : 1.0 - Initial Release
#                 : 1.1 - Changed default username/password to root/public
#                 : 1.2 - Modified help screen and CheckForModules() for demo version.
##############################################################################
use strict;
use Getopt::Std;

print "\ncte.pl - CLI Automation & Test Tool";

CheckForModules();            # Will exit the script if Telnet Module is not installed
require Net::Telnet;

#### Signatures
sub PrintHelp;
sub ValidateIP;

##############################################################################
#### Initialization ####
my $dbg     = 0;                                # Debug flag
my $quit    = 0;                                # Quit on first failure flag
my $strict  = 0;                                # Strict mode flag

my %args = ( i  => '127.0.0.1',                 # default ip address
             p  => 0,                           # telnet port
             l  => 0,                           # loops
             u  => '0',                         # Login name
             c  => 'cli_sanity.cfg' );          # default config file


# Read in command line arguments
getopts('?sSdDhHqQi:p:c:l:u:', \%args);

# Set the debug flag
if (defined $args{d} or $args{D}) { $|++; $dbg = 1; }

# Set the quit flag if it has been defined
if (defined $args{q} or $args{Q}) { $quit = 1; }

# Set the strict flag if it has been defined
if (defined $args{s} or $args{S}) { $strict = 1; }

# Check for help flag, PrintHelp() terminates script
if (defined ($args{h} or $args{H} or $args{'?'})) { PrintHelp(); }

# Create test log file, remove extension if necessary
my $testlog;
if ($args{c} =~ /(.*)(?:\.\w+$)/)
{
  $testlog = $1.".log";
}
else
{
  $testlog = $args{c}.".log";
}

open logFH, ">$testlog" or die "Failed opening log file. Error: $!\n";
print logFH "cte.pl - CLI Automation & Test Tool\n";

# Comparing for "MSWin32" instead of "linux" in case this is used on other Unix systems.
if (not $^O =~ /^MSWin32/i) { system("dos2unix $args{c} 2> /dev/nul"); }

# Open the config file and parse.
open cfgFH, $args{c} or die "Failed to open config file ".
  $args{c}. " Error: $!\n";

my $tcid    = 0;                                  # Test case index
my %testcases;                                    # Store TestCases
my %events;                                       # Store events to wait for
my %substrings;                                   # Store substrings to wait for
my %userargs;                                     # Store user variables
print "\nParsing Config File: $args{c}";
while (<cfgFH>)
{
  print ".";

  # Ignore lines that start with a comment or contain only whitespace
  if ( ($_ !~ /^\s*#/) and ($_ !~ /^\s+$/) )
  {
    chomp($_);

    # Look for arguments / testcases
    if ($_ =~ /^-[iI]\s*([\w\d\.]+)/i)
    {
      # IP address is specified, set to %args if not set on the command line
      $args{i} = $1 if ($args{i} eq '127.0.0.1');
    }
    elsif ($_ =~ /^-[pP]\s*(\d{2,5})/i)
    {
      # Port is specified, set to %args if not specified on the command line
      $args{p} = $1 if ($args{p} == 0);
    }
    elsif ($_ =~ /^-[lL]\s*(\d+)/i)
    {
      # Loops are specified, set to %args if not specified on the command line
      $args{l} = $1 if ($args{l} == 0);
    }
    elsif ($_ =~ /^-[uU]\s*(\w+)/i)
    {
      # Login name is specified, set to %args if not specified on the command line
      $args{u} = $1 if ($args{u} eq '0');
    }
    elsif ($_ =~ /^-[qQ]\s*(\w+)/i)
    {
      # Quit flag has been specified
      $quit = 1;
    }
    elsif ($_ =~ /^-[sS]\s*(\w+)/i)
    {
      # Strict flag has been specified
      $strict = 1;
    }
    elsif ($_ =~ /\s*'(.*)'\s*$/)
    {
      # Just a command has been found
      $tcid++;
      $testcases{$tcid}{Command}   = $1;
      $testcases{$tcid}{CompareOp} = undef;

      if ($dbg == 1)
      {
        print "\nDEBUG: Line: $. - Initializing TestCase $tcid".
              "\n\tCommand      - $testcases{$tcid}{Command}".
              "\n\tCompare Op   - 'Undefined'".
              "\n\tResponse     - 'Undefined'".
              "\n";
      }
    }
    elsif ($_ =~ /\s*'(.*)'\s*([=~!]>)\s*\|(.*)/)
    {
      # A command / comparison operator / response has been found
      $tcid++;
      $testcases{$tcid}{Command}   = $1;
      $testcases{$tcid}{CompareOp} = $2;

      # If the command operator is an exact match
      if ($testcases{$tcid}{CompareOp} eq '=>')
      {
        # Just push the entire string onto the array
        push (@{$testcases{$tcid}{Response}}, $3);
      }
      else
      {
        # The response string needs to be broken down into the substrings
        # and then pushed onto the array
        my @substrings = split(',', $3);
        foreach my $str (@substrings)
        {
          # Remove leading and trailing whitespace
          $str =~ s/^\s+//;
          $str =~ s/\s+$//;
          push (@{$testcases{$tcid}{Response}}, $str);
        }
      }

      if ($dbg == 1)
      {
        print "\nDEBUG: Line: $. - Initializing TestCase $tcid".
              "\n\tCommand      - $testcases{$tcid}{Command}".
              "\n\tCompare Op   - $testcases{$tcid}{CompareOp}".
              "\n\tResponse     - @{$testcases{$tcid}{Response}}".
              "\n";
      }
    }
    elsif ($_ =~ /\s*!EVENT\[(.*)\]$/i)
    {
      # An event has been defined, grab the entire string as is
      $events{$tcid} = $1;

      if ($dbg == 1)
      {
        print "\nDEBUG: Line: $. - Initializing Events".
            "\n\tEvent(s)     - $events{$tcid}".
            "\n";
      }
    }
    elsif ($_ =~ /\s*!SUBSTRING\[(.*)\]$/i)
    {
      # An substring has been defined, grab the entire string as is
      $substrings{$tcid} = $1;

      if ($dbg == 1)
      {
        print "\nDEBUG: Line: $. - Initializing Events".
            "\n\tEvent(s)     - $events{$tcid}".
            "\n";
      }
    }
    elsif ($_ =~ /\s*\|(.*)/)
    {
      # All other lines with a '|' will be considered part of the response to the
      # last test case found.
      if (defined $testcases{$tcid}{CompareOp})
      {
        # If the current command's compare operator is an exact match, just push
        # the entire string onto the array
        if ($testcases{$tcid}{CompareOp} eq '=>')
        {
          push (@{$testcases{$tcid}{Response}}, $1);
        }
        else
        {
          # Break up the substrings and push them onto the array
          my @substrings = split(',', $1);
          foreach my $str (@substrings)
          {
            $str =~ s/^\s+//;
            push (@{$testcases{$tcid}{Response}}, $str);
          }
        }

        if ($dbg == 1)
        {
          print "\nDEBUG: Line $.\n\tTestCase: $tcid  - Added Response \"$1\"\n";
        }
      }
      else
      {
        # If the last defined testcase is not to test the response then ignore the line
        print "\nWARNING: Line $.: Response string cannot be assigned to a ".
          "command that does not\n\t have a comparison operator defined. Ignoring.\n";
      }
    }
    elsif ($_ =~ /\s*\[\s*(&\w+)\s*(.*-.*)\s*"(.*)"\s*$/)
    {
      # A user defined variable has been defined, parse the location operator and the token
      $userargs{$tcid}{$1}{LocationOp} = $2;
      $userargs{$tcid}{$1}{Token}      = $3;
      if ($dbg == 1)
      {
        print "\nDEBUG: Line: $. - Initializing User Variable".
            "\n\tVariable     - $1".
            "\n\tLocation Op  - $userargs{$tcid}{$1}{LocationOp}".
            "\n\tToken        - $userargs{$tcid}{$1}{Token}".
            "\n";
      }
    }
    else
    {
      # All other lines will be ignored
      print "\nWARNING: Line $. contains unrecognized format.  Ignoring.\n";
    }
  }
}
close cfgFH;
print "Completed";

# If the IP address is not valid print the help menu
if (ValidateIP($args{i}) == 0) { PrintHelp(); }
# If the port is not specified use default telnet port
if ($args{p} == 0) { $args{p} = 23; }
# If loops are not specified set to 1 as default
if ($args{l} == 0) { $args{l} = 1; }
# If login name is not specified set to 'root' as default
if ($args{u} eq '0') { $args{u} = "root"; }

# Print final parameters to console if debugging
if ($dbg == 1) { print "\nDEBUG: '$_' : $args{$_}" foreach sort keys %args; }

# Log parameters
print logFH "\n*** Parameters ***".
            "\nConfig File  : ". $args{c}.
            "\nIP Address   : ". $args{i}.
            "\nPort         : ". $args{p}.
            "\nLogin name   : ". $args{u}.
            "\nPassword     : ". $args{u}.
            "\nLoops        : ". $args{l}.
            "\n\n*** Modes ***".
            "\nDebug        : ". ($dbg    == 0 ? "Off" : "On").
            "\nStrict       : ". ($strict == 0 ? "Off" : "On").
            "\nQuit         : ". ($quit   == 0 ? "Off" : "On");

# Sort the testcases hash, print to the screen
my @tcids = sort { $a <=> $b } keys %testcases;
if (@tcids == 0)
{
  # If the config file contained no valid test cases exit out
  print "\nFailed to initialize any commands. Check config file '$args{c}'.\n";
  exit 1;
}
else
{
  print "\nInitialization complete.  Found ". @tcids. " commands.";
}

# Log defined tests to the log file
print logFH "\n\n*** Test Case Definitions ***";
foreach my $tcid (@tcids)
{
  # Print the header
  print logFH "\n\nTestCase ID  : $tcid".
              "\nCommand      : $testcases{$tcid}{Command}";

  # If a compare operator has been defined
  print logFH "\nOperator     : ";
  if (defined $testcases{$tcid}{CompareOp})
  {
    print logFH "$testcases{$tcid}{CompareOp}";
  }
  else
  {
    print logFH "\'Undefined\'";
  }

  # If a response string has been defined
  if (defined @{$testcases{$tcid}{Response}})
  {
    foreach my $resp (@{$testcases{$tcid}{Response}})
    {
      print logFH "\nResponse     : $resp";
    }
  }
  else
  {
    print logFH "\nResponse     : \'Undefined\'";
  }
  
  if (defined $userargs{$tcid})
  {
    print logFH "\n\t*** User Variables ***";
    foreach my $var (keys %{$userargs{$tcid}})
    {
      print logFH "\n\tVariable     : $var".
                  "\n\tLocation Op  : $userargs{$tcid}{$var}{LocationOp}".
                  "\n\tToken        : \'$userargs{$tcid}{$var}{Token}\'";
    }
  }
  
  if (defined $events{$tcid})
  {
    print logFH "\n\t*** Events ***\n\tEvent(s)    : ".$events{$tcid};
  }
  
  if (defined $substrings{$tcid})
  {
    print logFH "\n\t*** Substrings ***\n\tSubstring(s)    : ".$substrings{$tcid};
  }

}
#### Initialization complete ####

##############################################################################
#### Start executing tests  ####

# Start CLI session
print "\n\nStarting CLI session\n";
my $username  = $args{u};
my $password  = $username;

if ($username =~ /^root$/i) {$password = "public";}

# Create the object, 5 minute timeout.  The use of event and substring handling
# should alleviate any need for strict timeouts.
my $mycli     = Net::Telnet->new( Host        => $args{i},
                                  Port        => $args{p},
                                  Timeout     => 300,
                                  Prompt      => '/CLI>\s*$/');
# Create CLI session log files
$mycli->buffer_empty;
my $cliFH = $mycli->input_log("cli_session.log");
if ($dbg == 1) { $mycli->dump_log("network_dump.log") };

# The telnet session needs a kick start to begin pushing data
$mycli->print ("");

# The CLI uses 'Login :' and 'Password :' for the login prompts.  The login
# method only supports /login[: ]*$/i/ so waitfor()/print() is used instead.
$mycli->waitfor('/Login\s*:/i');
$mycli->print($username);
$mycli->waitfor('/Password\s*:$/i');
$mycli->print($password);
$mycli->waitfor('/CLI>\s$/');

print logFH "\n\n*** Execution Log ***";
# Test statistics
my $totalcmds     = 0;              # Total commands issued
my $totalcmprs    = 0;              # Total string comparisons
my $failedcmprs   = 0;              # Total failed string comparisons
my $totalwarns    = 0;              # Total number of warnings issued

# Testcase specific
my $loopcnt;                        # Current loop
my $orgcmd;                         # Original command string
my $command;                        # Current command
my $compare;                        # Compare operator
my @expect;                         # Expected response from the CLI command
my $result;                         # Flag to denote if testcase passed

# User defined arguments
my %realargs;                       # User defined argument hash
my $realvar;                        # The real argument to pass into the command string
my @output;                         # CLI response
my $outputline;                     # Line element from the output array
my $var;                            # Placeholder for each var in the user variables hash

# Volatile data - Digits are considered volatile data when not in strict mode.
my @actual_ss;                      # Holds the actual output substrings
my @expect_ss;                      # Holds the expected output substrings
my $found = 0;                      # Flag to determine if substring is found

# Loop through each tcid for the number of loops specified
for ($loopcnt = 1; $loopcnt <= $args{l}; $loopcnt++)
{
  # Cycle through each test case
  foreach my $tcid (@tcids)
  {
    print "\n\nTestCase: $tcid";

    # Set the command, compare operator, and expected response
    $command = $testcases{$tcid}{Command};
    $orgcmd  = $command;                               # Used only for debugging
    $compare = $testcases{$tcid}{CompareOp};
    @expect  = @{$testcases{$tcid}{Response}};

    # Cycle through each argument replacing the user defined argument with the actual
    # argument
    foreach $realvar (keys %realargs)
    {
      # If the command string contains the argument switch the user defined argument
      # with the actual argument string
      if ($command =~ /$realvar\s|$realvar$/)
      {
        $command =~ s/$realvar/$realargs{$realvar}/g;
        if ($dbg == 1) { print "\nDEBUG: Replaced \'$realvar' with \'$realargs{$realvar}\'"; }
      }
    }


    # Now issue the command
    $result = 1;
    if ($dbg == 1) { print "\nDEBUG: Issuing Command: $command"; }
    @output = $mycli->cmd($command);
    $totalcmds++;
    if ($dbg == 1) { print "\nDEBUG: Response Received:\n@output\n"; }

    # If there are user defined arguments to be parsed in the response
    if ($userargs{$tcid})
    {
      # Cycle through each variable in the %userargs hash
      foreach $var (keys %{$userargs{$tcid}})
      {
        # Within each output line of the output array
        foreach $outputline (@output)
        {
          # Extract the substring and populate the %realargs hash
          if ($userargs{$tcid}{$var}{LocationOp} =~ /^\s*->\s*$/i and $outputline =~ /$userargs{$tcid}{$var}{Token}\s+(\S+)\s*/)
          {
            $realargs{$var} = $1;
          }
	        elsif ($userargs{$tcid}{$var}{LocationOp} =~ /^\s*<-\s*$/i and $outputline =~ /\s*(\S+)\s+$userargs{$tcid}{$var}{Token}/)
          {
            $realargs{$var} = $1;
          }
          
          # Trim off any non alphanumeric characters
          if (defined $realargs{$var} and $realargs{$var} =~ /(.*)[\W*]$/)
          {
            $realargs{$var} = $1;
          }
        }

        if (!defined $realargs{$var})
        {
          print "\n\tWARNING: User Variable: '$var\' NOT set.  Did not find ".
            "\'$userargs{$tcid}{$var}{Token}\'\n\tin response.";
          print logFH "\n\nTestCase: $tcid\tWARNING: Variable: '$var\' NOT set.  Did not find ".
            "\'$userargs{$tcid}{$var}{Token}\' in response.";
          $totalwarns++;
        }
        elsif ($dbg == 1)
        {
          print "\nDEBUG: Set Variable \'$var\' to \'$realargs{$var}\'";
        }
      }
	  }

    # If a substring has been defined test for it here.  Even though we are testing the output below
    # it is possible that a repsonse comparison has not be defined.
    if ($substrings{$tcid})
    {
      if ($dbg == 1)
      {
         print "\nDEBUG: Looking for substring \'".$substrings{$tcid}."\'";
      }
      
      foreach (@output)
      {
        if ($_ =~ /$substrings{$tcid}/)
        {
          # !SUBSTRING and !EVENT have an either/or approach, if the substring has been detected, undefine the events
          $events{$tcid} = undef;
          if ($dbg == 1)
          {
             print "\nDEBUG: Substring \'".$substrings{$tcid}."\' found in output.";
          }
          last;
        }
      }
    }

    # If events have been defined wait for them here, else strip them from the output
    if ($events{$tcid})
    {
      if ($dbg == 1)
      {
         print "\nDEBUG: Looking for event(s) \'".$events{$tcid}."\'";
      }

      # Sometimes the EVENTS get caught in the output returned in the cmd() method,
      # test for them here
      $found = 0;
      foreach (@output)
      {
        if ($_ =~ /$events{$tcid}/)
        {
          $found = 1;
          if ($dbg == 1)
          {
             print "\nDEBUG: Event(s) \'".$events{$tcid}."\' found in output";
          }

          # Allow the prompt to come back
          sleep 2;
        }
      }

      if ($found == 0)
      {
        my $rc = $mycli->waitfor('/'.$events{$tcid}.'/');
        $rc = $mycli->waitfor('/CLI> /');
        # TODO: Currently if the waitfor() fails it will exit the script.
        # Implement handler and set in telnet object
        if ($rc != 1)
        {
          print "\nEvent(s): ".$events{$tcid}." NOT detected";
        }
        else
        {
          if ($dbg == 1)
          {
             print "\nDEBUG: Found event(s) \'".$events{$tcid}."\' in waitfor()";
          }

          # Allow the prompt to come back
          sleep 2;
        }
      }
    }
    else
    {
      for (my $i = 0; $i < @output; $i++)
      {
        if ($output[$i]=~ /^EVENT--/) { splice (@output, $i, 1); }
      }
    }

    # Compare the results depending on the operator used in the config file
    if (defined $compare)
    {
      if ($compare eq '=>')       # Exact Comparison
      {
        # Remove the last element in the array if it contiains no alphanumeric characters.  In some instances the
        # CLI will return a '\n' which needs to be removed for an exact compare.
        if ($output[@output - 1] !~ /\w+/) { pop @output; }

        # Check to see if there is more output than expected
        if (@output > @expect)
        {
          print "\n\tWARNING - Actual output lines exceeded number of expected lines.  See log";
          print logFH "\n\nTestCase: $tcid\tWARNING: Actual output lines exceeded the ".
              "number of expected lines.  Only up to the number of expected ".
              "lines will be tested.".
              "\n\tCommand            : ". $command.
              "\n\tExpected Output    : ". @expect. " lines".
              "\n\tActual Output      : ". @output. " lines";

          $totalwarns++;
        }
        
        # Compare expected to response
        for (my $i = 0; $i < @expect; $i++)
        {
          # Test if number of expected lines exceeds CLI output
          if ($i == @output)
          {
            print "\n\tWARNING - Expected output lines exceeded the number of actual output lines. See log.";
            print logFH "\n\nTestCase: $tcid\tWARNING: Expected output lines exceeded ".
              "the number of actual output lines.  Skipping the rest of the comparisons.".
              "\n\tCommand            : ". $command.
              "\n\tExpected Output    : ". @expect. " lines".
              "\n\tActual Output      : ". @output. " lines";

            $totalwarns++;
            last;
          }
          else
          {
            # Increment the total comparisons
            $totalcmprs++;

            # Remove the <CR> from the end of the line
            chomp $output[$i];
            if ($dbg == 1) { print "\nDEBUG: Comparing output line ". ($i + 1). ": \"". $output[$i]. "\""; }
            
            # When not in strict mode leading and trailing whitespace is trimmed, and volatile data is not compared
            if ($strict == 0)
            {
              # Remove leading and trailing whitespace
              $output[$i] =~ s/^\s+//;
              $output[$i] =~ s/\s+$//;
              $expect[$i] =~ s/^\s+//;
              $expect[$i] =~ s/\s+$//;
              
              # Remove any left over <CR>
              $output[$i] =~ s/\^\x0D+//;
              $output[$i] =~ s/\\x0D+$//;
              $expect[$i] =~ s/\^\x0D+//;
              $expect[$i] =~ s/\\x0D+$//;

              # If the output contains digits there is volatile data, break the lines into substrings then
              # only compare the strings which have no digit characters
              if ($output[$i] =~ /\d+/)
              {
                print logFH "\n\nTestCase: $tcid\tWARNING: Output contained volatile data. The line ".
                "will be broken down into substrings with the non volatile strings being compared.".
                "\n\tCommand            : ". $command.
                "\n\tExpected Output    : ". $expect[$i].
                "\n\tActual Output      : ". $output[$i];
                $totalwarns++;

                @expect_ss = split(' ', $expect[$i]);
                @actual_ss = split(' ', $output[$i]);

                # NOTE: Total Compares will still only be incremented once per line and not individual substrings
                # when not in strict mode.
                for (my $i = 0; $i < @actual_ss; $i++)
                {
                  $found = 0;
                  if (not $actual_ss[$i] =~ /\d+/)
                  {
                    for (my $j = 0; $j < @expect_ss; $j++)
                    {
                      if ($dbg == 1) { print "\nDEBUG: Comparing \'". $actual_ss[$i]. "\' to \'". $expect_ss[$j]. "\'"; }
                      if ($actual_ss[$i] eq $expect_ss[$j])
                      {
                        $found = 1;
                        if ($dbg == 1) { print " - FOUND"; }
                        last;
                      }
                      else
                      {
                        if ($dbg == 1) { print " - NOT FOUND"; }
                      }
                    } # End foreach my $expect_ss[$j]

                    if ($found  == 0)
                    {
                      print "\n\tFAILED - Comparison ". ($i + 1). ". See log.";
                      print logFH "\n\nTestCase: $tcid\tFAILED Comparison".
                        "\n\tCommand        : $command".
                        "\n\tExpected Token : \'". $actual_ss[$i]. "\' - NOT FOUND";

                      $failedcmprs++;
                      $result = 0;
                      if ($quit == 1) { last; }
                    }
                    else
                    {
                      if ($dbg == 1) { print " - PASSED"; }
                    }
                  }
                } # End foreach my $actual_ss[$i]
              }
              else
              {
                # No digits in the string, compare normally
                if ($output[$i] ne $expect[$i])
                {
                  $result = 0;
                  $failedcmprs++;
                  print "\n\tFAILED - Comparison ". ($i + 1). ". See log.";

                  print logFH "\n\nTestCase: $tcid\tFAILED Comparison".
                    "\n\tCommand      : $command".
                    "\n\tOutput Line  : ". ($i + 1).
                    "\n\tReceived     : \"$output[$i]\"".
                    "\n\tExpected     : \"$expect[$i]\"";

                  # If quit is enabled quit the test case on failure
                  if ($quit == 1) { last; }
                }
              }
            }
            elsif ($output[$i] ne $expect[$i])
            {
              $result = 0;
              $failedcmprs++;
              print "\n\tFAILED - Comparison ". ($i + 1). ". See log.";

              print logFH "\n\nTestCase: $tcid\tFAILED Comparison".
                "\n\tCommand      : $command".
                "\n\tOutput Line  : ". ($i + 1).
                "\n\tReceived     : \"$output[$i]\"".
                "\n\tExpected     : \"$expect[$i]\"";

              # If quit is enabled quit the test case on failure
              if ($quit == 1) { last; }
            }
            else
            {
              if ($dbg == 1) { print " - PASSED"; }
            }
          }
        } # End for($i)
      }
      else     # Either a '~> or !>' comparison
      {
        # Loop through each expected substring
        for (my $i = 0; $i < @expect; $i++)
        {
          $totalcmprs++;
          $found = 0;

          # Loop through each line of output
          for (my $j = 0; $j < @output; $j++)
          {
            # If the expected substring has been matched
            if ($output[$j] =~ /$expect[$i]/)
            {
              $found = 1;
              last;
            }
          }

          # If $found = 0 the substring has not been found
          if ($found == 0)
          {
            # Fail the comparision test if the operator was to contain the substring
            if ($compare eq '~>')
            {
              $result = 0;
              $failedcmprs++;
              if ($dbg == 1)
              {
                print "\nDEBUG: '$expect[$i]' NOT Found";
              }
              
              print "\n\tFAILED - Comparison ". ($i + 1). ". See log.";

              chomp @output;
              print logFH "\n\nTestCase: $tcid\tFAILED Comparison".
                "\n\tCommand        : $command".
                "\n\tExpected Token : '$expect[$i]' NOT FOUND";

              # If quit is enabled quit the test case on failure
              if ($quit == 1) { last; }
            }
          }
          else
          {
            # Fail the comparison test if the operator was to 'NOT' contain the substring
            if ($compare eq '!>')
            {
               $result = 0;
               $failedcmprs++;

               if ($dbg == 1)
               {
                 print "\nDEBUG: '$expect[$i]' Found";
               }

               print "\n\tFAILED - Comparison ". ($i + 1). ". See log.";
               
               chomp @output;
               print logFH "\n\nTestCase: $tcid\tFAILED Comparison".
                "\n\tCommand          : $command".
                "\n\tUnexpected Token : '$expect[$i]' FOUND";

               # If quit is enabled quit the test case on failure
               if ($quit == 1) { last; }
            }
          } # End if ($found)
        } # End for ($i)
      } # End if ($compare == "=>") else
    } # End if (defined $compare)

    if ($result == 1) { print "\n\tPASSED"; }
  } # End foreach ($testcase)

  print "\nLoop: $loopcnt Completed";
} # End for ($loopcnt)

# Close down CLI session
$mycli->print('logout');

# Log counters and summary
print logFH "\n\n*** Execution Summary ***".
            "\nTotal Loops                  : ". ($loopcnt - 1).
            "\nTotal Commands Issued        : $totalcmds".
            "\nTotal Warnings               : $totalwarns".
            "\nTotal Response Comparisons   : $totalcmprs".
            "\n  Passed                     : ". ($totalcmprs - $failedcmprs).
            "\n  Failed                     : $failedcmprs";

close logFH;
print "\nCLI Test Engine script completed.\n\n";
#### Script is completed ####

##############################################################################
#### Subroutines ####

##############################################################################
#        NAME : PrintHelp()
# DESCRIPTION : Prints help menu
#       INPUT : None
#      OUTPUT : None
#     RETURNS : 0
#    CAUTIONS : Terminates script
#     HISTORY : 1.0 - BLC - Intial Version
##############################################################################
sub PrintHelp()
{
  print "\n\nOptions Menu:".
        "\nArgument             Default Value           Purpose".
        "\n-i = IP Address      - 127.0.0.1             CLI IP address".
        "\n-p = Port            - 23                    CLI Port".
        "\n-l = Loops           - 1".
        "\n-c = Config file     - cli_sanity.cfg".
        "\n-u = Login name      - 'root'                CLI login name".
        "\n-d = Debug flag      - Off                   Print execution flow to console".
        "\n-q = Quit flag       - Off                   Quit testcase on first failure".
        "\n-s = Strict flag     - Off                   Does not remove leading (trailing) whitespace and <CR>".
        "                                               from comparisons".
        "\n-h = Options Menu    - Off                   Show this menu, optional '?'".
        "\n\nNotes:".
        "\n * Arguments: '-c', '-d' and '-h' are NOT supported in the config file".
        "\n * Arguments: 'i', 'p', 'l', 'c', and 'u' are currently case ".
        "\n   sensitive when used on the command line.".
        "\n * Command line has the highest precedence".
        "\n * Login name will also be used as the password".
        "\n * <cfgfile>.log is the engine log file.".
        "\n * cli_session.log is the readable form of the CLI transactions.".
        "\n * network_dump.log is the complete dump of the network traffic.  ".
        "\n   Use Debug (-d) flag to generate.".
        "\n * 'Net-Telnet' package must be installed on Linux systems.  Windows".
        "\n   systems will have the module installed if it is not present.".
        "\n * Demo version of script must uncomment PPM portion in CheckForModules().".
        "\n\nKnown Issues:".
        "\n1.  While using strict mode (-s) leading whitespace and <CR> may be a probem, the ".
        "\n    comparison operation will fail if the config file does not perfectly match the CLI output for each".
        "\n    actual character of whitespace - \'\\w \\t \\n \\r \\f\'.".
        "\n2.  PERL specific characters are not being found in expected tokens.".
	      "\n*** For complete validation the output in <cfg_file>.log and cli_session.log must".
          "\n    be visually examined. ***\n\n";
  exit 1;
} # End PrintHelp()

##############################################################################
#        NAME : ValidateIP()
# DESCRIPTION : Validates the IP address
#       INPUT : string containing ip address
#      OUTPUT : Print message to console if argument is not a valid ip address
#     RETURNS : 0 - Not a valid IP address
#               1 - Valid IP address
#    CAUTIONS : None
#     HISTORY : 1.0 - BLC - Intial Version
##############################################################################
sub ValidateIP()
{
  my $string = shift;

  if ($string !~ /^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)
  {
    print "Invalid IP address: $string\n";
    return 0;
  }

  return 1;
} # End ValidateIP()


##############################################################################
#        NAME : CheckForModules
# DESCRIPTION : Checks for modules installed
#       INPUT : None
#      OUTPUT : Installs modules if not already installed
#     RETURNS : None
#    CAUTIONS : Not implemented for Unix, prints error then exits script
#               PPM not used for demo version, must be uncommented
#     HISTORY : 1.0 - BLC
##############################################################################
sub CheckForModules
{
#### Uncomment this block to use the PPM for Windows ####
#  if ($^O =~ /^MSWin/i)
#  {
#    my (@modules, $module, @out);
#    @modules = qw( Net-Telnet );                # Add more modules into this array
#    $ENV{'HTTP_proxy'}="<Your proxy goes here>";
#    foreach $module (@modules)
#    {
#      my $realmod = $module;
#      $realmod =~ s/\-/::/;
#      @out = eval "use $realmod; 1";
#      if (!@out) # if module does not exist eval returns ''.  if it does, '1'
#      {
#        system("ppm install $module");
#      } # end else @out
#    } # end foreach @modules
#  }
#  else # Unix
#  {
    if (not eval "use Net::Telnet; 1")
    {
      print "\nINIT ERROR: Did not find \'Net::Telnet\' module installed";

      if (-x "/testing/utils/go")
      {
        print "\n\t Installing the \'Net::Telnet\' module";
        system ("go load NET::TELNET");
      }
      else
      {
        print "\n\tPlease install the Net::Telnet module";
        exit 1;
      }
    }
#  }
#### End Uncomment ####
} # End CheckForModules
