/********************************************************************************
* DIALOGIC CONFIDENTIAL      
* Copyright (C) 2008 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code ("Material") are owned by Dialogic Corporation or its 
* suppliers or licensors.  Title to the Material remains with Dialogic 
* Corporation or its suppliers and licensors.  The Material contains trade
* secrets and proprietary and confidential information of Dialogic or its 
* suppliers and licensors.  The Material is protected by worldwide copyright
* and trade secret laws and treaty provisions.  No part of the Material may be
* used, copied, reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way without Dialogic's prior express written
* permission.
* 
* No license under any patent, copyright, trade secret or other intellectual 
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise.  Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
****************************************************************************** */

// RTCP Parser API Header
#ifdef _WIN32
#include <windows.h>
#endif

#include "rtcpparser.h"
/***************************************************************************
*        NAME: 	GetBits8
* DESCRIPTION: 	Return the number of bits required from an 8 Bit Integer
*	       	its header
*      INPUTS: 	in	- 8 bit integer
*	       	loc	- Start location from which the bits should be taken		 
*     OUTPUTS: 	num	- Number of bits required
*     RETURNS: 	Resultant 8 bit integer containing the bits required
*    CAUTIONS: 	none
****************************************************************************/
UInt8 GetBits8(UInt8 in, UInt8 loc, UInt8 num)
{
	UInt8 a1 	= 0;
	UInt8 a2 	= 0;
	UInt8 temp 	= 0;
		
	temp = in;
	a1 = temp << (loc-1);
	a2 = a1 >> (8-num);
	return a2;

}
/***************************************************************************
*        NAME: 	GetBits16
* DESCRIPTION: 	Return the number of bits required from an 16 Bit Integer
*		its header
*      INPUTS: 	in	- 16 bit integer
*		loc	- Start location from which the bits should be taken		 
*     OUTPUTS: 	num	- Number of bits required
*     RETURNS: 	Resultant 16 bit integer containing the bits required
*    CAUTIONS: 	none
***************************************************************************/
UInt16 GetBits16(UInt16 in, UInt8 loc, UInt8 num)
{
	UInt16 a1 	= 0;
	UInt16 a2 	= 0;
	UInt16 temp 	= 0;
		
	temp = in;
	a1 = temp << (loc-1);
	a2 = a1 >> (16-num);
	return a2;

}

/****************************************************************************
*        NAME: 	GetBits32
* DESCRIPTION: 	Return the number of bits required from an 32 Bit Integer
*		its header
*      INPUTS: 	in	- 32 bit integer
*		loc	- Start location from which the bits should be taken		 
*     OUTPUTS: 	num	- Number of bits required
*     RETURNS: 	Resultant 32 bit integer containing the bits required
*    CAUTIONS: 	none
*****************************************************************************/
UInt32 GetBits32(UInt32 in, UInt8 loc, UInt8 num)
{
	UInt32 a1 	= 0;
	UInt32 a2 	= 0;
	UInt32 temp 	= 0;
		
	temp = in;
	a1 = temp << (loc-1);
	a2 = a1 >> (32-num);
	return a2;
}

/****************************************************************************
*        NAME: 	Convert8bitsToDouble
* DESCRIPTION: 	Return the value in the right numeric format
*      INPUTS: 	in	- 8 bit integer
*				issigned - 1 if the format is signed, 0 if not
*				frac - Fractional portion of the numeric format
*     OUTPUTS: 	None
*     RETURNS: 	Required value in the requested format
*    CAUTIONS: 	Formats supported 8:8, 7:1
*****************************************************************************/
double Convert8bitsToDouble(UInt8 in, UInt8 issigned, UInt8 frac)
{
	if(0 == issigned)
	{
		if(8 == frac)
		{
			return (in / 256.0);
		}
		
		if(1 == frac)
		{
			return (in / 2.0);
		}
	}
	return 0;
}

/****************************************************************************
*        NAME: 	Convert16bitsToDouble
* DESCRIPTION: 	Return the value in the right numeric format
*      INPUTS: 	in	- 16 bit integer
*				issigned - 1 if the format is signed, 0 if not
*				frac - Fractional portion of the numeric format
*     OUTPUTS: 	None
*     RETURNS: 	Required value in the requested format
*    CAUTIONS: 	Formats supported S11:4, 0:16
*****************************************************************************/
double Convert16bitsToDouble(UInt16 in, UInt8 issigned, UInt8 frac)
{
	if(0 == issigned)
	{
		if(16 == frac)
		{
			return (in / 65536.0);
		}

		if(8 == frac)
		{
			return (in / 256.0);
		}

	}
	else
	{
		if(4 == frac)
		{
			return ( ((signed short)in) / 16.0);
		}
	}
	return 0;
}


/*****************************************************************************
*        NAME: 	rtcp_TestEndian
* DESCRIPTION: 	Find the Endianness of the system
*      INPUTS: 	None
*     OUTPUTS: 	result		- will be of the type LITTLE_ENDIAN/BIG_ENDIAN
*     RETURNS: 	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS: 	none
******************************************************************************/
eRTCP_Ret_Code rtcp_TestEndian(UInt8* result)
{

	UInt16 temp = 0x1234;
	UInt8* test	= (UInt8 *)&temp;

	if ( 2 != sizeof(UInt16)) 
		return RTCP_FAILURE;

	if (*test == 0x12) 
	{
		*result = LITTLE_END; // big endian
		return RTCP_SUCCESS;
	}

	if (*test == 0x34) 
	{
		*result = BIG_END; // little endian
		return RTCP_SUCCESS;
	}
	
	return RTCP_FAILURE;
}

/******************************************************************
*        NAME:	rtcp_NetToHost16
* DESCRIPTION: 	Conver the 16 bit digit from Network to Host Format
*	       	its header
*      INPUTS: 	in - 16 bit integer to be converted
*     OUTPUTS: 	None
*     RETURNS: 	Converted 16 bit integer
*    CAUTIONS: 	none
******************************************************************/
UInt16 rtcp_NetToHost16(UInt16 in)
{
	UInt8 result = 0;

	if(RTCP_SUCCESS == rtcp_TestEndian(&result))
	{
		if (BIG_END == result)
			return ( (in << 8) | (in >> 8) );
		
		if(LITTLE_END == result)
			return in;
	}
	return 0;
}

/******************************************************************
*        NAME: 	rtcp_NetToHost32
* DESCRIPTION: 	Conver the 32 bit digit from Network to Host Format
*		its header
*      INPUTS: 	in - 32 bit integer to be converted
*     OUTPUTS: 	None
*     RETURNS: 	Converted 32 bit integer
*    CAUTIONS: 	none
******************************************************************/
UInt32 rtcp_NetToHost32(UInt32 in)
{
	UInt32* out = &in;

	UInt16 temp1 = *((UInt16 *)out );
	UInt16 temp2 = *(((UInt16 *)out) + 1);

	*(((UInt16 *)out) + 1) = rtcp_NetToHost16(temp1);
	*((UInt16 *)out) = rtcp_NetToHost16(temp2);

	return (*out);
}

/******************************************************************
*        NAME: 	rtcp_GetPktLenInBytes
* DESCRIPTION:	Returns the length of the RTCP Packet from
*		its header
*      INPUTS: 	rtcp - Pointer to the start of the RTCP Packet 
*     OUTPUTS: 	None
*     RETURNS: 	Length of the RTCP packet as Number of Bytes
*    CAUTIONS: 	Unit of Length should be noted
******************************************************************/
UInt16 rtcp_GetPktLenInBytes(RTCP_Pkt_Header *rtcp)
{
	// Local variables
	UInt16 rtcplen = 0;
	
	// Validity Check
	if(NULL == rtcp)
	{
		return 0;
	}
    
	// Packet Length in always expressed as number of 32 bit units
	// we need to add 1 to get the total length
	rtcplen = rtcp_NetToHost16(rtcp->Length);
	return (4 * (rtcplen + 1));	 
}

/*********************************************************************
*        NAME: 	rtcp_GetNextPktHeaderPtr
* DESCRIPTION: 	Returns a pointer to the start of the next RTCP Packet
*      INPUTS: 	rtcp - Pointer to the start of the current RTCP Packet
*     OUTPUTS: 	None
*     RETURNS: 	Pointer to start of next RTCP Packet
*    CAUTIONS: 	None
*********************************************************************/
RTCP_Pkt_Header* rtcp_GetNextPktHeaderPtr(RTCP_Pkt_Header *rtcp)
{
   // Validity Check
   RTCP_Pkt_Header *temp = NULL;
   
   if( NULL != rtcp)
   {	    
	   	rtcp = (RTCP_Pkt_Header *)( (UInt8 *)rtcp + rtcp_GetPktLenInBytes(rtcp));
		if( (rtcp->PacketType >= RTCP_PKTTYPE_SR) && (rtcp->PacketType <= RTCP_PKTTYPE_XR) ) 
		{
			return rtcp;
		}
   }
   
   return NULL;
}

/****************************************************************************
*        NAME:	rtcp_ValidatePkt
* DESCRIPTION:	Retrieve the RTCP Packet Header with all the values
*      INPUTS:	rtcp		- Pointer to the raw RTCP data
*     OUTPUTS:	hdr		- Populated Packet Header
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
****************************************************************************/
eRTCP_Ret_Code rtcp_GetPktHeader(RTCP_Pkt_Header* rtcp, RTCP_Pkt_Header* hdr)
{
	RTCP_Pkt_Header* temp;
	UInt8 pkttype = 0;
	
	if( NULL == rtcp)
	{		
		return RTCP_FAILURE;
	}
		
	temp = (RTCP_Pkt_Header *)rtcp;
	if( (temp->PacketType < RTCP_PKTTYPE_SR) || (temp->PacketType > RTCP_PKTTYPE_XR) ) 
	{
		return RTCP_FAILURE;
	}

	memset(hdr, 0, sizeof(RTCP_Pkt_Header));


	hdr->Version		= GetBits8(*((UInt8 *)temp),1,2); 
	hdr->Padding		= GetBits8(*((UInt8 *)temp),3,1); 
	hdr->Count			= GetBits8(*((UInt8 *)temp),4,5); 
	hdr->PacketType		= *((UInt8 *)temp + 1); 
	hdr->Length			= rtcp_GetPktLenInBytes(temp); 
	
	return RTCP_SUCCESS;
}

/*******************************************************************************
*        NAME:	rtcp_ValidatePkt
* DESCRIPTION:	Validates the entire compound RTCP packet, one packet
*		at a time packetReturns the length of the RTCP Packet from
*		its header
*      INPUTS:	rtcp		- Pointer to the raw RTCP data
*		length		- Length of raw data in bytes
*     OUTPUTS:	result		- stores the result of the validity test. Can be
*				  RTCP_PKT_VALID or RTCP_PKT_INVALID
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
*******************************************************************************/
eRTCP_Ret_Code rtcp_ValidatePkt	(UInt8* rtcp, UInt32 length, UInt8* result)
{
	UInt8				ver		= 0;
	unsigned short		Len		= 0;
	UInt16				temp	= 0;
	RTCP_Pkt_Header*	start	= NULL;		// Common RTCP Report header
	RTCP_Pkt_Header*	end		= NULL;		// End of compound RTCP packet
	RTCP_Pkt_Header*	tmp		= NULL;
	
	// Validity Check
	if( (NULL == rtcp) || !(length > 0))
	{
		*result = RTCP_PKT_INVALID;
		return RTCP_FAILURE;
	}

	start = (RTCP_Pkt_Header *) rtcp;
	end = (RTCP_Pkt_Header *)((UInt8 *)start + length); 
	
	/*
	1. Payload type field of the first RTCP packet in a compound
	   packet must be equal to SR or RR
	2. Padding bit must be zero for the first RTCP Packet
	3. Check the version
	*/
	temp = rtcp_NetToHost16(*((UInt16 *)rtcp));
	
	if( (temp & RTCP_VALID_MASK) != RTCP_VALID_VALUE) 
	{		
		*result = RTCP_PKT_INVALID;
		return RTCP_FAILURE;
	}
      
	for(tmp = start; tmp < end; tmp = rtcp_GetNextPktHeaderPtr(tmp))
	{
		if( NULL != tmp )
		{
			ver = GetBits8(*((UInt8 *)tmp),1,2); 
			if(RTCP_VERSION == ver)
				*result = RTCP_PKT_VALID;
			else
			{
				*result = RTCP_PKT_INVALID;
				break;
			}
		}
		else
			break;
	}
	return RTCP_SUCCESS;
}

/***************************************************************************
*        NAME:	rtcp_GetNumPkts
* DESCRIPTION:	Calculates the number of individual RTCP Packets are
*		present in an compound packet.
*      INPUTS:	rtcp		- Pointer to the raw RTCP data
*		length		- Length of raw data in bytes
*     OUTPUTS:	numpkts		- Number of packets present
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
****************************************************************************/
eRTCP_Ret_Code rtcp_GetNumPkts(UInt8* rtcp, UInt32 length, UInt8* numpkts)
{
	// Local Variables
	RTCP_Pkt_Header*	curhdr 	= NULL;
	int 			rval 	= 0;
	*numpkts = 0;
	
	// Validity Check
	if( (NULL == rtcp) || (length <= 0) )
	{
		*numpkts = 0;
		return RTCP_FAILURE;
	}
	
	// Count the number of reports by checking for the Report Type
	for(curhdr = (RTCP_Pkt_Header *)rtcp; 
		(length >= 0), (NULL != curhdr); 
		curhdr = rtcp_GetNextPktHeaderPtr(curhdr))
	{
		if( (curhdr->PacketType >= RTCP_PKTTYPE_SR) && (curhdr->PacketType <= RTCP_PKTTYPE_XR) )
		{
			*numpkts = (*numpkts) + 1;
			length = length - rtcp_GetPktLenInBytes(curhdr);			
		}
		else // if packet type is not valid, the Report is assumed to be corrupt
		{
			*numpkts = 0;
			return RTCP_FAILURE;
		}		    		
	}
    return RTCP_SUCCESS;
}

/**********************************************************************************
*        NAME:	rtcp_GetPktsPtr
* DESCRIPTION:	Store and return an array of pointers to start of 
*		each RTCP Packet
*      INPUTS:	rtcp		- Pointer to the raw RTCP data
*		length		- Length of raw data in bytes
*		arrlen		- Number of elements of the type 
*				  RTCP_Pkt_Header that can be stored in the array
*     OUTPUTS:	hdrarr		- Array containing the pointers to the RTCP Packets
*				  stored as type RTCP_Pkt_Header
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	The memory for hdrarr should be accurate and must be set by the
*		application. The arrlen should correctly reflect the number of 
*		elements that can be stored
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetPktsPtr(UInt8* rtcp, UInt32 length, RTCP_Pkt_Header **hdrarr, UInt16 arrlen)
{
    	RTCP_Pkt_Header* 	curhdr = NULL;
   	UInt16			idx	= 0;
	UInt8			numpkts = 0;
	UInt8			rval	= 0;

	// Validity Check
	if( (NULL == rtcp) || (NULL == hdrarr) || (length == 0) || (arrlen <= 0))
	{
		return RTCP_FAILURE;
	}

	// Make sure the storage sent in by the application and storage
	// expected match
	if(RTCP_SUCCESS == rtcp_GetNumPkts(rtcp, length, &numpkts))
	{
		if(arrlen != numpkts)
			return RTCP_FAILURE;
	}
	else
		RTCP_FAILURE;

	// Initialize the variables
	memset(hdrarr, 0, arrlen);

   	// Get the packets after checking for the packet type
    	for( 	curhdr = (RTCP_Pkt_Header *)rtcp, idx = 0; 
         	length >=0, (NULL != curhdr); 
		curhdr = rtcp_GetNextPktHeaderPtr(curhdr),
		idx ++)
	{
		if( (curhdr->PacketType >= RTCP_PKTTYPE_SR) && (curhdr->PacketType <= RTCP_PKTTYPE_XR) )
		{
			hdrarr[idx] = (RTCP_Pkt_Header *) curhdr;
			length = length - rtcp_GetPktLenInBytes(curhdr);
            rval = RTCP_SUCCESS;
		}
		else // if packet type is not valid, the packet is assumed to be corrupt
		{
			rval = RTCP_FAILURE;
			break;
		}		    		
	}
    
   return rval;	  
}

/********************************************************************************
*        NAME:	rtcp_GetNumRptBlks
* DESCRIPTION:	Get the number of Report Blocks present in a Sender or Receiver 
*		Report RTCP Packet
*      INPUTS:	rtcp		- Pointer of the type RTCP_Pkt_Header that points 
*				  to either SR or RR RTCP packet
*     OUTPUTS:	numblks		- Number of Report Blocks found
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
********************************************************************************/
eRTCP_Ret_Code rtcp_GetNumRptBlks(RTCP_Pkt_Header* rtcp, UInt8* numblks)
{
	// Validity check. Report Blocks exist for only RTCP packets of the type SR/RR
	if( (NULL == rtcp) || (NULL == numblks) )
	{
		return RTCP_FAILURE;	
	}
	
	if( (rtcp->PacketType != RTCP_PKTTYPE_SR) && (rtcp->PacketType != RTCP_PKTTYPE_RR) )
    	{
		*numblks = 0;
		return RTCP_FAILURE;	
	}

	*numblks = GetBits8(*((UInt8 *)rtcp), 4, 5);
	return RTCP_SUCCESS;
}

/***************************************************************************
*        NAME:	rtcp_GetRptBlkSize
* DESCRIPTION:	Get the size of general Report Block in SR or RR RTCP Packet
*		ss it appears on the wire
*      INPUTS:	None 
*     OUTPUTS:	None
*     RETURNS:	size of the Report Block
*    CAUTIONS:	None
***************************************************************************/
UInt8 rtcp_GetRptBlkSize(void)
{
	return (sizeof(SRRR_ReportBlock) - 1);
}

/****************************************************************
*        NAME:	rtcp_GetNextRptBlockPtr
* DESCRIPTION:	Get the pointer to start of the next Report Block
*      INPUTS:	blk - Pointer to the current Report Block
*     OUTPUTS:	None
*     RETURNS:	Pointer to the next Report Block
*    CAUTIONS:	None
****************************************************************/
SRRR_ReportBlock* rtcp_GetNextRptBlockPtr(SRRR_ReportBlock* blk)
{ 
	// Validity Check
	if(NULL != blk)
	{
		return (SRRR_ReportBlock *)((UInt32 *)blk + rtcp_GetRptBlkSize());
	}
	else
		return NULL;
}

/*********************************************************************************
*        NAME:	rtcp_GetProfExtLen
* DESCRIPTION:	Determine if Profile Extension exists in the SR or RR RTCP Packet 
*      INPUTS:	rpthdr		- Pointer of the type RTCP_Pkt_Header that points 
*				  to either SR or RR RTCP packet
*     OUTPUTS:	length		- Length of Profile Extension in Bytes, zero if it
*				  absent
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
*********************************************************************************/
eRTCP_Ret_Code rtcp_GetProfExtLen(RTCP_Pkt_Header* rpthdr, UInt16* length)
{
	// Local Variables
	UInt16					offset		= 0;
	UInt8					rptblksize	= 0;
	UInt8					numblks		= 0;
	RTCP_Sender_Report*			sr		= NULL;
	RTCP_Receiver_Report*			rr		= NULL;
	RTCP_Pkt_Header				temp;
	UInt16					pktlen		= 0;
	UInt8					numpad		= 0;

	// Validity Check
	if( (NULL == rpthdr) || (NULL == length) )
	{
		return RTCP_FAILURE;
	}

	// Retrieve the populated Header, this is needed for some validation and padding check
	if(RTCP_SUCCESS != rtcp_GetPktHeader(rpthdr, &temp))
	{
		*length = 0;
		return RTCP_FAILURE;
	}

	if( (temp.PacketType != RTCP_PKTTYPE_SR) && (temp.PacketType != RTCP_PKTTYPE_RR) ) 
	{
		*length = 0;
		return RTCP_FAILURE;
	}

	// Calcualate the offset until all Report Blocks are accounted for
	if(temp.PacketType == RTCP_PKTTYPE_SR)
	{
		sr = (RTCP_Sender_Report *)rpthdr;
		
		offset = sizeof(UInt32) +	// Accounts for RTCP_Pkt_Header
				 sizeof(sr->Sender_SSRC) +
				 sizeof(sr->NTPTimeStamp_Intgr) +
				 sizeof(sr->NTPTimeStamp_Frac) +
				 sizeof(sr->RTPTimeStamp) +
				 sizeof(sr->PacketCount) +
				 sizeof(sr->OctetCount);

		if(RTCP_SUCCESS == rtcp_GetNumRptBlks(rpthdr, &numblks))
		{			
			offset = offset + (numblks * rtcp_GetRptBlkSize());		
			
		}

	}
	else
		if(rpthdr->PacketType == RTCP_PKTTYPE_RR)
		{
			rr = (RTCP_Receiver_Report *)rpthdr;

			offset = sizeof(UInt32) + // Accounts for RTCP_Pkt_Header
					 sizeof(rr->Sender_SSRC);
	
			if(RTCP_SUCCESS == rtcp_GetNumRptBlks(rpthdr, &numblks))
			{
				offset = offset + (numblks * rtcp_GetRptBlkSize());
			}			
		}


	// Check for Padding
    	if(1 == temp.Padding)
	{
		pktlen = rtcp_GetPktLenInBytes(rpthdr);
		numpad = *((UInt8 *)rpthdr + (pktlen - 1));
	}
		
	// Get the Length of remainder of the RTCP Packet which 
	// would actually be Profile Extension. Subtract the padding
	// bytes that is not part of the Profile Extension
	*length = rtcp_GetPktLenInBytes(rpthdr) - offset - numpad;
	
    return RTCP_SUCCESS;
}

/****************************************************************************************
*        NAME:	rtcp_GetProfExt
* DESCRIPTION:	Get the actual Profile Extension if it exists in the SR or RR RTCP Packet 
*      INPUTS:	rpthdr		- Pointer of the type RTCP_Pkt_Header that points 
*				  to either SR or RR RTCP packet
*		length		- Length of storage sent down
*     OUTPUTS:	profext		- Profile Extension data copied
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	Length of the array specified should match the required storage
****************************************************************************************/
eRTCP_Ret_Code rtcp_GetProfExt(RTCP_Pkt_Header* rpthdr, UInt8* profext, int length)
{
	UInt8					*curptr		= NULL;
	UInt16					offset		= 0;
	UInt16					proflen		= 0;
	UInt8					rptblksize	= 0;
	UInt8					numblks		= 0;
	RTCP_Sender_Report*			sr		= NULL;
	RTCP_Receiver_Report*			rr		= NULL;
	RTCP_Pkt_Header				temp;
	UInt16					pktlen		= 0;
	UInt8					numpad		= 0;

	// Validity Check
	if( (NULL == rpthdr) || (NULL == profext) )
	{
		return RTCP_FAILURE;
	}
	
	// Retrieve the populated Header, this is needed for some validation and padding check
	if(RTCP_SUCCESS != rtcp_GetPktHeader(rpthdr, &temp))
	{
		return RTCP_FAILURE;
	}

	if( (temp.PacketType != RTCP_PKTTYPE_SR) && (temp.PacketType != RTCP_PKTTYPE_RR) )  
	{
		return RTCP_FAILURE;
	}

	// Calculate the offset to which we have to move
	if(temp.PacketType == RTCP_PKTTYPE_SR)
	{
		sr = (RTCP_Sender_Report *)rpthdr;
		
		offset = sizeof(UInt32) +	// Accounts for RTCP_Pkt_Header
				 sizeof(sr->Sender_SSRC) +
				 sizeof(sr->NTPTimeStamp_Intgr) +
				 sizeof(sr->NTPTimeStamp_Frac) +
				 sizeof(sr->RTPTimeStamp) +
				 sizeof(sr->PacketCount) +
				 sizeof(sr->OctetCount);

		if(RTCP_SUCCESS == rtcp_GetNumRptBlks(rpthdr, &numblks))
		{			
			offset = offset + (numblks * rtcp_GetRptBlkSize());		
			
		}
	}
	else
		if(temp.PacketType == RTCP_PKTTYPE_RR)
		{
			rr = (RTCP_Receiver_Report *)rpthdr;

			offset = sizeof(UInt32) + // Accounts for RTCP_Pkt_Header
					 sizeof(rr->Sender_SSRC);
	
			if(RTCP_SUCCESS == rtcp_GetNumRptBlks(rpthdr, &numblks))
			{			
				offset = offset + (numblks * rtcp_GetRptBlkSize());
			}			
		}

	// Calculate the length of the Profile Extension
	
	// Check for Padding
	if(1 == temp.Padding)
	{
		pktlen = rtcp_GetPktLenInBytes(rpthdr);
		numpad = *((UInt8 *)rpthdr + (pktlen - 1));
	}
		
	// Get the Length of remainder of the RTCP Packet which 
	// would actually be Profile Extension. Subtract the padding
	// bytes that is not part of the Profile Extension
	proflen = rtcp_GetPktLenInBytes(rpthdr) - offset - numpad;
	
	// Cannot copy Profile Extension if it does not exist or
	// does not match the storage sent in
	if( ( proflen == 0) || (proflen != length) )
	{
		return RTCP_FAILURE;
	}
	else
	{
		memset(profext, 0, length);
		memcpy(profext, ((UInt8 *)rpthdr + offset), proflen);
		return RTCP_SUCCESS;
	}
}

/*********************************************************************************
*        NAME:	rtcp_GetSRRpt
* DESCRIPTION:	Retrieve the Sender Report RTCP Packet 
*      INPUTS:	rtcphdr		- Pointer to the start of the SR RTCP packet which
*				  of the type RTCP_Pkt_Header
*     OUTPUTS:	sr		- Storage for the SR Packet information which is of
*				  the type RTCP_Sender_Report
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	The memory for the SR must be allocated by the application with all
*		report blocks properly allocated
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetSRRpt(RTCP_Pkt_Header* rtcphdr, RTCP_Sender_Report* sr)
{
	// Local variables
	SRRR_ReportBlock*	currpt		= NULL;
	RTCP_Sender_Report*	srpt		= NULL;
	UInt16			proflen		= 0;
	UInt8			i		= 0;
	UInt32			temp		= 0;
	UInt32			*tempptr	= NULL;
	
	// Validity Check
	if( (NULL == rtcphdr)|| (NULL == sr))
	{
		return RTCP_FAILURE;
	}

	if( (rtcphdr->PacketType != RTCP_PKTTYPE_SR) )
	{
		return RTCP_FAILURE;
	}

	// Get the Header
	if (RTCP_FAILURE == rtcp_GetPktHeader(rtcphdr, sr->SR_Header))
	{
		return RTCP_FAILURE;
	}
	
	// Get the rest of the data
	srpt = (RTCP_Sender_Report *) rtcphdr;
	sr->Sender_SSRC			=	rtcp_NetToHost32((UInt32)(srpt->Sender_SSRC));
	sr->NTPTimeStamp_Intgr  	=	rtcp_NetToHost32((UInt32)(srpt->NTPTimeStamp_Intgr));
	sr->NTPTimeStamp_Frac		=	rtcp_NetToHost32((UInt32)(srpt->NTPTimeStamp_Frac));
	sr->RTPTimeStamp		=	rtcp_NetToHost32((UInt32)(srpt->RTPTimeStamp));
	sr->PacketCount			=	rtcp_NetToHost32((UInt32)(srpt->PacketCount));
	sr->OctetCount			=	rtcp_NetToHost32((UInt32)(srpt->OctetCount));
		
	// It is assumed that the app sent enough memory for the report blocks, so
	// only checking for the validity of the rptblk pointer
	if((sr->SR_Header->Count > 0) && (NULL != sr->rptblk))
	{
		currpt = (SRRR_ReportBlock *)&(srpt->rptblk);
	
		// Cycle through each report block and populate the values
		for(i = 0; 
		    i < sr->SR_Header->Count; 
			i++, currpt = currpt + rtcp_GetRptBlkSize())
		{
			sr->rptblk[i]->Source_SSRC	= rtcp_NetToHost32(currpt->Source_SSRC);				 
			sr->rptblk[i]->FractionLost	= currpt->FractionLost;				 
			
			tempptr = (UInt32 *)&currpt->FractionLost;
			
			// CumPktLost is an odd 3 byte value
			temp = rtcp_NetToHost32(*tempptr);
			sr->rptblk[i]->CumPktLost = temp & 0xFFFFFF;

			sr->rptblk[i]->ExtendedHighSeqNum	= rtcp_NetToHost32(*(tempptr + 1));				 			 
			sr->rptblk[i]->InterArrJitter		= rtcp_NetToHost32(*(tempptr + 2));				 				 
			sr->rptblk[i]->LSR			= rtcp_NetToHost32(*(tempptr + 3));				 						 
			sr->rptblk[i]->DLSR			= rtcp_NetToHost32(*(tempptr + 4));			 
		}		
	
	}
	
	// Get the Profile Extension if it exists
	if(RTCP_SUCCESS == rtcp_GetProfExtLen(rtcphdr, &proflen))
	{
		if( (proflen > 0) && (NULL != sr->ProfileExt))
		{
			rtcp_GetProfExt(rtcphdr, sr->ProfileExt, proflen);
		}
	
	}
	
	// Done collecting Sender Report
	return RTCP_SUCCESS;
}

/**********************************************************************************
*        NAME:	rtcp_GetRRRpt
* DESCRIPTION:	Retrieve the Receiver Report RTCP Packet 
*      INPUTS:	rtcphdr		- Pointer to the start of the RR RTCP packet which
*				  of the type RTCP_Pkt_Header
*     OUTPUTS:	rr		- Storage for the RR Packet information which is of
*				  the type RTCP_Sender_Report
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	The memory for the RR must be allocated by the application with all
*		report blocks properly allocated
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetRRRpt(RTCP_Pkt_Header* rtcphdr, RTCP_Receiver_Report* rr)
{
	// Local variables
	SRRR_ReportBlock*		currpt		= NULL;
	RTCP_Receiver_Report*		rrpt		= NULL;
	UInt16				proflen		= 0;
	UInt8				i		= 0;
	UInt32				temp		= 0;
	UInt32				*tempptr	= NULL;

	// Validity Check
	if( (NULL == rtcphdr) || (NULL == rr))
	{
		return RTCP_FAILURE;
	}

	if( (rtcphdr->PacketType != RTCP_PKTTYPE_RR) )
	{
		return RTCP_FAILURE;
	}

	// Get the Header
	if (RTCP_FAILURE == rtcp_GetPktHeader(rtcphdr, rr->RR_Header))
	{
		return RTCP_FAILURE;
	}
	
	// Get the rest of the data
	rrpt = (RTCP_Receiver_Report *) rtcphdr;
	rr->Sender_SSRC			=	rtcp_NetToHost32((UInt32)(rrpt->Sender_SSRC));

	
	// It is assumed that the app sent enough memory for the report blocks, so
	// only checking for the validity of the rptblk pointer
	if((rr->RR_Header->Count > 0) && (NULL != rr->rptblk))
	{
		currpt = (SRRR_ReportBlock *)&(rrpt->rptblk);

		// Cycle through each report block and populate the values
		for(i = 0; 
		    i < rr->RR_Header->Count; 
			i++, currpt = currpt + rtcp_GetRptBlkSize())
		{
			rr->rptblk[i]->Source_SSRC			= rtcp_NetToHost32(currpt->Source_SSRC);				 
			rr->rptblk[i]->FractionLost			= currpt->FractionLost;				 
			
			
			tempptr = (UInt32 *)&currpt->FractionLost;
			
			// CumPktLost is an odd 3 byte value
			temp = rtcp_NetToHost32(*tempptr);
			rr->rptblk[i]->CumPktLost = temp & 0xFFFFFF;

			rr->rptblk[i]->ExtendedHighSeqNum	= rtcp_NetToHost32(*(tempptr + 1));				 			 
			rr->rptblk[i]->InterArrJitter		= rtcp_NetToHost32(*(tempptr + 2));				 				 
			rr->rptblk[i]->LSR			= rtcp_NetToHost32(*(tempptr + 3));				 						 
			rr->rptblk[i]->DLSR			= rtcp_NetToHost32(*(tempptr + 4));				 
		}		
	
	}
	
	// Get the Profile Extension if it exists
	if(RTCP_SUCCESS == rtcp_GetProfExtLen(rtcphdr, &proflen))
	{
		if( (proflen > 0) && (NULL != rr->ProfileExt))
		{
			rtcp_GetProfExt(rtcphdr, rr->ProfileExt, proflen);
		}
	
	}
	
	// Done collecting Sender Report
	return RTCP_SUCCESS;
	
}

/***********************************************************************************
*        NAME:	rtcp_GetNumSDESChunks
* DESCRIPTION:	Retrieve the number of chunks present in the SDES RTCP Packet 
*      INPUTS:	sdes		- Pointer to the start of the SDES RTCP packet which
*				  of the type RTCP_SDES
*     OUTPUTS:	numchunks	- Number of chunks found
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
************************************************************************************/
eRTCP_Ret_Code rtcp_GetNumSDESChunks(RTCP_Pkt_Header* hdr, UInt8* numchunks)
{
	if( (NULL == hdr) && (NULL == numchunks))
	{
		return RTCP_FAILURE;
	}

	if( hdr->PacketType != RTCP_PKTTYPE_SDES)
	{
		*numchunks = 0;
		return RTCP_FAILURE;
	}
	
	*numchunks = GetBits8(*((UInt8 *)hdr), 4, 5);
	return RTCP_SUCCESS;	
}

/********************************************************************************
*        NAME:	rtcp_GetChunkLen
* DESCRIPTION:	Retrieve the length of SDES chunk 
*      INPUTS:	chunk		- Pointer to the start of the SDES chunk of the 
*				  type SDES_Chunk
*     OUTPUTS:	chunklen	- Length of the chunk
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetChunkLen(SDES_Chunk* chunk, UInt16* chunklen)
{
	// Local Variables
	SDES_Chunk* 	temp		= NULL;
	UInt16		offset		= 0;
	UInt16		curritemlen	= 0;
	SDES_Item*	curritem	= NULL;
	UInt8		rem		= 0;

	// Validity Check
	if((NULL == chunk) || (NULL == chunklen ))
		return RTCP_FAILURE;
	
	// Move past the fisrt element of the chunk
	offset = (UInt16) sizeof(temp->SSRC);
	
	// Calculate the length by adding the length of individual
	// items in the Chunk
	for( curritem = (SDES_Item *)&(chunk->ItemList);;)
	{
		if((curritem->Type > SDES_ITEM_END) && (curritem->Type <= SDES_ITEM_PRIV))
		{
			
			curritemlen = (UInt16)sizeof(curritem->Type) + 
						  (UInt16)sizeof(curritem->Length) +  
						  (UInt16)curritem->Length;
			
			curritem = (SDES_Item *)( (UInt8 *)curritem + curritemlen);
			offset = offset + curritemlen;

		}
		else
			if(curritem->Type == SDES_ITEM_END) // End item is still valid, hence the number is incremented
			{
				// Check if there is padding to align to 32 bit boundary
				rem = (offset%4);
				*chunklen = offset + (4 - rem);
				return RTCP_SUCCESS;
			}
			else
			{
				*chunklen = 0;
				return RTCP_FAILURE;
			}
	}
		
	*chunklen = offset;
	return RTCP_SUCCESS;
}

/********************************************************************************
*        NAME:	rtcp_GetNextChunkPtr
* DESCRIPTION:	Retrieve the next SDES chunk 
*      INPUTS:	chunk		- Pointer to the start of the current SDES chunk 
*							  of the type SDES_Chunk
*     OUTPUTS:	None
*     RETURNS:	SDES_Chunk	- Pointer to the start of the next SDES Chunk
*    CAUTIONS:	None
**********************************************************************************/
SDES_Chunk* rtcp_GetNextChunkPtr(SDES_Chunk* chunk)
{
	SDES_Chunk* temp		= NULL;
	UInt16		chunklen	= 0;
	UInt16		offset		= 0;
	UInt16		curritemlen	= 0;
	SDES_Item*	curritem	= NULL;
	

	if(NULL == chunk)
		return NULL;
	
	if(RTCP_SUCCESS == rtcp_GetChunkLen(chunk, &chunklen))
	{
		temp = (SDES_Chunk *)((UInt8 *)chunk + chunklen);
	}
	
	return temp;	
}


/************************************************************************************
*        NAME:	rtcp_GetNextItemPtr
* DESCRIPTION:	Retrieve the next SDES Item
*      INPUTS:	item		- Pointer to the start of the current SDES Item of the 
*				  type SDES_Item
*     OUTPUTS:	None
*     RETURNS:	SDES_Item	- Pointer to the start of the next SDES Item
*    CAUTIONS:	None
*************************************************************************************/
SDES_Item* rtcp_GetNextItemPtr(SDES_Item *item)
{
	SDES_Item*	temp = NULL;
	UInt16		offset = 0;

	if(NULL == item)
		return (NULL);
	
	temp = item;
	offset = (UInt16)sizeof(temp->Type) + 
			 (UInt16)sizeof(temp->Length) +  
			 (UInt16)temp->Length;

	temp = (SDES_Item *) ((UInt8 *)temp + offset);

	// Uncomment the check if the value of SDES_ITEM_END becomes non-zero
	// If zero produces a meaningless warning
	if( /* (temp->Type >= SDES_ITEM_END) && */ (temp->Type <= SDES_ITEM_PRIV))
	{
		return temp;
	}
	else
		return NULL;
	
}

/**********************************************************************************
*        NAME:	rtcp_GetNumItems
* DESCRIPTION:	Retrieve the total number SDES Items in a SDES chunk 
*      INPUTS:	chunk		- Pointer to the start of the SDES chunk of the type
*				  SDES_Chunk
*     OUTPUTS:	numitems	- Number of the items present in the chunk
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
***********************************************************************************/
eRTCP_Ret_Code rtcp_GetNumItems(SDES_Chunk* chunk, UInt8* numitems)
{
	SDES_Item* curritem = NULL;

	if((NULL == chunk) || (NULL == numitems))
	{
		return RTCP_FAILURE;
	}

	for( curritem = (SDES_Item *)&(chunk->ItemList);;
		 curritem = rtcp_GetNextItemPtr(curritem))
	{
		if((curritem->Type > SDES_ITEM_END) && (curritem->Type <= SDES_ITEM_PRIV))
		{
			*numitems = *numitems + 1;
		}
		else
			if(curritem->Type == SDES_ITEM_END)
			{
				return RTCP_SUCCESS;

			}
		else
		{
			*numitems = 0;
			return RTCP_FAILURE;
		}
	}
	return RTCP_SUCCESS;

}


/***********************************************************************************
*        NAME:	rtcp_GetSDES
* DESCRIPTION:	Retrieve the SDES RTCP Packet
*      INPUTS:	hdr		- Pointer to the start of the SDES Packet of the type
*				  RTCP_Pkt_Header
*     OUTPUTS:	sdes		- Storage for the SDES RTCP Packet
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	Application needs to take care of the memory allocation for the
*		entire SDES Packet including all its Chunks and Items
*************************************************************************************/
eRTCP_Ret_Code rtcp_GetSDES(RTCP_Pkt_Header* hdr, RTCP_SDES* sdes)
{
	// Local variables
	RTCP_Pkt_Header* 	curhdr		= NULL;
	RTCP_SDES*		pkt		= NULL;
	SDES_Chunk*		curchnk		= NULL;
	SDES_Item*		fromitem	= NULL;
	SDES_Item*		toitem		= NULL;
	UInt8			numchunks	= 0;
	UInt8			idx		= 0;
	UInt8			i		= 0;
	UInt8			j		= 0;
	UInt8			numitems	= 0;
	UInt16			chnklen		= 0;
	UInt16			currlen		= 0;
	UInt16			curritemlen 	= 0;
	
	// Validity Check
	if( (NULL == hdr) || (NULL == sdes) )
	{
		return RTCP_FAILURE;
	}
	
	if(hdr->PacketType != RTCP_PKTTYPE_SDES)
	{
		return RTCP_FAILURE;
	}

	// Get SDES Header   	
	if( sdes->SDES_Header != NULL)
	{
		if(RTCP_SUCCESS != rtcp_GetPktHeader(hdr, sdes->SDES_Header))
		{
			return RTCP_FAILURE;
		}
    	}

	// Get the Chunks if they exist
	numchunks = (UInt8)sdes->SDES_Header->Count;
	if( (numchunks > 0) && (NULL != sdes->ChunkList) )
	{
		// Get the Chunk
		for(i = 0, curchnk = (SDES_Chunk *) (&((RTCP_SDES *)hdr)->ChunkList); 
			(NULL != curchnk), (i <  numchunks);
			i ++, curchnk = rtcp_GetNextChunkPtr(curchnk))
		{
			// Get the length of this chunk
			if( RTCP_SUCCESS != rtcp_GetChunkLen(curchnk, &chnklen) )
			{
				return RTCP_FAILURE;
			}

			if( (chnklen > 0) && (NULL != sdes->ChunkList))
			{
				// Get SSRC for this chunk
				sdes->ChunkList[i].SSRC = rtcp_NetToHost32(curchnk->SSRC);
				
				// Find the number of Items in this chunk
				if(RTCP_SUCCESS != rtcp_GetNumItems(curchnk, &numitems))
				{
					return RTCP_FAILURE;
				}

				// Get the Items if they exist
				if(numitems > 0)
				{	
					sdes->ChunkList[i].NumItems = numitems;
					
					fromitem = (SDES_Item *)&(curchnk->ItemList);
					currlen = 0;
					
					for(j=0; 
						((j< numitems) && (NULL != fromitem)); 
						j++, (fromitem = rtcp_GetNextItemPtr(fromitem)) )
					{										
						if(fromitem->Type != SDES_ITEM_END)
						{
							sdes->ChunkList[i].ItemList[j].Type = fromitem->Type;
							sdes->ChunkList[i].ItemList[j].Length = fromitem->Length;						
							memcpy(sdes->ChunkList[i].ItemList[j].Text, &fromitem->Text, fromitem->Length);		
						}
						else
							if(fromitem->Type == SDES_ITEM_END)
							{
								break;
							}
					}
				}								
			}
		}
	}
	return RTCP_SUCCESS;
}

/********************************************************************************
*        NAME:	rtcp_GetAPPDataLen
* DESCRIPTION:	Retrieve the length of the Application data in the APP RTCP Packet
*      INPUTS:	app		- Pointer to the start of the APP Packet of the type
*				  RTCP_APP
*     OUTPUTS:	appdatalen	- Length of Application Data in Bytes
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetAPPDataLen(RTCP_Pkt_Header* hdr, UInt8* appdatalen)
{	
	RTCP_APP*		temp		= NULL;
	UInt8			numpad		= 0;
	UInt16			pktlen		= 0;
	RTCP_Pkt_Header 	pktheader;


	// Validity Check
	if( (NULL == hdr) || (NULL == appdatalen) )
	{
		return RTCP_FAILURE;
	}

	// Retrieve the populated Header, this is needed for some validation and padding check
	if(RTCP_SUCCESS != rtcp_GetPktHeader(hdr, &pktheader))
	{
		*appdatalen = 0;
		return RTCP_FAILURE;
	}

	if(pktheader.PacketType != RTCP_PKTTYPE_APP)
	{
		*appdatalen = 0;
		return RTCP_FAILURE;
	}

	temp = (RTCP_APP *)hdr;

	// Check for Padding
    	if(1 == pktheader.Padding)
	{
		pktlen = rtcp_GetPktLenInBytes(hdr);
		numpad = *((UInt8 *)hdr + (pktlen - 1));
	}
		
	// Get the Length of remainder of the RTCP Packet which 
	// would actually be Application data. Subtract the padding
	// bytes that is not part of the Application Data
	*appdatalen = rtcp_GetPktLenInBytes(hdr) - 
				( (UInt8)sizeof(RTCP_Pkt_Header) + 
				  (UInt8)sizeof(temp->SSRC) + 
				  (UInt8)sizeof(temp->Name) -
				  numpad);
	return RTCP_SUCCESS;
}

/**********************************************************************************
*        NAME:	rtcp_GetAPPData
* DESCRIPTION:	Retrieve the Application data from the APP RTCP Packet
*      INPUTS:	app		- Pointer to the start of the APP Packet of the type
*				  RTCP_APP
*     OUTPUTS:	appdata		- Pointer to the start of the Application Data
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	Application should pass in correct amount of storage by checking
*		with rtcp_GetAppDatalen first
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetAPPData(RTCP_APP* app, UInt8* appdata)
{	
	UInt8	appdatalen	= 0;
	UInt16	offset		= 0;

	// Validity Check
	if( (NULL == app) || (NULL == appdata))
	{
		return RTCP_FAILURE;
	}

	offset = (UInt32)sizeof(RTCP_Pkt_Header) + (UInt32)sizeof(app->SSRC) + (UInt32)sizeof(app->Name);
	memcpy(appdata, app + offset, appdatalen);
	
	return RTCP_SUCCESS;
}

/**********************************************************************************
*        NAME:	rtcp_GetAPP
* DESCRIPTION:	Retrieve the APP RTCP Packet
*      INPUTS:	hdr		- Pointer to the start of the APP Packet of the type
*				  RTCP_Pkt_Header
*     OUTPUTS:	sdes		- Storage for the APP RTCP Packet
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	Application needs to take care of the memory allocation for the
*		entire APP Packet including the Application Data
***********************************************************************************/
eRTCP_Ret_Code rtcp_GetAPP(RTCP_Pkt_Header* hdr, RTCP_APP* app)
{
	// Local variables
	RTCP_Pkt_Header* 	curhdr		= NULL;
	RTCP_APP*		 pkt		= NULL;
	UInt16			 offset		= 0;
	UInt8			 appdatalen	= 0;
   
	// Validity Check
	if( (NULL == hdr) || (NULL == app) )
	{
	    return RTCP_FAILURE;
	}

	if(hdr->PacketType != RTCP_PKTTYPE_APP) 
	{
		return RTCP_FAILURE;
	}

	if( NULL != app->APP_Header)
	{
		if(RTCP_SUCCESS != rtcp_GetPktHeader(hdr, app->APP_Header))
		{
			return RTCP_FAILURE;
		}
	}

	// Initialize the variable
	pkt = (RTCP_APP *)hdr;
	
	// Get the fixed length data
	app->SSRC = rtcp_NetToHost32(pkt->SSRC);
	
	memset(&app->Name, 0, sizeof(app->Name));
	memcpy(&app->Name, &pkt->Name, sizeof(app->Name));

	// Get the Application data
	if(RTCP_SUCCESS == rtcp_GetAPPDataLen(hdr, &appdatalen))
	{
		if((appdatalen > 0) && (NULL != app->ApplicationData))
		{
			memset(app->ApplicationData, 0, appdatalen);
			memcpy(app->ApplicationData, (UInt8 *)&pkt->ApplicationData, appdatalen);
		}
	}
	return RTCP_SUCCESS;	 
}

/**********************************************************************************
*        NAME:	rtcp_GetNumBYEBlks
* DESCRIPTION:	Retrieve the number of BYE Blocks in the BYE RTCP Packet
*      INPUTS:	hdr		- Pointer to the start of the BYE Packet of the type
*				  RTCP_BYE
*     OUTPUTS:	numblks		- Number of Bye Blocks
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
************************************************************************************/
eRTCP_Ret_Code rtcp_GetNumBYEBlks(RTCP_Pkt_Header* hdr, UInt8* numblks)
{
	RTCP_BYE* bye = NULL;
	if( (NULL == hdr) || (NULL == numblks))
    {
	    return RTCP_FAILURE;
    }
	
	if( hdr->PacketType != RTCP_PKTTYPE_APP)
	{
		*numblks = 0;
		return RTCP_FAILURE;
	}

	bye = (RTCP_BYE *)hdr;
	*numblks  = bye->BYE_Header->Count;
	
	return RTCP_SUCCESS;
}

/********************************************************************************
*        NAME:	rtcp_GetNumBYEBlks
* DESCRIPTION:	Get the next Bye Block in the BYE RTCP Packet
*      INPUTS:	BYE_Block	- Pointer to the start of the current Bye Block
*     OUTPUTS:	None
*     RETURNS:	BYE_Block	- Pointer to the start of the next Bye Block
*    CAUTIONS:	None
**********************************************************************************/
BYE_Block* rtcp_GetNextBYEBlock(BYE_Block* blk)
{
	if(NULL == blk)
		return(NULL);
	
	return (blk + sizeof(blk->SSRC));
}

/********************************************************************************
*        NAME:	rtcp_GetBYE
* DESCRIPTION:	Retrieve the BYE RTCP Packet
*      INPUTS:	hdr		- Pointer to the start of the BYE Packet of the type
*				  RTCP_Pkt_Header
*     OUTPUTS:	sdes		- Storage for the BYE RTCP Packet
*     RETURNS:	eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	Application needs to take care of the memory allocation for the
*		entire BYE Packet including all its blocks
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetBYE(RTCP_Pkt_Header* hdr, RTCP_BYE* bye)
{
	RTCP_Pkt_Header* curhdr	= NULL;
	RTCP_BYE*		 pkt	= NULL;
	UInt16			 offset	= 0;
	UInt16			 remlen	= 0;
	UInt8			 i		= 0;
	UInt8			 idx	= 0;
	BYE_Block*		 curblk	= NULL;
   
	// Validity Check
	if( (NULL == hdr) || (NULL == bye))
    {
	    return RTCP_FAILURE;
    }

	if (hdr->PacketType != RTCP_PKTTYPE_BYE)
	{
	   return RTCP_FAILURE;
	}

	pkt = (RTCP_BYE *)hdr;
	
	// Get BYE Header
	if( bye->BYE_Header != NULL)
	{
		if(RTCP_SUCCESS != rtcp_GetPktHeader(hdr, bye->BYE_Header))
		{
			return RTCP_FAILURE;
		}
	}

	// Get the rest of the data
	if(bye->BYE_Header->Count > 0)
	{
		if(NULL != curblk)
		{
			curblk = (BYE_Block *)&(pkt->ByeBlkList);
		
			// Cycle through each report block and populate the values
			for(i = 0; 
				i < bye->BYE_Header->Count; 
				i++, curblk = curblk + sizeof(BYE_Block))
			{
				bye->ByeBlkList[i].SSRC = rtcp_NetToHost32(curblk->SSRC);
			}
		}
    }
	return RTCP_SUCCESS;	 
}

/************************************************************************
*        NAME:	rtcp_GetXRBlkLenInBytes
* DESCRIPTION:	Retrieve the Length of the XR Block
*      INPUTS:	xrhdr - Pointer to the start of the XR Packet of the type
*			XR_Block_Header
*     OUTPUTS:	None
*     RETURNS:  Length of the XR Block
*    CAUTIONS:	None
**************************************************************************/
UInt16 rtcp_GetXRBlkLenInBytes(XR_Block_Header* xrblkhdr)
{
	UInt8	blklen	= 0;
	UInt16	offset	= 0;
	UInt8  *temp	= NULL;

	
	if(NULL == xrblkhdr)
	{
		return RTCP_FAILURE;
	}

	offset = (UInt16) sizeof(XR_Block_Header) + (UInt16)sizeof(UInt8);
	temp = (UInt8 *)xrblkhdr + offset;
	blklen = 4 * ( rtcp_NetToHost16(*((UInt16 *)temp)) + 1 );
	
	return blklen;
}

/********************************************************************************
*        NAME:	rtcp_GetNumXRBlks
* DESCRIPTION:	Retrieve the number of XR blocks present in the XR RTCP Packet
*      INPUTS:	xrhdr - Pointer to the start of the XR Packet of the type
*				RTCP_XR_Header
*     OUTPUTS:	numblks		- Number of XR Blocks found
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetNumXRBlks(RTCP_XR_Header* xrhdr, UInt8* numblks)
{
	UInt16				offset		= 0;
	UInt16				pktlen		= 0;
	UInt16				remlength	= 0;
	UInt16				blklen		= 0;
	UInt8				blocktype	= 0;
	UInt8				numpad		= 0;
	XR_Block_Header*		curp		= NULL;
	RTCP_XR_Header			temp;
	
	if( (NULL == xrhdr) || (NULL == numblks) )
	{
		return RTCP_FAILURE;
	}
	
	if( RTCP_SUCCESS != rtcp_GetXRHeader((RTCP_Pkt_Header *)xrhdr, &temp) )
	{
		return RTCP_FAILURE;
	}
	
	// Get the length of the entire XR RTCP Packet
	pktlen = rtcp_GetPktLenInBytes((RTCP_Pkt_Header *)xrhdr);
	
	// Check for padding
	if(1 == temp.Padding)
	{
		numpad = *((UInt8 *)xrhdr + (pktlen -1));
		pktlen = pktlen - numpad;
	}
	
	// Go the start of XR Blocks
	offset = (UInt16) sizeof(RTCP_XR_Header);
	curp = (XR_Block_Header *)((UInt8 *)xrhdr + offset); 

	// Restart from here
	*numblks = 0;
	remlength = pktlen - offset;
	while(remlength > 0)
	{
		if(	(curp->BlockType >= XR_LOSS_RLE_REPORT_BLOCK) && 
			(curp->BlockType <= XR_HR_VOIP_REPORT_BLOCK))
		{
			*numblks = *numblks + 1;
			blklen = rtcp_GetXRBlkLenInBytes(curp);
			curp = (XR_Block_Header *)((UInt8 *)curp + blklen);
			remlength = remlength - blklen;
		}
		else
		{
			*numblks = 0;
			return RTCP_FAILURE;
		}
	}
	return RTCP_SUCCESS;
}

/***********************************************************************************
*        NAME:	rtcp_GetXRHeader
* DESCRIPTION:	Retrieve the Header of the XR RTCP Packet
*      INPUTS:	hdr		- Pointer to the start of the XR Packet
*     OUTPUTS:	xrhdr	- Populated header of XR RTCP Packet of type RTCP_XR_Header
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
***********************************************************************************/
eRTCP_Ret_Code rtcp_GetXRHeader(RTCP_Pkt_Header* hdr, RTCP_XR_Header* xrhdr)
{
	// Validity Check
	if( (NULL == hdr)|| (NULL == xrhdr))
	{
		return RTCP_FAILURE;
	}

	if (hdr->PacketType != RTCP_PKTTYPE_XR)
	{
		return RTCP_FAILURE;
	}

	// Get the Header
	memset(xrhdr, 0, sizeof(RTCP_XR_Header));


	xrhdr->Version		= GetBits8(*((UInt8 *)hdr),1,2); 
	xrhdr->Padding		= GetBits8(*((UInt8 *)hdr),3,1); 
	xrhdr->Reserved		= GetBits8(*((UInt8 *)hdr),4,5); 
	xrhdr->PacketType	= *((UInt8 *)hdr + 1); 
	xrhdr->Length		= rtcp_GetPktLenInBytes(hdr); 
	xrhdr->SSRC			= rtcp_NetToHost32(*((UInt32 *)hdr + 1));

	return RTCP_SUCCESS;
}

/****************************************************************************************
*        NAME:	rtcp_GetNextXRBlkHeaderPtr
* DESCRIPTION:	Retrieve the pointer to the start of the next XR Block in a XR RTCP Packet
*      INPUTS:	blkhdr	- Pointer to the start of current XR Packet header of the type
*			  XR_Block_Header
*     OUTPUTS:	None
*     RETURNS:  Pointer to the start of next XR Block 
*    CAUTIONS:	None
******************************************************************************************/
XR_Block_Header* rtcp_GetNextXRBlkHeaderPtr(XR_Block_Header* blkhdr)
{
   XR_Block_Header *temp = NULL;
 
   if(NULL != blkhdr)
   {
	   temp = (XR_Block_Header *)((UInt8 *)blkhdr + rtcp_GetXRBlkLenInBytes(blkhdr));
	   if((temp->BlockType >= XR_LOSS_RLE_REPORT_BLOCK) && 
		  (temp->BlockType <= XR_HR_VOIP_REPORT_BLOCK))
	   {
		   return temp;
	   }
   }
   return NULL;
}

/**********************************************************************************************
*        NAME:	rtcp_GetXRBlocksPtr
* DESCRIPTION:	Retrieve the array of pointers to the start of each XR Block in a XR RTCP Packet
*      INPUTS:	hdr		- Pointer to the start of current XR Packet header of the type
*				XR_Block_Header
*     OUTPUTS:	blkhdrarr		- Populated XR Blocks pointer array
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	Memory allocation for the blkhdrarr should be done correctly after
*				checking the number of XR blocks present.
************************************************************************************************/
eRTCP_Ret_Code rtcp_GetXRBlocksPtr(RTCP_XR_Header* xrhdr, XR_Block_Header** blkhdrarr, UInt8 arrlen)
{
	XR_Block_Header*	curhdr		= NULL;
    UInt8				idx			= 0;
    UInt16				length		= 0;
	UInt16				offset		= 0;
	UInt8				numblks		= 0;
	UInt8				numpad		= 0;
	UInt16				pktlen		= 0;
	UInt16				remlength	= 0;
	RTCP_XR_Header		temp;
	
	if((NULL == xrhdr) || (NULL == blkhdrarr) || (0 == arrlen))
	{
		return RTCP_FAILURE;
	}

	// Get the Packet Header
	if( RTCP_SUCCESS != rtcp_GetXRHeader((RTCP_Pkt_Header *)xrhdr, &temp))
	{
		return RTCP_FAILURE;
	}
	
	// Make sure the array sent in can support the right number of blocks
	if(RTCP_SUCCESS != rtcp_GetNumXRBlks(xrhdr,&numblks))
	{
		return RTCP_FAILURE;
	}

	if(numblks != arrlen)
	{
		return RTCP_FAILURE;
	}

	// Get the Length of the entire XR Packet
	pktlen = rtcp_GetPktLenInBytes((RTCP_Pkt_Header *)xrhdr);

	// Check for padding and adjust the packet length
	if(1 == temp.Padding)
	{
		numpad = *((UInt8 *)xrhdr + (pktlen -1));
		pktlen = pktlen - numpad;
	}

	// Initialize the variables
	memset(blkhdrarr, 0, sizeof(numblks));
	
	// Move to start of XR Blocks
	offset = (UInt16)sizeof(RTCP_XR_Header);
	curhdr = (XR_Block_Header *)((UInt8 *)xrhdr + offset);
	
	remlength = pktlen - offset;
	for( idx = 0; 
         remlength >0, (NULL != curhdr); 
		 curhdr = rtcp_GetNextXRBlkHeaderPtr(curhdr),
		 idx ++)
	{
		if(	(curhdr->BlockType >= XR_LOSS_RLE_REPORT_BLOCK) && 
			(curhdr->BlockType <= XR_HR_VOIP_REPORT_BLOCK))
		{
			blkhdrarr[idx] = (XR_Block_Header *) curhdr;
			remlength = remlength - rtcp_GetXRBlkLenInBytes(curhdr);            
		}
		else // if packet type is not valid, the Report is assumed to be corrupt
		{
			return RTCP_FAILURE;
		}		    		
	}
	return RTCP_SUCCESS;
}

/********************************************************************************
*        NAME:	rtcp_GetNumXRRLEChunks
* DESCRIPTION:	Retrieve the number of RLE Chunks in XR Loss RLE / Duplicate
*		Loss RLE Block
*      INPUTS:	xrrle		- Pointer to the start of XR Loss RLE / Duplicate
*				  Loss RLE Block
*     OUTPUTS:	numchnks	- Number of RLE Chunks Found
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
********************************************************************************/
eRTCP_Ret_Code rtcp_GetNumXRRLEChunks(XR_Block_Header* xrrle, UInt8 *numchnks)
{
	XR_RLE_Chunk* curp			= NULL;
	UInt16	blklen				= 0;
	UInt16	remlength			= 0;
	UInt16	offset				= 0;
	UInt8	chnktype			= 0;
	UInt16	temp				= 0;

	*numchnks				= 0;
	

	if((NULL == xrrle) || (NULL == numchnks))
	{
		return RTCP_FAILURE;
	}

	blklen = rtcp_GetXRBlkLenInBytes((XR_Block_Header *)xrrle);

	
	if( xrrle->BlockType == XR_LOSS_RLE_REPORT_BLOCK)
	{
		offset = (UInt16) sizeof(XR_Block_Header) + sizeof(UInt8) +
			 (UInt16) sizeof(((XR_Loss_RLE_Block *)xrrle)->BlockLength) +
			 (UInt16) sizeof( ((XR_Loss_RLE_Block *)xrrle)->SSRC) +
			 (UInt16) sizeof(UInt16) +
			 (UInt16) sizeof(UInt16);
    }
	else
		if( xrrle->BlockType == XR_DUPLOSS_RLE_REPORT_BLOCK)
		{
	
			offset = (UInt16) sizeof(XR_Block_Header) + sizeof(UInt8) +
					 (UInt16) sizeof(((XR_DupLoss_RLE_Block *)xrrle)->BlockLength) +
			         (UInt16) sizeof(((XR_DupLoss_RLE_Block *)xrrle)->SSRC) +
					 (UInt16) sizeof(UInt16) +
					 (UInt16) sizeof(UInt16);
		}
	else
		return RTCP_FAILURE;

	curp = (XR_RLE_Chunk *)((UInt8 *)xrrle + offset);
	remlength = blklen - offset;
	while (remlength > 0)
	{		
		if((chnktype == RUNLENGTH) || (chnktype == BITVECTOR))
		{
			*numchnks = *numchnks + 1;
			curp = (XR_RLE_Chunk *)((UInt8 *) curp + (UInt8)sizeof(XR_RLE_Chunk));
			remlength = remlength - (UInt8)sizeof(XR_RLE_Chunk);
		}
		else
			if( (chnktype == TERMINATING) && (*((UInt16 *)curp) == 0) )
			{
				return RTCP_SUCCESS;
			}
		else
		{
			*numchnks = 0;
			return RTCP_FAILURE;
		}
	}
	return RTCP_SUCCESS;

}
/********************************************************************************
*        NAME:	rtcp_GetXRRcptTimeBlks
* DESCRIPTION:	Retrieve the RLE Chunks inside XR Xoss RLE Block or XR Dup Loss
*		RLE Block
*      INPUTS:	xrrle		- Pointer to the start of XR RLE/Dup RLE Block
*		numchnks	- Number of RLE Blocks
*     OUTPUTS:	xrrlechunk	- Populated array of RLE Chunks
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	Application should allocate memory for the array properly by
*		finding the number of chunks first using rtcp_GetNumXRRLEChunks()
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetXRRLEChunks(XR_Block_Header* xrrle, XR_RLE_Chunk** xrrlechunk, UInt8 numchnks)
{
	XR_RLE_Chunk* curp			= NULL;
	UInt16	blklen				= 0;
	UInt16	offset				= 0;
	UInt8	chnktype			= 0;
	UInt8	i					= 0;

	if((NULL == xrrle) || (NULL == xrrlechunk) || (0 == numchnks))
	{
		return RTCP_FAILURE;
	}

	memset(xrrlechunk, 0, numchnks);

	blklen = rtcp_GetXRBlkLenInBytes((XR_Block_Header *)xrrle);
	if( xrrle->BlockType == XR_LOSS_RLE_REPORT_BLOCK)
	{
		offset = (UInt16) sizeof(XR_Block_Header) + sizeof(UInt8) +
			 (UInt16) sizeof(((XR_Loss_RLE_Block *)xrrle)->BlockLength) +
			 (UInt16) sizeof( ((XR_Loss_RLE_Block *)xrrle)->SSRC) +
			 (UInt16) sizeof(UInt16) +
			 (UInt16) sizeof(UInt16);
    }
	else
		if( xrrle->BlockType == XR_DUPLOSS_RLE_REPORT_BLOCK)
		{
	
			offset = (UInt16) sizeof(XR_Block_Header) + sizeof(UInt8) +
					 (UInt16) sizeof(((XR_DupLoss_RLE_Block *)xrrle)->BlockLength) +
			         (UInt16) sizeof(((XR_DupLoss_RLE_Block *)xrrle)->SSRC) +
					 (UInt16) sizeof(UInt16) +
					 (UInt16) sizeof(UInt16);
		}
	else
		return RTCP_FAILURE;

	curp = (XR_RLE_Chunk *)((UInt8 *)xrrle + offset);
	i = 0;
	while ((offset < blklen) && (i < numchnks))
	{		
		chnktype = GetBits8(*((UInt8 *)curp), 1, 1);
		if((chnktype == RUNLENGTH))
		{
			xrrlechunk[i]->ChnkType = chnktype;
			xrrlechunk[i]->Chunk.Run_Length_Chunk.RunType = GetBits8(*((UInt8 *)curp), 2, 1);
			xrrlechunk[i]->Chunk.Run_Length_Chunk.RunLength = GetBits8(*((UInt8 *)curp), 3, 4);

			curp = (XR_RLE_Chunk *)((UInt8 *)curp + 1);
			offset = offset + 1;
		}
		else
			if(chnktype == BITVECTOR)
			{
				xrrlechunk[i]->ChnkType = chnktype;
				xrrlechunk[i]->Chunk.Bit_Vector_Chunk.BitVector = GetBits8(*((UInt8 *)curp), 2, 15);				
			}
		else
			if( (chnktype == TERMINATING) && (*((UInt16 *)curp) == 0) )
			{
				return RTCP_SUCCESS;
			}
		else
			return RTCP_FAILURE;
		
	}
	return RTCP_SUCCESS;
}

/********************************************************************************
*        NAME:	rtcp_GetNumRcptTimeBlks
* DESCRIPTION:	Retrieve the number of Receipt Times in XR Receipt Time Block
*      INPUTS:	dlrrblk		- Pointer to the start of XR DReceipt Time Block
*     OUTPUTS:	numblks		- Number of Receipt Times Found
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetNumRcptTimeBlks(XR_Rpt_RcptTime_Block* rcptblk, UInt8* numblks)
{
	UInt16				blklen		= 0;
	UInt16				offset		= 0;
	XR_Rpt_RcptTime*		curp		= NULL;
	UInt16				remlength 	= 0;
	*numblks					= 0;


	if( (NULL == rcptblk) && (NULL == numblks))
	{
		return RTCP_FAILURE;	
	}

	blklen = rtcp_GetXRBlkLenInBytes( (XR_Block_Header*)rcptblk);
	offset =	 (UInt16) sizeof(XR_Block_Header) + sizeof(UInt8) +
			 (UInt16) sizeof(rcptblk->BlockLength) +
			 (UInt16) sizeof(rcptblk->SSRC) +
			 (UInt16) sizeof(UInt16) +
			 (UInt16) sizeof(UInt16);

	curp = (XR_Rpt_RcptTime *)((UInt8 *)rcptblk + offset);
	remlength = blklen - offset;
	while (remlength > 0)
	{		
		*numblks = *numblks + 1;
		curp = curp + (UInt8)sizeof(XR_Rpt_RcptTime);
		remlength = remlength - (UInt8)sizeof(XR_Rpt_RcptTime);
	}
	return RTCP_SUCCESS;
}

/******************************************************************************
*        NAME:	rtcp_GetXRRcptTimeBlks
* DESCRIPTION:	Retrieve the Receipt Times in XR Receipt Time Blcok
*      INPUTS:	rcptblk		- Pointer to the start of XR Receipt Time Block
*		numblks		- Number of Receipt Times
*     OUTPUTS:	xrrcptarr	- Populated array of Receipt Times
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
*******************************************************************************/
eRTCP_Ret_Code rtcp_GetXRRcptTimeBlks(XR_Rpt_RcptTime_Block* rcptblk, XR_Rpt_RcptTime** xrrcptarr, UInt8 numblks)
{
	XR_Rpt_RcptTime* 	curp			= NULL;
	UInt16			blklen			= 0;
	UInt16			offset			= 0;
	UInt16  		remlength		= 0;
	UInt32 			temp 			= 0;
	UInt8			i			= 0;

	if((NULL == rcptblk) || (NULL == xrrcptarr) || (0 == numblks))
	{
		return RTCP_FAILURE;
	}

	blklen = rtcp_GetXRBlkLenInBytes((XR_Block_Header *)rcptblk);
	offset = (UInt16) sizeof(XR_Block_Header) + sizeof(UInt8) +
			 (UInt16) sizeof(rcptblk->BlockLength) +
			 (UInt16) sizeof(rcptblk->SSRC) +
			 (UInt16) sizeof(UInt16) +
			 (UInt16) sizeof(UInt16);

	
	
	remlength = blklen - offset;
	curp = (XR_Rpt_RcptTime *)((UInt8 *)rcptblk + offset);
	i = 0; 
	while ((remlength > 0) && (i < numblks))
	{	
		xrrcptarr[i]->ReceiptTime = rtcp_NetToHost32((UInt32)curp->ReceiptTime);
		curp = curp + (UInt8)sizeof(XR_Rpt_RcptTime);
		remlength = remlength - (UInt8)sizeof(XR_Rpt_RcptTime);
		i ++;
		
	}
	return RTCP_SUCCESS;
}

/****************************************************************************
*        NAME:	rtcp_GetNumXRLRRBlks
* DESCRIPTION:	Retrieve the number of LRR blocks in a XR DLRR Block
*      INPUTS:	dlrrblk		- Pointer to the start of XR DLRR Block
*     OUTPUTS:	numblks		- Number of LRR Blocks Found
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
****************************************************************************/
eRTCP_Ret_Code rtcp_GetNumXRLRRBlks(XR_DLRR_Block* dlrrblk, UInt8* numblks)
{
	UInt16		blklen		= 0;
	UInt16		remlength	= 0;
	UInt16		offset		= 0;
	XR_LRR*		curp		= NULL;
	UInt8 temp =0;
	*numblks				= 0;
	

	if( (NULL == dlrrblk) && (NULL == numblks))
	{
		return RTCP_FAILURE;	
	}

	blklen = rtcp_GetXRBlkLenInBytes((XR_Block_Header *)dlrrblk);
	offset = (UInt16) sizeof(XR_Block_Header) + sizeof(UInt8) +
			 (UInt16) sizeof(dlrrblk->BlockLength);

	remlength = blklen - offset;
	curp = (XR_LRR *)((UInt8 *)dlrrblk + offset);
	while (remlength > 0)
	{		
		*numblks = *numblks + 1;
		curp = curp + (UInt8)sizeof(XR_LRR);
		remlength = remlength - (UInt8)sizeof(XR_LRR);
	}
	return RTCP_SUCCESS;
}

/********************************************************************************
*        NAME:	rtcp_GetXRLRRBlks
* DESCRIPTION:	Retrieve the populated array of LRR Blocks inside XR DLRR Block
*      INPUTS:	dlrrblk		- Pointer to the start of XR DLRR Block
*     OUTPUTS:	xrlrrarrr	- Array of Pointers to storage for every LRR Block
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	The memory allocation for all LRR Blocks should be done correctly
*		by getting the number through rtcp_GetNumXRLRRBlks()
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetXRLRRBlks(XR_DLRR_Block* dlrrblk, XR_LRR** xrlrrarr, UInt8 numblks)
{
	XR_LRR* curp				= NULL;
	UInt16	remlength			= 0;
	UInt16	blklen				= 0;
	UInt16	offset				= 0;
	
	UInt8	i					= 0;

	if((NULL == dlrrblk) || (NULL == xrlrrarr) || (0 == numblks))
	{
		return RTCP_FAILURE;
	}

	blklen = rtcp_GetXRBlkLenInBytes((XR_Block_Header *)dlrrblk);
	offset = (UInt16) sizeof(XR_Block_Header) + sizeof(UInt8) +
			 (UInt16) sizeof(dlrrblk->BlockLength);


	curp = (XR_LRR *)((UInt8 *)dlrrblk + offset);
	remlength = blklen - offset;
	i = 0;
	while ((remlength > 0) && (i < numblks))
	{		
		xrlrrarr[i]->SSRC = rtcp_NetToHost32(curp->SSRC);
		xrlrrarr[i]->DLRR = rtcp_NetToHost32(curp->DLRR);
		xrlrrarr[i]->LRR = rtcp_NetToHost32(curp->LRR);
		curp = curp + (UInt8)sizeof(XR_LRR);
		remlength = remlength - (UInt8)sizeof(XR_LRR);
		i ++;		
	}
	return RTCP_SUCCESS;
}

/********************************************************************************
*        NAME:	rtcp_GetXRBlock
* DESCRIPTION:	Retrieve the populated XR Block by blocktype
*      INPUTS:	xrhdr		- Pointer to the start of XR Block header that we 
*				are interested in
*     OUTPUTS:	xr	- Pointer to the start of populated XR Block of 
*				the type XR_Block_Header
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetXRBlock(XR_Block_Header* xrhdr, XR_Block_Header* blkhdr)
{
	XR_Block_Header*		curhdr		= NULL;
	UInt8				rval		= RTCP_SUCCESS;
	UInt8				numchunks	= 0;
	UInt8				numblks		= 0;
	XR_RLE_Chunk**			lrle		= NULL;
	XR_Rpt_RcptTime**		rcptarr		= NULL;
	XR_LRR**			lrrarr		= NULL;
	XR_StatSummary_Block*		stat		= NULL;
	XR_StatSummary_Block*		statin		= NULL;
	XR_StatSummary_Block*		statout		= NULL;
	UInt8*				datastart	= NULL;
	XR_RLE_Chunk**			lrlearr		= NULL;
	XR_RLE_Chunk**			dlrlearr	= NULL;
	XR_ReceiverRefTime_Block*	refin		= NULL;
	XR_ReceiverRefTime_Block*	refout		= NULL;
	XR_VOIP_Metrics_Block*		voipin		= NULL;
	XR_VOIP_Metrics_Block*		voipout		= NULL;

	// Validity Check
	if((NULL == xrhdr) || (NULL == blkhdr))
	{
		return RTCP_FAILURE;
	}

	switch(xrhdr->BlockType)
	{

		case XR_LOSS_RLE_REPORT_BLOCK:
			datastart = (UInt8 *)xrhdr + sizeof(XR_Block_Header);
			((XR_Loss_RLE_Block *)blkhdr)->xrblkhdr.BlockType	= ((XR_Loss_RLE_Block *)xrhdr)->xrblkhdr.BlockType;
			((XR_Loss_RLE_Block *)blkhdr)->Reserved				= GetBits8(*((UInt8 *)datastart), 1, 4);
			((XR_Loss_RLE_Block *)blkhdr)->Thinning				= GetBits8(*((UInt8 *)datastart), 5, 4);
			((XR_Loss_RLE_Block *)blkhdr)->BlockLength			= rtcp_GetXRBlkLenInBytes(xrhdr);
			((XR_Loss_RLE_Block *)blkhdr)->SSRC				= rtcp_NetToHost32(((XR_Loss_RLE_Block *)xrhdr)->SSRC);
			((XR_Loss_RLE_Block *)blkhdr)->BeginSeq				= rtcp_NetToHost16(((XR_Loss_RLE_Block *)xrhdr)->BeginSeq);
			((XR_Loss_RLE_Block *)blkhdr)->EndSeq				= rtcp_NetToHost16(((XR_Loss_RLE_Block *)xrhdr)->EndSeq);
			
			if(RTCP_SUCCESS == rtcp_GetNumXRRLEChunks(xrhdr, &numchunks))
			{
				lrlearr = ((XR_Loss_RLE_Block *)blkhdr)->chnk;
				if( (0 < numchunks) && (NULL != lrlearr))
				{					
					rval = rtcp_GetXRRLEChunks(xrhdr, lrlearr, numchunks);
				}
			}			
			break;

		case XR_DUPLOSS_RLE_REPORT_BLOCK:
			datastart = (UInt8 *)xrhdr + sizeof(XR_Block_Header);
			((XR_DupLoss_RLE_Block *)blkhdr)->xrblkhdr.BlockType	= ((XR_DupLoss_RLE_Block *)xrhdr)->xrblkhdr.BlockType;
			((XR_DupLoss_RLE_Block *)blkhdr)->Reserved		= GetBits8(*((UInt8 *)datastart), 1, 4);
			((XR_DupLoss_RLE_Block *)blkhdr)->Thinning		= GetBits8(*((UInt8 *)datastart), 5, 4);
			((XR_DupLoss_RLE_Block *)blkhdr)->BlockLength		= rtcp_GetXRBlkLenInBytes(xrhdr);
			((XR_DupLoss_RLE_Block *)blkhdr)->SSRC			= rtcp_NetToHost32(((XR_DupLoss_RLE_Block *)xrhdr)->SSRC);
			((XR_DupLoss_RLE_Block *)blkhdr)->BeginSeq		= rtcp_NetToHost16(((XR_DupLoss_RLE_Block *)xrhdr)->BeginSeq);
			((XR_DupLoss_RLE_Block *)blkhdr)->EndSeq		= rtcp_NetToHost16(((XR_DupLoss_RLE_Block *)xrhdr)->EndSeq);
			
			if(RTCP_SUCCESS == rtcp_GetNumXRRLEChunks(xrhdr, &numchunks))
			{
				dlrlearr = ((XR_DupLoss_RLE_Block *)blkhdr)->chnk;
				if( (0 < numchunks) && (NULL != dlrlearr))
				{
					rval = rtcp_GetXRRLEChunks(xrhdr, dlrlearr, numchunks);
				}
			}			
			break;
	
		case XR_DLRR_REPORT_BLOCK:
			datastart = (UInt8 *)xrhdr + sizeof(XR_Block_Header);
			((XR_DLRR_Block *)blkhdr)->xrblkhdr.BlockType	= ((XR_DLRR_Block *)xrhdr)->xrblkhdr.BlockType;
			((XR_DLRR_Block *)blkhdr)->Reserved		= ((XR_DLRR_Block *)xrhdr)->Reserved;
			((XR_DLRR_Block *)blkhdr)->BlockLength		= rtcp_GetXRBlkLenInBytes(xrhdr);
		
			if(RTCP_SUCCESS == rtcp_GetNumXRLRRBlks((XR_DLRR_Block *)xrhdr, &numchunks))
			{
				if(numchunks > 0)
				{
					lrrarr = ((XR_DLRR_Block *)blkhdr)->lrr;
					rval = rtcp_GetXRLRRBlks((XR_DLRR_Block *)xrhdr, lrrarr, numchunks);
				}
			}
			break;
		
		case XR_RPT_RCPT_TIMES_REPORT_BLOCK:
		    
			datastart = (UInt8 *)xrhdr + sizeof(XR_Block_Header);
			((XR_Rpt_RcptTime_Block *)blkhdr)->xrblkhdr.BlockType	= ((XR_Rpt_RcptTime_Block *)xrhdr)->xrblkhdr.BlockType;
			((XR_Rpt_RcptTime_Block *)blkhdr)->Reserved		= GetBits8(*((UInt8 *)datastart), 1, 4);
			((XR_Rpt_RcptTime_Block *)blkhdr)->Thinning		= GetBits8(*((UInt8 *)datastart), 5, 4);
			((XR_Rpt_RcptTime_Block *)blkhdr)->BlockLength		= rtcp_GetXRBlkLenInBytes(xrhdr);
			((XR_Rpt_RcptTime_Block *)blkhdr)->SSRC			= rtcp_NetToHost32(((XR_Rpt_RcptTime_Block *)xrhdr)->SSRC);
			((XR_Rpt_RcptTime_Block *)blkhdr)->BeginSeq		= rtcp_NetToHost16(((XR_Rpt_RcptTime_Block *)xrhdr)->BeginSeq);
			((XR_Rpt_RcptTime_Block *)blkhdr)->EndSeq		= rtcp_NetToHost16(((XR_Rpt_RcptTime_Block *)xrhdr)->EndSeq);
			
			if(RTCP_SUCCESS == rtcp_GetNumRcptTimeBlks((XR_Rpt_RcptTime_Block *)xrhdr, &numblks))
			{
				rcptarr = ((XR_Rpt_RcptTime_Block *)blkhdr)->rptrcpttime;
				if( (0 < numblks) && (NULL != rcptarr))
				{
					rval = rtcp_GetXRRcptTimeBlks((XR_Rpt_RcptTime_Block *)xrhdr, rcptarr, numblks);
				}
			}

			break;
			
		case XR_RECV_REF_TIME_REPORT_BLOCK:
			refin = (XR_ReceiverRefTime_Block *)blkhdr;
			refout = (XR_ReceiverRefTime_Block *)xrhdr;
			datastart = (UInt8 *)xrhdr + sizeof(XR_Block_Header);
			((XR_ReceiverRefTime_Block *)blkhdr)->xrblkhdr.BlockType	= ((XR_ReceiverRefTime_Block *)xrhdr)->xrblkhdr.BlockType;
			((XR_ReceiverRefTime_Block *)blkhdr)->Reserved			= ((XR_ReceiverRefTime_Block *)xrhdr)->Reserved;
			((XR_ReceiverRefTime_Block *)blkhdr)->BlockLength		= rtcp_GetXRBlkLenInBytes(xrhdr);
			((XR_ReceiverRefTime_Block *)blkhdr)->NTPTimeMSW		= rtcp_NetToHost32(((XR_ReceiverRefTime_Block *)xrhdr)->NTPTimeMSW);
			((XR_ReceiverRefTime_Block *)blkhdr)->NTPTimeLSW		= rtcp_NetToHost32(((XR_ReceiverRefTime_Block *)xrhdr)->NTPTimeLSW);
			break;



		case XR_STATSUMMARY_REPORT_BLOCK:
			statin = (XR_StatSummary_Block *)xrhdr;
			statout = (XR_StatSummary_Block *)blkhdr;
			datastart = (UInt8 *)xrhdr + sizeof(XR_Block_Header);
			
			statout->xrblkhdr.BlockType		= statin->xrblkhdr.BlockType;
			statout->LossReportFlag			= GetBits8(*((UInt8 *)datastart), 1, 1);
			statout->DupReportFlag			= GetBits8(*((UInt8 *)datastart), 2, 1); 
			statout->JitterFlag			= GetBits8(*((UInt8 *)datastart), 3, 1);
			statout->ToHFlag			= GetBits8(*((UInt8 *)datastart), 4, 2);
			statout->Reserved			= GetBits8(*((UInt8 *)datastart), 6, 3);
			statout->BlockLength			= rtcp_NetToHost16(((XR_StatSummary_Block *)xrhdr)->BlockLength);
			statout->SSRC				= rtcp_NetToHost32(((XR_StatSummary_Block *)xrhdr)->SSRC);
			statout->BeginSeq			= rtcp_NetToHost16(((XR_StatSummary_Block *)xrhdr)->BeginSeq);
			statout->EndSeq				= rtcp_NetToHost16(((XR_StatSummary_Block *)xrhdr)->EndSeq);
			statout->LostPackets			= rtcp_NetToHost32(((XR_StatSummary_Block *)xrhdr)->LostPackets);
			statout->DupPackets			= rtcp_NetToHost32(((XR_StatSummary_Block *)xrhdr)->DupPackets);
			statout->MinJitter			= rtcp_NetToHost32(((XR_StatSummary_Block *)xrhdr)->MinJitter);
			statout->MaxJitter			= rtcp_NetToHost32(((XR_StatSummary_Block *)xrhdr)->MaxJitter);
			statout->MeanJitter			= rtcp_NetToHost32(((XR_StatSummary_Block *)xrhdr)->MeanJitter);	
			statout->DevJitter			= rtcp_NetToHost32(((XR_StatSummary_Block *)xrhdr)->DevJitter);
			statout->MinTOH				= ((XR_StatSummary_Block *)xrhdr)->MinTOH;
			statout->MeanTOH			= ((XR_StatSummary_Block *)xrhdr)->MeanTOH;
			statout->DevTOH				= ((XR_StatSummary_Block *)xrhdr)->DevTOH;
			break;

					
		case XR_VOIP_METRICS_REPORT_BLOCK:
			voipin = (XR_VOIP_Metrics_Block *)xrhdr;
			voipout = (XR_VOIP_Metrics_Block *)blkhdr;

			datastart = (UInt8 *)xrhdr + sizeof(XR_Block_Header);
			((XR_VOIP_Metrics_Block *)blkhdr)->xrblkhdr.BlockType	= ((XR_VOIP_Metrics_Block *)xrhdr)->xrblkhdr.BlockType;
			((XR_VOIP_Metrics_Block *)blkhdr)->Reserved		= ((XR_VOIP_Metrics_Block *)xrhdr)->Reserved;
			((XR_VOIP_Metrics_Block *)blkhdr)->BlockLength		= rtcp_GetXRBlkLenInBytes(xrhdr);
			((XR_VOIP_Metrics_Block *)blkhdr)->SSRC			= rtcp_NetToHost32(((XR_VOIP_Metrics_Block *)xrhdr)->SSRC);
			((XR_VOIP_Metrics_Block *)blkhdr)->LossRate		= (*((UInt8 *)datastart + 7));
			((XR_VOIP_Metrics_Block *)blkhdr)->DiscardRate		= (*((UInt8 *)datastart + 8));
			((XR_VOIP_Metrics_Block *)blkhdr)->BurstDensity		= (*((UInt8 *)datastart + 9));
			((XR_VOIP_Metrics_Block *)blkhdr)->GapDensity		= (*((UInt8 *)datastart + 10));
			((XR_VOIP_Metrics_Block *)blkhdr)->BurstDuration	= rtcp_NetToHost16((*((UInt16 *)(datastart + 11))));
			((XR_VOIP_Metrics_Block *)blkhdr)->GapDuration		= rtcp_NetToHost16((*((UInt16 *)(datastart + 13))));
			((XR_VOIP_Metrics_Block *)blkhdr)->RoundTripDelay	= rtcp_NetToHost16((*((UInt16 *)(datastart + 15))));
			((XR_VOIP_Metrics_Block *)blkhdr)->EndSystemDelay	= rtcp_NetToHost16((*((UInt16 *)(datastart + 17))));
			((XR_VOIP_Metrics_Block *)blkhdr)->SignalLevel		= (*((UInt8 *)datastart + 19));
			((XR_VOIP_Metrics_Block *)blkhdr)->NoiseLevel		= (*((UInt8 *)datastart + 20));
			((XR_VOIP_Metrics_Block *)blkhdr)->RERL			= (*((UInt8 *)datastart + 21)); 
			((XR_VOIP_Metrics_Block *)blkhdr)->GMin			= (*((UInt8 *)datastart + 22)); 
			((XR_VOIP_Metrics_Block *)blkhdr)->RFactor		= (*((UInt8 *)datastart + 23)); 
			((XR_VOIP_Metrics_Block *)blkhdr)->ExtRFactor		= (*((UInt8 *)datastart + 24)); 
			((XR_VOIP_Metrics_Block *)blkhdr)->MOS_LQ		= (*((UInt8 *)datastart + 25)); 
			((XR_VOIP_Metrics_Block *)blkhdr)->MOS_CQ		= (*((UInt8 *)datastart + 26)); 
			((XR_VOIP_Metrics_Block *)blkhdr)->PLC			= GetBits8(*((UInt8 *)datastart + 27), 1, 2); 
			((XR_VOIP_Metrics_Block *)blkhdr)->JBA			= GetBits8(*((UInt8 *)datastart + 27), 3, 2); 
			((XR_VOIP_Metrics_Block *)blkhdr)->JBRate		= GetBits8(*((UInt8 *)datastart + 27), 5, 4); 
			((XR_VOIP_Metrics_Block *)blkhdr)->Reserved		= (*((UInt8 *)datastart + 28)); 
			((XR_VOIP_Metrics_Block *)blkhdr)->JBNominal	= rtcp_NetToHost16((*((UInt16 *)(datastart + 29))));
			((XR_VOIP_Metrics_Block *)blkhdr)->JBMax		= rtcp_NetToHost16((*((UInt16 *)(datastart + 31))));
			((XR_VOIP_Metrics_Block *)blkhdr)->JBAbsMax		= rtcp_NetToHost16((*((UInt16 *)(datastart + 33))));
			break;
		
		case XR_HR_VOIP_REPORT_BLOCK:	
			((XR_HR_VOIP_Block *)blkhdr)->xrblkhdr.BlockType	= ((XR_HR_VOIP_Block *)xrhdr)->xrblkhdr.BlockType;
			((XR_HR_VOIP_Block *)blkhdr)->Map					= ((XR_HR_VOIP_Block *)xrhdr)->Map;					
			((XR_HR_VOIP_Block *)blkhdr)->BlockLength			= 4 * (  rtcp_NetToHost16(((XR_HR_VOIP_Block *)xrhdr)->BlockLength) + 1);
			((XR_HR_VOIP_Block *)blkhdr)->SourceSSRC			= rtcp_NetToHost32(((XR_HR_VOIP_Block *)xrhdr)->SourceSSRC);
			((XR_HR_VOIP_Block *)blkhdr)->Duration				= rtcp_NetToHost32(((XR_HR_VOIP_Block *)xrhdr)->Duration);
			break;

		case XR_HR_CONFIG_REPORT_BLOCK:	
			((XR_HR_Config_Block *)blkhdr)->xrblkhdr.BlockType	= ((XR_HR_Config_Block *)xrhdr)->xrblkhdr.BlockType;
			((XR_HR_Config_Block *)blkhdr)->Map					= ((XR_HR_Config_Block *)xrhdr)->Map;
			((XR_HR_Config_Block *)blkhdr)->BlockLength			= 4 * ( rtcp_NetToHost16(((XR_HR_VOIP_Block *)xrhdr)->BlockLength) + 1);
			((XR_HR_Config_Block *)blkhdr)->SourceSSRC			= rtcp_NetToHost32(((XR_HR_Config_Block *)xrhdr)->SourceSSRC);
			break;

		default:
			// if packet type is not valid, the Packet is assumed to be corrupt
			blkhdr = NULL;
			rval = RTCP_FAILURE;
			break;
			
	}
	return rval;
}


/****************************************************************************
*        NAME:	rtcp_GetXRHRVOIPJBConfig
* DESCRIPTION:	Retrieve the Jitter Buffer Configuration
*      INPUTS:	jb		- Jitter Buffer data
*     OUTPUTS:	cfg		- Type of JB configuration used
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
****************************************************************************/
eRTCP_Ret_Code rtcp_GetXRHRVOIPJBConfig(UInt8 jb, UInt8* cfg)
{
	UInt8 temp; 

	// Validity Check
	if(NULL == cfg)
	{
		return RTCP_FAILURE;
	}

	temp = jb;
	*cfg = GetBits8(temp, 1, 4);
	return RTCP_SUCCESS;
}

/****************************************************************************
*        NAME:	rtcp_GetXRHRVOIPPLCAlg
* DESCRIPTION:	Retrieve the Jitter Buffer Algorithm
*      INPUTS:	jb		- Jitter Buffer data
*     OUTPUTS:	cfg		- Type of JB Algorithm used
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
****************************************************************************/
eRTCP_Ret_Code rtcp_GetXRHRVOIPPLCAlg(UInt8 jb, UInt8* alg)
{
	UInt8 temp; 

	// Validity Check
	if(NULL == alg)
	{
		return RTCP_FAILURE;
	}

	temp = jb;
	*alg = GetBits8(temp, 5, 4);
	return RTCP_SUCCESS;
}

/*******************************************************************************
*        NAME:	rtcp_GetXRHRVOIPSubBlockLenInBytes
* DESCRIPTION:	Retrieve Length of the HR VOIP Sub Block based on the Block Type
*      INPUTS:	subblocktype - Block Type Requested
*     OUTPUTS:	None
*     RETURNS:  Returns the length of the requested Sub Block Type
*    CAUTIONS:	None
*******************************************************************************/
UInt8 rtcp_GetXRHRVOIPSubBlockLenInBytes(UInt8 subblocktype)
{
	if( 	((signed char)subblocktype > XR_HR_VOIP_DELAYPDV_SUBBLOCK) || 
		((signed char)subblocktype < (UInt8)XR_HR_VOIP_BURSTGAP_SUBBLOCK))
	{
		return 0;
	}

	if(subblocktype == XR_HR_VOIP_BASIC_LOSSDISCARD_SUBBLOCK)
		return ( (2 * sizeof(UInt16)) + (1 * sizeof(UInt32)) );
	
	if(subblocktype == XR_HR_VOIP_DELAYPDV_SUBBLOCK)
		return ( (2 * sizeof(UInt8)) + (11 * sizeof(UInt16)) );

	if(subblocktype == XR_HR_VOIP_BURSTGAP_SUBBLOCK)
		return ( (1 * sizeof(UInt8)) + (3 * sizeof(UInt16)) + (1 * sizeof(UInt32)) );

	if(subblocktype == XR_HR_VOIP_PLAYOUT_SUBBLOCK)
		return (4 * sizeof(UInt32));

	if(subblocktype == XR_HR_VOIP_CONCEALED_SECONDS_SUBBLOCK)
		return ( (2 * sizeof(UInt8)) + (1 * sizeof(UInt16)) + (2 * sizeof(UInt32)) );

	if(subblocktype == XR_HR_VOIP_CALLQUALITY_SUBBLOCK)
		return ( (10 * sizeof(UInt8)) + (5 * sizeof(UInt16)) );

	/*if(subblocktype == XR_HR_VOIP_VENDOR_SUBBLOCK)
		return ( (2 * sizeof(UInt8)) + (1 * sizeof(UInt16)) + (1 * sizeof(UInt32)) );
	*/
	return 0;

}

/********************************************************************************
*        NAME:	rtcp_GetXRHRVOIPSubBlockByType
* DESCRIPTION:	Retrieve the populated XR Sub Block on the Sub Block Type
*      INPUTS:	xrhrblk		- Pointer to the start of XR HR Block of the type
*				  XR_Block_Header
*		subblocktype	- Type of Sub Block requested
*     OUTPUTS:	xrhrblk		- Generic Pointer to the populated Sub Block
*     RETURNS:  eRTCP_Ret_Code	- Indicates if the function was successful
*				  or not. Can be RTCP_FAILURE or RTCP_SUCCESS
*    CAUTIONS:	None
**********************************************************************************/
eRTCP_Ret_Code rtcp_GetXRHRVOIPSubBlockByType(XR_Block_Header* xrhrblk, UInt8* subblk, UInt8 subblocktype)
{
	XR_HR_VOIP_Block*	voiphdr			= NULL;
	UInt16*			temp			= NULL;
	UInt16			temppdv			= 0;
	UInt16			tempvar			= 0;
	UInt16*			datastart16		= NULL;
	UInt32*			datastart32		= NULL;
	UInt8*			datastart8		= NULL;
	UInt16			map			= 0;
	double			tempd			= 0;
	
	// Validity Check
	if((NULL == xrhrblk) || (NULL == subblk))
	{
		return RTCP_FAILURE;
	}

	temp  = (UInt16 *)( (UInt8 *)xrhrblk + sizeof(XR_HR_VOIP_Block));
	map = ((XR_HR_VOIP_Block *)xrhrblk)->Map;
	
	// Per Draft the position of these subblocks are assumed to be in the same order specified
	switch(subblocktype)
	{
		// This is a mandatory block, will not be indicated in map
		case XR_HR_VOIP_BASIC_LOSSDISCARD_SUBBLOCK:
			
			
			datastart16	= temp;
			
			// 0:16 format
			tempvar = rtcp_NetToHost16(*datastart16);
            ((XR_HR_VOIP_BasicLossDiscard *)subblk)->LossProportion		= Convert16bitsToDouble(tempvar, 0, 16);
			
			// 0:16 format
			tempvar = rtcp_NetToHost16(*(datastart16 + 1));
			((XR_HR_VOIP_BasicLossDiscard *)subblk)->DiscardProportion	= Convert16bitsToDouble(tempvar, 0, 16);
			
			datastart32	= (UInt32 *)(datastart16 + 2);
			((XR_HR_VOIP_BasicLossDiscard *)subblk)->NoofframesExpected	= rtcp_NetToHost32(*datastart32);
			break;

		case XR_HR_VOIP_BURSTGAP_SUBBLOCK:
			
			if(!(map & XR_HR_VOIP_BURSTGAP_SUBBLOCK))
				return RTCP_FAILURE;

			temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_BASIC_LOSSDISCARD_SUBBLOCK));
			
			((XR_HR_VOIP_BurstGap *)subblk)->Threshold		= ((XR_HR_VOIP_BurstGap *)temp)->Threshold;
			((XR_HR_VOIP_BurstGap *)subblk)->BurstDuration	= rtcp_NetToHost32(GetBits32( *((UInt32 *)(((XR_HR_VOIP_BurstGap *)temp)->BurstDuration)), 1, 24) );
			((XR_HR_VOIP_BurstGap *)subblk)->GapDuration	= rtcp_NetToHost32(((XR_HR_VOIP_BurstGap *)temp)->GapDuration);
			
			datastart16	= (UInt16 *)(&((XR_HR_VOIP_BurstGap *)subblk)->BurstLoss_DiscProportion);

			// 0:16 format
			tempvar = rtcp_NetToHost16(*datastart16);
			((XR_HR_VOIP_BurstGap *)subblk)->BurstLoss_DiscProportion = Convert16bitsToDouble(tempvar, 0, 16);
           
			// 0:16 format
			tempvar = rtcp_NetToHost16(*(datastart16+1));
           	((XR_HR_VOIP_BurstGap *)subblk)->GapLoss_DiscProportion	= Convert16bitsToDouble(tempvar, 0, 16);
			break;

		case XR_HR_VOIP_PLAYOUT_SUBBLOCK:
			
			if(!(map & XR_HR_VOIP_PLAYOUT_SUBBLOCK))
				return RTCP_FAILURE;

			temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_BASIC_LOSSDISCARD_SUBBLOCK));
			
			if(map & XR_HR_VOIP_BURSTGAP_SUBBLOCK)
				temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_BURSTGAP_SUBBLOCK));		
				
			((XR_HR_VOIP_Playout *)subblk)->OntimePlayoutDur		= rtcp_NetToHost32(((XR_HR_VOIP_Playout *)temp)->OntimePlayoutDur);
			((XR_HR_VOIP_Playout *)subblk)->OntimeActiveSpchPlayoutDur	= rtcp_NetToHost32(((XR_HR_VOIP_Playout *)temp)->OntimeActiveSpchPlayoutDur);
			((XR_HR_VOIP_Playout *)subblk)->LossConcealmentDur		= rtcp_NetToHost32(((XR_HR_VOIP_Playout *)temp)->LossConcealmentDur);
			((XR_HR_VOIP_Playout *)subblk)->BufferAdjConcealmentDur		= rtcp_NetToHost32(((XR_HR_VOIP_Playout *)temp)->BufferAdjConcealmentDur);
			break;

		case XR_HR_VOIP_CONCEALED_SECONDS_SUBBLOCK:

			if (!(map & XR_HR_VOIP_CONCEALED_SECONDS_SUBBLOCK))
				return RTCP_FAILURE;

			temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_BASIC_LOSSDISCARD_SUBBLOCK));
			
			if(map & XR_HR_VOIP_BURSTGAP_SUBBLOCK)
				temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_BURSTGAP_SUBBLOCK));
				
			if(map & XR_HR_VOIP_PLAYOUT_SUBBLOCK)
				temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_PLAYOUT_SUBBLOCK));
				

			((XR_HR_VOIP_ConcealedSec *)subblk)->UnimpairedSec			= rtcp_NetToHost32(((XR_HR_VOIP_ConcealedSec *)temp)->UnimpairedSec);
			((XR_HR_VOIP_ConcealedSec *)subblk)->ConcealedSec			= rtcp_NetToHost32(((XR_HR_VOIP_ConcealedSec *)temp)->ConcealedSec);
			((XR_HR_VOIP_ConcealedSec *)subblk)->SeverelyConcealedSec	= rtcp_NetToHost16(((XR_HR_VOIP_ConcealedSec *)temp)->SeverelyConcealedSec);
			((XR_HR_VOIP_ConcealedSec *)subblk)->Reserved				= ((XR_HR_VOIP_ConcealedSec *)temp)->Reserved;
			((XR_HR_VOIP_ConcealedSec *)subblk)->SCSThreshold			= ((XR_HR_VOIP_ConcealedSec *)temp)->SCSThreshold;
			break;

		case XR_HR_VOIP_DELAYPDV_SUBBLOCK:
				
			temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_BASIC_LOSSDISCARD_SUBBLOCK));
				
			if(map & XR_HR_VOIP_BURSTGAP_SUBBLOCK)
				temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_BURSTGAP_SUBBLOCK));
				
			if(map & XR_HR_VOIP_PLAYOUT_SUBBLOCK)
				temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_PLAYOUT_SUBBLOCK));
				
			if(map & XR_HR_VOIP_CONCEALED_SECONDS_SUBBLOCK)
				temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_CONCEALED_SECONDS_SUBBLOCK));
				
			((XR_HR_VOIP_DelayPDV *)subblk)->NetworkRoundTripDelay		= rtcp_NetToHost16(((XR_HR_VOIP_DelayPDV *)temp)->NetworkRoundTripDelay);
			((XR_HR_VOIP_DelayPDV *)subblk)->EndSystemDelay				= rtcp_NetToHost16(((XR_HR_VOIP_DelayPDV *)temp)->EndSystemDelay);
			((XR_HR_VOIP_DelayPDV *)subblk)->ExternalDelay				= rtcp_NetToHost16(((XR_HR_VOIP_DelayPDV *)temp)->ExternalDelay);
			
			datastart16	= (UInt16 *)(&((XR_HR_VOIP_DelayPDV *)temp)->MeanPDV);
						
			// S11:4 format
			temppdv													= rtcp_NetToHost16(*datastart16);
			((XR_HR_VOIP_DelayPDV *)subblk)->MeanPDV				= Convert16bitsToDouble(temppdv, 1, 4);

			// S11:4 format
			temppdv													= rtcp_NetToHost16(*(datastart16 + 1));
			((XR_HR_VOIP_DelayPDV *)subblk)->PosThresholdPDV		= Convert16bitsToDouble(temppdv, 1, 4);
			
			//8:8 format
			temppdv													= rtcp_NetToHost16(*(datastart16 + 2));
			((XR_HR_VOIP_DelayPDV *)subblk)->PosPDVPercentile		= Convert16bitsToDouble(temppdv, 0, 8);
			
			// S11:4 format
			temppdv													= rtcp_NetToHost16(*(datastart16 + 3));
			((XR_HR_VOIP_DelayPDV *)subblk)->NegThresholdPDV		= Convert16bitsToDouble(temppdv, 1, 4);
			
			//8:8 format
			temppdv													= rtcp_NetToHost16(*(datastart16 + 4));
			((XR_HR_VOIP_DelayPDV *)subblk)->NegPDVPercentile		= Convert16bitsToDouble(temppdv, 0, 8);
			
			datastart8 = (UInt8 *)(datastart16 + 5);
			((XR_HR_VOIP_DelayPDV *)subblk)->PDVType			= *(datastart8);
			((XR_HR_VOIP_DelayPDV *)subblk)->JBConfig			= *(datastart8 + 1);
			
			datastart16 = (UInt16 *) (datastart8 + 2);
			((XR_HR_VOIP_DelayPDV *)subblk)->JBNominal			= rtcp_NetToHost16(*(datastart16));
			((XR_HR_VOIP_DelayPDV *)subblk)->JBMaximum			= rtcp_NetToHost16(*(datastart16 + 1));
			((XR_HR_VOIP_DelayPDV *)subblk)->JBAbsMax			= rtcp_NetToHost16(*(datastart16 + 2));
			((XR_HR_VOIP_DelayPDV *)subblk)->JBHighWaterMark	= rtcp_NetToHost16(*(datastart16 + 3));
			((XR_HR_VOIP_DelayPDV *)subblk)->JBLowWaterMark		= rtcp_NetToHost16(*(datastart16 + 4));

			break;

		case XR_HR_VOIP_CALLQUALITY_SUBBLOCK:
			
			if (!(map & XR_HR_VOIP_CALLQUALITY_SUBBLOCK))
				return RTCP_FAILURE;
			
			temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_BASIC_LOSSDISCARD_SUBBLOCK));
				
			if(map & XR_HR_VOIP_BURSTGAP_SUBBLOCK)
				temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_BURSTGAP_SUBBLOCK));
				
			if(map & XR_HR_VOIP_PLAYOUT_SUBBLOCK)
				temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_PLAYOUT_SUBBLOCK));
				
			if(map & XR_HR_VOIP_CONCEALED_SECONDS_SUBBLOCK)
				temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_CONCEALED_SECONDS_SUBBLOCK));
				
			temp = (UInt16 *)((UInt8 *)temp + rtcp_GetXRHRVOIPSubBlockLenInBytes(XR_HR_VOIP_DELAYPDV_SUBBLOCK));
			
			datastart16	= (UInt16 *)(&((XR_HR_VOIP_CallQuality *)temp)->R_LQ);
				
			// 8:8 Format
			tempvar												= rtcp_NetToHost16(*datastart16);
			((XR_HR_VOIP_CallQuality *)subblk)->R_LQ			= Convert16bitsToDouble(tempvar, 0, 8);

			// 8:8 Format
			tempvar												= rtcp_NetToHost16(*(datastart16 + 1));
			((XR_HR_VOIP_CallQuality *)subblk)->R_CQ			= Convert16bitsToDouble(tempvar, 0, 8);

			// 8:8 Format
			tempvar												= rtcp_NetToHost16(*(datastart16 + 2));
			((XR_HR_VOIP_CallQuality *)subblk)->MOS_LQ			= Convert16bitsToDouble(tempvar, 0, 8);

			// 8:8 Format
			tempvar												= rtcp_NetToHost16(*(datastart16 + 3));
			((XR_HR_VOIP_CallQuality *)subblk)->MOS_CQ			= Convert16bitsToDouble(tempvar, 0, 8);

			datastart8 = (UInt8 *)(&((XR_HR_VOIP_CallQuality *)temp)->R_LQExtIn);

			// 7:1 Format
			((XR_HR_VOIP_CallQuality *)subblk)->R_LQExtIn			= Convert8bitsToDouble(*datastart8, 0, 1);

			// 7:1 Format
			((XR_HR_VOIP_CallQuality *)subblk)->R_LQExtOut 			= Convert8bitsToDouble(*(datastart8 + 1), 0, 1);


			((XR_HR_VOIP_CallQuality *)subblk)->RFC3550Payload		= * (datastart8 + 2);
			((XR_HR_VOIP_CallQuality *)subblk)->MediaType			= * (datastart8 + 3);

			// S7 Format
			((XR_HR_VOIP_CallQuality *)subblk)->RxSigLevIP			= (Int8) *(datastart8 + 4);

			// S7 Format
			((XR_HR_VOIP_CallQuality *)subblk)->RxNoiseLevIP		= (Int8) *(datastart8 + 5);

			// S7 Format
			((XR_HR_VOIP_CallQuality *)subblk)->LocalRERL			= (Int8) *(datastart8 + 6);

			// S7 Format
			((XR_HR_VOIP_CallQuality *)subblk)->RemoteRERL			= (Int8) *(datastart8 + 7);
			
			// S7 Format
			((XR_HR_VOIP_CallQuality *)subblk)->RxSigLevExt			= (Int8) *(datastart8 + 8);
			
			// S7 Format
			((XR_HR_VOIP_CallQuality *)subblk)->RxNoiseLevExt		= (Int8) *(datastart8 + 9);
			
			datastart16	= (UInt16 *)(((XR_HR_VOIP_CallQuality *)temp)->MetricStatus);
			((XR_HR_VOIP_CallQuality *)subblk)->MetricStatus		= rtcp_NetToHost16(*datastart16);
			
			break;
	
		default:
			return RTCP_FAILURE;
	}
	return RTCP_SUCCESS;
}



