#****************************************************************************
#         DIALOGIC CONFIDENTIAL
# Copyright 2006-2007 Dialogic Corporation All Rights Reserved. 
# The source code contained or described herein and all documents related
# to the source code ("Material") are owned by Dialogic Corporation or its
# suppliers or licensors. Title to the Material remains with Dialogic
# Corporation or its suppliers and licensors. The Material contains trade
# secrets and proprietary and confidential information of Dialogic or its
# suppliers and licensors. The Material is protected by worldwide copyright
# and trade secret laws and treaty provisions. No part of the Material may
# be used, copied, reproduced, modified, published, uploaded, posted,
# transmitted, distributed, or disclosed in any way without Dialogic's prior
# express written permission.
# No license under any patent, copyright, trade secret or other intellectual
# property right is granted to or conferred upon you by disclosure or
# delivery of the Materials, either expressly, by implication, inducement,
# estoppel or otherwise. Any license under such intellectual property rights
# must be express and approved by Dialogic in writing.
#***************************************************************************
#****************************************************************************
# FILE:        amc.tcl
# PURPOSE:     This file contains procedures for tvl2_dbg program.
#
# NOTE: Do not use curly brackets inside this file header section. TCL
#       will consider the curly brackets as a start of a statement.
#
# HISTORY:
# Date         Description                                         Who
# 09/12/08     Merger: Added switchMACAdd9x						   Ray Bailey
# 02/22/06     Initial Creation                                    Mark Buser
# 07/26/07     Add support for Zionsville MMA card.                Naum Shapiro
# 09/07/07     Added support for Port Monitoring on 98DX106        Luke Kiernan
#              Added support for turning Port Monitoring OFF on
#              both the 98DX106 and 88E6185
#***************************************************************************

load ./libtvl2_dbg.so tvl2_dbg

proc amc_up {} {

    switch_reset
    phy_errata
    phy_config
    switch_config
}

proc phy_errata {} {
    global TVL2_HW_AMC_PHY1  TVL2_HW_AMC_PHY2  TVL2_HW_AMC_PHY3
    set p [new_uintp]

    puts "Apply PHY errata"
    for {set phy 1} {$phy <= 3} {incr phy} {
        puts [format "PHY%d" $phy]
        puts [format "Writing PHY%d 29 with 0" $phy]
        eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 29 0x0000
    
        puts [format "Writing PHY%d 30 with 0x0040" $phy]
        eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 30 0x0040
    
        puts [format "Writing PHY%d 30 with 0x0000" $phy]
        eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 30 0x0000
    
        puts [format "Writing PHY%d 00 with 0x8140" $phy]
        eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 00 0x8140
    
        eval tvl2_hw_read \$TVL2_HW_AMC_PHY$phy 0 20 $p
        set reg [uintp_value $p]
        puts [format "Reading PHY%d 20 got 0x%x" $phy $reg]
        set reg [expr $reg & ~0x20]
        set reg [expr $reg | 0x10]
        puts [format "Writing PHY%d 20 with 0x%x" $phy $reg]
        eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 20 $reg
    
        eval tvl2_hw_read \$TVL2_HW_AMC_PHY$phy 0 00 $p
        set reg [uintp_value $p]
        puts [format "Reading PHY%d 00 got 0x%x" $phy $reg]
        set reg [expr $reg | 0x4000]
        puts [format "Writing PHY%d 00 with 0x%x" $phy $reg]
        eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 00 $reg
        puts [format "PHY%d DONE" $phy]
    }
    delete_uintp $p
}

proc phy_config {} {
    global TVL2_HW_AMC_PHY1  TVL2_HW_AMC_PHY2  TVL2_HW_AMC_PHY3
    set p [new_uintp]
    
    puts "Apply PHY config"
    for {set phy 1} {$phy <= 3} {incr phy} {
        #puts [format "Writing PHY%d 22 with 0x%x (Fiber Page)" $phy 1]
        #puts [eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 22 1]

        # clear loopback
        eval tvl2_hw_read \$TVL2_HW_AMC_PHY$phy 0 00 $p
        set ctrlreg [uintp_value $p]
        puts [format "Reading PHY%d 00 got 0x%x" $phy $ctrlreg]
        # clear loopback
        set ctrlreg [expr ($ctrlreg & ~0x4000)]
        puts [format "Writing PHY%d 00 with 0x%x" $phy $ctrlreg]
        eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 00 $ctrlreg

        # LED config
        eval tvl2_hw_read \$TVL2_HW_AMC_PHY$phy 0 24 $p
        set ledreg [uintp_value $p]
        puts [format "Reading PHY%d 24 got 0x%x" $phy $ledreg]
        #set ledreg [expr ($ledreg | 0x0054)]
        set ledreg [expr ($ledreg | 0x0044)]
        puts [format "Writing PHY%d 24 with 0x%x" $phy $ledreg]
        eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 24 $ledreg
        
        # soft reset
        set ctrlreg [expr ($ctrlreg | 0x8000)]
        puts [format "Writing PHY%d 00 with 0x%x" $phy $ctrlreg]
        eval tvl2_hw_write \$TVL2_HW_AMC_PHY$phy 0 00 $ctrlreg

        puts [format "PHY%d CONFIG DONE" $phy]
    }
    delete_uintp $p
}

proc phyLink {} {
    global TVL2_HW_AMC_PHY1
    global TVL2_HW_AMC_PHY2
    global TVL2_HW_AMC_PHY3

    set p [new_uintp]
    set linkDesc [list "NONE" "CCP4" "CCP3" "CCP2"]
    for {set phy 1} {$phy <= 3} {incr phy} {
        eval tvl2_hw_read \$TVL2_HW_AMC_PHY$phy 0 1 $p
        set reg [uintp_value $p]
        puts -nonewline [format "PHY%d (%s) " $phy [lindex $linkDesc $phy]]
        if {($reg & 0x0004)} {
            puts "up"
        } else {
            puts "down "
        }
    }
}

proc switch_reset {} {
    global TVL2_HW_AMC_RCR
    puts "Resetting I2C/PHY/Switch"
    tvl2_hw_write $TVL2_HW_AMC_RCR 0 0 0x00
    tvl2_hw_write $TVL2_HW_AMC_RCR 0 0 0x30
    after 10
}

proc switch_config {} {
    global TVL2_HW_AMC_MDIO
    set p [new_uintp]

    if {[switchID] != "88E6185"} {
        error "Only 88E6185 switch supported"
    }

    puts "Set Switch Config"
    set portDesc [list "CCP4" "CCP3" "CCP2" "Fabric 1" \
                 "Fabric 2" "RTM 1" "RTM 2" "Ophir 1" "Ophir 2" "CCP1"]
    for {set port 0} {$port < 10} {incr port} {
        if {$port <= 2} {
            tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$port] 1 0x3d
        } elseif {$port == 3 || $port == 4 || $port == 7 || $port == 8} {
            tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$port] 1 0x0602
            after 50
            tvl2_hw_read $TVL2_HW_AMC_MDIO [expr 0x10+$port] 1 $p
            set portStat [uintp_value $p]
            puts [format "DONE: 0x%x" [expr $portStat & 0x0100]]
            #tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$port] 1 0x0433
            tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$port] 1 0x0432
        } elseif {$port == 9} {
            #tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$port] 1 0x0033
            tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$port] 1 0x0031
        }
        tvl2_hw_read $TVL2_HW_AMC_MDIO [expr 0x10+$port] 0 $p
        set portStat [uintp_value $p]
        puts -nonewline [format "Port %d (%s) " $port [lindex $portDesc $port]]
        if {($portStat & 0x0800)} {
            puts "up"
        } else {
            puts "down "
        }
    }
    delete_uintp $p
}

proc ifconfig {port enable} {
    global TVL2_HW_AMC_MDIO

    set p [new_uintp]

    set switchType [switchID]

    if {$switchType == "88E6185"} {
        tvl2_hw_read $TVL2_HW_AMC_MDIO [expr 0x10+$port] 4 $p
        set portControl [uintp_value $p]
        if { [string match -nocase "up" $enable] == 1 } {
            set portControl [expr ($portControl & ~0x0003) | 0x0003]
            tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$port] 4 $portControl
        } elseif { [string match -nocase "down" $enable] == 1 } {
            set portControl [expr ($portControl & ~0x0003)]
            tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$port] 4 $portControl
        } else {
            delete_uintp $p
            error "Expecting one of up/down, not '$enable'"
        }
        delete_uintp $p
    } elseif {$switchType == "9XDX106"} {
        if {$port < 9} {
            set portControl [read9x [expr 0x0A800000 + ($port*0x100)] 0]
        } else {
            set portControl [read9x 0x0A803F00 0]
        }

        if { [string match -nocase "up" $enable] == 1 } {
            set portControl [expr ($portControl & ~0x0001) | 0x0001]
        } elseif { [string match -nocase "down" $enable] == 1 } {
            set portControl [expr ($portControl & ~0x0001)]
        } else {
            error "Expecting one of up/down, not '$enable'"
        }

        if {$port < 9} {
            write9x [expr 0x0A800000 + ($port*0x100)] $portControl
        } else {
            write9x 0x0A803F00 $portControl
        }
    } else {
        error "switch $switchType not supported"
    }
}

proc switchLink {} {
    global TVL2_HW_AMC_MDIO

    set switchType [switchID]
    set p [new_uintp]
    set portDesc [list "CCP4" "CCP3" "CCP2" "FAB1" \
                 "FAB2" "RTM1" "RTM2" "CPU1" "CPU2" "CCP1"]

    for {set port 0} {$port < 10} {incr port} {
        if {$switchType == "88E6185"} {
            tvl2_hw_read $TVL2_HW_AMC_MDIO [expr 0x10+$port] 0 $p
            set portStat [uintp_value $p]
            if {($portStat & 0x0800)} {
                set link "up"
            } else {
                set link "down"
            }
            tvl2_hw_read $TVL2_HW_AMC_MDIO [expr 0x10+$port] 4 $p
            set portControl [uintp_value $p]
            if {($portControl & 0x0003) == 0x0003} {
                set control "enabled"
            } else {
                set control "disabled"
            }
            puts [format "Port %d (%4s) %4s %8s" \
                         $port [lindex $portDesc $port] $link $control]
        } elseif {$switchType == "9XDX106"} {
            if {$port < 9} {
                set portStat [read9x [expr 0x0A800010 + ($port*0x100)] 0]
            } else {
                set portStat [read9x 0x0A803F10 0]
            }
            if {($portStat & 0x0001)} {
                set link "up"
            } else {
                set link "down"
            }
            if {$port < 9} {
                set portControl [read9x [expr 0x0A800000 + ($port*0x100)] 0]
            } else {
                set portControl [read9x 0x0A803F00 0]
            }
            if {($portControl & 0x0001)} {
                set control "enabled"
            } else {
                set control "disabled"
            }
            puts [format "Port %d (%4s) %4s %8s" \
                         $port [lindex $portDesc $port] $link $control]
        } else {
            error "Unsupported switch $switchType"
        }
    }
}

proc cfgMonitorToPort9x { portNum dir } {
    #
    # Configure the "To" Monitor Port
    # This is done per Configuration guidelines at end of 
    # Section 16.2.1 of 9X Switch Functional Spec
    #
    # Set the Analyser Port Configuration Register
    # Table 453 p.709
    # Register Offset = 0x0B000008
    # Egress Port Number is specified at bit 5
    # Ingress Port is specified at bit 16
    #
    # Zero out the existing register since we do not allow monitoring
    # of more than one port
    set AnalyzerPortCfgReg 0
    
    # 
    # If egress or both set the appripriate bits
    #
    if { ($dir == "egress") || ($dir == "both") } {
        # add the new port number
        set AnalyzerPortCfgReg [expr ($AnalyzerPortCfgReg | (($portNum & 0x3F) << 5))]
    }

    # 
    # If ingress or both set the appripriate bits
    #
    if { ($dir == "ingress") || ($dir == "both") } {
        # add the new port number
        set AnalyzerPortCfgReg [expr ($AnalyzerPortCfgReg | (($portNum & 0x3F) << 16))]
    }

    # write to device
    write9x 0x0B000008 $AnalyzerPortCfgReg
    puts [format "<0x0B000008> <== 0x%x" $AnalyzerPortCfgReg]
    puts "Analyzer Port Configuration Complete"

    # read back as debug
    set AnalyzerPortCfgReg [read9x 0x0B000008 0]
    puts [format "<0x0B000008> ==> 0x%x" $AnalyzerPortCfgReg]
}

proc cfgMonitorFromPortIngress9x { portNum dir } {  
    # 
    # Ingress monitoring
    # This is done per Configuration guidelines at end of 
    # Section 16.2.3.1 of 9X Switch Functional Spec
    #
    # Need to set or reset the <MirrorToIngressAnalyzerPort> bit
    # (bit 23 of Word0) in the Port<n> VLAN and Qos Config Entry
    # Table 312 p.567
    # To do this we need to use protocol defined in 
    # C.13.1.4 and C.13.1.5 using Table 314 p.575
    #
    # First we read the existing entry values for this port
    # Then we OR in the new setting and write it back to the
    # switch
    #
    # Read protocol
    #
    set actionType 0x1
    set rdwr  0
    set trigger 1
    set writeVal [expr ($actionType << 10) | (($portNum & 0x3F) << 2) | ($rdwr << 1) | $trigger] 
    write9x 0x0B800328 $writeVal
    puts [format "<0x0B800328> <== 0x%x" $writeVal]
    set count 1
    while {($trigger == 1) && ($count < 50)} {
      set control [read9x 0x0B800328 0]
      set trigger [expr $control & 0x1]
      incr count 1
    }
    if {$count >= 50} {
        error "Timeout Configuring Ingress Port"
    }            
    set PortVQCfgRegWord0 [read9x 0x0B800320 0]
    puts [format "<0x0B800320> ==> 0x%x" $PortVQCfgRegWord0]
    set PortVQCfgRegWord1 [read9x 0x0B800324 0]
    puts [format "<0x0B800324> ==> 0x%x" $PortVQCfgRegWord1]

    #
    # Then depending on if we want to tuen on or off monitoring
    # we OR in or mask out the appropriate bit and write it out
    # Need to set/reset bit 23 of Word0 to 1
    #
    if { $dir == "off" } {
        set PortVQCfgRegWord0 [expr ($PortVQCfgRegWord0 &~ (0x01 << 23))]
    } else {
        set PortVQCfgRegWord0 [expr ($PortVQCfgRegWord0 | (0x01 << 23))]
    }
    write9x 0x0B800320 $PortVQCfgRegWord0
    puts [format "<0x0B800320> <== 0x%x" $PortVQCfgRegWord0]
    write9x 0x0B800324 $PortVQCfgRegWord1
    puts [format "<0x0B800324> <== 0x%x" $PortVQCfgRegWord1]
    set actionType 0x1
    set rdwr  1
    set trigger 1
    set writeVal [expr ($actionType << 10) | (($portNum & 0x3F) << 2) | ($rdwr << 1) | $trigger] 
    write9x 0x0B800328 $writeVal
    puts [format "<0x0B800328> <== 0x%x" $writeVal]
    set count 1
    while {($trigger == 1) && ($count < 50)} {
        set control [read9x 0x0B800328 0]
        set trigger [expr $control & 0x1]
        incr count 1
    }
    if {$count >= 50} {
        error "Timeout Configuring Ingress Port"
    }
    puts "Ingress Configuration Complete"
}
    
proc cfgMonitorFromPortEgress9x { portNum dir } {  
    # 
    # Egress monitoring
    # This is done per Configuration guidelines at end of 
    # Section 16.2.4.1 of 9X Switch Functional Spec
    #
    # First need to set/reset the right bit in the 
    # Egress Monitor Enable Configuration Register
    # Table 454 p.709
    # For this we read the existing register value
    # and OR in or MASK out the new bit
    #
    set EgressMonEnCfgReg [read9x 0x0B00000C 0]
    puts [format "<0x0B00000C> ==> 0x%x" $EgressMonEnCfgReg]
    if { $dir == "off" } {
        set EgressMonEnCfgReg [expr ($EgressMonEnCfgReg & ~(1 << $portNum))]
    } else {
        set EgressMonEnCfgReg [expr ($EgressMonEnCfgReg | (1 << $portNum))]
    }
    write9x 0x0B00000C $EgressMonEnCfgReg
    puts [format "<0x0B00000C> <== 0x%x" $EgressMonEnCfgReg]

    # read it back for debug
    set EgressMonEnCfgReg [read9x 0x0B00000C 0]
    puts [format "<0x0B00000C> ==> 0x%x" $EgressMonEnCfgReg]

    #
    # Next we configure the egress port to start or stop 
    # mirroring all packets to the egress analyser (monitor)
    # This is done by setting/resetting the 
    # <EgressMirror to Analyser En> bit in the 
    # Port<n>TxqConfigRegister
    # Table 466 p.722
    # Offset = 0x01800080 + (portNum * 0x200)
    # Bit to set/reset is bit 20
    #
    set offset [expr (0x01800080 + ($portNum * 0x200))]
    set PortTxqCfgReg [read9x $offset 0]
    puts [format "<0x%08x> ==> 0x%x" $offset $PortTxqCfgReg]
    if { $dir == "off" } {
        set PortTxqCfgReg [expr ($PortTxqCfgReg & ~(0x1 << 20))]
    } else {
        set PortTxqCfgReg [expr ($PortTxqCfgReg | (0x1 << 20))]
    }
      
    write9x $offset $PortTxqCfgReg
    puts [format "<0x%08x> <== 0x%x" $offset $PortTxqCfgReg]

    # read back for debug
    set PortTxqCfgReg [read9x $offset 0]
    puts [format "<0x%08x> ==> 0x%x" $offset $PortTxqCfgReg]
    puts "Egress Configuration Complete"
}
      
proc cfgMonitorFromPort9x { portNum dir } {    
    #
    # Configure the Monitor "From" Port
    # This configuration is done differently depending on 
    # whether or not we are monitoring Ingress or Egress traffic
    #
    if {($dir == "ingress") || ($dir == "both") || ($dir == "off")} {
        cfgMonitorFromPortIngress9x $portNum $dir
    }
    if {($dir == "egress") || ($dir == "both") || ($dir == "off")} {
        cfgMonitorFromPortEgress9x $portNum $dir
    }
    puts "Source Port Configuration Complete"
}

proc switchMonitor { from to {dir egress} } {
    switch [switchID] {
        9XDX106 { switchMonitor9x  $from $to $dir  }
        88E6185 { switchMonitor88  $from $to $dir  }
        default { error "Unknown switch type '[switchID]' unsupported" }
    }
}
                         
proc switchMonitor9x { from to {dir egress} } { 

    # 
    # Check that the input parameters are valid
    # For 98DX106 the port number can be 0 - 9
    #
    if {$from < 0 || $from > 9} {
        error "monitored port must be 0 >= from <= 9"
    }
    if {$to < 0 || $to > 9} {
        error "monitor port must be 0 >= to <= 9"
    }
    puts [format "Monitor From Port %d To Port %d" $from $to]
    puts [format "Direction = %s" $dir]
    if {($dir ne "egress") && ($dir ne "ingress") && ($dir ne "both") && ($dir ne "off")} {
        error "monitor dir must be one of: ingress, egress, both, off"
    }

    # Configure the monitoring "To" port
    cfgMonitorToPort9x $to $dir
                            
    # Configure the monitoring "From" port
    cfgMonitorFromPort9x $from $dir
}

proc switchMonitor88 { from to {dir egress} } {
    global TVL2_HW_AMC_MDIO

    if {$from < 0 || $from > 9} {
        error "monitored port must be 0 >= from <= 9"
    }
    if {$to < 0 || $to > 9} {
        error "monitor port must be 0 >= to <= 9"
    }
    puts $dir
    if {($dir ne "egress") && ($dir ne "ingress") && ($dir ne "both") && ($dir ne "off") } {
        error "monitor dir must be one of: ingress, egress, both, off"
    }
    set p [new_uintp]

    # maybe we don't need this
    if {0} {
        # Remove monitor destination port from normal traffic
        tvl2_hw_read $TVL2_HW_AMC_MDIO [expr 0x10+$to] 6 $p
        set vlanMap [uintp_value $p]
        puts [format "<0x%x>:<0x06> ==> 0x%x" [expr 0x10+$to] $vlanMap]
        set vlanMap [expr ($vlanMap & ~0x03FF) | (1<<$to)]
        tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$to] 6 $vlanMap
        puts [format "<0x%x>:<0x06> <== 0x%x" [expr 0x10+$to] $vlanMap]
    }

    # 
    # Set ingress/egress monitor destination
    # See Table 89 p.172 of Functional Spec
    #
    tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1B 0x1a $p
    set monControl [uintp_value $p]
    puts [format "<0x1B>:<0x1A> ==> 0x%x" $monControl]

    if {$dir == "off"} {
        set monControl [expr ($monControl & 0x00FF) & ~((0xF)<<12)]
        set monControl [expr ($monControl & 0x00FF) & ~((0xF)<<8)]
    } else {
        set monControl [expr ($monControl & 0x00FF) | (($to&0xF)<<12) | (($to&0xF)<<8)]
    }
    tvl2_hw_write $TVL2_HW_AMC_MDIO 0x1B 0x1a $monControl
    puts [format "<0x1B>:<0x1A> <== 0x%x" $monControl]

    # 
    # Set port monitor enable
    # See Table 53 p.141 of Functional Spec
    #
    tvl2_hw_read $TVL2_HW_AMC_MDIO [expr 0x10+$from] 0x8 $p
    set portControl2 [uintp_value $p]
    puts [format "<0x%x>:<0x08> ==> 0x%x" [expr 0x10+$from] $portControl2]

    # 
    # First we turn off monitoring on port
    # If we have to enable then set the appropriate bits
    #
    set portControl2 [expr ($portControl2 & ~0x0030)]
    if {$dir == "egress"} {
        set portControl2 [expr ($portControl2 | 0x0020) ]
    } elseif {$dir == "ingress"} {
        set portControl2 [expr ($portControl2 | 0x0010) ]
    } elseif {$dir == "both"} {
        set portControl2 [expr ($portControl2 | 0x0030) ]
    }
    tvl2_hw_write $TVL2_HW_AMC_MDIO [expr 0x10+$from] 0x8 $portControl2
    puts [format "<0x%x>:<0x08> <== 0x%x" [expr 0x10+$from] $portControl2]

}

proc switchID {} {
    set p [read9x 0x4C 0]
    if {($p & 0xFFFF0) == 0xD3D30} {
        puts "Switch 9xDx106"
        return "9XDX106"
    } else {
        global TVL2_HW_AMC_MDIO

        set p [new_uintp]
        tvl2_hw_read $TVL2_HW_AMC_MDIO 0x10 0x3 $p
        if {([uintp_value $p] & 0xFFF0) == 0x01A70} {
            puts "Switch 88E6185"
            return "88E6185"
        }
    }
    error "Unknown switch type"
}

proc portCount {port} {
    switch [switchID] {
        9XDX106 {portCount9x $port}
        88E6185 {portCount88 $port}
        default { error "Unknown switch type '[switchID]' unsupported" }
    }
}

proc portCount88 {port} {
    set p [new_uintp]
    set statReg [list \
         "InGoodOctetsLo"  "InGoodOctetsHi" "InBadOctets"    "OutFCSErr" \
         "InUnicase"       "Deferred"       "InBroadcasts"   "InMulticasts" \
         "64 Octets"       "65-127 Octets"  "128-255 Octets" "256-511 Octets" \
         "512-1023 Octets" "1024+ Octets"   "OutOctetsLo"    "OutOctetsHi" \
         "OutUnicast"      "Excessive"      "OutMulticasts"  "OutBroadcasts" \
         "Single"          "OutPause"       "InPause"        "Multiple" \
         "InUndersize"     "InFragments"    "InOversize"     "InJabber" \
         "InRxErr"         "InFCSErr"       "Collisions"     "Late"]

    global TVL2_HW_AMC_MDIO


    # capture all counters for port 'port'
    tvl2_hw_write $TVL2_HW_AMC_MDIO 0x1b 0x1d [expr 0xdc00 | $port]
    after 1
    tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 0x1d $p
    if {([uintp_value $p] & 0x8000)} {
        error [format "Port Statistics Unit Busy on CAPTURE (0x%x)" [uintp_value $p]]
    }
    # clear all counters for next time
    tvl2_hw_write $TVL2_HW_AMC_MDIO 0x1b 0x1d [expr 0xac00 | $port]
    after 1
    tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 0x1d $p
    if {([uintp_value $p] & 0x8000)} {
        error [format "Port Statistics Unit Busy on CLEAR (0x%x)" [uintp_value $p]]
    }

    puts [format "Port %d Statistics" $port]

    # read and print each counter in turn
    for {set reg 0} {$reg < [llength $statReg]} {incr reg} {
        tvl2_hw_write $TVL2_HW_AMC_MDIO 0x1b 0x1d [expr 0xcc00 | $reg]
        tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 0x1d $p
        if {([uintp_value $p] & 0x8000)} {
            error "Port Statistics Unit Busy on READ"
        }
        tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 0x1e $p
        set val [uintp_value $p]
        tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 0x1f $p
        set val [expr ($val << 16) | [uintp_value $p]]
        puts [format "%20s 0x%08x" [lindex $statReg $reg] $val]
    }
}

proc portCount9x {port} {
    set statReg [list \
         "InGoodOctetsLo"  "InGoodOctetsHi" "InBadOctets"    "OutFCSErr" \
         "InUnicase"       "Deferred"       "InBroadcasts"   "InMulticasts" \
         "64 Octets"       "65-127 Octets"  "128-255 Octets" "256-511 Octets" \
         "512-1023 Octets" "1024+ Octets"   "OutOctetsLo"    "OutOctetsHi" \
         "OutUnicast"      "Excessive"      "OutMulticasts"  "OutBroadcasts" \
         "Single"          "OutPause"       "InPause"        "Multiple" \
         "InUndersize"     "InFragments"    "InOversize"     "InJabber" \
         "InRxErr"         "InFCSErr"       "Collisions"     "Late"]
 
    if {$port <= 5 } {
        write9x 0x04004020  [expr (0x3f00 + ((($port-0)<<1) & 0x0e ) | 1 )]
    } elseif {$port < 9} {
        write9x 0x04804020  [expr (0x3f00 + ((($port-6)<<1) & 0x0e ) | 1 )]
    }
 
    after 100

    puts [format "Port %d Statistics" $port]
    if { $port < 9 } {
        for {set reg 0} {$reg < [llength $statReg]} {incr reg} {
            if {$port <=5} {
                set p [read9x [expr {0x04010300+ $reg*4} ] 0]
 
            }
            if {$port < 9 && $port >=6} {
                set p [read9x [ expr {0x04810300 + $reg*4} ] 0]
            }
 
         puts [format "%20s 0x%08x"  [lindex $statReg $reg] $p]
      }
   }
 
 
    if {$port == 0x3f || $port == 9} {
       puts [ format "Good frames Sent           :    %x " [read9x  0x60 0]]
       puts [ format "MAC TransError frames sent :    %x " [read9x  0x64 0]]
       puts [ format "Good octets sent           :    %x " [read9x  0x68 0]]
       puts [ format "Good frames received       :    %x " [read9x  0x70 0]]
       puts [ format "Bad frames received        :    %x " [read9x  0x74 0]]
       puts [ format "Good octets received       :    %x " [read9x  0x78 0]]
       puts [ format "Bad octets received        :    %x " [read9x  0x7C 0]]
       puts [ format "RX internal packet drop    :    %x " [read9x  0x6c 0]]
    }
    if {$port > 0x9 && $port !=0x3f} {
        error "Port not supported by 98DX106 "
    }
}

proc switchMACDump {} {
    switch [switchID] {
        9XDX106 {switchMACDump9x}
        88E6185 {switchMACDump88}
        default { error "Unknown switch type '[switchID]' unsupported" }
    }
}

proc switchMACAdd9x {port mac {cmd add} } {
    # TODO: Set port into right fields
    if {[regexp {([[:xdigit:]]+):([[:xdigit:]]+):([[:xdigit:]]+):([[:xdigit:]]+):([[:xdigit:]]+):([[:xdigit:]]+)} $mac match mac5 mac4 mac3 mac2 mac1 mac0] != 1} {
        error "Could not parse mac '$mac'"
    }
    if {$port < 9} {
        set SrcId $port
        set PortNum $port
    } else {
        # CPU port is special
        set SrcId 27
        set PortNum 63
    }
    set MACUpdateWord0 [expr (0x$mac1<<24) | (0x$mac0<<16)]
    set MACUpdateWord1 [expr (0x$mac5<<24) | (0x$mac4<<16) | (0x$mac3<<8) | 0x$mac2]
    if { ($cmd == "remove") } {
                                 # Skip      PortNum        VID
        set MACUpdateWord2 [expr  (1<<12) | ($PortNum<<18) | 1]
    } else {
                                 # PortNum        VID
        set MACUpdateWord2 [expr  ($PortNum<<18) | 1]
    }
                        # Static       SrcID
    set MACUpdateWord3 [expr (1<<18) | ($SrcId<<2)]

    puts [format "0x%.8x 0x%.8x 0x%.8x 0x%.8x" \
          $MACUpdateWord0 $MACUpdateWord1 $MACUpdateWord2 $MACUpdateWord3]

    write9x 0x06000040 $MACUpdateWord0
    write9x 0x06000044 $MACUpdateWord1
    write9x 0x06000048 $MACUpdateWord2
    write9x 0x0600004C $MACUpdateWord3

    write9x 0x06000050 1
}


proc switchMACDump9x {} {
 
    global TVL2_HW_AMC_MDIO
    set p [new_uintp]
 
    for {set entry 0} {$entry < 8192} {incr entry} {
        set trigger 1
        set count 1
        write9x 0x6000064 [expr ($entry << 2) | $trigger]
        while {$trigger == 1 && $count < 50} {
            set control [ read9x 0x6000064 0]
            set trigger [expr $control & 1]
            incr count 1
        }
        if {$count >= 50} {
            error "FBU Busy Timeout"
        }
        set FDBEntryWord0 [read9x 0x06000054 0]

        if {$FDBEntryWord0 & 1} {

            set FDBEntryWord1 [read9x 0x06000058 0]
            set FDBEntryWord2 [read9x 0x0600005C 0]
            set FDBEntryWord3 [read9x 0x06000060 0]

            switch [expr (($FDBEntryWord0 >> 3) & 0x3)] {
                0 {
                  # MAC Entry
                  set mac0 [expr ((($FDBEntryWord2 & 1) << 15) | ($FDBEntryWord1 >> 17) & 0xFFFF)]
                  set mac1 [expr ((($FDBEntryWord1 & 0x1FFFE) >> 1) & 0xFFFF)]
                  set mac2 [expr ((($FDBEntryWord1 & 1) << 15) | ($FDBEntryWord0 >> 17) & 0xFFFF)]
                  puts [format "Entry %.4x %.8x %.8x %.8x %.8x" \
                        $entry $FDBEntryWord0 $FDBEntryWord1 $FDBEntryWord2 $FDBEntryWord3]
                  puts -nonewline [format "MAC %.4x%.4x%.4x " $mac0 $mac1 $mac2]
                  puts -nonewline [format "VID %d " [expr ($FDBEntryWord0 >> 5) & 0x0FFF]]
                  puts -nonewline [format "SrcID %d " [expr ($FDBEntryWord2 >> 6) & 0x001F]]
                  puts [format "DevID %d " [expr ($FDBEntryWord2 >> 1) & 0x001F]]
                }
                1 {
                  # IPv4 multicast
                }
                2 {
                  # IPv6 multicast
                }
                default {
                }
            }
        }
        
    }
}


proc switchMACDump88 {} {
 
    global TVL2_HW_AMC_MDIO
    set p [new_uintp]
 
    set atu_mac0 0xFFFF
    set atu_mac1 0xFFFF
    set atu_mac2 0xFFFF
 
    set entryState 1
    set entry 1
    while {($entryState & 0x000F) != 0} {
        if {$entry == 1} {
            tvl2_hw_write $TVL2_HW_AMC_MDIO 0x1b 13 $atu_mac0
            tvl2_hw_write $TVL2_HW_AMC_MDIO 0x1b 14 $atu_mac1
            tvl2_hw_write $TVL2_HW_AMC_MDIO 0x1b 15 $atu_mac2
        }
 
        tvl2_hw_write $TVL2_HW_AMC_MDIO 0x1b 11 0xc000
        set atu_op 0
        set count 0
        while {($atu_op & 0x8000) == 0 && $count < 50} {
            tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 11 $p
            set atu_op [uintp_value $p]
            incr count 1
        }
        if {$count >= 50} {
            error "ATU Busy Timeout"
        }
        tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 12 $p
        set atu_data [uintp_value $p]
        set entryState [expr $atu_data & 0x000F]
        if {($entryState & 0x000F) == 0} {
            break
        }
 
        puts -nonewline [format "entry %2d port map " $entry]
        for {set i 9} {$i >= 0} {incr i -1} {
            puts -nonewline [format "%d" \
              [expr ($atu_data >> (4+$i)) & 0x1 ]]
        }
        puts [format "(0x%.4x) MAC %.4x%.4x%.4x" \
              $atu_data $atu_mac0 $atu_mac1 $atu_mac2]
 
        tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 13 $p
        set atu_mac0 [uintp_value $p]
        tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 14 $p
        set atu_mac1 [uintp_value $p]
        tvl2_hw_read $TVL2_HW_AMC_MDIO 0x1b 15 $p
        set atu_mac2 [uintp_value $p]
 
        incr entry
    }
    delete_uintp $p
}



proc addressOf { device offset {inst 0}  } {

    set matched [list]
    set devs [lsort -ascii [info vars "::TVL2_HW_\[AR\]*"]]
    lappend devs [info vars ::TVL2_HW_PRODUCT_TYPE ]
    foreach d $devs {
        if { [string match -nocase "::$device" $d] == 1 } {
            lappend matched $d
            break
        }
        if { [string match -nocase "*$device*" "$d"] == 1 } {
            lappend matched $d
        }
    }
    set numMatched [llength $matched]
    if {$numMatched == 0} {
        puts "Unrecognized device '$device', expected one of:" 
        foreach d [lsort -ascii [info vars "::TVL2_HW_\[AR\]*"]] { 
            puts $d  
        }        
        foreach d [lsort -ascii [info vars "::TVL2_HW_PRODUCT_TYPE"]] { 
            puts $d  
        }        
        return
    } elseif {$numMatched != 1} {
        puts "'$device' was not unique, matched $matched"  
        return
    }

    set p [new_uintp]
    set result [eval tvl2_hw_address \$$matched $inst [voidpp $p]]
    if {$result} {
        error [format "Address failed: %s" [tvl2_hw_errstr $result]]
    } else {
        puts [format "0x%x" [expr [uintp_value $p] + $offset]]
        return [expr [uintp_value $p] + $offset]
    }
    del_uintp $p
}
    
proc read { device offset {inst 0}  } {

    set matched [list]
    set devs [lsort -ascii [info vars "::TVL2_HW_\[AR\]*"]]
    lappend devs [info vars ::TVL2_HW_PRODUCT_TYPE ]
    foreach d $devs {
        if { [string match -nocase "::$device" $d] == 1 } {
            lappend matched $d
            break
        }
        if { [string match -nocase "*$device*" "$d"] == 1 } {
            lappend matched $d
        }
    }
    set numMatched [llength $matched]
    if {$numMatched == 0} {
        puts "Unrecognized device '$device', expected one of:" 
        foreach d [lsort -ascii [info vars "::TVL2_HW_\[AR\]*"]] { 
            puts $d  
        }        
        foreach d [lsort -ascii [info vars "::TVL2_HW_PRODUCT_TYPE"]] { 
            puts $d  
        }
        return
    } elseif {$numMatched != 1} {
        puts "'$device' was not unique, matched $matched"  
        return
    }

    
    set p [new_uintp]
    set result [eval tvl2_hw_read \$$matched $inst $offset $p]
    if {$result} {
        error [format "Read failed: %s" [tvl2_hw_errstr $result]]
    } else {
        puts [format "0x%x" [uintp_value $p]]
        return [uintp_value $p]
    }
    delete_uintp $p
}

proc write { device offset data {inst 0}} {

    set matched [list]
    set devs [lsort -ascii [info vars "::TVL2_HW_\[AR\]*"]]
    lappend devs [info vars ::TVL2_HW_PRODUCT_TYPE ]
    foreach d $devs {
        if { [string match -nocase "::$device" $d] == 1 } {
            lappend matched $d
            break
        }
        if { [string match -nocase "*$device*" "$d"] == 1 } {
            lappend matched $d
        }
    }
    set numMatched [llength $matched]
    if {$numMatched == 0} {
        puts "Unrecognized device '$device', expected one of:" 
        foreach d [lsort -ascii [info vars "::TVL2_HW_\[AR\]*"]] { 
            puts $d  
        }        
        foreach d [lsort -ascii [info vars "::TVL2_HW_PRODUCT_TYPE"]] { 
            puts $d  
        }        
        return
    } elseif {$numMatched != 1} {
        puts "'$device' was not unique, matched $matched"  
        return
    }

    set result [eval tvl2_hw_write \$$matched $inst $offset $data]
    if {$result} {
        error [format "Write failed: %s" [tvl2_hw_errstr $result]]
    }
}

proc read9x { offset {verbose 1} } {
    global TVL2_HW_AMC_MDIO
    global TVL2_HW_PRODUCT_TYPE
    global TVL2_HW_PRODUCT_TYPE_TVL2 TVL2_HW_PRODUCT_TYPE_ZVL

    #puts "read9x: enter"
    set p [new_uintp]
    set mdioAddr [new_uintp]

    eval tvl2_hw_read \$TVL2_HW_PRODUCT_TYPE 0 0 $p
    set productType [uintp_value $p]

    if {$productType == $TVL2_HW_PRODUCT_TYPE_TVL2} {
        set mdioAddr 0
    } elseif {$productType == $TVL2_HW_PRODUCT_TYPE_ZVL} {
        set mdioAddr 1
    } else {
        error "Invalid product type = $productType"
    }

    #puts "read9x: productType = $productType"
    #puts "read9x: mdioAddr    = $mdioAddr"

    tvl2_hw_write $TVL2_HW_AMC_MDIO $mdioAddr 4 [expr ($offset & 0xFFFF0000) >> 16]
    tvl2_hw_write $TVL2_HW_AMC_MDIO $mdioAddr 5 [expr ($offset & 0x0000FFFF)]

    set spin 0
    while { [incr spin 1] < 50 } {
        tvl2_hw_read  $TVL2_HW_AMC_MDIO $mdioAddr 0x1F $p
        set rdDone [expr [uintp_value $p] & 0x0001]
        if {$rdDone == 1 } {
            tvl2_hw_read  $TVL2_HW_AMC_MDIO $mdioAddr 6 $p
            set result [expr ([uintp_value $p] & 0xFFFF) << 16]
            tvl2_hw_read  $TVL2_HW_AMC_MDIO $mdioAddr 7 $p
            set result [expr $result | ([uintp_value $p] & 0xFFFF)]
            if {$verbose} {
                puts [format "0x%x" $result]
            }
            delete_uintp $p
            #puts "read9x: exit"
            return $result
        }
    }
    delete_uintp $p
    error "Timeout"
}

proc write9x { offset data } {
    global TVL2_HW_AMC_MDIO
    global TVL2_HW_PRODUCT_TYPE
    global TVL2_HW_PRODUCT_TYPE_TVL2 TVL2_HW_PRODUCT_TYPE_ZVL

    set p [new_uintp]

    set mdioAddr [new_uintp]

    eval tvl2_hw_read \$TVL2_HW_PRODUCT_TYPE 0 0 $p
    set productType [uintp_value $p]

    if {$productType == $TVL2_HW_PRODUCT_TYPE_TVL2} {
        set mdioAddr 0
    } elseif {$productType == $TVL2_HW_PRODUCT_TYPE_ZVL} {
        set mdioAddr 1
    } else {
        error "Invalid product type = $productType"
    }

    tvl2_hw_write $TVL2_HW_AMC_MDIO $mdioAddr 0 [expr ($offset & 0xFFFF0000) >> 16]
    tvl2_hw_write $TVL2_HW_AMC_MDIO $mdioAddr 1 [expr ($offset & 0x0000FFFF)]

    tvl2_hw_write $TVL2_HW_AMC_MDIO $mdioAddr 2 [expr ($data & 0xFFFF0000) >> 16]
    tvl2_hw_write $TVL2_HW_AMC_MDIO $mdioAddr 3 [expr ($data & 0x0000FFFF)]

    set spin 0
    while { [incr spin 1] < 50 } {
        tvl2_hw_read  $TVL2_HW_AMC_MDIO $mdioAddr 0x1F $p
        set wrDone [expr [uintp_value $p] & 0x0002]
        if {$wrDone == 2 } {
            delete_uintp $p
            return 0
        }
    }
    delete_uintp $p
    error "Timeout"
}

proc readpci { domain bar offset data} {

    if {[regexp {([[:xdigit:]]+):([[:xdigit:]]+).([[:xdigit:]]+)} $domain match bus slot function] != 1} {
        error "Could not parse domain '$domain'"
    }

    set d [new_uintp]
    set err [uPCIMemAccess [expr 0x$bus] $slot $function $bar 1 $offset $d]
    if {$err} {
        delete_uintp $d
        error "Could not read $domain Bar:$bar Offset:$offset"
    } else {
        upvar $data calledData
        set calledData [uintp_value $d]
        puts [format "0x%x" $calledData]
    }
    delete_uintp $d
    return $calledData
}

proc writepci { domain bar offset data} {

    if {[regexp {([[:xdigit:]]+):([[:xdigit:]]+).([[:xdigit:]]+)} $domain match bus slot function] != 1} {
        error "Could not parse domain '$domain'"
    }

    set d [new_uintp]
    uintp_assign $d $data
    set err [uPCIMemAccess [expr 0x$bus] $slot $function $bar 0 $offset $d]
    if {$err} {
        error "Could not write $domain Bar:$bar Offset:$offset"
    }
    delete_uintp $d
}

proc productTypeDisplay {} {
    global TVL2_HW_PRODUCT_TYPE
    global TVL2_HW_PRODUCT_TYPE_TVL2 TVL2_HW_PRODUCT_TYPE_ZVL
    set p [new_uintp]

    eval tvl2_hw_read \$TVL2_HW_PRODUCT_TYPE 0 0 $p
    set productType [uintp_value $p]

    if {$productType == $TVL2_HW_PRODUCT_TYPE_TVL2} {
        puts [format "The product type is Titusville 2"]
    } elseif {$productType == $TVL2_HW_PRODUCT_TYPE_ZVL} {
        puts [format "The product type is Zionsville"]
    } else {
        error "Invalid product type = $productType"
    }
}

proc help {} {
    puts " ifconfig {port enable} port: 0-9, enable: up,down"
    puts " switch_reset {} reset AMC switch & PHYS"
    puts " phyLink {}             show AMC PHY link status"
    puts " switchLink {}          show AMC Switch link status"
    puts " switchMonitor { from to {dir egress} } copy packets on port 'from' to"
    puts "                                        port 't' for dir: ingress,egress,both,off"
    puts " switchID {}            which AMC switch type is installed"
    puts " switchMACDump {}       dump AMC switch learned MAC addresses"
    puts " switchMACAdd9x {port mac {cmd add} }  add/remove MAC/port association from switch port"
    puts "                                        mac '01:02:03:04:05:06'"
    puts "                                        cmd 'add' or 'remove'"
    puts " portCount {port}       dump switch port statistics"
    puts ""
    puts " addressOf { device offset {inst 0}  } return address of device"
    puts " read { device offset {inst 0}  }      read a device location"
    puts " write { device offset data {inst 0}}  write a device location"
    puts " read9x { offset {verbose 1} }         read a 9xdx106 location"
    puts " write9x { offset data }               write a 9xdx106 location"
    puts ""
    puts " writepci { domain bar offset data }   write a PCI location"
    puts " readpci { domain bar offset data }    read a PCI location"
    puts "     domain is of the form \"bus:slot.func\""
    puts "     e.g. \"readpci 7:0.0 2 0 data\" read AMC FPGA ID register"
    puts ""
    puts " productTypeDisplay {}    display the underlying product type"
    puts ""
}

