/**********@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********************************
* Copyright (C) 2001-2013 Dialogic Corporation. All Rights Reserved.
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
* 1.    Redistributions of source code must retain the above copyright
* notice, this list of conditions and the following disclaimer.
*
* 2.    Redistributions in binary form must reproduce the above copyright
* notice, this list of conditions and the following disclaimer in
* the documentation and/or other materials provided with the
* distribution.
*
* 3.    Neither the name Dialogic nor the names of its
* contributors may be used to endorse or promote products derived from this
* software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
***********************************@@@SOFT@@@WARE@@@COPY@@@RIGHT@@@**********/
/**********************************************************************
 * File Name                    : mercd_linux_entry.c
 * Description                  : Driver Entry Points Module
 *
 *
 **********************************************************************/

#ifndef LINUX24
#include <linux/moduleparam.h>
#endif

#include <linux/module.h>
#include <linux/version.h>
#ifdef CONFIG_COMPAT
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
#include <linux/ioctl32.h>
#endif
#include <linux/compat.h>
#endif
#include <linux/timer.h>
#include <linux/proc_fs.h>

#include "msd.h"
#include "mercd.h"
#include "msdpciif.h"
#include "msdversion.ver"

#define MERCD_LINUX_ENTRY
#include "msdextern.h"
#undef MERCD_LINUX_ENTRY
#include "mercd_hsi.h"

static int mercdMajor = 0; /* Till 2.4.2 there is minor no. limit - 256 */

static char mercd_string[]       = "mercd";

static int dm3_feature = 0x302; /* Enable additional features at command line */
int NewCanTakeProtocol = 0; /* Optimizes CanTake protocol in play */
int No_Send_From_DPC = 0;   /* Do not send messages from DPC */
int ReceiveDMA = 0;
int SendDMA = 0;
int SendExitNotify = 0;
int NonDefaultId = 0;
int DPC_Disabled = 0;
int RH_Enabled = 0;
int HMP_Enabled = 0;
int HCS_Flag = 1;
int g_hsi_stub_enabled = 1;
int g_hsi_dpc_enabled = 1;

MODULE_AUTHOR("Dialogic Corp. (c) 2013");
MODULE_DESCRIPTION("DM3 Driver");
#ifdef LINUX24
MODULE_PARM(dm3_feature,"i");
#else
module_param(dm3_feature,int,0);
MODULE_PARM_DESC(mercd_string, "Mercd driver name");
MODULE_LICENSE("Dual BSD/GPL");
#endif
//For Linux kernels that do not support MSI
#ifndef CONFIG_PCI_MSI
#define pci_disable_msi(a)
#define pci_enable_msi(a)
#endif

static int mercd_probe(struct pci_dev *pdev, const struct pci_device_id *ent);
static void mercd_release(struct pci_dev *pdev);
int mercd_resume(struct pci_dev *pdev);
int mercd_suspend(struct pci_dev *pdev);
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,36)
static long mercd_unlocked_ioctl(struct file *, unsigned int, unsigned long);
#endif
#ifdef CONFIG_COMPAT
static int mercd_compat_ioctl(struct file *, unsigned int, unsigned long);
#else
#define mercd_compat_ioctl NULL
#endif

//========================================================================
//
// Set the low/high water marks to allow only one message to be queued
// for the write-side service routine at a time.
//
//========================================================================
#define LO_WATER        1
#define HI_WATER        (48*4*1024)
#define MIN_PSZ         0
#define MAX_PSZ         (16*4*1024)
#define MSD_ID          401

static struct file_operations mercd_fops =
{
        .owner   =    THIS_MODULE,
        .read    =    mercd_read,
        .write   =    mercd_write,
        .poll    =    mercd_poll,
        .open    =    mercd_open,
        .release =    mercd_close,
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,36)
        .unlocked_ioctl = mercd_unlocked_ioctl,
#else
        .ioctl   =    mercd_ioctl,
#endif
#ifdef CONFIG_COMPAT
        .compat_ioctl= mercd_compat_ioctl,
#endif
        .mmap    =    mercd_mmap
};

/*************************************************************************
*
*      Global Allocation
*
*************************************************************************/
pmercd_control_block_sT MsdControlBlock = NULL;

PMSD_MUTEX_STATS pmutex_stats = NULL;


static struct pci_device_id mercd_pci_tbl[] __devinitdata = {
        { PCI_VENDOR_ID_DLGC, PCI_DEVICE_ID_DLGC_906D,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_DM3, },
        { PCI_VENDOR_ID_DLGC, PCI_DEVICE_ID_DLGC_9054,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_DM3, },
        { PCI_VENDOR_ID_DLGC_NEW, PCI_DEVICE_ID_DLGC_0046,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_DTI16, },
        { PCI_VENDOR_ID_DLGC_NEW, PCI_DEVICE_ID_DLGC_530D,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_ROZETTA, },
        { PCI_VENDOR_ID_DLGC_NEW, PCI_DEVICE_ID_DLGC_0046,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_ROZETTA, },
        { PCI_VENDOR_ID_DLGC_21555, PCI_DEVICE_ID_DLGC_21555,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_DTI16, },
        { PCI_SUBVENDOR_ID_DLGC, PCI_DEVICE_ID_DLGC_4143,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_DM3, },
        { PCI_VENDOR_ID_DLGC,PCI_DEVICE_ID_DLGC_9054,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_DMVB, },
        { PCI_VENDOR_ID_DLGC,PCI_DEVICE_ID_DLGC_9056,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_DMVC, },
        { PCI_SUBVENDOR_ID_DLGC, PCI_DEVICE_ID_DLGC_5356,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_DISI, },
        { PCI_SUBVENDOR_ID_DLGC, PCI_DEVICE_ID_DLGC_5356,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_SEAV, },
        { PCI_SUBVENDOR_ID_DLGC, PCI_DEVICE_ID_GEM,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_GEM, },
        { PCI_VENDOR_ID_MPC, PCI_DEVICE_ID_GEMMPC,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_GEMMPC, },
        { PCI_SUBVENDOR_ID_DLGC, PCI_DEVICE_ID_GEM_AMC,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_GEM_AMC, },
        { PCI_VENDOR_ID_MPC, PCI_DEVICE_ID_GEMMPC_AMC,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_GEMMPC_AMC, },
        { PCI_VENDOR_ID_MPC, PCI_DEVICE_ID_GEMMPC,
                PCI_SUBVENDOR_ID_DLGC, PCI_SUBDEVICE_ID_GEM_OPAL, },
        { 0,}
};

MODULE_DEVICE_TABLE(pci, mercd_pci_tbl);

static struct pci_driver mercd_driver = {
        .name     =       mercd_string,
        .id_table =       mercd_pci_tbl,
        .probe    =       mercd_probe,
        .remove   =       mercd_release,
        .resume   =       mercd_resume,
        .suspend  =       mercd_suspend,
};

struct proc_dir_entry *mercd_proc_entry;
struct proc_dir_entry *mercd_proc_hsi_entry;
static int mercd_proc_info(char *buf, char **start, off_t offset, int len, int *eof, void *data);
static int mercd_proc_hsi_info(char *buf, char **start, off_t offset, int len, int *eof, void *data);

//========================================================================
//
//NAME         :  mercd_init_module
//DESCRIPTION     :  Streams Driver Initialization Routine.
//                   When driver is loaded.
// INPUTS      :  none
// OUTPUTS     :  none
// RETURNS     :  none
// CALLS       :
// CAUTIONS    :  Called by OS.
//
//========================================================================
static int __init mercd_init_module(void)
{
        int ret,i;
        mercd_dhal_drvr_identify_sT     drvr_identifyinfo = { 0 };

        // Initialize Control Block
        mercd_zalloc(MsdControlBlock, pmercd_control_block_sT, MERCD_CONTROL_BLOCK_STRUCT);

        mercd_zalloc(pmutex_stats, PMSD_MUTEX_STATS, sizeof(MSD_MUTEX_STATS));

        if (MsdControlBlock == NULL) {
            printk("mercd_init_module: Control Block Allocation Failure");
            return -ENOMEM;
        }

        MSD_INIT_MUTEX(&MsdControlBlock->mercd_ctrl_block_mutex, "Control Block Mutex", NULL);

        MsdControlBlock->maxbind = MSD_ABSOLUTE_MAX_BIND;
        MsdControlBlock->MsdState = MERCD_CTRL_BLOCK_PRE_INI;
        MsdControlBlock->maxstreams = MSD_INITIAL_MAX_STREAMS;

        /* Initialize the interface function table */
        INITIALIZE_OSAL();
        INITIALIZE_DHAL();

        /* Initialize Control block */
        DM3_SETUP_MSDCONTROLBLOCK(MsdControlBlock);
        
        /* This will give high major node to low */
	mercdMajor = abstract_register_dev(0, &mercd_fops, MSD_MAX_OPEN_ALLOWED, mercd_string);

	if (mercdMajor < 0) {
	    printk("mercd_init_module: Unable to register driver.\n");
	    return mercdMajor;
	}

	if (dm3_feature & 1) { NewCanTakeProtocol = 1; printk("NewCanTake Protocol Enabled \n"); }
        if (dm3_feature & 2) { ReceiveDMA = 1; printk("Receive DMA Enabled \n"); }
        if (dm3_feature & 4) { SendDMA = 1; printk("Send DMA Enabled \n"); }
        if (dm3_feature & 8) { No_Send_From_DPC  = 1; printk("30 mSec Send Only From Timer Enabled \n"); }
        if (dm3_feature & 16) { SendExitNotify = 1; printk("Exit Notification Enabled\n"); }
        if (dm3_feature & 32) {
           NonDefaultId = 1;
           printk("NonDefault mode of Driver attach order based on Configuration id\n");
        }
        if (dm3_feature & 64) { DPC_Disabled = 1; printk("DPC Disabled\n"); }
        if (dm3_feature & 128) { HCS_Flag = 0; printk("HCS Disabled\n"); }
        if (dm3_feature & 0x100) { g_hsi_stub_enabled = 0; }
        if (dm3_feature & 0x200) { g_hsi_dpc_enabled  = 0; }
        if (dm3_feature & 0x400) { HMP_Enabled  = 1; printk("HMP XMSG Enabled\n"  ); }
        if (ctimod_check_rh()) { RH_Enabled = 1; printk("RH Enabled \n"); }

        if(qDrvMsgPoolAllocate(NATIVE_MAX_POOL_SIZE) < 0)
		{
           printk(KERN_ERR "mercd_init_module: Cannot allocate qDrvMsgPool\n");
        }

        g_hsi_isr_callback = (HSI_ISR_CALLBACK) stub_hsi_isr_callback;

        if (g_hsi_stub_enabled) {
           stub_hsi_interface_timer();
        }

        ret = pci_register_driver(&mercd_driver);

        if (ret < 0)  {
            qDrvMsgPoolFree();
	    mercd_free(MsdControlBlock,MERCD_CONTROL_BLOCK_STRUCT, MERCD_FORCE);

	    if (abstract_unregister_dev(mercdMajor, mercd_string) < 0) {
		printk("mercd_init_module: Unable to remove driver err=%d.\n", ret);
 	    } else {
		printk("mercd_init_module: Driver removed\n");
            }

	    printk("mercd_init_module: No DM3 Cards found!\n");
        }

        mercd_proc_entry = create_proc_entry("mercd", S_IFREG | S_IRUGO, NULL);
        if (mercd_proc_entry) {
            mercd_proc_entry->read_proc = (read_proc_t *)mercd_proc_info;
        } 
        mercd_proc_hsi_entry = create_proc_entry("mercd_hsi", S_IFREG | S_IRUGO, NULL);
        if (mercd_proc_hsi_entry) {
            mercd_proc_hsi_entry->read_proc = (read_proc_t *)mercd_proc_hsi_info;
        } 

	MsdStringCopy(VersionString, MsdDriverVersion);
        supp_ver2string(VersionString, VER_WITHBUILD, MsdDriverBuild);
        (*mercd_dhal_func[MERCD_DHAL_DRVR_IDENTIFY])((void *)&drvr_identifyinfo);

        return ret;
}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,36)
static long mercd_unlocked_ioctl(struct file *filep, unsigned int cmd, unsigned long arg)
{
  return(mercd_ioctl(filep->f_dentry->d_inode, filep, cmd, arg));
}
#endif

/**
 * mercd_compat_ioctl        : 32bit to 64bit ioctl conversion routine
 */
#ifdef CONFIG_COMPAT
static int
mercd_compat_ioctl(struct file *filep, unsigned int cmd,
				unsigned long arg)
{
	int err;

	err = mercd_ioctl32(filep->f_dentry->d_inode, filep, cmd, arg);
	return err;
}
#endif

//========================================================================
//
//NAME         :  mercd_cleanup_module
//DESCRIPTION     :  Streams Driver Clean up Routine.
//                   When driver is unloaded.
// INPUTS      :  none
// OUTPUTS     :  none
// RETURNS     :  none
// CALLS       :
// CAUTIONS    :  Called by OS.
//
//========================================================================
static void __exit mercd_cleanup_module(void)
{
        mercd_osal_timeout_stop_sT timeoutinfo = { 0 };

   if (g_hsi_stub_enabled) {
      timeoutinfo.Handle = (struct timer_list *)&g_stub_hsi_interface_timer_list;
      g_stub_hsi_interface_timer_shutdown = 1;
      if (g_stub_hsi_interface_timer_on_queue == 1) {
         printk( "mercd_cleanup_module: calling del_timer_sync.\n");
         (*mercd_osal_func[MERCD_OSAL_TIMEOUT_STOP])((void *)&timeoutinfo);
      }
   }

	if (pmutex_stats) {
            mercd_free(pmutex_stats, sizeof(MSD_MUTEX_STATS), MERCD_FORCE);
	}

	if (abstract_unregister_dev(mercdMajor, mercd_string) < 0) {
	    printk( "mercd_cleanup_module: Unable to remove driver.\n");
	} else {
 	    printk("mercd_cleanup_module: Driver removed.\n");
	}

        pci_unregister_driver(&mercd_driver);
        mercd_free(MsdControlBlock->padapter_block_list, MERCD_ADAPTER_BLOCK_LIST, MERCD_FORCE);
        mercd_free(MsdControlBlock->pbind_block_list, MERCD_INITIAL_BIND_BLOCK_LIST, MERCD_FORCE);
        mercd_free(MsdControlBlock->pbind_free_block_list, sizeof(pmercd_bind_block_sT)*(MSD_MAX_BINDTOBE_FREED_INDEX+1), MERCD_FORCE);
        mercd_free(MsdControlBlock->popen_block_queue, MERCD_INITIAL_OPEN_BLOCK_LIST, MERCD_FORCE);
        mercd_free(MsdControlBlock->popen_free_block_queue, (MSD_MAX_OPENTOBE_FREED_INDEX+1) * sizeof(pmercd_open_block_sT), MERCD_FORCE);

        mercd_free(MsdControlBlock,MERCD_CONTROL_BLOCK_STRUCT, MERCD_FORCE);
        qDrvMsgPoolFree();

        if (mercd_proc_entry) {
            remove_proc_entry ("mercd", NULL);
        }
        if (mercd_proc_hsi_entry) {
            remove_proc_entry ("mercd_hsi", NULL);
        }
}

module_init(mercd_init_module);
module_exit(mercd_cleanup_module);

//========================================================================
//
//NAME			:	mercd_release
//DESCRIPTION		:	Streams Driver Release Routine.
//				Called for each board
//				Clean up board structures
// INPUTS		:       device info
// OUTPUTS		:	none
// RETURNS		:	none
// CALLS			:
// CAUTIONS		:	Called by OS.
//
//========================================================================
static void __devexit mercd_release(struct pci_dev *pdev) {
        mercd_dhal_drvr_free_sT drvr_free = { 0 };
        mercd_hs_t* hsd = NULL;

        hsd = (mercd_hs_t*)pci_get_drvdata(pdev);
        if(hsd == NULL) {
           return ;
        }

        if (hsd->state != MERCD_SUSPENDED ) {
           mercd_suspend(pdev);
        }

        drvr_free.pdevi = pdev;
        (*mercd_dhal_func[MERCD_DHAL_DRVR_FREE])((pmerc_void_t)&drvr_free);
        if (drvr_free.ret != MD_SUCCESS) {
           printk("mercd_release: mercd_dhal_drvr_free().. failed\n");
        }

        if (hsd) {
            MSD_FREE_KERNEL_MEMORY(hsd, sizeof(mercd_hs_t));
        }

        return;
}

//========================================================================
//
//NAME			:	mercd_probe
//DESCRIPTION		:	Streams Driver probe Routine.
//				Called for each board
//				Clean up board structures
// INPUTS		:       device info
// OUTPUTS		:	none
// RETURNS		:	none
// CALLS			:
// CAUTIONS		:	Called by OS.
//
//========================================================================
static int __devinit mercd_probe(struct pci_dev *pdev, const struct pci_device_id *ent)
{
        mercd_dhal_drvr_probe_sT drvr_probeinfo = { 0 };
        drvr_probeinfo.pdevi = pdev;
        mercd_hs_t* hsd = NULL;
        int ret = 0;

        hsd = kmalloc(sizeof(mercd_hs_t), GFP_KERNEL);
        if (hsd == NULL) {
          printk("mercd_probe: unable to allocate memory for hsd\n");
          return (0);
        }
        memset(hsd, 0x00, sizeof(mercd_hs_t));

        // Store a pointer to the device object in the kernels pci_dev object
        pci_set_drvdata(pdev, hsd);

        // Indicate successful probe completition
        hsd->state = MERCD_PROBED;

        (*mercd_dhal_func[MERCD_DHAL_DRVR_PROBE])((pmerc_void_t)&drvr_probeinfo);

        if (drvr_probeinfo.ret != MD_SUCCESS) {
            printk("mercd_probe: DM3 board probe failed.\n");
            MSD_FREE_KERNEL_MEMORY(hsd, sizeof(mercd_hs_t));
            hsd = NULL;
            pci_set_drvdata(pdev, hsd);
            return (0);
        }

        if (!RH_Enabled) {
            linux_pci_brd_probe((pmerc_void_t)&drvr_probeinfo);
        }

        return (0);
}

//========================================================================
//
//NAME                  :       mercd_resume
//DESCRIPTION           :       Streams Driver probe Routine.
//                              Called for each board
//                              Clean up board structures
// INPUTS               :       device info
// OUTPUTS              :       none
// RETURNS              :       none
// CALLS                        :
// CAUTIONS             :       Called by OS.
//
//========================================================================
int mercd_resume(struct pci_dev *pdev)
{
        mercd_dhal_drvr_probe_sT drvr_probeinfo = { 0 };
        drvr_probeinfo.pdevi = pdev;

        mercd_hs_t* hsd = NULL;

        hsd = (mercd_hs_t*)pci_get_drvdata(pdev);
        if(hsd == NULL) {
           printk("mercd_resume: error retreiving hsd object\n");
           return (0);
        }

        if (( hsd->state != MERCD_PROBED ) &&  ( hsd->state != MERCD_SUSPENDED )) {
           (*mercd_dhal_func[MERCD_DHAL_DRVR_PROBE])((pmerc_void_t)&drvr_probeinfo);
        }

        linux_pci_brd_probe((pmerc_void_t)&drvr_probeinfo);

        if (drvr_probeinfo.ret == MD_FAILURE) {
            printk("mercd_resume: board probe failed to resume\n");
            return (0);
        }

        // Indicate successful probe completition
        hsd->state = MERCD_RESUMED;

        return (0);
}


//========================================================================
//
//NAME                  :       mercd_suspend
//DESCRIPTION           :       Streams Driver probe Routine.
//                              Called for each board
//                              Clean up board structures
// INPUTS               :       device info
// OUTPUTS              :       none
// RETURNS              :       none
// CALLS                        :
// CAUTIONS             :       Called by OS.
//
//========================================================================
int mercd_suspend(struct pci_dev *pdev)
{
        pmercd_adapter_block_sT padapter;
        mercd_osal_timeout_stop_sT timeoutinfo = { 0 };
        mercd_dhal_intr_disable_sT intr_disableinfo = { 0 };
        mercd_dhal_dma_free_sT   dma_free = { 0 };
        mercd_hs_t* hsd = NULL;
        int j = 0;

        hsd = (mercd_hs_t*)pci_get_drvdata(pdev);
        if(hsd == NULL) {
           printk("mercd_suspend:  error retreiving hsd object\n");
           return (0);
        }

        if (RH_Enabled && (hsd->state != MERCD_RESUMED)) {
           printk("mercd_suspend:  Incorrect state %#x\n", hsd->state);
           return (0);
        }

        hsd->state = MERCD_SUSPENDED;
        padapter = hsd->padapter;

        if (padapter == NULL ) {
	    printk("mercd_suspend:  null adapter..\n");
	    return (0);
        }

        if ((padapter->phw_info->pciSubSysId == PCI_SUBDEVICE_ID_DMVB) ||
            (padapter->phw_info->pciSubSysId == PCI_SUBDEVICE_ID_SEAV) ||
            (padapter->phw_info->pciSubSysId == PCI_SUBDEVICE_ID_DMVC)) {
            // first do a soft reset
            linux_ww_generate_soft_reset(padapter);

            // now a hard reset as well as free up things
            if ((padapter->pww_info) &&
                (padapter->pww_info->state != MERCD_ADAPTER_WW_IN_SHUTDOWN)) {
                mid_wwmgr_process_cancel_request_intr_ack_from_board(padapter->pww_info);
            }
        }

        // remove timeout routine timer
        if (padapter->flags.SendTimeoutPending  & MERC_BOARD_FLAG_SEND_TIMEOUT_PEND) {
            timeoutinfo.Handle = (struct timer_list *)&padapter->phw_info->timer_info->timeout_id;
            (*mercd_osal_func[MERCD_OSAL_TIMEOUT_STOP])((void *)&timeoutinfo);

            if (timeoutinfo.ret != MD_SUCCESS) {
                printk("mercd_suspend: Untimeout failed on board %d\n", padapter->adapternumber);
            }
        }

        // remove host ram semaphore timeout timer
        if (padapter->flags.SendTimeoutPending  & MERC_ADAPTER_FLAG_HRAM_SEM_TIMEOUT_PEND) {
            timeoutinfo.Handle = (struct timer_list *)&padapter->phw_info->timer_info->timeout_id_sem;
            (*mercd_osal_func[MERCD_OSAL_TIMEOUT_STOP])((void *)&timeoutinfo);

            if (timeoutinfo.ret != MD_SUCCESS){
                printk("mercd_suspend: Untimeout failed for host Sem on board %d\n", padapter->adapternumber);
            }
        }

        // remove interrupt lines and PLX interrupt
        intr_disableinfo.ConfigId = padapter->adapternumber;

        // disable PLX interrupt
        (*mercd_dhal_func[MERCD_DHAL_INTR_DISABLE])((void *) &intr_disableinfo);

        // remove interrupt lines
        if (!( padapter->flags.LaunchIntr & MERC_ADAPTER_FLAG_LAUNCH_INTR)) {
            if (padapter->phw_info->pciSubSysId == PCI_SUBDEVICE_ID_GEM) {
	        mid_strmmgr_hsi_gemini(padapter, MSD_GEM_TDM_OFF, NULL);
	        mid_strmmgr_hsi_gemini(padapter, MSD_GEM_STOP, NULL);
	        free_irq(padapter->pdevi->irq, (void *)padapter->pdevi);
                free_irq(padapter->pdevi_gem->irq, (void *)padapter->pdevi_gem);
                padapter->MSI = MERCD_MSI_IRQ_FREE;
            } else {
                free_irq(pdev->irq, (void *)pdev);
            }
            padapter->flags.LaunchIntr &= ~MERC_ADAPTER_FLAG_LAUNCH_INTR;
        }

        if (padapter->phw_info->boardFamilyType != THIRD_ROCK_FAMILY) {
            dma_free.ConfigId = padapter->adapternumber;
            (*mercd_dhal_func[MERCD_DHAL_DMA_FREE_HANDLE])((void *) &dma_free);
        }

       for (j = 0; j < MSD_MAX_BOARD_ID_COUNT ; j++) {
            if (mercd_adapter_log_to_phy_map_table[j] == mercd_adapter_map[padapter->adapternumber]  ) {
                if (mercd_adapter_map[padapter->adapternumber] != 0xFF )
                    mercd_adapter_log_to_phy_map_table[j] = 0xFF;
                break;
            }
       }

       if (padapter->flags.WWFlags & MERCD_ADAPTER_WW_I20_MESSAGING_READY) {
           padapter->flags.WWFlags &= ~(MERCD_ADAPTER_WW_I20_MESSAGING_READY);
       }

       if (padapter->pww_info) {
           if (padapter->pww_info->pww_param) {
               mercd_free(padapter->pww_info->pww_param, MERCD_WW_PARAM, MERCD_FORCE);
	   }
           mercd_free(padapter->pww_info, MERCD_WW_DEV_INFO, MERCD_FORCE);
	   padapter->pww_info = NULL;
       }

        // Disable device
        if (padapter->phw_info->pciSubSysId == PCI_SUBDEVICE_ID_GEM) {
	   if (padapter->MSI != MERCD_MSI_IRQ_FREE) {
	       free_irq(padapter->pdevi->irq, (void *)padapter->pdevi);
               free_irq(padapter->pdevi_gem->irq, (void *)padapter->pdevi_gem);
               padapter->flags.LaunchIntr &= ~MERC_ADAPTER_FLAG_LAUNCH_INTR;
	    }
            pci_disable_msi(padapter->pdevi); 
            pci_disable_msi(padapter->pdevi_gem); 
            padapter->MSI = MERCD_MSI_DISABLED;
			pci_disable_device(padapter->pdevi); 
            pci_disable_device(padapter->pdevi_gem); 
	    if (padapter->pdevi == pdev)
                pci_set_drvdata(padapter->pdevi_gem, NULL);
            else if (padapter->pdevi_gem == pdev)
                pci_set_drvdata(padapter->pdevi, NULL);
	} else {
           pci_disable_device(pdev);
        }

        padapter->state = MERCD_ADAPTER_STATE_SUSPENDED;

        return (0);
}


//========================================================================
//
//NAME                  :       mercd_proc_info
//DESCRIPTION           :       Streams Driver proc Routine.
// INPUTS               :       proc
// OUTPUTS              :       none
// RETURNS              :       none
// CALLS                :
// CAUTIONS             :       Called by OS.
//
//========================================================================
static int mercd_proc_info(char *buf, char **start, off_t offset, int len, int *eof, void *data) {
   pmercd_adapter_block_sT padapter;
   pmercury_host_info_sT   HostIfPtr;
   PMERC_CIR_BUFFER        CirBuffer;
  
   struct pci_dev *pdev;
   merc_int_t started=0;
   merc_int_t hsiIndex=0;
   merc_int_t hsiCount=0;
   merc_int_t barIndex=0;
   merc_int_t hmpBoard=0;
   merc_int_t gemBoard=0;
   merc_int_t i=0, j, k;
   merc_int_t ulTmp;
   merc_uchar_t ucTmp;
   merc_ushort_t usTmp;
   pmerc_ushort_t sMemAddr;
   pmerc_uchar_t BaseAddr, MemAddr; 
  
   len = 0;
   len += sprintf(buf+len, "          --------------------------------\n");
   len += sprintf(buf+len, "           Native DM3 Linux Device Driver\n");
   len += sprintf(buf+len, "           Version: %s\n", MsdDriverBuild);
   len += sprintf(buf+len, "          --------------------------------\n\n");

   if (!MsdControlBlock->adapter_count) {
       len += sprintf(buf+len, "No DM3 Boards detected\n");
       return (len);
   }

   for (i = 0; i <= MsdControlBlock->adapter_count; i++) {
       if (mercd_adapter_map[i] > MSD_MAX_BOARD_ID_COUNT ||
           MsdControlBlock->padapter_block_list[mercd_adapter_map[i]] == NULL ) {
           continue;
       }
 
       padapter = MsdControlBlock->padapter_block_list[mercd_adapter_map[i]];
       if (!padapter)
           continue;
    
       switch (padapter->state) {
          case MERCD_ADAPTER_STATE_INIT:
            len += sprintf(buf+len, "\n-------< Board %d - INIT STATE >-------\n", padapter->adapternumber);
                break;
          case MERCD_ADAPTER_STATE_STARTED:
            started = 1;
            len += sprintf(buf+len, "\n-------< Board %d - STARTED STATE >-------\n", padapter->adapternumber);
                break;
          case MERCD_ADAPTER_STATE_DOWNLOADED:
            started = 1;
            len += sprintf(buf+len, "\n-------< Board %d - DOWNLOADED STATE >-------\n", padapter->adapternumber);
                break;
          default:
                len += sprintf(buf+len, "\n-------< Board %d - %#x STATE >-------\n", padapter->adapternumber, padapter->state);
                break;
       }

       pdev = padapter->pdevi;
       gemBoard = 0;
       switch (padapter->phw_info->pciDeviceId) {
          case PCI_DEVICE_ID_GEMRTM:
          case PCI_DEVICE_ID_GEM:
          case PCI_DEVICE_ID_GEMMPC:
               gemBoard = 1;
               // Check if FPGA/MPC were linked
               if (padapter->MSI != MERCD_MSI_ENABLED) {
                   len += sprintf(buf+len, "  Gemini Board Failed Detection\n");
                   break;
               }
               len += sprintf(buf+len, "  Motorola Power PC bus=%d  slot=%d  irq=%d\n", pdev->bus->number, PCI_SLOT(pdev->devfn), pdev->irq);
               pdev = padapter->pdevi_gem;
               len += sprintf(buf+len, "  Gemini FPGA       bus=%d  slot=%d  irq=%d\n", pdev->bus->number, PCI_SLOT(pdev->devfn), pdev->irq);
               break;
          case PCI_DEVICE_ID_DLGC_5356:
               len += sprintf(buf+len, "  Seaville FPGA     bus=%d  slot=%d  irq=%d\n", pdev->bus->number, PCI_SLOT(pdev->devfn), pdev->irq);
               break;
          default:
               len += sprintf(buf+len, "  PLX FPGA          bus=%d  slot=%d  irq=%d\n", pdev->bus->number, PCI_SLOT(pdev->devfn), pdev->irq);
               break;
       }                   

       if ((gemBoard) && (padapter->MSI != MERCD_MSI_ENABLED)) {
           continue;
       }

       if (padapter->hsiBuffers) {
           hmpBoard=1;
       }

       len += sprintf(buf+len, "\n  Board Information: \n");
       for (barIndex = 0; barIndex < 5; barIndex++) {
            if (padapter->phw_info->bar[barIndex].phys_addr) {
                len += sprintf(buf+len, "   Bar[%d] PhysAddr=%08x\n", barIndex, padapter->phw_info->bar[barIndex].phys_addr);
            }
       }

       if (started) {
           switch(padapter->phw_info->pciSubSysId) {
              case PCI_SUBDEVICE_ID_DM3:
              case PCI_SUBDEVICE_ID_GEM:
                 HostIfPtr = &padapter->phost_info->merc_host_info;
                 CirBuffer = (PMERC_CIR_BUFFER)HostIfPtr->host_to_fw_msg_start;
                 j = padapter->phost_info->host_config.maxHostMsgs;
                 len += sprintf(buf+len, "\n  DM3 SRAM Protocol: \n"
                                         "   H2B %#x[R] / %#x[W] (MaxMsg %#x)\n",
                              CirBuffer->Read, CirBuffer->Write, j);
                 CirBuffer = (PMERC_CIR_BUFFER)HostIfPtr->fw_to_host_msg_start;
                 j = padapter->phost_info->host_config.maxFwMsgs;
                 len += sprintf(buf+len, "   B2H %#x[R] / %#x[W] (MaxMsg %#x)\n",
                              CirBuffer->Read, CirBuffer->Write, j);
                 break;
              default:
                 break;
           }
       }
#if 0
       if (padapter->phw_info->typePCIExpress) {
           len += sprintf(buf+len, "\n  Power Information: \n"
                                   "   actualPower   %#x\n"
                                   "   powerReq      %#x\n"
                                   "   powerGood     %#x\n" 
                                   "   powerOverride %#x\n", 
                          padapter->phw_info->actualPowerProvided, padapter->phw_info->powerRequiredByBoard, 
                          padapter->phw_info->powerGoodReg, padapter->phw_info->powerOverrideReg);  
       }
#endif
   }

   *eof = 1;

   if (len > PAGE_SIZE-500) {
       printk("mercd_proc_info(): Error too much data\n");
       return(PAGE_SIZE-1000);
   }
   return len;

}


//========================================================================
//
//NAME                  :       mercd_proc_hsi_info
//DESCRIPTION           :       Streams Driver proc Routine.
// INPUTS               :       proc
// OUTPUTS              :       none
// RETURNS              :       none
// CALLS                :
// CAUTIONS             :       Called by OS.
//
//========================================================================
static int mercd_proc_hsi_info(char *buf, char **start, off_t offset, int len, int *eof, void *data) {
   pmercd_adapter_block_sT padapter;
   pmercury_host_info_sT   HostIfPtr;
   PMERC_CIR_BUFFER        CirBuffer;
  
   struct pci_dev *pdev;
   merc_int_t started=0;
   merc_int_t hsiIndex=0;
   merc_int_t hsiCount=0;
   merc_int_t barIndex=0;
   merc_int_t hmpBoard=0;
   merc_int_t gemBoard=0;
   merc_int_t i=0, j, k;
   merc_int_t ulTmp;
   merc_uchar_t ucTmp;
   merc_ushort_t usTmp;
   pmerc_ushort_t sMemAddr;
   pmerc_uchar_t BaseAddr, MemAddr; 
  
   len = 0;

   if (!MsdControlBlock->adapter_count) {
       len += sprintf(buf+len, "No DM3 Boards detected\n");
       return (len);
   }

   for (i = 0; i <= MsdControlBlock->adapter_count; i++) {
       if (mercd_adapter_map[i] > MSD_MAX_BOARD_ID_COUNT ||
           MsdControlBlock->padapter_block_list[mercd_adapter_map[i]] == NULL ) {
           continue;
       }
 
       padapter = MsdControlBlock->padapter_block_list[mercd_adapter_map[i]];
       if (!padapter)
           continue;
    
       pdev = padapter->pdevi;
       gemBoard = 0;
       switch (padapter->phw_info->pciDeviceId) {
          case PCI_DEVICE_ID_GEMRTM:
          case PCI_DEVICE_ID_GEM:
          case PCI_DEVICE_ID_GEMMPC:
               gemBoard = 1;
               // Check if FPGA/MPC were linked
               if (padapter->MSI != MERCD_MSI_ENABLED) {
                   len += sprintf(buf+len, "  Gemini Board Failed Detection\n");
                   break;
               }
               len += sprintf(buf+len, "Gemini %d\n", padapter->adapternumber);
               break;
          case PCI_DEVICE_ID_DLGC_5356:
               len += sprintf(buf+len, "DNI PCIe %d\n", padapter->adapternumber);
               break;
          default:
               len += sprintf(buf+len, "DNI PCI %d\n", padapter->adapternumber);
               break;
       }                   

       if ((gemBoard) && (padapter->MSI != MERCD_MSI_ENABLED)) {
           continue;
       }

       if (padapter->hsiBuffers) {
           hmpBoard=1;
       }

       if (hmpBoard) {
#if 0
           // Interrupt tracing disabled - enable counters in ISR routines
           hsiIndex = (padapter->hsiBoardCount >> 0x18);
           hsiCount = (padapter->hsiBoardCount & 0x00FFFFFF);
           len += sprintf(buf+len, "\n  HSI Information: \n"
                                   "   BoardIndex    %d\n"
                                   "   BoardCount    %d\n"
                                   "   DriverCount   %d\n"
                                   "   DriftCount    %d\n"
                                   "   Delta         %d\n",
                          hsiIndex, hsiCount, padapter->hsiDriverCount, padapter->hsiDrift, padapter->hsiDelta);
#endif

	   // HSI Buffer tracing disabled
           if (padapter->hsiBuffers) {
               len += sprintf(buf+len, " HSI Buffer (%d)\n", padapter->hsiBuffers);
	   
               for (barIndex=0; barIndex<padapter->hsiBuffers; barIndex++) {
                }

                // For board Sync Pattern
                if (padapter->hsiBuffers) {
                    for (j=0; j<padapter->hsiBuffers; j++) {
                         len += sprintf(buf+len, "   H2B[%d][0](%x): ", j, padapter->H2B[j]);
                         BaseAddr = sMemAddr = ioremap(padapter->H2B[j], 32);
                         for (k=0; k<16; k++) {
                              usTmp = readw(sMemAddr);
                              len += sprintf(buf+len, "%04x", usTmp);
                              sMemAddr++;
                         }
                         len += sprintf(buf+len, "\n");
                         iounmap((void *)BaseAddr);
                    }
                    if (gemBoard) {
                        for (j=0; j<padapter->hsiBuffers; j++) {
                             len += sprintf(buf+len, "   B2H[%d][0](%x): ", j, padapter->B2H[j]);
                             BaseAddr = sMemAddr = ioremap(padapter->B2H[j], 32);
                             for (k=0; k<16; k++) {
                                  usTmp = readw(sMemAddr);
                                  len += sprintf(buf+len, "%04x", usTmp);
                                  sMemAddr++;
                             }
                             len += sprintf(buf+len, "\n");
                             iounmap((void *)BaseAddr);
                        }
                        BaseAddr =(pmerc_uchar_t)(padapter->phw_info->bar[3].virt_addr);
                        MemAddr = BaseAddr + MSD_GEM_RATE_CNT;
                        ulTmp = readl(MemAddr);
                        len += sprintf(buf+len, " Buf Indx= %d\n\n", ulTmp>>24);
                    } else {
                        // For board Sync Lock
#if 0
                        BaseAddr =(pmerc_uchar_t)(padapter->phw_info->bar[3].virt_addr);
                        MemAddr = BaseAddr + 0x11C;
                        ulTmp = readl(MemAddr);
                        len += sprintf(buf+len, "  TDM ICSR (0x11C) %#x\n", ulTmp);
                        MsdWWPlxReadFromPlxRegister(MERCD_WW_PLX_MAILBOX_4_REG, ulTmp)
#endif
                    }

                    
                }
           }
       }   

   }

   *eof = 1;

   if (len > PAGE_SIZE-500) {
       return(PAGE_SIZE-1000);
   }
   return len;

}
