/**
 * @file  KMBC_low_linux.c
 * @brief Linux specific framework for KMBC_low Driver
 * @date  May 1, 2006
 *
 * Dialogic CONFIDENTIAL	
 * Copyright 2013 Dialogic Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to 
 * the source code ("Material") are owned by Dialogic Corporation or its suppliers
 * or licensors.  Title to the Material remains with Dialogic Corporation or its 
 * suppliers and licensors.  The Material contains trade secrets and proprietary
 * and confidential information of Dialogic or its suppliers and licensors.  The
 * Material is protected by worldwide copyright and trade secret laws and treaty
 * provisions.  No part of the Material may be used, copied, reproduced, 
 * modified, published, uploaded, posted, transmitted, distributed, or disclosed
 * in any way without Dialogic's prior express written permission.
 * 
 * No license under any patent, copyright, trade secret or other intellectual 
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise.  Any license under such intellectual property rights must be
 * express and approved by Dialogic in writing.
 * 
 * Unless otherwise agreed by Dialogic in writing, you may not remove or alter this
 * notice or any other notice embedded in Materials by Dialogic or Dialogic's 
 * suppliers or licensors in any way.
 */


#include <linux/moduleparam.h>

/* Header Files */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/cdev.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include <asm/uaccess.h>
#include <asm/atomic.h>

#include "KMBC_low.h"
#include "KMBC_low_mm.h"
#include "KMBC_low_mmap.h"




/* Custom Data Types */
typedef struct _tagKMBC_ISR_DATA
{
	int BridgeId;
	int SyncData;
}
KMBC_ISR_DATA, *PKMBC_ISR_DATA;




/* Function Prototypes */
int     kmbc_open( struct inode *inode, struct file *filp );
int     kmbc_release( struct inode *inode, struct file *filp );
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,36)
int     kmbc_ioctl( struct file *filp, unsigned int cmd, unsigned long arg );
#else
int     kmbc_ioctl( struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg );
#endif
ssize_t kmbc_read( struct file *filp, char __user *buff, size_t count, loff_t *offp );
void    kmbc_isr( int BridgeId, int SyncData );
void    kmbc_dpc( unsigned long data );
#ifdef CONFIG_COMPAT
static int kmbc_compat_ioctl(struct file *filp, unsigned int cmd, unsigned long arg );
#else
#define kmbc_compat_ioctl NULL
#endif
extern void kmbc_hsi_hotwire(int index, int bid);


/* Global Variables */
static DECLARE_WAIT_QUEUE_HEAD( kmbc_wait_queue );

atomic_t               kmbc_isr_bridge_id;
atomic_t               kmbc_isr_sync_data;
int                    kmbc_isr_flag     = 0;
int                    kmbc_isr_shutdown = 0;
int                    kmbc_major_num    = 0;
struct cdev            kmbc_cdev;
struct file_operations kmbc_fops         =
{
	.owner   = THIS_MODULE,
	.read    = kmbc_read,
	.mmap    = kmbc_mmap,
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,36)
	.unlocked_ioctl = kmbc_ioctl,
#else
	.ioctl   = kmbc_ioctl,
#endif
#ifdef CONFIG_COMPAT
	.compat_ioctl = kmbc_compat_ioctl,
#endif
	.open    = kmbc_open,
	.release = kmbc_release,
};

#define MAX_TS                       256
#define KMBC_UNPACK_SYNC_COUNTER(x)  ( x & 0xFFFFFF )
#define KMBC_UNPACK_SYNC_INDEX(x)    ((x & 0xFF000000UL ) >> 24 )
#define KMBC_CHECK_SET_TS(x,y)	     ((x > 0) && (x<MAX_TS)) ? y=x : 0

struct tasklet_struct kmbc_tasklet;

kmbc_ts        ts_data[MAX_TS];
unsigned int   ts_map[MAX_TS];
static char   *proc_buffer;
struct         proc_dir_entry *kmbc_proc_entry;
static int     kmbc_read_proc(char *p, char **s, off_t o, int c, int *e, void *d);
static ssize_t kmbc_write_proc(struct file *f, const char __user *b, unsigned long c, void *d);

#ifdef KMBC_DEBUG
/***************************************************************************************/
/* The KMBC_TRACE macro provides a way of compiling in a lot more trace data for       */
/* initial development and subsequent debugging.  Most errors will always be logged,   */
/* regardless of the KMBC_DEBUG build option.                                          */
/***************************************************************************************/
#define KMBC_TRACE(x)     printk x
#else
#define KMBC_TRACE(x)
#endif	/* #ifdef KMBC_DEBUG */




#ifdef KMBC_UNIT_TEST
/***************************************************************************************/
/* The KMBC_LOW device driver can be built in "Unit Test" mode which adds some code    */
/* that can simulate certain conditions, or add specific debug for the unit tests,     */
/* in cases where the KMBC_LOW device driver will be run in isolation.  When in this   */
/* mode, the KMBC_LOW will not have access to the DM3 driver for interrupt generation, */
/* so in order to test the rate interrupt plumbing, the KBMC_LOW can be built with     */
/* special code that simulates interrupts by using a recurring timer (which will be    */
/* disabled in response to a SHUTDOWN command.                                         */
/***************************************************************************************/
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/param.h>


#define INIT_KMBC_TIMER(x)     {init_timer( &x );}
#define ADD_KMBC_TIMER(x,y)    {x.data=0;x.function=kmbc_timer_callback;x.expires=jiffies+y;add_timer(&x);}
#define DEL_KMBC_TIMER(x)      {del_timer_sync( &x );}


static struct timer_list kmbc_timer;
static int               dbg_sync_data = 0;

void
kmbc_timer_callback( unsigned long context )
{
	/* All this timer does is call the ISR */
	kmbc_isr( 50001, dbg_sync_data++ );
}

#else

#define INIT_KMBC_TIMER(x)
#define ADD_KMBC_TIMER(x,y)
#define DEL_KMBC_TIMER(x)

#endif  /* #ifdef KMBC_UNIT_TEST */




int
kmbc_open( struct inode *inode, struct file *filp )
{
	KMBC_TRACE(( KERN_ALERT "kmbc_low:  kmbc_low_open( inode=0x%p, filp=0x%p ).\n",
				 inode, filp ));
	  
	return( 0 );
}




int
kmbc_release( struct inode *inode, struct file *filp )
{
	KMBC_TRACE(( KERN_ALERT "kmbc_low:  kmbc_low_release( inode=0x%p, filp=0x%p ).\n",
				 inode, filp ));
	  
	return( 0 ); 
}




ssize_t
kmbc_read( struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
	KMBC_ISR_DATA isr_payload;
	int           ret_val;
	 
	/********************************************************************************/
	/* The count should only ever be 8 bytes, so we will check that now, and reject */
	/* any read request that is not for 8 bytes. The *only* time that this request  */
	/* should be serviced is when the caller is making the request to pend on the   */
	/* rate interrupt.                                                              */
	/********************************************************************************/
	if( count != 8 )
	{
		return( -EINVAL );
	}  

	ADD_KMBC_TIMER( kmbc_timer, HZ/10 ); /* Only exists in UNIT_TEST build */


	/********************************************************************************/
	/* The only time this system call should be made is when the KMBC_HIGH library  */
	/* is waiting for the rate interrupt.  The library should read 64 bits of data. */
	/* This function will then sleep until our ISR is called, at which point, this  */
	/* function will be awoken and provided with the 64 bits of data to return to   */
	/* the user.                                                                    */
	/********************************************************************************/
	ret_val = wait_event_interruptible_timeout( kmbc_wait_queue, kmbc_isr_flag > 0, HZ*200 );
	kmbc_isr_flag = 0;
	  
	  
	/********************************************************************************/
	/* Retrieve the ISR data from our atomic variables and copy it to the buffer    */
	/* that was provided by the user.                                               */
	/********************************************************************************/
	if( kmbc_isr_shutdown == 0 )
	{
		if( ret_val != 0 )
		{
			isr_payload.BridgeId = atomic_read( &kmbc_isr_bridge_id );
			isr_payload.SyncData = atomic_read( &kmbc_isr_sync_data );
	      
			if( copy_to_user( buf, &isr_payload, count ))
			{
				return( -EFAULT );
			}
		}
		else
		{
			isr_payload.BridgeId = 0x0;
			isr_payload.SyncData = 0x0;

			if( copy_to_user( buf, &isr_payload, count ))
			{
				return( -EFAULT );
			}
		}
	}
	else
	{
		/* Shutdown flag is set, return an error to terminate user space thread */
		kmbc_isr_shutdown = 0;
		return( -EFAULT );
	}
	  
	return( count );
}



#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,36)
int kmbc_ioctl( struct file *filp, unsigned int cmd, unsigned long arg )
#else
int kmbc_ioctl( struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg )
#endif
{
	int status = 0;
	int type   = _IOC_TYPE( cmd );
	int code   = _IOC_NR( cmd );
	int compat = 0;
	
#if defined(__x86_64__)
	/* check to see if 32/64bit task */
	if(is_compat_task()) {
		compat = 1;
	}
#endif

	/* Validate Device Magic Number */
	if( (type == KMBCLOW_FILE_DEVICE) || (type == HSI_FILE_DEVICE) )
	{
		/* Validate IOCTL Code */
		if(( code <= KMBC_LOW_GET_INFO ) && ( code >= 0 ))
		{
			/* Process the Request */
			switch( cmd )
			{
			case( KMBC_LOW_INIT_IOCTL ):
				{   
					int                  status = 0;
					KMBC_LOW_INIT_STRUCT payload;
					KMBC_LOW_INIT_STRUCT32 payload32;
					
					KMBC_TRACE(( KERN_ALERT "kmbc_low:  IOCTL (KMBC_LOW_INIT_IOCTL).\n" ));
					
					/* Specifies parameters for the HSI buffer manager */
					if(compat) {
						status = copy_from_user( &payload32, (KMBC_LOW_INIT_STRUCT32 __user *)arg,
												 sizeof( KMBC_LOW_INIT_STRUCT32 ));
												 
						payload.NumBuffers = payload32.NumBuffers;
						payload.BufferSize = payload32.BufferSize;
					} else {
						status = copy_from_user( &payload, (KMBC_LOW_INIT_STRUCT __user *)arg,
												 sizeof( KMBC_LOW_INIT_STRUCT ));
					}
		
					if( status == 0 )
					{           
						KMBC_TRACE(( KERN_ALERT "kmbc_low:  num=%d, size=%d.\n",
									 payload.NumBuffers, payload.BufferSize ));
			
						status = kmbc_hsi_pool_init( payload.NumBuffers, payload.BufferSize );
			
						if( status == 0 )
						{
							if(compat) {
								payload32.NumBuffers = payload.NumBuffers;
								payload32.BufferSize = payload.BufferSize;
								payload32.CallbackAddr = kmbc_isr;
								
								status = copy_to_user( (KMBC_LOW_INIT_STRUCT32 __user *)arg,
													&payload32, sizeof( KMBC_LOW_INIT_STRUCT32 ));
							
							} else {
								payload.CallbackAddr = kmbc_isr;
					
								/* Buffer Pool allocated, we are good to go */
								status = copy_to_user( (KMBC_LOW_INIT_STRUCT __user *)arg,
													&payload, sizeof( KMBC_LOW_INIT_STRUCT ));
							}
					
							if( status != 0 )
							{
								printk( KERN_ALERT "kmbc_low:  KMBC_LOW_INIT: Failed to copy %d bytes to user buffer.\n",
										status );
						
								status = -ENOTTY;
							}
						}
						else
						{
							printk( KERN_ALERT "kmbc_low:  KMBC_LOW_INIT: Could not allocate HSI buffer pool (status=%d).\n",
									status );
						}
					}
					else
					{
						printk( KERN_ALERT "kmbc_low:  KMBC_LOW_INIT: Failed to copy %d bytes from user buffer.\n",
								status );
					
						status = -ENOTTY;
					}           
		
					break;
				}


			case( KMBC_LOW_SHUTDOWN_IOCTL ):
				{
					KMBC_TRACE(( KERN_ALERT "kmbc_low:  IOCTL( KMBC_LOW_SHUTDOWN_IOCTL ).\n" ));
		
					DEL_KMBC_TIMER( kmbc_timer );  /* Only exists in Unit Test Builds */
		
					/* Wakeup a read if it is pending */
					kmbc_isr_shutdown = 1;				
					kmbc_isr( 0, 0 );

					status = 0;
					break;
				}


			case( KMBC_LOW_GET_BUFFER_IOCTL ):
				{
					int                        status = 0;
					KMBC_LOW_GET_BUFFER_STRUCT payload;
					KMBC_LOW_GET_BUFFER_STRUCT32 payload32;


					/*************************************************************************/
					/* The KMBC_HIGH library needs a way of associating a particular virtual */
					/* that is passed to the CFSP with the bus address that is used by the   */
					/* DMA engine on the HIB.  This request provides that data to the user.  */
					/*************************************************************************/
					if(compat) {
						status = copy_from_user( &payload32, (KMBC_LOW_GET_BUFFER_STRUCT32 __user *)arg,
												 sizeof( KMBC_LOW_GET_BUFFER_STRUCT32 ));

						payload.VirtAddr = payload32.VirtAddr;
					} else {
						status = copy_from_user( &payload, (KMBC_LOW_GET_BUFFER_STRUCT __user *)arg,
												 sizeof( KMBC_LOW_GET_BUFFER_STRUCT ));
					}

					if( status == 0 )
					{
						payload.PhysAddr = (void *)kmbc_hsi_pool_get_bus_addr( payload.VirtAddr );

						if( payload.PhysAddr )
						{	
							if(compat) {
								payload32.VirtAddr = payload.VirtAddr;
								payload32.PhysAddr = payload.PhysAddr;

								status = copy_to_user( (KMBC_LOW_GET_BUFFER_STRUCT32 __user *)arg,
													   &payload32, sizeof( KMBC_LOW_GET_BUFFER_STRUCT32 ));

							} else {
								status = copy_to_user( (KMBC_LOW_GET_BUFFER_STRUCT __user *)arg,
													   &payload, sizeof( KMBC_LOW_GET_BUFFER_STRUCT ));
							}

							if( status != 0 )
							{
								printk( KERN_ALERT "kmbc_low:  KMBC_LOW_GET_BUFFER: Failed to copy %d bytes to user buffer.\n",
								(unsigned int) sizeof( KMBC_LOW_GET_BUFFER_STRUCT ));

								status = -ENOTTY;
							}
						}
					}
					else
					{
						printk( KERN_ALERT "kmbc_low:  KMBC_LOW_GET_BUFFER: Failed to copy %d bytes from user buffer.\n",
						(unsigned int) sizeof( KMBC_LOW_GET_BUFFER_STRUCT ));
						status = -ENOTTY;
					}

					break;
				}

			case( KMBC_LOW_GET_INFO_IOCTL ):
				{
					PKMBC_LOW_GET_INFO_STRUCT payload = (PKMBC_LOW_GET_INFO_STRUCT) arg;
					payload->CallbackAddr = kmbc_isr;
					break;
				}
			case( HSI_BRIDGE_INIT_IOCTL ):
				{
					PHSI_BRIDGE_INIT_INPUT payload = (PHSI_BRIDGE_INIT_INPUT) arg;
					kmbc_hsi_buffer_init(payload);
					break;
				}
			default:
				{
					status = -ENOTTY;
					break;
				}
			}
		}
		else
		{
			printk( KERN_ALERT "kmbc_low:  IOCTL code invalid (%d > %d or %d < 0).\n",
					code, KMBC_LOW_GET_BUFFER, code );
			status = -ENOTTY;
		}
	}
	else
	{   
		printk( KERN_ALERT "kmbc_low:  IOCTL type != KMBCLOW_FILE_DEVICE (%d != %d).\n",
				type, KMBCLOW_FILE_DEVICE );
	    
		status = -ENOTTY;
	}

	return( status );
}




void
kmbc_isr( int BridgeId, int SyncData )
{
	/* Store the BridgeId and SyncData values in our atomic global variables */
	atomic_set( &kmbc_isr_bridge_id, BridgeId );
	atomic_set( &kmbc_isr_sync_data, SyncData );
	  
	tasklet_schedule( &kmbc_tasklet );
	kmbc_hsi_hotwire(KMBC_UNPACK_SYNC_INDEX(SyncData), BridgeId);
	return;
}




void kmbc_dpc( unsigned long data )
{
	/* Set the ISR flag */
	kmbc_isr_flag = 1;
	  
	/* Wake up any waiting reads */
	wake_up_interruptible( &kmbc_wait_queue );
	return;
}




static int kmbc_read_proc(char *p, char **s, off_t o, int c, int *e, void *d)
{
	int i, len=0;
	kmbc_ts *ts_cfg = (kmbc_ts *)d;

	/* Output tx_data info into the proc file */
	for (i=0; i<MAX_TS; i++) {
	     if (ts_cfg[i].enable) {
		 len += sprintf(p+len, "%d:%d:%d\n", ts_cfg[i].rx, ts_cfg[i].tx, ts_cfg[i].enable);
	     }
	}

	*e = 1;
	return len;
}



static ssize_t kmbc_write_proc(struct file *f, const char __user *b, unsigned long c, void *d)
{
	int      i,j;
	long     ulTmp;
	char    *token;
	char    *curr;
	char    *endptr;
	kmbc_ts *ts_cfg = (kmbc_ts *)d;

	if (copy_from_user(proc_buffer, b, c)) {
            return -EFAULT;
        }

	/* traverse ASCII buffer from proc file and set up ts_data struct */
	memset(ts_data, 0, sizeof (ts_data));
	i = 0;
	j = 1; /* skip TS0 */
	curr = proc_buffer;
	while (curr) {
	   token = strsep(&curr, ":\n");
	   ulTmp=0;
	   /* 3 columns in proc file rx:tx:enable */
	   switch (i%3) {
		case 0: 
		   ulTmp = simple_strtol(token, &endptr, 0); 
		   KMBC_CHECK_SET_TS(ulTmp, ts_cfg[j].rx);
		   break;
		case 1:
		   ulTmp = simple_strtol(token, &endptr, 0); 
		   KMBC_CHECK_SET_TS(ulTmp, ts_cfg[j].tx);
		   break;
		case 2:
		   ulTmp = simple_strtol(token, &endptr, 0); 
		   ts_cfg[j].enable = 0;
		   if (ulTmp == 1) {
		       ts_cfg[j].enable = 1;
		   } 
		   j++; /* important - set on last column */
		   break;
		default: 
		   break;
	   }
	   i++;
	}

	/* setup hotwire map (skip TS 0) */
	j=1;
	for (i=1; i<MAX_TS; i++) {
	     if (ts_data[i].enable) {
	    	 ts_map[j] = i;
		 j++;     
		 printk("ENABLE: %d->%d\n",ts_data[i].rx, ts_data[i].tx);
	     }
	}
	ts_map[0] = j;

	return c;
}




int kmbc_init(void)
{
	int   i, status;
	dev_t dev_num = MKDEV( kmbc_major_num, 0 ); 

	/* Get our dynamic major number */
	status = alloc_chrdev_region( &dev_num, 0, 1, "kmbclow" );

	if( status == 0 )
	{
		kmbc_major_num = MAJOR( dev_num );
	    
		cdev_init( &kmbc_cdev, &kmbc_fops );
		kmbc_cdev.owner = THIS_MODULE;
	    
		status = cdev_add( &kmbc_cdev, dev_num, 1 );

		if( status != 0 )
		{
			printk( KERN_ALERT "kmbc_low:  cdev_add() failed (status=%d).\n", status );
		}
	}
	else
	{
		/* Failed to allocate a major number */
		printk( KERN_ALERT "kmbc_low:  alloc_chrdev_region() failed (status=%d).\n", status );
		return( status );
	}    


	/* Initialize the atomic integers that we use to get our bridge ID and sync data from ISR */
	atomic_set( &kmbc_isr_bridge_id, 0 );
	atomic_set( &kmbc_isr_sync_data, 0 );

	printk("******************\n");
	printk("*   KMBC - DPC   *\n");
	printk("******************\n");

	tasklet_init(&kmbc_tasklet, kmbc_dpc, 0);

	INIT_KMBC_TIMER( kmbc_timer ); /* Only exists in a Unit Test build */

	kmbc_proc_entry = create_proc_entry("kmbc_info", S_IFREG | S_IWUSR, NULL);
	if (kmbc_proc_entry) {
        kmbc_proc_entry->read_proc = (read_proc_t *)kmbc_read_proc;
        kmbc_proc_entry->write_proc = (write_proc_t *)kmbc_write_proc;
	    kmbc_proc_entry->data = (void *)ts_data;
	    proc_buffer = (char*) vmalloc(PAGE_SIZE);
	    if (proc_buffer == NULL) {
	        remove_proc_entry ("kmbc_info", NULL);
	        kmbc_proc_entry = NULL;
        }
	    for (i=0; i<MAX_TS; i++) {
	         ts_data[i].bridge=0;
	         ts_data[i].rx=i;
	         ts_data[i].tx=i;
	         ts_data[i].enable=0;
	    }
	} 

	KMBC_TRACE(( KERN_ALERT "kmbc_low:  Module Loaded at 0x%p (status=%d)\n", kmbc_init, status ));
	return( status );
}

#ifdef CONFIG_COMPAT
static int
kmbc_compat_ioctl(struct file *filep, unsigned int cmd,
                        unsigned long arg)
{
        int err;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,36)
        err = kmbc_ioctl(filep, cmd, arg);
#else
        err = kmbc_ioctl(filep->f_dentry->d_inode, filep, cmd, arg);
#endif
        return err;
}
#endif

void kmbc_cleanup(void)
{
	kmbc_hsi_pool_shutdown(); /* Shutdown here to avoid race condition with unmaps */

	tasklet_kill( &kmbc_tasklet );
	cdev_del( &kmbc_cdev );
	unregister_chrdev_region( MKDEV( kmbc_major_num, 0 ), 1 );

	if (kmbc_proc_entry) {
        remove_proc_entry ("kmbc_info", NULL);
	    vfree(proc_buffer);
	}
	  
	KMBC_TRACE(( KERN_ALERT "kmbc_low:  Module Unloaded from 0x%p\n", kmbc_cleanup ));
}


module_init( kmbc_init );
module_exit( kmbc_cleanup );


MODULE_AUTHOR( "Dialogic Corporation" );
MODULE_DESCRIPTION( "Implements kernel-mode portion of KMBC stack." );
MODULE_LICENSE( "BSD" );
