/**
 * @file  KMBC_low_mm.c
 * @brief Buffer Pool manager within the KMBC_low driver. (LINUX ONLY)
 * @date  June 26, 2006
 *
 * Dialogic CONFIDENTIAL	
 * Copyright 2013 Dialogic Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to 
 * the source code ("Material") are owned by Dialogic Corporation or its suppliers
 * or licensors.  Title to the Material remains with Dialogic Corporation or its 
 * suppliers and licensors.  The Material contains trade secrets and proprietary
 * and confidential information of Dialogic or its suppliers and licensors.  The
 * Material is protected by worldwide copyright and trade secret laws and treaty
 * provisions.  No part of the Material may be used, copied, reproduced, 
 * modified, published, uploaded, posted, transmitted, distributed, or disclosed
 * in any way without Dialogic's prior express written permission.
 * 
 * No license under any patent, copyright, trade secret or other intellectual 
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise.  Any license under such intellectual property rights must be
 * express and approved by Dialogic in writing.
 * 
 * Unless otherwise agreed by Dialogic in writing, you may not remove or alter this
 * notice or any other notice embedded in Materials by Dialogic or Dialogic's 
 * suppliers or licensors in any way.
 */




/* Header Files */
#include <linux/kernel.h>
#include <linux/spinlock.h>
#include "KMBC_low_mm.h"




/* Global Variables */
static int                g_NumBuffers      = 0;
static int                g_NumBridges      = 0;
int                       g_BufferSize      = 0;
static KMBC_LOW_MEM_DESC *g_BufferPool      = NULL;
static spinlock_t         g_PoolLock;
static int                g_LockInitialized = 0;
KMBC_LOW_MEM_BUF          g_BufferMap[8];
extern kmbc_ts            ts_data[];
extern unsigned int       ts_map[];
#define FRAME_SIZE        32

#ifdef KMBC_DEBUG
/***************************************************************************************/
/* The KMBC_TRACE macro provides a way of compiling in a lot more trace data for       */
/* initial development and subsequent debugging.  Most errors will always be logged,   */
/* regardless of the KMBC_DEBUG build option.                                          */
/***************************************************************************************/
#define KMBC_TRACE(x)     printk x
#else
#define KMBC_TRACE(x)
#endif	/* #ifdef KMBC_DEBUG */




int
kmbc_hsi_pool_init( int NumBuffers, int BufferSize )
{
	int            status     = 0;
	int            index      = 0;
	int			   page_index = 0;
	int            num_pages  = 0;
	int            order      = 0;
	void          *temp       = NULL;
	unsigned long  phys_addr  = 0;
	

	/* Because the driver persists longer than the calling application, the driver */
	/* will attempt to reuse a prior set of buffers as long as the number and size */
	/* match the current request (should always be true in practice).              */
	if( g_BufferPool && ( NumBuffers == g_NumBuffers ) && ( BufferSize == g_BufferSize ))
	{
		/* Do Nothing - Initialization is already complete */
		return( status );
	}
	else
	{
		/* If there is already a buffer pool, we need to clean it up first */
		if( g_BufferPool )
		{
			kmbc_hsi_pool_shutdown();
		}
	}


	/* Allocate the pool of buffer descriptors */
	g_BufferPool = kmalloc( NumBuffers * sizeof( KMBC_LOW_MEM_DESC ), GFP_ATOMIC );
		
	if( g_BufferPool )
	{
		memset( g_BufferPool, 0, NumBuffers * sizeof( KMBC_LOW_MEM_DESC ));

		/* If this is the first time in, initialize the spinlock */
		if( g_LockInitialized == 0 )
		{
			spin_lock_init( &g_PoolLock );
			g_LockInitialized = 1;
		}
		
		g_NumBuffers = NumBuffers;
		g_BufferSize = BufferSize;
		

		/* Calculate the order of the allocation for __get_free_pages() */
		if( BufferSize <= PAGE_SIZE )
		{
			order     = 0;
			num_pages = 1;  /* Allocate one page of memory - 4KB on x86 */
		}
		else if( BufferSize <= ( 2 * PAGE_SIZE ))
		{
			order     = 1;
			num_pages = 2;  /* Allocate two pages of memory - 8KB on x86 */
		}
		else if( BufferSize <= ( 4 * PAGE_SIZE ))
		{
			order     = 2;
			num_pages = 4;  /* Allocate four pages of memory - 16KB on x86 */
		}
		else
		{
			order     = 3;
			num_pages = 8;  /* Allocate eight pages of memory - 32KB on x86 */
		}


		KMBC_TRACE(( KERN_ALERT "kmbc_low:  Each HSI Buffer requires %d pages of memory.\n", num_pages ));


		/* Mgmt structure allocated, now allocate the individual buffers */
		spin_lock( &g_PoolLock );
		
		for( index = 0; index < g_NumBuffers; index++ )
		{
			void *page_addr;
			void *actual_buffer;

#ifdef CONFIG_COMPAT
		temp = kmalloc( num_pages * PAGE_SIZE, GFP_KERNEL | __GFP_DMA);
#else
		temp = kmalloc( num_pages * PAGE_SIZE, GFP_KERNEL);
#endif
			
			if( temp )
			{
				/* Round address up to the page boundary */
				actual_buffer = temp;

				/* Need to get the physical address */
				phys_addr = virt_to_phys( actual_buffer );

				/* Mark the pages reserved so we can use remap_page_range */
				for( page_index = 0; page_index < num_pages; page_index++ )
				{
					page_addr = (void *)((unsigned long)actual_buffer + ( page_index * PAGE_SIZE ));
					SetPageReserved( virt_to_page( page_addr ));

					KMBC_TRACE(( KERN_ALERT "kmbc_low:  Marking Page at %p as RESERVED.\n", page_addr ));
				}

				g_BufferPool[index].fUsed     = 0;
				g_BufferPool[index].size      = BufferSize;
				g_BufferPool[index].num_pages = num_pages;
				g_BufferPool[index].order     = order;
				g_BufferPool[index].UserAddr  = NULL; /* will set these when mapped */
				g_BufferPool[index].BlockAddr = temp;
				g_BufferPool[index].KernAddr  = actual_buffer;
				g_BufferPool[index].BusAddr   = phys_addr;

				KMBC_TRACE(( KERN_ALERT "kmbc_low:  Allocated new HSI Buffer (block_addr=%p, kern_addr=%p, phys_addr=0x%lx).\n",
							 temp, actual_buffer, phys_addr ));	

				memset( actual_buffer, 0, BufferSize );
			}
			else
			{
				KMBC_TRACE(( KERN_ALERT "kmbc_low:  Could only allocate %d Buffers.\n", index ));
				g_NumBuffers = index;
				break;
			}
		}
		
		spin_unlock( &g_PoolLock );
		
		KMBC_TRACE(( KERN_ALERT "kmbc_low:  HSI Buffer Pool Initialized (num=%d, size=%d, total_mem=%d).\n",
					g_NumBuffers, g_BufferSize, g_BufferSize * g_NumBuffers ));
	}
	else
	{
		KMBC_TRACE(( KERN_ALERT "kmbc_low:  Could not allocate HSI Buffer Descriptors.\n" ));
		g_NumBuffers = 0;
		g_BufferSize = 0;
		status       = -ENOMEM;
	}

	g_NumBridges=0;
	return( status );
}



void
kmbc_hsi_pool_shutdown( void )
{
	int   index = 0;
	int   i     = 0;
	void *page_addr;

	if( g_BufferPool != NULL )
	{
		spin_lock( &g_PoolLock );

		for( index = 0; index < g_NumBuffers; index++ )
		{
			if( g_BufferPool[index].fUsed )
			{
				KMBC_TRACE(( KERN_ALERT "kmbc_low:  MEMORY LEAK!!! (idx=%ul, addr=%p).\n",
							index, g_BufferPool[index].KernAddr ));
			}
			else
			{
				if( g_BufferPool[index].KernAddr )
				{
					/* Unreserve the pages so that we can free them */
					for( i = 0; i < g_BufferPool[index].num_pages; i++ )
					{
						page_addr = (void *)((unsigned long)g_BufferPool[index].KernAddr + ( i * PAGE_SIZE ));
						ClearPageReserved( virt_to_page( page_addr ));
					}

					kfree( g_BufferPool[index].BlockAddr );
//					free_pages( (unsigned long)g_BufferPool[index].KernAddr, g_BufferPool[index].order );

					memset( &g_BufferPool[index], 0, sizeof( KMBC_LOW_MEM_DESC ));
				}
			}
		}

		spin_unlock( &g_PoolLock );
		kfree( g_BufferPool );
		printk( KERN_ALERT "kmbc_low:  HSI Buffer Pool shut down.\n" );
	}
}



void *
kmbc_hsi_pool_alloc( void )
{
	void *temp  = NULL;
	int   index = 0;

	spin_lock( &g_PoolLock );

	/* Search the descriptors for a free block */
	for( index = 0; index < g_NumBuffers; index++ )
	{
		if( g_BufferPool[index].fUsed == 0 )
		{
			g_BufferPool[index].fUsed = 1;
			temp = g_BufferPool[index].KernAddr;
			break;
		}
	}

	spin_unlock( &g_PoolLock );
	return( temp );
}




void
kmbc_hsi_pool_free( void *buffer )
{
	int not_found = 1;
	int index     = 0;

	spin_lock( &g_PoolLock );

	for( index = 0; index < g_NumBuffers; index++ )
	{
		if(( g_BufferPool[index].KernAddr == buffer ) && ( g_BufferPool[index].fUsed == 1))
		{
//			memset( g_BufferPool[index].KernAddr, 0, g_BufferPool[index].size );
			g_BufferPool[index].fUsed = 0;
			not_found = 0;
			break;
		}
	}

	spin_unlock( &g_PoolLock );

	if( not_found )
	{
		KMBC_TRACE(( KERN_ALERT "kmbc_low:  Attempting to free unallocated HSI buffer (addr=%p).\n", buffer ));
	}
}




void
kmbc_hsi_pool_set_user_addr( void *kern_addr, void *user_addr )
{
	int not_found = 1;
	int index     = 0;

	spin_lock( &g_PoolLock );

	for( index = 0; index < g_NumBuffers; index++ )
	{
		if(( g_BufferPool[index].KernAddr == kern_addr ) && ( g_BufferPool[index].fUsed == 1 ))
		{
			g_BufferPool[index].UserAddr = user_addr;
			not_found = 0;
			break;
		}
	}

	spin_unlock( &g_PoolLock );

	if( not_found )
	{
		KMBC_TRACE(( KERN_ALERT "kmbc_low:  Can't find memory block (addr=%p).\n", kern_addr ));
	}
}



void *
kmbc_hsi_pool_convert_user_addr( void *user_addr )
{
	int   not_found = 1;
	int   index     = 0;
	void *kern_addr = NULL;

	spin_lock( &g_PoolLock );

	for( index = 0; index < g_NumBuffers; index++ )
	{
		if(( g_BufferPool[index].UserAddr == user_addr ) && ( g_BufferPool[index].fUsed == 1 ))
		{
			kern_addr = g_BufferPool[index].KernAddr;
			not_found = 0;
			break;
		}
	}

	spin_unlock( &g_PoolLock );

	if( not_found )
	{
		KMBC_TRACE(( KERN_ALERT "kmbc_low:  Can't find user addr (addr=%p).\n", user_addr ));
	}

	return( kern_addr );
}




unsigned long
kmbc_hsi_pool_get_bus_addr( void *user_addr )
{
	int           not_found = 1;
	int           index     = 0;
	unsigned long bus_addr  = 0;

	spin_lock( &g_PoolLock );

	for( index = 0; index < g_NumBuffers; index++ )
	{
		if(( g_BufferPool[index].UserAddr == user_addr ) && ( g_BufferPool[index].fUsed == 1 ))
		{
			bus_addr  = g_BufferPool[index].BusAddr;
			not_found = 0;
			break;
		}
	}

	spin_unlock( &g_PoolLock );

	if( not_found )
	{
		KMBC_TRACE(( KERN_ALERT "kmbc_low:  Can't find user addr (addr=%p).\n", user_addr ));
	}

	return( bus_addr );
}




void kmbc_hsi_buffer_init(PHSI_BRIDGE_INIT_INPUT info) {
   unsigned char ucTmp;
   unsigned int  index;   

   for (ucTmp = 0; ucTmp < info->numHsiBuffers; ucTmp++) {
        for (index = 0; index < g_NumBuffers; index++ ) {
             if (g_BufferPool[index].BusAddr == (unsigned long)info->hostToBoardBuffers[ucTmp]) {
                 g_BufferMap[info->bridgeId].H2B[ucTmp] = (unsigned int) g_BufferPool[index].KernAddr; 
             }
             if (g_BufferPool[index].BusAddr == (unsigned long)info->boardToHostBuffers[ucTmp]) {
                 g_BufferMap[info->bridgeId].B2H[ucTmp] = (unsigned int) g_BufferPool[index].KernAddr; 
             }
        }

   } 
   g_NumBridges++;

}




void kmbc_hsi_hotwire(int index, int bid) {
   int ts, rx, tx, i;
   void *src, *dest;

   if (ts_map[0] < 2) {
       return;
   }

   /* skip TS 0 */
   for (i=1; i<ts_map[0]; i++) {
        ts = ts_map[i];
        rx = ts_data[ts].rx;
        tx = ts_data[ts].tx;
        src =  (void*)g_BufferMap[bid].B2H[index]+(rx*FRAME_SIZE);
        dest = (void*)g_BufferMap[bid].H2B[index]+(tx*FRAME_SIZE);

        /* hotwire B2H[rx] --> H2B[tx] provided via proc */
        memcpy(dest, src, FRAME_SIZE);
   }
}
