/**
* @file bridgedcallconnected_state.cpp
* @brief Definition of BridgedCallConnectedState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "bridgedcallconnected_state.h"
#include "bridgedcall.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"

//*****************************************************************************
// Function: BridgedCallConnectedState::BridgedCallConnectedState(BridgedCall *pBridgedCall)
// Description: Initializing constructor
// Return:  BridgedCallConnectedState*
// Parameters: BridgedCall *pBridgedCall 
//*****************************************************************************
BridgedCallConnectedState::BridgedCallConnectedState(BridgedCall* pBridgedCall) :
   BridgedCallState(pBridgedCall)
{
}


//*****************************************************************************
// Function: BridgedCallConnectedState::~BridgedCallConnectedState()
// Description: Defstructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedCallConnectedState::~BridgedCallConnectedState()
{
}

//*****************************************************************************
// Function: void BridgedCallConnectedState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallConnectedState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      case APP_BRIDGEDCALL_DISCONNECT:   // fall-thru
      case APP_M3G_ENDPOINT_DISCONNECT:   // fall-thru
      case APP_M3G_ENDPOINT_DISCONNECTED: // fall-thru
      case APP_SIP_ENDPOINT_DISCONNECT:   // fall-thru
         ProcessDisconnect(metaevent);
         break;
      default:
         LOG_ERROR(metaevent.evtdev,"BridgedCall[%d] BridgedCallConnectedState: Unexpected event type: 0x%x\n",
         m_pBridgedCall->Index(),metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void BridgedCallConnectedState::ProcessDisconnect(METAEVENT metaevent)
// Description: Process a disconnect event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallConnectedState::ProcessDisconnect(METAEVENT metaevent)
{
   m_pBridgedCall->ChangeState(DISCONNECTING_STATE);

   switch ( metaevent.evttype )
   {
      case APP_BRIDGEDCALL_DISCONNECT:
         {
            LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallConnectedState rcvd:APP_BRIDGEDCALL_DISCONNECT\n", m_pBridgedCall->Index());
            // Start disconnection of both bridged endpoints
            SIPEndpoint *pSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx( m_pBridgedCall->SipIndex());
            if ( pSIPEndpoint )
            {
               LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallConnectedState: Preparing fake SIP GCEV_DISCONNECT to react to APP_BRIDGEDCALL_DISCONNECT\n");
               METAEVENT tmpMetaevent;
               tmpMetaevent.evttype =  GCEV_DISCONNECTED;
               tmpMetaevent.evtdev = 0;
               pSIPEndpoint->ProcessEvent(tmpMetaevent);
            }

            Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedCall->M3gIndex());
            if ( pEndpoint )
            {
               pEndpoint->StopMedia();

               // ISDN/SS7 disconnect
               if (pEndpoint->IsSigProtocolManaged())  {
                 LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallConnectedState: Preparing fake GCEV_DISCONNECT for ISDN/SS7 connection\n",
                           m_pBridgedCall->Index());
                 METAEVENT tmpMetaevent;
                 tmpMetaevent.evttype =  GCEV_DISCONNECTED;
                 tmpMetaevent.evtdev = 0;
                 ISDNEndpoint *pISDNEndpoint = pEndpoint->GetSigProtocolEndpoint();
                 if (pISDNEndpoint)
                   pISDNEndpoint->ProcessEvent(tmpMetaevent);
               }
            }
            // And if there is a loopback peer, do the same
            Endpoint *pLoopbackEndpoint = pEndpoint->GetLoopbackPeer();
            if (pLoopbackEndpoint && (pLoopbackEndpoint->GetState() != EPState::H245_INACTIVE_STATE)) 
            {
                LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d]BridgedCallConnectedState: Stopping loopback peer\n",m_pBridgedCall->Index());
                pLoopbackEndpoint->StopMedia();

                // A little roundabout to get the associated SIP endpoint...
                BridgedCall *bridgedCall =   EndpointMngr::Instance()->GetCallFromIdx( pLoopbackEndpoint->GetIndex());
                SIPEndpoint *pLoopbackSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx(bridgedCall->SipIndex());
                if ( pLoopbackSIPEndpoint )
                {
                   LOG_ENTRY(metaevent.evtdev, 
                     "BridgedCallConnectedState: Preparing fake SIP GCEV_DISCONNECT to react to APP_BRIDGEDCALL_DISCONNECT for loopback endpoint\n");
                   METAEVENT tmpMetaevent;
                   tmpMetaevent.evttype =  GCEV_DISCONNECTED;
                   tmpMetaevent.evtdev = 0;
                   pLoopbackSIPEndpoint->ProcessEvent(tmpMetaevent);
                }
            }
         }
         break;

      case APP_M3G_ENDPOINT_DISCONNECTED:
      case APP_M3G_ENDPOINT_DISCONNECT:
         {
            LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallConnectedState rcvd:APP_M3G_ENDPOINT_DISCONNECT\n", m_pBridgedCall->Index());

            // Pass the 3G disconnect along to the disconnecting state
            METAEVENT tmpMetaevent;
            tmpMetaevent.evttype =  APP_M3G_ENDPOINT_DISCONNECTED;
            tmpMetaevent.evtdev = 0;
            m_pBridgedCall->ProcessEvent(tmpMetaevent);
         }
         break;

      case APP_SIP_ENDPOINT_DISCONNECT:
         {
            LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallConnectedState rcvd:APP_SIP_ENDPOINT_DISCONNECT\n", m_pBridgedCall->Index());
            Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedCall->M3gIndex());
            if ( pEndpoint )
            {

               // JH Try ISDN/SS7 disconnect
               if (pEndpoint->IsSigProtocolManaged())  {
                 LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallConnectedState: Preparing fake GCEV_DISCONNECT for ISDN/SS7 connection\n",
                           m_pBridgedCall->Index());
                 METAEVENT tmpMetaevent;
                 tmpMetaevent.evttype =  GCEV_DISCONNECTED;
                 tmpMetaevent.evtdev = 0;
                 ISDNEndpoint *pISDNEndpoint = pEndpoint->GetSigProtocolEndpoint();
                 if (pISDNEndpoint)
                   pISDNEndpoint->ProcessEvent(tmpMetaevent);
               }
               else
               {
                 //must call m3g stop media if not ISDN. 
                 pEndpoint->StopMedia();
               }
            }
            // And if there is a loopback peer, do the same
            Endpoint *pLoopbackEndpoint = pEndpoint->GetLoopbackPeer();
            if (pLoopbackEndpoint && (pLoopbackEndpoint->GetState() != EPState::H245_INACTIVE_STATE))
              {
                LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallConnectedState: Stopping loopback peer\n",m_pBridgedCall->Index());
                pLoopbackEndpoint->StopMedia();
              }
         }
         break;
   }
}


