/**
* @file bridgedcalldisconnecting_state.cpp
* @brief Definition of BridgedCallDisconnectingState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "bridgedcalldisconnecting_state.h"
#include "bridgedcall.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"
#include "main.h"

// static constant members:
const int BridgedCallDisconnectingState::SIP_DISCONNECTED   = 0x01;
const int BridgedCallDisconnectingState::M3G_DISCONNECTED   = 0x02;
const int BridgedCallDisconnectingState::FULL_DISCONNECT_NO_CMD = 0x03;
const int BridgedCallDisconnectingState::DISCONNECT_CMD_SENT = 0x04;
const int BridgedCallDisconnectingState::FULL_DISCONNECT_WITH_CMD = 0x7;


//*****************************************************************************
// Function: BridgedCallDisconnectingState::BridgedCallDisconnectingState(BridgedCall *pBridgedCall)
// Description: Initializing constructor
// Return:  BridgedCallDisconnectingState*
// Parameters: BridgedCall *pBridgedCall 
//*****************************************************************************
BridgedCallDisconnectingState::BridgedCallDisconnectingState(BridgedCall* pBridgedCall) :
   BridgedCallState(pBridgedCall)
{
}


//*****************************************************************************
// Function: BridgedCallDisconnectingState::~BridgedCallDisconnectingState()
// Description: Default Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedCallDisconnectingState::~BridgedCallDisconnectingState()
{
}

//*****************************************************************************
// Function: void BridgedCallDisconnectingState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallDisconnectingState::ProcessEvent(METAEVENT metaevent)
{
SIPEndpoint *pSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx( m_pBridgedCall->SipIndex());

   switch ( metaevent.evttype )
   {
      case APP_M3G_ENDPOINT_DISCONNECTED:
      case APP_M3G_ENDPOINT_DISCONNECT:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallDisconnectingState rcvd:APP_M3G_ENDPOINT_DISCONNECT\n",m_pBridgedCall->Index());
         SetStatus(M3G_DISCONNECTED);
         // SetStatus of SIP endpoint if it has already been disconnected
         if(pSIPEndpoint->GetState() == SIPEPState::DROPPING_STATE)
         {
            LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallDisconnectingState SetStatus(SIP_DISCONNECTED)\n",m_pBridgedCall->Index());
            SetStatus(SIP_DISCONNECTED);
         }
         ProcessEndpointDisconnected(metaevent);
         break;

      case APP_SIP_ENDPOINT_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallDisconnectingState rcvd:APP_SIP_ENDPOINT_DISCONNECTED\n",m_pBridgedCall->Index());
         SetStatus(SIP_DISCONNECTED);
         ProcessEndpointDisconnected(metaevent);
         break;

      case APP_SIP_ENDPOINT_DISCONNECT:
         LOG_ENTRY(metaevent.evtdev, "BridgedCall[%d] BridgedCallDisconnectingState rcvd:APP_SIP_ENDPOINT_DISCONNECT - ignored\n",m_pBridgedCall->Index());
         break;

      default:
         LOG_ERROR(metaevent.evtdev,"BridgedCall[%d] BridgedCallDisconnectingState: Unexpected event type: 0x%x\n",
             m_pBridgedCall->Index(),metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void BridgedCallDisconnectingState::ProcessEndpointDisconnected(METAEVENT metaevent)
// Description: Process endpoint disconnected event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedCallDisconnectingState::ProcessEndpointDisconnected(METAEVENT metaevent)
{
   // Need to handle the following situations:
   // 1) Clear channel with a loopback peer
   // 2) Clear channel with no loopback peer. This implies an external gateway and that the SIP
   //    side has disconnected.  So, a "D" command must be sent to the external gateway.
   // 3) ISDN/SS7 internal connection (HMP only)

   Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedCall->M3gIndex());
   if (pEndpoint->IsClearChannel())
   {
      // Clear Channel E1 used, so expect external ISDN gateway 
      // To distinguish a crossover connection from one dti span to another on the MMP from
      // connections to an ISDN gateway, look at the Loopback peer colunm.  If it is zero,
      // the ISDN gateway will tell us what disconnection to make.  
      if (pEndpoint->GetLoopbackPeer() == 0)
      {
         // Send  command to ISDN Gateway to tell it that there is no more 3G activity on a clear channel DS0 
         if (false == IsStatusSet(DISCONNECT_CMD_SENT)) 
         {
            char command[80]; 
            // Use MapID in config file as device to disconnect
            sprintf(command, "D %s", pEndpoint->MapID());
            // TODO - Fix disconnect 3G endpoint
            // JM - Leave out for now. This message to ISDNGW kills ISDNGW app.
               //LOG_ENTRY(0, "BridgedCall[%d] Cmd socket - Sending disconnect command %s to ISDN gateway\n",                                                                               m_pBridgedCall->Index(), command);
               //sendCmdToISDNGW(command);
            SetStatus(DISCONNECT_CMD_SENT);
            LOG_ENTRY(metaevent.evtdev, 
               "BridgedCall[%d] Bitmask for state change to idle state = %d (ISDN GW - want 0x7) boolean returned = %d\n", 
               m_pBridgedCall->Index(),
               m_StatusBitmask, IsIsdnGwCallFullyDisconnected());
         } // end if disconnect command not sent
         if ( IsIsdnGwCallFullyDisconnected() ) 
         {
            m_pBridgedCall->ChangeState(IDLE_STATE);
         }
      } // end if no loopback peer 
      else
      {  // clear channel with loopback connection
         //LOG_ENTRY(metaevent.evtdev, 
         //      "BridgedCall[%d] Bitmask for state change to idle state = %d (No ISDN GW - want 0x3) boolean returned = %d\n", 
         //      m_pBridgedCall->Index(),
         //      m_StatusBitmask, IsNonIsdnGwCallFullyDisconnected());
         if ( IsNonIsdnGwCallFullyDisconnected() )
         {
            m_pBridgedCall->ChangeState(IDLE_STATE);
         }
      }
   }  // end if clear channel

   // if SIP endpoint is still connected
   if (false == IsStatusSet(SIP_DISCONNECTED))
   {
      SIPEndpoint *pSIPEndpoint = EndpointMngr::Instance()->GetSIPEPFromIdx( m_pBridgedCall->SipIndex());
      if ( pSIPEndpoint )
      {
         LOG_ENTRY(metaevent.evtdev, 
                   "BridgedCall[%d] BridgedCallDisconnectingState: Preparing fake SIP GCEV_DISCONNECT to react to APP_M3G_ENDPOINT_DISCONNECT\n",
                    m_pBridgedCall->Index());
         METAEVENT tmpMetaevent;
         tmpMetaevent.evttype =  GCEV_DISCONNECTED;
         tmpMetaevent.evtdev = pSIPEndpoint->GetGCSIPHandle();
         pSIPEndpoint->ProcessEvent(tmpMetaevent);
      }
   }

   //LOG_ENTRY(metaevent.evtdev, 
   //          "BridgedCall[%d] Bitmask for state change to idle state = %d (No ISDN GW - want 0x3) boolean returned = %d\n", 
   //      m_pBridgedCall->Index(),
   //      m_StatusBitmask, IsNonIsdnGwCallFullyDisconnected());

   if ( IsNonIsdnGwCallFullyDisconnected() ) 
   {
      m_pBridgedCall->ChangeState(IDLE_STATE);
   }
}

