/**
* @file bridgedhairp.cpp
* @brief Definition of BridgedHairp class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "logger.h"
#include "endpointmgr.h"
#include "bridgedhairp.h"
#include "locker.h"
#include <string.h>

#include "bridgedhairpopening_state.h"
#include "appevents.h"


// static constant members:
const int BridgedHairp::BRG_CALL_IDX_NOT_SET           = -1;


//*****************************************************************************
// Function: BridgedHairp::BridgedHairp()
// Description: Default constructor
// Return:  BridgedHairp*
// Parameters: none 
//*****************************************************************************
BridgedHairp::BridgedHairp()
{
}

//*****************************************************************************
// Function: BridgedHairp::BridgedHairp(int index, int m3gIndex, int sipIndex)
// Description: Initializing constructor
// Return:  BridgedHairp*
// Parameters: int index 
//             int m3gIndex 
//             int sipIndex 
//*****************************************************************************
BridgedHairp::BridgedHairp(int index, int m3gIndex1, int m3gIndex2):
   m_pCurrentState(0),
   m_Index(index),
   m_m3gIndex1(m3gIndex1),
   m_m3gIndex2(m3gIndex2),
   m_CallerIndex(-1),
   m_CalleeIndex(-1)
{
   strncpy(m_CurrentStateStr, "OPENING        ",MAX_CURRENT_STATE_STR);
   strcpy(m_LastMessageBuff, "");

   m_pBridgeCallOpeningState = new BridgedHairpOpeningState(this);
   m_pBridgedHairpIdleState = new BridgedHairpIdleState(this);
   m_pBridgedHairpConnectingState = new BridgedHairpConnectingState(this);
   m_pBridgeCallConnectedState = new BridgedHairpConnectedState(this);
   m_pBridgedHairpDisconnectingState = new BridgedHairpDisconnectingState(this);
}

//*****************************************************************************
// Function: BridgedHairp::~BridgedHairp()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedHairp::~BridgedHairp()
{
   delete m_pBridgeCallOpeningState;
   delete m_pBridgedHairpIdleState;
   delete m_pBridgedHairpConnectingState;
   delete m_pBridgeCallConnectedState;
   delete m_pBridgedHairpDisconnectingState;
}

//*****************************************************************************
// Function: void BridgedHairp::ChangeState(BridgedHairpState::E_BRIDGEDHAIRPSTATE e_NewCallState)
// Description: Change the object state
// Return: void 
// Parameters: BridgedHairpState::E_BRIDGEDHAIRPSTATE e_NewCallState 
//*****************************************************************************
void BridgedHairp::ChangeState(BridgedHairpState::E_BRIDGEDHAIRPSTATE e_NewCallState)
{
   char oldStateStr[MAX_CURRENT_STATE_STR];
   strcpy(oldStateStr, m_pCurrentState->GetStateStr());
   
   // reset the current state for the next call:
   m_pCurrentState->Reset();

   switch ( e_NewCallState )
   {
      case BridgedHairpState::IDLE_STATE:
         m_pCurrentState = m_pBridgedHairpIdleState;
         break;

      case BridgedHairpState::CONNECTING_STATE:
         m_pCurrentState = m_pBridgedHairpConnectingState;
         break;

      case BridgedHairpState::CONNECTED_STATE:
         m_pCurrentState = m_pBridgeCallConnectedState;
         break;

      case BridgedHairpState::DISCONNECTING_STATE:
         m_pCurrentState = m_pBridgedHairpDisconnectingState;
         break;

      default:
         LOG_ERROR(0,"Unexpected state: %d\n",static_cast<int>(e_NewCallState));
         break;
   }

   // cache the state string in the endpoint
   strcpy(m_CurrentStateStr,m_pCurrentState->GetStateStr());
   LOG_ENTRY(0, "BridgedHairp[%d] State transition: %s ---> %s\n",m_Index, oldStateStr,
		m_CurrentStateStr);
}


//*****************************************************************************
// Function: void BridgedHairp::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: bool 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedHairp::ProcessEvent(METAEVENT metaevent)
{
   m_StateLock.Lock();
   // dispatch event information to appropriate state:
   m_pCurrentState->ProcessEvent(metaevent);
   m_StateLock.Unlock();
}

//*****************************************************************************
// Function: BridgedHairpState::E_BRIDGEDHAIRPSTATE BridgedHairp::GetState()
// Description: Return the current state object
// Return: BridgedHairpState::E_BRIDGEDHAIRPSTATE 
// Parameters: none 
//*****************************************************************************
BridgedHairpState::E_BRIDGEDHAIRPSTATE BridgedHairp::GetState()
{
   m_StateLock.Lock();
   BridgedHairpState::E_BRIDGEDHAIRPSTATE state = m_pCurrentState->GetState();
   m_StateLock.Unlock();
   return state;
}

//*****************************************************************************
// Function: void BridgedHairp::SetLastMessageString(const char *pMessageStr)
// Description: Store as the last messages
// Return: void 
// Parameters: char *pMessageStr 
//*****************************************************************************
void BridgedHairp::SetLastMessageString(const char *pMessageStr)
{
   int length = (MAX_LAST_MSG_STRING > strlen(pMessageStr)) ? strlen(pMessageStr) : MAX_LAST_MSG_STRING-1;
   memset(m_LastMessageBuff, 0, sizeof(char)*MAX_LAST_MSG_STRING);
   strncpy(&m_LastMessageBuff[0], pMessageStr, length);
   m_LastMessageBuff[length] = '\0';
}

//*****************************************************************************
// Function: char* BridgedHairp::GetStateString()
// Description: Get a formatted string describing the current state
// Return: char* 
// Parameters: none 
//*****************************************************************************
char *BridgedHairp::GetStateString()
{
   memset(m_StateBuffer, 0, sizeof(char)*MAX_STATE_STRING);
//   cout << "EP:    STATE:    AUDIO:         Tx    VIDEO:        Tx        MISC." << endl;
//   cout << "                 TxLCN: RxLCN:  CAP:  TxLCN: RxLCN: CAP:    CMD/IND:" << endl;
   sprintf(m_StateBuffer, "BRH%2d: %15s M3G:%d <-> M3G:%d %s",
           m_Index,
           m_CurrentStateStr,
           m_m3gIndex1,
           m_m3gIndex2,
           m_LastMessageBuff);
   return m_StateBuffer;
}

//*****************************************************************************
// Function: void BridgedHairp::Open()
// Description: Set the BridgedHairp to opening
// Return: void 
// Parameters: none 
//*****************************************************************************
void BridgedHairp::Open()
{
   m_pCurrentState = m_pBridgeCallOpeningState;
}


void BridgedHairp::SetCallRoles(int connectionIndex)
{
   // if roles not set this is the first connection
   if (BRG_CALL_IDX_NOT_SET == m_CallerIndex)
   {
      // first connection must be the caller
      m_CallerIndex = connectionIndex;
      // other index must be callee
      m_CalleeIndex = (m_m3gIndex1 == connectionIndex) ? m_m3gIndex2 : m_m3gIndex1;
   }
}
