/**
* @file bridgedhairpconnecting_state.cpp
* @brief Definition of BridgedHairpConnectingState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "bridgedhairpconnecting_state.h"
#include "bridgedhairp.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"
#include "main.h"

// static constant members:
const int BridgedHairpConnectingState::H245_STARTED    = 0x01;
const int BridgedHairpConnectingState::M3G_CONNECTED   = 0x02;
const int BridgedHairpConnectingState::FULLY_CONNECTED = 0x03;


//*****************************************************************************
// Function: BridgedHairpConnectingState::BridgedHairpConnectingState(BridgedHairp *pBridgedHairp)
// Description: Initializing constructor
// Return:  BridgedHairpConnectingState*
// Parameters: BridgedHairp *pBridgedHairp 
//*****************************************************************************
BridgedHairpConnectingState::BridgedHairpConnectingState(BridgedHairp* pBridgedHairp) :
   BridgedHairpState(pBridgedHairp)
{
}


//*****************************************************************************
// Function: BridgedHairpConnectingState::~BridgedHairpConnectingState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedHairpConnectingState::~BridgedHairpConnectingState()
{
}


//*****************************************************************************
// Function: void BridgedHairpConnectingState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedHairpConnectingState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      case APP_H245_START_REQUEST:
         LOG_ENTRY(metaevent.evtdev, "BridgedHairp[%d] BridgedHairpConnectingState rcvd:APP_H245_START_REQUEST\n",m_pBridgedHairp->Index());
         break;

      case APP_H245_STARTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedHairp[%d] BridgedHairpConnectingState rcvd:APP_H245_STARTED\n",m_pBridgedHairp->Index());
         SetStatus(H245_STARTED);
         break;

      case APP_M3G_ENDPOINT_CONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedHairp[%d] BridgedHairpConnectingState rcvd:APP_M3G_ENDPOINT_CONNECTED\n",m_pBridgedHairp->Index());
         SetStatus(M3G_CONNECTED);
         ProcessEndpointConnected(metaevent);
         break;

      case APP_M3G_ENDPOINT_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedHairp[%d] BridgedHairpConnectingState rcvd:APP_M3G_ENDPOINT_DISCONNECTED\n", m_pBridgedHairp->Index());
         break;

      default:
         LOG_ERROR(metaevent.evtdev,"BridgedHairp[%d] BridgedHairpConnectingState: Unexpected event type: 0x%x\n",
         m_pBridgedHairp->Index(),metaevent.evttype);
         break;
   }
}


//*****************************************************************************
// Function: void BridgedHairpConnectingState::ProcessEndpointConnected(METAEVENT metaevent)
// Description: Process an endpoint connected event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedHairpConnectingState::ProcessEndpointConnected(METAEVENT metaevent)
{
  // if this is the first call, it must be the caller
  if (BridgedHairp::BRG_CALL_IDX_NOT_SET == m_pBridgedHairp->CallerIndex()) 
  {
     // the index of the caller is passed within the metaevent
     m_pBridgedHairp->SetCallRoles(*(int *)metaevent.evtdatap);

     int callerIndex = m_pBridgedHairp->CallerIndex();
     int calleeIndex = m_pBridgedHairp->CalleeIndex();
     Endpoint* pCallerEP = EndpointMngr::Instance()->GetEPFromIdx(callerIndex);;
     Endpoint* pCalleeEP = EndpointMngr::Instance()->GetEPFromIdx(calleeIndex);

     // This side of the 3G hairpin connection is ready.  Now trigger an outbound call
     // on the other side if it is idle and can be used for the new call
     if (pCalleeEP->GetState() == EPState::H245_INACTIVE_STATE)  
     {
        LOG_ENTRY(metaevent.evtdev, "BridgedHairp[%d] Starting call on 3G EP Index = %d\n",
                  m_pBridgedHairp->Index(), calleeIndex);

        // stop Rx media for the caller endpoint by re-starting to Tx-only
        pCallerEP->StopRxMedia();
        pCallerEP->ChangeState(EPState::RX_MEDIA_STOPPED_STATE);

        // start the callee endpoint associated with the hairpin
        pCalleeEP->ChangeState(EPState::H245_STARTING_STATE);
        pCalleeEP->StartH245();

        //  start its loopback peer or send connection command to ISDN GW
        if (pCalleeEP->GetLoopbackPeer())  
        {
           LOG_ENTRY(metaevent.evtdev, "BridgedHairp[%d] Starting call on 3G EP Index = %d\n",
                     m_pBridgedHairp->Index(), pCalleeEP->GetLoopbackPeer()->GetIndex());
           pCalleeEP->GetLoopbackPeer()->ChangeState(EPState::H245_STARTING_STATE);
           pCalleeEP->GetLoopbackPeer()->StartH245();
        }
        else  
        {
           char command[80]; 
           // What to connect to is taken from the MapID field in the config file 
           sprintf(command, "C %s", pCalleeEP->MapID());
           LOG_ENTRY(0, "BridgedHairp[%d] Sending connect command %s to ISDN gateway\n",
                     m_pBridgedHairp->Index(), command);
           sendCmdToISDNGW(command); 
        }
     }
     else
     {
       LOG_ERROR(metaevent.evtdev,"BridgedHairp[%d] Expected 3G endpoint not in inactive state."
                 "No call made\n", m_pBridgedHairp->Index());
     }
  }
  else  // else this must be the second connection for the hairpinned callee endpoint
  {
     // 3G endpoint above has been started
     m_pBridgedHairp->ChangeState(CONNECTED_STATE);

     Endpoint* pCallerEP = EndpointMngr::Instance()->GetEPFromIdx(m_pBridgedHairp->CallerIndex());
     Endpoint* pCalleeEP = EndpointMngr::Instance()->GetEPFromIdx(m_pBridgedHairp->CalleeIndex());

     // restart receiving media on caller call leg
     pCallerEP->RestartFDXMedia();

     // send VFU on both call legs to initiate I-Frames
     pCallerEP->SendFastVideoUpdate();
     pCalleeEP->SendFastVideoUpdate();
  }
}

