/**
* @file bridgedrtspidle_state.cpp
* @brief Definition of BridgedRtspIdleState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "bridgedrtspidle_state.h"
#include "bridgedrtsp.h"
#include "endpointmgr.h"
#include "logger.h"
#include "appevents.h"
#include "epstate.h"

// static constant members:
const int BridgedRtspIdleState::H245_STARTED    = 0x01;
const int BridgedRtspIdleState::M3G_CONNECTED   = 0x02;
const int BridgedRtspIdleState::RTSP_READY     = 0x03;

//*****************************************************************************
// Function: BridgedRtspIdleState::BridgedRtspIdleState(BridgedRtsp *pBridgedRtsp)
// Description: Initializing constructor
// Return:  BridgedRtspIdleState*
// Parameters: BridgedRtsp *pBridgedRtsp 
//*****************************************************************************
BridgedRtspIdleState::BridgedRtspIdleState(BridgedRtsp* pBridgedRtsp) :
   BridgedRtspState(pBridgedRtsp)
{
}


//*****************************************************************************
// Function: BridgedRtspIdleState::~BridgedRtspIdleState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
BridgedRtspIdleState::~BridgedRtspIdleState()
{
}

//*****************************************************************************
// Function: void BridgedRtspIdleState::ProcessEvent(METAEVENT metaevent)
// Description: Process an event
// Return: none
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedRtspIdleState::ProcessEvent(METAEVENT metaevent)
{
   switch ( metaevent.evttype )
   {
      case APP_H245_START_REQUEST:
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] BridgedRtspIdleState rcvd:APP_H245_START_REQUEST\n",m_pBridgedRtsp->RTSPIndex());
         break;
      case APP_RTSP_ENDPOINT_PLAYING:
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] BridgedRtspIdleState rcvd:APP_RTSP_ENDPOINT_PLAYING\n",m_pBridgedRtsp->RTSPIndex());
	 ProcessRTSPPlaying(metaevent);
         break;
      case APP_RTSP_ENDPOINT_RECORDING:
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] BridgedRtspIdleState rcvd:APP_RTSP_ENDPOINT_RECORDING\n",m_pBridgedRtsp->RTSPIndex());
	 ProcessRTSPRecording(metaevent);
         break;
      case APP_RTSP_ENDPOINT_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] BridgedRtspIdleState rcvd:APP_RTSP_ENDPOINT_DISCONNECTED\n",m_pBridgedRtsp->RTSPIndex());
         break;
      case APP_H245_STARTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] BridgedRtspIdleState rcvd:APP_H245_STARTED\n",m_pBridgedRtsp->RTSPIndex());
         m_StatusBitmask |= H245_STARTED;
         ProcessEndpointConnected(metaevent);
         break;
      case APP_M3G_ENDPOINT_CONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] BridgedRtspIdleState rcvd:APP_M3G_ENDPOINT_CONNECTED\n",m_pBridgedRtsp->RTSPIndex());
         m_StatusBitmask |= M3G_CONNECTED;
         ProcessEndpointConnected(metaevent);
         break;
      case APP_M3G_ENDPOINT_DISCONNECT:
         ProcessEndpointDisconnect(metaevent);
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] BridgedRtspIdleState rcvd:APP_M3G_ENDPOINT_DISCONNECT\n",m_pBridgedRtsp->RTSPIndex());
         break;
      case APP_M3G_ENDPOINT_DISCONNECTED:
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] BridgedRtspIdleState rcvd:APP_M3G_ENDPOINT_DISCONNECTED in BridgedRtspIdleState\n",m_pBridgedRtsp->RTSPIndex());
         break;
      break;

      default:
         LOG_ERROR(metaevent.evtdev,"BridgedRtsp[%d] BridgedRtspIdleState: Unexpected event type: 0x%x\n",
		   m_pBridgedRtsp->RTSPIndex(),metaevent.evttype);
         break;
   }
}

//*****************************************************************************
// Function: void BridgedRtspIdleState::ProcessRTSPPlaying(METAEVENT metaevent)
// Description: Process an RTSP busy  event
// Return: void 
// Parameters: METAEVENT metaevent 
//*****************************************************************************
void BridgedRtspIdleState::ProcessRTSPPlaying(METAEVENT metaevent)
{
      // Advance BridgedRtsp's state machine to PLAYING
      m_pBridgedRtsp->ChangeState(PLAYING_STATE);
}

void BridgedRtspIdleState::ProcessRTSPRecording(METAEVENT metaevent)
{
      // Advance BridgedRtsp's state machine to RECORDING
      m_pBridgedRtsp->ChangeState(RECORDING_STATE);
}

void BridgedRtspIdleState::ProcessEndpointConnected(METAEVENT metaevent)
{
   if (IsBridgedRtspReady())  {
     RTSPEndpoint *pRTSPEndpoint = EndpointMngr::Instance()->GetRTSPEPFromIdx( m_pBridgedRtsp->RTSPIndex());
     Endpoint *pEndpoint = EndpointMngr::Instance()->GetEPFromIdx( m_pBridgedRtsp->M3gIndex());
     if ( pRTSPEndpoint )   {
       // Set the selected 3G coder for RTSP
       LOG_ENTRY(0, "BridgedRtsp[%d] 3GSelectedVidCoder=%s, 3GSelectedAudioCoder=%s:\n",
                 m_pBridgedRtsp->RTSPIndex(),
                 pEndpoint->GetSelVidCoderStr(),
                 pEndpoint->GetSelAudCoderStr());
       pRTSPEndpoint->SetSelectedVideoCoder(pEndpoint->GetSelectedVideoCoder());
       pRTSPEndpoint->SetSelectedAudioCoder(pEndpoint->GetSelectedAudioCoder());
       
       // Start a play. Play terminates with a  single DTMF/UII or a disconnect.
       if (!strcmp(EndpointMngr::Instance()->GetRTSPOperation(), "PLAY"))  {
         pRTSPEndpoint->PlayVideoFile();
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] Starting video play\n",m_pBridgedRtsp->RTSPIndex());
       } 
       else  {
         // Start a record.  The recording is terminated with a single DTMF/UII or a disconnect.
         pRTSPEndpoint->RecordVideoFile();
         LOG_ENTRY(metaevent.evtdev, "BridgedRtsp[%d] Video record not supported yet.\n",m_pBridgedRtsp->RTSPIndex());
       }
     }
   }
}

void BridgedRtspIdleState::ProcessEndpointDisconnect(METAEVENT metaevent)
{
   if (IsBridgedRtspReady())  {
     RTSPEndpoint *pRTSPEndpoint = EndpointMngr::Instance()->GetRTSPEPFromIdx( m_pBridgedRtsp->RTSPIndex());
     if ( pRTSPEndpoint )   {
       // Stop IPM streaming
         pRTSPEndpoint->StopMedia();
     }
   }
}

