/**
* @file config.cpp
* @brief Definition of DemoConfig class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "main.h"
#include "config.h"
#include "logger.h"
#include "endpointmgr.h"
#include <iostream>
#include <sstream>
#include <fstream>
#include <string>

using namespace std;

//*****************************************************************************
// Function: DemoConfig::DemoConfig()
// Description: Initializing constructor
// Return:  DemoConfig*
// Parameters: none 
//*****************************************************************************
DemoConfig::DemoConfig() :
   m_LoggingEnabled(true)
{
   strcpy(m_ConfigFile,"config.txt");
   strcpy(m_LogFile,"logfile.txt");
}


//*****************************************************************************
// Function: DemoConfig::~DemoConfig()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
DemoConfig::~DemoConfig()
{
}


//*****************************************************************************
// Function: int DemoConfig::SetConfigFile(const char *fileName)
// Description: Set the configuration filename
// Return: int 
// Parameters: const char *fileName 
//*****************************************************************************
int DemoConfig::SetConfigFile(const char* fileName)
{
   if ((MAX_FILE_NAME-1) < strlen(fileName))
   {
      LOG_ERROR(0, "Configuration file name too long:%s(Max:%d chars) \n", fileName, MAX_FILE_NAME);
      cout << "ERROR: Configuration file name too long: " << fileName << endl;
      return B3G_ERROR;
   }

   strcpy(m_ConfigFile, fileName);
   return B3G_SUCCESS;
}


//*****************************************************************************
// Function: int DemoConfig::SetLogFile(const char *fileName)
// Description: Set the log filename
// Return: int 
// Parameters: const char *fileName 
//*****************************************************************************
int DemoConfig::SetLogFile(const char* fileName)
{
   // NULL fileName implies logging disabled
   if (NULL == fileName)
   {
      SetLoggingEnabled(false);
      return B3G_SUCCESS;
   }

   if ((MAX_FILE_NAME-1) < strlen(fileName))
   {
      LOG_ERROR(0, "Log file name too long:%s(Max:%d chars) \n", fileName, MAX_FILE_NAME);
      cout << "ERROR: Log file name too long: " << fileName << endl;
      return B3G_ERROR;
   }

   strcpy(m_LogFile, fileName);
   return B3G_SUCCESS;
}


//*****************************************************************************
// Function: int DemoConfig::ParseConfigFile()
// Description: Parse the configuration file`
// Return: int 
// Parameters: none 
//*****************************************************************************
int DemoConfig::ParseConfigFile()
{
   ifstream confFile(m_ConfigFile);
   int  retVal = B3G_SUCCESS;

   if(!confFile)
   {
      LOG_ERROR(0,"Cannot open configurtion file: %s\n", m_ConfigFile);
      cout << "ERROR: Cannot open configuration file " << m_ConfigFile << endl;
      return B3G_ERROR; 
   }

   // Parse configuration file line by line:
   unsigned int lineNum=0;
   string lineStr;

   while(getline(confFile, lineStr))
   {
      lineNum++;

      // ignore comments or blank lines
      if ((0 == lineStr.size()) || 
         (lineStr[0] == '#') || 
         (lineStr[0] == '\r') || 
         (lineStr[0] == '\n'))
      {
          continue;  // ignore and read the next line
      }

      // construct input stream to fascilitate logging
      istringstream istrLine(lineStr);
      char szLineStr[512] = {'\0'};
      istrLine.get(szLineStr,512);
      LOG_ENTRY(0,"%d: %s\n",lineNum,szLineStr);
      if (B3G_ERROR == ParseLine(lineStr))
      {
         LOG_ERROR(0,"Parse error in configuration file:%s, line:%d\n", m_ConfigFile, lineNum);
         cout << "ERROR: Parse error in configuration file " << m_ConfigFile 
              << " line:" << lineNum << endl;
         retVal = B3G_ERROR;
         break;
      }
   }

   confFile.close();
   return retVal;
}

//*****************************************************************************
// Function: int DemoConfig::ParseLine(string& lineStr)
// Description: Parse a line from the configuration file
// Return: int 
// Parameters: string& lineStr
//*****************************************************************************
int DemoConfig::ParseLine(string& lineStr)
{
   // construct input stream to simplify parsing:
   istringstream istrLine(lineStr);

   char command[TOKEN_SIZE] = {'\0'};
   istrLine >> command;

   Capitalize(command);

   // configuration parameters denoted by ':' character
   if (NULL != strstr(command, "LOG:"))
   {
      return ParseLogging(istrLine);
   }
   if (NULL != strstr(command, "EP:"))
   {
      return ParseEP(istrLine);
   }
   if (NULL != strstr(command, "PRM:"))
   {
      return ParseParms(istrLine);
   }
   if (NULL != strstr(command, "ENEV:"))
   {
      return ParseEvents(istrLine, true);
   }
   if (NULL != strstr(command, "DSEV:"))
   {
      return ParseEvents(istrLine);
   }
   if (NULL != strstr(command, "HSES:"))
   {
      return ParseH223Defaults(istrLine);
   }
   if (NULL != strstr(command, "HCAP:"))
   {
      return ParseH223Caps(istrLine);
   }
   if (NULL != strstr(command, "MCAP:"))
   {
      return ParseMediaCaps(istrLine);
   }
   if (NULL != strstr(command, "HOLC:"))
   {
      return ParseH223OLCParms(istrLine);
   }
   if (NULL != strstr(command, "H264:"))
   {
      return ParseH264Parms(istrLine);
   }
   if (NULL != strstr(command, "ETRC:"))
   {
      return ParseEPTracing(istrLine);
   }
   if (NULL != strstr(command, "BTRC:"))
   {
      return ParseBrdTracing(istrLine);
   }
   if (NULL != strstr(command, "DCI:"))
   {
      return ParseDCI(istrLine);
   }
   if (NULL != strstr(command, "VI:"))
   {
      return ParseVendorId(istrLine);
   }
   if (NULL != strstr(command, "SIP_INFO_DTMF:"))
   {
      return ParseSIPInfoDTMF(istrLine);
   }
   if (NULL != strstr(command, "SIPVID:"))
   {
      return ParseSIPVideo(istrLine);
   }
   if (NULL != strstr(command, "SIPAUD:"))
   {
      return ParseSIPAudio(istrLine);
   }
   if (NULL != strstr(command, "SIPINFO:"))
   {
      return ParseSIPInfo(istrLine);
   }
   if (NULL != strstr(command, "CMDINFO:"))
   {
      return ParseCmdInfo(istrLine);
   }
   if (NULL != strstr(command, "MMINFO:"))
   {
      return ParseMMInfo(istrLine);
   }
   if (NULL != strstr(command, "NBUPINFO:"))
   {
      return ParseNbUPInfo(istrLine);
   }
#ifdef USE_RTSP
   if (NULL != strstr(command, "RTSPINFO:"))
   {
      return ParseRTSPInfo(lineStr);
   }
#else
   if (NULL != strstr(command, "RTSPINFO:"))
   { 
      LOG_ENTRY(0, "RTSPINFO defined but no RTSP support (define USE_RTSP in Makefile).\n");
      return B3G_SUCCESS;
   }
#endif
   // no configuration field found on line, return error:
   return B3G_ERROR;
}

#ifdef USE_RTSP
int DemoConfig::ParseRTSPInfo(string& lineStr)
{
   char dtmf = '0';
   char command[FNAME_SIZE] = {'\0'};
   char rtspUrl[FNAME_SIZE] = {'\0'};

   istringstream istrLine(lineStr);

   if (istrLine >> command >> dtmf >> rtspUrl)
   {
      RTSPEndpoint::AddUrl (dtmf, rtspUrl);
      cerr << "RTSP URL, dtmf '"<< dtmf << "', URL = '"<< rtspUrl <<"'" << endl;
   }
   else
      return B3G_ERROR;

   return B3G_SUCCESS;
}
#endif

int DemoConfig::ParseLogging(istringstream& istrLine)
{
   char token[TOKEN_SIZE] = {'\0'};

   // parse line for boolean
   if ((istrLine >> token >> ws) && istrLine.eof())
   {
      Capitalize(token);
      bool isLoggingEnabled = strstr(token, "ON") ? true : false;
      if (false == isLoggingEnabled)
      {
          // a NULL file name will disable logging:
          SetLogFile(NULL);
      }
      return B3G_SUCCESS;
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseEP(istringstream& istrLine)
{
   char epType[TOKEN_SIZE] = {'\0'};
   char controlName[TOKEN_SIZE] = {'\0'};
   char streamName[TOKEN_SIZE] = {'\0'};
   char mapID[TOKEN_SIZE] = {'\0'};
   char audXcode[TOKEN_SIZE] = {'\0'};
   char vidXcode[TOKEN_SIZE] = {'\0'};
   unsigned int numEP = 0;
   unsigned int numPeer = 0;
   unsigned int numLoopback = 0;

   // parse tokens
   if (istrLine >> numEP >> epType >> controlName >> streamName >> numPeer 
                >> numLoopback >> mapID >> audXcode >> vidXcode)
   {
      // verify endpoint type
      switch (toupper(epType[0]))
      {
         case 'R':
            switch (toupper(streamName[0]))
            {
               case 'I':
               break;
               default:
                return B3G_ERROR;
            }
            break;
         // 3G (m3g):
         case 'M':
         {
            // verify stream name is either "dti", "Idti" (ISDN), ipm or "none"
            switch (toupper(streamName[0]))
            {
               case 'D':   // fall-thru
               case 'I':
                  break;

               case 'N':   // "none"
                  // if loopback specified, verify a device is specified and not "none"
                  if (0 == numLoopback) 
                  {
                      LOG_ENTRY(0, "Must have loopback endpoint if no stream device\n");
                      cout << "ERROR: Must have loopback endpoint if no stream device" << endl;
                      return B3G_ERROR;
                  }
                  break;

               default: // unrecognized
                  return B3G_ERROR;
            }
            break;
         }

         case 'S':  // sip (fall-thru)
         case 'V':  // video multimedia
            break;

         default: // unrecognized endpoint type
             return B3G_ERROR;
      }
      
      Capitalize(vidXcode);
      Capitalize(audXcode);
      bool isVidXcdrModeEnabled = (vidXcode[0] == 'T') ? true : false;
      bool isAudXcdrModeEnabled = (audXcode[0] == 'T') ? true : false;
      //LOG_ENTRY(0, "EP:%d, Axcode=%d, Vxcode=%d\n",numEP, isVidXcdrModeEnabled, isAudXcdrModeEnabled);
 
#ifdef ISDN_CALL_OUT
      char dialString[LINE_BUFFER_SIZE] = {'\0'};
      istrLine >> dialString;

      return (EndpointMngr::Instance()->AddEndpoint(numEP, epType, controlName, streamName,
              numPeer, numLoopback, mapID, isAudXcdrModeEnabled, isVidXcdrModeEnabled, dialString));
#else
      return (EndpointMngr::Instance()->AddEndpoint(numEP, epType, controlName, streamName,
              numPeer, numLoopback, mapID, isAudXcdrModeEnabled, isVidXcdrModeEnabled));
#endif
   }
   else  // invalid token format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseParms(istringstream& istrLine)
{
   static const char* parameterStr[] =
   {
      "DIGIT_TO_MEDIA",
      "DIGIT_TO_H245",
      "FASTUPDATE_TO_MEDIA",
      "FASTUPDATE_TO_H245",
      "unsupported",          // place holder for future supported parameter
      "unsupported",          // place holder for future supported parameter
      "unsupported",          // place holder for future supported parameter
      "unsupported",          // place holder for future supported parameter
      "SKEWINDICATION",
      "AUDIOVISUALSYNC",
      "H245_TERMINAL_TYPE",
      "MAX_CCSRL_SEGMENT",
      "RETRANSMIT_ON_IDLE",
      "AMR_PAYLOAD_FORMAT",
      "unsupported",          // use tracing function in lieu of debug parameters
      "unsupported",          // use tracing function in lieu of debug parameters
      "unsupported",          // use tracing function in lieu of debug parameters
      "unsupported",          // use tracing function in lieu of debug parameters
      "unsupported",          // use tracing function in lieu of debug parameters
      "placeholder",          // DCI set via separate command
      "placeholder",          // DCI set via separate command
      "TX_SKEW_ADJUSTMENT",
      "RX_SKEW_ADJUSTMENT",
      "PRM_VIDEO_BIT_RATE",
      "PRM_VIDEO_FRAME_RATE",
      "PRM_EARLY_MES",
      "PRM_AUTO_VFU_PERIOD",
      "PRM_H223_SYNC_TIMER",
      "placeholder",
      "placeholder",          // H264 DCI set via separate command
      "PRM_IFRAME_NOTIFY_CONTROL_MASK"

   };

   char token[TOKEN_SIZE] = {'\0'};
   int intVal = 0;

   istrLine >> token;
   Capitalize(token);

   for (unsigned int parmId=0; 
        parmId<(sizeof(parameterStr)/sizeof(const char *));
        parmId++)
   {
      if (NULL != strstr(token, parameterStr[parmId]))
      {
         istrLine >> intVal;
         return (EndpointMngr::Instance()->AddParameter(parmId, intVal));
      }
   }

   // no matching parameter found
   return B3G_ERROR;
}


int DemoConfig::ParseEvents(istringstream& istrLine, bool isEnableEvt)
{
   static const char* eventStr[] = 
   {
      "H245_UII_EVT",
      "FASTUPDATE_EVT",
      "TEMP_SPAT_TRDFF_EVT",
      "VIDEO_FREEZE_EVT",
      "SYNC_GOB_EVT",
      "unsupported",         // place holder for future supported event
      "SKEW_INDICATION_EVT",
      "VERBOSE_MSD_EVT",
      "MES_EVTS_EVT",
      "MONA_PREF_MSG_EVT",
      "REMOTE_VENDORID_EVT",
      "CALL_STATISTICS_EVT", 
      "IFRAME_EVT", 
      "IFRAME_DCI_CHANGE_EVT"
   };

   char token[TOKEN_SIZE] = {'\0'};
   Capitalize(token);

   // parse line for event token
   if ((istrLine >> token >> ws) && istrLine.eof())
   {
      for (unsigned int evtId=0; 
           evtId<(sizeof(eventStr)/sizeof(const char *)); 
           evtId++)
      {
         if (NULL != strstr(token, eventStr[evtId]))
            return (EndpointMngr::Instance()->AddEvent((1<<evtId), isEnableEvt));
      }
   }

   // no matching event string found or token had invalid format
   return B3G_ERROR;
}


int DemoConfig::ParseH223Defaults(istringstream& istrLine)
{
   static const char* muxLVLStr[] = 
   {
      "L0",
      "L1",
      "L1D",
      "L2",
      "L2H",
      "L3",
      "L3H"
   };

   char token[TOKEN_SIZE] = {'\0'};
   char defMuxLvlStr[TOKEN_SIZE] = {'\0'};
   unsigned int ALSDUSize = 0;
   char c_IsWNSRP = '\0';
   char c_IsMMPP = '\0';
   char c_IsMONA = '\0';

   // parse line for six tokens
   if ((istrLine >> token >> defMuxLvlStr >> ALSDUSize >> c_IsWNSRP >> c_IsMMPP >> c_IsMONA >> ws)
        && istrLine.eof())
   {
      unsigned int startEP = 0;
      unsigned int endEP = 0;
      if (0 == sscanf(token, "%d-%d",&startEP, &endEP))
      {
         return B3G_ERROR;
      }

      if (0 == endEP)
      {
         endEP = startEP;
      }

      M3G_H223_SESSION epH223Session;
      INIT_M3G_H223_SESSION(&epH223Session);

      Capitalize(defMuxLvlStr);
      for (unsigned int muxLVL =0; 
           muxLVL<(sizeof(muxLVLStr)/sizeof(const char *));
           muxLVL++)
      {
         if (0 == strcmp(defMuxLvlStr, muxLVLStr[muxLVL]))
         {
            epH223Session.defaultH223MuxLevel = static_cast<M3G_E_H223_MUX_LVL_TYPE>(muxLVL);
            break;
         }
      }

      epH223Session.maxALSDUSize = ALSDUSize;
      epH223Session.isWNSRPEnabled = ('T' == toupper(c_IsWNSRP)) ? M3G_TRUE : M3G_FALSE;
      epH223Session.isMultipleMsgsPerPdu = ('T' == toupper(c_IsMMPP)) ? M3G_TRUE : M3G_FALSE;
#ifdef MONA
      epH223Session.isMONAEnabled = ('T' == toupper(c_IsMONA)) ? M3G_TRUE : M3G_FALSE;
#endif

      return (EndpointMngr::Instance()->InitH223Session(startEP, endEP, &epH223Session));
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseH223Caps(istringstream& istrLine)
{
   char token[TOKEN_SIZE] = {'\0'};
   unsigned int AL = 0;
   unsigned int AL2Size = 0;
   unsigned int AL3Size = 0;
   unsigned int bitRate = 0;
   char c_IsAnnexA = '\0';
   char c_IsADblFlag = '\0';
   char c_IsAnnexB = '\0';
   char c_IsBHdr = '\0';
   char c_IsMblOpXmt = '\0';
   char c_IsRSCode = '\0';

   // parse line for 11 tokens
   if ((istrLine >> token >> hex >> AL >> dec >> AL2Size >> AL3Size >> c_IsAnnexA >> c_IsADblFlag
        >> c_IsAnnexB >> c_IsBHdr >> c_IsRSCode >> c_IsMblOpXmt >> bitRate >> ws)
        && istrLine.eof())
   {
      unsigned int startEP = 0;
      unsigned int endEP = 0;
      if (0 == sscanf(token, "%d-%d",&startEP, &endEP))
      {
         return B3G_ERROR;
      }

      if (0 == endEP)
      {
         endEP = startEP;
      }

      LOCAL_H223_CAPS epH223Caps;
      epH223Caps.AL = AL;
      epH223Caps.AL2Size = AL2Size;
      epH223Caps.AL3Size = AL3Size;
      epH223Caps.isAnnexA = ('T' == toupper(c_IsAnnexA)) ? true : false;
      epH223Caps.isADblFlag = ('T' == toupper(c_IsADblFlag)) ? true : false;
      epH223Caps.isAnnexB = ('T' == toupper(c_IsAnnexB)) ? true : false;
      epH223Caps.isBHdr = ('T' == toupper(c_IsBHdr)) ? true : false;
      epH223Caps.isRSCode = ('T' == toupper(c_IsRSCode)) ? true : false;
      epH223Caps.isMblOpXmt = ('T' == toupper(c_IsMblOpXmt)) ? true : false;;
      epH223Caps.bitRate = bitRate;

      return (EndpointMngr::Instance()->InitH223Caps(startEP, endEP, &epH223Caps));
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseMediaCaps(istringstream& istrLine)
{
   static const char* termCapStr[] =
   { 
      "G723",
      "AMR",
      "H263",
      "MPEG4",
      "H264",
   };

   char token[TOKEN_SIZE] = {'\0'};
   char cap1Str[TOKEN_SIZE] = {'\0'};
   char cap2Str[TOKEN_SIZE] = {'\0'};
   char cap3Str[TOKEN_SIZE] = {'\0'};
   char cap4Str[TOKEN_SIZE] = {'\0'};
   char cap5Str[TOKEN_SIZE] = {'\0'};

   // parse line for at least three tokens: one token plus minimum one audio and one video cap
   if (istrLine >> token >> cap1Str >> cap2Str )
   {
      // read optional a/v caps if specified:
      istrLine >> cap3Str >> cap4Str >> cap5Str;

      unsigned int startEP = 0;
      unsigned int endEP = 0;
      if (0 == sscanf(token, "%d-%d",&startEP, &endEP))
      {
         return B3G_ERROR;
      }

      if (0 == endEP)
      {
         endEP = startEP;
      }

      MEDIA_CAP preferredMediaList[MAX_MEDIA_CAPS];

      char* capStrTbl[MAX_MEDIA_CAPS];
      capStrTbl[0] = cap1Str;
      capStrTbl[1] = cap2Str;
      capStrTbl[2] = cap3Str;
      capStrTbl[3] = cap4Str;
      capStrTbl[4] = cap5Str;

      unsigned int parmNum;
      for (parmNum=0; parmNum<MAX_MEDIA_CAPS; parmNum++)
      {
         if ('\0' == capStrTbl[parmNum][0])
         {
            break;  //  exit loop if less than four capabilities
         }
         for (unsigned int capType =0; 
              capType<(sizeof(termCapStr)/sizeof(const char *));
              capType++)
         {
            Capitalize(capStrTbl[parmNum]);
            if (NULL != strstr(capStrTbl[parmNum], termCapStr[capType]))
            {
               preferredMediaList[parmNum] = static_cast<MEDIA_CAP>(capType+1);
               break;
            }
                
         }
      }
      return EndpointMngr::Instance()->InitMediaCaps(startEP, endEP, parmNum, preferredMediaList);
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseH223OLCParms(istringstream& istrLine)
{
   static const char* olcALTypeStr[] = 
   {
      "AL1_FRM",
      "AL1",
      "AL2_SEQ",
      "AL2",
      "AL3"
   };

   char token[TOKEN_SIZE] = {'\0'};
   char AAL[TOKEN_SIZE] = {'\0'};
   char VAL[TOKEN_SIZE] = {'\0'};
   char c_IsAudioSeg = '\0';
   char c_IsVideoSeg = '\0';
   unsigned int  AL3CntFldSz =0;
   unsigned int  AL3SndBufSz =0;

   // parse line for seven tokens
   if ((istrLine >> token >> AAL >> VAL >> c_IsAudioSeg >> c_IsVideoSeg >> AL3CntFldSz 
        >> AL3SndBufSz >> ws) && istrLine.eof())
   {
      unsigned int startEP = 0;
      unsigned int endEP = 0;
      if (0 == sscanf(token, "%d-%d",&startEP, &endEP))
      {
         return B3G_ERROR;
      }

      if (0 == endEP)
      {
         endEP = startEP;
      }

      LOCAL_H223_LC_PARAMS H223LcParams;

      H223LcParams.audioAL = M3G_E_AL2_WITH_SEQ_NUMS;  // default
      Capitalize(AAL);

      for (unsigned int AALTyp =0; 
           AALTyp<(sizeof(olcALTypeStr)/sizeof(const char *));
           AALTyp++)
      {
         if (NULL != strstr(AAL, olcALTypeStr[AALTyp]))
         {
            //H223LcParams.audioAL = static_cast<M3G_E_H223_MUX_LVL_TYPE>(AALTyp);
            H223LcParams.audioAL = static_cast<M3G_E_ADAPTATION_LYR_TYPE>(AALTyp);
            break;
         }
      }

      H223LcParams.videoAL = M3G_E_AL3;
      Capitalize(VAL);

      for (unsigned int VALTyp =0; 
           VALTyp<(sizeof(olcALTypeStr)/sizeof(const char *));
           VALTyp++)
      {
         if (NULL != strstr(VAL, olcALTypeStr[VALTyp]))
         {
            //H223LcParams.videoAL = static_cast<M3G_E_H223_MUX_LVL_TYPE>(VALTyp);
            H223LcParams.videoAL = static_cast<M3G_E_ADAPTATION_LYR_TYPE>(VALTyp);
            break;
         }
      }

      H223LcParams.isAudioSegmentable = ('T' == toupper(c_IsAudioSeg)) ? M3G_TRUE : M3G_FALSE;
      H223LcParams.isVideoSegmentable = ('T' == toupper(c_IsVideoSeg)) ? M3G_TRUE : M3G_FALSE;
      H223LcParams.AL3ControlFieldSize = AL3CntFldSz;
      H223LcParams.AL3SendBufferSize = AL3SndBufSz;

      return EndpointMngr::Instance()->InitH223LCParams(startEP, endEP, &H223LcParams);
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseH264Parms(istringstream& istrLine)
{

   char token[TOKEN_SIZE] = {'\0'};

   char c_acceptRedundantSlices = '\0';
   char c_nalAlignedMode = '\0';
   unsigned int  ProfileIOP = 0;


   // parse line for 3 tokens
   if ((istrLine >> token >> c_acceptRedundantSlices >> c_nalAlignedMode >> ProfileIOP  >> ws) && istrLine.eof())
   {
      unsigned int startEP = 0;
      unsigned int endEP = 0;
      if (0 == sscanf(token, "%d-%d",&startEP, &endEP))
      {
         return B3G_ERROR;
      }

      if (0 == endEP)
      {
         endEP = startEP;
      }

      LOCAL_H264_PARAMS H264Params;

 
      H264Params.acceptRedundantSlices = ('T' == toupper(c_acceptRedundantSlices)) ? M3G_TRUE : M3G_FALSE;
      H264Params.nalAlignedMode = ('T' == toupper(c_nalAlignedMode)) ? M3G_TRUE : M3G_FALSE;
      H264Params.profileIOP = ProfileIOP;

      return EndpointMngr::Instance()->InitH264Params(startEP, endEP, &H264Params);
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}

int DemoConfig::ParseEPTracing(istringstream& istrLine)
{
   char token[TOKEN_SIZE] = {'\0'};
   unsigned int traceLevel = 0;
   char fileName[MAX_FILE_NAME] = {'\0'};

   // parse line for three tokens
   if ((istrLine >> token >> hex >> traceLevel >> dec >> fileName >> ws) && istrLine.eof())
   {
      unsigned int startEP = 0;
      unsigned int endEP = 0;
      if (0 == sscanf(token, "%d-%d",&startEP, &endEP))
      {
         return B3G_ERROR;
      }

      if (0 == endEP)
      {
         endEP = startEP;
      }

      return (EndpointMngr::Instance()->InitEPTracing(startEP, endEP, traceLevel, fileName));
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseBrdTracing(istringstream& istrLine)
{
   unsigned int traceLevel = 0;
   char fileName[MAX_FILE_NAME] = {'\0'};

   // parse line for trace level and fileName tokens
   if ((istrLine >> hex >> traceLevel >> dec >> fileName >> ws) && istrLine.eof())
   {
      return (EndpointMngr::Instance()->InitBoardTracing(traceLevel, fileName));
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseDCI(istringstream& istrLine)
{
   char token[TOKEN_SIZE] = {'\0'};
   unsigned int dciMode = 0;

   // M3G API supports DCI sizes of 255 octets
   // this equates to 510 ASCII characters
   char dciAsciStr[2*OCTET_STRING_SIZE] = {'\0'};

   // parse line for three tokens
   if ((istrLine >> token >> dciMode >> dciAsciStr >> ws) && istrLine.eof())
   {
      unsigned int startEP = 0;
      unsigned int endEP = 0;
      if (0 == sscanf(token, "%d-%d",&startEP, &endEP))
      {
         return B3G_ERROR;
      }

      if (0 == endEP)
      {
         endEP = startEP;
      }

      if (3 < dciMode)
      {
         return B3G_ERROR;
      }

      unsigned char octetStrSize = (char)strlen(dciAsciStr)/2;

      // convert DCI from ASCII to binary:
      unsigned char dciOctetStr[OCTET_STRING_SIZE] = {0};
      for (unsigned int i = 0; i < octetStrSize; i++)
      {
         dciOctetStr[i] = AsciiOctetToHex(&dciAsciStr[2*i]);
      }

      return (EndpointMngr::Instance()->InitEPDCI(startEP, endEP, dciOctetStr,
                                                 octetStrSize, dciMode));
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseVendorId(istringstream& istrLine)
{
   M3G_VENDORID_INFO vendorId;
   INIT_M3G_VENDORID_INFO(&vendorId);

   char oidType[TOKEN_SIZE] = {'\0'};

   // M3G API supports OCTET STRING sizes of 255 octets
   // this equates to 510 ASCII characters
   static char productNumStr[2*OCTET_STRING_SIZE] = {'\0'};
   static char versionNumStr[2*OCTET_STRING_SIZE] = {'\0'};

   // parsing error flags:
   bool  isVendorValid = false;
   bool  isProdAndVersValid = false;

   // parse first token for identifier type
   if (istrLine >> oidType)
   {
      Capitalize(oidType);
      // if an H.245 OID type
      if (NULL != strstr(oidType, "OID"))
      {
         vendorId.vendor.oidType = M3G_E_OID_TYPE;

         // parse line for vendor
         static char vendorStr[2*OCTET_STRING_SIZE] = {'\0'};
         if (istrLine >> vendorStr)
         {
            // convert vendor OID from ASCII to binary:
            unsigned char vendorLength = (unsigned char)strlen(vendorStr)/2;
            vendorId.vendor.oidValue.oid.length = vendorLength;
            for (unsigned int i = 0; i < vendorLength ; i++)
            {
               vendorId.vendor.oidValue.oid.objectId[i] = 
                                            AsciiOctetToHex(&vendorStr[2*i]);
            }
            isVendorValid = true;
         }
      }
      // else if an H.221 NonStandard identifier
      else  if (NULL != strstr(oidType, "H221"))
      {
         vendorId.vendor.oidType = M3G_E_H221_ID_TYPE;

         char      t35CC[TOKEN_SIZE] = {'\0'};
         char      t35Ext[TOKEN_SIZE] = {'\0'};
         char      mnfCode[TOKEN_SIZE] = {'\0'};

         // parse line for H.221 identifier tokens
         if (istrLine >> t35CC >> t35Ext >> mnfCode)
         {
            // convert t35CountryCode string from ASCII to binary:
            vendorId.vendor.oidValue.h221NonStd.t35CountryCode = AsciiOctetToHex(t35CC);

            // convert t35Extension string from ASCII to binary:
            vendorId.vendor.oidValue.h221NonStd.t35Extension = AsciiOctetToHex(t35Ext);

            // convert manufacturerCode string from ASCII to binary:
            vendorId.vendor.oidValue.h221NonStd.manufacturerCode = AsciiOctetToHex(&mnfCode[2]);
            vendorId.vendor.oidValue.h221NonStd.manufacturerCode += (256 * AsciiOctetToHex(&mnfCode[0]));

            isVendorValid = true;
         }
      }

      // parse line for productNumber and versionNumber tokens
      if ((istrLine >> productNumStr >> versionNumStr >> ws) && istrLine.eof())
      {
         // convert productNumber octet string from ASCII to binary:
         unsigned char prodNumLength = (unsigned char)strlen(productNumStr)/2;
         vendorId.productNumber.length = prodNumLength;

         for (unsigned int i = 0; i < prodNumLength ; i++)
         {
            vendorId.productNumber.octet[i] = AsciiOctetToHex(&productNumStr[2*i]);
         }

         // convert versionNumber octet string from ASCII to binary:
         unsigned char versionNumLength = (unsigned char)strlen(versionNumStr)/2;
         vendorId.versionNumber.length = versionNumLength;

         for (unsigned int i = 0; i < versionNumLength ; i++)
         {
            vendorId.versionNumber.octet[i] = AsciiOctetToHex(&versionNumStr[2*i]);
         }

         isProdAndVersValid = true;
      }
   }

   if ((true == isVendorValid) && (true == isProdAndVersValid))
   {
      EndpointMngr::Instance()->InitVendorId(&vendorId);
      return B3G_SUCCESS;
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseSIPInfoDTMF(istringstream& istrLine)
{
   char token[TOKEN_SIZE] = {'\0'};

   // parse line for boolean
   if ((istrLine >> token >> ws) && istrLine.eof())
   {
      Capitalize(token);
      bool isSipInfoDtmfEnabled = strstr(token, "ON") ? true : false;
      EndpointMngr::Instance()->SetSipInfoDtmfEnabled(isSipInfoDtmfEnabled);
      return B3G_SUCCESS;
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}

int DemoConfig::ParseSIPVideo(istringstream& istrLine)
{
   static const char* SipVidStr[] =
   { 
      "NONE",
      "H263",
      "MPEG4",
      "H263+",
      "H264"
   };

   char sipvid1Str[TOKEN_SIZE] = {'\0'};
   char sipvid2Str[TOKEN_SIZE] = {'\0'};
   char* sipvidStrTbl[VID_CODER_LAST];

   // parse line for at least one video cap token:
   if (istrLine >> sipvid1Str)
   {
      // read optional second video cap if specified:
      istrLine >> sipvid2Str;

      Capitalize(sipvid1Str);
      Capitalize(sipvid2Str);
      
      E_SEL_VID_CODER preferredSipVideoList[VID_CODER_LAST];
      sipvidStrTbl[0] = sipvid1Str;
      sipvidStrTbl[1] = sipvid2Str;

      int parmNum;
      for ( parmNum=0; parmNum<VID_CODER_LAST; parmNum++ )
      {
         if ( '\0' == sipvidStrTbl[parmNum][0] )
         {
            preferredSipVideoList[parmNum]=VID_CODER_NONE;
            break;  
         }
         for ( unsigned int vidType =0;
             vidType<(sizeof(SipVidStr)/sizeof(const char *));
             vidType++ )
         {
            if ( NULL != strstr(sipvidStrTbl[parmNum], SipVidStr[vidType]) )
            {
               preferredSipVideoList[parmNum] = static_cast<E_SEL_VID_CODER>(vidType);
               break;
            }
         }
      }
      return EndpointMngr::Instance()->InitSipVideoCoderList(preferredSipVideoList);
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseSIPAudio(istringstream& istrLine)
{
   static const char* SipAudStr[] =
   {
      "NONE",
      "AMR",
      "G723",
      "G711U",
      "G711A",
      "G729"
   };

   char sipaud1Str[TOKEN_SIZE] = {'\0'};
   char sipaud2Str[TOKEN_SIZE] = {'\0'};
   char sipaud3Str[TOKEN_SIZE] = {'\0'};
   char sipaud4Str[TOKEN_SIZE] = {'\0'};
   char* sipaudStrTbl[AUD_CODER_LAST];

   // parse line for at least one audio cap token:
   if (istrLine >> sipaud1Str)
   {
      // read optional audio caps if specified:
      istrLine >> sipaud2Str >> sipaud3Str >> sipaud4Str;

      Capitalize(sipaud1Str);
      Capitalize(sipaud2Str);
      Capitalize(sipaud3Str);
      Capitalize(sipaud4Str);
      
      E_SEL_AUD_CODER preferredSipAudioList[AUD_CODER_LAST];
      sipaudStrTbl[0] = sipaud1Str;
      sipaudStrTbl[1] = sipaud2Str;
      sipaudStrTbl[2] = sipaud3Str;
      sipaudStrTbl[3] = sipaud4Str;

      int parmNum;
      for (parmNum=0; parmNum<AUD_CODER_LAST; parmNum++)
      {
         if ( '\0' == sipaudStrTbl[parmNum][0])
         {
            preferredSipAudioList[parmNum]=AUD_CODER_NONE;
            break;
         }
         for ( unsigned int audType =0;
             audType<(sizeof(SipAudStr)/sizeof(const char *));
             audType++ )
         {
            if ( NULL != strstr(sipaudStrTbl[parmNum], SipAudStr[audType]) )
            {
               preferredSipAudioList[parmNum] = static_cast<E_SEL_AUD_CODER>(audType);
               break;
            }
         }
      }
      return EndpointMngr::Instance()->InitSipAudioCoderList(preferredSipAudioList);
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseSIPInfo(istringstream& istrLine)
{
   // SIP endpoint addresses
   unsigned short localport;
   //char foo[SIP_ADDR_SIZE];
   char originSIPAddr[SIP_ADDR_SIZE] = {'\0'};
   char destSIPAddr[SIP_ADDR_SIZE] = {'\0'};

   // parse line for five tokens
   //if ((istrLine >> localport >> originSIPAddr >> destSIPAddr >> ws) &&
   if ((istrLine >> originSIPAddr >> destSIPAddr >> ws) &&
        istrLine.eof())
   {
      //sscanf(foo, "%s:%u", originSIPAddr, &localport);
      char* pFoo = strstr(originSIPAddr, ":");
      if (pFoo)
      {
         pFoo++;
         localport = atoi(pFoo);
      }
      else
      {
         LOG_ENTRY(0, "Local SIP port not specified, using 5062\n");
         strcat(originSIPAddr, ":5062");
         localport = 5062;
      }
      pFoo = strstr(originSIPAddr, "@");
      if (pFoo == NULL) {
          LOG_ENTRY(0, "Local phone number not specified, using 1000\n");
          char foo[SIP_ADDR_SIZE] = {'\0'};
          sprintf(foo, "1000@%s", originSIPAddr);
          memset(originSIPAddr, 0, sizeof(char)*SIP_ADDR_SIZE);
          strcpy(originSIPAddr, foo);
      }
      return(EndpointMngr::Instance()->SetSIPInfo (localport, originSIPAddr, destSIPAddr));
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


int DemoConfig::ParseCmdInfo(istringstream& istrLine)
{
   // Info for remote ISDN Gateway command socket 
   char connectCmdIPAddr[SIP_ADDR_SIZE] = {'\0'};
   int connectCmdRcvPort = 0;
   int connectCmdSndPort = 0;

   // parse line for three tokens
   if ((istrLine >> connectCmdIPAddr >> connectCmdRcvPort >> connectCmdSndPort >> ws) && istrLine.eof())
   {
      return(EndpointMngr::Instance()->SetCmdInfo ((const char *)connectCmdIPAddr, connectCmdRcvPort, connectCmdSndPort));
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}

int DemoConfig::ParseMMInfo(istringstream& istrLine)
{
   static const char* MMAudStr[] =
   {
      "NONE",
      "AMR",
      "G723",
      "G711U",
      "G711A",
      "G729",
      "PCM"
   };
   
   static const char* MMVidStr[] =
   { 
      "NONE",
      "H263",
      "MPEG4",
      "H263+", 
      "H264"
   };
   
   static const char* MMVidResStr[] =
   { 
      "NONE",
      "SQCIF",
      "QCIF",
      "CIF"
   };
 
   static const char* MMFileTypeStr[] =
   { 
      "DMF",
      "3GP"
   };

   // Info for Multimedia endpoint
   char dtmf;
   char mmOperation[10] = {'\0'};
   char mmAudCoderStr[10] = {'\0'};
   char mmVidCoderStr[10] = {'\0'};
   char mmVidResStr[10] = {'\0'};
   char mmFileTypeStr[10] = {'\0'};
   unsigned int mmBitRate = 0;
   unsigned int mmFrameRate = 0;
   eVIDEO_BITRATE  mmVideoBitRate = VIDEO_BITRATE_DEFAULT;
   eVIDEO_FRAMESPERSEC mmVideoFramesPerSec = VIDEO_FRAMESPERSEC_DEFAULT;
   E_SEL_AUD_CODER mmAudCoder = AUD_CODER_NONE;
   E_SEL_VID_CODER mmVidCoder = VID_CODER_NONE;
   E_SEL_VID_RES mmVidRes = VID_RES_NONE;
   E_SEL_MM_FILE_TYPE mmFileType = MM_FILE_TYPE_DMF;
   char mmPlayAudioFileName[FNAME_SIZE] = {'\0'};
   char mmPlayVideoFileName[FNAME_SIZE] = {'\0'};
   char mmRecordAudioFileName[FNAME_SIZE] = {'\0'};
   char mmRecordVideoFileName[FNAME_SIZE] = {'\0'};


   // parse line for 12 tokens
   if ((istrLine >> dtmf >> mmOperation >> mmFileTypeStr >> mmAudCoderStr >> mmVidCoderStr
         >> mmVidResStr >> mmFrameRate >> mmBitRate >> mmPlayAudioFileName >> mmPlayVideoFileName 
         >> mmRecordAudioFileName >> mmRecordVideoFileName >> ws) && istrLine.eof())
   {
     LOG_ENTRY(0,"ParseMMInfo: DTMF = '%c', MMoperation=%s, mmFileTypeStr=%s, mmAudCoderStr=%s, mmVidCoderStr=%s, mmVidResStr=%s, mmFpsStr=%d, mmKbpsStr=%d\n",  
                 dtmf, mmOperation, mmFileTypeStr, mmAudCoderStr, mmVidCoderStr, mmVidResStr, mmFrameRate, mmBitRate);

     Capitalize(mmOperation);
     Capitalize(mmAudCoderStr);
     Capitalize(mmVidCoderStr);
     Capitalize(mmVidResStr); 
     Capitalize(mmFileTypeStr); 
     if (!(dtmf < '0' && dtmf > '9')) {
     }

     // Get Selected MM File Type value from string value
     for ( unsigned int ft =0;
           ft<(sizeof(MMFileTypeStr)/sizeof(const char *));
           ft++ )
     {
       //printf("ft=%d,MMFileTypeStr[]=%s\n",ft,MMFileTypeStr[ft]);
       if ( NULL != strstr(mmFileTypeStr, MMFileTypeStr[ft]) )
       {
         //printf("!!ft=%d,MMFileTypeStr[]=%s\n",ft,MMFileTypeStr[ft]);
         mmFileType = static_cast<E_SEL_MM_FILE_TYPE>(ft);
         break;
       }
     }

     // Get Selected Audio Coder value from audio string
     for ( unsigned int audType =0;
           audType<(sizeof(MMAudStr)/sizeof(const char *));
           audType++ )
     {
       //printf("audType=%d,MMAudStr[]=%s\n",audType,MMAudStr[audType]);
       if ( NULL != strstr(mmAudCoderStr, MMAudStr[audType]) )
       {
         //printf("!!audType=%d,MMAudStr[]=%s\n",audType,MMAudStr[audType]);
         mmAudCoder = static_cast<E_SEL_AUD_CODER>(audType);
         break;
       }
     }
     
     // Get Selected Video Coder value from video string
     for ( unsigned int vidType =0;
           vidType<(sizeof(MMVidStr)/sizeof(const char *));
           vidType++ )
     {
       //printf("vidType=%d,MMVidStr[]=%s\n",vidType,MMVidStr[vidType]);
       if ( NULL != strstr(mmVidCoderStr, MMVidStr[vidType]) )
       {
         //printf("!!vidType=%d,MMVidStr[]=%s\n",vidType,MMVidStr[vidType]);
           mmVidCoder = static_cast<E_SEL_VID_CODER>(vidType);
         break;
       }
     }
    
     // Get Selected Video Resolution value from video res string
     for ( unsigned int vidRes =0;
           vidRes<(sizeof(MMVidResStr)/sizeof(const char *));
           vidRes++ )
     {
       //printf("vidRes=%d,MMVidResStr[]=%s\n",vidRes,MMVidResStr[vidRes]);
       if ( NULL != strstr(mmVidResStr, MMVidResStr[vidRes]) )
       {
         //printf("!! vidRes=%d,MMVidResStr[]=%s\n",vidRes,MMVidResStr[vidRes]);
         mmVidRes = static_cast<E_SEL_VID_RES>(vidRes);
         break;
       }
     }

     // Validate FrameRate and BitRate required for Record
     //if (0 == strcmp(mmOperation,"RECORD"))
     //{
       // Match the Framerate to the define values
       switch(mmFrameRate)
       {
         case 6:
           mmVideoFramesPerSec = VIDEO_FRAMESPERSEC_6;
           break; 
         case 7:
           mmVideoFramesPerSec = static_cast<eVIDEO_FRAMESPERSEC>(0x70032);  //No VIDEO_FRAMESPERSEC_75 in videodefs.h
           break;    
         case 10:
           mmVideoFramesPerSec = VIDEO_FRAMESPERSEC_10;
           break; 
         case 15:
           mmVideoFramesPerSec = VIDEO_FRAMESPERSEC_15;
           break; 
         case 25:
           mmVideoFramesPerSec = VIDEO_FRAMESPERSEC_25;
           break; 
         case 30:
           mmVideoFramesPerSec = VIDEO_FRAMESPERSEC_30;
           break; 
         default:
           LOG_ERROR(0,"Invalid Frame Rate value (%d), valid values=6,7,10,15,25,30 ( 7=7.5fps)\n", mmFrameRate);
           cout << "Error: Invalid Frame Rate value (" << mmFrameRate 
                << "), valid values=6,7,10,15,25,30 (7=7.5fps)" << endl;
           return B3G_ERROR;
       } 
	
       //Convert Bitrate into Kbps
       if (mmBitRate > 384)
           mmVideoBitRate = VIDEO_BITRATE_384K; 
       else
          mmVideoBitRate = static_cast<eVIDEO_BITRATE>(mmBitRate*1000); 

      
      //printf("FileType=%d, Audio codec =%d, Video codec=%d, VideoRes=%d\n", mmFileType, mmAudCoder, mmVidCoder, mmVidRes); 
      return(EndpointMngr::Instance()->SetMMInfo (dtmf, (const char *) mmOperation,(E_SEL_MM_FILE_TYPE) mmFileType,
             (E_SEL_AUD_CODER) mmAudCoder,(E_SEL_VID_CODER) mmVidCoder,(E_SEL_VID_RES) mmVidRes, 
             (eVIDEO_FRAMESPERSEC)mmVideoFramesPerSec,(eVIDEO_BITRATE)mmVideoBitRate, 
             (const char *)mmPlayAudioFileName, (const char *) mmPlayVideoFileName, 
             (const char *) mmRecordAudioFileName, (const char *) mmRecordVideoFileName));
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}

int DemoConfig::ParseNbUPInfo(istringstream& istrLine)
{
   // Info for Nbup transport for 3G endpoints
   // Note: expecting one NBUPINFO line per 3G device using Nbup
   unsigned short associated3GEndpoint = 0;
   char nbupType = '\0';   // Will be S)end or R)eceive
   char remoteIPAddr[SIP_ADDR_SIZE] = {'\0'};
   unsigned short remoteNbupPort = 0;

   // parse line for four tokens
   if ((istrLine >> associated3GEndpoint >> nbupType >> remoteIPAddr >> remoteNbupPort  >> ws) && istrLine.eof())
   {
      return(EndpointMngr::Instance()->SetNbupInfo (associated3GEndpoint, nbupType, (const char *)remoteIPAddr, 
             remoteNbupPort));
   }
   else  // invalid format
   {
      return B3G_ERROR;
   }
}


unsigned char AsciiOctetToHex(const char *pOctet)
{
   unsigned char hexDigit = 0;
   unsigned char value = 0;

   for (unsigned int numDigits = 0; numDigits < 2; numDigits++, pOctet++)
   {
      if (*pOctet >= '0' && *pOctet <= '9')
         hexDigit = *pOctet - '0';
      // lineBuffer was converted to upper case, but for flexibility in reuse:
      else if (*pOctet >= 'a' && *pOctet <= 'f')
         hexDigit = (*pOctet - 'a') + 10;
      else if (*pOctet >= 'A' && *pOctet <= 'F')
         hexDigit = (*pOctet - 'A') + 10;

      value =  (value << 4) + hexDigit;
   }

   return value;
}


void Capitalize(string& str)
{
   // capitalize all text:
   for (unsigned int i = 0; i < str.length(); i++)
   {
        str[i] = toupper(str[i]);
   }
}


void Capitalize(char *c_str)
{
   // capitalize all text:
   for (unsigned int i = 0; i < strlen(c_str); i++)
   {
        c_str[i] = toupper(c_str[i]);
   }
}
