/**
* @file connecting_state.cpp
* @brief Definition of ConnectingState class
* @date Feb. 16, 2007
*
* DIALOGIC CONFIDENTIAL
* Copyright  2007 Dialogic Corporation. All Rights Reserved.
* 
* The source code contained or described herein and all documents related to
* the source code (Material) are owned by Dialogic Corporation or its suppliers
* or licensors. Title to the Material remains with Dialogic Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Dialogic or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Dialogic's prior express written permission.
* 
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Dialogic in writing.
*/

#include "connecting_state.h"
#include "endpointmgr.h"
#include "logger.h"
#include "endpoint.h"
#include "appevents.h"


// static constant members:
const int ConnectingState::AUDIO_TX_PORTS_RCVD = 0x01;
const int ConnectingState::AUDIO_RX_PORTS_RCVD = 0x02;
const int ConnectingState::VIDEO_TX_PORTS_RCVD = 0x04;
const int ConnectingState::VIDEO_RX_PORTS_RCVD = 0x08;
const int ConnectingState::MEDIA_PORTS_RCVD    = 0x0F;

const int ConnectingState::NETWORK_CONNECTED   = 0x10;
const int ConnectingState::CONTROL_CONNECTED   = 0x20;
const int ConnectingState::AUDIO_CONNECTED     = 0x40;
const int ConnectingState::VIDEO_CONNECTED     = 0x80;
const int ConnectingState::LOOPBACK_CONNECTED  = 0xE0;
const int ConnectingState::M3G_CONNECTED       = 0xE0;
const int ConnectingState::FULLY_CONNECTED     = 0xF0;

const int ConnectingState::NBUP_3G_TX_PORTS_RCVD  = 0x1000;
const int ConnectingState::NBUP_3G_RX_PORTS_RCVD  = 0x2000;
const int ConnectingState::NBUP_IPM_TX_PORTS_RCVD = 0x4000;
const int ConnectingState::NBUP_IPM_RX_PORTS_RCVD = 0x8000;
const int ConnectingState::ALL_NBUP_PORTS_RCVD    = 0xF000;


//*****************************************************************************
// Function: ConnectingState::ConnectingState(Endpoint *pEndpoint)
// Description: Initializing constructor
// Return:  ConnectingState*
// Parameters: Endpoint *pEndpoint 
//*****************************************************************************
ConnectingState::ConnectingState(Endpoint* pEndpoint) :
   EPState(pEndpoint)
{
}

//*****************************************************************************
// Function: ConnectingState::~ConnectingState()
// Description: Destructor
// Return:  none
// Parameters: none 
//*****************************************************************************
ConnectingState::~ConnectingState()
{
}

//*****************************************************************************
// Function: void ConnectingState::ProcessEvent(long evtType, void *pEvtData, long evtLen, long evtDev)
// Description: Process an event
// Return: none
// Parameters: long evtType 
//             void *pEvtData 
//             long evtLen 
//             long evtDev 
//*****************************************************************************
void ConnectingState::ProcessEvent(long  evtType, void *pEvtData, long  evtLen, long  evtDev)
{
   switch ( evtType )
   {
      case DMEV_GET_TX_PORT_INFO:
         {
            DM_PORT_INFO_LIST* pTxPortInfoList = reinterpret_cast<DM_PORT_INFO_LIST*>(pEvtData);
            m_pEndpoint->RecordPortInfoList(evtDev, evtType, pTxPortInfoList);
            if ( evtDev == m_pEndpoint->GetAudioHandle() )
            {
               m_pEndpoint->SetAudioTxPortInfoList(pTxPortInfoList);
               SetStatus(AUDIO_TX_PORTS_RCVD);
               CheckToInitPortConnection();
            }
            else if ( evtDev == m_pEndpoint->GetVideoHandle())
            {
               m_pEndpoint->SetVideoTxPortInfoList(pTxPortInfoList);
               SetStatus(VIDEO_TX_PORTS_RCVD);
               CheckToInitPortConnection();
            }
            else if ( evtDev == m_pEndpoint->GetNbupHandle())
            {
                // Nbup (IPML) port info.  Only saving Nbup info, no audio/video
               for ( unsigned int i=0; i< pTxPortInfoList->unCount; i++ )
               {
                  if (pTxPortInfoList->port_info[i].port_media_type == DM_PORT_MEDIA_TYPE_NBUP)
                    m_pEndpoint->RecordIPMNbupPortInfo(evtDev, evtType, &(pTxPortInfoList->port_info[i]));
               }
               SetStatus(NBUP_IPM_TX_PORTS_RCVD);
               CheckToInitNbupPortConnection();
            }
            else if ( evtDev == m_pEndpoint->GetControlHandle())
            {
               // Nbup (3G) port info
               m_pEndpoint->Set3GNbupTxPortInfoList(pTxPortInfoList);
               SetStatus(NBUP_3G_TX_PORTS_RCVD);
               CheckToInitNbupPortConnection();
            }
         }
         break;

      case DMEV_GET_RX_PORT_INFO:
         {
            DM_PORT_INFO_LIST* pRxPortInfoList = reinterpret_cast<DM_PORT_INFO_LIST*>(pEvtData);
            m_pEndpoint->RecordPortInfoList(evtDev, evtType, pRxPortInfoList);
            if ( evtDev == m_pEndpoint->GetAudioHandle() )
            {
               m_pEndpoint->SetAudioRxPortInfoList(pRxPortInfoList);
               SetStatus(AUDIO_RX_PORTS_RCVD);
               CheckToInitPortConnection();
            }
            else if ( evtDev == m_pEndpoint->GetVideoHandle() )
            {
               m_pEndpoint->SetVideoRxPortInfoList(pRxPortInfoList);
               SetStatus(VIDEO_RX_PORTS_RCVD);
               CheckToInitPortConnection();
            }
            else if ( evtDev == m_pEndpoint->GetNbupHandle())
            {
                // Nbup (IPML) port info.  Only saving Nbup info, no audio/video
               for ( unsigned int i=0; i< pRxPortInfoList->unCount; i++ )
               {
                  if (pRxPortInfoList->port_info[i].port_media_type == DM_PORT_MEDIA_TYPE_NBUP)
                    m_pEndpoint->RecordIPMNbupPortInfo(evtDev, evtType, &(pRxPortInfoList->port_info[i]));
               }
               SetStatus(NBUP_IPM_RX_PORTS_RCVD);
               CheckToInitNbupPortConnection();
            }
            else if ( evtDev == m_pEndpoint->GetControlHandle())
            {
               // Nbup (3G) port info
               m_pEndpoint->Set3GNbupRxPortInfoList(pRxPortInfoList);
               SetStatus(NBUP_3G_RX_PORTS_RCVD);
               CheckToInitNbupPortConnection();
            }
         }
         break;

      case DMEV_CONNECT:      // fall-thru
      case DMEV_PORT_CONNECT:
         ProcessDeviceConnect(evtDev);
         break;

      case DMEV_GET_TX_PORT_INFO_FAIL:  // fall-thru
      case DMEV_GET_RX_PORT_INFO_FAIL:
      case DMEV_CONNECT_FAIL:           // fall-thru
      case DMEV_PORT_CONNECT_FAIL:      // fall-thru
         LOG_ERROR(evtDev, "connection failure\n");
         Shutdown();
         break;

      case M3GEV_RESET_CMPLT:
         ProcessResetComplete();
        break;

      // Nbup-related events on IPM devices used by 3G endpoint
      case IPMEV_EVENT_ENABLED:
         LOG_ENTRY(evtDev, "IPMEV_EVENT_ENABLED received\n");
         m_pEndpoint->StartNbupMedia();
         break;

      case IPMEV_STARTMEDIA:
         LOG_ENTRY(evtDev, "IPMEV_STARTMEDIA received\n");
         //LOG_ENTRY(evtDev, "ISNbup = %d\n", m_pEndpoint->IsNbupSend());
         // If this endpoint is designated as the Send side, start the Nbup protocol
         if (m_pEndpoint->IsNbupSend())
           m_pEndpoint->NbupInitSend();
         break;

      case IPMEV_INIT_SEND:
         LOG_ENTRY(evtDev, "IPMEV_INIT_SEND received\n");
         break;

      case IPMEV_INIT_RECEIVED:
        {
          LOG_ENTRY(evtDev, "IPMEV_INIT_RECEIVED received\n");
          // Check to see what type of init we received
           IPM_INIT_RECEIVED *pRecv = (IPM_INIT_RECEIVED *) sr_getevtdatap ();
           if (!pRecv)
             {
              LOG_ERROR(evtDev, "Invalid pointer returned from sr_getevtdatap()");
             }
           if (pRecv->eProtocol == RTP_PROTOCOL_NBUP)
             {
               // We've received an NBUP invitation. Send an ACK to start the session
               m_pEndpoint->NbupInitResponseSend();
             }
           else
             {
               LOG_ERROR (evtDev, "Unsupported protocol (%d) in init request.", pRecv->eProtocol);
             }
          } 
         break;

      case IPMEV_INIT_RESPONSE_SEND:
         LOG_ENTRY(evtDev, "IPMEV_INIT_RESPONSE_SEND received\n");
         break;

      case IPMEV_INIT_COMPLETE:
         LOG_ENTRY(evtDev, "IPMEV_INIT_COMPLETE received\n");
         // Nbup initialization complete, so can now move on to a 3G ready state
         m_pEndpoint->ChangeState(H245_INACTIVE_STATE);
         m_pEndpoint->Notify(APP_M3G_ENDPOINT_PORT_CONNECTED);
         break;

      case IPMEV_INIT_FAILED:
         LOG_ERROR(evtDev, "IPMEV_INIT_FAILED received\n");
         break;

      case IPMEV_ERROR:
         LOG_ERROR(evtDev, "IPMEV_ERROR received\n");
         break;

      case GCEV_UNBLOCKED:
         LOG_ENTRY(evtDev, "ConnectingState: GCEV_UNBLOCKED received.\n");
         break; 

      default:
         LOG_ERROR(evtDev,"ConnectingState: Unexpected event type: 0x%x\n", evtType);
         break;
   }
}

//*****************************************************************************
// Function: void ConnectingState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
// Description: Process a request from the user interface
// Return: void 
// Parameters: EPState::E_USER_PROMPT_TYPE eUserPrompt 
//*****************************************************************************
void  ConnectingState::ProcessUserPrompt(EPState::E_USER_PROMPT_TYPE eUserPrompt)
{
   LOG_ERROR(0,"Unexpected user prompt on endpoint[%d] in CONNECTING state: %d\n",
             m_pEndpoint->GetIndex(), static_cast<int>(eUserPrompt));
}

//*****************************************************************************
// Function: void ConnectingState::Shutdown()
// Description: Process a shutdown request
// Return: void 
// Parameters: none 
//*****************************************************************************
void ConnectingState::Shutdown()
{
  LOG_ENTRY(m_pEndpoint->GetControlHandle(), "Shutdown called in %s state.  Resetting M3g\n", GetStateStr());
  m_pEndpoint->Reinitialize();
  m_pEndpoint->StopNbupMedia();
}


//*****************************************************************************
// Function: void ConnectingState::ProcessDeviceConnect(long evtDev)
// Description: Process a device connect event
// Return: void 
// Parameters: long evtDev 
//*****************************************************************************
void ConnectingState::ProcessDeviceConnect(long evtDev)
{
   if ( evtDev == m_pEndpoint->GetNetworkDeviceHandle() )
   {
      SetStatus(NETWORK_CONNECTED);
   }
   else if ( evtDev == m_pEndpoint->GetControlHandle() )
   {
      SetStatus(CONTROL_CONNECTED);
   }
   else if ( evtDev == m_pEndpoint->GetAudioHandle() )
   {
      SetStatus(AUDIO_CONNECTED);
   }
   else if ( evtDev == m_pEndpoint->GetVideoHandle() )
   {
      SetStatus(VIDEO_CONNECTED);
   }

   LOG_ENTRY(evtDev, "Netwk handle = %d\n", m_pEndpoint->GetNetworkHandle());
   LOG_ENTRY(evtDev,"Endpoint[%d] ProcessDeviceConnect on %d status:%04X\n", m_pEndpoint->GetIndex(),evtDev, m_StatusBitmask);

   if ( true == IsEndpointFullyConnected() )
   {
      m_pEndpoint->ChangeState(H245_INACTIVE_STATE);
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_PORT_CONNECTED);
   }
}


//*****************************************************************************
// Function: void ConnectingState::CheckToInitPortConnection()
// Description: If all ports are initialized, notify the BridgedCall
// Return: void 
// Parameters: none 
//*****************************************************************************
void ConnectingState::CheckToInitPortConnection()
{
   bool haveLocalPorts = IsEndpointAllPortsRcvd();

      if (m_pEndpoint->GetLoopback())
        LOG_ENTRY(0,"Endpoint[%d] CheckToInitPortConnection Loopback Endpoint[%d] isallportsrecvd bool value = %d\n", 
              m_pEndpoint->GetIndex(), m_pEndpoint->GetLoopback()->GetIndex(),  m_pEndpoint->GetLoopback()->IsEndpointAllPortsRcvd());

   bool haveLoopbackPorts = m_pEndpoint->GetLoopback()
                            ? m_pEndpoint->GetLoopback()->IsEndpointAllPortsRcvd()
                            : true;
    LOG_ENTRY(0,"Endpoint[%d] CheckToInitPortConnection-haveLocalPorts=%d,haveLoopbackPorts=%d\n",
              m_pEndpoint->GetIndex(), haveLocalPorts, haveLoopbackPorts);
   if ( (true == haveLocalPorts) && (true == haveLoopbackPorts) )
   {
      LOG_ENTRY(0,"Endpoint[%d] CheckToInitPortConnection-all ports received\n",m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_OPENED);
      if ( m_pEndpoint->GetLoopback() )
      {
         m_pEndpoint->GetLoopback()->Notify(APP_M3G_ENDPOINT_OPENED);
      }
      else 
         LOG_ENTRY(0,"Endpoint[%d] haveLocalPorts = %d, haveLoopbackPorts = %d\n",m_pEndpoint->GetIndex(), haveLocalPorts, haveLoopbackPorts);
   }
}

void ConnectingState::CheckToInitNbupPortConnection()
{
   LOG_ENTRY(0,"Endpoint[%d] CheckToInitNbupPortConnection bitmask - 0x%x bool value = %d\n",
             m_pEndpoint->GetIndex(), m_StatusBitmask,
             IsStatusSet(ALL_NBUP_PORTS_RCVD));
   // Need a Tx and Rx return from both 3G and NBUP.  Then can move on to Port Connect.
   if (IsStatusSet(ALL_NBUP_PORTS_RCVD))
   {
     LOG_ENTRY(0,"Endpoint[%d] CheckToInitNbupPortConnection - all Nbup ports received. Doing dev_PortConnects \n",m_pEndpoint->GetIndex());

     // Ready for dev_PortConnects
     m_pEndpoint->ConnectIPMNbupTo3GNbup(m_pEndpoint->GetNetworkHandle(), 
                                         m_pEndpoint->GetIPMNbupTxPortInfo(), 
                                         m_pEndpoint->Get3GNbupRxPortInfo());
     m_pEndpoint->ConnectIPMNbupTo3GNbup(m_pEndpoint->GetControlHandle(), 
                                         m_pEndpoint->Get3GNbupTxPortInfo(), 
                                         m_pEndpoint->GetIPMNbupRxPortInfo());
   }
}

//*****************************************************************************
// Function: bool ConnectingState::IsEndpointFullyConnected()
// Description: Return true if all ports are connected
// Return: bool 
// Parameters: none 
//*****************************************************************************
bool ConnectingState::IsEndpointFullyConnected()
{
   bool retVal = false;
   // First check for loopback mode 
   if ( true == m_pEndpoint->GetIsLoopbackNotNetwork() )
   {
      retVal = (IsStatusSet(LOOPBACK_CONNECTED));
      LOG_ENTRY(0,"Endpoint[%d] Check for loopback connection = %d\n",
                m_pEndpoint->GetIndex(), retVal);
   }
   else  // else 3G-3G device over soft CT bus or Nbup
   {
      if (m_pEndpoint->IsClearChannel() || m_pEndpoint->IsSigProtocolManaged())
      {
        retVal = IsStatusSet(FULLY_CONNECTED);
        LOG_ENTRY(0,"Endpoint[%d] Check for full connection = %d\n",
                  m_pEndpoint->GetIndex(), retVal);
      }
      else
      {
        // Nbup - trigger the start of the Nbup connection if the 3G side is ready
        bool nbupRetVal = IsStatusSet(M3G_CONNECTED);
        LOG_ENTRY(0,"Endpoint[%d] Check to start Nbup connection process.  0x%x = boolean %d\n",
	          m_pEndpoint->GetIndex(), m_StatusBitmask, nbupRetVal);
        if (nbupRetVal)
        {
           // 3G side is ready; Begin Nbup connection process.
           m_pEndpoint->EnableNbupEvents();
        }
        // Not ready to move the 3G endpoint to a fully connected state.  That is done
        // after Nbup is ready
        return false;
      }
   }
   return retVal;
}

//*****************************************************************************
// Function: void ConnectingState::ProcessResetComplete()
// Description: M3G reset done; set state to inactive to trigger shutdown
// Return: void
// Parameters: none
//*****************************************************************************
void ConnectingState::ProcessResetComplete()
{
      LOG_ENTRY(0,"Endpoint[%d] M3g reset complete.  Triggering shutdown.\n",
                m_pEndpoint->GetIndex());
      LOG_ENTRY(0,"Endpoint[%d] Notify(APP_M3G_ENDPOINT_DISCONNECTED)\n",
                m_pEndpoint->GetIndex());
      m_pEndpoint->Notify(APP_M3G_ENDPOINT_DISCONNECTED);
      m_pEndpoint->ChangeState(H245_INACTIVE_STATE); 
}

